# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['backoff']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'backoff',
    'version': '2.1.2',
    'description': 'Function decoration for backoff and retry',
    'long_description': 'backoff\n=======\n\n.. image:: https://travis-ci.org/litl/backoff.svg\n    :target: https://travis-ci.org/litl/backoff\n.. image:: https://coveralls.io/repos/litl/backoff/badge.svg\n    :target: https://coveralls.io/r/litl/backoff?branch=python-3\n.. image:: https://github.com/litl/backoff/workflows/CodeQL/badge.svg\n    :target: https://github.com/litl/backoff/actions/workflows/codeql-analysis.yml\n.. image:: https://img.shields.io/pypi/v/backoff.svg\n    :target: https://pypi.python.org/pypi/backoff\n.. image:: https://img.shields.io/github/license/litl/backoff\n    :target: https://github.com/litl/backoff/blob/master/LICENSE\n\n**Function decoration for backoff and retry**\n\nThis module provides function decorators which can be used to wrap a\nfunction such that it will be retried until some condition is met. It\nis meant to be of use when accessing unreliable resources with the\npotential for intermittent failures i.e. network resources and external\nAPIs. Somewhat more generally, it may also be of use for dynamically\npolling resources for externally generated content.\n\nDecorators support both regular functions for synchronous code and\n`asyncio <https://docs.python.org/3/library/asyncio.html>`__\'s coroutines\nfor asynchronous code.\n\nExamples\n========\n\nSince Kenneth Reitz\'s `requests <http://python-requests.org>`_ module\nhas become a defacto standard for synchronous HTTP clients in Python,\nnetworking examples below are written using it, but it is in no way required\nby the backoff module.\n\n@backoff.on_exception\n---------------------\n\nThe ``on_exception`` decorator is used to retry when a specified exception\nis raised. Here\'s an example using exponential backoff when any\n``requests`` exception is raised:\n\n.. code-block:: python\n\n    @backoff.on_exception(backoff.expo,\n                          requests.exceptions.RequestException)\n    def get_url(url):\n        return requests.get(url)\n\nThe decorator will also accept a tuple of exceptions for cases where\nthe same backoff behavior is desired for more than one exception type:\n\n.. code-block:: python\n\n    @backoff.on_exception(backoff.expo,\n                          (requests.exceptions.Timeout,\n                           requests.exceptions.ConnectionError))\n    def get_url(url):\n        return requests.get(url)\n\n**Give Up Conditions**\n\nOptional keyword arguments can specify conditions under which to give\nup.\n\nThe keyword argument ``max_time`` specifies the maximum amount\nof total time in seconds that can elapse before giving up.\n\n.. code-block:: python\n\n    @backoff.on_exception(backoff.expo,\n                          requests.exceptions.RequestException,\n                          max_time=60)\n    def get_url(url):\n        return requests.get(url)\n\n\nKeyword argument ``max_tries`` specifies the maximum number of calls\nto make to the target function before giving up.\n\n.. code-block:: python\n\n    @backoff.on_exception(backoff.expo,\n                          requests.exceptions.RequestException,\n                          max_tries=8,\n                          jitter=None)\n    def get_url(url):\n        return requests.get(url)\n\n\nIn some cases the raised exception instance itself may need to be\ninspected in order to determine if it is a retryable condition. The\n``giveup`` keyword arg can be used to specify a function which accepts\nthe exception and returns a truthy value if the exception should not\nbe retried:\n\n.. code-block:: python\n\n    def fatal_code(e):\n        return 400 <= e.response.status_code < 500\n\n    @backoff.on_exception(backoff.expo,\n                          requests.exceptions.RequestException,\n                          max_time=300,\n                          giveup=fatal_code)\n    def get_url(url):\n        return requests.get(url)\n\nBy default, when a give up event occurs, the exception in question is reraised\nand so code calling an `on_exception`-decorated function may still\nneed to do exception handling. This behavior can optionally be disabled\nusing the `raise_on_giveup` keyword argument.\n\nIn the code below, `requests.exceptions.RequestException` will not be raised\nwhen giveup occurs. Note that the decorated function will return `None` in this\ncase, regardless of the logic in the `on_exception` handler.\n\n.. code-block:: python\n\n    def fatal_code(e):\n        return 400 <= e.response.status_code < 500\n\n    @backoff.on_exception(backoff.expo,\n                          requests.exceptions.RequestException,\n                          max_time=300,\n                          raise_on_giveup=False,\n                          giveup=fatal_code)\n    def get_url(url):\n        return requests.get(url)\n\nThis is useful for non-mission critical code where you still wish to retry\nthe code inside of `backoff.on_exception` but wish to proceed with execution\neven if all retries fail.\n\n@backoff.on_predicate\n---------------------\n\nThe ``on_predicate`` decorator is used to retry when a particular\ncondition is true of the return value of the target function.  This may\nbe useful when polling a resource for externally generated content.\n\nHere\'s an example which uses a fibonacci sequence backoff when the\nreturn value of the target function is the empty list:\n\n.. code-block:: python\n\n    @backoff.on_predicate(backoff.fibo, lambda x: x == [], max_value=13)\n    def poll_for_messages(queue):\n        return queue.get()\n\nExtra keyword arguments are passed when initializing the\nwait generator, so the ``max_value`` param above is passed as a keyword\narg when initializing the fibo generator.\n\nWhen not specified, the predicate param defaults to the falsey test,\nso the above can more concisely be written:\n\n.. code-block:: python\n\n    @backoff.on_predicate(backoff.fibo, max_value=13)\n    def poll_for_message(queue):\n        return queue.get()\n\nMore simply, a function which continues polling every second until it\ngets a non-falsey result could be defined like like this:\n\n.. code-block:: python\n\n    @backoff.on_predicate(backoff.constant, jitter=None, interval=1)\n    def poll_for_message(queue):\n        return queue.get()\n\nThe jitter is disabled in order to keep the polling frequency fixed.  \n\nJitter\n------\n\nA jitter algorithm can be supplied with the ``jitter`` keyword arg to\neither of the backoff decorators. This argument should be a function\naccepting the original unadulterated backoff value and returning it\'s\njittered counterpart.\n\nAs of version 1.2, the default jitter function ``backoff.full_jitter``\nimplements the \'Full Jitter\' algorithm as defined in the AWS\nArchitecture Blog\'s `Exponential Backoff And Jitter\n<https://www.awsarchitectureblog.com/2015/03/backoff.html>`_ post.\nNote that with this algorithm, the time yielded by the wait generator\nis actually the *maximum* amount of time to wait.\n\nPrevious versions of backoff defaulted to adding some random number of\nmilliseconds (up to 1s) to the raw sleep value. If desired, this\nbehavior is now available as ``backoff.random_jitter``.\n\nUsing multiple decorators\n-------------------------\n\nThe backoff decorators may also be combined to specify different\nbackoff behavior for different cases:\n\n.. code-block:: python\n\n    @backoff.on_predicate(backoff.fibo, max_value=13)\n    @backoff.on_exception(backoff.expo,\n                          requests.exceptions.HTTPError,\n                          max_time=60)\n    @backoff.on_exception(backoff.expo,\n                          requests.exceptions.Timeout,\n                          max_time=300)\n    def poll_for_message(queue):\n        return queue.get()\n\n\nRuntime Configuration\n---------------------\n\nThe decorator functions ``on_exception`` and ``on_predicate`` are\ngenerally evaluated at import time. This is fine when the keyword args\nare passed as constant values, but suppose we want to consult a\ndictionary with configuration options that only become available at\nruntime. The relevant values are not available at import time. Instead,\ndecorator functions can be passed callables which are evaluated at\nruntime to obtain the value:\n\n.. code-block:: python\n\n    def lookup_max_time():\n        # pretend we have a global reference to \'app\' here\n        # and that it has a dictionary-like \'config\' property\n        return app.config["BACKOFF_MAX_TIME"]\n\n    @backoff.on_exception(backoff.expo,\n                          ValueError,\n                          max_time=lookup_max_time)\n\nEvent handlers\n--------------\n\nBoth backoff decorators optionally accept event handler functions\nusing the keyword arguments ``on_success``, ``on_backoff``, and ``on_giveup``.\nThis may be useful in reporting statistics or performing other custom\nlogging.\n\nHandlers must be callables with a unary signature accepting a dict\nargument. This dict contains the details of the invocation. Valid keys\ninclude:\n\n* *target*: reference to the function or method being invoked\n* *args*: positional arguments to func\n* *kwargs*: keyword arguments to func\n* *tries*: number of invocation tries so far\n* *elapsed*: elapsed time in seconds so far\n* *wait*: seconds to wait (``on_backoff`` handler only)\n* *value*: value triggering backoff (``on_predicate`` decorator only)\n\nA handler which prints the details of the backoff event could be\nimplemented like so:\n\n.. code-block:: python\n\n    def backoff_hdlr(details):\n        print ("Backing off {wait:0.1f} seconds after {tries} tries "\n               "calling function {target} with args {args} and kwargs "\n               "{kwargs}".format(**details))\n\n    @backoff.on_exception(backoff.expo,\n                          requests.exceptions.RequestException,\n                          on_backoff=backoff_hdlr)\n    def get_url(url):\n        return requests.get(url)\n\n**Multiple handlers per event type**\n\nIn all cases, iterables of handler functions are also accepted, which\nare called in turn. For example, you might provide a simple list of\nhandler functions as the value of the ``on_backoff`` keyword arg:\n\n.. code-block:: python\n\n    @backoff.on_exception(backoff.expo,\n                          requests.exceptions.RequestException,\n                          on_backoff=[backoff_hdlr1, backoff_hdlr2])\n    def get_url(url):\n        return requests.get(url)\n\n**Getting exception info**\n\nIn the case of the ``on_exception`` decorator, all ``on_backoff`` and\n``on_giveup`` handlers are called from within the except block for the\nexception being handled. Therefore exception info is available to the\nhandler functions via the python standard library, specifically\n``sys.exc_info()`` or the ``traceback`` module.\n\nAsynchronous code\n-----------------\n\nBackoff supports asynchronous execution in Python 3.5 and above.\n\nTo use backoff in asynchronous code based on\n`asyncio <https://docs.python.org/3/library/asyncio.html>`__\nyou simply need to apply ``backoff.on_exception`` or ``backoff.on_predicate``\nto coroutines.\nYou can also use coroutines for the ``on_success``, ``on_backoff``, and\n``on_giveup`` event handlers, with the interface otherwise being identical.\n\nThe following examples use `aiohttp <https://aiohttp.readthedocs.io/>`__\nasynchronous HTTP client/server library.\n\n.. code-block:: python\n\n    @backoff.on_exception(backoff.expo, aiohttp.ClientError, max_time=60)\n    async def get_url(url):\n        async with aiohttp.ClientSession(raise_for_status=True) as session:\n            async with session.get(url) as response:\n                return await response.text()\n\nLogging configuration\n---------------------\n\nBy default, backoff and retry attempts are logged to the \'backoff\'\nlogger. By default, this logger is configured with a NullHandler, so\nthere will be nothing output unless you configure a handler.\nProgrammatically, this might be accomplished with something as simple\nas:\n\n.. code-block:: python\n\n    logging.getLogger(\'backoff\').addHandler(logging.StreamHandler())\n\nThe default logging level is INFO, which corresponds to logging\nanytime a retry event occurs. If you would instead like to log\nonly when a giveup event occurs, set the logger level to ERROR.\n\n.. code-block:: python\n\n    logging.getLogger(\'backoff\').setLevel(logging.ERROR)\n\nIt is also possible to specify an alternate logger with the ``logger``\nkeyword argument.  If a string value is specified the logger will be\nlooked up by name.\n\n.. code-block:: python\n\n   @backoff.on_exception(backoff.expo,\n                         requests.exceptions.RequestException,\n\t\t\t logger=\'my_logger\')\n   # ...\n\nIt is also supported to specify a Logger (or LoggerAdapter) object\ndirectly.\n\n.. code-block:: python\n\n    my_logger = logging.getLogger(\'my_logger\')\n    my_handler = logging.StreamHandler()\n    my_logger.addHandler(my_handler)\n    my_logger.setLevel(logging.ERROR)\n\n    @backoff.on_exception(backoff.expo,\n                          requests.exceptions.RequestException,\n\t\t\t  logger=my_logger)\n    # ...\n\nDefault logging can be disabled all together by specifying\n``logger=None``. In this case, if desired alternative logging behavior\ncould be defined by using custom event handlers.\n',
    'author': 'Bob Green',
    'author_email': 'rgreen@aquent.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/litl/backoff',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
