# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['json_excel_converter',
 'json_excel_converter.csv',
 'json_excel_converter.xlsx']

package_data = \
{'': ['*']}

extras_require = \
{'xlsxwriter': ['xlsxwriter>=1.2,<2.0']}

setup_kwargs = {
    'name': 'json-excel-converter',
    'version': '1.0.0',
    'description': 'JSON to excel convertor',
    'long_description': '# JSON to excel converter\n\n[![](https://img.shields.io/github/license/oarepo/json-excel-converter.svg)](https://github.com/oarepo/json-excel-converter/blob/master/LICENSE)\n[![](https://img.shields.io/travis/oarepo/json-excel-converter.svg)](https://travis-ci.org/oarepo/json-excel-converter)\n[![](https://img.shields.io/coveralls/oarepo/json-excel-converter.svg)](https://coveralls.io/r/oarepo/json-excel-converter)\n[![](https://img.shields.io/pypi/v/json-excel-converter.svg)](https://pypi.org/pypi/json-excel-converter)\n\nA package that converts json to CSV, excel or other table formats\n\n## Sample output\n\n### Simple json\n\n```json\n[\n  {\n    "col1": "val1",\n    "col2": "val2" \n  }\n]\n```\n\nthe generated CSV/excel is:\n\n```\ncol1          col2\n==================\nval1          val2\n```\n\n### Nested json\n\n```json\n[\n  {\n    "col1": "val1",\n    "col2": {\n      "col21": "val21",\n      "col22": "val22"\n    }\n  }\n]\n```\n\nthe generated CSV/excel is (in excel, col2 spans two cells horizontally):\n\n```\ncol1          col2\n              col21         col22\n=================================\nval1          val21         val22\n```\n\n### json with array property\n\n```json\n[\n  {\n    "col1": "val1",\n    "col2": [\n      {\n        "col21": "val21"\n      },\n      {\n        "col21": "val22"\n      }\n    ]\n  }\n]\n```\n\nthe generated CSV/excel is (in excel, col2 spans two cells horizontally):\n\n```\ncol1          col2         \n              col21         col21\n=================================\nval1          val21         val22\n```\n\n\n## Installation\n\n```bash\npip install json-excel-converter[extra]\n```\n\nwhere extra is:\n\n * ``xlsxwriter`` to use the xlsxwriter\n\n## Usage\n\n### Simple usage\n\n```python\n\nfrom json_excel_converter import Converter \nfrom json_excel_converter.xlsx import Writer\n\ndata = [\n    {\'a\': [1], \'b\': \'hello\'},\n    {\'a\': [1, 2, 3], \'b\': \'world\'}\n]\n\nconv = Converter()\nconv.convert(data, Writer(file=\'/tmp/test.xlsx\'))\n```\n\n### Streaming usage with restarts\n\n```python\n\nfrom json_excel_converter import Converter, LinearizationError \nfrom json_excel_converter.csv import Writer\n\nconv = Converter()\nwriter = Writer(file=\'/tmp/test.csv\')\nwhile True:\n    try:\n        data = get_streaming_data()     # custom function to get iterator of data\n        conv.convert_streaming(data, writer)\n        break\n    except LinearizationError:\n        pass\n```\n\n### Arrays\n\nWhen the first row is processed, the library guesses the columns layout. In case of arrays,\na column (or more columns if the array contains json objects) is created for each\nof the items in the array, as shown in the example above.\n\nOn subsequent rows the array might contain more items. The library reacts by adjusting \nthe number of columns in the layout and raising ``LinearizationError`` as previous rows might\nbe already output.\n\n``Converter.convert_streaming`` just raises this exception - it is the responsibility of caller\nto take the right action.\n\n``Converter.convert`` captures this error and restarts the processing. In case of CSV\nthis means truncating the output file to 0 bytes and processing the data again. XLSX writer\ncaches all the data before writing them to excel so the restart just means discarding the cache.\n\nIf you know the size of the array in advance, you should pass it in options. Then no\nprocessing restarts are required and ``LinearizationError`` is not raised.\n\n ```python\n\nfrom json_excel_converter import Converter, Options\nfrom json_excel_converter.xlsx import Writer\n\ndata = [\n    {\'a\': [1]},\n    {\'a\': [1, 2, 3]}\n]\noptions = Options()\noptions[\'a\'].cardinality = 3\n\nconv = Converter(options=options)\nwriter = Writer(file=\'/tmp/test.xlsx\')\nconv.convert(data, writer)\n# or\nconv.convert_streaming(data, writer)    # no exception occurs here\n```\n\n### XLSX Formatting\n\n#### Cell format\n\nXLSX writer enables you to format the header and data by passing an array of header_formatters or\ndata_formatters. Take these from ``json_excel_converter.xlsx.formats`` package or create your own.\n\n```python\nfrom json_excel_converter import Converter\n\nfrom json_excel_converter.xlsx import Writer\nfrom json_excel_converter.xlsx.formats import LastUnderlined, Bold, \\\n    Centered, Format\n\ndata = [\n    {\'a\': \'Hello\'},\n    {\'a\': \'World\'}\n]\n\nw = Writer(\'/tmp/test3.xlsx\',\n           header_formats=(\n               Centered, Bold, LastUnderlined,\n               Format({\n                   \'font_color\': \'red\'\n               })),\n           data_formats=(\n               Format({\n                   \'font_color\': \'green\'\n               }),)\n           )\n\nconv = Converter()\nconv.convert(data, w)\n```\n\nSee https://xlsxwriter.readthedocs.io/format.html for details on formats in xlsxwriter\n\n#### Column widths\n\nPass the required column widths to writer:\n\n```python\nw = Writer(\'/tmp/test3.xlsx\', column_widths={\n    \'a\': 20\n})\n```\n\nWidth of nested data can be specified as well: \n\n```python\ndata = [\n    {\'a\': {\'b\': 1, \'c\': 2}}\n]\n\nw = Writer(\'/tmp/test3.xlsx\', column_widths={\n    \'a.b\': 20,\n    \'a.c\': 30,\n})\n```\n\nTo set the default column width, pass it as ``DEFAULT_COLUMN_WIDTH`` property:\n\n```python\nw = Writer(\'/tmp/test3.xlsx\', column_widths={\n    DEFAULT_COLUMN_WIDTH: 20\n})\n```\n',
    'author': 'Mirek Simek',
    'author_email': 'miroslav.simek@vscht.cz',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
