import { IAspect, IConstruct, CfnResource } from 'monocdk';
/**
 * Interface for creating a Nag rule set.
 *
 * @stability stable
 */
export interface NagPackProps {
    /**
     * Whether or not to enable extended explanatory descriptions on warning, error, and logged ignore messages (default: false).
     *
     * @stability stable
     */
    readonly verbose?: boolean;
    /**
     * Whether or not to log triggered rules that have been suppressed as informational messages (default: false).
     *
     * @stability stable
     */
    readonly logIgnores?: boolean;
}
/**
 * Interface for JSII interoperability for passing parameters and the Rule Callback to @applyRule method.
 *
 * @stability stable
 */
export interface IApplyRule {
    /**
     * The id of the rule to ignore.
     *
     * @stability stable
     */
    ruleId: string;
    /**
     * Why the rule was triggered.
     *
     * @stability stable
     */
    info: string;
    /**
     * Why the rule exists.
     *
     * @stability stable
     */
    explanation: string;
    /**
     * The annotations message level to apply to the rule if triggered.
     *
     * @stability stable
     */
    level: NagMessageLevel;
    /**
     * Ignores listed in cdk-nag metadata.
     *
     * @stability stable
     */
    node: CfnResource;
    /**
     * The callback to the rule.
     *
     * @param node the CfnResource to check.
     * @stability stable
     */
    rule(node: CfnResource): boolean;
}
/**
 * The level of the message that the rule applies.
 *
 * @stability stable
 */
export declare enum NagMessageLevel {
    /**
     * @stability stable
     */
    WARN = 0,
    /**
     * @stability stable
     */
    ERROR = 1
}
/**
 * Base class for all rule sets.
 *
 * @stability stable
 */
export declare abstract class NagPack implements IAspect {
    /**
     * @stability stable
     */
    protected verbose: boolean;
    /**
     * @stability stable
     */
    protected logIgnores: boolean;
    /**
     * @stability stable
     */
    constructor(props?: NagPackProps);
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    abstract visit(node: IConstruct): void;
    /**
     * Create a rule to be used in the NagPack.
     *
     * @param params The.
     * @stability stable
     * @IApplyRule interface with rule details
     */
    applyRule(params: IApplyRule): void;
    /**
     * Check whether a specific rule should be ignored
     * @param ignores ignores listed in cdkNag metadata
     * @param ruleId the id of the rule to ignore
     * @returns the reason the rule was ignored, or an empty string
     */
    private ignoreRule;
    /**
     * The message to output to the console when a rule is triggered
     * @param ruleId the id of the rule
     * @param info why the rule was triggered
     * @param explanation why the rule exists
     * @returns string
     */
    private createMessage;
}
/**
 * Use in cases where a primitive value must be known to pass a rule
 * https://developer.mozilla.org/en-US/docs/Glossary/Primitive
 * @param node the CfnResource to check
 * @param parameter the value to attempt to resolve
 * @returns Return a value if resolves to a primitive data type, otherwise throw an error.
 */
export declare function resolveIfPrimitive(node: CfnResource, parameter: any): any;
