"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const monocdk_1 = require("monocdk");
const aws_ec2_1 = require("monocdk/aws-ec2");
const nag_pack_1 = require("../../../nag-pack");
const BLOCKED_PORTS = [20, 21, 3389, 3309, 3306, 4333];
/**
 * EC2 instances have all common TCP ports restricted for ingress IPv4 traffic - (Control IDs: AC-4, CM-2, SC-7, SC-7(3)).
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/default-custom-security-groups.html
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnSecurityGroup) {
        const ingressRules = monocdk_1.Stack.of(node).resolve(node.securityGroupIngress);
        if (ingressRules != undefined) {
            //For each ingress rule, ensure that it does not allow unrestricted SSH traffic.
            for (const rule of ingressRules) {
                const resolvedRule = monocdk_1.Stack.of(node).resolve(rule);
                for (const portNum of BLOCKED_PORTS) {
                    if (!testPort(node, resolvedRule, portNum)) {
                        return false;
                    }
                }
            }
        }
    }
    else if (node instanceof aws_ec2_1.CfnSecurityGroupIngress) {
        for (const portNum of BLOCKED_PORTS) {
            if (!testPort(node, node, portNum)) {
                return false;
            }
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to identify if the given port number is unrestricted
 * @param node the CfnResource to check
 * @param rule the CfnSecurityGroupIngress rule to check
 * @param portNum the number of the port to check
 */
function testPort(node, rule, portNum) {
    //Does this rule apply to TCP traffic?
    const ipProtocol = nag_pack_1.resolveIfPrimitive(node, rule.ipProtocol);
    const cidrIp = nag_pack_1.resolveIfPrimitive(node, rule.cidrIp);
    const fromPort = nag_pack_1.resolveIfPrimitive(node, rule.fromPort);
    const toPort = nag_pack_1.resolveIfPrimitive(node, rule.toPort);
    if (ipProtocol === 'tcp') {
        //Does this rule allow all IPv4 addresses (unrestricted access)?
        if (cidrIp != undefined && cidrIp.includes('/0')) {
            //Is a port range specified?
            if (fromPort != undefined && toPort != undefined) {
                if ((fromPort <= portNum && toPort >= portNum) ||
                    fromPort == -1 ||
                    toPort == -1) {
                    return false;
                }
            }
            else {
                if (fromPort == portNum) {
                    return false;
                }
            }
        }
    }
    //Are all ports allowed?
    if (ipProtocol === '-1') {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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