"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NIST80053Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const apigw_1 = require("./rules/apigw");
const autoscaling_1 = require("./rules/autoscaling");
const cloudtrail_1 = require("./rules/cloudtrail");
const cloudwatch_1 = require("./rules/cloudwatch");
const codebuild_1 = require("./rules/codebuild");
const dms_1 = require("./rules/dms");
const dynamodb_1 = require("./rules/dynamodb");
const ec2_1 = require("./rules/ec2");
const efs_1 = require("./rules/efs");
const elasticache_1 = require("./rules/elasticache");
const elb_1 = require("./rules/elb");
const emr_1 = require("./rules/emr");
const iam_1 = require("./rules/iam");
const lambda_1 = require("./rules/lambda");
const opensearch_1 = require("./rules/opensearch");
const rds_1 = require("./rules/rds");
const redshift_1 = require("./rules/redshift");
const s3_1 = require("./rules/s3");
const sagemaker_1 = require("./rules/sagemaker");
const sns_1 = require("./rules/sns");
/**
 * Check for NIST 800-53 compliance.
 *
 * Based on the NIST 800-53 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_4.html
 *
 * @stability stable
 */
class NIST80053Checks extends nag_pack_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkAPIGW(node);
            this.checkAutoScaling(node);
            this.checkCloudTrail(node);
            this.checkCloudWatch(node);
            this.checkCodeBuild(node);
            this.checkDMS(node);
            this.checkDynamoDB(node);
            this.checkEC2(node);
            this.checkEFS(node);
            this.checkElastiCache(node);
            this.checkELB(node);
            this.checkEMR(node);
            this.checkIAM(node);
            this.checkLambda(node);
            this.checkOpenSearch(node);
            this.checkRDS(node);
            this.checkRedshift(node);
            this.checkS3(node);
            this.checkSageMaker(node);
            this.checkSNS(node);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-APIGWCacheEnabledAndEncrypted',
            info: 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: SC-13, SC-28).',
            explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.nist80053APIGWCacheEnabledAndEncrypted,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-APIGWExecutionLoggingEnabled',
            info: 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.nist80053APIGWExecutionLoggingEnabled,
            node: node,
        });
    }
    /**
     * Check Auto Scaling Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-AutoScalingHealthChecks',
            info: 'The Auto Scaling group utilizes a load balancer and does not have an ELB health check configured - (Control IDs: SC-5).',
            explanation: 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.nist80053AutoScalingHealthChecks,
            node: node,
        });
    }
    /**
     * Check CloudTrail Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-CloudTrailCloudWatchLogsEnabled',
            info: 'The trail does not have CloudWatch logs enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-6(1)(3), AU-7(1), AU-12(a)(c), CA-7(a)(b), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).',
            explanation: 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.nist80053CloudTrailCloudWatchLogsEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-CloudTrailEncryptionEnabled',
            info: 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: AU-9).',
            explanation: 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.nist80053CloudTrailEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-CloudTrailLogFileValidationEnabled',
            info: 'The trail does not have log file validation enabled - (Control ID: AC-6).',
            explanation: 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.nist80053CloudTrailLogFileValidationEnabled,
            node: node,
        });
    }
    /**
     * Check CloudWatch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-CloudWatchAlarmAction',
            info: 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control IDs: AC-2(4), AU-6(1)(3), AU-7(1), CA-7(a)(b), IR-4(1), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).',
            explanation: 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.nist80053CloudWatchAlarmAction,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-CloudWatchLogGroupEncrypted',
            info: 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control IDs: AU-9, SC-13, SC-28).',
            explanation: 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.nist80053CloudWatchLogGroupEncrypted,
            node: node,
        });
    }
    /**
     * Check CodeBuild Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-CodeBuildCheckEnvVars',
            info: 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control IDs: AC-6, IA-5(7), SA-3(a)).',
            explanation: 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.nist80053CodeBuildCheckEnvVars,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-CodeBuildURLCheck',
            info: 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAUTH - (Control ID: SA-3(a)).',
            explanation: 'OAUTH is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.nist80053CodeBuildURLCheck,
            node: node,
        });
    }
    /**
     * Check DMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDMS(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-DMSReplicationNotPublic',
            info: 'The DMS replication instance is public - (Control IDs: AC-3).',
            explanation: 'DMS replication instances can contain sensitive information and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dms_1.nist80053DMSReplicationNotPublic,
            node: node,
        });
    }
    /**
     * Check DynamoDB Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-DynamoDBPITREnabled',
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'The recovery maintains continuous backups of your table for the last 35 days.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.nist80053DynamoDBPITREnabled,
            node: node,
        });
    }
    /**
     * Check EC2 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEC2(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-EC2CheckDefaultSecurityGroupClosed',
            info: "The VPC's default security group allows inbound or outbound traffic - (Control IDs: AC-4, SC-7, SC-7(3)).",
            explanation: 'When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: ec2_1.nist80053EC2CheckDefaultSecurityGroupClosed,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-EC2CheckDetailedMonitoring',
            info: 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: CA-7(a)(b), SI-4(2), SI-4(a)(b)(c)).',
            explanation: 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.nist80053EC2CheckDetailedMonitoring,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-EC2CheckInsideVPC',
            info: 'The EC2 instance is not within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.nist80053EC2CheckInsideVPC,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-EC2CheckNoPublicIPs',
            info: 'The EC2 instance is associated with a public IP address - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)). ',
            explanation: 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.nist80053EC2CheckNoPublicIPs,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-EC2CheckSSHRestricted',
            info: 'The Security Group allows unrestricted SSH access - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.nist80053EC2CheckSSHRestricted,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-EC2CheckCommonPortsRestricted',
            info: 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on common ports (20, 21, 3389, 3306, 4333) - (Control IDs: AC-4, CM-2, SC-7, SC-7(3)).',
            explanation: 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.nist80053EC2CheckCommonPortsRestricted,
            node: node,
        });
    }
    /**
     * Check EFS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEFS(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-EFSEncrypted',
            info: 'The EFS does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.nist80053EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check ElastiCache Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-ElastiCacheRedisClusterAutomaticBackup',
            info: 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.nist80053ElastiCacheRedisClusterAutomaticBackup,
            node: node,
        });
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkELB(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-ALBHttpDropInvalidHeaderEnabled',
            info: 'The ALB does not have invalid HTTP header dropping enabled - (Control ID: AC-17(2)).',
            explanation: 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ALBHttpDropInvalidHeaderEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-ALBHttpToHttpsRedirection',
            info: "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13, SC-23).",
            explanation: 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ALBHttpToHttpsRedirection,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-ELBCrossZoneBalancing',
            info: 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: SC-5, CP-10).',
            explanation: 'The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ELBCrossZoneBalancing,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-ELBDeletionProtectionEnabled',
            info: 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: CM-2, CP-10).',
            explanation: 'Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ELBDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-ELBListenersUseSSLOrHTTPS',
            info: 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-23).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ELBListenersUseSSLOrHTTPS,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-ELBLoggingEnabled',
            info: 'The ELB does not have logging enabled - (Control ID: AU-2(a)(d)).',
            explanation: "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-ELBUseACMCerts',
            info: 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.nist80053ELBUseACMCerts,
            node: node,
        });
    }
    /**
     * Check EMR Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEMR(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-EMRKerberosEnabled',
            info: 'The EMR cluster does not have Kerberos enabled - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).',
            explanation: 'The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.nist80053EMRKerberosEnabled,
            node: node,
        });
    }
    /**
     * Check IAM Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkIAM(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-IAMGroupMembership',
            info: 'The IAM user does not belong to any group(s) - (Control IDs: AC-2(1), AC-2(j), AC-3, and AC-6).',
            explanation: 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations, by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.nist80053IAMGroupMembership,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-IAMNoInlinePolicy',
            info: 'The IAM Group, User, or Role contains an inline policy - (Control ID: AC-6).',
            explanation: 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.nist80053IAMNoInlinePolicy,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-IAMPolicyNoStatementsWithAdminAccess',
            info: 'The IAM policy grants admin access - (Control IDs: AC-2(1), AC-2(j), AC-3, AC-6).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, restricting policies from containing "Effect": "Allow" with "Action": "*" over "Resource": "*". Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.nist80053IAMPolicyNoStatementsWithAdminAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-IAMUserNoPolicies',
            info: 'The IAM policy is attached at the user level - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).',
            explanation: 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.nist80053IAMUserNoPolicies,
            node: node,
        });
    }
    /**
     * Check Lambda Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-LambdaFunctionsInsideVPC',
            info: 'The Lambda function is not VPC enabled - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: lambda_1.nist80053LambdaFunctionsInsideVPC,
            node: node,
        });
    }
    /**
     * Check OpenSearch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-OpenSearchEncryptedAtRest',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.nist80053OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-OpenSearchNodeToNodeEncrypted',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control IDs: SC-7, SC-8, SC-8(1)).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.nist80053OpenSearchNodeToNodeEncrypted,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-OpenSearchRunningWithinVPC',
            info: 'The OpenSearch Service domain is not running within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'VPCs help secure your AWS resources and provide an extra layer of protection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.nist80053OpenSearchRunningWithinVPC,
            node: node,
        });
    }
    /**
     * Check Amazon RDS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRDS(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-RDSEnhancedMonitoringEnabled',
            info: 'The RDS DB instance does not enhanced monitoring enabled - (Control ID: CA-7(a)(b)).',
            explanation: 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSEnhancedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RDSInstanceBackupEnabled',
            info: 'The RDS DB instance does not have backups enabled - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSInstanceBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RDSInstanceDeletionProtectionEnabled',
            info: 'The RDS DB instance or Aurora DB cluster does not have deletion protection enabled - (Control ID: SC-5).',
            explanation: 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RDSInstanceMultiAZSupport',
            info: 'The non-Aurora RDS DB instance does not have multi-AZ support enabled - (Control IDs: CP-10, SC-5, SC-36).',
            explanation: 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSInstanceMultiAZSupport,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RDSInstancePublicAccess',
            info: 'The RDS DB instance allows public access - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSInstancePublicAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RDSLoggingEnabled',
            info: 'The RDS DB instance does not have all CloudWatch log types exported - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RDSStorageEncrypted',
            info: 'The RDS DB instance or Aurora DB cluster does not have storage encrypted - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in Amazon RDS DB instances and clusters, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.nist80053RDSStorageEncrypted,
            node: node,
        });
    }
    /**
     * Check Redshift Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-RedshiftClusterConfiguration',
            info: 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c), SC-13).',
            explanation: 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.nist80053RedshiftClusterConfiguration,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-RedshiftClusterPublicAccess',
            info: 'The Redshift cluster allows public access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.nist80053RedshiftClusterPublicAccess,
            node: node,
        });
    }
    /**
     * Check Amazon S3 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkS3(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketDefaultLockEnabled',
            info: 'The S3 Bucket does not have object lock enabled - (Control ID: SC-28).',
            explanation: 'Because sensitive data can exist at rest in S3 buckets, enforce object locks at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketDefaultLockEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketLoggingEnabled',
            info: 'The S3 Bucket does not have server access logs enabled - (Control IDs: AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketPublicReadProhibited',
            info: 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketPublicReadProhibited,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketPublicWriteProhibited',
            info: 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketPublicWriteProhibited,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketReplicationEnabled',
            info: 'The S3 Bucket does not have replication enabled - (Control IDs: AU-9(2), CP-9(b), CP-10, SC-5, SC-36).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketReplicationEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketServerSideEncryptionEnabled',
            info: 'The S3 Bucket does not have default server-side encryption enabled - (Control IDs: AU-9(2), CP-9(b), CP-10, SC-5, SC-36).',
            explanation: 'Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketServerSideEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-S3BucketVersioningEnabled',
            info: 'The S3 Bucket does not have versioning enabled - (Control IDs: CP-10, SI-12).',
            explanation: 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.nist80053S3BucketVersioningEnabled,
            node: node,
        });
    }
    /**
     * Check SageMaker Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-SageMakerEndpointKMS',
            info: 'The SageMaker endpoint is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.nist80053SageMakerEndpointKMS,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-SageMakerNotebookDirectInternetAccessDisabled',
            info: 'The SageMaker notebook does not disable direct internet access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.nist80053SageMakerNotebookDirectInternetAccessDisabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'NIST.800.53-SageMakerNotebookKMS',
            info: 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.nist80053SageMakerNotebookKMS,
            node: node,
        });
    }
    /**
     * Check Amazon SNS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSNS(node) {
        this.applyRule({
            ruleId: 'NIST.800.53-SNSEncryptedKMS',
            info: 'The SNS topic does not have KMS encryption enabled - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.nist80053SNSEncryptedKMS,
            node: node,
        });
    }
}
exports.NIST80053Checks = NIST80053Checks;
_a = JSII_RTTI_SYMBOL_1;
NIST80053Checks[_a] = { fqn: "monocdk-nag.NIST80053Checks", version: "0.0.131" };
//# sourceMappingURL=data:application/json;base64,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