"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const monocdk_1 = require("monocdk");
const aws_ec2_1 = require("monocdk/aws-ec2");
const nag_pack_1 = require("../../../nag-pack");
const BLOCKED_PORTS = [20, 21, 3389, 3309, 3306, 4333];
/**
 * EC2 instances have all common TCP ports restricted for ingress IPv4 traffic - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1))).
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/default-custom-security-groups.html
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnSecurityGroup) {
        const ingressRules = monocdk_1.Stack.of(node).resolve(node.securityGroupIngress);
        if (ingressRules != undefined) {
            //For each ingress rule, ensure that it does not allow unrestricted SSH traffic.
            for (const rule of ingressRules) {
                const resolvedRule = monocdk_1.Stack.of(node).resolve(rule);
                for (const portNum of BLOCKED_PORTS) {
                    if (!testPort(node, resolvedRule, portNum)) {
                        return false;
                    }
                }
            }
        }
    }
    else if (node instanceof aws_ec2_1.CfnSecurityGroupIngress) {
        for (const portNum of BLOCKED_PORTS) {
            if (!testPort(node, node, portNum)) {
                return false;
            }
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to identify if the given port number is unrestricted
 * @param node the CfnResource to check
 * @param rule the CfnSecurityGroupIngress rule to check
 * @param portNum the number of the port to check
 */
function testPort(node, rule, portNum) {
    //Does this rule apply to TCP traffic?
    const ipProtocol = nag_pack_1.resolveIfPrimitive(node, rule.ipProtocol);
    const cidrIp = nag_pack_1.resolveIfPrimitive(node, rule.cidrIp);
    const fromPort = nag_pack_1.resolveIfPrimitive(node, rule.fromPort);
    const toPort = nag_pack_1.resolveIfPrimitive(node, rule.toPort);
    if (ipProtocol === 'tcp') {
        //Does this rule allow all IPv4 addresses (unrestricted access)?
        if (cidrIp != undefined && cidrIp.includes('/0')) {
            //Is a port range specified?
            if (fromPort != undefined && toPort != undefined) {
                if ((fromPort <= portNum && toPort >= portNum) ||
                    fromPort == -1 ||
                    toPort == -1) {
                    return false;
                }
            }
            else {
                if (fromPort == portNum) {
                    return false;
                }
            }
        }
    }
    //Are all ports allowed?
    if (ipProtocol === '-1') {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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