"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HIPAASecurityChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const apigw_1 = require("./rules/apigw");
const autoscaling_1 = require("./rules/autoscaling");
const cloudtrail_1 = require("./rules/cloudtrail");
const cloudwatch_1 = require("./rules/cloudwatch");
const codebuild_1 = require("./rules/codebuild");
const dms_1 = require("./rules/dms");
const dynamodb_1 = require("./rules/dynamodb");
const ec2_1 = require("./rules/ec2");
const ecs_1 = require("./rules/ecs");
const efs_1 = require("./rules/efs");
const elasticache_1 = require("./rules/elasticache");
const elasticbeanstalk_1 = require("./rules/elasticbeanstalk");
const elb_1 = require("./rules/elb");
const emr_1 = require("./rules/emr");
const iam_1 = require("./rules/iam");
const lambda_1 = require("./rules/lambda");
const opensearch_1 = require("./rules/opensearch");
const rds_1 = require("./rules/rds");
const redshift_1 = require("./rules/redshift");
const s3_1 = require("./rules/s3");
const sagemaker_1 = require("./rules/sagemaker");
const secretsmanager_1 = require("./rules/secretsmanager");
const sns_1 = require("./rules/sns");
const vpc_1 = require("./rules/vpc");
/**
 * Check for HIPAA Security compliance.
 *
 * Based on the HIPAA Security AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-hipaa_security.html
 *
 * @stability stable
 */
class HIPAASecurityChecks extends nag_pack_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkAPIGW(node);
            this.checkAutoScaling(node);
            this.checkCloudTrail(node);
            this.checkCloudWatch(node);
            this.checkCodeBuild(node);
            this.checkDMS(node);
            this.checkDynamoDB(node);
            this.checkEC2(node);
            this.checkECS(node);
            this.checkEFS(node);
            this.checkElastiCache(node);
            this.checkElasticBeanstalk(node);
            this.checkELB(node);
            this.checkEMR(node);
            this.checkIAM(node);
            this.checkLambda(node);
            this.checkOpenSearch(node);
            this.checkRDS(node);
            this.checkRedshift(node);
            this.checkS3(node);
            this.checkSageMaker(node);
            this.checkSecretsManager(node);
            this.checkSNS(node);
            this.checkVPC(node);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-APIGWCacheEnabledAndEncrypted',
            info: 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.hipaaSecurityAPIGWCacheEnabledAndEncrypted,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-APIGWExecutionLoggingEnabled',
            info: 'The API Gateway stage does not have execution logging enabled for all methods - (Control ID: 164.312(b)).',
            explanation: 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.hipaaSecurityAPIGWExecutionLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-APIGWSSLEnabled',
            info: 'The API Gateway REST API stage is not configured with SSL certificates - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).',
            explanation: 'Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.hipaaSecurityAPIGWSSLEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-APIGWXrayEnabled',
            info: 'The API Gateway REST API stage does not have X-Ray enabled - (Control ID: 164.312(b)).',
            explanation: 'AWS X-Ray collects data about requests that your application serves, and provides tools you can use to view, filter, and gain insights into that data to identify issues and opportunities for optimization. Ensure X-Ray is enables so you can see detailed information not only about the request and response, but also about calls that your application makes to downstream AWS resources, microservices, databases and HTTP web APIs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.hipaaSecurityAPIGWXrayEnabled,
            node: node,
        });
    }
    /**
     * Check Auto Scaling Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-AutoscalingGroupELBHealthCheckRequired',
            info: 'The Auto Scaling group utilizes a load balancer and does not have an ELB health check configured - (Control ID: 164.312(b)).',
            explanation: 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.hipaaSecurityAutoscalingGroupELBHealthCheckRequired,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-AutoscalingLaunchConfigPublicIpDisabled',
            info: 'The Auto Scaling launch configuration does not have public IP addresses disabled - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.hipaaSecurityAutoscalingLaunchConfigPublicIpDisabled,
            node: node,
        });
    }
    /**
     * Check CloudTrail Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-CloudTrailCloudWatchLogsEnabled',
            info: 'The trail does not have CloudWatch logs enabled - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).',
            explanation: 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.hipaaSecurityCloudTrailCloudWatchLogsEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-CloudTrailEncryptionEnabled',
            info: 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.hipaaSecurityCloudTrailEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-CloudTrailLogFileValidationEnabled',
            info: 'The trail does not have log file validation enabled - (Control ID: 164.312(c)(1), 164.312(c)(2)).',
            explanation: 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.hipaaSecurityCloudTrailLogFileValidationEnabled,
            node: node,
        });
    }
    /**
     * Check CloudWatch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-CloudWatchAlarmAction',
            info: 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control ID: 164.312(b)).',
            explanation: 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.hipaaSecurityCloudWatchAlarmAction,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-CloudWatchLogGroupEncrypted',
            info: 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.hipaaSecurityCloudWatchLogGroupEncrypted,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-CloudWatchLogGroupRetentionPeriod',
            info: 'The CloudWatch Log Group does not have an explicit retention period configured - (Control ID: 164.312(b)).',
            explanation: 'Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.hipaaSecurityCloudWatchLogGroupRetentionPeriod,
            node: node,
        });
    }
    /**
     * Check CodeBuild Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-CodeBuildProjectEnvVarAwsCred',
            info: 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.hipaaSecurityCodeBuildProjectEnvVarAwsCred,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-CodeBuildProjectSourceRepoUrl',
            info: 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAUTH - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'OAUTH is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.hipaaSecurityCodeBuildProjectSourceRepoUrl,
            node: node,
        });
    }
    /**
     * Check DMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDMS(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-DMSReplicationNotPublic',
            info: 'The DMS replication instance is public - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'DMS replication instances can contain sensitive information and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dms_1.hipaaSecurityDMSReplicationNotPublic,
            node: node,
        });
    }
    /**
     * Check DynamoDB Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-DynamoDBPITREnabled',
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).',
            explanation: 'The recovery maintains continuous backups of your table for the last 35 days.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.hipaaSecurityDynamoDBPITREnabled,
            node: node,
        });
    }
    /**
     * Check EC2 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEC2(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2EBSOptimizedInstance',
            info: "The EC2 instance type 'supports' EBS optimization and does not have EBS optimization enabled - (Control ID: 164.308(a)(7)(i)).",
            explanation: 'An optimized instance in Amazon Elastic Block Store (Amazon EBS) provides additional, dedicated capacity for Amazon EBS I/O operations. This optimization provides the most efficient performance for your EBS volumes by minimizing contention between Amazon EBS I/O operations and other traffic from your instance.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2EBSOptimizedInstance,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2InstanceDetailedMonitoringEnabled',
            info: 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: 164.312(b)).',
            explanation: 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2InstanceDetailedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2InstanceNoPublicIps',
            info: 'The EC2 instance is associated with a public IP address - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2InstanceNoPublicIps,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2InstanceProfileAttached',
            info: 'The EC2 instance does not have an instance profile attached - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2InstanceProfileAttached,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2InstancesInVPC',
            info: 'The EC2 instance is not within a VPC - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2InstancesInVPC,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2RestrictedCommonPorts',
            info: 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on common ports (20, 21, 3389, 3306, 4333) - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2RestrictedCommonPorts,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-EC2RestrictedSSH',
            info: 'The Security Group allows unrestricted SSH access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1).',
            explanation: 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.hipaaSecurityEC2RestrictedSSH,
            node: node,
        });
    }
    /**
     * Check ECS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkECS(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-ECSTaskDefinitionUserForHostMode',
            info: "The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)).",
            explanation: 'If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.hipaaSecurityECSTaskDefinitionUserForHostMode,
            node: node,
        });
    }
    /**
     * Check EFS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEFS(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-EFSEncrypted',
            info: 'The EFS does not have encryption at rest enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.hipaaSecurityEFSEncrypted,
            node: node,
        });
    }
    /**
     * Check ElastiCache Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-ElastiCacheRedisClusterAutomaticBackup',
            info: 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).',
            explanation: 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.hipaaSecurityElastiCacheRedisClusterAutomaticBackup,
            node: node,
        });
    }
    /**
     * Check Elastic Beanstalk Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElasticBeanstalk(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-ElasticBeanstalkEnhancedHealthReportingEnabled',
            info: 'The Elastic Beanstalk environment does not have enhanced health reporting enabled - (Control ID: 164.312(b)).',
            explanation: 'AWS Elastic Beanstalk enhanced health reporting enables a more rapid response to changes in the health of the underlying infrastructure. These changes could result in a lack of availability of the application. Elastic Beanstalk enhanced health reporting provides a status descriptor to gauge the severity of the identified issues and identify possible causes to investigate.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.hipaaSecurityElasticBeanstalkEnhancedHealthReportingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ElasticBeanstalkManagedUpdatesEnabled',
            info: 'The Elastic Beanstalk environment does not have managed updates enabled - (Control ID: 164.308(a)(5)(ii)(A)).',
            explanation: 'Enabling managed platform updates for an Amazon Elastic Beanstalk environment ensures that the latest available platform fixes, updates, and features for the environment are installed. Keeping up to date with patch installation is a best practice in securing systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.hipaaSecurityElasticBeanstalkManagedUpdatesEnabled,
            node: node,
        });
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkELB(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-ALBHttpDropInvalidHeaderEnabled',
            info: 'The ALB does not have invalid HTTP header dropping enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).',
            explanation: 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityALBHttpDropInvalidHeaderEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ALBHttpToHttpsRedirection',
            info: "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).",
            explanation: 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityALBHttpToHttpsRedirection,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ELBACMCertificateRequired',
            info: 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityELBACMCertificateRequired,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ELBCrossZoneBalancingEnabled',
            info: 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).',
            explanation: 'The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityELBCrossZoneBalancingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ELBDeletionProtectionEnabled',
            info: 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).',
            explanation: 'Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityELBDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ELBLoggingEnabled',
            info: 'The ELB does not have logging enabled - (Control ID: 164.312(b)).',
            explanation: "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ELBTlsHttpsListenersOnly',
            info: 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityELBTlsHttpsListenersOnly,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-ELBv2ACMCertificateRequired',
            info: 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.hipaaSecurityELBv2ACMCertificateRequired,
            node: node,
        });
    }
    /**
     * Check EMR Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEMR(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-EMRKerberosEnabled',
            info: 'The EMR cluster does not have Kerberos enabled - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.hipaaSecurityEMRKerberosEnabled,
            node: node,
        });
    }
    /**
     * Check IAM Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkIAM(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-IAMNoInlinePolicy',
            info: 'The IAM Group, User, or Role contains an inline policy - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.hipaaSecurityIAMNoInlinePolicy,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-IAMPolicyNoStatementsWithAdminAccess',
            info: 'The IAM policy grants admin access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, restricting policies from containing "Effect": "Allow" with "Action": "*" over "Resource": "*". Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.hipaaSecurityIAMPolicyNoStatementsWithAdminAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-IAMPolicyNoStatementsWithFullAccess',
            info: 'The IAM policy grants full access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.hipaaSecurityIAMPolicyNoStatementsWithFullAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-IAMUserGroupMembership',
            info: 'The IAM user does not belong to any group(s) - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations, by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.hipaaSecurityIAMUserGroupMembership,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-IAMUserNoPolicies',
            info: 'The IAM policy is attached at the user level - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).',
            explanation: 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.hipaaSecurityIAMUserNoPolicies,
            node: node,
        });
    }
    /**
     * Check Lambda Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-LambdaConcurrency',
            info: 'The Lambda function is not configured with function-level concurrent execution limits - (Control ID: 164.312(b)).',
            explanation: "Ensure that a Lambda function's concurrency high and low limits are established. This can assist in baselining the number of requests that your function is serving at any given time.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: lambda_1.hipaaSecurityLambdaConcurrency,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-LambdaDlq',
            info: 'The Lambda function is not configured with a dead-letter configuration - (Control ID: 164.312(b)).',
            explanation: 'Notify the appropriate personnel through Amazon Simple Queue Service (Amazon SQS) or Amazon Simple Notification Service (Amazon SNS) when a function has failed.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: lambda_1.hipaaSecurityLambdaDlq,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-LambdaInsideVPC',
            info: 'The Lambda function is not VPC enabled - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: lambda_1.hipaaSecurityLambdaInsideVPC,
            node: node,
        });
    }
    /**
     * Check OpenSearch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-OpenSearchEncryptedAtRest',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.hipaaSecurityOpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-OpenSearchInVPCOnly',
            info: 'The OpenSearch Service domain is not running within a VPC - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'VPCs help secure your AWS resources and provide an extra layer of protection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.hipaaSecurityOpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-OpenSearchLogsToCloudWatch',
            info: 'The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).',
            explanation: 'Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.hipaaSecurityOpenSearchLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-OpenSearchNodeToNodeEncryption',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.hipaaSecurityOpenSearchNodeToNodeEncryption,
            node: node,
        });
    }
    /**
     * Check RDS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRDS(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSAutomaticMinorVersionUpgradeEnabled',
            info: 'The RDS DB instance does not have automatic minor version upgrades enabled - (Control ID: 164.308(a)(5)(ii)(A)).',
            explanation: 'Enable automatic minor version upgrades on your Amazon Relational Database Service (RDS) instances to ensure the latest minor version updates to the Relational Database Management System (RDBMS) are installed, which may include security patches and bug fixes.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSAutomaticMinorVersionUpgradeEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSEnhancedMonitoringEnabled',
            info: 'The RDS DB instance does not enhanced monitoring enabled - (Control ID: 164.312(b)).',
            explanation: 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSEnhancedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSInstanceBackupEnabled',
            info: 'The RDS DB instance does not have backups enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).',
            explanation: 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSInstanceBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSInstanceDeletionProtectionEnabled',
            info: 'The RDS DB instance or Aurora DB cluster does not have deletion protection enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).',
            explanation: 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSInstanceMultiAZSupport',
            info: 'The non-Aurora RDS DB instance does not have multi-AZ support enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).',
            explanation: 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSInstanceMultiAZSupport,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSInstancePublicAccess',
            info: 'The RDS DB instance allows public access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSInstancePublicAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSLoggingEnabled',
            info: 'The RDS DB instance does not have all CloudWatch log types exported - (Control IDs: 164.308(a)(3)(ii)(A), 164.308(a)(5)(ii)(C)).',
            explanation: 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RDSStorageEncrypted',
            info: 'The RDS DB instance or Aurora DB cluster does not have storage encrypted - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist at rest in Amazon RDS DB instances and clusters, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.hipaaSecurityRDSStorageEncrypted,
            node: node,
        });
    }
    /**
     * Check Redshift Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-RedshiftBackupEnabled',
            info: 'The Redshift cluster does not have automated snapshots enabled or the retention period is not between 1 and 35 days - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Redshift clusters have automated snapshots. When automated snapshots are enabled for a cluster, Redshift periodically takes snapshots of that cluster. By default, Redshift takes a snapshot every eight hours or every 5 GB per node of data changes, or whichever comes first.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.hipaaSecurityRedshiftBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RedshiftClusterConfiguration',
            info: 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(b), 164.312(e)(2)(ii)).',
            explanation: 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.hipaaSecurityRedshiftClusterConfiguration,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RedshiftClusterMaintenanceSettings',
            info: 'The Redshift cluster has version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured - (Control IDs: 164.308(a)(5)(ii)(A), 164.308(a)(7)(ii)(A)).',
            explanation: 'Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.hipaaSecurityRedshiftClusterMaintenanceSettings,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RedshiftClusterPublicAccess',
            info: 'The Redshift cluster allows public access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.hipaaSecurityRedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-RedshiftEnhancedVPCRoutingEnabled',
            info: 'The Redshift cluster does not have enhanced VPC routing enabled - (Control IDs: 164.312(e)(1)).',
            explanation: 'Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.hipaaSecurityRedshiftEnhancedVPCRoutingEnabled,
            node: node,
        });
    }
    /**
     * Check Amazon S3 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkS3(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketLevelPublicAccessProhibited',
            info: 'The S3 bucket does not prohibit public access through bucket level settings - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketLevelPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketLoggingEnabled',
            info: 'The S3 Bucket does not have server access logs enabled - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketPublicReadProhibited',
            info: 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketPublicReadProhibited,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketPublicWriteProhibited',
            info: 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketPublicWriteProhibited,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketReplicationEnabled',
            info: 'The S3 Bucket does not have replication enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketReplicationEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketServerSideEncryptionEnabled',
            info: 'The S3 Bucket does not have default server-side encryption enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(c)(2), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketServerSideEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3BucketVersioningEnabled',
            info: 'The S3 Bucket does not have versioning enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B), 164.312(c)(1), 164.312(c)(2)).',
            explanation: 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3BucketVersioningEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-S3DefaultEncryptionKMS',
            info: 'The S3 Bucket is not encrypted with a KMS Key by default - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.hipaaSecurityS3DefaultEncryptionKMS,
            node: node,
        });
    }
    /**
     * Check SageMaker Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-SageMakerEndpointConfigurationKMSKeyConfigured',
            info: 'The SageMaker endpoint is not encrypted with a KMS key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.hipaaSecuritySageMakerEndpointConfigurationKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-SageMakerNotebookInstanceKMSKeyConfigured',
            info: 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.hipaaSecuritySageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-SageMakerNotebookNoDirectInternetAccess',
            info: 'The SageMaker notebook does not disable direct internet access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.hipaaSecuritySageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Secrets Manager Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecretsManager(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-SecretsManagerUsingKMSKey',
            info: 'The secret is not encrypted with a KMS Customer managed key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.hipaaSecuritySecretsManagerUsingKMSKey,
            node: node,
        });
    }
    /**
     * Check Amazon SNS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSNS(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-SNSEncryptedKMS',
            info: 'The SNS topic does not have KMS encryption enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).',
            explanation: 'Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.hipaaSecuritySNSEncryptedKMS,
            node: node,
        });
    }
    /**
     * Check VPC Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkVPC(node) {
        this.applyRule({
            ruleId: 'HIPAA.Security-VPCDefaultSecurityGroupClosed',
            info: "The VPC's default security group allows inbound or outbound traffic - (Control ID: 164.312(e)(1)).",
            explanation: 'When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: vpc_1.hipaaSecurityVPCDefaultSecurityGroupClosed,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-VPCNoUnrestrictedRouteToIGW',
            info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0') - (Control ID: 164.312(e)(1)).",
            explanation: 'Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.hipaaSecurityVPCNoUnrestrictedRouteToIGW,
            node: node,
        });
        this.applyRule({
            ruleId: 'HIPAA.Security-VPCSubnetAutoAssignPublicIpDisabled',
            info: 'The subnet auto-assigns public IP addresses - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.hipaaSecurityVPCSubnetAutoAssignPublicIpDisabled,
            node: node,
        });
    }
}
exports.HIPAASecurityChecks = HIPAASecurityChecks;
_a = JSII_RTTI_SYMBOL_1;
HIPAASecurityChecks[_a] = { fqn: "monocdk-nag.HIPAASecurityChecks", version: "0.0.131" };
//# sourceMappingURL=data:application/json;base64,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