__doc__ = """
This program requires python 3.6 or higher.

print(functionName.__doc__) to see the

documentation for the function or usage.

Send enquiries to osita@protein-science.com
"""
import os
import sys
import itertools
import numpy as np
from .MissingHeavyAtoms import *
from .ReadMaster import getChains
from .ForcefieldParam import param

def repairNotes(resNo,chain,missing):
    """
    PRAS reminds users of residues whose atoms are fixed.
    This will help user check the quality of the fix.

    Arguments
    ----------
    resNo:   the amino acid residue number
    chain:   the chain or chain number
    missing: a list of the missing atoms

    Returns
    -------
    None:    it writes to a text file
    """
    with open('log.txt', 'a') as log:
        log.write('The residue No {} in chain {} have missing atom(s) {}.'
        ' We have fixed it'.format(resNo, chain,missing)+'\n\n')

def exitNotes(resNo,chain):
    """
    PRAS can fix missing backbone O but demands that N, CA, & C be present.
    This function will terminate the program if a main-chain atom is missing.

    Arguments
    ----------
    resNo:   the amino acid residue number
    chain:   the chain or chain number

    Returns
    -------
    None:    a premature termination of the program
    """
    with open('missing_backbone_atom.txt', 'w') as f:
        f.write('The residue No {} in chain {} is missing'
            ' backbone heavy atom. Program has terminated'
            ' abnormally'.format(resNo,chain)+'\n\n')
        sys.exit()

def exitError():
    """
    PRAS will read PDBs with as much flaws as covered in the program.
    Unfortunately, if the file is severely damaged or illegal, PRAS will fail.

    Returns
    -------
    None:   a premature termination of the program
    """
    with open('invalid_pdb_file.txt', 'w') as f:
        f.write('Error occured while reading the PDB file'+'\n')
        f.write('Visit www.protein-science.com and click How-tos to see what PRAS expects'+'\n')
        f.write('Each chain must end with TER record and last chain TER and END'+'\n')
        sys.exit()

def checkpdbAtoms (pdb_pras,rotamer,mutation,pdb_faspr):
    """
    This function obtains all chains in the PDB file as read by ReadMAster.py
    It then loops through each chain and checks if a residue has missing heavy atom(s)
    If it has, this function will replace it by calling MissingHeavyAtoms.py
    See PRAS.py for the detailed explanation of the arguments passed.

    Arguments
    ----------
    pdb_pras:  the PDB file you intend to repair/analyze

    rotamer:   supply "no" if you need to generate lower occupancy conformers,
               if not supply "". PRAS uses atoms with highest occupancy
               by default.

    mutation:  supply "no" if you need to generate lower occupancy conformers,
               if not supply "". PRAS uses the residue with highest
               occupancy by default if two residues are given same residue number

    pdb_faspr: the output PDB obtained by running your PDB with FASPR

    Returns
    -------
    A list:    a list that contains all chains with complete heavy atom of each residue
    """

    for i in os.listdir(os.getcwd()):
        if i == 'log.txt':
            os.remove(i)

    ter_all = []

    with open('log.txt', 'a') as log:

        log.write('When using files generated by this program in a publication'
        ' please cite this program as "O.S. Nnyigide, T.O. Nnyigide,'+'\n'
        'S.G. Lee, K. Hyun. PRAS: A Web Server to Repair PDB Files,'
        ' Add Missing Heavy Atoms and Hydrogen Atoms and Assign'+'\n'
        'Secondary Structure by Amide Interactions. Submitted'+'\n')
        log.write('#'*118+'\n')
        log.write('PRAS 1.0.5 This is a PRAS-generated log file.'
                ' For your information, fixed atoms (if any) are appended below '+'\n')
        log.write('#'*118+'\n\n')

    """read user's input file obtained from faspr
        but if it is not readable inform the user and
        use default chi from Dunbrack 2011 rotamer library"""
    try:
        faspr = getChains("","","",pdb_faspr)
    except:
        faspr = []

    if not faspr:
        with open('log.txt', 'a') as log:
            log.write('No FASPR PDB output supplied. Default chi will be used'+'\n\n')

    """read user's input file to be repaired or analyzed
        but if it is not readable inform the user and
        terminate the program"""
    #try:
    chains = getChains(pdb_pras,rotamer,mutation,"")
    #except:
        #chains = []

    #if not chains:
    #    exitError()

    for n in range(len(chains)):
        resn,atoms,atmpos,resNo,chain_id = chains[n][0],\
        chains[n][1],chains[n][2],chains[n][3],chains[n][4]
        resseq = [i[:3] for i in resn]

        # first, check if cterminal is missing a backbone atom and terminate program
        if not  ('N' in atoms[-1] or 'CA' in atoms[-1] or 'C' in atoms[-1]):
            exitNotes(resNo[-1],n+1)
        # now loop through all the chains, checking and fixing missing heavy  atoms
        for k in range(len(atoms)):
            # the expected complete atoms of any residue
            x = list(param[resseq[k]].keys())[:-1]

            if resseq[k] == 'ASP':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0], atmpos[k-1][1],atmpos[k-1][2], atmpos[k][0]]
                    atoms[k],atmpos[k]=repairAsp(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'ASN':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0], atmpos[k-1][1],atmpos[k-1][2], atmpos[k][0]]
                    atoms[k],atmpos[k]=repairAsn(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'ALA':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairAla(atoms[k],atmpos[k],atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'ARG':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairArg(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'LEU':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairLeu(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'ILE':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairIle(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'PRO':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairPro(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'SER':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairSer(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'VAL':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairVal(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'LYS':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairLys(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'TYR':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairTyr(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'TRP':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairTrp(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'MET':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0], atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairMet(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'HIS':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairHis(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'GLY':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0], atmpos[k-1][1],atmpos[k-1][2], atmpos[k][0]]
                    atoms[k],atmpos[k]=repairGly(atoms[k],atmpos[k],atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'CYS':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairCys(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'GLU':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairGlu(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'PHE':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairPhe(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'GLN':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairGln(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

            elif resseq[k] == 'THR':
                missing=[i for i in x if not i in atoms[k]]
                if not missing:
                    pass
                elif missing and not ('N' in missing or 'CA' in missing or 'C' in missing):
                    psi = [atmpos[k-1][0],atmpos[k-1][1],atmpos[k-1][2],atmpos[k][0]]
                    atoms[k],atmpos[k]=repairThr(atoms[k],atmpos[k],resNo[k],n,missing,faspr,atmpos[(k+1)%len(atmpos)],psi)
                    repairNotes(resNo[k], n+1,missing)
                elif ('N' in missing or 'CA' in missing or 'C' in missing):
                    exitNotes(resNo[k],n+1)

        # add C-terminal O [OXT] if it is missing
        if not 'OXT' in atoms[-1] or 'OC2' in atoms[-1]:
            atmpos[-1].extend(c_termini(atoms[-1],atmpos[-1]))
            atoms[-1].extend(['OXT'])
            repairNotes(resNo[-1], n+1,['OXT'])

        # get all CYS SG coordinates for disulfide bond checks.
        # notice that this is done when all missing atoms are fixed.
        atom_pos =  list(zip(atoms,atmpos));sg_coord = []
        for i in range(len(atom_pos)):
            for j,k in enumerate(atom_pos[i][0]):
                if k == 'SG':
                    sg_coord.append(atom_pos[i][1][j])

        ter_all.append([resn,atoms,atmpos,resNo,resseq,sg_coord,chain_id])
    return  ter_all