"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk_security_group_1 = require("@pepperize/cdk-security-group");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const constructs_1 = require("constructs");
const cache_1 = require("./cache");
const job_runner_1 = require("./job-runner");
const manager_1 = require("./manager");
const network_1 = require("./network");
/**
 * The Gitlab Runner autoscaling on EC2 by Docker Machine.
 *
 * @example <caption>Provisioning a basic Runner</caption>
 * const app = new cdk.App();
 * const stack = new cdk.Stack(app, "RunnerStack", {
 *   env: {
 *     account: "000000000000",
 *     region: "us-east-1",
 *   }
 * });
 *
 * const token = new StringParameter(stack, "imported-token", {
 *   parameterName: "/gitlab-runner/token1",
 *   stringValue: gitlabToken,
 *   type: ParameterType.SECURE_STRING,
 *   tier: ParameterTier.STANDARD,
 * });
 *
 * new GitlabRunnerAutoscaling(stack, "GitlabRunner", {
 *   runners: [{
 *     token: "xxxxxxxxxxxxxxxxxxxx"
 *     }],
 * });
 */
class GitlabRunnerAutoscaling extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { concurrent, checkInterval, logFormat, logLevel, manager, cache, runners, network, } = props;
        /**
         * Global Configuration
         * @link GlobalConfiguration
         */
        this.concurrent = concurrent ?? 10;
        this.checkInterval = checkInterval ?? 0;
        this.logFormat = logFormat ?? "runner";
        this.logLevel = logLevel ?? "info";
        /**
         * S3 Bucket for Runners' cache
         */
        this.cacheBucket = cache?.bucket || new cache_1.Cache(scope, "Cache", cache?.options).bucket;
        /**
         * Network
         */
        this.network = new network_1.Network(scope, "Network", network);
        /**
         * Security groups
         */
        const runnersSecurityGroupName = `${scope.stackName}-RunnersSecurityGroup`;
        const runnersSecurityGroup = new cdk_security_group_1.SecurityGroup(scope, "RunnersSecurityGroup", {
            securityGroupName: runnersSecurityGroupName,
            description: "Security group for GitLab Runners.",
            vpc: this.network.vpc,
        });
        const managerSecurityGroup = new cdk_security_group_1.SecurityGroup(scope, "ManagerSecurityGroup", {
            vpc: this.network.vpc,
            description: "Security group for GitLab Runners Manager.",
        });
        managerSecurityGroup.connections.allowTo(runnersSecurityGroup, aws_ec2_1.Port.tcp(22), "SSH traffic from Manager");
        managerSecurityGroup.connections.allowTo(runnersSecurityGroup, aws_ec2_1.Port.tcp(2376), "SSH traffic from Docker");
        this.runners = runners.map((runnerProps, index) => {
            return new job_runner_1.GitlabRunnerAutoscalingJobRunner(scope, `GitlabRunnerAutoscalingJobRunner${index}`, runnerProps);
        });
        /**
         * GitLab Manager
         */
        this.manager = new manager_1.GitlabRunnerAutoscalingManager(scope, "Manager", {
            ...manager,
            globalConfiguration: {
                concurrent: this.concurrent,
                checkInterval: this.checkInterval,
                logFormat: this.logFormat,
                logLevel: this.logLevel,
            },
            runnersSecurityGroup: runnersSecurityGroup,
            network: this.network,
            cacheBucket: this.cacheBucket,
            runners: this.runners,
        });
        new aws_autoscaling_1.AutoScalingGroup(scope, "ManagerAutoscalingGroup", {
            vpc: this.network.vpc,
            vpcSubnets: {
                subnets: [this.network.subnet],
            },
            instanceType: this.manager.instanceType,
            machineImage: this.manager.machineImage,
            keyName: this.manager.keyPairName,
            securityGroup: managerSecurityGroup,
            role: this.manager.role,
            userData: this.manager.userData,
            init: this.manager.initConfig,
            initOptions: {
                ignoreFailures: false,
            },
            maxCapacity: 1,
            minCapacity: 1,
            desiredCapacity: 1,
            signals: aws_autoscaling_1.Signals.waitForCount(1, { timeout: aws_cdk_lib_1.Duration.minutes(15) }),
            requireImdsv2: true,
        });
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscaling[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.GitlabRunnerAutoscaling", version: "0.2.187" };
//# sourceMappingURL=data:application/json;base64,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