"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscalingManager = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const runner_configuration_1 = require("../runner-configuration");
const DEFAULT_SSH_KEY_PATH = "/etc/gitlab-runner/ssh";
/**
 * Settings for the manager (coordinator)
 *
 *  Manager coordinates the placement of runner (job executor) instances
 */
class GitlabRunnerAutoscalingManager extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.globalConfiguration =
            props.globalConfiguration ||
                {
                    concurrent: 10,
                    checkInterval: 0,
                    logFormat: "runner",
                    logLevel: "info",
                };
        this.machineImage =
            props.machineImage ??
                aws_ec2_1.MachineImage.latestAmazonLinux({
                    generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
                    edition: aws_ec2_1.AmazonLinuxEdition.STANDARD,
                    virtualization: aws_ec2_1.AmazonLinuxVirt.HVM,
                    storage: aws_ec2_1.AmazonLinuxStorage.EBS,
                    cpuType: aws_ec2_1.AmazonLinuxCpuType.X86_64,
                });
        this.instanceType = props.instanceType ?? aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.NANO);
        this.keyPairName = props.keyPairName;
        this.runners = props.runners;
        this.network = props.network;
        this.cacheBucket = props.cacheBucket;
        this.runnersSecurityGroupName = props.runnersSecurityGroup.securityGroupName;
        this.role =
            props.role ||
                new aws_iam_1.Role(scope, "ManagerRole", {
                    assumedBy: new aws_iam_1.ServicePrincipal("ec2.amazonaws.com", {}),
                    managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName("AmazonSSMManagedInstanceCore")],
                    inlinePolicies: {
                        Cache: aws_iam_1.PolicyDocument.fromJson({
                            Version: "2012-10-17",
                            Statement: [
                                {
                                    Effect: "Allow",
                                    Action: ["s3:ListObjects*", "s3:GetObject*", "s3:DeleteObject*", "s3:PutObject*"],
                                    Resource: [`${this.cacheBucket.bucketArn}/*`],
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["s3:ListBucket"],
                                    Resource: [`${this.cacheBucket.bucketArn}`],
                                },
                            ],
                        }),
                        Runners: aws_iam_1.PolicyDocument.fromJson({
                            Version: "2012-10-17",
                            Statement: [
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:CreateKeyPair", "ec2:DeleteKeyPair", "ec2:ImportKeyPair", "ec2:Describe*"],
                                    Resource: ["*"],
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:CreateTags", "ssm:UpdateInstanceInformation"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringLike: {
                                            "aws:RequestTag/Name": "*gitlab-runner-*",
                                        },
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["Name"],
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:RequestSpotInstances", "ec2:CancelSpotInstanceRequests"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringEqualsIfExists: {
                                            "ec2:Region": `${scope.region}`,
                                        },
                                        ArnEqualsIfExists: {
                                            "ec2:Vpc": `arn:aws:ec2:${scope.region}:${scope.account}:vpc/${this.network.vpc.vpcId}`,
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:RunInstances"],
                                    Resource: ["*"],
                                    Condition: {
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["InstanceProfile"],
                                            "ec2:InstanceType": (this.runners || []).map((runner) => {
                                                const runnersInstanceType = (this.runners && runner.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
                                                return runnersInstanceType.toString();
                                            }),
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:TerminateInstances", "ec2:StopInstances", "ec2:StartInstances", "ec2:RebootInstances"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringLike: {
                                            "ec2:ResourceTag/Name": "*gitlab-runner-*",
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["iam:PassRole"],
                                    Resource: ["*"],
                                    Condition: {
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["RunnersRole"],
                                        },
                                    },
                                },
                            ],
                        }),
                    },
                });
        this.userData = aws_ec2_1.UserData.forLinux({});
        this.userData.addCommands(`yum update -y aws-cfn-bootstrap` // !/bin/bash -xe
        );
        // https://github.com/awslabs/amazon-ecr-credential-helper
        const userDataRunners = aws_ec2_1.UserData.forLinux({});
        userDataRunners.addCommands(`[ ! -z "$(which apt-get)" ] && apt-get install -y amazon-ecr-credential-helper`, `[ ! -z "$(which yum)" ] && yum install -y amazon-ecr-credential-helper`);
        const gitlabRunnerConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        gitlabRunnerConfigRestartHandle._addFile("/etc/gitlab-runner/config.toml");
        const rsyslogConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        rsyslogConfigRestartHandle._addFile("/etc/rsyslog.d/25-gitlab-runner.conf");
        /**
         * Config set keys
         * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html#aws-resource-init-configsets
         */
        const REPOSITORIES = "repositories";
        const PACKAGES = "packages";
        const CONFIG = "config";
        const RESTART = "restart";
        this.initConfig = aws_ec2_1.CloudFormationInit.fromConfigSets({
            configSets: {
                default: [REPOSITORIES, PACKAGES, CONFIG, RESTART],
            },
            configs: {
                [REPOSITORIES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://packages.gitlab.com/install/repositories/runner/gitlab-runner/script.rpm.sh | bash", { key: "10-gitlab-runner" }),
                ]),
                [PACKAGES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitPackage.yum("docker"),
                    aws_ec2_1.InitPackage.yum("gitlab-runner"),
                    aws_ec2_1.InitPackage.yum("tzdata"),
                    aws_ec2_1.InitPackage.yum("jq"),
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://gitlab-docker-machine-downloads.s3.amazonaws.com/v0.16.2-gitlab.12/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine", 
                    //"curl -L https://github.com/docker/machine/releases/download/v0.16.2/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine",
                    { key: "10-docker-machine" }),
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner start", {
                        key: "20-gitlab-runner-start",
                    }),
                ]),
                [CONFIG]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitFile.fromString("/etc/gitlab-runner/config.toml", runner_configuration_1.ConfigurationMapper.withDefaults({
                        globalConfiguration: this.globalConfiguration,
                        runnersConfiguration: this.runners.map((runner) => {
                            const configuration = runner.configuration;
                            return {
                                ...configuration,
                                machine: {
                                    ...configuration.machine,
                                    machineOptions: {
                                        sshKeypath: runner.keyPair ? DEFAULT_SSH_KEY_PATH : "",
                                        ...configuration.machine?.machineOptions,
                                        instanceType: runner.instanceType.toString(),
                                        ami: runner.machineImage.getImage(scope).imageId,
                                        region: scope.region,
                                        vpcId: this.network.vpc.vpcId,
                                        zone: this.network.availabilityZone.slice(-1),
                                        subnetId: this.network.subnet.subnetId,
                                        securityGroup: this.runnersSecurityGroupName,
                                        privateAddressOnly: configuration.machine?.machineOptions?.privateAddressOnly ?? this.network.hasPrivateSubnets(),
                                        usePrivateAddress: configuration.machine?.machineOptions?.usePrivateAddress ?? true,
                                        iamInstanceProfile: runner.instanceProfile.ref,
                                        userdata: "/etc/gitlab-runner/user_data_runners",
                                    },
                                },
                                cache: {
                                    s3: {
                                        serverAddress: `s3.${scope.urlSuffix}`,
                                        bucketName: `${this.cacheBucket.bucketName}`,
                                        bucketLocation: `${scope.region}`,
                                        authenticationType: "iam",
                                    },
                                },
                            };
                        }),
                    }).toToml(), {
                        owner: "gitlab-runner",
                        group: "gitlab-runner",
                        mode: "000600",
                    }),
                    aws_ec2_1.InitFile.fromString("/etc/rsyslog.d/25-gitlab-runner.conf", `:programname, isequal, "gitlab-runner" /var/log/gitlab-runner.log`, {
                        owner: "root",
                        group: "root",
                        mode: "000644",
                    }),
                    aws_ec2_1.InitService.enable("gitlab-runner", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: gitlabRunnerConfigRestartHandle,
                    }),
                    aws_ec2_1.InitService.enable("rsyslog", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: rsyslogConfigRestartHandle,
                    }),
                    aws_ec2_1.InitCommand.shellCommand(
                    // Download custom EC2 key pair for manager <> runner ssh connect
                    this.runners
                        .map((runner) => {
                        if (!runner.keyPair) {
                            return "";
                        }
                        runner.keyPair.grantRead(this.role);
                        const region = aws_cdk_lib_1.Stack.of(this).region;
                        const secretArn = runner.keyPair.secretArn;
                        const keyPairName = runner.configuration.machine.machineOptions.keypairName;
                        const sshKeyPath = runner.configuration.machine.machineOptions.sshKeypath ?? DEFAULT_SSH_KEY_PATH;
                        return [
                            `mkdir -p ${sshKeyPath};`,
                            `echo $(aws secretsmanager get-secret-value --region ${region} --secret-id ${secretArn} --query SecretString --output text | jq -r ."${keyPairName}") > ${sshKeyPath}/${keyPairName};`,
                            `echo $(aws secretsmanager get-secret-value --region ${region} --secret-id ${secretArn} --query SecretString --output text | jq -r ."${keyPairName}.pub") > ${sshKeyPath}/${keyPairName}.pub;`,
                        ].join("\n");
                    })
                        .filter((s) => s.length > 0)
                        .join("\n"), {
                        key: "999-retrieve-ec2-key-pair",
                    }),
                    aws_ec2_1.InitFile.fromString("/etc/gitlab-runner/user_data_runners", userDataRunners.render(), {
                        owner: "gitlab-runner",
                        group: "gitlab-runner",
                        mode: "000600",
                    }),
                ]),
                [RESTART]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner restart", {
                        key: "10-gitlab-runner-restart",
                    }),
                ]),
            },
        });
    }
}
exports.GitlabRunnerAutoscalingManager = GitlabRunnerAutoscalingManager;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscalingManager[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.GitlabRunnerAutoscalingManager", version: "0.2.187" };
//# sourceMappingURL=data:application/json;base64,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