import { aws_iam as iam, aws_lambda as lambda, aws_secretsmanager as secretsmanager, aws_kms as kms, aws_s3 as s3 } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * The properties of a new set of SMTP Credentials
 */
export interface AccessKeyProps {
    /**
     * The IAM user the access key will be created for.
     */
    readonly user: iam.User;
    /**
     * If a secret is pending deletion should it be restored?
     *
     * This helps in cases where cloudformation roll backs puts a secret in pending delete state.
     *
     * @default true
     */
    readonly restoreSecret?: boolean;
    /**
     * If a secret already exists should it be overwritten?
     *
     * This helps in cases where cloudformation creates a secret successfully but it gets orphaned for some reason.
     *
     * @default true
     */
    readonly overwriteSecret?: boolean;
    /**
     * The KMS key to use for the secret
     *
     * @default - default key
     */
    readonly kmsKey?: kms.IKey;
    /**
     * The lambda function code to use for the custom resource.
     *
     * For most use cases this should be left as the default, but
     * in cases where the custom resource needs to be deployed through
     * something like CloudFormation StackSets you may need to source
     * the lambda function code from S3 or some other location because
     * the CDK cant upload the local code to the correct asset location
     * for the StackSet target accounts.
     *
     * You can use the included `AccessKeyFunctionCodeCache` class to
     * cache the lambda function code in S3 and create a cross
     * account access policy to allow the StackSet target accounts
     * to access the code.
     *
     * @default - default lambda function code
     */
    readonly lambdaCode?: lambda.Code;
}
export declare class AccessKey extends Construct {
    /**
     * The AWS secrets manager secret that contains the access key.
     */
    readonly secret: secretsmanager.ISecret;
    constructor(scope: Construct, id: string, props: AccessKeyProps);
}
export interface AccessKeyFunctionCodeCacheProps extends s3.BucketProps {
}
export declare class AccessKeyFunctionCodeCache extends s3.Bucket {
    /**
     * The lambda.Code object that represents the contents of the bucket.
     */
    readonly lambdaCode: lambda.Code;
    constructor(scope: Construct, id: string, props: AccessKeyFunctionCodeCacheProps);
    /**
     * Add access to the whole organization to get
     * the lambda function code from the bucket.
     *
     * @param principalOrgId The organization ID to require for any accounts accessing the bucket.
     */
    addOrgWideAccessPolicy(principalOrgId: string): iam.AddToResourcePolicyResult;
    /**
     * Add access to the specified accounts to get
     * the lambda function code from the bucket.
     *
     * @param principalAccountId The account ID to add access for.
     * @param principalOrgId (Optional) The organization ID to require for the account accessing the bucket.
     */
    addAccountAccessPolicy(principalAccountId: string, principalOrgId?: string): iam.AddToResourcePolicyResult;
    /**
     * Add access to the specified organizational units to get
     * the lambda function code from the bucket.
     *
     * @param principalOrgPaths The organizational unit paths to add access for.
     */
    addOrgOuAccessPolicy(principalOrgPaths: string[]): iam.AddToResourcePolicyResult;
}
