"""
The General Aviation Aircraft (GAA) problem.

T. W. Simpson, W. Chen, J. K. Allen, and F. Mistree (1996), 
"Conceptual design of a family of products through the use of the robust
concept exploration method," in 6th AIAA/USAF/NASA/ ISSMO Symposium on 
Multidiciplinary Analysis and Optimization, vol. 2, pp. 1535-1545.

T. W. Simpson, B. S. D'Souza (2004), "Assessing variable levels of platform 
commonality within a product family using a multiobjective genetic algorithm," 
Concurrent Engineering: Research and Applications, vol. 12, no. 2, pp. 119-130.

R. Shah, P. M. Reed, and T. W. Simpson (2011), "Many-objective evolutionary optimization 
and visual analytics for product family design," Multiobjective Evolutionary Optimisation 
for Product Design and Manufacturing, Springer, London, pp. 137-159.

M. Woodruff, T. W. Simpson, P. M. Reed (2013), "Diagnostic Analysis of Metamodels' 
Multivariate Dependencies and their Impacts in Many-Objective Design Optimization," 
Proceedings of the ASME 2013 IDETC/CIE Conference, Paper No. DETC2013-13125.

https://github.com/matthewjwoodruff/generalaviation

"""

from desdeo_problem.problem.Variable import Variable
from desdeo_problem.problem.Objective import ScalarObjective
from desdeo_problem.problem.Problem import MOProblem, ProblemBase
from desdeo_problem import ScalarConstraint, problem

import numpy as np

def gaa(var_iv: np.array = np.array([
    0.36, 9, 3, 5.7, 22, 97, 17, 3.4, 0.75,
    0.36, 9, 3, 5.7, 22, 97, 17, 3.4, 0.75,
    0.36, 9, 3, 5.7, 22, 97, 17, 3.4, 0.75
    ])) -> MOProblem:
    """The General Aviation Aircraft (GAA) problem.
    27 design variables, 10 objectives and 1 constraint.

    Arguments:
        var_iv (np.array): Optional, initial variable values.

    Returns:
        MOProblem: a problem object.
    """
    
    # Check the number of variables
    if (np.shape(np.atleast_2d(var_iv)[0]) != (27,)):
        raise RuntimeError("Number of variables must be 27")

    # Lower bounds
    lb = np.array([
        0.24, 7, 0, 5.5, 19, 85, 14, 3, 0.46,
	    0.24, 7, 0, 5.5, 19, 85, 14, 3, 0.46,
	    0.24, 7, 0, 5.5, 19, 85, 14, 3, 0.46
    ])
    
    # Upper bounds
    ub = np.array([
        0.48, 11, 6, 5.968, 25, 110, 20, 3.75, 1,
	    0.48, 11, 6, 5.968, 25, 110, 20, 3.75, 1,
	    0.48, 11, 6, 5.968, 25, 110, 20, 3.75, 1
    ])

    # Check the variable bounds
    if np.any(lb > var_iv) or np.any(ub < var_iv):
        raise ValueError("Initial variable values need to be between lower and upper bounds")

    def scaled_variables(x: np.ndarray) -> np.ndarray:
        scaled_x = np.zeros(x.shape)
        scaled_x[:,0] = (x[:,0] - 0.36) / 0.12
        scaled_x[:,1] = (x[:,1] - 9) / 2
        scaled_x[:,2] = (x[:,2] - 3) / 3
        scaled_x[:,3] = (x[:,3] - 5.734) / 0.234
        scaled_x[:,4] = (x[:,4] - 22) / 3
        scaled_x[:,5] = (x[:,5] - 97.5) / 12.5
        scaled_x[:,6] = (x[:,6] - 17) / 3
        scaled_x[:,7] = (x[:,7] - 3.375) / 0.375
        scaled_x[:,8] = (x[:,8] - 0.73) / 0.27
        scaled_x[:,9] = (x[:,9] - 0.36) / 0.12
        scaled_x[:,10] = (x[:,10] - 9) / 2
        scaled_x[:,11] = (x[:,11] - 3) / 3
        scaled_x[:,12] = (x[:,12] - 5.734) / 0.234
        scaled_x[:,13] = (x[:,13] - 22) / 3
        scaled_x[:,14] = (x[:,14] - 97.5) / 12.5
        scaled_x[:,15] = (x[:,15] - 17) / 3
        scaled_x[:,16] = (x[:,16] - 3.375) / 0.375
        scaled_x[:,17] = (x[:,17] - 0.73) / 0.27
        scaled_x[:,18] = (x[:,18] - 0.36) / 0.12
        scaled_x[:,19] = (x[:,19] - 9) / 2
        scaled_x[:,20] = (x[:,20] - 3) / 3
        scaled_x[:,21] = (x[:,21] - 5.734) / 0.234
        scaled_x[:,22] = (x[:,22] - 22) / 3
        scaled_x[:,23] = (x[:,23] - 97.5) / 12.5
        scaled_x[:,24] = (x[:,24] - 17) / 3
        scaled_x[:,25] = (x[:,25] - 3.375) / 0.375
        return scaled_x

    # Response variables
    def noise_2(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        74.099998 - 0.0004 * x[:,0] - 0.0156 * x[:,1] 
        + 0.0003 * x[:,2] + 0.9684 * x[:,3] 
        + 0.0316 * x[:,4] - 0.0053 * x[:,5] - 0.0015 * x[:,6] - 0.0002 * x[:,7] 
        + 0.0007 * x[:,8] - 0.0001 * x[:,0] * x[:,3] - 0.0001 * x[:,0] * x[:,4] 
        - 0.0001 * x[:,0] * x[:,5] + 0 * x[:,0] * x[:,5] 
        + 0.0001 * x[:,0] * x[:,7] + 0 * x[:,1] * x[:,2] - 0.0012 * x[:,1] * x[:,3] 
        - 0.0014 * x[:,1] * x[:,4] - 0.0002 * x[:,1] * x[:,5] 
        + 0 * x[:,1] * x[:,6] + 0.0003 * x[:,1] * x[:,7]
		+ 0.0001 * x[:,1] * x[:,8] + 0 * x[:,2] * x[:,8] - 0.0003 * x[:,3]
		* x[:,4] + 0.0058 * x[:,3] * x[:,5] - 0.0001 * x[:,3] * x[:,6]
		+ 0.0002 * x[:,3] * x[:,7] - 0.0001 * x[:,3] * x[:,8] + 0.0002
		* x[:,4] * x[:,5] - 0.0003 * x[:,4] * x[:,6] - 0.0002 * x[:,4]
		* x[:,7] + 0.0001 * x[:,4] * x[:,8] - 0.0001 * x[:,5] * x[:,6]
		+ 0 * x[:,5] * x[:,8] - 0.0001 * x[:,6] * x[:,7] - 0.0001 * x[:,6]
		* x[:,8] + 0 * x[:,7] * x[:,8] + 0.0008 * pow(x[:,0], 2)
		+ 0.0016 * pow(x[:,1], 2) + 0.0011 * pow(x[:,2], 2) + 0.1105
		* pow(x[:,3], 2) + 0.0004 * pow(x[:,4], 2) - 0.0019
		* pow(x[:,5], 2) + 0.0009 * pow(x[:,6], 2) + 0.001
		* pow(x[:,7], 2) + 0.0007 * pow(x[:,8], 2)
        )

    def wemp_2(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        1917 + 5.979 * x[:,0] + 35.130001 * x[:,1] - 0.7119 * x[:,2]
		+ 11.11 * x[:,3] - 32.290001 * x[:,4] + 5.739 * x[:,5]
		+ 48.110001 * x[:,6] + 0.3376 * x[:,7] + 15.28 * x[:,8] + 1.244
		* x[:,0] * x[:,1] - 0.1315 * x[:,0] * x[:,2] + 1.129 * x[:,0]
		* x[:,3] - 2.393 * x[:,0] * x[:,4] + 0.3954 * x[:,0] * x[:,5]
		- 0.4978 * x[:,0] * x[:,6] - 0.3882 * x[:,0] * x[:,7] + 0.5742
		* x[:,0] * x[:,8] - 0.2236 * x[:,1] * x[:,2] - 0.0739 * x[:,1]
		* x[:,3] - 3.805 * x[:,1] * x[:,4] - 0.0164 * x[:,1] * x[:,5] - 0.0923
		* x[:,1] * x[:,6] - 0.9326 * x[:,1] * x[:,7] + 3.135 * x[:,1] * x[:,8]
		- 0.0385 * x[:,2] * x[:,3] + 0.4376 * x[:,2] * x[:,4] + 0.0259
		* x[:,2] * x[:,5] + 0.4009 * x[:,2] * x[:,6] + 0.3002 * x[:,2]
		* x[:,7] + 0.7036 * x[:,2] * x[:,8] - 0.2083 * x[:,3] * x[:,4]
		+ 1.165 * x[:,3] * x[:,5] - 0.2119 * x[:,3] * x[:,6] - 0.1934
		* x[:,3] * x[:,7] + 0.1462 * x[:,3] * x[:,8] - 0.0644 * x[:,4]
		* x[:,5] + 3.194 * x[:,4] * x[:,6] + 2.672 * x[:,4] * x[:,7]
		- 0.4407 * x[:,4] * x[:,8] - 0.057 * x[:,5] * x[:,6] - 0.0529
		* x[:,5] * x[:,7] + 0.0341 * x[:,5] * x[:,8] + 4.88 * x[:,6] * x[:,7]
		+ 1.349 * x[:,6] * x[:,8] + 0.8836 * x[:,7] * x[:,8] + 0.958
		* pow(x[:,0], 2) - 1.812 * pow(x[:,1], 2) + 1.173 * pow(x[:,2], 2)
		+ 0.753 * pow(x[:,3], 2) + 3.638 * pow(x[:,4], 2) + 0.133
		* pow(x[:,5], 2) + 5.323 * pow(x[:,6], 2) + 1.478 * pow(x[:,7], 2)
		- 0.192 * pow(x[:,8], 2)
        )

    def doc_2(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        83.17 + 12.53 * x[:,0] - 0.0477 * x[:,1] - 0.0215 * x[:,2]
		+ 3.597 * x[:,3] - 0.7367 * x[:,4] + 0.7481 * x[:,5] + 0.733
		* x[:,6] - 0.2029 * x[:,7] + 0.0393 * x[:,8] + 0.6526 * x[:,0]
		* x[:,1] + 0.0481 * x[:,0] * x[:,2] + 1.208 * x[:,0] * x[:,3]
		+ 0.6802 * x[:,0] * x[:,4] + 0.0992 * x[:,0] * x[:,5] - 0.7074
		* x[:,0] * x[:,6] + 0.2768 * x[:,0] * x[:,7] + 0.0109 * x[:,0]
		* x[:,8] + 0.0031 * x[:,1] * x[:,2] + 0.2146 * x[:,1] * x[:,3]
		- 0.0721 * x[:,1] * x[:,4] - 0.2445 * x[:,1] * x[:,5] - 0.0172 * x[:,1]
		* x[:,6] + 0.0127 * x[:,1] * x[:,7] + 0.0087 * x[:,1] * x[:,8]
		+ 0.0169 * x[:,2] * x[:,3] + 0.0151 * x[:,2] * x[:,4] - 0.0063
		* x[:,2] * x[:,5] - 0.0001 * x[:,2] * x[:,6] - 0.0042 * x[:,2]
		* x[:,7] - 0.0059 * x[:,2] * x[:,8] + 0.0789 * x[:,3] * x[:,4]
		+ 0.676 * x[:,3] * x[:,5] - 0.1912 * x[:,3] * x[:,6] + 0.0519
		* x[:,3] * x[:,7] - 0.0265 * x[:,3] * x[:,8] + 0.0136 * x[:,4]
		* x[:,5] + 0.0804 * x[:,4] * x[:,6] + 0.0577 * x[:,4] * x[:,7]
		+ 0.017 * x[:,4] * x[:,8] - 0.0617 * x[:,5] * x[:,6] + 0.0058
		* x[:,5] * x[:,7] - 0.0178 * x[:,5] * x[:,8] + 0.0901 * x[:,6]
		* x[:,7] + 0.0047 * x[:,6] * x[:,8] - 0.003 * x[:,7] * x[:,8]
		- 11.37 * pow(x[:,0], 2) - 0.2836 * pow(x[:,1], 2) - 0.3149
		* pow(x[:,2], 2) + 5.337 * pow(x[:,3], 2) - 0.3711
		* pow(x[:,4], 2) - 0.071 * pow(x[:,5], 2) - 0.2177
		* pow(x[:,6], 2) - 0.2354 * pow(x[:,7], 2) - 0.238
		* pow(x[:,8], 2)
        )

    def rough_2(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        2.197 - 0.0002 * x[:,0] + 0.1541 * x[:,1] - 0.0012 * x[:,2]
		+ 0.0222 * x[:,3] - 0.1611 * x[:,4] - 0.0012 * x[:,5] - 0.0628
		* x[:,6] - 0.011 * x[:,7] + 0.0068 * x[:,8] + 0.0006 * x[:,0]
		* x[:,1] + 0.0001 * x[:,0] * x[:,2] + 0.001 * x[:,0] * x[:,4]
		+ 0.0001 * x[:,0] * x[:,6] - 0.0002 * x[:,0] * x[:,7] - 0.0006
		* x[:,0] * x[:,8] - 0.0001 * x[:,1] * x[:,2] + 0.0006 * x[:,1]
		* x[:,3] - 0.0113 * x[:,1] * x[:,4] - 0.0001 * x[:,1] * x[:,5] - 0.0045
		* x[:,1] * x[:,6] - 0.0017 * x[:,1] * x[:,7] - 0.0011 * x[:,1] * x[:,8]
		- 0.0002 * x[:,2] * x[:,3] - 0.0004 * x[:,2] * x[:,4] - 0.0006
		* x[:,2] * x[:,6] - 0.0003 * x[:,2] * x[:,7] - 0.0001 * x[:,2]
		* x[:,8] - 0.0051 * x[:,3] * x[:,4] - 0.0038 * x[:,3] * x[:,5]
		+ 0.0022 * x[:,3] * x[:,6] - 0.0012 * x[:,3] * x[:,7] - 0.0002
		* x[:,3] * x[:,8] - 0.001 * x[:,4] * x[:,5] - 0.0036 * x[:,4]
		* x[:,6] - 0.0025 * x[:,4] * x[:,7] - 0.0025 * x[:,4]
		* x[:,8] + 0.0008 * x[:,5] * x[:,6] - 0.0003 * x[:,5] * x[:,7]
		- 0.0001 * x[:,5] * x[:,8] - 0.001 * x[:,6] * x[:,7] - 0.0007
		* x[:,6] * x[:,8] + 0.0012 * pow(x[:,0], 2) - 0.0273
		* pow(x[:,1], 2) - 0.0048 * pow(x[:,2], 2) + 0.0033
		* pow(x[:,3], 2) + 0.0062 * pow(x[:,4], 2) + 0.0021
		* pow(x[:,5], 2) - 0.0016 * pow(x[:,6], 2) - 0.0011
		* pow(x[:,7], 2) - 0.0048 * pow(x[:,8], 2)
        )
    
    def wfuel_2(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        416.399994 - 6.093 * x[:,0] - 31.91 * x[:,1] + 0.7968
		* x[:,2] - 19.17 * x[:,3] + 34.189999 * x[:,4] - 7.57 * x[:,5]
		- 49.610001 * x[:,6] + 0.2331 * x[:,7] - 15.33 * x[:,8] - 1.201
		* x[:,0] * x[:,1] + 0.1735 * x[:,0] * x[:,2] - 1.247 * x[:,0]
		* x[:,3] + 1.703 * x[:,0] * x[:,4] - 0.4588 * x[:,0] * x[:,5]
		+ 0.1585 * x[:,0] * x[:,6] + 0.6156 * x[:,0] * x[:,7] - 0.528
		* x[:,0] * x[:,8] + 0.2215 * x[:,1] * x[:,2] - 0.4976 * x[:,1]
		* x[:,3] + 4.058 * x[:,1] * x[:,4] - 0.108 * x[:,1] * x[:,5] + 0.2679
		* x[:,1] * x[:,6] + 0.8514 * x[:,1] * x[:,7] - 3.182 * x[:,1] * x[:,8]
		+ 0.0359 * x[:,2] * x[:,3] - 0.482 * x[:,2] * x[:,4] - 0.0207
		* x[:,2] * x[:,5] - 0.3878 * x[:,2] * x[:,6] - 0.3249 * x[:,2]
		* x[:,7] - 0.715 * x[:,2] * x[:,8] + 0.3374 * x[:,3] * x[:,4]
		- 2.403 * x[:,3] * x[:,5] + 0.4519 * x[:,3] * x[:,6] + 0.1352
		* x[:,3] * x[:,7] - 0.123 * x[:,3] * x[:,8] + 0.2498 * x[:,4]
		* x[:,5] - 2.896 * x[:,4] * x[:,6] - 3.016 * x[:,4] * x[:,7]
		+ 0.3662 * x[:,4] * x[:,8] - 0.114 * x[:,5] * x[:,6] + 0.0571
		* x[:,5] * x[:,7] - 0.0222 * x[:,5] * x[:,8] - 4.689 * x[:,6]
		* x[:,7] - 1.339 * x[:,6] * x[:,8] - 0.9311 * x[:,7] * x[:,8]
		- 0.7538 * pow(x[:,0], 2) + 1.13 * pow(x[:,1], 2) - 1.078
		* pow(x[:,2], 2) - 5.989 * pow(x[:,3], 2) - 3.043
		* pow(x[:,4], 2) + 0.0627 * pow(x[:,5], 2) - 4.958
		* pow(x[:,6], 2) - 1.41 * pow(x[:,7], 2) + 0.3532
		* pow(x[:,8], 2)
        )

    def purch_2(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        43280 + 133.1 * x[:,0] + 780.400024 * x[:,1] - 1.501
		* x[:,2] + 494 * x[:,3] + 191.2 * x[:,5] + 786 * x[:,6] + 102
		* x[:,7] + 333.9 * x[:,8] + 25.95 * x[:,0] * x[:,1] - 2.442 * x[:,0]
		* x[:,2] + 28.280001 * x[:,0] * x[:,3] - 50.509998 * x[:,0]
		* x[:,4] + 11.55 * x[:,0] * x[:,5] - 6.188 * x[:,0] * x[:,6]
		- 4.326 * x[:,0] * x[:,7] + 13.78 * x[:,0] * x[:,8] - 3.335 * x[:,1]
		* x[:,2] + 0.4272 * x[:,1] * x[:,3] - 69.110001 * x[:,1] * x[:,4]
		- 331.8 * x[:,4] - 0.8834 * x[:,1] * x[:,5] + 7.221 * x[:,1] * x[:,6]
		- 20.389999 * x[:,1] * x[:,7] + 69.580002 * x[:,1] * x[:,8] - 1.107
		* x[:,2] * x[:,3] + 7.078 * x[:,2] * x[:,4] + 0.0705 * x[:,2]
		* x[:,5] + 5.359 * x[:,2] * x[:,6] + 4.936 * x[:,2] * x[:,7]
		+ 14.77 * x[:,2] * x[:,8] - 21.82 * x[:,3] * x[:,4] + 22.17
		* x[:,3] * x[:,5] + 14.2 * x[:,3] * x[:,6] - 7.537 * x[:,3]
		* x[:,7] + 5.813 * x[:,3] * x[:,8] - 6.771 * x[:,4] * x[:,5]
		+ 27.99 * x[:,4] * x[:,6] + 47.080002 * x[:,4] * x[:,7]
		- 9.888 * x[:,4] * x[:,8] + 3.135 * x[:,5] * x[:,6] - 2.658 * x[:,5]
		* x[:,7] + 1.033 * x[:,5] * x[:,8] + 82.230003 * x[:,6] * x[:,7]
		+ 27.969999 * x[:,6] * x[:,8] + 16.110001 * x[:,7] * x[:,8]
		+ 10.51 * pow(x[:,0], 2) - 49.189999 * pow(x[:,1], 2) + 26.51
		* pow(x[:,2], 2) + 35.310001 * pow(x[:,3], 2) + 33.560001
		* pow(x[:,4], 2) + 10.66 * pow(x[:,5], 2) + 74.309998
		* pow(x[:,6], 2) + 8.461 * pow(x[:,7], 2) - 0.7392
		* pow(x[:,8], 2)
        )

    def range_2(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        1971 - 6.807 * x[:,0] - 76.230003 * x[:,1] + 2.885 * x[:,2]
		- 363.799988 * x[:,3] - 383.100006 * x[:,4] - 68.099998 * x[:,5]
		- 67.480003 * x[:,6] + 18.09 * x[:,7] - 165.699997 * x[:,8]
		+ 0.084 * x[:,0] * x[:,1] + 0.2793 * x[:,0] * x[:,2] + 1.674 * x[:,0]
		* x[:,3] + 3.342 * x[:,0] * x[:,4] + 0.1543 * x[:,0] * x[:,5]
		- 0.4629 * x[:,0] * x[:,6] + 0.8574 * x[:,0] * x[:,7] + 2.537
		* x[:,0] * x[:,8] + 0.0098 * x[:,1] * x[:,2] - 23.23 * x[:,1] * x[:,3]
		- 25.799999 * x[:,1] * x[:,4] - 1.787 * x[:,1] * x[:,5] + 9.432 * x[:,1]
		* x[:,6] - 1.912 * x[:,1] * x[:,7] - 40.200001 * x[:,1] * x[:,8]
		- 1.479 * x[:,2] * x[:,3] - 1.311 * x[:,2] * x[:,4] - 0.2168
		* x[:,2] * x[:,5] - 0.7168 * x[:,2] * x[:,6] - 0.584 * x[:,2]
		* x[:,7] - 0.8652 * x[:,2] * x[:,8] + 121.099998 * x[:,3]
		* x[:,4] - 26.709999 * x[:,3] * x[:,5] + 25.280001 * x[:,3]
		* x[:,6] - 10.74 * x[:,3] * x[:,7] + 37.080002 * x[:,3]
		* x[:,8] + 21.290001 * x[:,4] * x[:,5] + 19.299999 * x[:,4]
		* x[:,6] - 4.318 * x[:,4] * x[:,7] - 20.360001 * x[:,4]
		* x[:,8] + 3.611 * x[:,5] * x[:,6] - 1.678 * x[:,5] * x[:,7] + 7.541
		* x[:,5] * x[:,8] - 7.232 * x[:,6] * x[:,7] + 16.860001 * x[:,6]
		* x[:,8] - 3.006 * x[:,7] * x[:,8] - 11.61 * pow(x[:,0], 2)
		+ 4.892 * pow(x[:,1], 2) - 11.61 * pow(x[:,2], 2) - 84.110001
		* pow(x[:,3], 2) + 65.389999 * pow(x[:,4], 2) - 7.108
		* pow(x[:,5], 2) - 15.61 * pow(x[:,6], 2) - 13.11 * pow(x[:,7], 2)
		+ 8.892 * pow(x[:,8], 2)
        )

    def ldmax_2(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        17.780001 + 0.4845 * x[:,0] + 1.625 * x[:,1] + 0.0267
		* x[:,2] - 0.0153 * x[:,3] - 0.5289 * x[:,4] - 0.007 * x[:,5]
		- 0.4965 * x[:,6] + 0.2108 * x[:,7] + 0.0302 * x[:,8] + 0.0598
		* x[:,0] * x[:,1] + 0.0019 * x[:,0] * x[:,2] - 0.0085 * x[:,0]
		* x[:,3] - 0.0146 * x[:,0] * x[:,4] - 0.0018 * x[:,0] * x[:,5]
		- 0.0014 * x[:,0] * x[:,6] + 0.0104 * x[:,0] * x[:,7] + 0.0044
		* x[:,0] * x[:,8] + 0.0053 * x[:,1] * x[:,2] - 0.0024 * x[:,1]
		* x[:,3] - 0.0612 * x[:,1] * x[:,4] - 0.001 * x[:,1] * x[:,5] - 0.0517
		* x[:,1] * x[:,6] + 0.0128 * x[:,1] * x[:,7] + 0.0043 * x[:,1] * x[:,8]
		- 0.0001 * x[:,2] * x[:,3] - 0.0054 * x[:,2] * x[:,4] + 0
		* x[:,2] * x[:,5] - 0.0072 * x[:,2] * x[:,6] - 0.0039 * x[:,2]
		* x[:,7] + 0 * x[:,2] * x[:,8] + 0.0002 * x[:,3] * x[:,4]
		- 0.0012 * x[:,3] * x[:,5] + 0.0016 * x[:,3] * x[:,6] - 0.0001
		* x[:,3] * x[:,7] + 0.0007 * x[:,3] * x[:,8] - 0.0001 * x[:,4]
		* x[:,5] - 0.0727 * x[:,4] * x[:,6] - 0.0256 * x[:,4] * x[:,7]
		- 0.0033 * x[:,4] * x[:,8] + 0.0008 * x[:,5] * x[:,6] - 0.0003
		* x[:,5] * x[:,7] + 0.0003 * x[:,5] * x[:,8] - 0.0525 * x[:,6]
		* x[:,7] - 0.0071 * x[:,6] * x[:,8] - 0.0059 * x[:,7] * x[:,8]
		- 0.0897 * pow(x[:,0], 2) - 0.1488 * pow(x[:,1], 2) - 0.0116
		* pow(x[:,2], 2) - 0.0009 * pow(x[:,3], 2) - 0.0305
		* pow(x[:,7], 2) - 0.0009 * pow(x[:,8], 2) + 0.0157
		* pow(x[:,4], 2) - 0.001 * pow(x[:,5], 2) - 0.0577
		* pow(x[:,6], 2)
        )

    def vcmax_2(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        200.4 - 0.3799 * x[:,0] + 0.8236 * x[:,1] + 0.2168 * x[:,2]
		+ 1.74 * x[:,3] + 5.589 * x[:,4] - 0.1683 * x[:,5] - 3.522
		* x[:,6] + 1.559 * x[:,7] + 0.2442 * x[:,8] - 0.0215 * x[:,0]
		* x[:,1] + 0.0054 * x[:,0] * x[:,2] - 0.0465 * x[:,0] * x[:,3]
		- 0.0144 * x[:,0] * x[:,4] - 0.0069 * x[:,0] * x[:,5] + 0.0578
		* x[:,0] * x[:,6] + 0.0557 * x[:,0] * x[:,7] + 0.0221 * x[:,0]
		* x[:,8] + 0.0175 * x[:,1] * x[:,2] - 0.0942 * x[:,1] * x[:,3]
		+ 0.1402 * x[:,1] * x[:,4] - 0.0079 * x[:,1] * x[:,5] + 0.0147 * x[:,1]
		* x[:,6] - 0.071 * x[:,1] * x[:,7] - 0.0104 * x[:,1] * x[:,8]
		- 0.0067 * x[:,2] * x[:,3] - 0.0306 * x[:,2] * x[:,4] - 0.0078
		* x[:,2] * x[:,5] - 0.0534 * x[:,2] * x[:,6] - 0.0224 * x[:,2]
		* x[:,7] + 0.0026 * x[:,2] * x[:,8] - 0.2771 * x[:,3] * x[:,4]
		- 0.3476 * x[:,3] * x[:,5] + 0.2093 * x[:,3] * x[:,6] - 0.0864
		* x[:,3] * x[:,7] - 0.0051 * x[:,3] * x[:,8] - 0.0948 * x[:,4]
		* x[:,5] - 0.7081 * x[:,4] * x[:,6] - 0.1219 * x[:,4] * x[:,7]
		- 0.016 * x[:,4] * x[:,8] + 0.0555 * x[:,5] * x[:,6] - 0.0342
		* x[:,5] * x[:,7] - 0.0049 * x[:,5] * x[:,8] - 0.3658 * x[:,6]
		* x[:,7] - 0.046 * x[:,6] * x[:,8] - 0.0504 * x[:,7] * x[:,8]
		- 0.1452 * pow(x[:,0], 2) - 0.2937 * pow(x[:,1], 2) + 0.0303
		* pow(x[:,2], 2) + 0.1378 * pow(x[:,3], 2) - 0.6522
		* pow(x[:,4], 2) + 0.1378 * pow(x[:,5], 2) - 0.6102
		* pow(x[:,6], 2) - 0.3722 * pow(x[:,7], 2) + 0.0303
		* pow(x[:,8], 2)
        )

    def noise_4(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        74.099998 - 0.0005 * x[:,9] - 0.0158 * x[:,10] + 0.0003
		* x[:,11] + 0.9682 * x[:,12] + 0.0316 * x[:,13] - 0.0053 * x[:,14]
		- 0.0014 * x[:,15] - 0.0003 * x[:,16] + 0.0008 * x[:,17] + 0
		* x[:,9] * x[:,11] - 0.0001 * x[:,9] * x[:,12] - 0.0001 * x[:,9]
		* x[:,13] - 0.0001 * x[:,9] * x[:,14] + 0.0001 * x[:,9] * x[:,16] + 0
		* x[:,10] * x[:,11] - 0.0012 * x[:,10] * x[:,12] - 0.0014 * x[:,10] * x[:,13]
		- 0.0002 * x[:,10] * x[:,14] + 0.0002 * x[:,10] * x[:,16] + 0.0002 * x[:,10]
		* x[:,17] + 0 * x[:,11] * x[:,17] - 0.0003 * x[:,12] * x[:,13]
		+ 0.0057 * x[:,12] * x[:,14] - 0.0001 * x[:,12] * x[:,15] + 0.0001
		* x[:,12] * x[:,16] + 0.0002 * x[:,13] * x[:,14] - 0.0002 * x[:,13]
		* x[:,15] - 0.0001 * x[:,13] * x[:,16] + 0.0001 * x[:,13]
		* x[:,17] - 0.0001 * x[:,14] * x[:,15] + 0 * x[:,14] * x[:,17] - 0.0001
		* x[:,15] * x[:,16] + 0.0008 * pow(x[:,9], 2) + 0.0016
		* pow(x[:,10], 2) + 0.0011 * pow(x[:,11], 2) + 0.1104
		* pow(x[:,12], 2) + 0.0003 * pow(x[:,13], 2) - 0.0019
		* pow(x[:,14], 2) + 0.001 * pow(x[:,15], 2) + 0.001 * pow(x[:,16], 2)
		+ 0.0007 * pow(x[:,17], 2)
        )

    def wemp_4(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        1947 + 6.338 * x[:,9] + 33.869999 * x[:,10] - 0.448 * x[:,11]
		+ 11 * x[:,12] - 30.85 * x[:,13] + 5.723 * x[:,14] + 53.220001
		* x[:,15] + 1.896 * x[:,16] + 15.26 * x[:,17] + 1.963 * x[:,9]
		* x[:,10] - 0.1599 * x[:,9] * x[:,11] + 1.073 * x[:,9] * x[:,12]
		- 1.699 * x[:,9] * x[:,13] + 0.462 * x[:,9] * x[:,14] - 0.9528
		* x[:,9] * x[:,15] - 0.9851 * x[:,9] * x[:,16] + 0.5956 * x[:,9]
		* x[:,17] + 0.0065 * x[:,10] * x[:,11] + 0.0874 * x[:,10] * x[:,12]
		- 3.447 * x[:,10] * x[:,13] + 0.1024 * x[:,10] * x[:,14] - 0.1814 * x[:,10]
		* x[:,15] + 0.7878 * x[:,10] * x[:,16] + 1.592 * x[:,10] * x[:,17]
		+ 0.0288 * x[:,11] * x[:,12] + 0.3498 * x[:,11] * x[:,13] + 0.034
		* x[:,11] * x[:,14] + 0.3134 * x[:,11] * x[:,15] + 0.2289 * x[:,11]
		* x[:,16] + 0.5603 * x[:,11] * x[:,17] - 0.1862 * x[:,12] * x[:,13]
		+ 1.061 * x[:,12] * x[:,14] - 0.0774 * x[:,12] * x[:,15] - 0.2335
		* x[:,12] * x[:,16] + 0.1385 * x[:,12] * x[:,17] - 0.0914 * x[:,13]
		* x[:,14] + 1.932 * x[:,13] * x[:,15] + 1.853 * x[:,13] * x[:,16]
		- 0.8019 * x[:,13] * x[:,17] - 0.0754 * x[:,14] * x[:,15] - 0.1306
		* x[:,14] * x[:,16] + 0.1086 * x[:,14] * x[:,17] + 4.81 * x[:,15] * x[:,16]
		+ 1.309 * x[:,15] * x[:,17] + 1.265 * x[:,16] * x[:,17] + 0.4046
		* pow(x[:,9], 2) + 1.065 * pow(x[:,10], 2) + 0.7346 * pow(x[:,11], 2)
		+ 0.4896 * pow(x[:,12], 2) + 5.815 * pow(x[:,13], 2) - 0.1304
		* pow(x[:,14], 2) + 3.595 * pow(x[:,15], 2) + 0.6296
		* pow(x[:,16], 2) + 1.26 * pow(x[:,17], 2)
        )

    def doc_4(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        83.150002 + 12.02 * x[:,9] - 0.072 * x[:,10] - 0.0126 * x[:,11]
		+ 3.428 * x[:,12] - 0.704 * x[:,13] + 0.7248 * x[:,14] + 0.7224
		* x[:,15] - 0.1421 * x[:,16] + 0.0407 * x[:,17] + 0.613 * x[:,9]
		* x[:,10] + 0.046 * x[:,9] * x[:,11] + 1.155 * x[:,9] * x[:,12]
		+ 0.7144 * x[:,9] * x[:,13] + 0.0944 * x[:,9] * x[:,14] - 0.8399
		* x[:,9] * x[:,15] + 0.2251 * x[:,9] * x[:,16] + 0.0082 * x[:,9]
		* x[:,17] + 0.0082 * x[:,10] * x[:,11] + 0.1826 * x[:,10] * x[:,12]
		- 0.0606 * x[:,10] * x[:,13] - 0.2352 * x[:,10] * x[:,14] - 0.0486 * x[:,10]
		* x[:,15] + 0.0162 * x[:,10] * x[:,16] + 0.0211 * x[:,10] * x[:,17]
		+ 0.0199 * x[:,11] * x[:,12] + 0.0088 * x[:,11] * x[:,13] - 0.0152
		* x[:,11] * x[:,14] + 0.0082 * x[:,11] * x[:,15] - 0.0123 * x[:,11]
		* x[:,16] - 0.0142 * x[:,11] * x[:,17] + 0.1207 * x[:,12] * x[:,13]
		+ 0.6576 * x[:,12] * x[:,14] - 0.2728 * x[:,12] * x[:,15] + 0.0435
		* x[:,12] * x[:,16] - 0.0326 * x[:,12] * x[:,17] + 0.0196 * x[:,13]
		* x[:,14] + 0.0952 * x[:,13] * x[:,15] + 0.0351 * x[:,13] * x[:,16]
		+ 0.0091 * x[:,13] * x[:,17] - 0.0628 * x[:,14] * x[:,15] - 0.0201
		* x[:,14] * x[:,16] - 0.0244 * x[:,14] * x[:,17] + 0.0651 * x[:,15]
		* x[:,16] + 0.0048 * x[:,15] * x[:,17] - 0.0042 * x[:,16] * x[:,17]
		- 10.95 * pow(x[:,9], 2) - 0.2401 * pow(x[:,10], 2) - 0.2203
		* pow(x[:,11], 2) + 5.223 * pow(x[:,12], 2) - 0.2669
		* pow(x[:,13], 2) - 0.0884 * pow(x[:,14], 2) - 0.2169
		* pow(x[:,15], 2) - 0.306 * pow(x[:,16], 2) - 0.2231
		* pow(x[:,17], 2) 
        )

    def rough_4(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        2.191 - 0.0001 * x[:,9] + 0.1584 * x[:,10] - 0.0017 * x[:,11]
		+ 0.0238 * x[:,12] - 0.1632 * x[:,13] - 0.0008 * x[:,14] - 0.0666
		* x[:,15] - 0.0142 * x[:,16] + 0.0069 * x[:,17] - 0.0002 * x[:,9]
		* x[:,10] + 0.0001 * x[:,9] * x[:,11] + 0.0001 * x[:,9] * x[:,13]
		- 0.0002 * x[:,9] * x[:,14] + 0.0008 * x[:,9] * x[:,15] + 0.0005
		* x[:,9] * x[:,16] - 0.0007 * x[:,9] * x[:,17] - 0.0003 * x[:,10]
		* x[:,11] + 0.0006 * x[:,10] * x[:,12] - 0.012 * x[:,10] * x[:,13]
		- 0.0003 * x[:,10] * x[:,14] - 0.0051 * x[:,10] * x[:,15] - 0.0037 * x[:,10]
		* x[:,16] + 0.0007 * x[:,10] * x[:,17] - 0.0002 * x[:,11] * x[:,12]
		- 0.0003 * x[:,11] * x[:,13] - 0.0001 * x[:,11] * x[:,14] - 0.0004
		* x[:,11] * x[:,15] - 0.0002 * x[:,11] * x[:,16] - 0.0001 * x[:,11]
		* x[:,17] - 0.005 * x[:,12] * x[:,13] - 0.0037 * x[:,12] * x[:,14]
		+ 0.0022 * x[:,12] * x[:,15] - 0.0008 * x[:,12] * x[:,16] - 0.0003
		* x[:,12] * x[:,17] - 0.0008 * x[:,13] * x[:,14] - 0.0018 * x[:,13]
		* x[:,15] - 0.0012 * x[:,13] * x[:,16] - 0.0021 * x[:,13]
		* x[:,17] + 0.0009 * x[:,14] * x[:,15] - 0.0001 * x[:,14] * x[:,16]
		- 0.0002 * x[:,14] * x[:,17] - 0.0011 * x[:,15] * x[:,16] - 0.0012
		* x[:,15] * x[:,17] - 0.0008 * x[:,16] * x[:,17] - 0.0001
		* pow(x[:,9], 2) - 0.0302 * pow(x[:,10], 2) - 0.0015
		* pow(x[:,11], 2) + 0.0041 * pow(x[:,12], 2) + 0.005
		* pow(x[:,13], 2) + 0.0029 * pow(x[:,14], 2) + 0.0009
		* pow(x[:,15], 2) - 0.0007 * pow(x[:,16], 2) - 0.0058
		* pow(x[:,17], 2)
        )

    def wfuel_4(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        385.5 - 6.707 * x[:,9] - 30.57 * x[:,10] + 0.5048 * x[:,11]
		- 18.91 * x[:,12] + 33.009998 * x[:,13] - 7.543 * x[:,14]
		- 55.169998 * x[:,15] - 1.447 * x[:,16] - 15.32 * x[:,17] - 1.938
		* x[:,9] * x[:,10] + 0.18 * x[:,9] * x[:,11] - 1.201 * x[:,9] * x[:,12]
		+ 1.434 * x[:,9] * x[:,13] - 0.5591 * x[:,9] * x[:,14] + 0.272
		* x[:,9] * x[:,15] + 1.172 * x[:,9] * x[:,16] - 0.5566 * x[:,9]
		* x[:,17] - 0.0178 * x[:,10] * x[:,11] - 0.6466 * x[:,10] * x[:,12]
		+ 3.714 * x[:,10] * x[:,13] - 0.1913 * x[:,10] * x[:,14] + 0.3522 * x[:,10]
		* x[:,15] - 0.828 * x[:,10] * x[:,16] - 1.633 * x[:,10] * x[:,17] - 0.0415
		* x[:,11] * x[:,12] - 0.3772 * x[:,11] * x[:,13] - 0.0287 * x[:,11]
		* x[:,14] - 0.3237 * x[:,11] * x[:,15] - 0.2653 * x[:,11] * x[:,16]
		- 0.5519 * x[:,11] * x[:,17] + 0.5867 * x[:,12] * x[:,13] - 2.332
		* x[:,12] * x[:,14] + 0.0168 * x[:,12] * x[:,15] + 0.2074 * x[:,12]
		* x[:,16] - 0.1206 * x[:,12] * x[:,17] + 0.3465 * x[:,13] * x[:,14]
		- 1.509 * x[:,13] * x[:,15] - 2.134 * x[:,13] * x[:,16] + 0.7418
		* x[:,13] * x[:,17] - 0.0803 * x[:,14] * x[:,15] + 0.1825 * x[:,14]
		* x[:,16] - 0.1052 * x[:,14] * x[:,17] - 4.86 * x[:,15] * x[:,16]
		- 1.355 * x[:,15] * x[:,17] - 1.289 * x[:,16] * x[:,17] + 0.1451
		* pow(x[:,9], 2) - 1.698 * pow(x[:,10], 2) - 0.6064 * pow(x[:,11], 2)
		- 6.025 * pow(x[:,12], 2) - 4.978 * pow(x[:,13], 2) + 0.2611
		* pow(x[:,14], 2) - 3.229 * pow(x[:,15], 2) - 0.4999
		* pow(x[:,16], 2) - 1.068 * pow(x[:,17], 2)
        )

    def purch_4(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
            43730 + 142.5 * x[:,9] + 756.5 * x[:,10] + 2.004 * x[:,11]
		+ 504.799988 * x[:,12] - 314.799988 * x[:,13] + 194.100006 * x[:,14]
		+ 890.5 * x[:,15] + 114.099998 * x[:,16] + 334.5 * x[:,17]
		+ 43.029999 * x[:,9] * x[:,10] - 2.865 * x[:,9] * x[:,11] + 27.42
		* x[:,9] * x[:,12] - 35.060001 * x[:,9] * x[:,13] + 13.28 * x[:,9]
		* x[:,14] - 17.02 * x[:,9] * x[:,15] - 18.83 * x[:,9] * x[:,16] + 14.26
		* x[:,9] * x[:,17] + 0.8277 * x[:,10] * x[:,11] + 3.724 * x[:,10] * x[:,12]
		- 60.380001 * x[:,10] * x[:,13] + 3.039 * x[:,10] * x[:,14] + 7.395 * x[:,10]
		* x[:,15] + 18.809999 * x[:,10] * x[:,16] + 35.360001 * x[:,10] * x[:,17]
		- 0.0758 * x[:,11] * x[:,12] + 6.451 * x[:,11] * x[:,13] + 0.5043
		* x[:,11] * x[:,14] + 4.745 * x[:,11] * x[:,15] + 3.97 * x[:,11]
		* x[:,16] + 11.97 * x[:,11] * x[:,17] - 20.6 * x[:,12] * x[:,13]
		+ 20.549999 * x[:,12] * x[:,14] + 18.09 * x[:,12] * x[:,15] - 7.554
		* x[:,12] * x[:,16] + 5.429 * x[:,12] * x[:,17] - 8.093 * x[:,13]
		* x[:,14] + 2.433 * x[:,13] * x[:,15] + 29.82 * x[:,13] * x[:,16]
		- 17.860001 * x[:,13] * x[:,17] + 3.676 * x[:,14] * x[:,15] - 3.442
		* x[:,14] * x[:,16] + 2.995 * x[:,14] * x[:,17] + 86.110001 * x[:,15]
		* x[:,16] + 29.129999 * x[:,15] * x[:,17] + 26.49 * x[:,16]
		* x[:,17] - 1.456 * pow(x[:,9], 2) + 16.049999 * pow(x[:,10], 2)
		- 0.7055 * pow(x[:,11], 2) + 45.990002 * pow(x[:,12], 2) + 87.14
		* pow(x[:,13], 2) + 0.8945 * pow(x[:,14], 2) + 51.650002
		* pow(x[:,15], 2) - 2.256 * pow(x[:,16], 2) + 25.59
		* pow(x[:,17], 2)
        )

    def range_4(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
            1941 - 6.768 * x[:,9] - 68.910004 * x[:,10] + 2.315 * x[:,11]
		- 346.799988 * x[:,12] - 365 * x[:,13] - 65.25 * x[:,14] - 77.599998
		* x[:,15] + 13.19 * x[:,16] - 155.300003 * x[:,17] + 0.1113
		* x[:,9] * x[:,10] + 0.2207 * x[:,9] * x[:,11] + 1.229 * x[:,9]
		* x[:,12] + 3.721 * x[:,9] * x[:,13] + 0.0176 * x[:,9] * x[:,14]
		- 0.0527 * x[:,9] * x[:,15] + 0.6777 * x[:,9] * x[:,16] + 2.674
		* x[:,9] * x[:,17] + 0.0762 * x[:,10] * x[:,11] - 24.07 * x[:,10] * x[:,12]
		- 31.309999 * x[:,10] * x[:,13] - 1.979 * x[:,10] * x[:,14] + 11.8 * x[:,10]
		* x[:,15] - 0.623 * x[:,10] * x[:,16] - 38.310001 * x[:,10] * x[:,17]
		- 1.15 * x[:,11] * x[:,12] - 0.9785 * x[:,11] * x[:,13] - 0.2051
		* x[:,11] * x[:,14] - 0.5879 * x[:,11] * x[:,15] - 0.4512 * x[:,11]
		* x[:,16] - 0.7754 * x[:,11] * x[:,17] + 115.400002 * x[:,12]
		* x[:,13] - 26.84 * x[:,12] * x[:,14] + 27.91 * x[:,12] * x[:,15]
		- 7.897 * x[:,12] * x[:,16] + 34.349998 * x[:,12] * x[:,17]
		+ 20.610001 * x[:,13] * x[:,14] + 28.92 * x[:,13] * x[:,15] - 3.006
		* x[:,13] * x[:,16] - 27.870001 * x[:,13] * x[:,17] + 4.107 * x[:,14]
		* x[:,15] - 1.17 * x[:,14] * x[:,16] + 7.186 * x[:,14] * x[:,17] - 7.389
		* x[:,15] * x[:,16] + 18.02 * x[:,15] * x[:,17] - 2.451 * x[:,16]
		* x[:,17] - 12.83 * pow(x[:,9], 2) + 4.667 * pow(x[:,10], 2) - 12.33
		* pow(x[:,11], 2) - 85.330002 * pow(x[:,12], 2) + 57.169998
		* pow(x[:,13], 2) - 8.833 * pow(x[:,14], 2) - 16.33
		* pow(x[:,15], 2) - 13.33 * pow(x[:,16], 2) + 8.167
		* pow(x[:,17], 2)
        )

    def ldmax_4(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
            17.43 + 0.4811 * x[:,9] + 1.584 * x[:,10] + 0.0212 * x[:,11]
		- 0.0128 * x[:,12] - 0.5456 * x[:,13] - 0.0054 * x[:,14] - 0.4984
		* x[:,15] + 0.1607 * x[:,16] + 0.0288 * x[:,17] + 0.0635 * x[:,9]
		* x[:,10] + 0.0013 * x[:,9] * x[:,11] - 0.0078 * x[:,9] * x[:,12]
		- 0.0123 * x[:,9] * x[:,13] - 0.0011 * x[:,9] * x[:,14] - 0.0065
		* x[:,9] * x[:,15] + 0.0049 * x[:,9] * x[:,16] + 0.0038 * x[:,9]
		* x[:,17] + 0.0038 * x[:,10] * x[:,11] - 0.0014 * x[:,10] * x[:,12]
		- 0.0601 * x[:,10] * x[:,13] - 0.0003 * x[:,10] * x[:,14] - 0.0503 * x[:,10]
		* x[:,15] + 0.017 * x[:,10] * x[:,16] - 0.0029 * x[:,10] * x[:,17]
		+ 0.0002 * x[:,11] * x[:,12] - 0.0032 * x[:,11] * x[:,13] + 0.0001
		* x[:,11] * x[:,14] - 0.0043 * x[:,11] * x[:,15] - 0.0021 * x[:,11]
		* x[:,16] - 0.0001 * x[:,11] * x[:,17] - 0.0007 * x[:,12] * x[:,13]
		- 0.0015 * x[:,12] * x[:,14] + 0.0009 * x[:,12] * x[:,15] - 0.0002
		* x[:,12] * x[:,16] + 0.0008 * x[:,12] * x[:,17] - 0.0008 * x[:,13]
		* x[:,14] - 0.0682 * x[:,13] * x[:,15] - 0.0243 * x[:,13] * x[:,16]
		- 0.0047 * x[:,13] * x[:,17] + 0 * x[:,14] * x[:,15] - 0.0004 * x[:,14]
		* x[:,16] + 0.0007 * x[:,14] * x[:,17] - 0.0406 * x[:,15] * x[:,16]
		- 0.0052 * x[:,15] * x[:,17] - 0.0018 * x[:,16] * x[:,17] - 0.0923
		* pow(x[:,9], 2) - 0.1312 * pow(x[:,10], 2) - 0.0127
		* pow(x[:,11], 2) - 0.0041 * pow(x[:,12], 2) + 0.0304
		* pow(x[:,13], 2) - 0.0044 * pow(x[:,14], 2) - 0.0545
		* pow(x[:,15], 2) - 0.0239 * pow(x[:,16], 2) + 0.0066
		* pow(x[:,17], 2)
        )

    def vcmax_4(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
            197.800003 - 0.3562 * x[:,9] + 0.7729 * x[:,10] + 0.1807
		* x[:,11] + 1.886 * x[:,12] + 5.332 * x[:,13] - 0.1262 * x[:,14]
		- 3.585 * x[:,15] + 1.18 * x[:,16] + 0.2323 * x[:,17] + 0.0123
		* x[:,9] * x[:,10] + 0.0073 * x[:,9] * x[:,11] - 0.0462 * x[:,9]
		* x[:,12] + 0.0075 * x[:,9] * x[:,13] - 0.003 * x[:,9] * x[:,14]
		+ 0.0375 * x[:,9] * x[:,15] + 0.0225 * x[:,9] * x[:,16] + 0.017
		* x[:,9] * x[:,17] + 0.0071 * x[:,10] * x[:,11] - 0.0914 * x[:,10]
		* x[:,12] + 0.1716 * x[:,10] * x[:,13] - 0.0115 * x[:,10] * x[:,14] + 0.0278
		* x[:,10] * x[:,15] - 0.0074 * x[:,10] * x[:,16] - 0.0583 * x[:,10] * x[:,17]
		- 0.0124 * x[:,11] * x[:,12] - 0.0116 * x[:,11] * x[:,13] - 0.0041
		* x[:,11] * x[:,14] - 0.0337 * x[:,11] * x[:,15] - 0.0132 * x[:,11]
		* x[:,16] + 0.0028 * x[:,11] * x[:,17] - 0.2578 * x[:,12] * x[:,13]
		- 0.3341 * x[:,12] * x[:,14] + 0.2057 * x[:,12] * x[:,15] - 0.0696
		* x[:,12] * x[:,16] - 0.0125 * x[:,12] * x[:,17] - 0.1043 * x[:,13]
		* x[:,14] - 0.6842 * x[:,13] * x[:,15] - 0.1276 * x[:,13] * x[:,16]
		- 0.0217 * x[:,13] * x[:,17] + 0.0663 * x[:,14] * x[:,15] - 0.0185
		* x[:,14] * x[:,16] + 0.0001 * x[:,14] * x[:,17] - 0.2758 * x[:,15]
		* x[:,16] - 0.039 * x[:,15] * x[:,17] - 0.0162 * x[:,16] * x[:,17]
		- 0.1479 * pow(x[:,9], 2) - 0.1989 * pow(x[:,10], 2) - 0.2574
		* pow(x[:,11], 2) + 0.3991 * pow(x[:,12], 2) - 0.5109
		* pow(x[:,13], 2) + 0.0706 * pow(x[:,14], 2) - 0.4174
		* pow(x[:,15], 2) - 0.2574 * pow(x[:,16], 2) - 0.0384
		* pow(x[:,17], 2)
        )

    def noise_6(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
        74.099998 - 0.0004 * x[:,18] - 0.0156 * x[:,19] + 0.0003
	    * x[:,20] + 0.9682 * x[:,21] + 0.0314 * x[:,22] - 0.0053 * x[:,23]
		- 0.0015 * x[:,24] - 0.0004 * x[:,25] + 0.0007 * x[:,26] + 0
		* x[:,18] * x[:,19] + 0 * x[:,18] * x[:,20] - 0.0001 * x[:,18] * x[:,21]
		- 0.0001 * x[:,18] * x[:,22] - 0.0001 * x[:,18] * x[:,23] + 0 * x[:,19]
		* x[:,20] - 0.0013 * x[:,19] * x[:,21] - 0.0014 * x[:,19] * x[:,22]
		- 0.0002 * x[:,19] * x[:,23] - 0.0001 * x[:,19] * x[:,24] + 0.0002 * x[:,19]
		* x[:,25] + 0.0002 * x[:,19] * x[:,26] + 0 * x[:,20] * x[:,26] - 0.0003
		* x[:,21] * x[:,22] + 0.0057 * x[:,21] * x[:,23] - 0.0002 * x[:,21]
		* x[:,24] + 0.0001 * x[:,21] * x[:,25] - 0.0001 * x[:,21] * x[:,26]
		+ 0.0002 * x[:,22] * x[:,23] - 0.0002 * x[:,22] * x[:,24] - 0.0002
		* x[:,22] * x[:,25] + 0.0001 * x[:,22] * x[:,26] - 0.0001 * x[:,23]
		* x[:,24] + 0 * x[:,23] * x[:,25] + 0 * x[:,23] * x[:,26] - 0.0001
		* x[:,24] * x[:,25] + 0.0006 * pow(x[:,18], 2) + 0.0017
		* pow(x[:,19], 2) + 0.0011 * pow(x[:,20], 2) + 0.1103
		* pow(x[:,21], 2) + 0.0004 * pow(x[:,22], 2) - 0.0021
		* pow(x[:,23], 2) + 0.001 * pow(x[:,24], 2) + 0.001 * pow(x[:,25], 2)
		+ 0.0008 * pow(x[:,26], 2)
        )

    def wemp_6(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
           1972 + 5.386 * x[:,18] + 33.290001 * x[:,19] - 0.0222 * x[:,20]
		+ 10.82 * x[:,21] - 28.889999 * x[:,22] + 5.588 * x[:,23] + 61.32
		* x[:,24] + 4.65 * x[:,25] + 16.620001 * x[:,26] + 1.32 * x[:,18]
		* x[:,19] - 0.2549 * x[:,18] * x[:,20] + 0.9089 * x[:,18] * x[:,21]
		- 1.403 * x[:,18] * x[:,22] + 0.3601 * x[:,18] * x[:,23] - 0.0118
		* x[:,18] * x[:,24] + 0.0123 * x[:,18] * x[:,25] + 0.4761 * x[:,18]
		* x[:,26] + 0.0455 * x[:,19] * x[:,20] - 0.0596 * x[:,19] * x[:,21]
		- 3.818 * x[:,19] * x[:,22] - 0.0408 * x[:,19] * x[:,23] + 0.5044 * x[:,19]
		* x[:,24] + 0.2867 * x[:,19] * x[:,25] + 2.497 * x[:,19] * x[:,26]
		+ 0.0011 * x[:,20] * x[:,21] + 0.151 * x[:,20] * x[:,22] + 0.0524
		* x[:,20] * x[:,23] + 0.1164 * x[:,20] * x[:,24] + 0.0491 * x[:,20]
		* x[:,25] + 0.391 * x[:,20] * x[:,26] + 0.1823 * x[:,21] * x[:,22]
		+ 1.254 * x[:,21] * x[:,23] + 0.0491 * x[:,21] * x[:,24] + 0.1011
		* x[:,21] * x[:,25] + 0.0538 * x[:,21] * x[:,26] + 0.098 * x[:,22]
		* x[:,23] + 1.807 * x[:,22] * x[:,24] + 1.119 * x[:,22] * x[:,25]
		- 1.437 * x[:,22] * x[:,26] + 0.0664 * x[:,23] * x[:,24] + 0.0566
		* x[:,23] * x[:,25] + 0.0508 * x[:,23] * x[:,26] + 2.914 * x[:,24]
		* x[:,25] + 0.0969 * x[:,24] * x[:,26] - 0.1483 * x[:,25] * x[:,26]
		+ 1.446 * pow(x[:,18], 2) - 2.524 * pow(x[:,19], 2) + 1.646
		* pow(x[:,20], 2) + 1.381 * pow(x[:,21], 2) + 2.651
		* pow(x[:,22], 2) + 0.7561 * pow(x[:,23], 2) + 3.116
		* pow(x[:,24], 2) + 1.211 * pow(x[:,25], 2) - 0.4489
		* pow(x[:,26], 2) 
        )

    def doc_6(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
            83.260002 + 11.86 * x[:,18] - 0.0805 * x[:,19] - 0.0218
		* x[:,20] + 3.345 * x[:,21] - 0.6443 * x[:,22] + 0.7039 * x[:,23]
		+ 0.8256 * x[:,24] - 0.0905 * x[:,25] + 0.0305 * x[:,26] + 0.6
		* x[:,18] * x[:,19] + 0.0284 * x[:,18] * x[:,20] + 1.109 * x[:,18]
		* x[:,21] + 0.6928 * x[:,18] * x[:,22] + 0.0795 * x[:,18] * x[:,23]
		- 0.9926 * x[:,18] * x[:,24] + 0.1119 * x[:,18] * x[:,25] - 0.0183
		* x[:,18] * x[:,26] + 0.0018 * x[:,19] * x[:,20] + 0.1731 * x[:,19]
		* x[:,21] - 0.0631 * x[:,19] * x[:,22] - 0.246 * x[:,19] * x[:,23] - 0.0581
		* x[:,19] * x[:,24] - 0.0008 * x[:,19] * x[:,25] - 0.0048 * x[:,19] * x[:,26]
		+ 0.0024 * x[:,20] * x[:,21] + 0.0011 * x[:,20] * x[:,22] - 0.0043
		* x[:,20] * x[:,23] + 0.0048 * x[:,20] * x[:,24] - 0.0019 * x[:,20]
		* x[:,25] + 0.0025 * x[:,20] * x[:,26] + 0.1331 * x[:,21] * x[:,22]
		+ 0.6487 * x[:,21] * x[:,23] - 0.353 * x[:,21] * x[:,24] - 0.0025
		* x[:,21] * x[:,25] - 0.0586 * x[:,21] * x[:,26] + 0.005 * x[:,22]
		* x[:,23] + 0.1121 * x[:,22] * x[:,24] + 0.0345 * x[:,22] * x[:,25]
		+ 0.0103 * x[:,22] * x[:,26] - 0.0674 * x[:,23] * x[:,24] + 0.0003
		* x[:,23] * x[:,25] - 0.0133 * x[:,23] * x[:,26] + 0.0316 * x[:,24]
		* x[:,25] - 0.0007 * x[:,24] * x[:,26] + 0.0063 * x[:,25] * x[:,26]
		- 10.77 * pow(x[:,18], 2) - 0.288 * pow(x[:,19], 2) - 0.288
		* pow(x[:,20], 2) + 5.182 * pow(x[:,21], 2) - 0.2313
		* pow(x[:,22], 2) - 0.0895 * pow(x[:,23], 2) - 0.2136
		* pow(x[:,24], 2) - 0.2562 * pow(x[:,25], 2) - 0.2686
		* pow(x[:,26], 2)
        )

    def rough_6(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
            2.161 + 0.0007 * x[:,18] + 0.156 * x[:,19] - 0.0022 * x[:,20]
		+ 0.0239 * x[:,21] - 0.1649 * x[:,22] - 0.0007 * x[:,23] - 0.0675
		* x[:,24] - 0.0135 * x[:,25] + 0.0056 * x[:,26] + 0.0006 * x[:,18]
		* x[:,19] + 0.0001 * x[:,18] * x[:,20] + 0.0001 * x[:,18] * x[:,21] + 0
		* x[:,18] * x[:,24] - 0.0003 * x[:,18] * x[:,25] - 0.0003 * x[:,18]
		* x[:,26] - 0.0003 * x[:,19] * x[:,20] + 0.0008 * x[:,19] * x[:,21]
		- 0.0119 * x[:,19] * x[:,22] + 0 * x[:,19] * x[:,23] - 0.0054 * x[:,19]
		* x[:,24] - 0.0026 * x[:,19] * x[:,25] - 0.0003 * x[:,19] * x[:,26]
		- 0.0002 * x[:,20] * x[:,21] - 0.0002 * x[:,20] * x[:,22] - 0.0001
		* x[:,20] * x[:,23] - 0.0002 * x[:,20] * x[:,24] - 0.0002 * x[:,20]
		* x[:,25] + 0.0001 * x[:,20] * x[:,26] - 0.0051 * x[:,21] * x[:,22]
		- 0.0038 * x[:,21] * x[:,23] + 0.0024 * x[:,21] * x[:,24] - 0.0009
		* x[:,21] * x[:,25] - 0.0002 * x[:,21] * x[:,26] - 0.001 * x[:,22]
		* x[:,23] - 0.0018 * x[:,22] * x[:,24] - 0.0009 * x[:,22] * x[:,25]
		- 0.0017 * x[:,22] * x[:,26] + 0.0009 * x[:,23] * x[:,24] - 0.0002
		* x[:,23] * x[:,25] - 0.0001 * x[:,23] * x[:,26] - 0.0005 * x[:,24]
		* x[:,25] - 0.0001 * x[:,24] * x[:,26] + 0.0003 * x[:,25] * x[:,26]
		- 0.0006 * pow(x[:,18], 2) - 0.0255 * pow(x[:,19], 2) - 0.0043
		* pow(x[:,20], 2) + 0.0039 * pow(x[:,21], 2) + 0.0092
		* pow(x[:,22], 2) + 0.0014 * pow(x[:,23], 2) + 0.0003
		* pow(x[:,24], 2) - 0.0007 * pow(x[:,25], 2) - 0.0052
		* pow(x[:,26], 2)
        )

    def wfuel_6(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
            359.700012 - 5.78 * x[:,18] - 29.99 * x[:,19] + 0.0682
		* x[:,20] - 18.709999 * x[:,21] + 31 * x[:,22] - 7.332 * x[:,23]
		- 63.599998 * x[:,24] - 4.44 * x[:,25] - 16.709999 * x[:,26]
		- 1.317 * x[:,18] * x[:,19] + 0.2715 * x[:,18] * x[:,20] - 1.069 * x[:,18]
		* x[:,21] + 1.185 * x[:,18] * x[:,22] - 0.3931 * x[:,18] * x[:,23]
		- 0.8091 * x[:,18] * x[:,24] + 0.0543 * x[:,18] * x[:,25] - 0.448
		* x[:,18] * x[:,26] - 0.0418 * x[:,19] * x[:,20] - 0.5179 * x[:,19]
		* x[:,21] + 4.113 * x[:,19] * x[:,22] - 0.0407 * x[:,19] * x[:,23] - 0.3222
		* x[:,19] * x[:,24] - 0.3059 * x[:,19] * x[:,25] - 2.526 * x[:,19] * x[:,26]
		- 0.0077 * x[:,20] * x[:,21] - 0.1761 * x[:,20] * x[:,22] - 0.0543
		* x[:,20] * x[:,23] - 0.1319 * x[:,20] * x[:,24] - 0.0633 * x[:,20]
		* x[:,25] - 0.3999 * x[:,20] * x[:,26] + 0.2791 * x[:,21] * x[:,22]
		- 2.456 * x[:,21] * x[:,23] - 0.1367 * x[:,21] * x[:,24] - 0.155
		* x[:,21] * x[:,25] - 0.0405 * x[:,21] * x[:,26] + 0.0736 * x[:,22]
		* x[:,23] - 1.298 * x[:,22] * x[:,24] - 1.258 * x[:,22] * x[:,25]
		+ 1.389 * x[:,22] * x[:,26] - 0.1952 * x[:,23] * x[:,24] - 0.0011
		* x[:,23] * x[:,25] - 0.0391 * x[:,23] * x[:,26] - 2.965 * x[:,24]
		* x[:,25] - 0.1262 * x[:,24] * x[:,26] + 0.1433 * x[:,25] * x[:,26]
		- 0.8956 * pow(x[:,18], 2) + 1.729 * pow(x[:,19], 2) - 1.613
		* pow(x[:,20], 2) - 6.309 * pow(x[:,21], 2) - 1.949
		* pow(x[:,22], 2) - 0.6581 * pow(x[:,23], 2) - 2.723
		* pow(x[:,24], 2) - 1.181 * pow(x[:,25], 2) + 0.5244
		* pow(x[:,26], 2)
        )

    def purch_6(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
            44220 + 124.400002 * x[:,18] + 749.200012 * x[:,19] + 9.32
		* x[:,20] + 506.899994 * x[:,21] - 290.600006 * x[:,22]
		+ 192.800003 * x[:,23] + 1034 * x[:,24] + 144.5 * x[:,25]
		+ 364.799988 * x[:,26] + 27.370001 * x[:,18] * x[:,19] - 5.918 * x[:,18]
		* x[:,20] + 24.51 * x[:,18] * x[:,21] - 29.690001 * x[:,18] * x[:,22]
		+ 10.97 * x[:,18] * x[:,23] + 2.652 * x[:,18] * x[:,24] + 2.262 * x[:,18]
		* x[:,25] + 11.44 * x[:,18] * x[:,26] + 2.046 * x[:,19] * x[:,20]
		+ 0.2715 * x[:,19] * x[:,21] - 67.760002 * x[:,19] * x[:,22] - 0.3336
		* x[:,19] * x[:,23] + 25.07 * x[:,19] * x[:,24] + 7.377 * x[:,19] * x[:,25]
		+ 56.369999 * x[:,19] * x[:,26] - 0.1879 * x[:,20] * x[:,21] + 1.594
		* x[:,20] * x[:,22] + 1.086 * x[:,20] * x[:,23] + 1.375 * x[:,20]
		* x[:,24] + 0.4313 * x[:,20] * x[:,25] + 8.3 * x[:,20] * x[:,26]
		- 11.7 * x[:,21] * x[:,22] + 24.440001 * x[:,21] * x[:,23] + 23.85
		* x[:,21] * x[:,24] + 1.195 * x[:,21] * x[:,25] + 3.797 * x[:,21]
		* x[:,26] - 3.027 * x[:,22] * x[:,23] + 4.452 * x[:,22] * x[:,24]
		+ 18.059999 * x[:,22] * x[:,25] - 30.709999 * x[:,22] * x[:,26]
		+ 7.103 * x[:,23] * x[:,24] + 1.01 * x[:,23] * x[:,25] + 1.463 * x[:,23]
		* x[:,26] + 53.560001 * x[:,24] * x[:,25] + 4.063 * x[:,24]
		* x[:,26] - 3.483 * x[:,25] * x[:,26] + 15.24 * pow(x[:,18], 2)
		- 71.910004 * pow(x[:,19], 2) + 33.990002 * pow(x[:,20], 2)
		+ 60.540001 * pow(x[:,21], 2) + 15.24 * pow(x[:,22], 2) + 28.49
		* pow(x[:,23], 2) + 50.490002 * pow(x[:,24], 2) + 18.190001
		* pow(x[:,25], 2) - 9.011 * pow(x[:,26], 2)
        )

    def range_6(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
            1932 - 6.389 * x[:,18] - 64.400002 * x[:,19] + 1.778 * x[:,20]
		- 342.799988 * x[:,21] - 356.399994 * x[:,22] - 64.639999 * x[:,23]
		- 91.739998 * x[:,24] + 5.658 * x[:,25] - 150.5 * x[:,26] - 0.2363
		* x[:,18] * x[:,19] + 0.0566 * x[:,18] * x[:,20] + 1.346 * x[:,18]
		* x[:,21] + 3.529 * x[:,18] * x[:,22] + 0.0215 * x[:,18] * x[:,23]
		- 0.291 * x[:,18] * x[:,24] + 0.0293 * x[:,18] * x[:,25] + 2.416
		* x[:,18] * x[:,26] + 0.0449 * x[:,19] * x[:,20] - 24.780001 * x[:,19]
		* x[:,21] - 35.150002 * x[:,19] * x[:,22] - 2.131 * x[:,19] * x[:,23]
		+ 13.65 * x[:,19] * x[:,24] - 0.0059 * x[:,19] * x[:,25] - 35.599998
		* x[:,19] * x[:,26] - 0.9512 * x[:,20] * x[:,21] - 0.6113 * x[:,20]
		* x[:,22] - 0.166 * x[:,20] * x[:,23] - 0.3457 * x[:,20] * x[:,24]
		- 0.1191 * x[:,20] * x[:,25] - 0.5762 * x[:,20] * x[:,26]
		+ 114.300003 * x[:,21] * x[:,22] - 26.83 * x[:,21] * x[:,23]
		+ 33.290001 * x[:,21] * x[:,24] - 4.76 * x[:,21] * x[:,25]
		+ 33.189999 * x[:,21] * x[:,26] + 20.6 * x[:,22] * x[:,23]
		+ 38.630001 * x[:,22] * x[:,24] + 0.4238 * x[:,22] * x[:,25]
		- 31.940001 * x[:,22] * x[:,26] + 5.057 * x[:,23] * x[:,24] - 0.8809
		* x[:,23] * x[:,25] + 6.873 * x[:,23] * x[:,26] - 3.928 * x[:,24] * x[:,25]
		+ 20.360001 * x[:,24] * x[:,26] + 0.0918 * x[:,25] * x[:,26]
		- 11.73 * pow(x[:,18], 2) + 4.268 * pow(x[:,19], 2) - 10.73
		* pow(x[:,20], 2) - 83.730003 * pow(x[:,21], 2) + 39.27
		* pow(x[:,22], 2) - 7.232 * pow(x[:,23], 2) - 14.23
		* pow(x[:,24], 2) - 11.73 * pow(x[:,25], 2) + 9.268
		* pow(x[:,26], 2)
        )

    def ldmax_6(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
            17.34 + 0.4817 * x[:,18] + 1.573 * x[:,19] + 0.0179 * x[:,20]
		- 0.0117 * x[:,21] - 0.5804 * x[:,22] - 0.0053 * x[:,23] - 0.5638
		* x[:,24] + 0.0986 * x[:,25] + 0.0278 * x[:,26] + 0.0603 * x[:,18]
		* x[:,19] + 0.0004 * x[:,18] * x[:,20] - 0.0074 * x[:,18] * x[:,21]
		- 0.0132 * x[:,18] * x[:,22] - 0.0014 * x[:,18] * x[:,23] - 0.0059
		* x[:,18] * x[:,24] + 0.0053 * x[:,18] * x[:,25] + 0.0024 * x[:,18]
		* x[:,26] + 0.0032 * x[:,19] * x[:,20] - 0.0021 * x[:,19] * x[:,21]
		- 0.0632 * x[:,19] * x[:,22] - 0.0009 * x[:,19] * x[:,23] - 0.0553 * x[:,19]
		* x[:,24] + 0.0072 * x[:,19] * x[:,25] + 0.0011 * x[:,19] * x[:,26]
		+ 0.0002 * x[:,20] * x[:,21] - 0.0028 * x[:,20] * x[:,22] + 0
		* x[:,20] * x[:,23] - 0.0035 * x[:,20] * x[:,24] - 0.0016 * x[:,20]
		* x[:,25] - 0.0006 * x[:,20] * x[:,26] + 0.0003 * x[:,21] * x[:,22]
		- 0.0006 * x[:,21] * x[:,23] + 0.0014 * x[:,21] * x[:,24] + 0.001
		* x[:,21] * x[:,25] + 0.0004 * x[:,21] * x[:,26] - 0.0001 * x[:,22]
		* x[:,23] - 0.0563 * x[:,22] * x[:,24] - 0.0173 * x[:,22] * x[:,25]
		- 0.0049 * x[:,22] * x[:,26] + 0.0006 * x[:,23] * x[:,24] + 0.0004
		* x[:,23] * x[:,25] + 0.0001 * x[:,23] * x[:,26] - 0.0254 * x[:,24]
		* x[:,25] - 0.0069 * x[:,24] * x[:,26] - 0.0037 * x[:,25] * x[:,26]
		- 0.0838 * pow(x[:,18], 2) - 0.1499 * pow(x[:,19], 2) - 0.0058
		* pow(x[:,20], 2) + 0.0016 * pow(x[:,21], 2) + 0.0214
		* pow(x[:,22], 2) + 0.0013 * pow(x[:,23], 2) - 0.0337
		* pow(x[:,24], 2) - 0.0133 * pow(x[:,25], 2) + 0.0028
		* pow(x[:,26], 2)
        )

    def vcmax_6(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        return (
            197.100006 - 0.3331 * x[:,18] + 0.7564 * x[:,19] + 0.153
		* x[:,20] + 1.918 * x[:,21] + 5.044 * x[:,22] - 0.1139 * x[:,23]
		- 4.07 * x[:,24] + 0.7276 * x[:,25] + 0.2242 * x[:,26] - 0.0231
		* x[:,18] * x[:,19] - 0.009 * x[:,18] * x[:,20] - 0.0365 * x[:,18]
		* x[:,21] - 0.0011 * x[:,18] * x[:,22] - 0.0064 * x[:,18] * x[:,23]
		+ 0.0355 * x[:,18] * x[:,24] + 0.028 * x[:,18] * x[:,25] + 0.013
		* x[:,18] * x[:,26] + 0.013 * x[:,19] * x[:,20] - 0.0947 * x[:,19] * x[:,21]
		+ 0.1791 * x[:,19] * x[:,22] - 0.0178 * x[:,19] * x[:,23] + 0.0598 * x[:,19]
		* x[:,24] - 0.0216 * x[:,19] * x[:,25] - 0.0274 * x[:,19] * x[:,26]
		- 0.006 * x[:,20] * x[:,21] - 0.0243 * x[:,20] * x[:,22] - 0.001
		* x[:,20] * x[:,23] - 0.0233 * x[:,20] * x[:,24] - 0.0108 * x[:,20]
		* x[:,25] - 0.0011 * x[:,20] * x[:,26] - 0.2368 * x[:,21] * x[:,22]
		- 0.334 * x[:,21] * x[:,23] + 0.2329 * x[:,21] * x[:,24] - 0.0419
		* x[:,21] * x[:,25] - 0.0148 * x[:,21] * x[:,26] - 0.0851 * x[:,22]
		* x[:,23] - 0.6179 * x[:,22] * x[:,24] - 0.0865 * x[:,22] * x[:,25]
		- 0.0158 * x[:,22] * x[:,26] + 0.0691 * x[:,23] * x[:,24] - 0.0088
		* x[:,23] * x[:,25] - 0.0056 * x[:,23] * x[:,26] - 0.1699 * x[:,24]
		* x[:,25] - 0.0462 * x[:,24] * x[:,26] - 0.028 * x[:,25] * x[:,26]
		- 0.2315 * pow(x[:,18], 2) - 0.384 * pow(x[:,19], 2) - 0.013
		* pow(x[:,20], 2) + 0.315 * pow(x[:,21], 2) - 0.612
		* pow(x[:,22], 2) + 0.206 * pow(x[:,23], 2) - 0.2825
		* pow(x[:,24], 2) - 0.1225 * pow(x[:,25], 2) - 0.013
		* pow(x[:,26], 2)
        )
    
    # Objective functions
    def f_1(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        x = scaled_variables(x)
        a = np.array([[noise_2(x)],[noise_4(x)],[noise_6(x)]])
        return(
            np.amax(a, axis=0)
        )
    
    def f_2(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        x = scaled_variables(x)
        a = np.array([[wemp_2(x)],[wemp_4(x)],[wemp_6(x)]])
        return(
            np.amax(a, axis=0)
        )

    def f_3(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        x = scaled_variables(x)
        a = np.array([[doc_2(x)],[doc_4(x)],[doc_6(x)]])
        return(
            np.amax(a, axis=0)
        )

    def f_4(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        x = scaled_variables(x)
        a = np.array([[rough_2(x)],[rough_4(x)],[rough_6(x)]])
        return(
            np.amax(a, axis=0)
        )

    def f_5(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        x = scaled_variables(x)
        a = np.array([[wfuel_2(x)],[wfuel_4(x)],[wfuel_6(x)]])
        return(
            np.amax(a, axis=0)
        )

    def f_6(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        x = scaled_variables(x)
        a = np.array([[purch_2(x)],[purch_4(x)],[purch_6(x)]])
        return(
            np.amax(a, axis=0)
        )

    def f_7(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        x = scaled_variables(x)
        a = np.array([[range_2(x)],[range_4(x)],[range_6(x)]])
        return -(
            np.amin(a, axis=0)
        )

    def f_8(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        x = scaled_variables(x)
        a = np.array([[ldmax_2(x)],[ldmax_4(x)],[ldmax_6(x)]])
        return -(
            np.amin(a, axis=0)
        )

    def f_9(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        x = scaled_variables(x)
        a = np.array([[vcmax_2(x)],[vcmax_4(x)],[vcmax_6(x)]])
        return -(
            np.amin(a, axis=0)
        )

    def f_10(x: np.ndarray) -> np.ndarray:
        x = np.atleast_2d(x)
        x = scaled_variables(x)
        cspd = (x[:,0] + x[:,9] + x[:,18]) / 3.0
        ar = (x[:,1] + x[:,10] + x[:,19]) / 3.0
        sweep = (x[:,2] + x[:,11] + x[:,20]) / 3.0
        dprop = (x[:,3] + x[:,12] + x[:,21]) / 3.0
        wingld = (x[:,4] + x[:,13] + x[:,22]) / 3.0
        af = (x[:,5] + x[:,14] + x[:,23]) / 3.0
        seatw = (x[:,6] + x[:,15] + x[:,24]) / 3.0
        elodt = (x[:,7] + x[:,16] + x[:,25]) / 3.0
        taper = (x[:,8] + x[:,17] + x[:,26]) / 3.0
        return(
            pow(
		    pow(x[:,0] - cspd, 2) +
		    pow(x[:,9] - cspd, 2) +
		    pow(x[:,18] - cspd, 2) +
		    pow(x[:,1] - ar, 2) + 
		    pow(x[:,10] - ar, 2) +
		    pow(x[:,19] - ar, 2) +
		    pow(x[:,2] - sweep, 2) +
		    pow(x[:,11] - sweep, 2) +
		    pow(x[:,20] - sweep, 2) +
		    pow(x[:,3] - dprop, 2) +
		    pow(x[:,12] - dprop, 2) +
		    pow(x[:,21] - dprop, 2) +
		    pow(x[:,4] - wingld, 2) +
		    pow(x[:,13] - wingld, 2) +
		    pow(x[:,22] - wingld, 2) +
		    pow(x[:,5] - af, 2) +
		    pow(x[:,14] - af, 2) +
		    pow(x[:,23] - af, 2) +
		    pow(x[:,6] - seatw, 2) +
		    pow(x[:,15] - seatw, 2) +
		    pow(x[:,24] - seatw, 2) +
		    pow(x[:,7] - elodt, 2) +
		    pow(x[:,16] - elodt, 2) +
	 	    pow(x[:,25] - elodt, 2) +
		    pow(x[:,8] - taper, 2) +
		    pow(x[:,17] - taper, 2) +
		    pow(x[:,26] - taper, 2), 
            0.5)
        )

    # Constraints
    def g_1(x: np.ndarray, _ = None) -> np.ndarray:
        x = np.atleast_2d(x)
        x = scaled_variables(x)
        NOISE2_CV = (noise_2(x)- 75) / 75
        WEMP2_CV = (wemp_2(x) - 2200) / 2200
        DOC2_CV = (doc_2(x) - 80) / 80
        ROUGH2_CV = (rough_2(x) - 2) / 2
        WFUEL2_CV = (wfuel_2(x) - 450) / 450
        RANGE2_CV = -(range_2(x) - 2000) / 2000

        NOISE4_CV = (noise_4(x) - 75) / 75
        WEMP4_CV = (wemp_4(x) - 2200) / 2200
        DOC4_CV = (doc_4(x) - 80) / 80
        ROUGH4_CV = (rough_4(x) - 2) / 2
        WFUEL4_CV = (wfuel_4(x) - 475) / 475
        RANGE4_CV = -(range_4(x) - 2000) / 2000

        NOISE6_CV = (noise_6(x) - 75) / 75
        WEMP6_CV = (wemp_6(x) - 2200) / 2200
        DOC6_CV = (doc_6(x) - 80) / 80
        ROUGH6_CV = (rough_6(x) - 2) / 2
        WFUEL6_CV = (wfuel_6(x) - 500) / 500
        RANGE6_CV = -(range_6(x) - 2000) / 2000

        NOISE2_CV = np.where(NOISE2_CV > 0, NOISE2_CV, 0)
        WEMP2_CV = np.where(WEMP2_CV > 0, WEMP2_CV, 0)
        DOC2_CV = np.where(DOC2_CV > 0, DOC2_CV, 0)
        ROUGH2_CV = np.where(ROUGH2_CV > 0, ROUGH2_CV, 0)
        WFUEL2_CV = np.where(WFUEL2_CV > 0, WFUEL2_CV, 0)
        RANGE2_CV = np.where(RANGE2_CV > 0, RANGE2_CV, 0)

        NOISE4_CV = np.where(NOISE4_CV > 0, NOISE4_CV, 0)
        WEMP4_CV = np.where(WEMP4_CV > 0, WEMP4_CV, 0)
        DOC4_CV = np.where(DOC4_CV > 0, DOC4_CV, 0)
        ROUGH4_CV = np.where(ROUGH4_CV > 0, ROUGH4_CV, 0)
        WFUEL4_CV = np.where(WFUEL4_CV > 0, WFUEL4_CV, 0)
        RANGE4_CV = np.where(RANGE4_CV > 0, RANGE4_CV, 0)

        NOISE6_CV = np.where(NOISE6_CV > 0, NOISE6_CV, 0)
        WEMP6_CV = np.where(WEMP6_CV > 0, WEMP6_CV, 0)
        DOC6_CV = np.where(DOC6_CV > 0, DOC6_CV, 0)
        ROUGH6_CV = np.where(ROUGH6_CV > 0, ROUGH6_CV, 0)
        WFUEL6_CV = np.where(WFUEL6_CV > 0, WFUEL6_CV, 0)
        RANGE6_CV = np.where(RANGE6_CV > 0, RANGE6_CV, 0)

        return -(
        NOISE2_CV + WEMP2_CV + DOC2_CV + ROUGH2_CV
		+ WFUEL2_CV + RANGE2_CV + NOISE4_CV + WEMP4_CV + DOC4_CV
		+ ROUGH4_CV + WFUEL4_CV + RANGE4_CV + NOISE6_CV + WEMP6_CV
		+ DOC6_CV + ROUGH6_CV + WFUEL6_CV + RANGE6_CV
        )

    objective_1 = ScalarObjective(name="", evaluator=f_1, maximize=[False])
    objective_2 = ScalarObjective(name="", evaluator=f_2, maximize=[False])
    objective_3 = ScalarObjective(name="", evaluator=f_3, maximize=[False])
    objective_4 = ScalarObjective(name="", evaluator=f_4, maximize=[False])
    objective_5 = ScalarObjective(name="", evaluator=f_5, maximize=[False])
    objective_6 = ScalarObjective(name="", evaluator=f_6, maximize=[False])
    objective_7 = ScalarObjective(name="", evaluator=f_7, maximize=[True])
    objective_8 = ScalarObjective(name="", evaluator=f_8, maximize=[True])
    objective_9 = ScalarObjective(name="", evaluator=f_9, maximize=[True])
    objective_10 = ScalarObjective(name="", evaluator=f_10, maximize=[False])

    objectives = [objective_1, objective_2, objective_3, objective_4, objective_5, 
                objective_6, objective_7, objective_8, objective_9, objective_10]

    cons_1 = ScalarConstraint("c_1", 27, 10, g_1)
    
    constraints = [cons_1]

    x_1 = Variable("cruise speed 2 seats", 0.36, 0.24, 0.48)
    x_2 = Variable("aspect ratio 2 seats", 9, 7, 11)
    x_3 = Variable("sweep angle 2 seats", 3, 0, 6)
    x_4 = Variable("propeller diameter 2 seats", 5.7, 5.5, 5.968)
    x_5 = Variable("wing loading 2 seats", 22, 19, 25)
    x_6 = Variable("engine activity factor 2 seats", 97, 85, 110)
    x_7 = Variable("seat width 2 seats", 17, 14, 20)
    x_8 = Variable("tail length/diameter ratio 2 seats", 3.4, 3, 3.75)
    x_9 = Variable("taper ratio 2 seats", 0.75, 0.46, 1)

    x_10 = Variable("cruise speed 4 seats", 0.36, 0.24, 0.48)
    x_11 = Variable("aspect ratio 4 seats", 9, 7, 11)
    x_12 = Variable("sweep angle 4 seats", 3, 0, 6)
    x_13 = Variable("propeller diameter 4 seats", 5.7, 5.5, 5.968)
    x_14 = Variable("wing loading 4 seats", 22, 19, 25)
    x_15 = Variable("engine activity factor 4 seats", 97, 85, 110)
    x_16 = Variable("seat width 4 seats", 17, 14, 20)
    x_17 = Variable("tail length/diameter ratio 4 seats", 3.4, 3, 3.75)
    x_18 = Variable("taper ratio 4 seats", 0.75, 0.46, 1)

    x_19 = Variable("cruise speed 6 seats", 0.36, 0.24, 0.48)
    x_20 = Variable("aspect ratio 6 seats", 9, 7, 11)
    x_21 = Variable("sweep angle 6 seats", 3, 0, 6)
    x_22 = Variable("propeller diameter 6 seats", 5.7, 5.5, 5.968)
    x_23 = Variable("wing loading 6 seats", 22, 19, 25)
    x_24 = Variable("engine activity factor 6 seats", 97, 85, 110)
    x_25 = Variable("seat width 6 seats", 17, 14, 20)
    x_26 = Variable("tail length/diameter ratio 6 seats", 3.4, 3, 3.75)
    x_27 = Variable("taper ratio 6 seats", 0.75, 0.46, 1)

    variables = [x_1, x_2, x_3, x_4, x_5, x_6, x_7, x_8, x_9, x_10, 
                x_11, x_12, x_13, x_14, x_15, x_16, x_17, x_18, x_19, 
                x_20, x_21, x_22, x_23, x_24, x_25, x_26, x_27]

    problem = MOProblem(variables=variables, objectives=objectives, constraints=constraints)


    return problem
