"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscalingManager = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const runner_configuration_1 = require("../runner-configuration");
const DEFAULT_SSH_KEY_PATH = "/etc/gitlab-runner/ssh-custom";
/**
 * Settings for the manager (coordinator)
 *
 *  Manager coordinates the placement of runner (job executor) instances
 */
class GitlabRunnerAutoscalingManager extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.globalConfiguration =
            props.globalConfiguration ||
                {
                    concurrent: 10,
                    checkInterval: 0,
                    logFormat: "runner",
                    logLevel: "info",
                };
        this.machineImage =
            props.machineImage ??
                aws_ec2_1.MachineImage.latestAmazonLinux({
                    generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
                    edition: aws_ec2_1.AmazonLinuxEdition.STANDARD,
                    virtualization: aws_ec2_1.AmazonLinuxVirt.HVM,
                    storage: aws_ec2_1.AmazonLinuxStorage.EBS,
                    cpuType: aws_ec2_1.AmazonLinuxCpuType.X86_64,
                });
        this.instanceType = props.instanceType ?? aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.NANO);
        this.keyPairName = props.keyPairName;
        this.runners = props.runners;
        this.network = props.network;
        this.cacheBucket = props.cacheBucket;
        this.runnersSecurityGroupName = props.runnersSecurityGroup.securityGroupName;
        this.role =
            props.role ||
                new aws_iam_1.Role(scope, "ManagerRole", {
                    assumedBy: new aws_iam_1.ServicePrincipal("ec2.amazonaws.com", {}),
                    managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName("AmazonSSMManagedInstanceCore")],
                    inlinePolicies: {
                        Cache: aws_iam_1.PolicyDocument.fromJson({
                            Version: "2012-10-17",
                            Statement: [
                                {
                                    Effect: "Allow",
                                    Action: ["s3:ListObjects*", "s3:GetObject*", "s3:DeleteObject*", "s3:PutObject*"],
                                    Resource: [`${this.cacheBucket.bucketArn}/*`],
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["s3:ListBucket"],
                                    Resource: [`${this.cacheBucket.bucketArn}`],
                                },
                            ],
                        }),
                        Runners: aws_iam_1.PolicyDocument.fromJson({
                            Version: "2012-10-17",
                            Statement: [
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:CreateKeyPair", "ec2:DeleteKeyPair", "ec2:ImportKeyPair", "ec2:Describe*"],
                                    Resource: ["*"],
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:CreateTags", "ssm:UpdateInstanceInformation"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringLike: {
                                            "aws:RequestTag/Name": "*gitlab-runner-*",
                                        },
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["Name"],
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:RequestSpotInstances", "ec2:CancelSpotInstanceRequests"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringEqualsIfExists: {
                                            "ec2:Region": `${aws_cdk_lib_1.Stack.of(this).region}`,
                                        },
                                        ArnEqualsIfExists: {
                                            "ec2:Vpc": `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(this).region}:${aws_cdk_lib_1.Stack.of(this).account}:vpc/${this.network.vpc.vpcId}`,
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:RunInstances"],
                                    Resource: ["*"],
                                    Condition: {
                                        "ForAllValues:StringEquals": {
                                            "ec2:InstanceType": (this.runners || []).map((runner) => {
                                                const runnersInstanceType = (this.runners && runner.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
                                                return runnersInstanceType.toString();
                                            }),
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:TerminateInstances", "ec2:StopInstances", "ec2:StartInstances", "ec2:RebootInstances"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringLike: {
                                            "ec2:ResourceTag/Name": "*gitlab-runner-*",
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["iam:PassRole"],
                                    Resource: ["*"],
                                    Condition: {
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["RunnersRole"],
                                        },
                                    },
                                },
                            ],
                        }),
                    },
                });
        this.userData = aws_ec2_1.UserData.forLinux({});
        this.userData.addCommands(`yum update -y aws-cfn-bootstrap` // !/bin/bash -xe
        );
        // https://github.com/awslabs/amazon-ecr-credential-helper
        const userDataRunners = aws_ec2_1.UserData.forLinux({});
        userDataRunners.addCommands(`[ ! -z "$(which apt-get)" ] && apt-get install -y amazon-ecr-credential-helper`, `[ ! -z "$(which yum)" ] && yum install -y amazon-ecr-credential-helper`);
        const gitlabRunnerConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        gitlabRunnerConfigRestartHandle._addFile("/etc/gitlab-runner/config.toml");
        const rsyslogConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        rsyslogConfigRestartHandle._addFile("/etc/rsyslog.d/25-gitlab-runner.conf");
        /**
         * Config set keys
         * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html#aws-resource-init-configsets
         */
        const REPOSITORIES = "repositories";
        const PACKAGES = "packages";
        const CONFIG = "config";
        const RESTART = "restart";
        this.initConfig = aws_ec2_1.CloudFormationInit.fromConfigSets({
            configSets: {
                default: [REPOSITORIES, PACKAGES, CONFIG, RESTART],
            },
            configs: {
                [REPOSITORIES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://packages.gitlab.com/install/repositories/runner/gitlab-runner/script.rpm.sh | bash", { key: "10-gitlab-runner" }),
                ]),
                [PACKAGES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitPackage.yum("docker"),
                    aws_ec2_1.InitPackage.yum("gitlab-runner"),
                    aws_ec2_1.InitPackage.yum("tzdata"),
                    aws_ec2_1.InitPackage.yum("jq"),
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://gitlab-docker-machine-downloads.s3.amazonaws.com/v0.16.2-gitlab.19/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine", 
                    //"curl -L https://github.com/docker/machine/releases/download/v0.16.2/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine",
                    { key: "10-docker-machine" }),
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner start", {
                        key: "20-gitlab-runner-start",
                    }),
                ]),
                [CONFIG]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitFile.fromString("/etc/gitlab-runner/config.toml", runner_configuration_1.ConfigurationMapper.withDefaults({
                        globalConfiguration: this.globalConfiguration,
                        runnersConfiguration: this.runners.map((runner) => {
                            const configuration = runner.configuration;
                            return {
                                ...configuration,
                                machine: {
                                    ...configuration.machine,
                                    machineOptions: {
                                        sshKeypath: 
                                        // drivers/amazonec2/amazonec2.go SSHPrivateKeyPath
                                        runner.keyPair
                                            ? `${DEFAULT_SSH_KEY_PATH}/${configuration.machine?.machineOptions?.keypairName}`
                                            : "",
                                        ...configuration.machine?.machineOptions,
                                        instanceType: runner.instanceType.toString(),
                                        ami: runner.machineImage.getImage(scope).imageId,
                                        region: aws_cdk_lib_1.Stack.of(this).region,
                                        vpcId: this.network.vpc.vpcId,
                                        zone: this.network.availabilityZone.slice(-1),
                                        subnetId: this.network.subnet.subnetId,
                                        securityGroup: this.runnersSecurityGroupName,
                                        privateAddressOnly: configuration.machine?.machineOptions?.privateAddressOnly ?? this.network.hasPrivateSubnets(),
                                        usePrivateAddress: configuration.machine?.machineOptions?.usePrivateAddress ?? true,
                                        iamInstanceProfile: runner.instanceProfile.ref,
                                        userdata: "/etc/gitlab-runner/user_data_runners",
                                    },
                                },
                                cache: {
                                    s3: {
                                        serverAddress: `s3.${aws_cdk_lib_1.Stack.of(this).urlSuffix}`,
                                        bucketName: `${this.cacheBucket.bucketName}`,
                                        bucketLocation: `${aws_cdk_lib_1.Stack.of(this).region}`,
                                        authenticationType: "iam",
                                    },
                                },
                            };
                        }),
                    }).toToml(), {
                        owner: "gitlab-runner",
                        group: "gitlab-runner",
                        mode: "000600",
                    }),
                    aws_ec2_1.InitFile.fromString("/etc/rsyslog.d/25-gitlab-runner.conf", `:programname, isequal, "gitlab-runner" /var/log/gitlab-runner.log`, {
                        owner: "root",
                        group: "root",
                        mode: "000644",
                    }),
                    aws_ec2_1.InitService.enable("gitlab-runner", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: gitlabRunnerConfigRestartHandle,
                    }),
                    aws_ec2_1.InitService.enable("rsyslog", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: rsyslogConfigRestartHandle,
                    }),
                    aws_ec2_1.InitCommand.shellCommand(
                    // Download custom EC2 key pair for manager <> runner ssh connect
                    this.runners
                        .map((runner) => {
                        if (!runner.keyPair) {
                            return "";
                        }
                        runner.keyPair.grantRead(this.role);
                        const region = aws_cdk_lib_1.Stack.of(this).region;
                        const secretArn = runner.keyPair.secretArn;
                        const keyPairName = runner.configuration.machine.machineOptions.keypairName;
                        const sshKeyPath = runner.configuration.machine.machineOptions.sshKeypath ?? DEFAULT_SSH_KEY_PATH;
                        return [
                            `mkdir -p ${sshKeyPath};`,
                            `echo $(aws secretsmanager get-secret-value --region ${region} --secret-id ${secretArn} --query SecretString --output text) | jq -r '."${keyPairName}"' > ${sshKeyPath}/${keyPairName};`,
                            `echo $(aws secretsmanager get-secret-value --region ${region} --secret-id ${secretArn} --query SecretString --output text) | jq -r '."${keyPairName}.pub"' > ${sshKeyPath}/${keyPairName}.pub;`,
                        ].join("\n");
                    })
                        .filter((s) => s.length > 0)
                        .join("\n"), {
                        key: "999-retrieve-ec2-key-pair",
                    }),
                    aws_ec2_1.InitFile.fromString("/etc/gitlab-runner/user_data_runners", userDataRunners.render(), {
                        owner: "gitlab-runner",
                        group: "gitlab-runner",
                        mode: "000600",
                    }),
                ]),
                [RESTART]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner restart", {
                        key: "10-gitlab-runner-restart",
                    }),
                ]),
            },
        });
    }
}
exports.GitlabRunnerAutoscalingManager = GitlabRunnerAutoscalingManager;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscalingManager[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.GitlabRunnerAutoscalingManager", version: "0.2.365" };
//# sourceMappingURL=data:application/json;base64,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