import datetime
from typing import Any, Dict, List, Type, TypeVar, Union

import attr
from dateutil.parser import isoparse

from ..models.exploit_only_report_data_source_enum import (
    ExploitOnlyReportDataSourceEnum,
)
from ..models.maturity_preliminary_enum import MaturityPreliminaryEnum
from ..types import UNSET, OSIDBModel, Unset

T = TypeVar("T", bound="ExploitOnlyReportData")


@attr.s(auto_attribs=True)
class ExploitOnlyReportData(OSIDBModel):
    """ """

    cve: str
    source: ExploitOnlyReportDataSourceEnum
    maturity_preliminary: MaturityPreliminaryEnum
    flaw: bool
    date: Union[Unset, None, datetime.date] = UNSET
    reference: Union[Unset, str] = UNSET
    additional_properties: Dict[str, Any] = attr.ib(init=False, factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        cve = self.cve
        source: str = UNSET
        if not isinstance(self.source, Unset):

            source = ExploitOnlyReportDataSourceEnum(self.source).value

        maturity_preliminary: int = UNSET
        if not isinstance(self.maturity_preliminary, Unset):

            maturity_preliminary = MaturityPreliminaryEnum(
                self.maturity_preliminary
            ).value

        flaw = self.flaw
        date: Union[Unset, None, str] = UNSET
        if not isinstance(self.date, Unset):
            date = self.date.isoformat() if self.date else None

        reference = self.reference

        field_dict: Dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        if cve is not UNSET:
            field_dict["cve"] = cve
        if source is not UNSET:
            field_dict["source"] = source
        if maturity_preliminary is not UNSET:
            field_dict["maturity_preliminary"] = maturity_preliminary
        if flaw is not UNSET:
            field_dict["flaw"] = flaw
        if date is not UNSET:
            field_dict["date"] = date
        if reference is not UNSET:
            field_dict["reference"] = reference

        return field_dict

    @classmethod
    def from_dict(cls: Type[T], src_dict: Dict[str, Any]) -> T:
        d = src_dict.copy()
        cve = d.pop("cve", UNSET)

        _source = d.pop("source", UNSET)
        source: ExploitOnlyReportDataSourceEnum
        if isinstance(_source, Unset):
            source = UNSET
        else:
            source = ExploitOnlyReportDataSourceEnum(_source)

        _maturity_preliminary = d.pop("maturity_preliminary", UNSET)
        maturity_preliminary: MaturityPreliminaryEnum
        if isinstance(_maturity_preliminary, Unset):
            maturity_preliminary = UNSET
        else:
            maturity_preliminary = MaturityPreliminaryEnum(_maturity_preliminary)

        flaw = d.pop("flaw", UNSET)

        _date = d.pop("date", UNSET)
        date: Union[Unset, None, datetime.date]
        if _date is None:
            date = None
        elif isinstance(_date, Unset):
            date = UNSET
        else:
            date = isoparse(_date).date()

        reference = d.pop("reference", UNSET)

        exploit_only_report_data = cls(
            cve=cve,
            source=source,
            maturity_preliminary=maturity_preliminary,
            flaw=flaw,
            date=date,
            reference=reference,
        )

        exploit_only_report_data.additional_properties = d
        return exploit_only_report_data

    @staticmethod
    def get_fields():
        return {
            "cve": str,
            "source": ExploitOnlyReportDataSourceEnum,
            "maturity_preliminary": MaturityPreliminaryEnum,
            "flaw": bool,
            "date": datetime.date,
            "reference": str,
        }

    @property
    def additional_keys(self) -> List[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties
