"""Test adding EXIF attributes."""

import binascii
import os
import textwrap
import unittest

from exif import GpsAltitudeRef, Image
from exif.tests.add_exif_baselines.add_short import ADD_SHORT_BASELINE, ADD_SHORT_LE_BASELINE
from exif.tests.add_exif_baselines.add_ascii import ADD_ASCII_BASELINE, ADD_ASCII_LE_BASELINE
from exif.tests.add_exif_baselines.add_gps import ADD_GPS_BASELINE
from exif.tests.add_exif_baselines.add_to_scan import ADD_TO_SCANNED_IMAGE_BASELINE
from exif.tests.test_little_endian import read_attributes as read_attributes_little_endian
from exif.tests.test_read_exif import read_attributes_florida_beach

# pylint: disable=protected-access


class TestAddExif(unittest.TestCase):

    """Test cases for adding EXIF attributes."""

    def setUp(self):
        """Open sample image file in binary mode for use in test cases."""
        florida = os.path.join(os.path.dirname(__file__), 'florida_beach.jpg')
        little_endian = os.path.join(os.path.dirname(__file__), 'little_endian.jpg')
        noise = os.path.join(os.path.dirname(__file__), 'noise.jpg')
        self.image = Image(florida)
        self.image_alt = Image(noise)
        self.image_le = Image(little_endian)

        assert self.image.has_exif
        assert self.image_le.has_exif

    def test_add_ascii(self):
        """Test adding new ASCII tags to an image."""
        self.image_alt.image_description = "Noise Generated by Photoshop"
        self.image_alt.make = "Test Make"
        self.image_alt.model = "Test Model"

        assert self.image_alt.image_description == "Noise Generated by Photoshop"
        assert self.image_alt.make == "Test Make"
        assert self.image_alt.model == "Test Model"

        segment_hex = binascii.hexlify(self.image_alt._segments['APP1'].get_segment_bytes()).decode("utf-8").upper()
        self.assertEqual('\n'.join(textwrap.wrap(segment_hex, 90)),
                         ADD_ASCII_BASELINE)

    def test_add_ascii_le(self):
        """Test adding new ASCII tags to a little endian image."""
        self.image_le.datetime = "2020:07:09 21:09:53"
        self.image_le.datetime_original = "2020:07:09 21:09:53"
        self.image_le.datetime_digitized = "2020:07:09 21:09:53"

        assert self.image_le.datetime == "2020:07:09 21:09:53"
        assert self.image_le.datetime_original == "2020:07:09 21:09:53"
        assert self.image_le.datetime_digitized == "2020:07:09 21:09:53"

        segment_hex = binascii.hexlify(self.image_alt._segments['APP1'].get_segment_bytes()).decode("utf-8").upper()
        self.assertEqual('\n'.join(textwrap.wrap(segment_hex, 90)),
                         ADD_ASCII_LE_BASELINE)

    def test_add_gps(self):
        """Test adding GPS-related tags to an image."""
        self.image_alt.gps_longitude = (112.0, 5.0, 4.18)
        self.image_alt.gps_latitude = (36.0, 3.0, 11.08)
        self.image_alt.gps_longitude_ref = "W"
        self.image_alt.gps_latitude_ref = "N"
        self.image_alt.gps_altitude = 2189.9896907216494
        self.image_alt.gps_altitude_ref = GpsAltitudeRef.ABOVE_SEA_LEVEL

        assert self.image_alt.gps_longitude == (112.0, 5.0, 4.18)
        assert self.image_alt.gps_latitude == (36.0, 3.0, 11.08)
        assert self.image_alt.gps_longitude_ref == "W"
        assert self.image_alt.gps_latitude_ref == "N"
        assert self.image_alt.gps_altitude == 2189.9896907216494
        assert self.image_alt.gps_altitude_ref == GpsAltitudeRef.ABOVE_SEA_LEVEL

        segment_hex = binascii.hexlify(self.image_alt._segments['APP1'].get_segment_bytes()).decode("utf-8").upper()
        self.assertEqual('\n'.join(textwrap.wrap(segment_hex, 90)),
                         ADD_GPS_BASELINE)

    def test_add_shorts(self):
        """Test adding two new SHORT tags to an image."""
        self.image.light_source = 1
        self.image.contrast = 0

        assert self.image.light_source == 1
        assert self.image.contrast == 0

        # Verify pre-existing attributes can still be read as expected.
        for attribute, func, value in read_attributes_florida_beach:
            assert func(getattr(self.image, attribute)) == value

        segment_hex = binascii.hexlify(self.image._segments['APP1'].get_segment_bytes()).decode("utf-8").upper()
        self.assertEqual('\n'.join(textwrap.wrap(segment_hex, 90)),
                         ADD_SHORT_BASELINE)

    def test_add_shorts_and_srational_le(self):
        """Test adding two new SHORT tags and an SRATIONAL to a little endian image."""
        self.image_le.contrast = 1
        self.image_le.light_source = 24
        self.image_le.brightness_value = 5.4321

        assert self.image_le.light_source == 24
        assert self.image_le.contrast == 1
        assert self.image_le.brightness_value == 5.4321

        # Verify pre-existing attributes can still be read as expected.
        for attribute, func, value in read_attributes_little_endian:
            assert func(getattr(self.image_le, attribute)) == value

        segment_hex = binascii.hexlify(self.image_le._segments['APP1'].get_segment_bytes()).decode("utf-8").upper()
        self.assertEqual('\n'.join(textwrap.wrap(segment_hex, 90)),
                         ADD_SHORT_LE_BASELINE)


def test_add_to_scanner_image():
    """Test adding metadata to a scanner-produced JPEG without any pre-existing APP1 or EXIF."""
    image = Image(os.path.join(os.path.dirname(__file__), 'scanner_without_app1.jpg'))
    assert not image.has_exif
    image.gps_latitude = (41.0, 29.0, 57.48)
    image.gps_latitude_ref = "N"
    image.gps_longitude = (81.0, 41.0, 39.84)
    image.gps_longitude_ref = "W"
    image.gps_altitude = 199.034
    image.gps_altitude_ref = GpsAltitudeRef.ABOVE_SEA_LEVEL
    image.make = "Acme Scanner Company"
    image.model = "Scan-o-Matic 5000"
    image.datetime_original = "1999:12:31 23:49:12"
    image.datetime_digitized = "2020:07:11 10:11:37"
    image.brightness_value = 10.9876  # provides coverage for SRATIONAL

    assert image.has_exif
    assert image.gps_latitude == (41.0, 29.0, 57.48)
    assert image.gps_latitude_ref == "N"
    assert image.gps_longitude == (81.0, 41.0, 39.84)
    assert image.gps_longitude_ref == "W"
    assert image.gps_altitude == 199.034
    assert image.gps_altitude_ref == GpsAltitudeRef.ABOVE_SEA_LEVEL
    assert image.make == "Acme Scanner Company"
    assert image.model == "Scan-o-Matic 5000"
    assert image.datetime_original == "1999:12:31 23:49:12"
    assert image.datetime_digitized == "2020:07:11 10:11:37"
    assert image.brightness_value == 10.9876  # provides coverage for SRATIONAL

    segment_hex = binascii.hexlify(image._segments['APP1'].get_segment_bytes()).decode("utf-8").upper()
    assert '\n'.join(textwrap.wrap(segment_hex, 90)) == ADD_TO_SCANNED_IMAGE_BASELINE
