# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/online.ipynb (unless otherwise specified).

__all__ = ['RollingMean', 'RollingMax', 'RollingMin', 'RollingStd', 'SeasonalRollingMean', 'SeasonalRollingStd',
           'SeasonalRollingMin', 'SeasonalRollingMax', 'ExpandingMean', 'ExpandingMax', 'ExpandingMin', 'ExpandingStd',
           'SeasonalExpandingMean', 'SeasonalExpandingStd', 'SeasonalExpandingMin', 'SeasonalExpandingMax', 'EWMMean']

# Cell
from math import ceil, sqrt
from typing import Callable, List, Optional, Union

import numpy as np

from .expanding import *
from .ewm import *
from .rolling import *
from .utils import _rolling_std

# Internal Cell
class BaseOnlineRolling:

    def __init__(self, rolling_op: Callable, window_size: int, min_samples: Optional[int] = None):
        self.rolling_op = rolling_op
        self.window_size = window_size
        self.min_samples = min_samples

    def fit_transform(self, x: np.ndarray) -> np.ndarray:
        self.x = x[-self.window_size:].tolist()
        return self.rolling_op(x, self.window_size, self.min_samples)

    def _update_x(self, x: np.ndarray) -> None:
        self.x = self.x[1:] + [x]

# Cell
class RollingMean(BaseOnlineRolling):

    def __init__(self, window_size: int, min_samples: Optional[int] = None):
        super().__init__(rolling_mean, window_size, min_samples)

    def update(self, x: float) -> float:
        self._update_x(x)
        return sum(self.x) / self.window_size

# Cell
class RollingMax(BaseOnlineRolling):

    def __init__(self, window_size: int, min_samples: Optional[int] = None):
        super().__init__(rolling_max, window_size, min_samples)

    def update(self, x) -> float:
        self._update_x(x)
        return max(self.x)

# Cell
class RollingMin(BaseOnlineRolling):

    def __init__(self, window_size, min_samples):
        super().__init__(rolling_min, window_size, min_samples)

    def update(self, x) -> float:
        self.x = self.x[1:] + [x]
        return min(self.x)

# Cell
class RollingStd(BaseOnlineRolling):

    def __init__(self, window_size, min_samples):
        super().__init__(rolling_std, window_size, min_samples)

    def update(self, x) -> float:
        self._update_x(x)
        return np.std(self.x, ddof=1)

# Internal Cell
class BaseOnlineSeasonalRolling:

    def __init__(self,
                 seasonal_rolling_op: Callable,
                 season_length: int,
                 window_size: int,
                 min_samples: Optional[int] = None):
        self.seasonal_rolling_op = seasonal_rolling_op
        self.season_length = season_length
        self.window_size = window_size
        self.min_samples = min_samples

    def fit_transform(self, x: np.ndarray) -> np.ndarray:
        n_samples = x.size
        min_samples = self.season_length * self.window_size
        if n_samples < min_samples:
            return np.full(n_samples, np.nan)
        self.x = x[-min_samples:].tolist()
        return self.seasonal_rolling_op(x, self.season_length, self.window_size, self.min_samples)

    def _update_and_get_seasonal_terms(self, x: float) -> List[float]:
        self.x = self.x[1:] + [x]
        return self.x[self.season_length-1::self.season_length]

# Cell
class SeasonalRollingMean(BaseOnlineSeasonalRolling):

    def __init__(self,
                 season_length: int,
                 window_size: int,
                 min_samples: Optional[int] = None):
        super().__init__(seasonal_rolling_mean, season_length, window_size, min_samples)

    def update(self, x: float) -> float:
        seasonal_terms = self._update_and_get_seasonal_terms(x)
        return sum(seasonal_terms) / self.window_size

# Cell
class SeasonalRollingStd(BaseOnlineSeasonalRolling):

    def __init__(self,
                 season_length: int,
                 window_size: int,
                 min_samples: Optional[int] = None):
        super().__init__(seasonal_rolling_std, season_length, window_size, min_samples)

    def update(self, x: float) -> float:
        seasonal_terms = self._update_and_get_seasonal_terms(x)
        return np.std(seasonal_terms, ddof=1)

# Cell
class SeasonalRollingMin(BaseOnlineSeasonalRolling):

    def __init__(self,
                 season_length: int,
                 window_size: int,
                 min_samples: Optional[int] = None):
        super().__init__(seasonal_rolling_min, season_length, window_size, min_samples)

    def update(self, x: float) -> float:
        seasonal_terms = self._update_and_get_seasonal_terms(x)
        return min(seasonal_terms)

# Cell
class SeasonalRollingMax(BaseOnlineSeasonalRolling):

    def __init__(self,
                 season_length: int,
                 window_size: int,
                 min_samples: Optional[int] = None):
        super().__init__(seasonal_rolling_max, season_length, window_size, min_samples)

    def update(self, x: float) -> float:
        seasonal_terms = self._update_and_get_seasonal_terms(x)
        return max(seasonal_terms)

# Cell
class ExpandingMean:

    def fit_transform(self, x: np.ndarray) -> np.ndarray:
        exp_mean = expanding_mean(x)
        self.n = x.size
        self.cumsum = exp_mean[-1] * self.n
        return exp_mean

    def update(self, x: float) -> float:
        self.cumsum += x
        self.n += 1
        return self.cumsum / self.n

# Cell
class ExpandingMax:

    def fit_transform(self, x: np.ndarray) -> np.ndarray:
        exp_max = expanding_max(x)
        self.max = exp_max[-1]
        return exp_max

    def update(self, x: float) -> float:
        if x > self.max:
            self.max = x
        return self.max

# Cell
class ExpandingMin:

    def fit_transform(self, x: np.ndarray) -> np.ndarray:
        exp_min = expanding_min(x)
        self.min = exp_min[-1]
        return exp_min

    def update(self, x: float) -> float:
        if x < self.min:
            self.min = x
        return self.min

# Cell
class ExpandingStd:

    def fit_transform(self, x):
        self.n = x.size
        exp_std, self.curr_avg, self.x_m2n = _rolling_std(x,
                                                          window_size=self.n,
                                                          min_samples=2)
        return exp_std

    def update(self, x):
        prev_avg = self.curr_avg
        self.n += 1
        self.curr_avg = prev_avg + (x - prev_avg) / self.n
        self.x_m2n += (x - prev_avg) * (x - self.curr_avg)
        return sqrt(self.x_m2n / (self. n - 1))

# Internal Cell
class BaseSeasonalExpanding:

    def __init__(self,
                 ExpandingOp: Union[ExpandingMean, ExpandingMax, ExpandingMin, ExpandingStd],
                 season_length: int):
        self.ExpandingOp = ExpandingOp
        self.season_length = season_length

    def fit_transform(self, x: np.ndarray) -> np.ndarray:
        self.expanding_ops = []
        self.n_samples = x.size
        result = np.empty(self.n_samples)
        for season in range(self.season_length):
            exp_op = self.ExpandingOp()
            result[season::self.season_length] = exp_op.fit_transform(x[season::self.season_length])
            self.expanding_ops.append(exp_op)
        return result

    def update(self, x: float) -> float:
        season = self.n_samples % self.season_length
        self.n_samples += 1
        return self.expanding_ops[season].update(x)

# Cell
class SeasonalExpandingMean(BaseSeasonalExpanding):

    def __init__(self, season_length):
        super().__init__(ExpandingMean, 7)

# Cell
class SeasonalExpandingStd(BaseSeasonalExpanding):

    def __init__(self, season_length):
        super().__init__(ExpandingStd, 7)

# Cell
class SeasonalExpandingMin(BaseSeasonalExpanding):

    def __init__(self, season_length):
        super().__init__(ExpandingMin, 7)

# Cell
class SeasonalExpandingMax(BaseSeasonalExpanding):

    def __init__(self, season_length):
        super().__init__(ExpandingMax, 7)

# Cell
class EWMMean:

    def __init__(self, alpha):
        self.alpha = alpha

    def fit_transform(self, x):
        mn = ewm_mean(x, self.alpha)
        self.smoothed = mn[-1]
        return mn

    def update(self, x):
        self.smoothed = self.alpha * x + (1 - self.alpha) * self.smoothed
        return self.smoothed