# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pardoc']

package_data = \
{'': ['*']}

install_requires = \
['diot>=0.1.6,<0.2.0', 'lark-parser>=0.12,<0.13']

setup_kwargs = {
    'name': 'pardoc',
    'version': '0.1.0',
    'description': 'Yet another docstring parser for python',
    'long_description': '# pardoc\n\n[![pypi][3]][4] [![tag][5]][6] ![pyver][12] [![build][7]][8] [![codacy quality][9]][10] [![codacy quality][11]][10]\n\nYet another docstring parser for python, using [`lark-parser`][1]\n\n## Installation\n\n```python\npip install pardoc\n```\n\n## A quick look\n\n```python console\n>>> from pardoc import google_parser, pretty\n\n>>> docstring = """This is an example of a module level function.\n\nFunction parameters should be documented in the ``Args`` section. The name\nof each parameter is required. The type and description of each parameter\nis optional, but should be included if not obvious.\n\nIf \\*args or \\*\\*kwargs are accepted,\nthey should be listed as ``*args`` and ``**kwargs``.\n\nThe format for a parameter is::\n\n    name (type): description\n        The description may span multiple lines. Following\n        lines should be indented. The "(type)" is optional.\n\n        Multiple paragraphs are supported in parameter\n        descriptions.\n\nArgs:\n    param1 (int): The first parameter.\n    param2 (:obj:`str`, optional): The second parameter. Defaults to None.\n        Second line of description should be indented.\n    *args: Variable length argument list.\n    **kwargs: Arbitrary keyword arguments.\n\nReturns:\n    bool: True if successful, False otherwise.\n\n    The return type is optional and may be specified at the beginning of\n    the ``Returns`` section followed by a colon.\n\n    The ``Returns`` section may span multiple lines and paragraphs.\n    Following lines should be indented to match the first line.\n\n    The ``Returns`` section supports any reStructuredText formatting,\n    including literal blocks::\n\n        {\n            \'param1\': param1,\n            \'param2\': param2\n        }\n\nRaises:\n    AttributeError: The ``Raises`` section is a list of all exceptions\n        that are relevant to the interface.\n    ValueError: If `param2` is equal to `param1`.\n\n"""\n\n>>> parsed = google_parser.parse(docstring)\n>>> pretty(parsed, print_=True)\n\nParsedSection(title=SUMMARY)\n   ParsedPara(lines=1)\n      This is an example of a module level function.\n   ParsedPara(lines=3)\n      Function parameters should be documented in the ``Args`` section. The name\n      of each parameter is required. The type and description of each parameter\n      is optional, but should be included if not obvious.\n   ParsedPara(lines=2)\n      If \\*args or \\*\\*kwargs are accepted,\n      they should be listed as ``*args`` and ``**kwargs``.\n   ParsedPara(lines=1)\n      The format for a parameter is::\n   ParsedPara(lines=2)\n      ParsedPara(lines=1)\n         name (type): description\n      ParsedPara(lines=2)\n         ParsedPara(lines=2)\n            The description may span multiple lines. Following\n            lines should be indented. The "(type)" is optional.\n         ParsedPara(lines=2)\n            Multiple paragraphs are supported in parameter\n            descriptions.\n\nParsedSection(title=Args)\n   ParsedItem(name=param1, type=int, desc=The first parameter.)\n   ParsedItem(name=param2, type=:obj:`str`, optional, desc=The second parameter. Defaults to None.)\n      ParsedPara(lines=1)\n         Second line of description should be indented.\n   ParsedItem(name=*args, type=None, desc=Variable length argument list.)\n   ParsedItem(name=**kwargs, type=None, desc=Arbitrary keyword arguments.)\n\nParsedSection(title=Returns)\n   ParsedItem(name=bool, type=None, desc=True if successful, False otherwise.)\n   ParsedPara(lines=2)\n      The return type is optional and may be specified at the beginning of\n      the ``Returns`` section followed by a colon.\n   ParsedPara(lines=2)\n      The ``Returns`` section may span multiple lines and paragraphs.\n      Following lines should be indented to match the first line.\n   ParsedPara(lines=2)\n      The ``Returns`` section supports any reStructuredText formatting,\n      including literal blocks::\n   ParsedPara(lines=2)\n      ParsedPara(lines=1)\n         {\n      ParsedPara(lines=1)\n         ParsedPara(lines=2)\n            \'param1\': param1,\n            \'param2\': param2\n   ParsedPara(lines=1)\n      ParsedPara(lines=1)\n         }\n\nParsedSection(title=Raises)\n   ParsedItem(name=AttributeError, type=None, desc=The ``Raises`` section is a list of all exceptions)\n      ParsedPara(lines=1)\n         that are relevant to the interface.\n   ParsedItem(name=ValueError, type=None, desc=If `param2` is equal to `param1`.)\n\n```\n\n## Usage\n\n### Parsing a known style docstring\n\n```python\nfrom pardoc import google_parser, numpy_parser\nparsed = google_parser(docstring)\n# or\nparsed = numpy_parser(docstring)\n```\n\n### Parsing an unknown style docstring\n\n```python\nfrom pardoc import auto_parser\n\nparser = auto_parser(docstring)\n# parsing results from auto_parser is cached and reused.\nparsed = parser.parse(docstring)\n```\n\n### Parsed object\n\nThere are 6 types of parsed objects, include the final `Parsed` object that\nattaches all sections\n\nThe first 5 are all `namedtuple`s:\n\n```python\nParsedItem = namedtuple(\'ParsedItem\',\n                        [\'name\', \'type\', \'desc\', \'more\'])\nParsedTodo = namedtuple(\'ParsedTodo\', [\'todo\', \'more\'])\nParsedSection = namedtuple(\'ParsedSection\', [\'title\', \'section\'])\nParsedPara = namedtuple(\'ParsedPara\', [\'lines\'])\nParsedCode = namedtuple(\'ParsedCode\', [\'lang\', \'codes\'])\n\n```\n\nThe `Parsed` is an ordered dictionary (`OrderedDiot`) from [`diot`][2], which\nallows dot access to keys:\n\n```python\nfrom diot import OrderedDiot\n\nclass Parsed(OrderedDiot):\n    """Parsed object"""\n```\n\n### Formatting a parsed object to the original style\n\n```python console\n>>> from pardoc import google_parser\n>>> docstring = """Example function with types documented in the docstring.\n\n\nArgs:\n\n\n    param0: No type\n\n    param1 (int): The first parameter.\n    param2 (str): The second parameter.\n\nReturns:\n\n\n    bool: The return value. True for success, False otherwise.\n\n"""\n>>> # note the arbitrary empty lines\n>>> reformatted = google_parser.format(docstring)\n>>> # or\n>>> reformatted = google_parser.format(google_parser.parse(docstring))\n>>> print(reformatted)\nExample function with types documented in the docstring.\n\nArgs:\n    param0: No type\n    param1 (int): The first parameter.\n    param2 (str): The second parameter.\n\nReturns:\n    bool: The return value. True for success, False otherwise.\n```\n\n### Pretty printing the parsed objects\n\nSee `A quick look`\n\n[1]: https://github.com/lark-parser/lark\n[2]: https://github.com/pwwang/diot\n[3]: https://img.shields.io/pypi/v/pardoc?style=flat-square\n[4]: https://pypi.org/project/pardoc/\n[5]: https://img.shields.io/github/tag/pwwang/pardoc?style=flat-square\n[6]: https://github.com/pwwang/pardoc\n[7]: https://img.shields.io/github/workflow/status/pwwang/pardoc/Build%20and%20Deploy?style=flat-square\n[8]: https://github.com/pwwang/pardoc\n[9]: https://img.shields.io/codacy/grade/a1ba6573a5fa4fc589ce3cf7daa5ddea?style=flat-square\n[10]: https://app.codacy.com/project/pwwang/pardoc/dashboard\n[11]: https://img.shields.io/codacy/coverage/a1ba6573a5fa4fc589ce3cf7daa5ddea?style=flat-square\n[12]: https://img.shields.io/pypi/pyversions/pardoc?style=flat-square\n',
    'author': 'pwwang',
    'author_email': 'pwwang@pwwang.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/pwwang/pardoc',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
