from functools import lru_cache
from pathlib import Path
from typing import TYPE_CHECKING, Dict, List, Any

import yaml

import arcade_gui
from arcade_gui.utils import parse_value

if TYPE_CHECKING:
    pass


class UIStyle:
    """
    Used as singleton in the UIView, style changes are applied by changing the values of the singleton.

    Use `.load()` to update UIStyle instance from YAML-file


    """
    __default_style = None

    def __init__(self, data: Dict, **kwargs):
        super().__init__(**kwargs)
        self.data = data

    @staticmethod
    def from_file(path: Path):
        """
        Load style from a file, overwriting existing data

        :param path: Path to a valid style YAML file
        """
        ui_style = UIStyle({})
        ui_style.load(path)
        return ui_style

    def load(self, path: Path):
        with path.open() as file:
            data: Dict[str, Dict[str, Any]] = yaml.safe_load(file)
            assert isinstance(data, dict)

        # parse values, expected structure Dict[class, Dict[key, value]]
        for style_class, style_data in data.items():
            for key, value in style_data.items():
                style_data[key] = parse_value(value)

        self.data = data

    @classmethod
    def default_style(cls):
        """
        :return: empty style # TODO maybe load the real default style once
        """
        if cls.__default_style is None:
            cls.__default_style = UIStyle.from_file(arcade_gui.resources.path('style/default.yml'))
        return cls.__default_style

    def get_class(self, style_classes: str):
        return self.data.setdefault(style_classes, {})

    def set_class_attrs(self, style_classes: str, **kwargs):
        style_data = self.get_class(style_classes)
        for key, value in kwargs.items():
            if value is None:
                if key in style_data:
                    del style_data[key]
            else:
                style_data[key] = value

    def get_attr(self, style_classes: List[str], attr: str):
        """
        Retrieves an attribute, resolved from style by style_classes

        :param style_classes: List of style classes, resolving from right to left
        :param attr: attribute name to get value for
        :return: value of the attribute, first found
        """
        style_classes = reversed(style_classes)
        for style_class in style_classes:
            style_data = self.data.get(style_class, {})
            attr_value = style_data.get(attr)
            if attr_value:
                return attr_value
        else:
            return None
