from typing import Optional

import arcade

from arcade_gui.elements import UIClickable
from arcade_gui.ui_style import UIStyle
from arcade_gui.utils import get_text_image


class UILabel(UIClickable):
    def __init__(self,
                 text: str,
                 center_x: int,
                 center_y: int,
                 width: int = 0,
                 align='center',
                 id: Optional[str] = None,
                 style: UIStyle = None,
                 **kwargs):
        super().__init__(
            center_x=center_x,
            center_y=center_y,
            id=id,
            style=style,
            **kwargs
        )
        self.style_classes.append('label')

        self.align = align
        self.width = width
        self.text = text

    @property
    def text(self):
        return self._text

    @text.setter
    def text(self, value):
        self._text = value
        self.render_texture()

    def render_texture(self):
        font_name = self.style_attr('font_name', ['Calibri', 'Arial'])
        font_size = self.style_attr('font_size', 12)

        font_color = self.style_attr('font_color', arcade.color.GRAY)
        font_color_hover = self.style_attr('font_color_hover', None)
        font_color_press = self.style_attr('font_color_press', None)

        if font_color_hover is None:
            font_color_hover = font_color
        if font_color_press is None:
            font_color_press = font_color_hover

        text_image_normal = get_text_image(text=self.text,
                                           font_color=font_color,
                                           font_size=font_size,
                                           font_name=font_name,
                                           align=self.align,
                                           width=int(self.width),
                                           )
        text_image_mouse_over = get_text_image(text=self.text,
                                               font_color=font_color_hover,
                                               font_size=font_size,
                                               font_name=font_name,
                                               align=self.align,
                                               width=int(self.width),
                                               )
        text_image_mouse_press = get_text_image(text=self.text,
                                                font_color=font_color_press,
                                                font_size=font_size,
                                                font_name=font_name,
                                                align=self.align,
                                                width=int(self.width),
                                                )

        self.normal_texture = arcade.Texture(image=text_image_normal, name=self.text + '1')
        self.press_texture = arcade.Texture(image=text_image_mouse_press, name=self.text + '2')
        self.hover_texture = arcade.Texture(image=text_image_mouse_over, name=self.text + '3')
