from typing import Optional
from uuid import uuid4

import arcade
from arcade import Texture

from arcade_gui import UIClickable, utils, UIView
from arcade_gui.ui_style import UIStyle


class UIImageButton(UIClickable):
    def __init__(self,
                 center_x,
                 center_y,
                 normal_texture: Texture,
                 hover_texture: Texture,
                 press_texture: Texture,
                 text='',
                 id: Optional[str] = None,
                 style: UIStyle = None,
                 **kwargs
                 ):
        super().__init__(
            center_x=center_x,
            center_y=center_y,
            id=id,
            style=style,
            **kwargs
        )
        self.style_classes.append('imagebutton')

        self._normal_texture = normal_texture
        self._hover_texture = hover_texture
        self._press_texture = press_texture
        if text:
            self.render_with_text(text)
        else:
            self.normal_texture = normal_texture
            self.hover_texture = hover_texture
            self.press_texture = press_texture

    def render_with_text(self, text: str):
        font_size = self.style_attr('font_size', 22)
        font_color = self.style_attr('font_color', arcade.color.GRAY)
        font_color_hover = self.style_attr('font_color_hover', arcade.color.GRAY)
        font_color_press = self.style_attr('font_color_press', arcade.color.GRAY)

        if not font_color_hover:
            font_color_hover = font_color
        if not font_color_press:
            font_color_press = font_color_hover

        normal_image = utils.get_image_with_text(text,
                                                 background_image=self._normal_texture.image,
                                                 font_color=font_color,
                                                 font_size=font_size,
                                                 align='center',
                                                 valign='middle'
                                                 )
        self.normal_texture = Texture(str(uuid4()), image=normal_image)

        hover_image = utils.get_image_with_text(text,
                                                background_image=self._hover_texture.image,
                                                font_color=font_color_hover,
                                                font_size=font_size,
                                                align='center',
                                                valign='middle'
                                                )
        self.hover_texture = Texture(str(uuid4()), image=hover_image)

        press_image = utils.get_image_with_text(text,
                                                background_image=self._press_texture.image,
                                                font_color=font_color_press,
                                                font_size=font_size,
                                                align='center',
                                                valign='middle'
                                                )
        self.press_texture = Texture(str(uuid4()), image=press_image)
