"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DnsValidatedCertificate = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const certificate_base_1 = require("./certificate-base");
/**
 * A certificate managed by AWS Certificate Manager.  Will be automatically
 * validated using DNS validation against the specified Route 53 hosted zone.
 *
 * @resource AWS::CertificateManager::Certificate
 */
class DnsValidatedCertificate extends certificate_base_1.CertificateBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_certificatemanager_DnsValidatedCertificateProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.region = props.region;
        this.domainName = props.domainName;
        this.normalizedZoneName = props.hostedZone.zoneName;
        // Remove trailing `.` from zone name
        if (this.normalizedZoneName.endsWith('.')) {
            this.normalizedZoneName = this.normalizedZoneName.substring(0, this.normalizedZoneName.length - 1);
        }
        // Remove any `/hostedzone/` prefix from the Hosted Zone ID
        this.hostedZoneId = props.hostedZone.hostedZoneId.replace(/^\/hostedzone\//, '');
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'AWS::CertificateManager::Certificate');
        const requestorFunction = new lambda.Function(this, 'CertificateRequestorFunction', {
            code: lambda.Code.fromAsset(path.resolve(__dirname, '..', 'lambda-packages', 'dns_validated_certificate_handler', 'lib')),
            handler: 'index.certificateRequestHandler',
            runtime: lambda.Runtime.NODEJS_12_X,
            timeout: cdk.Duration.minutes(15),
            role: props.customResourceRole,
        });
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['acm:RequestCertificate', 'acm:DescribeCertificate', 'acm:DeleteCertificate', 'acm:AddTagsToCertificate'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:GetChange'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:changeResourceRecordSets'],
            resources: [`arn:${cdk.Stack.of(requestorFunction).partition}:route53:::hostedzone/${this.hostedZoneId}`],
        }));
        const certificate = new cdk.CustomResource(this, 'CertificateRequestorResource', {
            serviceToken: requestorFunction.functionArn,
            properties: {
                DomainName: props.domainName,
                SubjectAlternativeNames: cdk.Lazy.list({ produce: () => props.subjectAlternativeNames }, { omitEmpty: true }),
                HostedZoneId: this.hostedZoneId,
                Region: props.region,
                Route53Endpoint: props.route53Endpoint,
                // Custom resources properties are always converted to strings; might as well be explict here.
                CleanupRecords: props.cleanupRoute53Records ? 'true' : undefined,
                Tags: cdk.Lazy.list({ produce: () => this.tags.renderTags() }),
            },
        });
        this.certificateArn = certificate.getAtt('Arn').toString();
    }
    validate() {
        const errors = [];
        // Ensure the zone name is a parent zone of the certificate domain name
        if (!cdk.Token.isUnresolved(this.normalizedZoneName) &&
            this.domainName !== this.normalizedZoneName &&
            !this.domainName.endsWith('.' + this.normalizedZoneName)) {
            errors.push(`DNS zone ${this.normalizedZoneName} is not authoritative for certificate domain name ${this.domainName}`);
        }
        return errors;
    }
}
exports.DnsValidatedCertificate = DnsValidatedCertificate;
_a = JSII_RTTI_SYMBOL_1;
DnsValidatedCertificate[_a] = { fqn: "@aws-cdk/aws-certificatemanager.DnsValidatedCertificate", version: "1.157.0" };
//# sourceMappingURL=data:application/json;base64,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