"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = exports.CertificateValidation = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const certificate_base_1 = require("./certificate-base");
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * How to validate a certificate
 */
class CertificateValidation {
    /** @param props Certification validation properties */
    constructor(props) {
        var _c;
        this.props = props;
        this.method = (_c = props.method) !== null && _c !== void 0 ? _c : ValidationMethod.EMAIL;
    }
    /**
     * Validate the certificate with DNS
     *
     * IMPORTANT: If `hostedZone` is not specified, DNS records must be added
     * manually and the stack will not complete creating until the records are
     * added.
     *
     * @param hostedZone the hosted zone where DNS records must be created
     */
    static fromDns(hostedZone) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZone,
        });
    }
    /**
     * Validate the certificate with automatically created DNS records in multiple
     * Amazon Route 53 hosted zones.
     *
     * @param hostedZones a map of hosted zones where DNS records must be created
     * for the domains in the certificate
     */
    static fromDnsMultiZone(hostedZones) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZones,
        });
    }
    /**
     * Validate the certificate with Email
     *
     * IMPORTANT: if you are creating a certificate as part of your stack, the stack
     * will not complete creating until you read and follow the instructions in the
     * email that you will receive.
     *
     * ACM will send validation emails to the following addresses:
     *
     *  admin@domain.com
     *  administrator@domain.com
     *  hostmaster@domain.com
     *  postmaster@domain.com
     *  webmaster@domain.com
     *
     * For every domain that you register.
     *
     * @param validationDomains a map of validation domains to use for domains in the certificate
     */
    static fromEmail(validationDomains) {
        return new CertificateValidation({
            method: ValidationMethod.EMAIL,
            validationDomains,
        });
    }
}
exports.CertificateValidation = CertificateValidation;
_a = JSII_RTTI_SYMBOL_1;
CertificateValidation[_a] = { fqn: "@aws-cdk/aws-certificatemanager.CertificateValidation", version: "1.157.0" };
/**
 * A certificate managed by AWS Certificate Manager
 */
class Certificate extends certificate_base_1.CertificateBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_certificatemanager_CertificateProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        let validation;
        if (props.validation) {
            validation = props.validation;
        }
        else { // Deprecated props
            if (props.validationMethod === ValidationMethod.DNS) {
                validation = CertificateValidation.fromDns();
            }
            else {
                validation = CertificateValidation.fromEmail(props.validationDomains);
            }
        }
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: renderDomainValidation(validation, allDomainNames),
            validationMethod: validation.method,
        });
        this.certificateArn = cert.ref;
    }
    /**
     * Import a certificate
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends certificate_base_1.CertificateBase {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
_b = JSII_RTTI_SYMBOL_1;
Certificate[_b] = { fqn: "@aws-cdk/aws-certificatemanager.Certificate", version: "1.157.0" };
/**
 * Method used to assert ownership of the domain
 */
var ValidationMethod;
(function (ValidationMethod) {
    /**
     * Send email to a number of email addresses associated with the domain
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-email.html
     */
    ValidationMethod["EMAIL"] = "EMAIL";
    /**
     * Validate ownership by adding appropriate DNS records
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-dns.html
     */
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
// eslint-disable-next-line max-len
function renderDomainValidation(validation, domainNames) {
    var _c, _d, _e;
    const domainValidation = [];
    switch (validation.method) {
        case ValidationMethod.DNS:
            for (const domainName of getUniqueDnsDomainNames(domainNames)) {
                const hostedZone = (_d = (_c = validation.props.hostedZones) === null || _c === void 0 ? void 0 : _c[domainName]) !== null && _d !== void 0 ? _d : validation.props.hostedZone;
                if (hostedZone) {
                    domainValidation.push({ domainName, hostedZoneId: hostedZone.hostedZoneId });
                }
            }
            break;
        case ValidationMethod.EMAIL:
            for (const domainName of domainNames) {
                const validationDomain = (_e = validation.props.validationDomains) === null || _e === void 0 ? void 0 : _e[domainName];
                if (!validationDomain && core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                domainValidation.push({ domainName, validationDomain: validationDomain !== null && validationDomain !== void 0 ? validationDomain : util_1.apexDomain(domainName) });
            }
            break;
        default:
            throw new Error(`Unknown validation method ${validation.method}`);
    }
    return domainValidation.length !== 0 ? domainValidation : undefined;
}
/**
 * Removes wildcard domains (*.example.com) where the base domain (example.com) is present.
 * This is because the DNS validation treats them as the same thing, and the automated CloudFormation
 * DNS validation errors out with the duplicate records.
 */
function getUniqueDnsDomainNames(domainNames) {
    return domainNames.filter(domain => {
        return core_1.Token.isUnresolved(domain) || !domain.startsWith('*.') || !domainNames.includes(domain.replace('*.', ''));
    });
}
//# sourceMappingURL=data:application/json;base64,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