"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addToDeadLetterQueueResourcePolicy = exports.addLambdaPermission = exports.singletonEventRole = exports.bindBaseTargetConfig = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * Bind props to base rule target config.
 * @internal
 */
function bindBaseTargetConfig(props) {
    let { deadLetterQueue, retryAttempts, maxEventAge } = props;
    return {
        deadLetterConfig: deadLetterQueue ? { arn: deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueArn } : undefined,
        retryPolicy: retryAttempts || maxEventAge
            ? {
                maximumRetryAttempts: retryAttempts,
                maximumEventAgeInSeconds: maxEventAge === null || maxEventAge === void 0 ? void 0 : maxEventAge.toSeconds({ integral: true }),
            }
            : undefined,
    };
}
exports.bindBaseTargetConfig = bindBaseTargetConfig;
/**
 * Obtain the Role for the EventBridge event
 *
 * If a role already exists, it will be returned. This ensures that if multiple
 * events have the same target, they will share a role.
 * @internal
 */
function singletonEventRole(scope, policyStatements) {
    const id = 'EventsRole';
    const existing = scope.node.tryFindChild(id);
    if (existing) {
        return existing;
    }
    const role = new iam.Role(scope, id, {
        roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
        assumedBy: new iam.ServicePrincipal('events.amazonaws.com'),
    });
    policyStatements.forEach(role.addToPolicy.bind(role));
    return role;
}
exports.singletonEventRole = singletonEventRole;
/**
 * Allows a Lambda function to be called from a rule
 * @internal
 */
function addLambdaPermission(rule, handler) {
    let scope;
    let node = handler.permissionsNode;
    let permissionId = `AllowEventRule${core_1.Names.nodeUniqueId(rule.node)}`;
    if (rule instanceof core_2.Construct) {
        // Place the Permission resource in the same stack as Rule rather than the Function
        // This is to reduce circular dependency when the lambda handler and the rule are across stacks.
        scope = rule;
        node = rule.node;
        permissionId = `AllowEventRule${core_1.Names.nodeUniqueId(handler.node)}`;
    }
    if (!node.tryFindChild(permissionId)) {
        handler.addPermission(permissionId, {
            scope,
            action: 'lambda:InvokeFunction',
            principal: new iam.ServicePrincipal('events.amazonaws.com'),
            sourceArn: rule.ruleArn,
        });
    }
}
exports.addLambdaPermission = addLambdaPermission;
/**
 * Allow a rule to send events with failed invocation to an Amazon SQS queue.
 * @internal
 */
function addToDeadLetterQueueResourcePolicy(rule, queue) {
    if (!sameEnvDimension(rule.env.region, queue.env.region)) {
        throw new Error(`Cannot assign Dead Letter Queue in region ${queue.env.region} to the rule ${core_1.Names.nodeUniqueId(rule.node)} in region ${rule.env.region}. Both the queue and the rule must be in the same region.`);
    }
    // Skip Resource Policy creation if the Queue is not in the same account.
    // There is no way to add a target onto an imported rule, so we can assume we will run the following code only
    // in the account where the rule is created.
    if (sameEnvDimension(rule.env.account, queue.env.account)) {
        const policyStatementId = `AllowEventRule${core_1.Names.nodeUniqueId(rule.node)}`;
        queue.addToResourcePolicy(new iam.PolicyStatement({
            sid: policyStatementId,
            principals: [new iam.ServicePrincipal('events.amazonaws.com')],
            effect: iam.Effect.ALLOW,
            actions: ['sqs:SendMessage'],
            resources: [queue.queueArn],
            conditions: {
                ArnEquals: {
                    'aws:SourceArn': rule.ruleArn,
                },
            },
        }));
    }
    else {
        core_1.Annotations.of(rule).addWarning(`Cannot add a resource policy to your dead letter queue associated with rule ${rule.ruleName} because the queue is in a different account. You must add the resource policy manually to the dead letter queue in account ${queue.env.account}.`);
    }
}
exports.addToDeadLetterQueueResourcePolicy = addToDeadLetterQueueResourcePolicy;
/**
 * Whether two string probably contain the same environment dimension (region or account)
 *
 * Used to compare either accounts or regions, and also returns true if both
 * are unresolved (in which case both are expted to be "current region" or "current account").
 * @internal
 */
function sameEnvDimension(dim1, dim2) {
    return [core_1.TokenComparison.SAME, core_1.TokenComparison.BOTH_UNRESOLVED].includes(core_1.Token.compareStrings(dim1, dim2));
}
//# sourceMappingURL=data:application/json;base64,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