"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsQueue = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const util_1 = require("./util");
/**
 * Use an SQS Queue as a target for Amazon EventBridge rules.
 *
 * @example
 *   /// fixture=withRepoAndSqsQueue
 *   // publish to an SQS queue every time code is committed
 *   // to a CodeCommit repository
 *   repository.onCommit('onCommit', { target: new targets.SqsQueue(queue) });
 *
 */
class SqsQueue {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        jsiiDeprecationWarnings._aws_cdk_aws_events_targets_SqsQueueProps(props);
        if (props.messageGroupId !== undefined && !queue.fifo) {
            throw new Error('messageGroupId cannot be specified for non-FIFO queues');
        }
    }
    /**
     * Returns a RuleTarget that can be used to trigger this SQS queue as a
     * result from an EventBridge event.
     *
     * @see https://docs.aws.amazon.com/eventbridge/latest/userguide/resource-based-policies-eventbridge.html#sqs-permissions
     */
    bind(rule, _id) {
        // Only add the rule as a condition if the queue is not encrypted, to avoid circular dependency. See issue #11158.
        const principalOpts = this.queue.encryptionMasterKey ? {} : {
            conditions: {
                ArnEquals: { 'aws:SourceArn': rule.ruleArn },
            },
        };
        // deduplicated automatically
        this.queue.grantSendMessages(new iam.ServicePrincipal('events.amazonaws.com', principalOpts));
        if (this.props.deadLetterQueue) {
            util_1.addToDeadLetterQueueResourcePolicy(rule, this.props.deadLetterQueue);
        }
        return {
            ...util_1.bindBaseTargetConfig(this.props),
            arn: this.queue.queueArn,
            input: this.props.message,
            targetResource: this.queue,
            sqsParameters: this.props.messageGroupId ? { messageGroupId: this.props.messageGroupId } : undefined,
        };
    }
}
exports.SqsQueue = SqsQueue;
_a = JSII_RTTI_SYMBOL_1;
SqsQueue[_a] = { fqn: "@aws-cdk/aws-events-targets.SqsQueue", version: "1.147.0" };
//# sourceMappingURL=data:application/json;base64,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