"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodePipelineDockerfileLinterAction = void 0;
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_codepipeline_1 = require("@aws-cdk/aws-codepipeline");
const aws_codepipeline_actions_1 = require("@aws-cdk/aws-codepipeline-actions");
const aws_iam_1 = require("@aws-cdk/aws-iam");
class CodePipelineDockerfileLinterAction extends aws_codepipeline_actions_1.Action {
    constructor(props) {
        super({
            ...props,
            category: aws_codepipeline_1.ActionCategory.TEST,
            provider: 'CodeBuild',
            artifactBounds: {
                minInputs: 1,
                maxInputs: 1,
                minOutputs: 0,
                maxOutputs: 0,
            },
            inputs: [props.input],
        });
        this.props = props;
    }
    bound(scope, _stage, options) {
        const buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_4_0;
        const version = this.props.version || 'v1.18.0';
        const project = new aws_codebuild_1.PipelineProject(scope, 'LinterProject', {
            cache: aws_codebuild_1.Cache.local(aws_codebuild_1.LocalCacheMode.DOCKER_LAYER),
            environment: {
                buildImage,
                computeType: this.props.computeType || buildImage.defaultComputeType,
                privileged: true,
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    pre_build: {
                        commands: ['echo Pulling the hadolint docker image', `docker pull hadolint/hadolint:${version}`],
                    },
                    build: {
                        commands: [],
                        finally: ['echo Scan started on `date`', `result=$(docker run --rm -i hadolint/hadolint:${version} hadolint -f json - < Dockerfile)`],
                    },
                    post_build: {
                        commands: ['if [ "$result" != "[]" ]; then echo $result | jq .; else echo "Awesome! No findings!"; fi', 'echo Scan completed on `date`'],
                    },
                },
            }),
        });
        // grant the Pipeline role the required permissions to this Project
        options.role.addToPolicy(new aws_iam_1.PolicyStatement({
            resources: [project.projectArn],
            actions: ['codebuild:BatchGetBuilds', 'codebuild:StartBuild', 'codebuild:StopBuild'],
        }));
        // allow the Project access to the Pipeline's artifact Bucket
        // but only if the project is not imported
        // (ie., has a role) - otherwise, the IAM library throws an error
        if (project.role) {
            options.bucket.grantRead(project);
        }
        return {
            configuration: {
                ProjectName: project.projectName,
            },
        };
    }
}
exports.CodePipelineDockerfileLinterAction = CodePipelineDockerfileLinterAction;
//# sourceMappingURL=data:application/json;base64,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