""""""
# ----------------------------------------------------------------------------
# Description    : SCPI interface
# Git repository : https://gitlab.com/qblox/packages/software/qblox_instruments.git
# Copyright (C) Qblox BV (2020)
# ----------------------------------------------------------------------------


# ----------------------------------------------------------------------------
# THIS FILE IS AUTOMATICALLY GENERATED. DO NOT MODIFY THIS FILE MANUALLY!
# ----------------------------------------------------------------------------


# -- include ------------------------------------------------------------------

import sys
import struct
import re
import json
from functools import wraps
from typing    import Any, Dict, List, Optional, Tuple

# Add IEEE488.2 support
from qblox_instruments.ieee488_2 import Ieee488_2, Transport


# -- decorator-----------------------------------------------------------------

def scpi_error_check(func):
    """
    Decorator that catches and checks for errors on an SCPI call.

    Parameters
    ----------
    func
        Class method that performs an SCPI call

    Returns
    ----------

    Raises
    ----------
    RuntimeError
        An error was found in system error.
    """

    @wraps(func)
    def decorator_wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except OSError:
            raise
        except Exception as err:
            args[0]._check_error_queue(err)
        finally:
            args[0]._check_error_queue()
    return decorator_wrapper


# -- class --------------------------------------------------------------------

class Cluster(Ieee488_2):
    """
    This interface provides an API for the mandatory and required SCPI calls and adds
    Pulsar related functionality
    (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).
    """

    # -------------------------------------------------------------------------
    def __init__(self, transport: Transport, debug: int=0):
        """
        Creates SCPI interface object.

        Parameters
        ----------
        transport : Transport
            Transport class responsible for the lowest level of communication
            (e.g. Ethernet).
        debug : int
            Debug level (0 = normal, 1 = no version check, >1 = no version or
            error checking).

        Returns
        ----------

        Raises
        ----------
        ConnectionError
            Debug level is 0 and there is a device or version mismatch.
        """

        # Store parameters for later use.
        self._debug = debug

        # Initialize parent class.
        super(Cluster, self).__init__(transport)

        if self._debug == 0:
            # Check if device and build are compatible.
            _, device, *_, build = self._read("*IDN?").split(',')
            if device.replace("Cluster", "Pulsar") != 'Pulsar MM':
                raise ConnectionError("This class expects a Cluster MM device, but found a {}. Please use the right device class.".format(device))
            build_ref = "fwVersion=0.2.2 fwBuild=28/07/2022-16:02:41 fwHash=0x22F638E7 fwDirty=0 kmodVersion=0.2.2 kmodBuild=28/07/2022-16:02:41 kmodHash=0x22F638E7 kmodDirty=0 swVersion=0.2.2 swBuild=28/07/2022-16:02:41 swHash=0x22F638E7 swDirty=0 cfgManVersion=0.2.0 cfgManBuild=28/07/2022-16:02:41 cfgManHash=0x22F638E7 cfgManDirty=0"
            if build != build_ref:
                build     = re.split(' |=', build)
                build_ref = re.split(' |=', build_ref)
                raise ConnectionError("Qblox Instruments package version is not compatible with device version:\n" + \
                                      "\n" + \
                                      "                        {:<25} {}\n".format("Expected:", "Actual:")  + \
                                      "Firmware;      Version: {:<25} {}\n".format(build_ref[1],  build[1])  + \
                                      "               Date:    {:<25} {}\n".format(build_ref[3],  build[3])  + \
                                      "               Hash:    {:<25} {}\n".format(build_ref[5],  build[5])  + \
                                      "               Dirty:   {:<25} {}\n".format(build_ref[7],  build[7])  + \
                                      "Kernel module; Version: {:<25} {}\n".format(build_ref[9],  build[9])  + \
                                      "               Date:    {:<25} {}\n".format(build_ref[11], build[11]) + \
                                      "               Hash:    {:<25} {}\n".format(build_ref[13], build[13]) + \
                                      "               Dirty:   {:<25} {}\n".format(build_ref[15], build[15]) + \
                                      "Application;   Version: {:<25} {}\n".format(build_ref[17], build[17]) + \
                                      "               Date:    {:<25} {}\n".format(build_ref[19], build[19]) + \
                                      "               Hash:    {:<25} {}\n".format(build_ref[21], build[21]) + \
                                      "               Dirty:   {:<25} {}\n".format(build_ref[23], build[23]) + \
                                      "\n" + \
                                      "Please update your device's firmware or the Qblox Instruments package.")

            # Clear SCPI error queue.
            while int(self._read('SYSTem:ERRor:COUNt?')) != 0:
                self._read('SYSTem:ERRor:NEXT?')

    # -------------------------------------------------------------------------
    def get_system_error(self) -> str:
        """
        Get system error from queue (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        str
            System error description string.

        """

        # SCPI call
        var0 = self._read('SYSTem:ERRor:NEXT?')

        return var0

    # -------------------------------------------------------------------------
    def get_num_system_error(self) -> int:
        """
        Get number of system errors (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Current number of system errors.

        """

        # SCPI call
        var0 = self._read('SYSTem:ERRor:COUNt?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_idn(self) -> str:
        """
        Get device identity and build information.

        Parameters
        ----------
        None

        Returns
        -------
        str
            Concatenated list of strings separated by the semicolon character. The IDN consists of four strings respectively ordered as:

            - Manufacturer
            - Model
            - Serial number
            - Build information

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*IDN?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_system_version(self) -> str:
        """
        Get SCPI system version (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        str
            SCPI system version.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('SYSTem:VERSion?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_scpi_commands(self) -> str:
        """
        Get SCPI commands.

        Parameters
        ----------
        None

        Returns
        -------
        str
            Concatenated list of strings separated by the semicolon character. Each command consists of nine sections respectively ordered as:

             - SCPI command pattern
             - SCPI input type
             - SCPI output type
             - Python function
             - Python input types (comma separated)
             - Python input variable names (comma separated)
             - Python output types (comma separated)
             - User access level (0 = public, >=1 = private)
             - Comment


        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*CMDS?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _temp_file_new(self) -> int:
        """
        Creates a temporary file for the purpose of sending a large batch of data to the instrument, such as an update file.

        Parameters
        ----------
        None

        Returns
        -------
        int
            Handle by which the file can be identified.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('TMPFILe:NEW?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _temp_file_delete(self, handle: int) -> None:
        """
        Deletes a temporary file.

        Parameters
        ----------
        handle : int
            Temporary file handle.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('TMPFILe:DELete {}'.format(handle))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _temp_file_size(self, handle: int) -> int:
        """
        Returns the size of a temporary file in bytes.

        Parameters
        ----------
        handle : int
            Temporary file handle.

        Returns
        -------
        int
            Size of the file in bytes.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('TMPFILe:SIZe? {}'.format(handle))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _temp_file_block_count(self, handle: int) -> int:
        """
        Returns the number of blocks that can be read from a temporary file.

        Parameters
        ----------
        handle : int
            Temporary file handle.

        Returns
        -------
        int
            Number of blocks that can be read.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('TMPFILe:BLOCK:COUNT? {}'.format(handle))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _temp_file_block_read(self, handle: int, block: int) -> bytes:
        """
        Reads the nth block of a temporary file.

        Parameters
        ----------
        handle : int
            Temporary file handle.
        block : int
            Block index.

        Returns
        -------
        bytes
            Contents of the block.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        var0 = self._read_bin('TMPFILe:BLOCK:READ? {},{}'.format(handle, block))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _temp_file_append(self, handle: int, data: bytes) -> None:
        """
        Appends a block of data to a temporary file.

        Parameters
        ----------
        handle : int
            Temporary file handle.
        data : bytes
            Data to append.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'bytes'])

        # SCPI call
        self._write_bin('TMPFILe:APPend {},'.format(handle), data)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _task_poll(self, handle: int) -> float:
        """
        Polls for completion of the given task.

        Parameters
        ----------
        handle : int
            Task handle.

        Returns
        -------
        float
            Task progress from 0.0 to 1.0. 1.0 indicates completion.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('TASK:POLL? {}'.format(handle))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _task_get_result(self, handle: int) -> Any:
        """
        Retrieves the value returned by a task. The task must be complete. If the task yielded an error, the error will be returned by this call. The task handle is deleted once the value is returned.

        Parameters
        ----------
        handle : int
            Task handle.

        Returns
        -------
        any
            The result of the task.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read_bin('TASK:RESult? {}'.format(handle))

        return json.loads(var0.decode('utf-8'))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_name(self, name: str) -> None:
        """
        Sets the customer-specified name of the instrument. The name must not contain any newlines, backslashes, or double quotes.

        Parameters
        ----------
        name : str
            The new name for the device.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['str'])

        # SCPI call
        self._write('SYSTem:NAME "{}"'.format(name))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_name(self) -> str:
        """
        Returns the customer-specified name of the instrument.

        Parameters
        ----------
        None

        Returns
        -------
        str
            The name of the device.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('SYSTem:NAME?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _download_log(self, source: str, format: int) -> int:
        """
        Prepares a batch of log data for retrieval via temp_file_*(). This may take some time, so instead of returning immediately, this spawns a task. task_poll() may be used to poll the state of the task, once this returns True (task complete), use task_get_result() to get the temp_file handle.

        Parameters
        ----------
        source : str
            The log source. Must be "system" for the Linux system log,"cfg_man" for the configuration manager log, or "app" for the main application log.
        format : int
            The format. Specify a positive integer to get up to that number of lines from the log. Specify zero to get the complete current log file (older log entries may exist via logrotate, only the file currently being written is returned). Specify a negative number to get all available log data in .tgz format.

        Returns
        -------
        int
            Task handle.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['str', 'int'])

        # SCPI call
        var0 = self._read('SYSTem:LOG? "{}",{}'.format(source, format))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_ip_config(self, config: str) -> None:
        """
        Reconfigures the IP address of this device. The configuration will not go into effect until reboot() is called or the device is power-cycled.

        Parameters
        ----------
        config : str
            IP configuration. May be one of the following things:
             - an IPv4 address including prefix length, for example 192.168.0.2/24, - the string `dhcp` to enable IPv4 DHCP, - an IPv6 address including prefix length, for example 1:2::3:4/64, or - a semicolon-separated combination of an IPv4 configuration (IP address or `dhcp`) and an IPv6 address.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['str'])

        # SCPI call
        self._write('SYSTem:IPCONFig "{}"'.format(config))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_ip_config(self) -> str:
        """
        Returns the IP address configuration that will go into effect when the device reboots.

        Parameters
        ----------
        None

        Returns
        -------
        str
            IP configuration. Can be one of the following things:
             - an IPv4 address including prefix length, for example 192.168.0.2/24, - the string `dhcp` to enable IPv4 DHCP, - an IPv6 address including prefix length, for example 1:2::3:4/64, or - a semicolon-separated combination of an IPv4 configuration (IP address or `dhcp`) and an IPv6 address.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('SYSTem:IPCONFig?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_update_module_types(self) -> str:
        """
        Returns a comma-separated list of device types that may be updated in addition to our own device type in the form of cluster modules. Returns ``not_applicable`` if this is not applicable to this device type.

        Parameters
        ----------
        None

        Returns
        -------
        str
            Comma-separated list of device types.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('SYSTem:UPDATE:MODules?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _update(self, handle: int) -> int:
        """
        Updates the device with an update tarball that was previously uploaded using temp_file_*() commands. This command merely starts the update process, and returns a task handle that can polled for completion via task_poll(), at which point any errors that may have occurred must be retrieved using task_get_result(). The task will return the string "OK" if successful. If this happens, the device must be rebooted via the reboot() command or via a power cycle to fully apply the update. Do not power down the device while the update task is running.

        Parameters
        ----------
        handle : int
            Temporary file handle for the update tarball.

        Returns
        -------
        int
            Task handle.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SYSTem:UPDATE:PREPare? {}'.format(handle))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _rollback(self) -> int:
        """
        Rolls back the device to the previous version, if rollback information is available. This command merely starts the rollback process, and returns a task handle that can polled for completion via task_poll(), at which point any errors that may have occurred must be retrieved using task_get_result(). The task will return the string "OK" if successful. If this happens, the device must be rebooted via the reboot() command or via a power cycle to fully apply the rollback. Do not power down the device while the rollback task is running.

        Parameters
        ----------
        None

        Returns
        -------
        int
            Task handle.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('SYSTem:ROLLBACK:PREPare?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def reboot(self) -> None:
        """
        Reboots the instrument.

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('SYSTem:REBOOT')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _reset(self) -> None:
        """
        Reset Cluster and all modules and clear all status and event registers (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('*RST')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def clear(self) -> None:
        """
        Clear all status and event registers (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('*CLS')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_status_byte(self) -> int:
        """
        Get status byte register. Register is only cleared when feeding registers are cleared (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status byte register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*STB?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_service_request_enable(self, reg: int) -> None:
        """
        Set service request enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        reg : int
            Service request enable register.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('*SRE {}'.format(reg))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_service_request_enable(self) -> int:
        """
        Get service request enable register. The register is cleared after reading it (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Service request enable register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*SRE?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_standard_event_status_enable(self, reg: int) -> None:
        """
        Set standard event status enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        reg : int
            Standard event status enable register.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('*ESE {}'.format(reg))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_standard_event_status_enable(self) -> int:
        """
        Get standard event status enable register. The register is cleared after reading it (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Standard event status enable register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*ESE?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_standard_event_status(self) -> int:
        """
        Get standard event status register. The register is cleared after reading it (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Standard event status register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*ESR?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_operation_complete(self) -> None:
        """
        Set device in operation complete query active state (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('*OPC')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_operation_complete(self) -> bool:
        """
        Get operation complete state (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        bool
            Operation complete state (False = running, True = completed).

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*OPC?')

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def test(self) -> bool:
        """
        Run self-test. Currently not implemented (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        bool
            Test result (False = failed, True = success).

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('*TST?')

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def wait(self) -> None:
        """
        Wait until operations completed before continuing (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('*WAI')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def preset_system_status(self) -> None:
        """
        Preset system status registers. Connects general system status flags for PLL unlock and temperature out-of-range indications
        to event status enable, status questionable temperature and status questionable frequency registers respectively
        (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('STATus:PRESet')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_questionable_condition(self) -> int:
        """
        Get status questionable condition register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status questionable condition register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:CONDition?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_questionable_event(self) -> int:
        """
        Get status questionable event register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status questionable event register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:EVENt?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_questionable_enable(self, reg: int) -> None:
        """
        Set status questionable enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        reg : int
            Status questionable enable register.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('STATus:QUEStionable:ENABle {}'.format(reg))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_questionable_enable(self) -> int:
        """
        Get status questionable enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status questionable enable register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:ENABle?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_operation_condition(self) -> int:
        """
        Get status operation condition register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status operation condition register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:OPERation:CONDition?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_operation_events(self) -> int:
        """
        Get status operation event register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status operation event register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:OPERation:EVENt?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def set_operation_enable(self, reg: int) -> None:
        """
        Set status operation enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        reg : int
            Status operation enable register.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('STATus:OPERation:ENABle {}'.format(reg))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_operation_enable(self) -> int:
        """
        Get status operation enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        None

        Returns
        -------
        int
            Status operation enable register.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:OPERation:ENABle?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def identify(self) -> None:
        """
        Toggle frontpanel LEDs to visually identify the instrument.

        Parameters
        ----------
        None

        Returns
        -------
        None

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        self._write('*IDENtify')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_system_state(self) -> str:
        """
        Get general system state.

        Parameters
        ----------
        None

        Returns
        -------
        str
            Concatenated list of strings separated by the semicolon character. Status is indicated by one status string and an optional number of flags respectively ordered as:

            :Status:

             - ``OKAY``: System is okay.
             - ``CRITICAL``: An error indicated by the flags occurred, but has been resolved.
             - ``ERROR``: An error indicated by the flags is occurring.

            :Flags:

             - ``CARRIER PLL UNLOCKED``
             - ``FPGA PLL UNLOCKED``
             - ``LO PLL UNLOCKED`` (only for RF-modules)
             - ``FPGA TEMPERATURE OUT-OF-RANGE``
             - ``CARRIER TEMPERATURE OUT-OF-RANGE``
             - ``AFE TEMPERATURE OUT-OF-RANGE``
             - ``LO TEMPERATURE OUT-OF-RANGE``
             - ``BACKPLANE TEMPERATURE OUT-OF-RANGE`` (only for Cluster)


        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:GENeral:STATE?')

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_reference_source(self, internal: bool) -> None:
        """
        Set reference (10MHz) clock source.

        Parameters
        ----------
        internal : bool
            Reference clock source (False = External, True = Internal).

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['bool'])

        # SCPI call
        self._write('STATus:QUEStionable:FREQuency:REFerence:SRC {}'.format(0 if internal == False else 1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_reference_source(self) -> bool:
        """
        Get reference (10MHz) clock source.

        Parameters
        ----------
        None

        Returns
        -------
        bool
            Reference clock source (False = External, True = Internal).

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:FREQuency:REFerence:SRC?')

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_current_fpga_temperature(self, slot: int) -> float:
        """
        Get current FPGA junction temperature (inside device).

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            Current FPGA junction temperature.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:QUEStionable:TEMPerature:FPGA:CURRent?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_maximum_fpga_temperature(self, slot: int) -> float:
        """
        Get maximum FPGA junction temperature since boot or clear (inside device).

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            Maximum FPGA junction temperature.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:QUEStionable:TEMPerature:FPGA:MAXimum?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_current_carrier_temperature(self, slot: int) -> float:
        """
        Get current carrier board temperature (inside device).

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            Current carrier board temperature.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:QUEStionable:TEMPerature:CARRier:CURRent?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_maximum_carrier_temperature(self, slot: int) -> float:
        """
        Get maximum carrier board temperature since boot or clear (inside device).

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            Maximum carrier board temperature.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:QUEStionable:TEMPerature:CARRier:MAXimum?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_current_bp_temperature_0(self) -> float:
        """
        Get current backplane board temperature from sensor 0 (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Current backplane board temperature from sensor 0.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:BP0:CURRent?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_maximum_bp_temperature_0(self) -> float:
        """
        Get maximum backplane board temperature from sensor 0 since boot or clear (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Maximum backplane board temperature from sensor 0.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:BP0:MAXimum?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_current_bp_temperature_1(self) -> float:
        """
        Get current backplane board temperature from sensor 1 (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Current backplane board temperature from sensor 1.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:BP1:CURRent?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_maximum_bp_temperature_1(self) -> float:
        """
        Get maximum backplane board temperature from sensor 1 since boot or clear (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Maximum backplane board temperature from sensor 1.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:BP1:MAXimum?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_current_bp_temperature_2(self) -> float:
        """
        Get current backplane board temperature from sensor 2 (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Current backplane board temperature from sensor 2.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:BP2:CURRent?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_maximum_bp_temperature_2(self) -> float:
        """
        Get maximum backplane board temperature from sensor 2 since boot or clear (inside device).

        Parameters
        ----------
        None

        Returns
        -------
        float
            Maximum backplane board temperature from sensor 2.

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('STATus:QUEStionable:TEMPerature:BP2:MAXimum?')

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_modules_present(self) -> int:
        """
        Get an indication of module presence for each slot of the Cluster.

        Parameters
        ----------
        None

        Returns
        -------
        int
            Module present indication where each bit represents a single slot and the LSB is slot 1 (False = not present, True = present).

        Raises
        ------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read('BP:MODules?')

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_mods_info(self) -> Any:
        """
        Get information about the available modules

        Parameters
        ----------
        None

        Returns
        -------
        any
            Module types available per slot

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # SCPI call
        var0 = self._read_bin('*MODS?')

        return json.loads(var0.decode('utf-8'))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _slot_reset(self, slot: int) -> None:
        """
        Reset the module in the slot specified by slot_index,and clear all its status and event registers (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('SLOT{}:RST'.format(slot))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_slot_idn(self, slot: int) -> str:
        """
        Get IDN from given slot number

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        str
            Concatenated list of strings separated by the semicolon character, related to the slot provided in the argument. The IDN consists of four strings respectively ordered as:

            - Manufacturer
            - Model
            - Serial number
            - Build information

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:IDN?'.format(slot))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_assembler_status(self, slot: int) -> bool:
        """
        Get assembler status. Refer to the assembler log to get more information regarding the assembler result.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        bool
            Assembler status (False = failed, True = success).

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:ASSEMbler:SUCCess?'.format(slot))

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_assembler_log(self, slot: int) -> str:
        """
        Get assembler log.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        str
            Assembler log.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read_bin('SLOT{}:STATus:ASSEMbler:LOG?'.format(slot))

        return var0.decode('utf-8', 'ignore')

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_in_amp_gain_0(self, slot: int) -> int:
        """
        Get input amplifier gain for input 0.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        int
            Current input amplifier gain for input 0.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:GAIN:INAMP0?'.format(slot))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_in_amp_gain_1(self, slot: int) -> int:
        """
        Get input amplifier gain for input 0.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        int
            Current input amplifier gain for input 0.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:GAIN:INAMP1?'.format(slot))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_out_amp_offset_0(self, slot: int) -> float:
        """
        Get output amplifier offset for output 0.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            Current output amplifier offset for output 0.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:OFFSet:OUTAMP0?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_out_amp_offset_1(self, slot: int) -> float:
        """
        Get output amplifier offset for output 1.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            Current output amplifier offset for output 1.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:OFFSet:OUTAMP1?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_out_amp_offset_2(self, slot: int) -> float:
        """
        Get output amplifier offset for output 2.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            Current output amplifier offset for output 2.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:OFFSet:OUTAMP2?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_out_amp_offset_3(self, slot: int) -> float:
        """
        Get output amplifier offset for output 3.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            Current output amplifier offset for output 3.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:OFFSet:OUTAMP3?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_dac_offset_0(self, slot: int) -> float:
        """
        Get dac offset for output 0.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            Current dac offset for output 0.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:OFFSet:DAC0?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_dac_offset_1(self, slot: int) -> float:
        """
        Get dac offset for output 1.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            Current dac offset for output 1.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:OFFSet:DAC1?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_dac_offset_2(self, slot: int) -> float:
        """
        Get dac offset for output 2.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            Current dac offset for output 2.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:OFFSet:DAC2?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_dac_offset_3(self, slot: int) -> float:
        """
        Get dac offset for output 3.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            Current dac offset for output 3.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:OFFSet:DAC3?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_scope_config(self, slot: int) -> bytes:
        """
        Get Set configuration of the scope acquisition. The configuration consists of multiple parameters in a C struct format. If an invalid sequencer index is given or the configation struct does not have the correct format, an error is set in system error..

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        bytes
            Current configuration struct.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read_bin('SLOT{}:ACQ:SCOpe:CONFiguration?'.format(slot))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_sequencer_channel_map(self, slot: int, sequencer: int) -> bytes:
        """
        Get channel map of the indexed sequencer. The channel map consists of list of associated output channels for the given sequencer index. If an invalid sequencer index is given or the channel map is not valid, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.

        Returns
        -------
        bytes
            Current Sequencer index.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        var0 = self._read_bin('SLOT{}:SEQuencer{}:CHANnelmap?'.format(slot, sequencer))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_sequencer_config(self, slot: int, sequencer: int) -> bytes:
        """
        Get configuration of the indexed sequencer. The configuration consists of multiple parameters in a C struct format. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.

        Returns
        -------
        bytes
            Current Configuration struct.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        var0 = self._read_bin('SLOT{}:SEQuencer{}:CONFiguration?'.format(slot, sequencer))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_enable_0(self, slot: int) -> bool:
        """
        Get LO 0 status.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        bool
            Current LO 0 status.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:LO0:ENAble?'.format(slot))

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_enable_1(self, slot: int) -> bool:
        """
        Get LO 1 status.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        bool
            Current LO 1 status.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:LO1:ENAble?'.format(slot))

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_in_att_0(self, slot: int) -> int:
        """
        Get input attenuator.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        int
            Attenuation in dB in a range of 0 to 30 in steps of 2dB.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:ATT:IN0?'.format(slot))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_out_att_0(self, slot: int) -> int:
        """
        Get output attenuator for output 0.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        int
            Attenuation in dB in a range of 0 to 60 in steps of 2dB.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:ATT:OUT0?'.format(slot))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_out_att_1(self, slot: int) -> int:
        """
        Get output attenuator for output 1.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        int
            Attenuation in dB in a range of 0 to 60 in steps of 2dB.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:AFE:ATT:OUT1?'.format(slot))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_freq_0(self, slot: int) -> int:
        """
        Get LO Frequency for output 0.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        int
            Frequency in Hertz

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:QUEStionable:FREQuency:LO0?'.format(slot))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_freq_1(self, slot: int) -> int:
        """
        Get LO Frequency for output 1.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        int
            Frequency in Hertz

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:QUEStionable:FREQuency:LO1?'.format(slot))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_pwr_0(self, slot: int) -> int:
        """
        Get LO Power for output 0.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        int
            Power in dBm

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:QUEStionable:POWer:LO0?'.format(slot))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_pwr_1(self, slot: int) -> int:
        """
        Get LO Power for output 1.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        int
            Power in dBm

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:QUEStionable:POWer:LO1?'.format(slot))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_in_amp_gain_0(self, slot: int, in_amp_gain_0: int) -> None:
        """
        Set input amplifier gain for input 0.

        Parameters
        ----------
        slot : int
            slot index.
        in_amp_gain_0 : int
            Current input amplifier gain for input 0 int.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        self._write('SLOT{}:AFE:GAIN:INAMP0 {}'.format(slot, in_amp_gain_0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_in_amp_gain_1(self, slot: int, in_amp_gain_1: int) -> None:
        """
        Set input amplifier gain for input 0.

        Parameters
        ----------
        slot : int
            slot index.
        in_amp_gain_1 : int
            Current input amplifier gain for input 0 int.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        self._write('SLOT{}:AFE:GAIN:INAMP1 {}'.format(slot, in_amp_gain_1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_out_amp_offset_0(self, slot: int, out_amp_offset_0: float) -> None:
        """
        Set output amplifier offset for output 0.

        Parameters
        ----------
        slot : int
            slot index.
        out_amp_offset_0 : float
            Current output amplifier offset for output 0 float.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'float'])

        # SCPI call
        self._write('SLOT{}:AFE:OFFSet:OUTAMP0 {}'.format(slot, out_amp_offset_0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_out_amp_offset_1(self, slot: int, out_amp_offset_1: float) -> None:
        """
        Set output amplifier offset for output 1.

        Parameters
        ----------
        slot : int
            slot index.
        out_amp_offset_1 : float
            Current output amplifier offset for output 1 float.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'float'])

        # SCPI call
        self._write('SLOT{}:AFE:OFFSet:OUTAMP1 {}'.format(slot, out_amp_offset_1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_out_amp_offset_2(self, slot: int, out_amp_offset_2: float) -> None:
        """
        Set output amplifier offset for output 2.

        Parameters
        ----------
        slot : int
            slot index.
        out_amp_offset_2 : float
            Current output amplifier offset for output 2 float.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'float'])

        # SCPI call
        self._write('SLOT{}:AFE:OFFSet:OUTAMP2 {}'.format(slot, out_amp_offset_2))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_out_amp_offset_3(self, slot: int, out_amp_offset_3: float) -> None:
        """
        Set output amplifier offset for output 3.

        Parameters
        ----------
        slot : int
            slot index.
        out_amp_offset_3 : float
            Current output amplifier offset for output 3 float.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'float'])

        # SCPI call
        self._write('SLOT{}:AFE:OFFSet:OUTAMP3 {}'.format(slot, out_amp_offset_3))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_dac_offset_0(self, slot: int, dac_offset_0: float) -> None:
        """
        Set dac offset for output 0.

        Parameters
        ----------
        slot : int
            slot index.
        dac_offset_0 : float
            Current dac offset for output 0 float.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'float'])

        # SCPI call
        self._write('SLOT{}:AFE:OFFSet:DAC0 {}'.format(slot, dac_offset_0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_dac_offset_1(self, slot: int, dac_offset_1: float) -> None:
        """
        Set dac offset for output 1.

        Parameters
        ----------
        slot : int
            slot index.
        dac_offset_1 : float
            Current dac offset for output 1 float.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'float'])

        # SCPI call
        self._write('SLOT{}:AFE:OFFSet:DAC1 {}'.format(slot, dac_offset_1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_dac_offset_2(self, slot: int, dac_offset_2: float) -> None:
        """
        Set dac offset for output 2.

        Parameters
        ----------
        slot : int
            slot index.
        dac_offset_2 : float
            Current dac offset for output 2 float.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'float'])

        # SCPI call
        self._write('SLOT{}:AFE:OFFSet:DAC2 {}'.format(slot, dac_offset_2))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_dac_offset_3(self, slot: int, dac_offset_3: float) -> None:
        """
        Set dac offset for output 3.

        Parameters
        ----------
        slot : int
            slot index.
        dac_offset_3 : float
            Current dac offset for output 3 float.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'float'])

        # SCPI call
        self._write('SLOT{}:AFE:OFFSet:DAC3 {}'.format(slot, dac_offset_3))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_acq_scope_config(self, slot: int, acq_scope_config: bytes) -> None:
        """
        Set Set configuration of the scope acquisition. The configuration consists of multiple parameters in a C struct format. If an invalid sequencer index is given or the configation struct does not have the correct format, an error is set in system error..

        Parameters
        ----------
        slot : int
            slot index.
        acq_scope_config : bytes
            Current configuration struct bytes.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'bytes'])

        # SCPI call
        self._write_bin('SLOT{}:ACQ:SCOpe:CONFiguration '.format(slot), acq_scope_config)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_sequencer_channel_map(self, slot: int, sequencer: int, sequencer_channel_map: bytes) -> None:
        """
        Set channel map of the indexed sequencer. The channel map consists of list of associated output channels for the given sequencer index. If an invalid sequencer index is given or the channel map is not valid, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        sequencer_channel_map : bytes
            Current Sequencer index bytes.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'bytes'])

        # SCPI call
        self._write_bin('SLOT{}:SEQuencer{}:CHANnelmap '.format(slot, sequencer), sequencer_channel_map)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_sequencer_config(self, slot: int, sequencer: int, sequencer_config: bytes) -> None:
        """
        Set configuration of the indexed sequencer. The configuration consists of multiple parameters in a C struct format. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        sequencer_config : bytes
            Current Configuration struct bytes.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'bytes'])

        # SCPI call
        self._write_bin('SLOT{}:SEQuencer{}:CONFiguration '.format(slot, sequencer), sequencer_config)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_lo_enable_0(self, slot: int, lo_enable_0: bool) -> None:
        """
        Set LO 0 status.

        Parameters
        ----------
        slot : int
            slot index.
        lo_enable_0 : bool
            Current LO 0 status bool.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'bool'])

        # SCPI call
        self._write('SLOT{}:LO0:ENAble {}'.format(slot, 0 if lo_enable_0 == False else 1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_lo_enable_1(self, slot: int, lo_enable_1: bool) -> None:
        """
        Set LO 1 status.

        Parameters
        ----------
        slot : int
            slot index.
        lo_enable_1 : bool
            Current LO 1 status bool.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'bool'])

        # SCPI call
        self._write('SLOT{}:LO1:ENAble {}'.format(slot, 0 if lo_enable_1 == False else 1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_in_att_0(self, slot: int, in_att_0: int) -> None:
        """
        Set input attenuator.

        Parameters
        ----------
        slot : int
            slot index.
        in_att_0 : int
            Attenuation in dB in a range of 0 to 30 in steps of 2dB.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        self._write('SLOT{}:AFE:ATT:IN0 {}'.format(slot, in_att_0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_out_att_0(self, slot: int, out_att_0: int) -> None:
        """
        Set output attenuator for output 0.

        Parameters
        ----------
        slot : int
            slot index.
        out_att_0 : int
            Attenuation in dB in a range of 0 to 60 in steps of 2dB.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        self._write('SLOT{}:AFE:ATT:OUT0 {}'.format(slot, out_att_0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_out_att_1(self, slot: int, out_att_1: int) -> None:
        """
        Set output attenuator for output 1.

        Parameters
        ----------
        slot : int
            slot index.
        out_att_1 : int
            Attenuation in dB in a range of 0 to 60 in steps of 2dB.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        self._write('SLOT{}:AFE:ATT:OUT1 {}'.format(slot, out_att_1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_lo_freq_0(self, slot: int, lo_freq_0: int) -> None:
        """
        Set LO Frequency for output 0.

        Parameters
        ----------
        slot : int
            slot index.
        lo_freq_0 : int
            Frequency in Hertz

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        self._write('SLOT{}:STATus:QUEStionable:FREQuency:LO0 {}'.format(slot, lo_freq_0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_lo_freq_1(self, slot: int, lo_freq_1: int) -> None:
        """
        Set LO Frequency for output 1.

        Parameters
        ----------
        slot : int
            slot index.
        lo_freq_1 : int
            Frequency in Hertz

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        self._write('SLOT{}:STATus:QUEStionable:FREQuency:LO1 {}'.format(slot, lo_freq_1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_lo_pwr_0(self, slot: int, lo_pwr_0: int) -> None:
        """
        Set LO Power for output 0.

        Parameters
        ----------
        slot : int
            slot index.
        lo_pwr_0 : int
            Power in dBm

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        self._write('SLOT{}:STATus:QUEStionable:POWer:LO0 {}'.format(slot, lo_pwr_0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_lo_pwr_1(self, slot: int, lo_pwr_1: int) -> None:
        """
        Set LO Power for output 1.

        Parameters
        ----------
        slot : int
            slot index.
        lo_pwr_1 : int
            Power in dBm

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        self._write('SLOT{}:STATus:QUEStionable:POWer:LO1 {}'.format(slot, lo_pwr_1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_current_afe_temperature(self, slot: int) -> float:
        """
        Get current AFE temperature.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            current

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:QUEStionable:TEMPerature:AFE:CURRent?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_maximum_afe_temperature(self, slot: int) -> float:
        """
        Get maximum AFE temperature.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            maximum

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:QUEStionable:TEMPerature:AFE:MAXimum?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_current_lo_temperature(self, slot: int) -> float:
        """
        Get current LO temperature.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            current

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:QUEStionable:TEMPerature:LO:CURRent?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def get_maximum_lo_temperature(self, slot: int) -> float:
        """
        Get maximum LO temperature.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        float
            maximum

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:STATus:QUEStionable:TEMPerature:LO:MAXimum?'.format(slot))

        return float(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_sequencer_state(self, slot: int, sequencer: int) -> str:
        """
        Get Get the sequencer state. If an invalid sequencer index is given, an error is set in system error..

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.

        Returns
        -------
        str
            Concatenated list of strings separated by the semicolon character. Status is indicated by one status string and an optional number of flags respectively ordered as:

            :Status:

             - IDLE: Sequencer waiting to be armed and started.
             - ARMED: Sequencer is armed and ready to start.
             - RUNNING: Sequencer is running.
             - Q1 STOPPED: Classical part of the sequencer has stopped, waiting for real-time part to stop.
             - STOPPED: Sequencer has completely stopped.

            :Flags:

             - DISARMED: Sequencer was disarmed.
             - FORCED STOP: Sequencer was stopped while still running.
             - SEQUENCE PROCESSOR Q1 ILLEGAL INSTRUCTION: Classical sequencer part executed an unknown instruction.
             - SEQUENCE PROCESSOR RT EXEC ILLEGAL INSTRUCTION: Real-time sequencer part executed an unknown instruction.
             - AWG WAVE PLAYBACK INDEX INVALID PATH 0: AWG path 0 tried to play an unknown waveform.
             - AWG WAVE PLAYBACK INDEX INVALID PATH 1: AWG path 1 tried to play an unknown waveform.
             - ACQ WEIGHT PLAYBACK INDEX INVALID PATH 0: Acquisition path 0 tried to play an unknown weight.
             - ACQ WEIGHT PLAYBACK INDEX INVALID PATH 1: Acquisition path 1 tried to play an unknown weight.
             - ACQ SCOPE DONE PATH 0: Scope acquisition for path 0 has finished.
             - ACQ SCOPE OUT-OF-RANGE PATH 0: Scope acquisition data for path 0 was out-of-range.
             - ACQ SCOPE OVERWRITTEN PATH 0: Scope acquisition data for path 0 was overwritten.
             - ACQ SCOPE DONE PATH 1: Scope acquisition for path 1 has finished.
             - ACQ SCOPE OUT-OF-RANGE PATH 1: Scope acquisition data for path 1 was out-of-range.
             - ACQ SCOPE OVERWRITTEN PATH 1: Scope acquisition data for path 1 was overwritten.
             - ACQ BINNING DONE: Acquisition binning completed.
             - ACQ BINNING FIFO ERROR: Acquisition binning encountered internal FIFO error.
             - ACQ BINNING COMM ERROR: Acquisition binning encountered internal communication error.
             - ACQ BINNING OUT-OF-RANGE: Acquisition binning data out-of-range.
             - ACQ INDEX INVALID: Acquisition tried to process an invalid acquisition.
             - ACQ BIN INDEX INVALID: Acquisition tried to process an invalid bin.
             - CLOCK INSTABILITY: Clock source instability occurred.


        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:STATE?'.format(slot, sequencer))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_num_acquisitions(self, slot: int, sequencer: int) -> int:
        """
        Get Get number of acquisitions in acquisition list of indexed sequencer's acquisition path. If an invalid sequencer index is given, an error is set in system error..

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.

        Returns
        -------
        int
            Number of acquisitions.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:ACQ:ALISt:SIZE?'.format(slot, sequencer))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_acquisitions(self, slot: int, sequencer: int) -> List[bytes]:
        """
        Get Get all acquisitions in acquisition list of indexed sequencer's acquisition path. If an invalid sequencer index is given, an error is set in system error..

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.

        Returns
        -------
        list[bytes]
            Dictionary with acquisitions

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:ACQ:ALISt?'.format(slot, sequencer))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_sequencer_program(self, slot: int, sequencer: int, program: str) -> None:
        """
        Assemble and set the resulting Q1ASM program for the indexed sequencer. If an invalid sequencer index is given or assembling fails, an error is set in system error. The assembler status and log can be retrieved using separate commands. To set the Q1ASM program, the sequencer will be stopped.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        program : str
            Q1ASM program.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        self._write_bin('SLOT{}:SEQuencer{}:PROGram '.format(slot, sequencer), program.encode('ascii'))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def start_adc_calib(self, slot: int) -> None:
        """
        Calibrates ADC delay and offset values. This method sets the correct delay values for every input data index (IO data lane) in order to avoid timing violations which occur while sampling ADC data. It also calibrates offsets internal to the ADC.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        self._write('SLOT{}:AFE:ADC:CALibration'.format(slot))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _add_awg_waveform(self, slot: int, sequencer: int, name: str, size: int, is_integer: bool) -> None:
        """
        Add new waveform to waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if the waveform causes the waveform memory limit to be exceeded, an error is set in system error. The waveform names 'all' and 'ALL' are reserved and adding those will also result in an error being set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        size : int
            Waveform length in number of samples.
        is_integer : bool
            Waveform is provided as integers (False = floats, True = integers).

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str', 'int', 'bool'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:AWG:WLISt:WAVeform:NEW "{}",{},{}'.format(slot, sequencer, name, size, 0 if is_integer == False else 1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _delete_awg_waveform(self, slot: int, sequencer: int, name: str) -> None:
        """
        Delete waveform from waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if a non-existing waveform name is given, an error is set in system error. The waveform names 'all' and 'ALL' are reserved and deleting those will cause a purge of the entire waveform list.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:AWG:WLISt:WAVeform:DELete "{}"'.format(slot, sequencer, name))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_awg_waveform_data(self, slot: int, sequencer: int, name: str, waveform: List[float]) -> None:
        """
        Set waveform data of waveform in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if a non-existing waveform name is given, an error is set in system error. If the waveform size does not match the size specified while adding the waveform or if the waveform samples are out-of-range, an error is set in the system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        waveform : list[float]
            List of floats in the range of 1.0 to -1.0 representing the waveform samples.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str', 'float'])

        # SCPI call
        self._write_bin('SLOT{}:SEQuencer{}:AWG:WLISt:WAVeform:DATA "{}",'.format(slot, sequencer, name), struct.pack('f'*len(waveform), *waveform))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_awg_waveform_data(self, slot: int, sequencer: int, name: str, start: int, size: int) -> List[float]:
        """
        Get waveform data of waveform in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if a non-existing waveform name is given, an error is set in system error. The start and size arguments can be used to return a section of the waveform. If the start argument is not given, zero is used as the default start sample. If the size argument is not given, all samples starting from the start sample are returned.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        start : int
            First sample within the waveform to be returned.
        size : int
            Number of samples starting from the start sample to be returned.

        Returns
        -------
        list[float]
            List of floats in the range of 1.0 to -1.0 representing the waveform samples.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str', 'int', 'int'])

        # SCPI call
        var0 = self._read_bin('SLOT{}:SEQuencer{}:AWG:WLISt:WAVeform:DATA? "{}",{},{}'.format(slot, sequencer, name, start, size))

        return struct.unpack('f'*int(len(var0)/4), var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_awg_waveform_index(self, slot: int, sequencer: int, name: str, index: int) -> None:
        """
        Set waveform index of waveform in waveform list of indexed sequencer's AWG path. The index is used by the sequencer Q1ASM program to refer to the waveform. If an invalid sequencer or waveform index is given or if a non-existing waveform name is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        index : int
            Waveform index of the waveform in the waveform list.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str', 'int'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:AWG:WLISt:WAVeform:INDex "{}",{}'.format(slot, sequencer, name, index))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_awg_waveform_index(self, slot: int, sequencer: int, name: str) -> int:
        """
        Get waveform index of waveform in waveform list of indexed sequencer's AWG path. The index is used by the sequencer Q1ASM program to refer to the waveform. If an invalid sequencer index is given or if a non-existing waveform name is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.

        Returns
        -------
        int
            Waveform index of the waveform in the waveform list.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:AWG:WLISt:WAVeform:INDex? "{}"'.format(slot, sequencer, name))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_awg_waveform_length(self, slot: int, sequencer: int, name: str) -> int:
        """
        Get length of waveform in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if a non-existing waveform name is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.

        Returns
        -------
        int
            Waveform length in number of samples.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:AWG:WLISt:WAVeform:LENGth? "{}"'.format(slot, sequencer, name))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_awg_waveform_name(self, slot: int, sequencer: int, index: int) -> str:
        """
        Get name of waveform in waveform list of indexed sequencer's AWG path based on its waveform index. If an invalid sequencer index is given or if a non-existing waveform index is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        index : int
            Waveform index.

        Returns
        -------
        str
            Waveform name.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'int'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:AWG:WLISt:WAVeform:NAME? {}'.format(slot, sequencer, index))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_awg_num_waveforms(self, slot: int, sequencer: int) -> int:
        """
        Get number of waveforms in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.

        Returns
        -------
        int
            Number of waveforms.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:AWG:WLISt:SIZE?'.format(slot, sequencer))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_awg_waveforms(self, slot: int, sequencer: int) -> List[bytes]:
        """
        Get all waveforms in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.

        Returns
        -------
        list[bytes]
            Dictionary with waveforms.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:AWG:WLISt?'.format(slot, sequencer))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _add_acq_weight(self, slot: int, sequencer: int, name: str, size: int, is_integer: bool) -> None:
        """
        Add new weight to weight list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if the weight causes the weight memory limit to be exceeded, an error is set in system error. The weight names 'all' and 'ALL' are reserved and adding those will also result in an error being set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Weight name.
        size : int
            Weight length in number of samples.
        is_integer : bool
            Weight is provided as integers (False = floats, True = integers).

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str', 'int', 'bool'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:ACQ:WLISt:WEIght:NEW "{}",{},{}'.format(slot, sequencer, name, size, 0 if is_integer == False else 1))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _delete_acq_weight(self, slot: int, sequencer: int, name: str) -> None:
        """
        Delete weight from weight list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if a non-existing weight name is given, an error is set in system error. The weight names 'all' and 'ALL' are reserved and deleting those will cause a purge of the entire weight list.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Weight name.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:ACQ:WLISt:WEIght:DELete "{}"'.format(slot, sequencer, name))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_acq_weight_data(self, slot: int, sequencer: int, name: str, weight: List[float]) -> None:
        """
        Set weight data of weight in weight list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if a non-existing weight name is given, an error is set in system error. If the weight size does not match the size specified while adding the weight or if the weight samples are out-of-range, an error is set in the system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Weight name.
        weight : list[float]
            List of floats in the range of 1.0 to -1.0 representing the weight samples.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str', 'float'])

        # SCPI call
        self._write_bin('SLOT{}:SEQuencer{}:ACQ:WLISt:WEIght:DATA "{}",'.format(slot, sequencer, name), struct.pack('f'*len(weight), *weight))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_weight_data(self, slot: int, sequencer: int, name: str, start: int, size: int) -> List[float]:
        """
        Get weight data of weight in weight list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if a non-existing weight name is given, an error is set in system error. The start and size arguments can be used to return a section of the weight. If the start argument is not given, zero is used as the default start sample. If the size argument is not given, all samples starting from the start sample are returned.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Weight name.
        start : int
            First sample within the weight to be returned.
        size : int
            Number of samples starting from the start sample to be returned.

        Returns
        -------
        list[float]
            List of floats in the range of 1.0 to -1.0 representing the weight samples.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str', 'int', 'int'])

        # SCPI call
        var0 = self._read_bin('SLOT{}:SEQuencer{}:ACQ:WLISt:WEIght:DATA? "{}",{},{}'.format(slot, sequencer, name, start, size))

        return struct.unpack('f'*int(len(var0)/4), var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_acq_weight_index(self, slot: int, sequencer: int, name: str, index: int) -> None:
        """
        Set weight index of weight in weight list of indexed sequencer's acquisition path. The index is used by the sequencer Q1ASM program to refer to the weight. If an invalid sequencer or weight index is given or if a non-existing weight name is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Weight name.
        index : int
            Weight index of the weight in the weight list.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str', 'int'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:ACQ:WLISt:WEIght:INDex "{}",{}'.format(slot, sequencer, name, index))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_weight_index(self, slot: int, sequencer: int, name: str) -> int:
        """
        Get weight index of weight in weight list of indexed sequencer's acquisition path. The index is used by the sequencer Q1ASM program to refer to the weight. If an invalid sequencer index is given or if a non-existing weight name is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Weight name.

        Returns
        -------
        int
            Weight index of the weight in the weight list.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:ACQ:WLISt:WEIght:INDex? "{}"'.format(slot, sequencer, name))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_weight_length(self, slot: int, sequencer: int, name: str) -> int:
        """
        Get length of weight in weight list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if a non-existing weight name is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Weight name.

        Returns
        -------
        int
            Weight length in number of samples.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:ACQ:WLISt:WEIght:LENGth? "{}"'.format(slot, sequencer, name))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_weight_name(self, slot: int, sequencer: int, index: int) -> str:
        """
        Get name of weight in weight list of indexed sequencer's acquisition path based on its weight index. If an invalid sequencer index is given or if a non-existing weight index is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        index : int
            Weight index.

        Returns
        -------
        str
            Weight name.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'int'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:ACQ:WLISt:WEIght:NAME? {}'.format(slot, sequencer, index))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_num_weights(self, slot: int, sequencer: int) -> int:
        """
        Get number of weights in weight list of indexed sequencer's acquisition path. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.

        Returns
        -------
        int
            Number of weights.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:ACQ:WLISt:SIZE?'.format(slot, sequencer))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_weights(self, slot: int, sequencer: int) -> List[bytes]:
        """
        Get all weights in weight list of indexed sequencer's acquisition path. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.

        Returns
        -------
        list[bytes]
            Dictionary with weights.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:ACQ:WLISt?'.format(slot, sequencer))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_lo_hw_present(self, slot: int) -> bool:
        """
        Check if LO hardware is present.

        Parameters
        ----------
        slot : int
            slot index.

        Returns
        -------
        bool
            LO hardware presence (False = not present, True = present).

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int'])

        # SCPI call
        var0 = self._read('SLOT{}:LO:PRESent?'.format(slot))

        return bool(int(var0))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _arm_sequencer(self, slot: int, sequencer: int) -> None:
        """
        Prepare the indexed sequencer to start by putting it in the armed state. If no sequencer index is given, all sequencers are armed. Any sequencer that was already running is stopped and rearmed. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:ARM'.format(slot, sequencer))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _start_sequencer(self, slot: int, sequencer: int) -> None:
        """
        Start the indexed sequencer, thereby putting it in the running state. If an invalid sequencer index is given or the indexed sequencer was not yet armed, an error is set in system error. If no sequencer index is given, all armed sequencers are started and any sequencer not in the armed state is ignored. However, if no sequencer index is given and no sequencers are armed, and error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:START'.format(slot, sequencer))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _stop_sequencer(self, slot: int, sequencer: int) -> None:
        """
        Stop the indexed sequencer, thereby putting it in the stopped state. If an invalid sequencer index is given, an error is set in system error. If no sequencer index is given, all sequencers are stopped.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:STOP'.format(slot, sequencer))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _add_acq_acquisition(self, slot: int, sequencer: int, name: str, num_bins: int) -> None:
        """
        Add new acquisition to acquisition list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if the required acquisition memory cannot be allocated, an error is set in system error. The acquisition names 'all' and 'ALL' are reserved and adding those will also result in an error being set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Acquisition name.
        num_bins : int
            Number of bins in acquisition. Maximum is 2^24.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str', 'int'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:ACQ:ALISt:ACQuisition:NEW "{}",{}'.format(slot, sequencer, name, num_bins))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _delete_acq_acquisition(self, slot: int, sequencer: int, name: str) -> None:
        """
        Delete acquisition from acquisition list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if a non-existing acquisition name is given, an error is set in system error. The acquisition names 'all' and 'ALL' are reserved and deleting those will cause a purge of the entire acquisition list.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Acquisition name.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:ACQ:ALISt:ACQuisition:DELete "{}"'.format(slot, sequencer, name))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_acq_acquisition_data(self, slot: int, sequencer: int, name: str) -> None:
        """
        Move scope mode (raw) acquisition data into acquisition in acquisition list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if a non-existing acquisition name is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Acquisition name.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:ACQ:ALISt:ACQuisition:DATA "{}"'.format(slot, sequencer, name))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _delete_acq_acquisition_data(self, slot: int, sequencer: int, name: str) -> None:
        """
        Delete acquisition data from acquisition in acquisition list. If an invalid sequencer index is given or if a non-existing acquisition name is given, an error is set in system error. The acquisition names 'all' and 'ALL' are reserved and deleting those will cause a purge of the data within the entire acquisition list.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Acquisition name.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:ACQ:ALISt:ACQuisition:DATA:DELete "{}"'.format(slot, sequencer, name))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_acquisition_data(self, slot: int, sequencer: int, name: str) -> List[bytes]:
        """
        Get acquisition data of acquisition in acquisition list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if a non-existing acquisition name is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Acquisition name.

        Returns
        -------
        list[bytes]
            Binary structure containing the acquisition.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:ACQ:ALISt:ACQuisition:DATA? "{}"'.format(slot, sequencer, name))

        return var0

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _set_acq_acquisition_index(self, slot: int, sequencer: int, name: str, index: int) -> None:
        """
        Set acquisition index of acquisition in acquisition list of indexed sequencer's acquisition path. The index is used by the sequencer Q1ASM program to refer to the acquisition. If an invalid sequencer or acquisition index is given or if a non-existing acquisition name is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Acquisition name.
        index : int
            Acquisition index of the acquisition in the acquisition list.

        Returns
        -------
        None

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str', 'int'])

        # SCPI call
        self._write('SLOT{}:SEQuencer{}:ACQ:ALISt:ACQuisition:INDex "{}",{}'.format(slot, sequencer, name, index))

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_acquisition_index(self, slot: int, sequencer: int, name: str) -> int:
        """
        Get acquisition index of acquisition in acquisition list of indexed sequencer's acquisition path. The index is used by the sequencer Q1ASM program to refer to the acquisition. If an invalid sequencer index is given or if a non-existing acquisition name is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Acquisition name.

        Returns
        -------
        int
            Acquisition index of the acquisition in the acquisition list.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:ACQ:ALISt:ACQuisition:INDex? "{}"'.format(slot, sequencer, name))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_acquisition_num_bins(self, slot: int, sequencer: int, name: str) -> int:
        """
        Get number of bins in acquisition in acquisition list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if a non-existing acquisition name is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        name : str
            Acquisition name.

        Returns
        -------
        int
            Number of bins in acquisition.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:ACQ:ALISt:ACQuisition:BINs? "{}"'.format(slot, sequencer, name))

        return int(var0)

    # -------------------------------------------------------------------------
    @scpi_error_check
    def _get_acq_acquisition_name(self, slot: int, sequencer: int, index: int) -> str:
        """
        Get name of acquisition in acquisition list of indexed sequencer's acquisition path based on its acquisition index. If an invalid sequencer index is given or if a non-existing acquisition index is given, an error is set in system error.

        Parameters
        ----------
        slot : int
            slot index.
        sequencer : int
            Sequencer index.
        index : int
            acquisition index.

        Returns
        -------
        str
            acquisition name.

        Raises
        ------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        # Check input types.
        self._check_in_type(locals(), ['int', 'int', 'int'])

        # SCPI call
        var0 = self._read('SLOT{}:SEQuencer{}:ACQ:ALISt:ACQuisition:NAME? {}'.format(slot, sequencer, index))

        return var0

    # ---------------------------------------------------------------------
    def _check_in_type(self, in_arg_dict: Dict, in_type_list: List) -> None:
        """
        Checks input argument types against reference types.

        Parameters
        ----------
        in_arg_dict : dict
            Dictionary with input arguments created by locals().
        in_type_list : list
            List of reference input argument types.

        Returns
        ----------

        Raises
        ----------
        TypeError
            Input argument type mismatch.
        """

        if self._debug <= 1:
            del in_arg_dict['self']
            in_val_list = [in_arg_dict[name] for name in in_arg_dict]
            for i, (in_val, in_type) in enumerate(zip(in_val_list, in_type_list)):
                if (type(in_val).__name__ == "list" or
                   type(in_val).__name__ == "ndarray"):
                    if len(in_val) > 0:
                        in_val = in_val[0]
                    else:
                        raise TypeError(
                            "Unexpected type for input argument {}, expected {} but got empty {}.".format(i, in_type, str(type(in_val).__name__))
                        )
                if type(in_val).__name__[:len(in_type)] != in_type:
                    raise TypeError(
                        "Unexpected type for input argument {}, expected {} but got {}.".format(i, in_type, str(type(in_val).__name__))
                    )

    # -------------------------------------------------------------------------
    def _check_error_queue(self, err: Optional[Exception]=None) -> None:
        """
        Check system error for errors. Empties and prints the complete error
        queue.

        Parameters
        ----------
        err : Optional[Exception]
            Exception to reraise.

        Returns
        ----------

        Raises
        ----------
        Exception
            An exception was passed as input argument.
        RuntimeError:
            An error was found in system error.
        """

        if self._debug <= 1:
            errors = [str(err)] if err is not None else []
            while int(self._read('SYSTem:ERRor:COUNt?')) != 0:
                errors.append(','.join(self._read('SYSTem:ERRor:NEXT?').split(',')[1:]))

            if len(errors) > 0:
                if err is not None:
                    err_type = type(err)
                else:
                    err_type = RuntimeError
                raise err_type('\n'.join(errors)).with_traceback(sys.exc_info()[2]) from None
