# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['xbotlib']
install_requires = \
['Jinja2>=2.11.2,<3.0.0',
 'aiohttp>=3.7.3,<4.0.0',
 'humanize>=3.2.0,<4.0.0',
 'redis>=3.5.3,<4.0.0',
 'slixmpp>=1.6.0,<2.0.0']

setup_kwargs = {
    'name': 'xbotlib',
    'version': '0.14.1',
    'description': 'XMPP bots for humans',
    'long_description': '# xbotlib\n\n[![PyPI version](https://badge.fury.io/py/xbotlib.svg)](https://badge.fury.io/py/xbotlib)\n[![Build Status](https://drone.autonomic.zone/api/badges/decentral1se/xbotlib/status.svg?ref=refs/heads/main)](https://drone.autonomic.zone/decentral1se/xbotlib)\n\n## XMPP bots for humans\n\n> status: experimental\n\nA friendly lightweight wrapper around\n[slixmpp](https://slixmpp.readthedocs.io/) for writing XMPP bots in Python. The\ngoal is to make writing and running XMPP bots easy and fun. `xbotlib` is a\n[single file implementation](./xbotlib.py) which can easily be understood and\nextended. It provides a small API surface which reflects the `slixmpp` way of\ndoing things. The `xbotlib` source code and ideas are largely\nborrowed/stolen/adapted/reimagined from the XMPP bot experiments that have gone\non and are still going on in\n[Varia](https://git.vvvvvvaria.org/explore/repos?tab=&sort=recentupdate&q=bots).\n\nWe\'re lurking in\n[xbotlibtest@muc.vvvvvvaria.org](xmpp:xbotlibtest@muc.vvvvvvaria.org?join) if\nyou want to chat or just invite your bots for testing.\n\n- [Install](#install)\n- [Example](#example)\n- [API Reference](#api-reference)\n  - [Bot.direct(message)](#bot-direct-message)\n  - [Bot.group(message)](#bot-group-message)\n  - [Bot.serve(request)](#bot-serve-request)\n  - [SimpleMessage](#simplemessage)\n  - [Bot](#bot)\n- [Working with your bot](#working-with-your-bot)\n  - [Documentation](#documentation)\n  - [Commands](#commands)\n  - [Avatars](#avatars)\n  - [Configuration](#configuration)\n    - [Using the `.conf` configuration file](#using-the--conf--configuration-file)\n    - [Using the command-line interface](#using-the-command-line-interface)\n    - [Using the environment](#using-the-environment)\n  - [Persistent storage](#persistent-storage)\n    - [File system](#file-system)\n    - [Redis key/value storage](#redis-key-value-storage)\n  - [Loading Plugins](#loading-plugins)\n  - [Serving HTTP](#serving-http)\n- [Deploy your bots](#deploy-your-bots)\n- [Roadmap](#roadmap)\n- [Changes](#changes)\n- [License](#license)\n\n## Install\n\n```sh\n$ pip install xbotlib\n```\n\n## Example\n\nPut the following in a `echo.py` file. This bot is pretty simple: it echoes\nback whatever message you send it. It is an easy way to get started.\n\n```python\nfrom xbotlib import Bot\n\nclass EchoBot(Bot):\n\n    def direct(self, message):\n        return self.reply(message.text, to=message.sender)\n\n    def group(self, message):\n        return self.reply(message.content, room=message.room)\n```\n\nAnd then `python echo.py`. You will be asked a few questions in order to load\nthe account details that your bot will be using. This will generate an\n`echobot.conf` file in the same working directory for further use. See the\n[configuration](#configure-your-bot) section for more.\n\nRead more in the [API reference](#api-reference) for how to write your own bots.\n\nSee more examples on [git.vvvvvvaria.org](https://git.vvvvvvaria.org/explore/repos?q=xbotlib&topic=1).\n\n## API Reference\n\nWhen writing your own bot, you always sub-class the `Bot` class provided from\n`xbotlib`. Then if you want to respond to a direct message, you write a\n[direct](#botdirectmessage) function. If you want to respond to a group chat\nmessage, you write a [group](#botgroupmessage) function. That\'s it for the\nbasics.\n\n### Bot.direct(message)\n\nRespond to direct messages.\n\nArguments:\n\n- **message**: received message (see [SimpleMessage](#simplemessage) below for available attributes)\n\n### Bot.group(message)\n\nRespond to a message in a group chat.\n\nArguments:\n\n- **message**: received message (see [SimpleMessage](#simplemessage) below for available attributes)\n\n### Bot.serve(request)\n\nServe requests via the built-in web server.\n\nArguments:\n\n- **request**: the web request\n\n### SimpleMessage\n\nA simple message interface.\n\nAttributes:\n\n- **text**: the entire text of the message\n- **content**: the text of the message after the nick\n- **sender**: the user the message came from\n- **room**: the room the message came from\n- **receiver**: the receiver of the message\n- **nick**: the nickname of the sender\n- **type**: the type of message\n- **url**: The URL of a sent file\n\n### Bot\n\n> Bot.reply(message, to=None, room=None)\n\nSend a reply back.\n\nArguments:\n\n- **message**: the message that is sent\n- **to**: the user to send the reply to\n- **room**: the room to send the reply to\n\n> Bot.respond(response, content_type="text/html")\n\nReturn a response via the web server.\n\nArguments:\n\n- **response**: the text of the response\n- **content_type**: the type of response\n\nOther useful attributes on the `Bot` class are:\n\n- **self.db**: The [Redis database](#redis-key-value-storage) if you\'re using it\n\n## Working with your bot\n\n### Documentation\n\nAdd a `help = "my help"` to your `Bot` class like so.\n\n```python\nclass MyBot(Bot):\n    help = "My help"\n```\n\nSee more in the [commands](#commands) section on how to use this.\n\n### Commands\n\nUsing `@<command>` in direct messages and `<nick>, @<command>` (the `,` is\noptional, anything will be accepted here and there doesn\'t seem to be a\nconsensus on what is most common way to "at" another user in XMPP) in group chats,\nhere are the supported commands.\n\n- `@uptime`: how long the bot has been running\n- `@help`: the help text for what the bot does\n\nThere are also more general status commands which all bots respond to.\n\n- `@bots`: status check on who is a bot in the group chat\n\nThese commands will be detected in any part of the message sent to the bot. So\nyou can write `echobot, can we see your @uptime`, or `I\'d love to know which @bots are here.`\n\n### Avatars\n\nBy default, `xbotlib` will look for an `avatar.png` (so far tested with `.png`\nbut other file types may work) file alongside your Python script which contains\nyour bot implementation. You can also specify another path using the `--avatar`\noption on the command-line interface. The images should ideally have a height\nof `64` and a width of `64` pixels each.\n\n## Configuration\n\nAll the ways you can pass configuration details to your bot. There are three\nways to configure your bot, the configuration file, command-line interface and\nthe environment. Use whichever one suits you best. The values are loaded in the\nfollowing order: command-line > configuration file > environment.\n\n#### Using the `.conf` configuration file\n\nIf you run simply run your Python script which contains the bot then `xbotlib`\nwill generate a configuration for you by asking a few questions. This is the\nsimplest way to run your bot locally.\n\nHere is an example of a working configuration.\n\n```conf\n[echobot]\naccount = echobot@vvvvvvaria.org\npassword = ...thepassword...\nnick = echobot\nrooms = test1@muc.example.com, test2@muc.example.com\n```\n\n#### Using the command-line interface\n\nEvery bot accepts a number of comand-line arguments to load configuration. You\ncan use the `--help` option to see what is available (e.g. `python bot.py --help`).\n\n```\nusage: bot.py [-h] [-d] [-a ACCOUNT] [-p PASSWORD] [-n NICK]\n              [-av AVATAR] [-ru REDIS_URL] [-r ROOMS [ROOMS ...]]\n              [--no-auto-join]\n\nXMPP bots for humans\n\noptional arguments:\n  -h, --help            show this help message and exit\n  -d, --debug           Enable verbose debug logs\n  -a ACCOUNT, --account ACCOUNT\n                        Account for the bot account\n  -p PASSWORD, --password PASSWORD\n                        Password for the bot account\n  -n NICK, --nick NICK  Nickname for the bot account\n  -av AVATAR, --avatar AVATAR\n                        Avatar for the bot account\n  -ru REDIS_URL, --redis-url REDIS_URL\n                        Redis storage connection URL\n  -r ROOMS [ROOMS ...], --rooms ROOMS [ROOMS ...]\n                        Rooms to automatically join\n  --no-auto-join        Disable automatically joining rooms when invited\n  -pt PORT, --port PORT\n                        The port to serve from\n  -t TEMPLATE, --template TEMPLATE\n                        The template to render\n```\n\n#### Using the environment\n\n`xbotlib` will try to read the following configuration values from the\nenvironment if it cannot read them from a configuration file or the\ncommand-line interface. This can be useful when doing remote server\ndeployments.\n\n- **XBOT_ACCOUNT**: The bot account\n- **XBOT_PASSWORD**: The bot password\n- **XBOT_NICK**: The bot nickname\n- **XBOT_AVATAR**: The bot avatar icon\n- **XBOT_REDIS_URL**: Redis key store connection URL\n- **XBOT_ROOMS**: The rooms to automatically join\n- **XBOT_NO_AUTO_JOIN**: Disable auto-joining on invite\n- **XBOT_PORT**: The port to serve from\n\n### Persistent storage\n\n#### File system\n\nJust use your local file system as you would in any other Python script. Please\nnote that when you deploy your bot, you might not have access to this local\nfilesystem in the same location. For remote server deployments\n[Redis](#redis-key-value-storage) can be more convenient.\n\n#### Redis key/value storage\n\n`xbotlib` supports using [Redis](https://redis.io/) as a storage back-end. It\nis simple to work with because the interface is exactly like a dictionary. You\ncan quickly run Redis locally using [Docker](https://docs.docker.com/engine/install/debian/)\n(`docker run --network=host --name redis -d redis`) or if you\'re on a Debian system you can\nalso `sudo apt install -y redis`.\n\nYou can configure the connection URL using the command-line interface,\nconfiguration or environment. Here is an example using the environment.\n\n```bash\n$ export XBOT_REDIS_URL=redis://localhost:6379/0\n```\n\nAnd you access the interface via the `self.db` attribute.\n\n```python\ndef direct(self, message):\n    self.db["mykey"] = message.text\n```\n\nYou should see `INFO Successfully connected to storage` when your bot\ninitialises. Please see the\n[redis-py](https://redis-py.readthedocs.io/en/stable/) API documentation for\nmore.\n\n### Loading Plugins\n\nYou can specify a `plugins = [...]` on your bot definition and they will be\nautomatically loaded.\n\n```python\nclass MyBot(Bot):\n    plugins = ["xep_0066"]\n```\n\n### Serving HTTP\n\nYour bot will automatically be running a web server at port `8080` when it is\nrun. If you\'re running your bot locally, just visit\n[0.0.0.0:8080](http://0.0.0.0:8080) to see. The default response is just some\nplaceholder text. You can write your own responses using the\n[Bot.serve](#bot-serve-request) function.\n\n`xbotlib` provides a small wrapper API for\n[Jinja2](https://jinja.palletsprojects.com/en/2.11.x/) which allows you to\neasily template and generate HTML. The web server is provided by\n[aiohttp](https://docs.aiohttp.org/).\n\nThe default template search path is `index.html.j2` in the current working\ndirectory. This can be configured through the usual configuration entrypoints.\n\nHere\'s a small example that renders a random ASCII letter.\n\n```jinja\n<h1>{{ letter }}</h1>\n```\n\n```python\nfrom string import ascii_letters\n\ndef serve(self, request):\n    letter = choice(ascii_letters)\n    rendered = self.template.render(letter=letter)\n    return self.respond(body=rendered)\n```\n\nIf you want to pass data from your `direct`/`group` functions to the `serve`\nfunction, you\'ll need to make use of [some type of persistent\nstorage](#persistent-storage). Your `serve` function can read from the database\nor file system and then respond with generated HTML from there.\n\nHaving your bot avaible on the web is useful for doing healthchecks with\nsomething like [statping](https://statping.com/) so you be sure that your bot\nis up and running.\n\n## Deploy your bots\n\nSee [bots.varia.zone](https://bots.varia.zone/).\n\n## Roadmap\n\nSee the [issue tracker](https://git.autonomic.zone/decentral1se/xbotlib/issues).\n\n## Changes\n\nSee the [CHANGELOG.md](./CHANGELOG.md).\n\n## License\n\nSee the [LICENSE](./LICENSE.md).\n',
    'author': 'decentral1se',
    'author_email': 'lukewm@riseup.net',
    'maintainer': 'decentral1se',
    'maintainer_email': 'lukewm@riseup.net',
    'url': 'https://git.autonomic.zone/decentral1se/xbotlib',
    'py_modules': modules,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
