"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsEventSource = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
/**
 * Use an Amazon SQS queue as an event source for AWS Lambda.
 *
 * @stability stable
 */
class SqsEventSource {
    /**
     * @stability stable
     */
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        this._eventSourceMappingId = undefined;
        if (this.props.maxBatchingWindow !== undefined) {
            if (queue.fifo) {
                throw new Error('Batching window is not supported for FIFO queues');
            }
            if (!this.props.maxBatchingWindow.isUnresolved() && this.props.maxBatchingWindow.toSeconds() > 300) {
                throw new Error(`Maximum batching window must be 300 seconds or less (given ${this.props.maxBatchingWindow.toHumanString()})`);
            }
        }
        if (this.props.batchSize !== undefined && !core_1.Token.isUnresolved(this.props.batchSize)) {
            if (this.props.maxBatchingWindow !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 10000)) {
                throw new Error(`Maximum batch size must be between 1 and 10000 inclusive (given ${this.props.batchSize}) when batching window is specified.`);
            }
            if (this.props.maxBatchingWindow === undefined && (this.props.batchSize < 1 || this.props.batchSize > 10)) {
                throw new Error(`Maximum batch size must be between 1 and 10 inclusive (given ${this.props.batchSize}) when batching window is not specified.`);
            }
        }
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     *
     * @stability stable
     */
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`SqsEventSource:${core_1.Names.nodeUniqueId(this.queue.node)}`, {
            batchSize: this.props.batchSize,
            maxBatchingWindow: this.props.maxBatchingWindow,
            enabled: this.props.enabled,
            eventSourceArn: this.queue.queueArn,
        });
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.queue.grantConsumeMessages(target);
    }
    /**
     * The identifier for this EventSourceMapping.
     *
     * @stability stable
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('SqsEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.SqsEventSource = SqsEventSource;
_a = JSII_RTTI_SYMBOL_1;
SqsEventSource[_a] = { fqn: "@aws-cdk/aws-lambda-event-sources.SqsEventSource", version: "1.129.0" };
//# sourceMappingURL=data:application/json;base64,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