"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceReadReplica = exports.DatabaseInstanceFromSnapshot = exports.DatabaseInstance = exports.StorageType = exports.LicenseModel = exports.DatabaseInstanceBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const parameter_group_1 = require("./parameter-group");
const util_1 = require("./private/util");
const props_1 = require("./props");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * A new or imported database instance.
 */
class DatabaseInstanceBase extends core_1.Resource {
    /**
     * Import an existing database instance.
     */
    static fromDatabaseInstanceAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseInstanceAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDatabaseInstanceAttributes);
            }
            throw error;
        }
        class Import extends DatabaseInstanceBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: this.defaultPort,
                });
                this.instanceIdentifier = attrs.instanceIdentifier;
                this.dbInstanceEndpointAddress = attrs.instanceEndpointAddress;
                this.dbInstanceEndpointPort = core_1.Tokenization.stringifyNumber(attrs.port);
                this.instanceEndpoint = new endpoint_1.Endpoint(attrs.instanceEndpointAddress, attrs.port);
                this.engine = attrs.engine;
                this.enableIamAuthentication = true;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a new db proxy to this instance.
     */
    addProxy(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseProxyOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addProxy);
            }
            throw error;
        }
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromInstance(this),
            ...options,
        });
    }
    grantConnect(grantee) {
        if (this.enableIamAuthentication === false) {
            throw new Error('Cannot grant connect when IAM authentication is disabled');
        }
        this.enableIamAuthentication = true;
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['rds-db:connect'],
            resourceArns: [this.instanceArn],
        });
    }
    /**
     * Defines a CloudWatch event rule which triggers for instance events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.rds'],
            resources: [this.instanceArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * The instance arn.
     */
    get instanceArn() {
        const commonAnComponents = {
            service: 'rds',
            resource: 'db',
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        };
        const localArn = core_1.Stack.of(this).formatArn({
            ...commonAnComponents,
            resourceName: this.instanceIdentifier,
        });
        return this.getResourceArnAttribute(localArn, {
            ...commonAnComponents,
            resourceName: this.physicalName,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.instanceIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_INSTANCE,
        };
    }
}
exports.DatabaseInstanceBase = DatabaseInstanceBase;
_a = JSII_RTTI_SYMBOL_1;
DatabaseInstanceBase[_a] = { fqn: "@aws-cdk/aws-rds.DatabaseInstanceBase", version: "1.190.0" };
/**
 * The license model.
 */
var LicenseModel;
(function (LicenseModel) {
    /**
     * License included.
     */
    LicenseModel["LICENSE_INCLUDED"] = "license-included";
    /**
     * Bring your own licencse.
     */
    LicenseModel["BRING_YOUR_OWN_LICENSE"] = "bring-your-own-license";
    /**
     * General public license.
     */
    LicenseModel["GENERAL_PUBLIC_LICENSE"] = "general-public-license";
})(LicenseModel = exports.LicenseModel || (exports.LicenseModel = {}));
/**
 * The type of storage.
 */
var StorageType;
(function (StorageType) {
    /**
     * Standard.
     */
    StorageType["STANDARD"] = "standard";
    /**
     * General purpose (SSD).
     */
    StorageType["GP2"] = "gp2";
    /**
     * Provisioned IOPS (SSD).
     */
    StorageType["IO1"] = "io1";
})(StorageType = exports.StorageType || (exports.StorageType = {}));
/**
 * A new database instance.
 */
class DatabaseInstanceNew extends DatabaseInstanceBase {
    constructor(scope, id, props) {
        // RDS always lower-cases the ID of the database, so use that for the physical name
        // (which is the name used for cross-environment access, so it needs to be correct,
        // regardless of the feature flag that changes it in the template for the L1)
        const instancePhysicalName = core_1.Token.isUnresolved(props.instanceIdentifier)
            ? props.instanceIdentifier
            : props.instanceIdentifier?.toLowerCase();
        super(scope, id, {
            physicalName: instancePhysicalName,
        });
        this.vpc = props.vpc;
        if (props.vpcSubnets && props.vpcPlacement) {
            throw new Error('Only one of `vpcSubnets` or `vpcPlacement` can be specified');
        }
        this.vpcPlacement = props.vpcSubnets ?? props.vpcPlacement;
        if (props.multiAz === true && props.availabilityZone) {
            throw new Error('Requesting a specific availability zone is not valid for Multi-AZ instances');
        }
        const subnetGroup = props.subnetGroup ?? new subnet_group_1.SubnetGroup(this, 'SubnetGroup', {
            description: `Subnet group for ${this.node.id} database`,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            removalPolicy: util_1.renderUnless(util_1.helperRemovalPolicy(props.removalPolicy), core_1.RemovalPolicy.DESTROY),
        });
        const securityGroups = props.securityGroups || [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: `Security group for ${this.node.id} database`,
                vpc: props.vpc,
            })];
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(core_1.Lazy.number({ produce: () => this.instanceEndpoint.port })),
        });
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new iam.Role(this, 'MonitoringRole', {
                assumedBy: new iam.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')],
            });
        }
        const storageType = props.storageType || StorageType.GP2;
        const iops = storageType === StorageType.IO1 ? (props.iops || 1000) : undefined;
        this.cloudwatchLogsExports = props.cloudwatchLogsExports;
        this.cloudwatchLogsRetention = props.cloudwatchLogsRetention;
        this.cloudwatchLogsRetentionRole = props.cloudwatchLogsRetentionRole;
        this.enableIamAuthentication = props.iamAuthentication;
        const enablePerformanceInsights = props.enablePerformanceInsights
            || props.performanceInsightRetention !== undefined || props.performanceInsightEncryptionKey !== undefined;
        if (enablePerformanceInsights && props.enablePerformanceInsights === false) {
            throw new Error('`enablePerformanceInsights` disabled, but `performanceInsightRetention` or `performanceInsightEncryptionKey` was set');
        }
        if (props.domain) {
            this.domainId = props.domain;
            this.domainRole = props.domainRole || new iam.Role(this, 'RDSDirectoryServiceRole', {
                assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSDirectoryServiceAccess'),
                ],
            });
        }
        const maybeLowercasedInstanceId = core_1.FeatureFlags.of(this).isEnabled(cxapi.RDS_LOWERCASE_DB_IDENTIFIER)
            && !core_1.Token.isUnresolved(props.instanceIdentifier)
            ? props.instanceIdentifier?.toLowerCase()
            : props.instanceIdentifier;
        const instanceParameterGroupConfig = props.parameterGroup?.bindToInstance({});
        this.newCfnProps = {
            autoMinorVersionUpgrade: props.autoMinorVersionUpgrade,
            availabilityZone: props.multiAz ? undefined : props.availabilityZone,
            backupRetentionPeriod: props.backupRetention?.toDays(),
            copyTagsToSnapshot: props.copyTagsToSnapshot ?? true,
            dbInstanceClass: core_1.Lazy.string({ produce: () => `db.${this.instanceType}` }),
            dbInstanceIdentifier: core_1.Token.isUnresolved(props.instanceIdentifier)
                // if the passed identifier is a Token,
                // we need to use the physicalName of the database
                // (we cannot change its case anyway),
                // as it might be used in a cross-environment fashion
                ? this.physicalName
                : maybeLowercasedInstanceId,
            dbSubnetGroupName: subnetGroup.subnetGroupName,
            deleteAutomatedBackups: props.deleteAutomatedBackups,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            enableCloudwatchLogsExports: this.cloudwatchLogsExports,
            enableIamDatabaseAuthentication: core_1.Lazy.any({ produce: () => this.enableIamAuthentication }),
            enablePerformanceInsights: enablePerformanceInsights || props.enablePerformanceInsights,
            iops,
            monitoringInterval: props.monitoringInterval?.toSeconds(),
            monitoringRoleArn: monitoringRole?.roleArn,
            multiAz: props.multiAz,
            dbParameterGroupName: instanceParameterGroupConfig?.parameterGroupName,
            optionGroupName: props.optionGroup?.optionGroupName,
            performanceInsightsKmsKeyId: props.performanceInsightEncryptionKey?.keyArn,
            performanceInsightsRetentionPeriod: enablePerformanceInsights
                ? (props.performanceInsightRetention || props_1.PerformanceInsightRetention.DEFAULT)
                : undefined,
            port: props.port !== undefined ? core_1.Tokenization.stringifyNumber(props.port) : undefined,
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            processorFeatures: props.processorFeatures && renderProcessorFeatures(props.processorFeatures),
            publiclyAccessible: props.publiclyAccessible ?? (this.vpcPlacement && this.vpcPlacement.subnetType === ec2.SubnetType.PUBLIC),
            storageType,
            vpcSecurityGroups: securityGroups.map(s => s.securityGroupId),
            maxAllocatedStorage: props.maxAllocatedStorage,
            domain: this.domainId,
            domainIamRoleName: this.domainRole?.roleName,
        };
    }
    setLogRetention() {
        if (this.cloudwatchLogsExports && this.cloudwatchLogsRetention) {
            for (const log of this.cloudwatchLogsExports) {
                new logs.LogRetention(this, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/instance/${this.instanceIdentifier}/${log}`,
                    retention: this.cloudwatchLogsRetention,
                    role: this.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * A new source database instance (not a read replica)
 */
class DatabaseInstanceSource extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        this.engine = props.engine;
        const engineType = props.engine.engineType;
        // only Oracle and SQL Server require the import and export Roles to be the same
        const combineRoles = engineType.startsWith('oracle-') || engineType.startsWith('sqlserver-');
        let { s3ImportRole, s3ExportRole } = util_1.setupS3ImportExport(this, props, combineRoles);
        const engineConfig = props.engine.bindToInstance(this, {
            ...props,
            s3ImportRole,
            s3ExportRole,
        });
        const instanceAssociatedRoles = [];
        const engineFeatures = engineConfig.features;
        if (s3ImportRole) {
            if (!engineFeatures?.s3Import) {
                throw new Error(`Engine '${util_1.engineDescription(props.engine)}' does not support S3 import`);
            }
            instanceAssociatedRoles.push({ roleArn: s3ImportRole.roleArn, featureName: engineFeatures?.s3Import });
        }
        if (s3ExportRole) {
            if (!engineFeatures?.s3Export) {
                throw new Error(`Engine '${util_1.engineDescription(props.engine)}' does not support S3 export`);
            }
            // only add the export feature if it's different from the import feature
            if (engineFeatures.s3Import !== engineFeatures?.s3Export) {
                instanceAssociatedRoles.push({ roleArn: s3ExportRole.roleArn, featureName: engineFeatures?.s3Export });
            }
        }
        this.instanceType = props.instanceType ?? ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
        if (props.parameterGroup && props.parameters) {
            throw new Error('You cannot specify both parameterGroup and parameters');
        }
        const dbParameterGroupName = props.parameters
            ? new parameter_group_1.ParameterGroup(this, 'ParameterGroup', {
                engine: props.engine,
                parameters: props.parameters,
            }).bindToInstance({}).parameterGroupName
            : this.newCfnProps.dbParameterGroupName;
        this.sourceCfnProps = {
            ...this.newCfnProps,
            associatedRoles: instanceAssociatedRoles.length > 0 ? instanceAssociatedRoles : undefined,
            optionGroupName: engineConfig.optionGroup?.optionGroupName,
            allocatedStorage: props.allocatedStorage?.toString() ?? '100',
            allowMajorVersionUpgrade: props.allowMajorVersionUpgrade,
            dbName: props.databaseName,
            engine: engineType,
            engineVersion: props.engine.engineVersion?.fullVersion,
            licenseModel: props.licenseModel,
            timezone: props.timezone,
            dbParameterGroupName,
        };
    }
    /**
     * Adds the single user rotation of the master password to this instance.
     *
     * @param options the options for the rotation,
     *                if you want to override the defaults
     */
    addRotationSingleUser(options = {}) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for an instance without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this instance.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...util_1.applyDefaultRotationOptions(options, this.vpcPlacement),
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this instance.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for an instance without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...util_1.applyDefaultRotationOptions(options, this.vpcPlacement),
            secret: options.secret,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            target: this,
        });
    }
}
/**
 * A database instance
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstance extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseInstanceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseInstance);
            }
            throw error;
        }
        const credentials = util_1.renderCredentials(this, props.engine, props.credentials);
        const secret = credentials.secret;
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            characterSetName: props.characterSetName,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            masterUsername: credentials.username,
            masterUserPassword: credentials.password?.unsafeUnwrap(),
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = this.getResourceNameAttribute(instance.ref);
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        instance.applyRemovalPolicy(props.removalPolicy ?? core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstance = DatabaseInstance;
_b = JSII_RTTI_SYMBOL_1;
DatabaseInstance[_b] = { fqn: "@aws-cdk/aws-rds.DatabaseInstance", version: "1.190.0" };
/**
 * A database instance restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceFromSnapshot extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseInstanceFromSnapshotProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseInstanceFromSnapshot);
            }
            throw error;
        }
        let credentials = props.credentials;
        let secret = credentials?.secret;
        if (!secret && credentials?.generatePassword) {
            if (!credentials.username) {
                throw new Error('`credentials` `username` must be specified when `generatePassword` is set to true');
            }
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: credentials.username,
                encryptionKey: credentials.encryptionKey,
                excludeCharacters: credentials.excludeCharacters,
                replaceOnPasswordCriteriaChanges: credentials.replaceOnPasswordCriteriaChanges,
                replicaRegions: credentials.replicaRegions,
            });
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            dbSnapshotIdentifier: props.snapshotIdentifier,
            masterUserPassword: secret?.secretValueFromJson('password')?.unsafeUnwrap() ?? credentials?.password?.unsafeUnwrap(),
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        instance.applyRemovalPolicy(props.removalPolicy ?? core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstanceFromSnapshot = DatabaseInstanceFromSnapshot;
_c = JSII_RTTI_SYMBOL_1;
DatabaseInstanceFromSnapshot[_c] = { fqn: "@aws-cdk/aws-rds.DatabaseInstanceFromSnapshot", version: "1.190.0" };
/**
 * A read replica database instance.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceReadReplica extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.engine = undefined;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_rds_DatabaseInstanceReadReplicaProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseInstanceReadReplica);
            }
            throw error;
        }
        if (props.sourceDatabaseInstance.engine
            && !props.sourceDatabaseInstance.engine.supportsReadReplicaBackups
            && props.backupRetention) {
            throw new Error(`Cannot set 'backupRetention', as engine '${util_1.engineDescription(props.sourceDatabaseInstance.engine)}' does not support automatic backups for read replicas`);
        }
        // The read replica instance always uses the same engine as the source instance
        // but some CF validations require the engine to be explicitely passed when some
        // properties are specified.
        const shouldPassEngine = props.domain != null;
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.newCfnProps,
            // this must be ARN, not ID, because of https://github.com/terraform-providers/terraform-provider-aws/issues/528#issuecomment-391169012
            sourceDbInstanceIdentifier: props.sourceDatabaseInstance.instanceArn,
            kmsKeyId: props.storageEncryptionKey?.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
            engine: shouldPassEngine ? props.sourceDatabaseInstance.engine?.engineType : undefined,
        });
        this.instanceType = props.instanceType;
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        instance.applyRemovalPolicy(props.removalPolicy ?? core_1.RemovalPolicy.SNAPSHOT);
        this.setLogRetention();
    }
}
exports.DatabaseInstanceReadReplica = DatabaseInstanceReadReplica;
_d = JSII_RTTI_SYMBOL_1;
DatabaseInstanceReadReplica[_d] = { fqn: "@aws-cdk/aws-rds.DatabaseInstanceReadReplica", version: "1.190.0" };
/**
 * Renders the processor features specifications
 *
 * @param features the processor features
 */
function renderProcessorFeatures(features) {
    const featuresList = Object.entries(features).map(([name, value]) => ({ name, value: value.toString() }));
    return featuresList.length === 0 ? undefined : featuresList;
}
//# sourceMappingURL=data:application/json;base64,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