'''
# Amazon Relational Database Service Construct Library

<!--BEGIN STABILITY BANNER-->---


![cfn-resources: Stable](https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge)

![cdk-constructs: Stable](https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge)

---
<!--END STABILITY BANNER-->

```python
import aws_cdk.aws_rds as rds
```

## Starting a clustered database

To set up a clustered database (like Aurora), define a `DatabaseCluster`. You must
always launch a database in a VPC. Use the `vpcSubnets` attribute to control whether
your instances will be launched privately or publicly:

```python
# vpc: ec2.Vpc

cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_2_08_1),
    credentials=rds.Credentials.from_generated_secret("clusteradmin"),  # Optional - will default to 'admin' username and generated password
    instance_props=rds.InstanceProps(
        # optional , defaults to t3.medium
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
        vpc_subnets=ec2.SubnetSelection(
            subnet_type=ec2.SubnetType.PRIVATE_WITH_NAT
        ),
        vpc=vpc
    )
)
```

If there isn't a constant for the exact version you want to use,
all of the `Version` classes have a static `of` method that can be used to create an arbitrary version.

```python
custom_engine_version = rds.AuroraMysqlEngineVersion.of("5.7.mysql_aurora.2.08.1")
```

By default, the master password will be generated and stored in AWS Secrets Manager with auto-generated description.

Your cluster will be empty by default. To add a default database upon construction, specify the
`defaultDatabaseName` attribute.

Use `DatabaseClusterFromSnapshot` to create a cluster from a snapshot:

```python
# vpc: ec2.Vpc

rds.DatabaseClusterFromSnapshot(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora(version=rds.AuroraEngineVersion.VER_1_22_2),
    instance_props=rds.InstanceProps(
        vpc=vpc
    ),
    snapshot_identifier="mySnapshot"
)
```

## Starting an instance database

To set up a instance database, define a `DatabaseInstance`. You must
always launch a database in a VPC. Use the `vpcSubnets` attribute to control whether
your instances will be launched privately or publicly:

```python
# vpc: ec2.Vpc

instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    # optional, defaults to m5.large
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.SMALL),
    credentials=rds.Credentials.from_generated_secret("syscdk"),  # Optional - will default to 'admin' username and generated password
    vpc=vpc,
    vpc_subnets=ec2.SubnetSelection(
        subnet_type=ec2.SubnetType.PRIVATE_WITH_NAT
    )
)
```

If there isn't a constant for the exact engine version you want to use,
all of the `Version` classes have a static `of` method that can be used to create an arbitrary version.

```python
custom_engine_version = rds.OracleEngineVersion.of("19.0.0.0.ru-2020-04.rur-2020-04.r1", "19")
```

By default, the master password will be generated and stored in AWS Secrets Manager.

To use the storage auto scaling option of RDS you can specify the maximum allocated storage.
This is the upper limit to which RDS can automatically scale the storage. More info can be found
[here](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling)
Example for max storage configuration:

```python
# vpc: ec2.Vpc

instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_12_3),
    # optional, defaults to m5.large
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
    vpc=vpc,
    max_allocated_storage=200
)
```

Use `DatabaseInstanceFromSnapshot` and `DatabaseInstanceReadReplica` to create an instance from snapshot or
a source database respectively:

```python
# vpc: ec2.Vpc

# source_instance: rds.DatabaseInstance

rds.DatabaseInstanceFromSnapshot(self, "Instance",
    snapshot_identifier="my-snapshot",
    engine=rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_12_3),
    # optional, defaults to m5.large
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
    vpc=vpc
)
rds.DatabaseInstanceReadReplica(self, "ReadReplica",
    source_database_instance=source_instance,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
    vpc=vpc
)
```

Automatic backups of read replica instances are only supported for MySQL and MariaDB. By default,
automatic backups are disabled for read replicas and can only be enabled (using `backupRetention`)
if also enabled on the source instance.

Creating a "production" Oracle database instance with option and parameter groups:

```python
# Set open cursors with parameter group
parameter_group = rds.ParameterGroup(self, "ParameterGroup",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    parameters={
        "open_cursors": "2500"
    }
)

option_group = rds.OptionGroup(self, "OptionGroup",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    configurations=[rds.OptionConfiguration(
        name="LOCATOR"
    ), rds.OptionConfiguration(
        name="OEM",
        port=1158,
        vpc=vpc
    )
    ]
)

# Allow connections to OEM
option_group.option_connections.OEM.connections.allow_default_port_from_any_ipv4()

# Database instance with production values
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    license_model=rds.LicenseModel.BRING_YOUR_OWN_LICENSE,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.MEDIUM),
    multi_az=True,
    storage_type=rds.StorageType.IO1,
    credentials=rds.Credentials.from_username("syscdk"),
    vpc=vpc,
    database_name="ORCL",
    storage_encrypted=True,
    backup_retention=cdk.Duration.days(7),
    monitoring_interval=cdk.Duration.seconds(60),
    enable_performance_insights=True,
    cloudwatch_logs_exports=["trace", "audit", "alert", "listener"
    ],
    cloudwatch_logs_retention=logs.RetentionDays.ONE_MONTH,
    auto_minor_version_upgrade=True,  # required to be true if LOCATOR is used in the option group
    option_group=option_group,
    parameter_group=parameter_group,
    removal_policy=RemovalPolicy.DESTROY
)

# Allow connections on default port from any IPV4
instance.connections.allow_default_port_from_any_ipv4()

# Rotate the master user password every 30 days
instance.add_rotation_single_user()

# Add alarm for high CPU
cloudwatch.Alarm(self, "HighCPU",
    metric=instance.metric_cPUUtilization(),
    threshold=90,
    evaluation_periods=1
)

# Trigger Lambda function on instance availability events
fn = lambda_.Function(self, "Function",
    code=lambda_.Code.from_inline("exports.handler = (event) => console.log(event);"),
    handler="index.handler",
    runtime=lambda_.Runtime.NODEJS_14_X
)

availability_rule = instance.on_event("Availability", target=targets.LambdaFunction(fn))
availability_rule.add_event_pattern(
    detail={
        "EventCategories": ["availability"
        ]
    }
)
```

Add XMLDB and OEM with option group

```python
# Set open cursors with parameter group
parameter_group = rds.ParameterGroup(self, "ParameterGroup",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    parameters={
        "open_cursors": "2500"
    }
)

option_group = rds.OptionGroup(self, "OptionGroup",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    configurations=[rds.OptionConfiguration(
        name="LOCATOR"
    ), rds.OptionConfiguration(
        name="OEM",
        port=1158,
        vpc=vpc
    )
    ]
)

# Allow connections to OEM
option_group.option_connections.OEM.connections.allow_default_port_from_any_ipv4()

# Database instance with production values
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.oracle_se2(version=rds.OracleEngineVersion.VER_19_0_0_0_2020_04_R1),
    license_model=rds.LicenseModel.BRING_YOUR_OWN_LICENSE,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.MEDIUM),
    multi_az=True,
    storage_type=rds.StorageType.IO1,
    credentials=rds.Credentials.from_username("syscdk"),
    vpc=vpc,
    database_name="ORCL",
    storage_encrypted=True,
    backup_retention=cdk.Duration.days(7),
    monitoring_interval=cdk.Duration.seconds(60),
    enable_performance_insights=True,
    cloudwatch_logs_exports=["trace", "audit", "alert", "listener"
    ],
    cloudwatch_logs_retention=logs.RetentionDays.ONE_MONTH,
    auto_minor_version_upgrade=True,  # required to be true if LOCATOR is used in the option group
    option_group=option_group,
    parameter_group=parameter_group,
    removal_policy=RemovalPolicy.DESTROY
)

# Allow connections on default port from any IPV4
instance.connections.allow_default_port_from_any_ipv4()

# Rotate the master user password every 30 days
instance.add_rotation_single_user()

# Add alarm for high CPU
cloudwatch.Alarm(self, "HighCPU",
    metric=instance.metric_cPUUtilization(),
    threshold=90,
    evaluation_periods=1
)

# Trigger Lambda function on instance availability events
fn = lambda_.Function(self, "Function",
    code=lambda_.Code.from_inline("exports.handler = (event) => console.log(event);"),
    handler="index.handler",
    runtime=lambda_.Runtime.NODEJS_14_X
)

availability_rule = instance.on_event("Availability", target=targets.LambdaFunction(fn))
availability_rule.add_event_pattern(
    detail={
        "EventCategories": ["availability"
        ]
    }
)
```

## Setting Public Accessibility

You can set public accessibility for the database instance or cluster using the `publiclyAccessible` property.
If you specify `true`, it creates an instance with a publicly resolvable DNS name, which resolves to a public IP address.
If you specify `false`, it creates an internal instance with a DNS name that resolves to a private IP address.
The default value depends on `vpcSubnets`.
It will be `true` if `vpcSubnets` is `subnetType: SubnetType.PUBLIC`, `false` otherwise.

```python
# vpc: ec2.Vpc

# Setting public accessibility for DB instance
rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.mysql(
        version=rds.MysqlEngineVersion.VER_8_0_19
    ),
    vpc=vpc,
    vpc_subnets=ec2.SubnetSelection(
        subnet_type=ec2.SubnetType.PRIVATE_WITH_NAT
    ),
    publicly_accessible=True
)

# Setting public accessibility for DB cluster
rds.DatabaseCluster(self, "DatabaseCluster",
    engine=rds.DatabaseClusterEngine.AURORA,
    instance_props=rds.InstanceProps(
        vpc=vpc,
        vpc_subnets=ec2.SubnetSelection(
            subnet_type=ec2.SubnetType.PRIVATE_WITH_NAT
        ),
        publicly_accessible=True
    )
)
```

## Instance events

To define Amazon CloudWatch event rules for database instances, use the `onEvent`
method:

```python
# instance: rds.DatabaseInstance
# fn: lambda.Function

rule = instance.on_event("InstanceEvent", target=targets.LambdaFunction(fn))
```

## Login credentials

By default, database instances and clusters (with the exception of `DatabaseInstanceFromSnapshot` and `ServerlessClusterFromSnapshot`) will have `admin` user with an auto-generated password.
An alternative username (and password) may be specified for the admin user instead of the default.

The following examples use a `DatabaseInstance`, but the same usage is applicable to `DatabaseCluster`.

```python
# vpc: ec2.Vpc

engine = rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_12_3)
rds.DatabaseInstance(self, "InstanceWithUsername",
    engine=engine,
    vpc=vpc,
    credentials=rds.Credentials.from_generated_secret("postgres")
)

rds.DatabaseInstance(self, "InstanceWithUsernameAndPassword",
    engine=engine,
    vpc=vpc,
    credentials=rds.Credentials.from_password("postgres", SecretValue.ssm_secure("/dbPassword", "1"))
)

my_secret = secretsmanager.Secret.from_secret_name(self, "DBSecret", "myDBLoginInfo")
rds.DatabaseInstance(self, "InstanceWithSecretLogin",
    engine=engine,
    vpc=vpc,
    credentials=rds.Credentials.from_secret(my_secret)
)
```

Secrets generated by `fromGeneratedSecret()` can be customized:

```python
# vpc: ec2.Vpc

engine = rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_12_3)
my_key = kms.Key(self, "MyKey")

rds.DatabaseInstance(self, "InstanceWithCustomizedSecret",
    engine=engine,
    vpc=vpc,
    credentials=rds.Credentials.from_generated_secret("postgres",
        secret_name="my-cool-name",
        encryption_key=my_key,
        exclude_characters="!&*^#@()",
        replica_regions=[secretsmanager.ReplicaRegion(region="eu-west-1"), secretsmanager.ReplicaRegion(region="eu-west-2")]
    )
)
```

### Snapshot credentials

As noted above, Databases created with `DatabaseInstanceFromSnapshot` or `ServerlessClusterFromSnapshot` will not create user and auto-generated password by default because it's not possible to change the master username for a snapshot. Instead, they will use the existing username and password from the snapshot. You can still generate a new password - to generate a secret similarly to the other constructs, pass in credentials with `fromGeneratedSecret()` or `fromGeneratedPassword()`.

```python
# vpc: ec2.Vpc

engine = rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_12_3)
my_key = kms.Key(self, "MyKey")

rds.DatabaseInstanceFromSnapshot(self, "InstanceFromSnapshotWithCustomizedSecret",
    engine=engine,
    vpc=vpc,
    snapshot_identifier="mySnapshot",
    credentials=rds.SnapshotCredentials.from_generated_secret("username",
        encryption_key=my_key,
        exclude_characters="!&*^#@()",
        replica_regions=[secretsmanager.ReplicaRegion(region="eu-west-1"), secretsmanager.ReplicaRegion(region="eu-west-2")]
    )
)
```

## Connecting

To control who can access the cluster or instance, use the `.connections` attribute. RDS databases have
a default port, so you don't need to specify the port:

```python
# cluster: rds.DatabaseCluster

cluster.connections.allow_from_any_ipv4(ec2.Port.all_traffic(), "Open to the world")
```

The endpoints to access your database cluster will be available as the `.clusterEndpoint` and `.readerEndpoint`
attributes:

```python
# cluster: rds.DatabaseCluster

write_address = cluster.cluster_endpoint.socket_address
```

For an instance database:

```python
# instance: rds.DatabaseInstance

address = instance.instance_endpoint.socket_address
```

## Rotating credentials

When the master password is generated and stored in AWS Secrets Manager, it can be rotated automatically:

```python
import aws_cdk.core as cdk

# instance: rds.DatabaseInstance

instance.add_rotation_single_user(
    automatically_after=cdk.Duration.days(7),  # defaults to 30 days
    exclude_characters="!@#$%^&*"
)
```

```python
cluster = rds.DatabaseCluster(stack, "Database",
    engine=rds.DatabaseClusterEngine.AURORA,
    instance_props=rds.InstanceProps(
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.SMALL),
        vpc=vpc
    )
)

cluster.add_rotation_single_user()
```

The multi user rotation scheme is also available:

```python
# instance: rds.DatabaseInstance
# my_imported_secret: rds.DatabaseSecret

instance.add_rotation_multi_user("MyUser",
    secret=my_imported_secret
)
```

It's also possible to create user credentials together with the instance/cluster and add rotation:

```python
# instance: rds.DatabaseInstance

my_user_secret = rds.DatabaseSecret(self, "MyUserSecret",
    username="myuser",
    secret_name="my-user-secret",  # optional, defaults to a CloudFormation-generated name
    master_secret=instance.secret,
    exclude_characters="{}[]()'\"/\\"
)
my_user_secret_attached = my_user_secret.attach(instance) # Adds DB connections information in the secret

instance.add_rotation_multi_user("MyUser",  # Add rotation using the multi user scheme
    secret=my_user_secret_attached)
```

**Note**: This user must be created manually in the database using the master credentials.
The rotation will start as soon as this user exists.

Access to the Secrets Manager API is required for the secret rotation. This can be achieved either with
internet connectivity (through NAT) or with a VPC interface endpoint. By default, the rotation Lambda function
is deployed in the same subnets as the instance/cluster. If access to the Secrets Manager API is not possible from
those subnets or using the default API endpoint, use the `vpcSubnets` and/or `endpoint` options:

```python
# instance: rds.DatabaseInstance
# my_endpoint: ec2.InterfaceVpcEndpoint


instance.add_rotation_single_user(
    vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PRIVATE_WITH_NAT),  # Place rotation Lambda in private subnets
    endpoint=my_endpoint
)
```

See also [@aws-cdk/aws-secretsmanager](https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-secretsmanager/README.md) for credentials rotation of existing clusters/instances.

## IAM Authentication

You can also authenticate to a database instance using AWS Identity and Access Management (IAM) database authentication;
See [https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html) for more information
and a list of supported versions and limitations.

**Note**: `grantConnect()` does not currently work - see [this GitHub issue](https://github.com/aws/aws-cdk/issues/11851).

The following example shows enabling IAM authentication for a database instance and granting connection access to an IAM role.

```python
# vpc: ec2.Vpc

instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.mysql(version=rds.MysqlEngineVersion.VER_8_0_19),
    vpc=vpc,
    iam_authentication=True
)
role = iam.Role(self, "DBRole", assumed_by=iam.AccountPrincipal(self.account))
instance.grant_connect(role)
```

The following example shows granting connection access for RDS Proxy to an IAM role.

```python
# vpc: ec2.Vpc

cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.AURORA,
    instance_props=rds.InstanceProps(vpc=vpc)
)

proxy = rds.DatabaseProxy(self, "Proxy",
    proxy_target=rds.ProxyTarget.from_cluster(cluster),
    secrets=[cluster.secret],
    vpc=vpc
)

role = iam.Role(self, "DBProxyRole", assumed_by=iam.AccountPrincipal(self.account))
proxy.grant_connect(role, "admin")
```

**Note**: In addition to the setup above, a database user will need to be created to support IAM auth.
See [https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.DBAccounts.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.DBAccounts.html) for setup instructions.

## Kerberos Authentication

You can also authenticate using Kerberos to a database instance using AWS Managed Microsoft AD for authentication;
See [https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html) for more information
and a list of supported versions and limitations.

The following example shows enabling domain support for a database instance and creating an IAM role to access
Directory Services.

```python
# vpc: ec2.Vpc

role = iam.Role(self, "RDSDirectoryServicesRole",
    assumed_by=iam.ServicePrincipal("rds.amazonaws.com"),
    managed_policies=[
        iam.ManagedPolicy.from_aws_managed_policy_name("service-role/AmazonRDSDirectoryServiceAccess")
    ]
)
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.mysql(version=rds.MysqlEngineVersion.VER_8_0_19),
    vpc=vpc,
    domain="d-????????",  # The ID of the domain for the instance to join.
    domain_role=role
)
```

**Note**: In addition to the setup above, you need to make sure that the database instance has network connectivity
to the domain controllers. This includes enabling cross-VPC traffic if in a different VPC and setting up the
appropriate security groups/network ACL to allow traffic between the database instance and domain controllers.
Once configured, see [https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html) for details
on configuring users for each available database engine.

## Metrics

Database instances and clusters both expose metrics (`cloudwatch.Metric`):

```python
# The number of database connections in use (average over 5 minutes)
# instance: rds.DatabaseInstance

# Average CPU utilization over 5 minutes
# cluster: rds.DatabaseCluster

db_connections = instance.metric_database_connections()
cpu_utilization = cluster.metric_cPUUtilization()

# The average amount of time taken per disk I/O operation (average over 1 minute)
read_latency = instance.metric("ReadLatency", statistic="Average", period=Duration.seconds(60))
```

## Enabling S3 integration

Data in S3 buckets can be imported to and exported from certain database engines using SQL queries. To enable this
functionality, set the `s3ImportBuckets` and `s3ExportBuckets` properties for import and export respectively. When
configured, the CDK automatically creates and configures IAM roles as required.
Additionally, the `s3ImportRole` and `s3ExportRole` properties can be used to set this role directly.

You can read more about loading data to (or from) S3 here:

* Aurora MySQL - [import](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.LoadFromS3.html)
  and [export](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.SaveIntoS3.html).
* Aurora PostgreSQL - [import](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html#USER_PostgreSQL.S3Import)
  and [export](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html).
* Microsoft SQL Server - [import and export](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/SQLServer.Procedural.Importing.html)
* PostgreSQL - [import](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html)
  and [export](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/postgresql-s3-export.html)
* Oracle - [import and export](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html)

The following snippet sets up a database cluster with different S3 buckets where the data is imported and exported -

```python
import aws_cdk.aws_s3 as s3

# vpc: ec2.Vpc

import_bucket = s3.Bucket(self, "importbucket")
export_bucket = s3.Bucket(self, "exportbucket")
rds.DatabaseCluster(self, "dbcluster",
    engine=rds.DatabaseClusterEngine.AURORA,
    instance_props=rds.InstanceProps(
        vpc=vpc
    ),
    s3_import_buckets=[import_bucket],
    s3_export_buckets=[export_bucket]
)
```

## Creating a Database Proxy

Amazon RDS Proxy sits between your application and your relational database to efficiently manage
connections to the database and improve scalability of the application. Learn more about at [Amazon RDS Proxy](https://aws.amazon.com/rds/proxy/)

The following code configures an RDS Proxy for a `DatabaseInstance`.

```python
# vpc: ec2.Vpc
# security_group: ec2.SecurityGroup
# secrets: List[secretsmanager.Secret[]]
# db_instance: rds.DatabaseInstance


proxy = db_instance.add_proxy("proxy",
    borrow_timeout=Duration.seconds(30),
    max_connections_percent=50,
    secrets=secrets,
    vpc=vpc
)
```

## Exporting Logs

You can publish database logs to Amazon CloudWatch Logs. With CloudWatch Logs, you can perform real-time analysis of the log data,
store the data in highly durable storage, and manage the data with the CloudWatch Logs Agent. This is available for both database
instances and clusters; the types of logs available depend on the database type and engine being used.

```python
import aws_cdk.aws_logs as logs
# my_logs_publishing_role: iam.Role
# vpc: ec2.Vpc


# Exporting logs from a cluster
cluster = rds.DatabaseCluster(self, "Database",
    engine=rds.DatabaseClusterEngine.aurora(
        version=rds.AuroraEngineVersion.VER_1_17_9
    ),
    instance_props=rds.InstanceProps(
        vpc=vpc
    ),
    cloudwatch_logs_exports=["error", "general", "slowquery", "audit"],  # Export all available MySQL-based logs
    cloudwatch_logs_retention=logs.RetentionDays.THREE_MONTHS,  # Optional - default is to never expire logs
    cloudwatch_logs_retention_role=my_logs_publishing_role
)

# Exporting logs from an instance
instance = rds.DatabaseInstance(self, "Instance",
    engine=rds.DatabaseInstanceEngine.postgres(
        version=rds.PostgresEngineVersion.VER_12_3
    ),
    vpc=vpc,
    cloudwatch_logs_exports=["postgresql"]
)
```

## Option Groups

Some DB engines offer additional features that make it easier to manage data and databases, and to provide additional security for your database.
Amazon RDS uses option groups to enable and configure these features. An option group can specify features, called options,
that are available for a particular Amazon RDS DB instance.

```python
# vpc: ec2.Vpc
# security_group: ec2.SecurityGroup


rds.OptionGroup(self, "Options",
    engine=rds.DatabaseInstanceEngine.oracle_se2(
        version=rds.OracleEngineVersion.VER_19
    ),
    configurations=[rds.OptionConfiguration(
        name="OEM",
        port=5500,
        vpc=vpc,
        security_groups=[security_group]
    )
    ]
)
```

## Parameter Groups

Database parameters specify how the database is configured.
For example, database parameters can specify the amount of resources, such as memory, to allocate to a database.
You manage your database configuration by associating your DB instances with parameter groups.
Amazon RDS defines parameter groups with default settings.

You can create your own parameter group for your cluster or instance and associate it with your database:

```python
# vpc: ec2.Vpc


parameter_group = rds.ParameterGroup(self, "ParameterGroup",
    engine=rds.DatabaseInstanceEngine.sql_server_ee(
        version=rds.SqlServerEngineVersion.VER_11
    ),
    parameters={
        "locks": "100"
    }
)

rds.DatabaseInstance(self, "Database",
    engine=rds.DatabaseInstanceEngine.SQL_SERVER_EE,
    vpc=vpc,
    parameter_group=parameter_group
)
```

Another way to specify parameters is to use the inline field `parameters` that creates an RDS parameter group for you.
You can use this if you do not want to reuse the parameter group instance for different instances:

```python
# vpc: ec2.Vpc


rds.DatabaseInstance(self, "Database",
    engine=rds.DatabaseInstanceEngine.sql_server_ee(version=rds.SqlServerEngineVersion.VER_11),
    vpc=vpc,
    parameters={
        "locks": "100"
    }
)
```

You cannot specify a parameter map and a parameter group at the same time.

## Serverless

[Amazon Aurora Serverless](https://aws.amazon.com/rds/aurora/serverless/) is an on-demand, auto-scaling configuration for Amazon
Aurora. The database will automatically start up, shut down, and scale capacity
up or down based on your application's needs. It enables you to run your database
in the cloud without managing any database instances.

The following example initializes an Aurora Serverless PostgreSql cluster.
Aurora Serverless clusters can specify scaling properties which will be used to
automatically scale the database cluster seamlessly based on the workload.

```python
# vpc: ec2.Vpc


cluster = rds.ServerlessCluster(self, "AnotherCluster",
    engine=rds.DatabaseClusterEngine.AURORA_POSTGRESQL,
    parameter_group=rds.ParameterGroup.from_parameter_group_name(self, "ParameterGroup", "default.aurora-postgresql10"),
    vpc=vpc,
    scaling=rds.ServerlessScalingOptions(
        auto_pause=Duration.minutes(10),  # default is to pause after 5 minutes of idle time
        min_capacity=rds.AuroraCapacityUnit.ACU_8,  # default is 2 Aurora capacity units (ACUs)
        max_capacity=rds.AuroraCapacityUnit.ACU_32
    )
)
```

Aurora Serverless Clusters do not support the following features:

* Loading data from an Amazon S3 bucket
* Saving data to an Amazon S3 bucket
* Invoking an AWS Lambda function with an Aurora MySQL native function
* Aurora replicas
* Backtracking
* Multi-master clusters
* Database cloning
* IAM database cloning
* IAM database authentication
* Restoring a snapshot from MySQL DB instance
* Performance Insights
* RDS Proxy

Read more about the [limitations of Aurora Serverless](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations)

Learn more about using Amazon Aurora Serverless by reading the [documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html)

Use `ServerlessClusterFromSnapshot` to create a serverless cluster from a snapshot:

```python
# vpc: ec2.Vpc

rds.ServerlessClusterFromSnapshot(self, "Cluster",
    engine=rds.DatabaseClusterEngine.AURORA_MYSQL,
    vpc=vpc,
    snapshot_identifier="mySnapshot"
)
```

### Data API

You can access your Aurora Serverless DB cluster using the built-in Data API. The Data API doesn't require a persistent connection to the DB cluster. Instead, it provides a secure HTTP endpoint and integration with AWS SDKs.

The following example shows granting Data API access to a Lamba function.

```python
# vpc: ec2.Vpc

# code: lambda.Code


cluster = rds.ServerlessCluster(self, "AnotherCluster",
    engine=rds.DatabaseClusterEngine.AURORA_MYSQL,
    vpc=vpc,  # this parameter is optional for serverless Clusters
    enable_data_api=True
)
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_14_X,
    handler="index.handler",
    code=code,
    environment={
        "CLUSTER_ARN": cluster.cluster_arn,
        "SECRET_ARN": cluster.secret.secret_arn
    }
)
cluster.grant_data_api_access(fn)
```

**Note**: To invoke the Data API, the resource will need to read the secret associated with the cluster.

To learn more about using the Data API, see the [documentation](https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html).

### Default VPC

The `vpc` parameter is optional.

If not provided, the cluster will be created in the default VPC of the account and region.
As this VPC is not deployed with AWS CDK, you can't configure the `vpcSubnets`, `subnetGroup` or `securityGroups` of the Aurora Serverless Cluster.
If you want to provide one of `vpcSubnets`, `subnetGroup` or `securityGroups` parameter, please provide a `vpc`.
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from ._jsii import *

import aws_cdk.aws_cloudwatch as _aws_cdk_aws_cloudwatch_9b88bb94
import aws_cdk.aws_ec2 as _aws_cdk_aws_ec2_67de8e8d
import aws_cdk.aws_events as _aws_cdk_aws_events_efcdfa54
import aws_cdk.aws_iam as _aws_cdk_aws_iam_940a1ce0
import aws_cdk.aws_kms as _aws_cdk_aws_kms_e491a92b
import aws_cdk.aws_logs as _aws_cdk_aws_logs_6c4320fb
import aws_cdk.aws_s3 as _aws_cdk_aws_s3_55f001a5
import aws_cdk.aws_secretsmanager as _aws_cdk_aws_secretsmanager_72af8d6f
import aws_cdk.core as _aws_cdk_core_f4b25747
import constructs as _constructs_77d1e7e8


@jsii.enum(jsii_type="@aws-cdk/aws-rds.AuroraCapacityUnit")
class AuroraCapacityUnit(enum.Enum):
    '''Aurora capacity units (ACUs).

    Each ACU is a combination of processing and memory capacity.

    :see: https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.architecture
    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        
        
        cluster = rds.ServerlessCluster(self, "AnotherCluster",
            engine=rds.DatabaseClusterEngine.AURORA_POSTGRESQL,
            parameter_group=rds.ParameterGroup.from_parameter_group_name(self, "ParameterGroup", "default.aurora-postgresql10"),
            vpc=vpc,
            scaling=rds.ServerlessScalingOptions(
                auto_pause=Duration.minutes(10),  # default is to pause after 5 minutes of idle time
                min_capacity=rds.AuroraCapacityUnit.ACU_8,  # default is 2 Aurora capacity units (ACUs)
                max_capacity=rds.AuroraCapacityUnit.ACU_32
            )
        )
    '''

    ACU_1 = "ACU_1"
    '''1 Aurora Capacity Unit.'''
    ACU_2 = "ACU_2"
    '''2 Aurora Capacity Units.'''
    ACU_4 = "ACU_4"
    '''4 Aurora Capacity Units.'''
    ACU_8 = "ACU_8"
    '''8 Aurora Capacity Units.'''
    ACU_16 = "ACU_16"
    '''16 Aurora Capacity Units.'''
    ACU_32 = "ACU_32"
    '''32 Aurora Capacity Units.'''
    ACU_64 = "ACU_64"
    '''64 Aurora Capacity Units.'''
    ACU_128 = "ACU_128"
    '''128 Aurora Capacity Units.'''
    ACU_192 = "ACU_192"
    '''192 Aurora Capacity Units.'''
    ACU_256 = "ACU_256"
    '''256 Aurora Capacity Units.'''
    ACU_384 = "ACU_384"
    '''384 Aurora Capacity Units.'''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.AuroraClusterEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class AuroraClusterEngineProps:
    def __init__(self, *, version: "AuroraEngineVersion") -> None:
        '''Creation properties of the plain Aurora database cluster engine.

        Used in {@link DatabaseClusterEngine.aurora}.

        :param version: The version of the Aurora cluster engine.

        :exampleMetadata: infused

        Example::

            # vpc: ec2.Vpc
            
            rds.DatabaseClusterFromSnapshot(self, "Database",
                engine=rds.DatabaseClusterEngine.aurora(version=rds.AuroraEngineVersion.VER_1_22_2),
                instance_props=rds.InstanceProps(
                    vpc=vpc
                ),
                snapshot_identifier="mySnapshot"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9ae80a743176ebbe12efd3ed0e3370647b08c300b1027e3c3b10537b12f75e46)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> "AuroraEngineVersion":
        '''The version of the Aurora cluster engine.'''
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return typing.cast("AuroraEngineVersion", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuroraClusterEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AuroraEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.AuroraEngineVersion",
):
    '''The versions for the Aurora cluster engine (those returned by {@link DatabaseClusterEngine.aurora}).

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        
        rds.DatabaseClusterFromSnapshot(self, "Database",
            engine=rds.DatabaseClusterEngine.aurora(version=rds.AuroraEngineVersion.VER_1_22_2),
            instance_props=rds.InstanceProps(
                vpc=vpc
            ),
            snapshot_identifier="mySnapshot"
        )
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        aurora_full_version: builtins.str,
        aurora_major_version: typing.Optional[builtins.str] = None,
    ) -> "AuroraEngineVersion":
        '''Create a new AuroraEngineVersion with an arbitrary version.

        :param aurora_full_version: the full version string, for example "5.6.mysql_aurora.1.78.3.6".
        :param aurora_major_version: the major version of the engine, defaults to "5.6".
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__91d57e2d296f33b46760deaf4ff334ed7626e66d1c77ee1b55be14002562d48c)
            check_type(argname="argument aurora_full_version", value=aurora_full_version, expected_type=type_hints["aurora_full_version"])
            check_type(argname="argument aurora_major_version", value=aurora_major_version, expected_type=type_hints["aurora_major_version"])
        return typing.cast("AuroraEngineVersion", jsii.sinvoke(cls, "of", [aurora_full_version, aurora_major_version]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_17_9")
    def VER_1_17_9(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.17.9".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_17_9"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_19_0")
    def VER_1_19_0(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.19.0".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_19_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_19_1")
    def VER_1_19_1(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.19.1".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_19_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_19_2")
    def VER_1_19_2(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.19.2".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_19_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_19_5")
    def VER_1_19_5(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.19.5".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_19_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_19_6")
    def VER_1_19_6(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.19.6".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_19_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_20_0")
    def VER_1_20_0(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.20.0".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_20_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_20_1")
    def VER_1_20_1(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.20.1".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_20_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_21_0")
    def VER_1_21_0(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.21.0".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_21_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_22_0")
    def VER_1_22_0(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.22.0".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_22_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_22_1")
    def VER_1_22_1(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.22.1".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_22_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_22_1_3")
    def VER_1_22_1_3(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.22.1.3".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_22_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_1_22_2")
    def VER_1_22_2(cls) -> "AuroraEngineVersion":
        '''Version "5.6.mysql_aurora.1.22.2".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_1_22_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10A")
    def VER_10_A(cls) -> "AuroraEngineVersion":
        '''Version "5.6.10a".'''
        return typing.cast("AuroraEngineVersion", jsii.sget(cls, "VER_10A"))

    @builtins.property
    @jsii.member(jsii_name="auroraFullVersion")
    def aurora_full_version(self) -> builtins.str:
        '''The full version string, for example, "5.6.mysql_aurora.1.78.3.6".'''
        return typing.cast(builtins.str, jsii.get(self, "auroraFullVersion"))

    @builtins.property
    @jsii.member(jsii_name="auroraMajorVersion")
    def aurora_major_version(self) -> builtins.str:
        '''The major version of the engine.

        Currently, it's always "5.6".
        '''
        return typing.cast(builtins.str, jsii.get(self, "auroraMajorVersion"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.AuroraMysqlClusterEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class AuroraMysqlClusterEngineProps:
    def __init__(self, *, version: "AuroraMysqlEngineVersion") -> None:
        '''Creation properties of the Aurora MySQL database cluster engine.

        Used in {@link DatabaseClusterEngine.auroraMysql}.

        :param version: The version of the Aurora MySQL cluster engine.

        :exampleMetadata: infused

        Example::

            # vpc: ec2.Vpc
            
            cluster = rds.DatabaseCluster(self, "Database",
                engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_2_08_1),
                credentials=rds.Credentials.from_generated_secret("clusteradmin"),  # Optional - will default to 'admin' username and generated password
                instance_props=rds.InstanceProps(
                    # optional , defaults to t3.medium
                    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                    vpc_subnets=ec2.SubnetSelection(
                        subnet_type=ec2.SubnetType.PRIVATE_WITH_NAT
                    ),
                    vpc=vpc
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e424a3527d3dd2caa0d12fa9e61b91063c3b19d5f1d9bf677b3e085f043b76e2)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> "AuroraMysqlEngineVersion":
        '''The version of the Aurora MySQL cluster engine.'''
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return typing.cast("AuroraMysqlEngineVersion", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuroraMysqlClusterEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AuroraMysqlEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.AuroraMysqlEngineVersion",
):
    '''The versions for the Aurora MySQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraMysql}).

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        
        cluster = rds.DatabaseCluster(self, "Database",
            engine=rds.DatabaseClusterEngine.aurora_mysql(version=rds.AuroraMysqlEngineVersion.VER_2_08_1),
            credentials=rds.Credentials.from_generated_secret("clusteradmin"),  # Optional - will default to 'admin' username and generated password
            instance_props=rds.InstanceProps(
                # optional , defaults to t3.medium
                instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc_subnets=ec2.SubnetSelection(
                    subnet_type=ec2.SubnetType.PRIVATE_WITH_NAT
                ),
                vpc=vpc
            )
        )
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        aurora_mysql_full_version: builtins.str,
        aurora_mysql_major_version: typing.Optional[builtins.str] = None,
    ) -> "AuroraMysqlEngineVersion":
        '''Create a new AuroraMysqlEngineVersion with an arbitrary version.

        :param aurora_mysql_full_version: the full version string, for example "5.7.mysql_aurora.2.78.3.6".
        :param aurora_mysql_major_version: the major version of the engine, defaults to "5.7".
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__09e38f708c02a55873835c133072905bdf35919f5f77e8a25669b706424f0b25)
            check_type(argname="argument aurora_mysql_full_version", value=aurora_mysql_full_version, expected_type=type_hints["aurora_mysql_full_version"])
            check_type(argname="argument aurora_mysql_major_version", value=aurora_mysql_major_version, expected_type=type_hints["aurora_mysql_major_version"])
        return typing.cast("AuroraMysqlEngineVersion", jsii.sinvoke(cls, "of", [aurora_mysql_full_version, aurora_mysql_major_version]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_03_2")
    def VER_2_03_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.03.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_03_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_03_3")
    def VER_2_03_3(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.03.3".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_03_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_03_4")
    def VER_2_03_4(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.03.4".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_03_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_0")
    def VER_2_04_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_1")
    def VER_2_04_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_2")
    def VER_2_04_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_3")
    def VER_2_04_3(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.3".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_4")
    def VER_2_04_4(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.4".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_5")
    def VER_2_04_5(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.5".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_6")
    def VER_2_04_6(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.6".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_7")
    def VER_2_04_7(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.7".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_04_8")
    def VER_2_04_8(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.04.8".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_04_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_05_0")
    def VER_2_05_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.05.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_05_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_06_0")
    def VER_2_06_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.06.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_06_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_07_0")
    def VER_2_07_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.07.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_07_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_07_1")
    def VER_2_07_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.07.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_07_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_07_2")
    def VER_2_07_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.07.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_07_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_08_0")
    def VER_2_08_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.08.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_08_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_08_1")
    def VER_2_08_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.08.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_08_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_08_2")
    def VER_2_08_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.08.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_08_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_09_0")
    def VER_2_09_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.09.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_09_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_09_1")
    def VER_2_09_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.09.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_09_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_09_2")
    def VER_2_09_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.09.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_09_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_09_3")
    def VER_2_09_3(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.09.3".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_09_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_10_0")
    def VER_2_10_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.10.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_10_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_10_1")
    def VER_2_10_1(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.10.1".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_10_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_2_10_2")
    def VER_2_10_2(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.mysql_aurora.2.10.2".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_2_10_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_3_01_0")
    def VER_3_01_0(cls) -> "AuroraMysqlEngineVersion":
        '''Version "8.0.mysql_aurora.3.01.0".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_3_01_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_5_7_12")
    def VER_5_7_12(cls) -> "AuroraMysqlEngineVersion":
        '''Version "5.7.12".'''
        return typing.cast("AuroraMysqlEngineVersion", jsii.sget(cls, "VER_5_7_12"))

    @builtins.property
    @jsii.member(jsii_name="auroraMysqlFullVersion")
    def aurora_mysql_full_version(self) -> builtins.str:
        '''The full version string, for example, "5.7.mysql_aurora.1.78.3.6".'''
        return typing.cast(builtins.str, jsii.get(self, "auroraMysqlFullVersion"))

    @builtins.property
    @jsii.member(jsii_name="auroraMysqlMajorVersion")
    def aurora_mysql_major_version(self) -> builtins.str:
        '''The major version of the engine.

        Currently, it's either "5.7", or "8.0".
        '''
        return typing.cast(builtins.str, jsii.get(self, "auroraMysqlMajorVersion"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.AuroraPostgresClusterEngineProps",
    jsii_struct_bases=[],
    name_mapping={"version": "version"},
)
class AuroraPostgresClusterEngineProps:
    def __init__(self, *, version: "AuroraPostgresEngineVersion") -> None:
        '''Creation properties of the Aurora PostgreSQL database cluster engine.

        Used in {@link DatabaseClusterEngine.auroraPostgres}.

        :param version: The version of the Aurora PostgreSQL cluster engine.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            # aurora_postgres_engine_version: rds.AuroraPostgresEngineVersion
            
            aurora_postgres_cluster_engine_props = rds.AuroraPostgresClusterEngineProps(
                version=aurora_postgres_engine_version
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__684b2dfbf317fdfc72e61bb4e5223b952b8aa6bd9b2106689fd7d826e4af46e8)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "version": version,
        }

    @builtins.property
    def version(self) -> "AuroraPostgresEngineVersion":
        '''The version of the Aurora PostgreSQL cluster engine.'''
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return typing.cast("AuroraPostgresEngineVersion", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuroraPostgresClusterEngineProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.AuroraPostgresEngineFeatures",
    jsii_struct_bases=[],
    name_mapping={"s3_export": "s3Export", "s3_import": "s3Import"},
)
class AuroraPostgresEngineFeatures:
    def __init__(
        self,
        *,
        s3_export: typing.Optional[builtins.bool] = None,
        s3_import: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Features supported by this version of the Aurora Postgres cluster engine.

        :param s3_export: Whether this version of the Aurora Postgres cluster engine supports the S3 data export feature. Default: false
        :param s3_import: Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature. Default: false

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            aurora_postgres_engine_features = rds.AuroraPostgresEngineFeatures(
                s3_export=False,
                s3_import=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c9f8248d9b77212c4625cb03e7cd480cdf162df09f37f12260ad0203f591b8ff)
            check_type(argname="argument s3_export", value=s3_export, expected_type=type_hints["s3_export"])
            check_type(argname="argument s3_import", value=s3_import, expected_type=type_hints["s3_import"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if s3_export is not None:
            self._values["s3_export"] = s3_export
        if s3_import is not None:
            self._values["s3_import"] = s3_import

    @builtins.property
    def s3_export(self) -> typing.Optional[builtins.bool]:
        '''Whether this version of the Aurora Postgres cluster engine supports the S3 data export feature.

        :default: false
        '''
        result = self._values.get("s3_export")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def s3_import(self) -> typing.Optional[builtins.bool]:
        '''Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature.

        :default: false
        '''
        result = self._values.get("s3_import")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuroraPostgresEngineFeatures(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AuroraPostgresEngineVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.AuroraPostgresEngineVersion",
):
    '''The versions for the Aurora PostgreSQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraPostgres}).

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        aurora_postgres_engine_version = rds.AuroraPostgresEngineVersion.VER_10_11
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        aurora_postgres_full_version: builtins.str,
        aurora_postgres_major_version: builtins.str,
        *,
        s3_export: typing.Optional[builtins.bool] = None,
        s3_import: typing.Optional[builtins.bool] = None,
    ) -> "AuroraPostgresEngineVersion":
        '''Create a new AuroraPostgresEngineVersion with an arbitrary version.

        :param aurora_postgres_full_version: the full version string, for example "9.6.25.1".
        :param aurora_postgres_major_version: the major version of the engine, for example "9.6".
        :param s3_export: Whether this version of the Aurora Postgres cluster engine supports the S3 data export feature. Default: false
        :param s3_import: Whether this version of the Aurora Postgres cluster engine supports the S3 data import feature. Default: false
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__57587751e479b0374162ed9c61d6b35a9dcd6320325ff3d3ce8d149cf8c2b4a7)
            check_type(argname="argument aurora_postgres_full_version", value=aurora_postgres_full_version, expected_type=type_hints["aurora_postgres_full_version"])
            check_type(argname="argument aurora_postgres_major_version", value=aurora_postgres_major_version, expected_type=type_hints["aurora_postgres_major_version"])
        aurora_postgres_features = AuroraPostgresEngineFeatures(
            s3_export=s3_export, s3_import=s3_import
        )

        return typing.cast("AuroraPostgresEngineVersion", jsii.sinvoke(cls, "of", [aurora_postgres_full_version, aurora_postgres_major_version, aurora_postgres_features]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_11")
    def VER_10_11(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.11".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_11"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_12")
    def VER_10_12(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.12".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_12"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_13")
    def VER_10_13(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.13".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_13"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_14")
    def VER_10_14(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.14".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_14"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_16")
    def VER_10_16(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.16".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_16"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_18")
    def VER_10_18(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.18".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_18"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_19")
    def VER_10_19(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.19".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_19"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_20")
    def VER_10_20(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.20".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_20"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_4")
    def VER_10_4(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.4".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_5")
    def VER_10_5(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.5".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_6")
    def VER_10_6(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.6".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_10_7")
    def VER_10_7(cls) -> "AuroraPostgresEngineVersion":
        '''Version "10.7".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_10_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_11")
    def VER_11_11(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.11".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_11"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_13")
    def VER_11_13(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.13".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_13"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_14")
    def VER_11_14(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.14".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_14"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_15")
    def VER_11_15(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.15".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_15"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_4")
    def VER_11_4(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.4".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_6")
    def VER_11_6(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.6".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_7")
    def VER_11_7(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.7".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_8")
    def VER_11_8(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.8".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_11_9")
    def VER_11_9(cls) -> "AuroraPostgresEngineVersion":
        '''Version "11.9".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_11_9"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_10")
    def VER_12_10(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.10".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_10"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_4")
    def VER_12_4(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.4".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_6")
    def VER_12_6(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.6".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_8")
    def VER_12_8(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.8".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_12_9")
    def VER_12_9(cls) -> "AuroraPostgresEngineVersion":
        '''Version "12.8".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_12_9"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_13_3")
    def VER_13_3(cls) -> "AuroraPostgresEngineVersion":
        '''Version "13.3".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_13_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_13_4")
    def VER_13_4(cls) -> "AuroraPostgresEngineVersion":
        '''Version "13.4".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_13_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_13_5")
    def VER_13_5(cls) -> "AuroraPostgresEngineVersion":
        '''Version "13.5".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_13_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_13_6")
    def VER_13_6(cls) -> "AuroraPostgresEngineVersion":
        '''Version "13.6".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_13_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_11")
    def VER_9_6_11(cls) -> "AuroraPostgresEngineVersion":
        '''Version "9.6.11".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_11"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_12")
    def VER_9_6_12(cls) -> "AuroraPostgresEngineVersion":
        '''Version "9.6.12".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_12"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_16")
    def VER_9_6_16(cls) -> "AuroraPostgresEngineVersion":
        '''Version "9.6.16".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_16"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_17")
    def VER_9_6_17(cls) -> "AuroraPostgresEngineVersion":
        '''Version "9.6.17".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_17"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_18")
    def VER_9_6_18(cls) -> "AuroraPostgresEngineVersion":
        '''Version "9.6.18".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_18"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_19")
    def VER_9_6_19(cls) -> "AuroraPostgresEngineVersion":
        '''Version "9.6.19".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_19"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_8")
    def VER_9_6_8(cls) -> "AuroraPostgresEngineVersion":
        '''Version "9.6.8".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_8"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VER_9_6_9")
    def VER_9_6_9(cls) -> "AuroraPostgresEngineVersion":
        '''Version "9.6.9".'''
        return typing.cast("AuroraPostgresEngineVersion", jsii.sget(cls, "VER_9_6_9"))

    @builtins.property
    @jsii.member(jsii_name="auroraPostgresFullVersion")
    def aurora_postgres_full_version(self) -> builtins.str:
        '''The full version string, for example, "9.6.25.1".'''
        return typing.cast(builtins.str, jsii.get(self, "auroraPostgresFullVersion"))

    @builtins.property
    @jsii.member(jsii_name="auroraPostgresMajorVersion")
    def aurora_postgres_major_version(self) -> builtins.str:
        '''The major version of the engine, for example, "9.6".'''
        return typing.cast(builtins.str, jsii.get(self, "auroraPostgresMajorVersion"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.BackupProps",
    jsii_struct_bases=[],
    name_mapping={"retention": "retention", "preferred_window": "preferredWindow"},
)
class BackupProps:
    def __init__(
        self,
        *,
        retention: _aws_cdk_core_f4b25747.Duration,
        preferred_window: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Backup configuration for RDS databases.

        :param retention: How many days to retain the backup.
        :param preferred_window: A daily time range in 24-hours UTC format in which backups preferably execute. Must be at least 30 minutes long. Example: '01:00-02:00' Default: - a 30-minute window selected at random from an 8-hour block of time for each AWS Region. To see the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow

        :default:

        - The retention period for automated backups is 1 day.
        The preferred backup window will be a 30-minute window selected at random
        from an 8-hour block of time for each AWS Region.

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            import aws_cdk.core as cdk
            
            backup_props = rds.BackupProps(
                retention=cdk.Duration.minutes(30),
            
                # the properties below are optional
                preferred_window="preferredWindow"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c42dc36b6851e49c39c3133d675adccd092642777c6958d1a0dd69fd0fa125c6)
            check_type(argname="argument retention", value=retention, expected_type=type_hints["retention"])
            check_type(argname="argument preferred_window", value=preferred_window, expected_type=type_hints["preferred_window"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "retention": retention,
        }
        if preferred_window is not None:
            self._values["preferred_window"] = preferred_window

    @builtins.property
    def retention(self) -> _aws_cdk_core_f4b25747.Duration:
        '''How many days to retain the backup.'''
        result = self._values.get("retention")
        assert result is not None, "Required property 'retention' is missing"
        return typing.cast(_aws_cdk_core_f4b25747.Duration, result)

    @builtins.property
    def preferred_window(self) -> typing.Optional[builtins.str]:
        '''A daily time range in 24-hours UTC format in which backups preferably execute.

        Must be at least 30 minutes long.

        Example: '01:00-02:00'

        :default:

        - a 30-minute window selected at random from an 8-hour block of
        time for each AWS Region. To see the time blocks available, see
        https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
        '''
        result = self._values.get("preferred_window")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BackupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnDBCluster(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBCluster",
):
    '''A CloudFormation ``AWS::RDS::DBCluster``.

    The ``AWS::RDS::DBCluster`` resource creates an Amazon Aurora DB cluster or Multi-AZ DB cluster.

    For more information about creating an Aurora DB cluster, see `Creating an Amazon Aurora DB cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.CreateInstance.html>`_ in the *Amazon Aurora User Guide* .

    For more information about creating a Multi-AZ DB cluster, see `Creating a Multi-AZ DB cluster <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/create-multi-az-db-cluster.html>`_ in the *Amazon RDS User Guide* .
    .. epigraph::

       You can only create this resource in AWS Regions where Amazon Aurora or Multi-AZ DB clusters are supported.

    *Updating DB clusters*

    When properties labeled " *Update requires:* `Replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ " are updated, AWS CloudFormation first creates a replacement DB cluster, then changes references from other dependent resources to point to the replacement DB cluster, and finally deletes the old DB cluster.
    .. epigraph::

       We highly recommend that you take a snapshot of the database before updating the stack. If you don't, you lose the data when AWS CloudFormation replaces your DB cluster. To preserve your data, perform the following procedure:

       - Deactivate any applications that are using the DB cluster so that there's no activity on the DB instance.
       - Create a snapshot of the DB cluster. For more information, see `Creating a DB Cluster Snapshot <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_CreateSnapshotCluster.html>`_ .
       - If you want to restore your DB cluster using a DB cluster snapshot, modify the updated template with your DB cluster changes and add the ``SnapshotIdentifier`` property with the ID of the DB cluster snapshot that you want to use.

       After you restore a DB cluster with a ``SnapshotIdentifier`` property, you must specify the same ``SnapshotIdentifier`` property for any future updates to the DB cluster. When you specify this property for an update, the DB cluster is not restored from the DB cluster snapshot again, and the data in the database is not changed. However, if you don't specify the ``SnapshotIdentifier`` property, an empty DB cluster is created, and the original DB cluster is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB cluster is restored from the specified ``SnapshotIdentifier`` property, and the original DB cluster is deleted.

       - Update the stack.

    Currently, when you are updating the stack for an Aurora Serverless DB cluster, you can't include changes to any other properties when you specify one of the following properties: ``PreferredBackupWindow`` , ``PreferredMaintenanceWindow`` , and ``Port`` . This limitation doesn't apply to provisioned DB clusters.

    For more information about updating other properties of this resource, see ``[ModifyDBCluster](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_ModifyDBCluster.html)`` . For more information about updating stacks, see `AWS CloudFormation Stacks Updates <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks.html>`_ .

    *Deleting DB clusters*

    The default ``DeletionPolicy`` for ``AWS::RDS::DBCluster`` resources is ``Snapshot`` . For more information about how AWS CloudFormation deletes resources, see `DeletionPolicy Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html>`_ .

    :cloudformationResource: AWS::RDS::DBCluster
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        cfn_dBCluster = rds.CfnDBCluster(self, "MyCfnDBCluster",
            allocated_storage=123,
            associated_roles=[rds.CfnDBCluster.DBClusterRoleProperty(
                role_arn="roleArn",
        
                # the properties below are optional
                feature_name="featureName"
            )],
            auto_minor_version_upgrade=False,
            availability_zones=["availabilityZones"],
            backtrack_window=123,
            backup_retention_period=123,
            copy_tags_to_snapshot=False,
            database_name="databaseName",
            db_cluster_identifier="dbClusterIdentifier",
            db_cluster_instance_class="dbClusterInstanceClass",
            db_cluster_parameter_group_name="dbClusterParameterGroupName",
            db_instance_parameter_group_name="dbInstanceParameterGroupName",
            db_subnet_group_name="dbSubnetGroupName",
            db_system_id="dbSystemId",
            deletion_protection=False,
            domain="domain",
            domain_iam_role_name="domainIamRoleName",
            enable_cloudwatch_logs_exports=["enableCloudwatchLogsExports"],
            enable_http_endpoint=False,
            enable_iam_database_authentication=False,
            engine="engine",
            engine_mode="engineMode",
            engine_version="engineVersion",
            global_cluster_identifier="globalClusterIdentifier",
            iops=123,
            kms_key_id="kmsKeyId",
            manage_master_user_password=False,
            master_username="masterUsername",
            master_user_password="masterUserPassword",
            master_user_secret=rds.CfnDBCluster.MasterUserSecretProperty(
                kms_key_id="kmsKeyId",
                secret_arn="secretArn"
            ),
            monitoring_interval=123,
            monitoring_role_arn="monitoringRoleArn",
            network_type="networkType",
            performance_insights_enabled=False,
            performance_insights_kms_key_id="performanceInsightsKmsKeyId",
            performance_insights_retention_period=123,
            port=123,
            preferred_backup_window="preferredBackupWindow",
            preferred_maintenance_window="preferredMaintenanceWindow",
            publicly_accessible=False,
            replication_source_identifier="replicationSourceIdentifier",
            restore_type="restoreType",
            scaling_configuration=rds.CfnDBCluster.ScalingConfigurationProperty(
                auto_pause=False,
                max_capacity=123,
                min_capacity=123,
                seconds_before_timeout=123,
                seconds_until_auto_pause=123,
                timeout_action="timeoutAction"
            ),
            serverless_v2_scaling_configuration=rds.CfnDBCluster.ServerlessV2ScalingConfigurationProperty(
                max_capacity=123,
                min_capacity=123
            ),
            snapshot_identifier="snapshotIdentifier",
            source_db_cluster_identifier="sourceDbClusterIdentifier",
            source_region="sourceRegion",
            storage_encrypted=False,
            storage_type="storageType",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            use_latest_restorable_time=False,
            vpc_security_group_ids=["vpcSecurityGroupIds"]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        allocated_storage: typing.Optional[jsii.Number] = None,
        associated_roles: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[typing.Union["CfnDBCluster.DBClusterRoleProperty", typing.Dict[builtins.str, typing.Any]], _aws_cdk_core_f4b25747.IResolvable]]]] = None,
        auto_minor_version_upgrade: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        availability_zones: typing.Optional[typing.Sequence[builtins.str]] = None,
        backtrack_window: typing.Optional[jsii.Number] = None,
        backup_retention_period: typing.Optional[jsii.Number] = None,
        copy_tags_to_snapshot: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        database_name: typing.Optional[builtins.str] = None,
        db_cluster_identifier: typing.Optional[builtins.str] = None,
        db_cluster_instance_class: typing.Optional[builtins.str] = None,
        db_cluster_parameter_group_name: typing.Optional[builtins.str] = None,
        db_instance_parameter_group_name: typing.Optional[builtins.str] = None,
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        db_system_id: typing.Optional[builtins.str] = None,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_iam_role_name: typing.Optional[builtins.str] = None,
        enable_cloudwatch_logs_exports: typing.Optional[typing.Sequence[builtins.str]] = None,
        enable_http_endpoint: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        enable_iam_database_authentication: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_mode: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        global_cluster_identifier: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        manage_master_user_password: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        master_username: typing.Optional[builtins.str] = None,
        master_user_password: typing.Optional[builtins.str] = None,
        master_user_secret: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDBCluster.MasterUserSecretProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        monitoring_interval: typing.Optional[jsii.Number] = None,
        monitoring_role_arn: typing.Optional[builtins.str] = None,
        network_type: typing.Optional[builtins.str] = None,
        performance_insights_enabled: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        performance_insights_kms_key_id: typing.Optional[builtins.str] = None,
        performance_insights_retention_period: typing.Optional[jsii.Number] = None,
        port: typing.Optional[jsii.Number] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        publicly_accessible: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        replication_source_identifier: typing.Optional[builtins.str] = None,
        restore_type: typing.Optional[builtins.str] = None,
        scaling_configuration: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDBCluster.ScalingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        serverless_v2_scaling_configuration: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDBCluster.ServerlessV2ScalingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        snapshot_identifier: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        source_region: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        storage_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
        use_latest_restorable_time: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        vpc_security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBCluster``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param allocated_storage: The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster. This setting is required to create a Multi-AZ DB cluster. Valid for: Multi-AZ DB clusters only
        :param associated_roles: Provides a list of the AWS Identity and Access Management (IAM) roles that are associated with the DB cluster. IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other Amazon Web Services on your behalf. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param auto_minor_version_upgrade: A value that indicates whether minor engine upgrades are applied automatically to the DB cluster during the maintenance window. By default, minor engine upgrades are applied automatically. Valid for: Multi-AZ DB clusters only
        :param availability_zones: A list of Availability Zones (AZs) where instances in the DB cluster can be created. For information on AWS Regions and Availability Zones, see `Choosing the Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param backtrack_window: The target backtrack window, in seconds. To disable backtracking, set this value to 0. .. epigraph:: Currently, Backtrack is only supported for Aurora MySQL DB clusters. Default: 0 Constraints: - If specified, this value must be set to a number from 0 to 259,200 (72 hours). Valid for: Aurora MySQL DB clusters only
        :param backup_retention_period: The number of days for which automated backups are retained. Default: 1 Constraints: - Must be a value from 1 to 35 Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param copy_tags_to_snapshot: A value that indicates whether to copy all tags from the DB cluster to snapshots of the DB cluster. The default is not to copy them. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param database_name: The name of your database. If you don't provide a name, then Amazon RDS won't create a database in this DB cluster. For naming constraints, see `Naming Constraints <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_cluster_identifier: The DB cluster identifier. This parameter is stored as a lowercase string. Constraints: - Must contain from 1 to 63 letters, numbers, or hyphens. - First character must be a letter. - Can't end with a hyphen or contain two consecutive hyphens. Example: ``my-cluster1`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_cluster_instance_class: The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example db.m6gd.xlarge. Not all DB instance classes are available in all AWS Regions , or for all database engines. For the full list of DB instance classes and availability for your engine, see `DB instance class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide* . This setting is required to create a Multi-AZ DB cluster. Valid for: Multi-AZ DB clusters only
        :param db_cluster_parameter_group_name: The name of the DB cluster parameter group to associate with this DB cluster. .. epigraph:: If you apply a parameter group to an existing DB cluster, then its DB instances might need to reboot. This can result in an outage while the DB instances are rebooting. If you apply a change to parameter group associated with a stopped DB cluster, then the update stack waits until the DB cluster is started. To list all of the available DB cluster parameter group names, use the following command: ``aws rds describe-db-cluster-parameter-groups --query "DBClusterParameterGroups[].DBClusterParameterGroupName" --output text`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_instance_parameter_group_name: The name of the DB parameter group to apply to all instances of the DB cluster. .. epigraph:: When you apply a parameter group using the ``DBInstanceParameterGroupName`` parameter, the DB cluster isn't rebooted automatically. Also, parameter changes are applied immediately rather than during the next maintenance window. Default: The existing name setting Constraints: - The DB parameter group must be in the same DB parameter group family as this DB cluster.
        :param db_subnet_group_name: A DB subnet group that you want to associate with this DB cluster. If you are restoring a DB cluster to a point in time with ``RestoreType`` set to ``copy-on-write`` , and don't specify a DB subnet group name, then the DB cluster is restored with a default DB subnet group. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_system_id: Reserved for future use.
        :param deletion_protection: A value that indicates whether the DB cluster has deletion protection enabled. The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param domain: Indicates the directory ID of the Active Directory to create the DB cluster. For Amazon Aurora DB clusters, Amazon RDS can use Kerberos authentication to authenticate users that connect to the DB cluster. For more information, see `Kerberos authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/kerberos-authentication.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param domain_iam_role_name: Specifies the name of the IAM role to use when making API calls to the Directory Service. Valid for: Aurora DB clusters only
        :param enable_cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Aurora User Guide* . *Aurora MySQL* Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery`` *Aurora PostgreSQL* Valid values: ``postgresql`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param enable_http_endpoint: A value that indicates whether to enable the HTTP endpoint for an Aurora Serverless DB cluster. By default, the HTTP endpoint is disabled. When enabled, the HTTP endpoint provides a connectionless web service API for running SQL queries on the Aurora Serverless DB cluster. You can also query your database from inside the RDS console with the query editor. For more information, see `Using the Data API for Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param enable_iam_database_authentication: A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts. By default, mapping is disabled. For more information, see `IAM Database Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon Aurora User Guide.* Valid for: Aurora DB clusters only
        :param engine: The name of the database engine to be used for this DB cluster. Valid Values: - ``aurora`` (for MySQL 5.6-compatible Aurora) - ``aurora-mysql`` (for MySQL 5.7-compatible Aurora) - ``aurora-postgresql`` - ``mysql`` - ``postgres`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param engine_mode: The DB engine mode of the DB cluster, either ``provisioned`` , ``serverless`` , ``parallelquery`` , ``global`` , or ``multimaster`` . The ``parallelquery`` engine mode isn't required for Aurora MySQL version 1.23 and higher 1.x versions, and version 2.09 and higher 2.x versions. The ``global`` engine mode isn't required for Aurora MySQL version 1.22 and higher 1.x versions, and ``global`` engine mode isn't required for any 2.x versions. The ``multimaster`` engine mode only applies for DB clusters created with Aurora MySQL version 5.6.10a. For Aurora PostgreSQL, the ``global`` engine mode isn't required, and both the ``parallelquery`` and the ``multimaster`` engine modes currently aren't supported. Limitations and requirements apply to some DB engine modes. For more information, see the following sections in the *Amazon Aurora User Guide* : - `Limitations of Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations>`_ - `Limitations of Parallel Query <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-mysql-parallel-query.html#aurora-mysql-parallel-query-limitations>`_ - `Limitations of Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html#aurora-global-database.limitations>`_ - `Limitations of Multi-Master Clusters <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-multi-master.html#aurora-multi-master-limitations>`_ Valid for: Aurora DB clusters only
        :param engine_version: The version number of the database engine to use. To list all of the available engine versions for ``aurora`` (for MySQL 5.6-compatible Aurora), use the following command: ``aws rds describe-db-engine-versions --engine aurora --query "DBEngineVersions[].EngineVersion"`` To list all of the available engine versions for ``aurora-mysql`` (for MySQL 5.7-compatible Aurora), use the following command: ``aws rds describe-db-engine-versions --engine aurora-mysql --query "DBEngineVersions[].EngineVersion"`` To list all of the available engine versions for ``aurora-postgresql`` , use the following command: ``aws rds describe-db-engine-versions --engine aurora-postgresql --query "DBEngineVersions[].EngineVersion"`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param global_cluster_identifier: If you are configuring an Aurora global database cluster and want your Aurora DB cluster to be a secondary member in the global database cluster, specify the global cluster ID of the global database cluster. To define the primary database cluster of the global cluster, use the `AWS::RDS::GlobalCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html>`_ resource. If you aren't configuring a global database cluster, don't specify this property. .. epigraph:: To remove the DB cluster from a global database cluster, specify an empty value for the ``GlobalClusterIdentifier`` property. For information about Aurora global databases, see `Working with Amazon Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param iops: The amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster. For information about valid IOPS values, see `Amazon RDS Provisioned IOPS storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* . This setting is required to create a Multi-AZ DB cluster. Constraints: Must be a multiple between .5 and 50 of the storage amount for the DB cluster. Valid for: Multi-AZ DB clusters only
        :param kms_key_id: The Amazon Resource Name (ARN) of the AWS KMS key that is used to encrypt the database instances in the DB cluster, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` . If you enable the ``StorageEncrypted`` property but don't specify this property, the default KMS key is used. If you specify this property, you must set the ``StorageEncrypted`` property to ``true`` . If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param manage_master_user_password: A value that indicates whether to manage the master user password with AWS Secrets Manager. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.* Constraints: - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param master_username: The name of the master user for the DB cluster. .. epigraph:: If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param master_user_password: The master password for the DB instance. .. epigraph:: If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param master_user_secret: Contains the secret managed by RDS in AWS Secrets Manager for the master user password. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*
        :param monitoring_interval: The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB cluster. To turn off collecting Enhanced Monitoring metrics, specify 0. The default is 0. If ``MonitoringRoleArn`` is specified, also set ``MonitoringInterval`` to a value other than 0. Valid Values: ``0, 1, 5, 10, 15, 30, 60`` Valid for: Multi-AZ DB clusters only
        :param monitoring_role_arn: The Amazon Resource Name (ARN) for the IAM role that permits RDS to send Enhanced Monitoring metrics to Amazon CloudWatch Logs. An example is ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting up and enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* . If ``MonitoringInterval`` is set to a value other than 0, supply a ``MonitoringRoleArn`` value. Valid for: Multi-AZ DB clusters only
        :param network_type: The network type of the DB cluster. Valid values: - ``IPV4`` - ``DUAL`` The network type is determined by the ``DBSubnetGroup`` specified for the DB cluster. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ). For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon Aurora User Guide.* Valid for: Aurora DB clusters only
        :param performance_insights_enabled: A value that indicates whether to turn on Performance Insights for the DB cluster. For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* . Valid for: Multi-AZ DB clusters only
        :param performance_insights_kms_key_id: The AWS KMS key identifier for encryption of Performance Insights data. The AWS KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. If you don't specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account . Your AWS account has a different default KMS key for each AWS Region . Valid for: Multi-AZ DB clusters only
        :param performance_insights_retention_period: The number of days to retain Performance Insights data. The default is 7 days. The following values are valid:. - 7 - *month* * 31, where *month* is a number of months from 1-23 - 731 For example, the following values are valid: - 93 (3 months * 31) - 341 (11 months * 31) - 589 (19 months * 31) - 731 If you specify a retention period such as 94, which isn't a valid value, RDS issues an error. Valid for: Multi-AZ DB clusters only
        :param port: The port number on which the DB instances in the DB cluster accept connections. Default: - When ``EngineMode`` is ``provisioned`` , ``3306`` (for both Aurora MySQL and Aurora PostgreSQL) - When ``EngineMode`` is ``serverless`` : - ``3306`` when ``Engine`` is ``aurora`` or ``aurora-mysql`` - ``5432`` when ``Engine`` is ``aurora-postgresql`` .. epigraph:: The ``No interruption`` on update behavior only applies to DB clusters. If you are updating a DB instance, see `Port <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-port>`_ for the AWS::RDS::DBInstance resource. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param preferred_backup_window: The daily time range during which automated backups are created. For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html#Aurora.Managing.Backups.BackupWindow>`_ in the *Amazon Aurora User Guide.* Constraints: - Must be in the format ``hh24:mi-hh24:mi`` . - Must be in Universal Coordinated Time (UTC). - Must not conflict with the preferred maintenance window. - Must be at least 30 minutes. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param preferred_maintenance_window: The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC). Format: ``ddd:hh24:mi-ddd:hh24:mi`` The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Cluster Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow.Aurora>`_ in the *Amazon Aurora User Guide.* Valid Days: Mon, Tue, Wed, Thu, Fri, Sat, Sun. Constraints: Minimum 30-minute window. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param publicly_accessible: A value that indicates whether the DB cluster is publicly accessible. When the DB cluster is publicly accessible, its Domain Name System (DNS) endpoint resolves to the private IP address from within the DB cluster's virtual private cloud (VPC). It resolves to the public IP address from outside of the DB cluster's VPC. Access to the DB cluster is ultimately controlled by the security group it uses. That public access isn't permitted if the security group assigned to the DB cluster doesn't permit it. When the DB cluster isn't publicly accessible, it is an internal DB cluster with a DNS name that resolves to a private IP address. Default: The default behavior varies depending on whether ``DBSubnetGroupName`` is specified. If ``DBSubnetGroupName`` isn't specified, and ``PubliclyAccessible`` isn't specified, the following applies: - If the default VPC in the target Region doesn’t have an internet gateway attached to it, the DB cluster is private. - If the default VPC in the target Region has an internet gateway attached to it, the DB cluster is public. If ``DBSubnetGroupName`` is specified, and ``PubliclyAccessible`` isn't specified, the following applies: - If the subnets are part of a VPC that doesn’t have an internet gateway attached to it, the DB cluster is private. - If the subnets are part of a VPC that has an internet gateway attached to it, the DB cluster is public. Valid for: Multi-AZ DB clusters only
        :param replication_source_identifier: The Amazon Resource Name (ARN) of the source DB instance or DB cluster if this DB cluster is created as a read replica. Valid for: Aurora DB clusters only
        :param restore_type: The type of restore to be performed. You can specify one of the following values:. - ``full-copy`` - The new DB cluster is restored as a full copy of the source DB cluster. - ``copy-on-write`` - The new DB cluster is restored as a clone of the source DB cluster. Constraints: You can't specify ``copy-on-write`` if the engine version of the source DB cluster is earlier than 1.11. If you don't specify a ``RestoreType`` value, then the new DB cluster is restored as a full copy of the source DB cluster. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param scaling_configuration: The ``ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless DB cluster. This property is only supported for Aurora Serverless v1. For Aurora Serverless v2, use ``ServerlessV2ScalingConfiguration`` property. Valid for: Aurora DB clusters only
        :param serverless_v2_scaling_configuration: The ``ServerlessV2ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless V2 DB cluster. This property is only supported for Aurora Serverless v2. For Aurora Serverless v1, use ``ScalingConfiguration`` property. Valid for: Aurora DB clusters only
        :param snapshot_identifier: The identifier for the DB snapshot or DB cluster snapshot to restore from. You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot. However, you can use only the ARN to specify a DB snapshot. After you restore a DB cluster with a ``SnapshotIdentifier`` property, you must specify the same ``SnapshotIdentifier`` property for any future updates to the DB cluster. When you specify this property for an update, the DB cluster is not restored from the snapshot again, and the data in the database is not changed. However, if you don't specify the ``SnapshotIdentifier`` property, an empty DB cluster is created, and the original DB cluster is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB cluster is restored from the specified ``SnapshotIdentifier`` property, and the original DB cluster is deleted. If you specify the ``SnapshotIdentifier`` property to restore a DB cluster (as opposed to specifying it for DB cluster updates), then don't specify the following properties: - ``GlobalClusterIdentifier`` - ``MasterUsername`` - ``MasterUserPassword`` - ``ReplicationSourceIdentifier`` - ``RestoreType`` - ``SourceDBClusterIdentifier`` - ``SourceRegion`` - ``StorageEncrypted`` (for an encrypted snapshot) - ``UseLatestRestorableTime`` Constraints: - Must match the identifier of an existing Snapshot. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param source_db_cluster_identifier: When restoring a DB cluster to a point in time, the identifier of the source DB cluster from which to restore. Constraints: - Must match the identifier of an existing DBCluster. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param source_region: The AWS Region which contains the source DB cluster when replicating a DB cluster. For example, ``us-east-1`` . Valid for: Aurora DB clusters only
        :param storage_encrypted: Indicates whether the DB cluster is encrypted. If you specify the ``KmsKeyId`` property, then you must enable encryption. If you specify the ``SourceDBClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB cluster is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB cluster to be encrypted, then don't set this property or set it to ``false`` . Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param storage_type: Specifies the storage type to be associated with the DB cluster. This setting is required to create a Multi-AZ DB cluster. Valid values: ``io1`` When specified, a value for the ``Iops`` parameter is required. Default: ``io1`` Valid for: Multi-AZ DB clusters only
        :param tags: An optional array of key-value pairs to apply to this DB cluster. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param use_latest_restorable_time: A value that indicates whether to restore the DB cluster to the latest restorable backup time. By default, the DB cluster is not restored to the latest restorable backup time. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param vpc_security_group_ids: A list of EC2 VPC security groups to associate with this DB cluster. If you plan to update the resource, don't specify VPC security groups in a shared VPC. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b543cb60a42192f0805eba0f04ddaba91da0e5750ae106e7befd7838a9ede570)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBClusterProps(
            allocated_storage=allocated_storage,
            associated_roles=associated_roles,
            auto_minor_version_upgrade=auto_minor_version_upgrade,
            availability_zones=availability_zones,
            backtrack_window=backtrack_window,
            backup_retention_period=backup_retention_period,
            copy_tags_to_snapshot=copy_tags_to_snapshot,
            database_name=database_name,
            db_cluster_identifier=db_cluster_identifier,
            db_cluster_instance_class=db_cluster_instance_class,
            db_cluster_parameter_group_name=db_cluster_parameter_group_name,
            db_instance_parameter_group_name=db_instance_parameter_group_name,
            db_subnet_group_name=db_subnet_group_name,
            db_system_id=db_system_id,
            deletion_protection=deletion_protection,
            domain=domain,
            domain_iam_role_name=domain_iam_role_name,
            enable_cloudwatch_logs_exports=enable_cloudwatch_logs_exports,
            enable_http_endpoint=enable_http_endpoint,
            enable_iam_database_authentication=enable_iam_database_authentication,
            engine=engine,
            engine_mode=engine_mode,
            engine_version=engine_version,
            global_cluster_identifier=global_cluster_identifier,
            iops=iops,
            kms_key_id=kms_key_id,
            manage_master_user_password=manage_master_user_password,
            master_username=master_username,
            master_user_password=master_user_password,
            master_user_secret=master_user_secret,
            monitoring_interval=monitoring_interval,
            monitoring_role_arn=monitoring_role_arn,
            network_type=network_type,
            performance_insights_enabled=performance_insights_enabled,
            performance_insights_kms_key_id=performance_insights_kms_key_id,
            performance_insights_retention_period=performance_insights_retention_period,
            port=port,
            preferred_backup_window=preferred_backup_window,
            preferred_maintenance_window=preferred_maintenance_window,
            publicly_accessible=publicly_accessible,
            replication_source_identifier=replication_source_identifier,
            restore_type=restore_type,
            scaling_configuration=scaling_configuration,
            serverless_v2_scaling_configuration=serverless_v2_scaling_configuration,
            snapshot_identifier=snapshot_identifier,
            source_db_cluster_identifier=source_db_cluster_identifier,
            source_region=source_region,
            storage_encrypted=storage_encrypted,
            storage_type=storage_type,
            tags=tags,
            use_latest_restorable_time=use_latest_restorable_time,
            vpc_security_group_ids=vpc_security_group_ids,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c7b0707fdad5269063399f7850153cdf93088727c79f2ed6a1c1460b941a67d4)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6fa5aceb5a30860a82b41ba3d23c362e84c166a07a83a07635df349391d390c7)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDbClusterArn")
    def attr_db_cluster_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) for the DB cluster.

        :cloudformationAttribute: DBClusterArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbClusterArn"))

    @builtins.property
    @jsii.member(jsii_name="attrDbClusterResourceId")
    def attr_db_cluster_resource_id(self) -> builtins.str:
        '''The AWS Region -unique, immutable identifier for the DB cluster.

        This identifier is found in AWS CloudTrail log entries whenever the KMS key for the DB cluster is accessed.

        :cloudformationAttribute: DBClusterResourceId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbClusterResourceId"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpointAddress")
    def attr_endpoint_address(self) -> builtins.str:
        '''The connection endpoint for the DB cluster.

        For example: ``mystack-mydbcluster-123456789012.us-east-2.rds.amazonaws.com``

        :cloudformationAttribute: Endpoint.Address
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpointAddress"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpointPort")
    def attr_endpoint_port(self) -> builtins.str:
        '''The port number that will accept connections on this DB cluster.

        For example: ``3306``

        :cloudformationAttribute: Endpoint.Port
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpointPort"))

    @builtins.property
    @jsii.member(jsii_name="attrMasterUserSecretSecretArn")
    def attr_master_user_secret_secret_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the secret.

        :cloudformationAttribute: MasterUserSecret.SecretArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMasterUserSecretSecretArn"))

    @builtins.property
    @jsii.member(jsii_name="attrReadEndpointAddress")
    def attr_read_endpoint_address(self) -> builtins.str:
        '''The reader endpoint for the DB cluster.

        For example: ``mystack-mydbcluster-ro-123456789012.us-east-2.rds.amazonaws.com``

        :cloudformationAttribute: ReadEndpoint.Address
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrReadEndpointAddress"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _aws_cdk_core_f4b25747.TagManager:
        '''An optional array of key-value pairs to apply to this DB cluster.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-tags
        '''
        return typing.cast(_aws_cdk_core_f4b25747.TagManager, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="allocatedStorage")
    def allocated_storage(self) -> typing.Optional[jsii.Number]:
        '''The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster.

        This setting is required to create a Multi-AZ DB cluster.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-allocatedstorage
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "allocatedStorage"))

    @allocated_storage.setter
    def allocated_storage(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__859b98f84f55fe7e65ee83a6df7170e44026c2963f87ebb1fb3b6d42fee432e9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allocatedStorage", value)

    @builtins.property
    @jsii.member(jsii_name="associatedRoles")
    def associated_roles(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union["CfnDBCluster.DBClusterRoleProperty", _aws_cdk_core_f4b25747.IResolvable]]]]:
        '''Provides a list of the AWS Identity and Access Management (IAM) roles that are associated with the DB cluster.

        IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other Amazon Web Services on your behalf.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-associatedroles
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union["CfnDBCluster.DBClusterRoleProperty", _aws_cdk_core_f4b25747.IResolvable]]]], jsii.get(self, "associatedRoles"))

    @associated_roles.setter
    def associated_roles(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union["CfnDBCluster.DBClusterRoleProperty", _aws_cdk_core_f4b25747.IResolvable]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d6ff1687c27120fd2f85f7d3d0362a957552e18c3ad7f3c42bbc3a6d7c6e69c6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "associatedRoles", value)

    @builtins.property
    @jsii.member(jsii_name="autoMinorVersionUpgrade")
    def auto_minor_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether minor engine upgrades are applied automatically to the DB cluster during the maintenance window.

        By default, minor engine upgrades are applied automatically.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-autominorversionupgrade
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "autoMinorVersionUpgrade"))

    @auto_minor_version_upgrade.setter
    def auto_minor_version_upgrade(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0d97345fd4820b3521f75d1bb379f58a6ac1fa2640f46d937948f114f3540cef)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoMinorVersionUpgrade", value)

    @builtins.property
    @jsii.member(jsii_name="availabilityZones")
    def availability_zones(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of Availability Zones (AZs) where instances in the DB cluster can be created.

        For information on AWS Regions and Availability Zones, see `Choosing the Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-availabilityzones
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "availabilityZones"))

    @availability_zones.setter
    def availability_zones(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9aef7cbc887a4aae7d48979ae15bf81b69ce63588f7f4ead72f036014fecdb65)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "availabilityZones", value)

    @builtins.property
    @jsii.member(jsii_name="backtrackWindow")
    def backtrack_window(self) -> typing.Optional[jsii.Number]:
        '''The target backtrack window, in seconds. To disable backtracking, set this value to 0.

        .. epigraph::

           Currently, Backtrack is only supported for Aurora MySQL DB clusters.

        Default: 0

        Constraints:

        - If specified, this value must be set to a number from 0 to 259,200 (72 hours).

        Valid for: Aurora MySQL DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-backtrackwindow
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "backtrackWindow"))

    @backtrack_window.setter
    def backtrack_window(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6aea09837759b098be8a2821347dc29b122fac753a4b4f4c20fe81db7bdc677b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "backtrackWindow", value)

    @builtins.property
    @jsii.member(jsii_name="backupRetentionPeriod")
    def backup_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days for which automated backups are retained.

        Default: 1

        Constraints:

        - Must be a value from 1 to 35

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-backupretentionperiod
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "backupRetentionPeriod"))

    @backup_retention_period.setter
    def backup_retention_period(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__30f4683cc521423a6cae730caf24305c58cf6364a668381591f23453b7a738c7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "backupRetentionPeriod", value)

    @builtins.property
    @jsii.member(jsii_name="copyTagsToSnapshot")
    def copy_tags_to_snapshot(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to copy all tags from the DB cluster to snapshots of the DB cluster.

        The default is not to copy them.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-copytagstosnapshot
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "copyTagsToSnapshot"))

    @copy_tags_to_snapshot.setter
    def copy_tags_to_snapshot(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4743697d4d426f7ae9df2656e45fca91975ec26e80d042dfa13c525fe24d3fcb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "copyTagsToSnapshot", value)

    @builtins.property
    @jsii.member(jsii_name="databaseName")
    def database_name(self) -> typing.Optional[builtins.str]:
        '''The name of your database.

        If you don't provide a name, then Amazon RDS won't create a database in this DB cluster. For naming constraints, see `Naming Constraints <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-databasename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "databaseName"))

    @database_name.setter
    def database_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__234d299919c54c6c1c31dc61301c74f5808c4092af17b6f36eb26898f0832422)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "databaseName", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterIdentifier")
    def db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The DB cluster identifier. This parameter is stored as a lowercase string.

        Constraints:

        - Must contain from 1 to 63 letters, numbers, or hyphens.
        - First character must be a letter.
        - Can't end with a hyphen or contain two consecutive hyphens.

        Example: ``my-cluster1``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbClusterIdentifier"))

    @db_cluster_identifier.setter
    def db_cluster_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__891db01653c2ab88788e3bdf9bbe26c24dd1debaaaf79a3f36eb274c57811c6b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterInstanceClass")
    def db_cluster_instance_class(self) -> typing.Optional[builtins.str]:
        '''The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example db.m6gd.xlarge. Not all DB instance classes are available in all AWS Regions , or for all database engines.

        For the full list of DB instance classes and availability for your engine, see `DB instance class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide* .

        This setting is required to create a Multi-AZ DB cluster.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusterinstanceclass
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbClusterInstanceClass"))

    @db_cluster_instance_class.setter
    def db_cluster_instance_class(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ec0a69b00f5661b57bd7365d3449ef6fd164b1e1ca2d0cf4ec8a0320dd2b47ea)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterInstanceClass", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterParameterGroupName")
    def db_cluster_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB cluster parameter group to associate with this DB cluster.

        .. epigraph::

           If you apply a parameter group to an existing DB cluster, then its DB instances might need to reboot. This can result in an outage while the DB instances are rebooting.

           If you apply a change to parameter group associated with a stopped DB cluster, then the update stack waits until the DB cluster is started.

        To list all of the available DB cluster parameter group names, use the following command:

        ``aws rds describe-db-cluster-parameter-groups --query "DBClusterParameterGroups[].DBClusterParameterGroupName" --output text``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusterparametergroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbClusterParameterGroupName"))

    @db_cluster_parameter_group_name.setter
    def db_cluster_parameter_group_name(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__22e70685341157bf52bd2a790a6fa69843b21f49993584472ab53ae63b5410bf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterParameterGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="dbInstanceParameterGroupName")
    def db_instance_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB parameter group to apply to all instances of the DB cluster.

        .. epigraph::

           When you apply a parameter group using the ``DBInstanceParameterGroupName`` parameter, the DB cluster isn't rebooted automatically. Also, parameter changes are applied immediately rather than during the next maintenance window.

        Default: The existing name setting

        Constraints:

        - The DB parameter group must be in the same DB parameter group family as this DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbinstanceparametergroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbInstanceParameterGroupName"))

    @db_instance_parameter_group_name.setter
    def db_instance_parameter_group_name(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8f271ba0e9f910f72245494057f4b594c9034f2815abcc8b5866988fbb24a404)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbInstanceParameterGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="dbSubnetGroupName")
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        '''A DB subnet group that you want to associate with this DB cluster.

        If you are restoring a DB cluster to a point in time with ``RestoreType`` set to ``copy-on-write`` , and don't specify a DB subnet group name, then the DB cluster is restored with a default DB subnet group.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbsubnetgroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbSubnetGroupName"))

    @db_subnet_group_name.setter
    def db_subnet_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__73cd72f370b9b8fd4006a9a0801a6116a63ad25848606bdcabdcd5acbdb461fd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSubnetGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="dbSystemId")
    def db_system_id(self) -> typing.Optional[builtins.str]:
        '''Reserved for future use.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbsystemid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbSystemId"))

    @db_system_id.setter
    def db_system_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f704946f2e521645a3d19aa7571bfb954f8c51f815bab9ae5ee21af1a55532c6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSystemId", value)

    @builtins.property
    @jsii.member(jsii_name="deletionProtection")
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether the DB cluster has deletion protection enabled.

        The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-deletionprotection
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "deletionProtection"))

    @deletion_protection.setter
    def deletion_protection(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__84a87b5d34b516f02a5f6ee178addea84b57376d7b21fee654b4444b9d6e9c8f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deletionProtection", value)

    @builtins.property
    @jsii.member(jsii_name="domain")
    def domain(self) -> typing.Optional[builtins.str]:
        '''Indicates the directory ID of the Active Directory to create the DB cluster.

        For Amazon Aurora DB clusters, Amazon RDS can use Kerberos authentication to authenticate users that connect to the DB cluster.

        For more information, see `Kerberos authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/kerberos-authentication.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-domain
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domain"))

    @domain.setter
    def domain(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c4ae6b897175d39007b6f5110955004ed79888a1bf4250d2ee6c2914e4bd87a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domain", value)

    @builtins.property
    @jsii.member(jsii_name="domainIamRoleName")
    def domain_iam_role_name(self) -> typing.Optional[builtins.str]:
        '''Specifies the name of the IAM role to use when making API calls to the Directory Service.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-domainiamrolename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domainIamRoleName"))

    @domain_iam_role_name.setter
    def domain_iam_role_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d922113c3d3f6a2ea04ffb090cb8fbdd4b0ced1422bc42a0e6863f7e4f9f2b69)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainIamRoleName", value)

    @builtins.property
    @jsii.member(jsii_name="enableCloudwatchLogsExports")
    def enable_cloudwatch_logs_exports(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of log types that need to be enabled for exporting to CloudWatch Logs.

        The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Aurora User Guide* .

        *Aurora MySQL*

        Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery``

        *Aurora PostgreSQL*

        Valid values: ``postgresql``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enablecloudwatchlogsexports
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "enableCloudwatchLogsExports"))

    @enable_cloudwatch_logs_exports.setter
    def enable_cloudwatch_logs_exports(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e04c59e27679cdd1385f72d99b9933e3bb9980449da9e25525ebad16b3c3ef40)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableCloudwatchLogsExports", value)

    @builtins.property
    @jsii.member(jsii_name="enableHttpEndpoint")
    def enable_http_endpoint(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to enable the HTTP endpoint for an Aurora Serverless DB cluster.

        By default, the HTTP endpoint is disabled.

        When enabled, the HTTP endpoint provides a connectionless web service API for running SQL queries on the Aurora Serverless DB cluster. You can also query your database from inside the RDS console with the query editor.

        For more information, see `Using the Data API for Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enablehttpendpoint
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "enableHttpEndpoint"))

    @enable_http_endpoint.setter
    def enable_http_endpoint(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9203a3ca9e6051c60f219c5c5f196d20b81a43b1e6aa0484c5edb4bba0b8ed05)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableHttpEndpoint", value)

    @builtins.property
    @jsii.member(jsii_name="enableIamDatabaseAuthentication")
    def enable_iam_database_authentication(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.

        By default, mapping is disabled.

        For more information, see `IAM Database Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon Aurora User Guide.*

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enableiamdatabaseauthentication
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "enableIamDatabaseAuthentication"))

    @enable_iam_database_authentication.setter
    def enable_iam_database_authentication(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__65edebea3296f54520c24221777d8ec4c2ac7fd7321482a9b89fb49afdcebe4a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableIamDatabaseAuthentication", value)

    @builtins.property
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional[builtins.str]:
        '''The name of the database engine to be used for this DB cluster.

        Valid Values:

        - ``aurora`` (for MySQL 5.6-compatible Aurora)
        - ``aurora-mysql`` (for MySQL 5.7-compatible Aurora)
        - ``aurora-postgresql``
        - ``mysql``
        - ``postgres``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engine
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engine"))

    @engine.setter
    def engine(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__70f9d741428ffc2a6c1df7e99d7080ef00b8a11e15ddeb03e762af793859d018)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engine", value)

    @builtins.property
    @jsii.member(jsii_name="engineMode")
    def engine_mode(self) -> typing.Optional[builtins.str]:
        '''The DB engine mode of the DB cluster, either ``provisioned`` , ``serverless`` , ``parallelquery`` , ``global`` , or ``multimaster`` .

        The ``parallelquery`` engine mode isn't required for Aurora MySQL version 1.23 and higher 1.x versions, and version 2.09 and higher 2.x versions.

        The ``global`` engine mode isn't required for Aurora MySQL version 1.22 and higher 1.x versions, and ``global`` engine mode isn't required for any 2.x versions.

        The ``multimaster`` engine mode only applies for DB clusters created with Aurora MySQL version 5.6.10a.

        For Aurora PostgreSQL, the ``global`` engine mode isn't required, and both the ``parallelquery`` and the ``multimaster`` engine modes currently aren't supported.

        Limitations and requirements apply to some DB engine modes. For more information, see the following sections in the *Amazon Aurora User Guide* :

        - `Limitations of Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations>`_
        - `Limitations of Parallel Query <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-mysql-parallel-query.html#aurora-mysql-parallel-query-limitations>`_
        - `Limitations of Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html#aurora-global-database.limitations>`_
        - `Limitations of Multi-Master Clusters <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-multi-master.html#aurora-multi-master-limitations>`_

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enginemode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engineMode"))

    @engine_mode.setter
    def engine_mode(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5dc4c1639abe880fdb9f2c369364a2612900e06c40cb571041a87b2593ab3c61)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engineMode", value)

    @builtins.property
    @jsii.member(jsii_name="engineVersion")
    def engine_version(self) -> typing.Optional[builtins.str]:
        '''The version number of the database engine to use.

        To list all of the available engine versions for ``aurora`` (for MySQL 5.6-compatible Aurora), use the following command:

        ``aws rds describe-db-engine-versions --engine aurora --query "DBEngineVersions[].EngineVersion"``

        To list all of the available engine versions for ``aurora-mysql`` (for MySQL 5.7-compatible Aurora), use the following command:

        ``aws rds describe-db-engine-versions --engine aurora-mysql --query "DBEngineVersions[].EngineVersion"``

        To list all of the available engine versions for ``aurora-postgresql`` , use the following command:

        ``aws rds describe-db-engine-versions --engine aurora-postgresql --query "DBEngineVersions[].EngineVersion"``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engineversion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engineVersion"))

    @engine_version.setter
    def engine_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ca85f1b0bf46697056a4bf55dd465fe6a346a84f4013cd50f3ce055295496d6a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engineVersion", value)

    @builtins.property
    @jsii.member(jsii_name="globalClusterIdentifier")
    def global_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''If you are configuring an Aurora global database cluster and want your Aurora DB cluster to be a secondary member in the global database cluster, specify the global cluster ID of the global database cluster.

        To define the primary database cluster of the global cluster, use the `AWS::RDS::GlobalCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html>`_ resource.

        If you aren't configuring a global database cluster, don't specify this property.
        .. epigraph::

           To remove the DB cluster from a global database cluster, specify an empty value for the ``GlobalClusterIdentifier`` property.

        For information about Aurora global databases, see `Working with Amazon Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-globalclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "globalClusterIdentifier"))

    @global_cluster_identifier.setter
    def global_cluster_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3adf62942cc49eb95f26bae95f4dc930a1cb425c400031a251dd528bd4731748)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "globalClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="iops")
    def iops(self) -> typing.Optional[jsii.Number]:
        '''The amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster.

        For information about valid IOPS values, see `Amazon RDS Provisioned IOPS storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* .

        This setting is required to create a Multi-AZ DB cluster.

        Constraints: Must be a multiple between .5 and 50 of the storage amount for the DB cluster.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-iops
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "iops"))

    @iops.setter
    def iops(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6409dfb520fba62ffbd31c806fd8ca229cb694b56d86436f2434e8f771f9ef18)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "iops", value)

    @builtins.property
    @jsii.member(jsii_name="kmsKeyId")
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the AWS KMS key that is used to encrypt the database instances in the DB cluster, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` .

        If you enable the ``StorageEncrypted`` property but don't specify this property, the default KMS key is used. If you specify this property, you must set the ``StorageEncrypted`` property to ``true`` .

        If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-kmskeyid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kmsKeyId"))

    @kms_key_id.setter
    def kms_key_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__739a3d34cac838c1debb06a3783836ec2e5922f071c10522c4c31ec5ff5e8a7e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kmsKeyId", value)

    @builtins.property
    @jsii.member(jsii_name="manageMasterUserPassword")
    def manage_master_user_password(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to manage the master user password with AWS Secrets Manager.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*

        Constraints:

        - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-managemasteruserpassword
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "manageMasterUserPassword"))

    @manage_master_user_password.setter
    def manage_master_user_password(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__324750fc3f929bcb7d38b0331f629748a74fe6e197ebc19042ad415b01f055ef)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "manageMasterUserPassword", value)

    @builtins.property
    @jsii.member(jsii_name="masterUsername")
    def master_username(self) -> typing.Optional[builtins.str]:
        '''The name of the master user for the DB cluster.

        .. epigraph::

           If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masterusername
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "masterUsername"))

    @master_username.setter
    def master_username(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9b87c37343c8d38d353be0b5b66bd9bda0287dc6d68d01bbf4c095a38e3c7cb9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "masterUsername", value)

    @builtins.property
    @jsii.member(jsii_name="masterUserPassword")
    def master_user_password(self) -> typing.Optional[builtins.str]:
        '''The master password for the DB instance.

        .. epigraph::

           If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masteruserpassword
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "masterUserPassword"))

    @master_user_password.setter
    def master_user_password(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__57b220b6c8de8edd1486d3f32463fb44f3cfa77e387faa4faf3dab8b8cc57373)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "masterUserPassword", value)

    @builtins.property
    @jsii.member(jsii_name="masterUserSecret")
    def master_user_secret(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBCluster.MasterUserSecretProperty"]]:
        '''Contains the secret managed by RDS in AWS Secrets Manager for the master user password.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masterusersecret
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBCluster.MasterUserSecretProperty"]], jsii.get(self, "masterUserSecret"))

    @master_user_secret.setter
    def master_user_secret(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBCluster.MasterUserSecretProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dc20613fca6aed093506d09922a4b9f0c2a8c97bb0d1d5d60c9e9770a8bc1d11)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "masterUserSecret", value)

    @builtins.property
    @jsii.member(jsii_name="monitoringInterval")
    def monitoring_interval(self) -> typing.Optional[jsii.Number]:
        '''The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB cluster.

        To turn off collecting Enhanced Monitoring metrics, specify 0. The default is 0.

        If ``MonitoringRoleArn`` is specified, also set ``MonitoringInterval`` to a value other than 0.

        Valid Values: ``0, 1, 5, 10, 15, 30, 60``

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-monitoringinterval
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "monitoringInterval"))

    @monitoring_interval.setter
    def monitoring_interval(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__81f7983018e1d22a01a163ffb4a1e6ab32ec25877a8a713d5edf4203b06040ce)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "monitoringInterval", value)

    @builtins.property
    @jsii.member(jsii_name="monitoringRoleArn")
    def monitoring_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) for the IAM role that permits RDS to send Enhanced Monitoring metrics to Amazon CloudWatch Logs.

        An example is ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting up and enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* .

        If ``MonitoringInterval`` is set to a value other than 0, supply a ``MonitoringRoleArn`` value.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-monitoringrolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "monitoringRoleArn"))

    @monitoring_role_arn.setter
    def monitoring_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__939fb0e55684648a37fb924f1f013de5db693ffcf47597e64095a56c592d241c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "monitoringRoleArn", value)

    @builtins.property
    @jsii.member(jsii_name="networkType")
    def network_type(self) -> typing.Optional[builtins.str]:
        '''The network type of the DB cluster.

        Valid values:

        - ``IPV4``
        - ``DUAL``

        The network type is determined by the ``DBSubnetGroup`` specified for the DB cluster. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ).

        For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon Aurora User Guide.*

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-networktype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "networkType"))

    @network_type.setter
    def network_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4978b63527773db38542911f13e844f33cca41330202796d80984199d895cd15)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "networkType", value)

    @builtins.property
    @jsii.member(jsii_name="performanceInsightsEnabled")
    def performance_insights_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to turn on Performance Insights for the DB cluster.

        For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* .

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-performanceinsightsenabled
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "performanceInsightsEnabled"))

    @performance_insights_enabled.setter
    def performance_insights_enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__128672514bbd63d4eb7d822ca97fa3d8ed92dc6a102d44020e89224acde1a3ac)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "performanceInsightsEnabled", value)

    @builtins.property
    @jsii.member(jsii_name="performanceInsightsKmsKeyId")
    def performance_insights_kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key identifier for encryption of Performance Insights data.

        The AWS KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.

        If you don't specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account . Your AWS account has a different default KMS key for each AWS Region .

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-performanceinsightskmskeyid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "performanceInsightsKmsKeyId"))

    @performance_insights_kms_key_id.setter
    def performance_insights_kms_key_id(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8f82005be35f7fe840adb883cee04c03b7d6efe70c3b47e557e208d20055ed75)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "performanceInsightsKmsKeyId", value)

    @builtins.property
    @jsii.member(jsii_name="performanceInsightsRetentionPeriod")
    def performance_insights_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days to retain Performance Insights data. The default is 7 days. The following values are valid:.

        - 7
        - *month* * 31, where *month* is a number of months from 1-23
        - 731

        For example, the following values are valid:

        - 93 (3 months * 31)
        - 341 (11 months * 31)
        - 589 (19 months * 31)
        - 731

        If you specify a retention period such as 94, which isn't a valid value, RDS issues an error.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-performanceinsightsretentionperiod
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "performanceInsightsRetentionPeriod"))

    @performance_insights_retention_period.setter
    def performance_insights_retention_period(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__809b7f9277787ef1142cde7eb28925d3c97f5071437be3542bcd48a0a5ab19cb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "performanceInsightsRetentionPeriod", value)

    @builtins.property
    @jsii.member(jsii_name="port")
    def port(self) -> typing.Optional[jsii.Number]:
        '''The port number on which the DB instances in the DB cluster accept connections.

        Default:

        - When ``EngineMode`` is ``provisioned`` , ``3306`` (for both Aurora MySQL and Aurora PostgreSQL)
        - When ``EngineMode`` is ``serverless`` :
        - ``3306`` when ``Engine`` is ``aurora`` or ``aurora-mysql``
        - ``5432`` when ``Engine`` is ``aurora-postgresql``

        .. epigraph::

           The ``No interruption`` on update behavior only applies to DB clusters. If you are updating a DB instance, see `Port <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-port>`_ for the AWS::RDS::DBInstance resource.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-port
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "port"))

    @port.setter
    def port(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__869d3b2b4fb6921728a4b58dae2b2b944545f453ade636f5c427bc10ff5c1bce)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "port", value)

    @builtins.property
    @jsii.member(jsii_name="preferredBackupWindow")
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        '''The daily time range during which automated backups are created.

        For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html#Aurora.Managing.Backups.BackupWindow>`_ in the *Amazon Aurora User Guide.*

        Constraints:

        - Must be in the format ``hh24:mi-hh24:mi`` .
        - Must be in Universal Coordinated Time (UTC).
        - Must not conflict with the preferred maintenance window.
        - Must be at least 30 minutes.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-preferredbackupwindow
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "preferredBackupWindow"))

    @preferred_backup_window.setter
    def preferred_backup_window(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e1ff742ff86da03084aa959f89e8d80614a7cc3b79334e5f00b28a245f2f0907)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "preferredBackupWindow", value)

    @builtins.property
    @jsii.member(jsii_name="preferredMaintenanceWindow")
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        '''The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).

        Format: ``ddd:hh24:mi-ddd:hh24:mi``

        The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Cluster Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow.Aurora>`_ in the *Amazon Aurora User Guide.*

        Valid Days: Mon, Tue, Wed, Thu, Fri, Sat, Sun.

        Constraints: Minimum 30-minute window.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-preferredmaintenancewindow
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "preferredMaintenanceWindow"))

    @preferred_maintenance_window.setter
    def preferred_maintenance_window(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4448fa3b0e982ffa4a814ad2bb8bc04437cbc9f6b03e8f4e13911b7d52598d71)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "preferredMaintenanceWindow", value)

    @builtins.property
    @jsii.member(jsii_name="publiclyAccessible")
    def publicly_accessible(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether the DB cluster is publicly accessible.

        When the DB cluster is publicly accessible, its Domain Name System (DNS) endpoint resolves to the private IP address from within the DB cluster's virtual private cloud (VPC). It resolves to the public IP address from outside of the DB cluster's VPC. Access to the DB cluster is ultimately controlled by the security group it uses. That public access isn't permitted if the security group assigned to the DB cluster doesn't permit it.

        When the DB cluster isn't publicly accessible, it is an internal DB cluster with a DNS name that resolves to a private IP address.

        Default: The default behavior varies depending on whether ``DBSubnetGroupName`` is specified.

        If ``DBSubnetGroupName`` isn't specified, and ``PubliclyAccessible`` isn't specified, the following applies:

        - If the default VPC in the target Region doesn’t have an internet gateway attached to it, the DB cluster is private.
        - If the default VPC in the target Region has an internet gateway attached to it, the DB cluster is public.

        If ``DBSubnetGroupName`` is specified, and ``PubliclyAccessible`` isn't specified, the following applies:

        - If the subnets are part of a VPC that doesn’t have an internet gateway attached to it, the DB cluster is private.
        - If the subnets are part of a VPC that has an internet gateway attached to it, the DB cluster is public.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-publiclyaccessible
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "publiclyAccessible"))

    @publicly_accessible.setter
    def publicly_accessible(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a06efa997ca7c5a98f61f64a9e9c14afff9fd1d3ef3098e43ea6a2691db29c12)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "publiclyAccessible", value)

    @builtins.property
    @jsii.member(jsii_name="replicationSourceIdentifier")
    def replication_source_identifier(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the source DB instance or DB cluster if this DB cluster is created as a read replica.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-replicationsourceidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "replicationSourceIdentifier"))

    @replication_source_identifier.setter
    def replication_source_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__291fc65caf4929cd61b5579d32c93005a4eff51eff50aef216da748a66ea1aa1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "replicationSourceIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="restoreType")
    def restore_type(self) -> typing.Optional[builtins.str]:
        '''The type of restore to be performed. You can specify one of the following values:.

        - ``full-copy`` - The new DB cluster is restored as a full copy of the source DB cluster.
        - ``copy-on-write`` - The new DB cluster is restored as a clone of the source DB cluster.

        Constraints: You can't specify ``copy-on-write`` if the engine version of the source DB cluster is earlier than 1.11.

        If you don't specify a ``RestoreType`` value, then the new DB cluster is restored as a full copy of the source DB cluster.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-restoretype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "restoreType"))

    @restore_type.setter
    def restore_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__428a25131746d49a34a9bf48c42cdb9fb1ef2b8b9b9f7e95b396ba486c5cbdcb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restoreType", value)

    @builtins.property
    @jsii.member(jsii_name="scalingConfiguration")
    def scaling_configuration(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBCluster.ScalingConfigurationProperty"]]:
        '''The ``ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless DB cluster.

        This property is only supported for Aurora Serverless v1. For Aurora Serverless v2, use ``ServerlessV2ScalingConfiguration`` property.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-scalingconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBCluster.ScalingConfigurationProperty"]], jsii.get(self, "scalingConfiguration"))

    @scaling_configuration.setter
    def scaling_configuration(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBCluster.ScalingConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2e810a02502a81fd539d12db9d09244808e45857d0d015817c52a7d1043a7c8b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "scalingConfiguration", value)

    @builtins.property
    @jsii.member(jsii_name="serverlessV2ScalingConfiguration")
    def serverless_v2_scaling_configuration(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBCluster.ServerlessV2ScalingConfigurationProperty"]]:
        '''The ``ServerlessV2ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless V2 DB cluster.

        This property is only supported for Aurora Serverless v2. For Aurora Serverless v1, use ``ScalingConfiguration`` property.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-serverlessv2scalingconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBCluster.ServerlessV2ScalingConfigurationProperty"]], jsii.get(self, "serverlessV2ScalingConfiguration"))

    @serverless_v2_scaling_configuration.setter
    def serverless_v2_scaling_configuration(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBCluster.ServerlessV2ScalingConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c8974c7b11198cf416847a7d8696038f2a436c83d95bb5428248472fb13db2c5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "serverlessV2ScalingConfiguration", value)

    @builtins.property
    @jsii.member(jsii_name="snapshotIdentifier")
    def snapshot_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier for the DB snapshot or DB cluster snapshot to restore from.

        You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot. However, you can use only the ARN to specify a DB snapshot.

        After you restore a DB cluster with a ``SnapshotIdentifier`` property, you must specify the same ``SnapshotIdentifier`` property for any future updates to the DB cluster. When you specify this property for an update, the DB cluster is not restored from the snapshot again, and the data in the database is not changed. However, if you don't specify the ``SnapshotIdentifier`` property, an empty DB cluster is created, and the original DB cluster is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB cluster is restored from the specified ``SnapshotIdentifier`` property, and the original DB cluster is deleted.

        If you specify the ``SnapshotIdentifier`` property to restore a DB cluster (as opposed to specifying it for DB cluster updates), then don't specify the following properties:

        - ``GlobalClusterIdentifier``
        - ``MasterUsername``
        - ``MasterUserPassword``
        - ``ReplicationSourceIdentifier``
        - ``RestoreType``
        - ``SourceDBClusterIdentifier``
        - ``SourceRegion``
        - ``StorageEncrypted`` (for an encrypted snapshot)
        - ``UseLatestRestorableTime``

        Constraints:

        - Must match the identifier of an existing Snapshot.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-snapshotidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "snapshotIdentifier"))

    @snapshot_identifier.setter
    def snapshot_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cb7282ba3308de10bea9e0040c23bb9db99ba5a0dcc60531db9eca001e73366e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "snapshotIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="sourceDbClusterIdentifier")
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''When restoring a DB cluster to a point in time, the identifier of the source DB cluster from which to restore.

        Constraints:

        - Must match the identifier of an existing DBCluster.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-sourcedbclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDbClusterIdentifier"))

    @source_db_cluster_identifier.setter
    def source_db_cluster_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8dfb6e43921c53a91251426270af5cf81a5ca8ce49eed51b3a764e1328167655)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceDbClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="sourceRegion")
    def source_region(self) -> typing.Optional[builtins.str]:
        '''The AWS Region which contains the source DB cluster when replicating a DB cluster. For example, ``us-east-1`` .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-sourceregion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceRegion"))

    @source_region.setter
    def source_region(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2a5e7016259b5992bf75f81aa691de404f3147d4d5e35bef2f25fa2c1d4ccdd1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceRegion", value)

    @builtins.property
    @jsii.member(jsii_name="storageEncrypted")
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''Indicates whether the DB cluster is encrypted.

        If you specify the ``KmsKeyId`` property, then you must enable encryption.

        If you specify the ``SourceDBClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB cluster is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB cluster to be encrypted, then don't set this property or set it to ``false`` .

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-storageencrypted
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "storageEncrypted"))

    @storage_encrypted.setter
    def storage_encrypted(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__709ed8bf3da2250f7c6cca9241cd7ad8bf46fbfeb3f88a5cfbe81dc6736c06b3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageEncrypted", value)

    @builtins.property
    @jsii.member(jsii_name="storageType")
    def storage_type(self) -> typing.Optional[builtins.str]:
        '''Specifies the storage type to be associated with the DB cluster.

        This setting is required to create a Multi-AZ DB cluster.

        Valid values: ``io1``

        When specified, a value for the ``Iops`` parameter is required.

        Default: ``io1``

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-storagetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "storageType"))

    @storage_type.setter
    def storage_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__06ff258edf307546cc996ff47b59ac6e6f36eda8e2a160d3ac9773233a62c556)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageType", value)

    @builtins.property
    @jsii.member(jsii_name="useLatestRestorableTime")
    def use_latest_restorable_time(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to restore the DB cluster to the latest restorable backup time.

        By default, the DB cluster is not restored to the latest restorable backup time.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-uselatestrestorabletime
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "useLatestRestorableTime"))

    @use_latest_restorable_time.setter
    def use_latest_restorable_time(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9bd59627aef7cc84dc755d861bce8f5e62683bcd78aa1a5fb5b8e5fb373329b0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "useLatestRestorableTime", value)

    @builtins.property
    @jsii.member(jsii_name="vpcSecurityGroupIds")
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of EC2 VPC security groups to associate with this DB cluster.

        If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-vpcsecuritygroupids
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "vpcSecurityGroupIds"))

    @vpc_security_group_ids.setter
    def vpc_security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b7cc3e6f1a8ccad57c0fb57074fe2175a3aac3416217710626ecfc6907001b32)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcSecurityGroupIds", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBCluster.DBClusterRoleProperty",
        jsii_struct_bases=[],
        name_mapping={"role_arn": "roleArn", "feature_name": "featureName"},
    )
    class DBClusterRoleProperty:
        def __init__(
            self,
            *,
            role_arn: builtins.str,
            feature_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Describes an AWS Identity and Access Management (IAM) role that is associated with a DB cluster.

            :param role_arn: The Amazon Resource Name (ARN) of the IAM role that is associated with the DB cluster.
            :param feature_name: The name of the feature associated with the AWS Identity and Access Management (IAM) role. IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other AWS services on your behalf. For the list of supported feature names, see the ``SupportedFeatureNames`` description in `DBEngineVersion <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html>`_ in the *Amazon RDS API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-dbclusterrole.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                d_bCluster_role_property = rds.CfnDBCluster.DBClusterRoleProperty(
                    role_arn="roleArn",
                
                    # the properties below are optional
                    feature_name="featureName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b089eb1c5730b363a16f841c8ec009bef14d35f5803a19ac0a238ac3bce463c0)
                check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
                check_type(argname="argument feature_name", value=feature_name, expected_type=type_hints["feature_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "role_arn": role_arn,
            }
            if feature_name is not None:
                self._values["feature_name"] = feature_name

        @builtins.property
        def role_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the IAM role that is associated with the DB cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-dbclusterrole.html#cfn-rds-dbcluster-dbclusterrole-rolearn
            '''
            result = self._values.get("role_arn")
            assert result is not None, "Required property 'role_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def feature_name(self) -> typing.Optional[builtins.str]:
            '''The name of the feature associated with the AWS Identity and Access Management (IAM) role.

            IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other AWS services on your behalf. For the list of supported feature names, see the ``SupportedFeatureNames`` description in `DBEngineVersion <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html>`_ in the *Amazon RDS API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-dbclusterrole.html#cfn-rds-dbcluster-dbclusterrole-featurename
            '''
            result = self._values.get("feature_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DBClusterRoleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBCluster.EndpointProperty",
        jsii_struct_bases=[],
        name_mapping={"address": "address", "port": "port"},
    )
    class EndpointProperty:
        def __init__(
            self,
            *,
            address: typing.Optional[builtins.str] = None,
            port: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies the connection endpoint for the primary instance of the DB cluster.

            :param address: Specifies the connection endpoint for the primary instance of the DB cluster.
            :param port: Specifies the port that the database engine is listening on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-endpoint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                endpoint_property = rds.CfnDBCluster.EndpointProperty(
                    address="address",
                    port="port"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__78d7e57d7cd7e1588df8806c0ee029eea42a57f00fd77a988eae755429fdee84)
                check_type(argname="argument address", value=address, expected_type=type_hints["address"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if address is not None:
                self._values["address"] = address
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def address(self) -> typing.Optional[builtins.str]:
            '''Specifies the connection endpoint for the primary instance of the DB cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-endpoint.html#cfn-rds-dbcluster-endpoint-address
            '''
            result = self._values.get("address")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[builtins.str]:
            '''Specifies the port that the database engine is listening on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-endpoint.html#cfn-rds-dbcluster-endpoint-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EndpointProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBCluster.MasterUserSecretProperty",
        jsii_struct_bases=[],
        name_mapping={"kms_key_id": "kmsKeyId", "secret_arn": "secretArn"},
    )
    class MasterUserSecretProperty:
        def __init__(
            self,
            *,
            kms_key_id: typing.Optional[builtins.str] = None,
            secret_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains the secret managed by RDS in AWS Secrets Manager for the master user password.

            For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*

            :param kms_key_id: The AWS KMS key identifier that is used to encrypt the secret.
            :param secret_arn: The Amazon Resource Name (ARN) of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-masterusersecret.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                master_user_secret_property = rds.CfnDBCluster.MasterUserSecretProperty(
                    kms_key_id="kmsKeyId",
                    secret_arn="secretArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d9dfdb205fa3a622d90a543b0c47ad01507c5204b76f1e66f2a51061d45e608e)
                check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
                check_type(argname="argument secret_arn", value=secret_arn, expected_type=type_hints["secret_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if kms_key_id is not None:
                self._values["kms_key_id"] = kms_key_id
            if secret_arn is not None:
                self._values["secret_arn"] = secret_arn

        @builtins.property
        def kms_key_id(self) -> typing.Optional[builtins.str]:
            '''The AWS KMS key identifier that is used to encrypt the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-masterusersecret.html#cfn-rds-dbcluster-masterusersecret-kmskeyid
            '''
            result = self._values.get("kms_key_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def secret_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-masterusersecret.html#cfn-rds-dbcluster-masterusersecret-secretarn
            '''
            result = self._values.get("secret_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MasterUserSecretProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBCluster.ReadEndpointProperty",
        jsii_struct_bases=[],
        name_mapping={"address": "address"},
    )
    class ReadEndpointProperty:
        def __init__(self, *, address: typing.Optional[builtins.str] = None) -> None:
            '''The ``ReadEndpoint`` return value specifies the reader endpoint for the DB cluster.

            The reader endpoint for a DB cluster load-balances connections across the Aurora Replicas that are available in a DB cluster. As clients request new connections to the reader endpoint, Aurora distributes the connection requests among the Aurora Replicas in the DB cluster. This functionality can help balance your read workload across multiple Aurora Replicas in your DB cluster.

            If a failover occurs, and the Aurora Replica that you are connected to is promoted to be the primary instance, your connection is dropped. To continue sending your read workload to other Aurora Replicas in the cluster, you can then reconnect to the reader endpoint.

            For more information about Aurora endpoints, see `Amazon Aurora connection management <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Overview.Endpoints.html>`_ in the *Amazon Aurora User Guide* .

            :param address: The host address of the reader endpoint.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-readendpoint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                read_endpoint_property = rds.CfnDBCluster.ReadEndpointProperty(
                    address="address"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__91503a3bfea5d213804a659d92be4592e065cb7c39ad683890bdac4e3b67eb20)
                check_type(argname="argument address", value=address, expected_type=type_hints["address"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if address is not None:
                self._values["address"] = address

        @builtins.property
        def address(self) -> typing.Optional[builtins.str]:
            '''The host address of the reader endpoint.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-readendpoint.html#cfn-rds-dbcluster-readendpoint-address
            '''
            result = self._values.get("address")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ReadEndpointProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBCluster.ScalingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "auto_pause": "autoPause",
            "max_capacity": "maxCapacity",
            "min_capacity": "minCapacity",
            "seconds_before_timeout": "secondsBeforeTimeout",
            "seconds_until_auto_pause": "secondsUntilAutoPause",
            "timeout_action": "timeoutAction",
        },
    )
    class ScalingConfigurationProperty:
        def __init__(
            self,
            *,
            auto_pause: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            max_capacity: typing.Optional[jsii.Number] = None,
            min_capacity: typing.Optional[jsii.Number] = None,
            seconds_before_timeout: typing.Optional[jsii.Number] = None,
            seconds_until_auto_pause: typing.Optional[jsii.Number] = None,
            timeout_action: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless DB cluster.

            For more information, see `Using Amazon Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html>`_ in the *Amazon Aurora User Guide* .

            This property is only supported for Aurora Serverless v1. For Aurora Serverless v2, use ``ServerlessV2ScalingConfiguration`` property.

            Valid for: Aurora DB clusters only

            :param auto_pause: A value that indicates whether to allow or disallow automatic pause for an Aurora DB cluster in ``serverless`` DB engine mode. A DB cluster can be paused only when it's idle (it has no connections). .. epigraph:: If a DB cluster is paused for more than seven days, the DB cluster might be backed up with a snapshot. In this case, the DB cluster is restored when there is a request to connect to it.
            :param max_capacity: The maximum capacity for an Aurora DB cluster in ``serverless`` DB engine mode. For Aurora MySQL, valid capacity values are ``1`` , ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``128`` , and ``256`` . For Aurora PostgreSQL, valid capacity values are ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``192`` , and ``384`` . The maximum capacity must be greater than or equal to the minimum capacity.
            :param min_capacity: The minimum capacity for an Aurora DB cluster in ``serverless`` DB engine mode. For Aurora MySQL, valid capacity values are ``1`` , ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``128`` , and ``256`` . For Aurora PostgreSQL, valid capacity values are ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``192`` , and ``384`` . The minimum capacity must be less than or equal to the maximum capacity.
            :param seconds_before_timeout: The amount of time, in seconds, that Aurora Serverless v1 tries to find a scaling point to perform seamless scaling before enforcing the timeout action. The default is 300. Specify a value between 60 and 600 seconds.
            :param seconds_until_auto_pause: The time, in seconds, before an Aurora DB cluster in ``serverless`` mode is paused. Specify a value between 300 and 86,400 seconds.
            :param timeout_action: The action to take when the timeout is reached, either ``ForceApplyCapacityChange`` or ``RollbackCapacityChange`` . ``ForceApplyCapacityChange`` sets the capacity to the specified value as soon as possible. ``RollbackCapacityChange`` , the default, ignores the capacity change if a scaling point isn't found in the timeout period. .. epigraph:: If you specify ``ForceApplyCapacityChange`` , connections that prevent Aurora Serverless v1 from finding a scaling point might be dropped. For more information, see `Autoscaling for Aurora Serverless v1 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.how-it-works.auto-scaling>`_ in the *Amazon Aurora User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                scaling_configuration_property = rds.CfnDBCluster.ScalingConfigurationProperty(
                    auto_pause=False,
                    max_capacity=123,
                    min_capacity=123,
                    seconds_before_timeout=123,
                    seconds_until_auto_pause=123,
                    timeout_action="timeoutAction"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e0781370579c041e63a9c86b467deedb7afa953c5536af30896c031617dc13e3)
                check_type(argname="argument auto_pause", value=auto_pause, expected_type=type_hints["auto_pause"])
                check_type(argname="argument max_capacity", value=max_capacity, expected_type=type_hints["max_capacity"])
                check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
                check_type(argname="argument seconds_before_timeout", value=seconds_before_timeout, expected_type=type_hints["seconds_before_timeout"])
                check_type(argname="argument seconds_until_auto_pause", value=seconds_until_auto_pause, expected_type=type_hints["seconds_until_auto_pause"])
                check_type(argname="argument timeout_action", value=timeout_action, expected_type=type_hints["timeout_action"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if auto_pause is not None:
                self._values["auto_pause"] = auto_pause
            if max_capacity is not None:
                self._values["max_capacity"] = max_capacity
            if min_capacity is not None:
                self._values["min_capacity"] = min_capacity
            if seconds_before_timeout is not None:
                self._values["seconds_before_timeout"] = seconds_before_timeout
            if seconds_until_auto_pause is not None:
                self._values["seconds_until_auto_pause"] = seconds_until_auto_pause
            if timeout_action is not None:
                self._values["timeout_action"] = timeout_action

        @builtins.property
        def auto_pause(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''A value that indicates whether to allow or disallow automatic pause for an Aurora DB cluster in ``serverless`` DB engine mode.

            A DB cluster can be paused only when it's idle (it has no connections).
            .. epigraph::

               If a DB cluster is paused for more than seven days, the DB cluster might be backed up with a snapshot. In this case, the DB cluster is restored when there is a request to connect to it.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-autopause
            '''
            result = self._values.get("auto_pause")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def max_capacity(self) -> typing.Optional[jsii.Number]:
            '''The maximum capacity for an Aurora DB cluster in ``serverless`` DB engine mode.

            For Aurora MySQL, valid capacity values are ``1`` , ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``128`` , and ``256`` .

            For Aurora PostgreSQL, valid capacity values are ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``192`` , and ``384`` .

            The maximum capacity must be greater than or equal to the minimum capacity.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-maxcapacity
            '''
            result = self._values.get("max_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min_capacity(self) -> typing.Optional[jsii.Number]:
            '''The minimum capacity for an Aurora DB cluster in ``serverless`` DB engine mode.

            For Aurora MySQL, valid capacity values are ``1`` , ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``128`` , and ``256`` .

            For Aurora PostgreSQL, valid capacity values are ``2`` , ``4`` , ``8`` , ``16`` , ``32`` , ``64`` , ``192`` , and ``384`` .

            The minimum capacity must be less than or equal to the maximum capacity.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-mincapacity
            '''
            result = self._values.get("min_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def seconds_before_timeout(self) -> typing.Optional[jsii.Number]:
            '''The amount of time, in seconds, that Aurora Serverless v1 tries to find a scaling point to perform seamless scaling before enforcing the timeout action.

            The default is 300.

            Specify a value between 60 and 600 seconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-secondsbeforetimeout
            '''
            result = self._values.get("seconds_before_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def seconds_until_auto_pause(self) -> typing.Optional[jsii.Number]:
            '''The time, in seconds, before an Aurora DB cluster in ``serverless`` mode is paused.

            Specify a value between 300 and 86,400 seconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-secondsuntilautopause
            '''
            result = self._values.get("seconds_until_auto_pause")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def timeout_action(self) -> typing.Optional[builtins.str]:
            '''The action to take when the timeout is reached, either ``ForceApplyCapacityChange`` or ``RollbackCapacityChange`` .

            ``ForceApplyCapacityChange`` sets the capacity to the specified value as soon as possible.

            ``RollbackCapacityChange`` , the default, ignores the capacity change if a scaling point isn't found in the timeout period.
            .. epigraph::

               If you specify ``ForceApplyCapacityChange`` , connections that prevent Aurora Serverless v1 from finding a scaling point might be dropped.

            For more information, see `Autoscaling for Aurora Serverless v1 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.how-it-works.auto-scaling>`_ in the *Amazon Aurora User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-scalingconfiguration.html#cfn-rds-dbcluster-scalingconfiguration-timeoutaction
            '''
            result = self._values.get("timeout_action")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBCluster.ServerlessV2ScalingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"max_capacity": "maxCapacity", "min_capacity": "minCapacity"},
    )
    class ServerlessV2ScalingConfigurationProperty:
        def __init__(
            self,
            *,
            max_capacity: typing.Optional[jsii.Number] = None,
            min_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``ServerlessV2ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless V2 DB cluster.

            For more information, see `Using Amazon Aurora Serverless v2 <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.html>`_ in the *Amazon Aurora User Guide* .

            If you have an Aurora cluster, you must set the ``ScalingConfigurationInfo`` attribute before you add a DB instance that uses the ``db.serverless`` DB instance class. For more information, see `Clusters that use Aurora Serverless v2 must have a capacity range specified <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.requirements.html#aurora-serverless-v2.requirements.capacity-range>`_ in the *Amazon Aurora User Guide* .

            This property is only supported for Aurora Serverless v2. For Aurora Serverless v1, use ``ScalingConfiguration`` property.

            :param max_capacity: The maximum number of Aurora capacity units (ACUs) for a DB instance in an Aurora Serverless v2 cluster. You can specify ACU values in half-step increments, such as 40, 40.5, 41, and so on. The largest value that you can use is 128. The maximum capacity must be higher than 0.5 ACUs. For more information, see `Choosing the maximum Aurora Serverless v2 capacity setting for a cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.setting-capacity.html#aurora-serverless-v2.max_capacity_considerations>`_ in the *Amazon Aurora User Guide* .
            :param min_capacity: The minimum number of Aurora capacity units (ACUs) for a DB instance in an Aurora Serverless v2 cluster. You can specify ACU values in half-step increments, such as 8, 8.5, 9, and so on. The smallest value that you can use is 0.5.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-serverlessv2scalingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                serverless_v2_scaling_configuration_property = rds.CfnDBCluster.ServerlessV2ScalingConfigurationProperty(
                    max_capacity=123,
                    min_capacity=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fc422a3da996b6d11dd8b19f27f8065e6aa48f550eae9c008b1297c8d18afc24)
                check_type(argname="argument max_capacity", value=max_capacity, expected_type=type_hints["max_capacity"])
                check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max_capacity is not None:
                self._values["max_capacity"] = max_capacity
            if min_capacity is not None:
                self._values["min_capacity"] = min_capacity

        @builtins.property
        def max_capacity(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of Aurora capacity units (ACUs) for a DB instance in an Aurora Serverless v2 cluster.

            You can specify ACU values in half-step increments, such as 40, 40.5, 41, and so on. The largest value that you can use is 128.

            The maximum capacity must be higher than 0.5 ACUs. For more information, see `Choosing the maximum Aurora Serverless v2 capacity setting for a cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless-v2.setting-capacity.html#aurora-serverless-v2.max_capacity_considerations>`_ in the *Amazon Aurora User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-serverlessv2scalingconfiguration.html#cfn-rds-dbcluster-serverlessv2scalingconfiguration-maxcapacity
            '''
            result = self._values.get("max_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min_capacity(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of Aurora capacity units (ACUs) for a DB instance in an Aurora Serverless v2 cluster.

            You can specify ACU values in half-step increments, such as 8, 8.5, 9, and so on. The smallest value that you can use is 0.5.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbcluster-serverlessv2scalingconfiguration.html#cfn-rds-dbcluster-serverlessv2scalingconfiguration-mincapacity
            '''
            result = self._values.get("min_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServerlessV2ScalingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnDBClusterParameterGroup(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBClusterParameterGroup",
):
    '''A CloudFormation ``AWS::RDS::DBClusterParameterGroup``.

    The ``AWS::RDS::DBClusterParameterGroup`` resource creates a new Amazon RDS DB cluster parameter group.

    For information about configuring parameters for Amazon Aurora DB clusters, see `Working with parameter groups <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon Aurora User Guide* .
    .. epigraph::

       If you apply a parameter group to a DB cluster, then its DB instances might need to reboot. This can result in an outage while the DB instances are rebooting.

       If you apply a change to parameter group associated with a stopped DB cluster, then the update stack waits until the DB cluster is started.

    :cloudformationResource: AWS::RDS::DBClusterParameterGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        # parameters: Any
        
        cfn_dBCluster_parameter_group = rds.CfnDBClusterParameterGroup(self, "MyCfnDBClusterParameterGroup",
            description="description",
            family="family",
            parameters=parameters,
        
            # the properties below are optional
            db_cluster_parameter_group_name="dbClusterParameterGroupName",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        description: builtins.str,
        family: builtins.str,
        parameters: typing.Any,
        db_cluster_parameter_group_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBClusterParameterGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param description: A friendly description for this DB cluster parameter group.
        :param family: The DB cluster parameter group family name. A DB cluster parameter group can be associated with one and only one DB cluster parameter group family, and can be applied only to a DB cluster running a DB engine and engine version compatible with that DB cluster parameter group family. .. epigraph:: The DB cluster parameter group family can't be changed when updating a DB cluster parameter group. To list all of the available parameter group families, use the following command: ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"`` The output contains duplicates. For more information, see ``[CreateDBClusterParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html)`` .
        :param parameters: Provides a list of parameters for the DB cluster parameter group.
        :param db_cluster_parameter_group_name: The name of the DB cluster parameter group. Constraints: - Must not match the name of an existing DB cluster parameter group. If you don't specify a value for ``DBClusterParameterGroupName`` property, a name is automatically created for the DB cluster paramter group. .. epigraph:: This value is stored as a lowercase string.
        :param tags: An optional array of key-value pairs to apply to this DB cluster parameter group.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__57be4651a4ecc626576cdebda567fb8f2be1558049cbd22f511fe777779ac93f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBClusterParameterGroupProps(
            description=description,
            family=family,
            parameters=parameters,
            db_cluster_parameter_group_name=db_cluster_parameter_group_name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a645a33c96898b172320e1468ec90c2265e95af171f578c78e0ea8f1d30843d6)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8ce0d917bb4134f234a602024393b778872f2252a7f602bc651dc146e329ebe2)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _aws_cdk_core_f4b25747.TagManager:
        '''An optional array of key-value pairs to apply to this DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-tags
        '''
        return typing.cast(_aws_cdk_core_f4b25747.TagManager, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> builtins.str:
        '''A friendly description for this DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-description
        '''
        return typing.cast(builtins.str, jsii.get(self, "description"))

    @description.setter
    def description(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2b7ad02ac255bae8f020ba74cfc4e5249f8ea3fed8d1bef21b3e213e1f1bff6a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="family")
    def family(self) -> builtins.str:
        '''The DB cluster parameter group family name.

        A DB cluster parameter group can be associated with one and only one DB cluster parameter group family, and can be applied only to a DB cluster running a DB engine and engine version compatible with that DB cluster parameter group family.
        .. epigraph::

           The DB cluster parameter group family can't be changed when updating a DB cluster parameter group.

        To list all of the available parameter group families, use the following command:

        ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"``

        The output contains duplicates.

        For more information, see ``[CreateDBClusterParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html)`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-family
        '''
        return typing.cast(builtins.str, jsii.get(self, "family"))

    @family.setter
    def family(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__504eb5afd3fdb71e19084ef43f5c9acb7a1e4ebcc8e803a4343ea84be7b3754a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "family", value)

    @builtins.property
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Any:
        '''Provides a list of parameters for the DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-parameters
        '''
        return typing.cast(typing.Any, jsii.get(self, "parameters"))

    @parameters.setter
    def parameters(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d4d64874991cc5cd0f85c7fd532e43422315dab29191d64e3dbab9257c9f4d9b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "parameters", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterParameterGroupName")
    def db_cluster_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB cluster parameter group.

        Constraints:

        - Must not match the name of an existing DB cluster parameter group.

        If you don't specify a value for ``DBClusterParameterGroupName`` property, a name is automatically created for the DB cluster paramter group.
        .. epigraph::

           This value is stored as a lowercase string.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-dbclusterparametergroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbClusterParameterGroupName"))

    @db_cluster_parameter_group_name.setter
    def db_cluster_parameter_group_name(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__976641a0b17acc815921db6e1ebf97de876a9fa2de848aaaea8e1b79e5f1b3dd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterParameterGroupName", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBClusterParameterGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "family": "family",
        "parameters": "parameters",
        "db_cluster_parameter_group_name": "dbClusterParameterGroupName",
        "tags": "tags",
    },
)
class CfnDBClusterParameterGroupProps:
    def __init__(
        self,
        *,
        description: builtins.str,
        family: builtins.str,
        parameters: typing.Any,
        db_cluster_parameter_group_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBClusterParameterGroup``.

        :param description: A friendly description for this DB cluster parameter group.
        :param family: The DB cluster parameter group family name. A DB cluster parameter group can be associated with one and only one DB cluster parameter group family, and can be applied only to a DB cluster running a DB engine and engine version compatible with that DB cluster parameter group family. .. epigraph:: The DB cluster parameter group family can't be changed when updating a DB cluster parameter group. To list all of the available parameter group families, use the following command: ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"`` The output contains duplicates. For more information, see ``[CreateDBClusterParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html)`` .
        :param parameters: Provides a list of parameters for the DB cluster parameter group.
        :param db_cluster_parameter_group_name: The name of the DB cluster parameter group. Constraints: - Must not match the name of an existing DB cluster parameter group. If you don't specify a value for ``DBClusterParameterGroupName`` property, a name is automatically created for the DB cluster paramter group. .. epigraph:: This value is stored as a lowercase string.
        :param tags: An optional array of key-value pairs to apply to this DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            # parameters: Any
            
            cfn_dBCluster_parameter_group_props = rds.CfnDBClusterParameterGroupProps(
                description="description",
                family="family",
                parameters=parameters,
            
                # the properties below are optional
                db_cluster_parameter_group_name="dbClusterParameterGroupName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__82e0849af650e24aff64cd6cb987f04e824ebcd4a729ed30ecdcf0e68a41d026)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument family", value=family, expected_type=type_hints["family"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument db_cluster_parameter_group_name", value=db_cluster_parameter_group_name, expected_type=type_hints["db_cluster_parameter_group_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "description": description,
            "family": family,
            "parameters": parameters,
        }
        if db_cluster_parameter_group_name is not None:
            self._values["db_cluster_parameter_group_name"] = db_cluster_parameter_group_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def description(self) -> builtins.str:
        '''A friendly description for this DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-description
        '''
        result = self._values.get("description")
        assert result is not None, "Required property 'description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def family(self) -> builtins.str:
        '''The DB cluster parameter group family name.

        A DB cluster parameter group can be associated with one and only one DB cluster parameter group family, and can be applied only to a DB cluster running a DB engine and engine version compatible with that DB cluster parameter group family.
        .. epigraph::

           The DB cluster parameter group family can't be changed when updating a DB cluster parameter group.

        To list all of the available parameter group families, use the following command:

        ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"``

        The output contains duplicates.

        For more information, see ``[CreateDBClusterParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBClusterParameterGroup.html)`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-family
        '''
        result = self._values.get("family")
        assert result is not None, "Required property 'family' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def parameters(self) -> typing.Any:
        '''Provides a list of parameters for the DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-parameters
        '''
        result = self._values.get("parameters")
        assert result is not None, "Required property 'parameters' is missing"
        return typing.cast(typing.Any, result)

    @builtins.property
    def db_cluster_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB cluster parameter group.

        Constraints:

        - Must not match the name of an existing DB cluster parameter group.

        If you don't specify a value for ``DBClusterParameterGroupName`` property, a name is automatically created for the DB cluster paramter group.
        .. epigraph::

           This value is stored as a lowercase string.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-dbclusterparametergroupname
        '''
        result = self._values.get("db_cluster_parameter_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]]:
        '''An optional array of key-value pairs to apply to this DB cluster parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbclusterparametergroup.html#cfn-rds-dbclusterparametergroup-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBClusterParameterGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "allocated_storage": "allocatedStorage",
        "associated_roles": "associatedRoles",
        "auto_minor_version_upgrade": "autoMinorVersionUpgrade",
        "availability_zones": "availabilityZones",
        "backtrack_window": "backtrackWindow",
        "backup_retention_period": "backupRetentionPeriod",
        "copy_tags_to_snapshot": "copyTagsToSnapshot",
        "database_name": "databaseName",
        "db_cluster_identifier": "dbClusterIdentifier",
        "db_cluster_instance_class": "dbClusterInstanceClass",
        "db_cluster_parameter_group_name": "dbClusterParameterGroupName",
        "db_instance_parameter_group_name": "dbInstanceParameterGroupName",
        "db_subnet_group_name": "dbSubnetGroupName",
        "db_system_id": "dbSystemId",
        "deletion_protection": "deletionProtection",
        "domain": "domain",
        "domain_iam_role_name": "domainIamRoleName",
        "enable_cloudwatch_logs_exports": "enableCloudwatchLogsExports",
        "enable_http_endpoint": "enableHttpEndpoint",
        "enable_iam_database_authentication": "enableIamDatabaseAuthentication",
        "engine": "engine",
        "engine_mode": "engineMode",
        "engine_version": "engineVersion",
        "global_cluster_identifier": "globalClusterIdentifier",
        "iops": "iops",
        "kms_key_id": "kmsKeyId",
        "manage_master_user_password": "manageMasterUserPassword",
        "master_username": "masterUsername",
        "master_user_password": "masterUserPassword",
        "master_user_secret": "masterUserSecret",
        "monitoring_interval": "monitoringInterval",
        "monitoring_role_arn": "monitoringRoleArn",
        "network_type": "networkType",
        "performance_insights_enabled": "performanceInsightsEnabled",
        "performance_insights_kms_key_id": "performanceInsightsKmsKeyId",
        "performance_insights_retention_period": "performanceInsightsRetentionPeriod",
        "port": "port",
        "preferred_backup_window": "preferredBackupWindow",
        "preferred_maintenance_window": "preferredMaintenanceWindow",
        "publicly_accessible": "publiclyAccessible",
        "replication_source_identifier": "replicationSourceIdentifier",
        "restore_type": "restoreType",
        "scaling_configuration": "scalingConfiguration",
        "serverless_v2_scaling_configuration": "serverlessV2ScalingConfiguration",
        "snapshot_identifier": "snapshotIdentifier",
        "source_db_cluster_identifier": "sourceDbClusterIdentifier",
        "source_region": "sourceRegion",
        "storage_encrypted": "storageEncrypted",
        "storage_type": "storageType",
        "tags": "tags",
        "use_latest_restorable_time": "useLatestRestorableTime",
        "vpc_security_group_ids": "vpcSecurityGroupIds",
    },
)
class CfnDBClusterProps:
    def __init__(
        self,
        *,
        allocated_storage: typing.Optional[jsii.Number] = None,
        associated_roles: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[typing.Union[CfnDBCluster.DBClusterRoleProperty, typing.Dict[builtins.str, typing.Any]], _aws_cdk_core_f4b25747.IResolvable]]]] = None,
        auto_minor_version_upgrade: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        availability_zones: typing.Optional[typing.Sequence[builtins.str]] = None,
        backtrack_window: typing.Optional[jsii.Number] = None,
        backup_retention_period: typing.Optional[jsii.Number] = None,
        copy_tags_to_snapshot: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        database_name: typing.Optional[builtins.str] = None,
        db_cluster_identifier: typing.Optional[builtins.str] = None,
        db_cluster_instance_class: typing.Optional[builtins.str] = None,
        db_cluster_parameter_group_name: typing.Optional[builtins.str] = None,
        db_instance_parameter_group_name: typing.Optional[builtins.str] = None,
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        db_system_id: typing.Optional[builtins.str] = None,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_iam_role_name: typing.Optional[builtins.str] = None,
        enable_cloudwatch_logs_exports: typing.Optional[typing.Sequence[builtins.str]] = None,
        enable_http_endpoint: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        enable_iam_database_authentication: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_mode: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        global_cluster_identifier: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        manage_master_user_password: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        master_username: typing.Optional[builtins.str] = None,
        master_user_password: typing.Optional[builtins.str] = None,
        master_user_secret: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnDBCluster.MasterUserSecretProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        monitoring_interval: typing.Optional[jsii.Number] = None,
        monitoring_role_arn: typing.Optional[builtins.str] = None,
        network_type: typing.Optional[builtins.str] = None,
        performance_insights_enabled: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        performance_insights_kms_key_id: typing.Optional[builtins.str] = None,
        performance_insights_retention_period: typing.Optional[jsii.Number] = None,
        port: typing.Optional[jsii.Number] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        publicly_accessible: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        replication_source_identifier: typing.Optional[builtins.str] = None,
        restore_type: typing.Optional[builtins.str] = None,
        scaling_configuration: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnDBCluster.ScalingConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        serverless_v2_scaling_configuration: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnDBCluster.ServerlessV2ScalingConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        snapshot_identifier: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        source_region: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        storage_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
        use_latest_restorable_time: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        vpc_security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBCluster``.

        :param allocated_storage: The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster. This setting is required to create a Multi-AZ DB cluster. Valid for: Multi-AZ DB clusters only
        :param associated_roles: Provides a list of the AWS Identity and Access Management (IAM) roles that are associated with the DB cluster. IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other Amazon Web Services on your behalf. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param auto_minor_version_upgrade: A value that indicates whether minor engine upgrades are applied automatically to the DB cluster during the maintenance window. By default, minor engine upgrades are applied automatically. Valid for: Multi-AZ DB clusters only
        :param availability_zones: A list of Availability Zones (AZs) where instances in the DB cluster can be created. For information on AWS Regions and Availability Zones, see `Choosing the Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param backtrack_window: The target backtrack window, in seconds. To disable backtracking, set this value to 0. .. epigraph:: Currently, Backtrack is only supported for Aurora MySQL DB clusters. Default: 0 Constraints: - If specified, this value must be set to a number from 0 to 259,200 (72 hours). Valid for: Aurora MySQL DB clusters only
        :param backup_retention_period: The number of days for which automated backups are retained. Default: 1 Constraints: - Must be a value from 1 to 35 Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param copy_tags_to_snapshot: A value that indicates whether to copy all tags from the DB cluster to snapshots of the DB cluster. The default is not to copy them. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param database_name: The name of your database. If you don't provide a name, then Amazon RDS won't create a database in this DB cluster. For naming constraints, see `Naming Constraints <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_cluster_identifier: The DB cluster identifier. This parameter is stored as a lowercase string. Constraints: - Must contain from 1 to 63 letters, numbers, or hyphens. - First character must be a letter. - Can't end with a hyphen or contain two consecutive hyphens. Example: ``my-cluster1`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_cluster_instance_class: The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example db.m6gd.xlarge. Not all DB instance classes are available in all AWS Regions , or for all database engines. For the full list of DB instance classes and availability for your engine, see `DB instance class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide* . This setting is required to create a Multi-AZ DB cluster. Valid for: Multi-AZ DB clusters only
        :param db_cluster_parameter_group_name: The name of the DB cluster parameter group to associate with this DB cluster. .. epigraph:: If you apply a parameter group to an existing DB cluster, then its DB instances might need to reboot. This can result in an outage while the DB instances are rebooting. If you apply a change to parameter group associated with a stopped DB cluster, then the update stack waits until the DB cluster is started. To list all of the available DB cluster parameter group names, use the following command: ``aws rds describe-db-cluster-parameter-groups --query "DBClusterParameterGroups[].DBClusterParameterGroupName" --output text`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_instance_parameter_group_name: The name of the DB parameter group to apply to all instances of the DB cluster. .. epigraph:: When you apply a parameter group using the ``DBInstanceParameterGroupName`` parameter, the DB cluster isn't rebooted automatically. Also, parameter changes are applied immediately rather than during the next maintenance window. Default: The existing name setting Constraints: - The DB parameter group must be in the same DB parameter group family as this DB cluster.
        :param db_subnet_group_name: A DB subnet group that you want to associate with this DB cluster. If you are restoring a DB cluster to a point in time with ``RestoreType`` set to ``copy-on-write`` , and don't specify a DB subnet group name, then the DB cluster is restored with a default DB subnet group. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param db_system_id: Reserved for future use.
        :param deletion_protection: A value that indicates whether the DB cluster has deletion protection enabled. The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param domain: Indicates the directory ID of the Active Directory to create the DB cluster. For Amazon Aurora DB clusters, Amazon RDS can use Kerberos authentication to authenticate users that connect to the DB cluster. For more information, see `Kerberos authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/kerberos-authentication.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param domain_iam_role_name: Specifies the name of the IAM role to use when making API calls to the Directory Service. Valid for: Aurora DB clusters only
        :param enable_cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Aurora User Guide* . *Aurora MySQL* Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery`` *Aurora PostgreSQL* Valid values: ``postgresql`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param enable_http_endpoint: A value that indicates whether to enable the HTTP endpoint for an Aurora Serverless DB cluster. By default, the HTTP endpoint is disabled. When enabled, the HTTP endpoint provides a connectionless web service API for running SQL queries on the Aurora Serverless DB cluster. You can also query your database from inside the RDS console with the query editor. For more information, see `Using the Data API for Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param enable_iam_database_authentication: A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts. By default, mapping is disabled. For more information, see `IAM Database Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon Aurora User Guide.* Valid for: Aurora DB clusters only
        :param engine: The name of the database engine to be used for this DB cluster. Valid Values: - ``aurora`` (for MySQL 5.6-compatible Aurora) - ``aurora-mysql`` (for MySQL 5.7-compatible Aurora) - ``aurora-postgresql`` - ``mysql`` - ``postgres`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param engine_mode: The DB engine mode of the DB cluster, either ``provisioned`` , ``serverless`` , ``parallelquery`` , ``global`` , or ``multimaster`` . The ``parallelquery`` engine mode isn't required for Aurora MySQL version 1.23 and higher 1.x versions, and version 2.09 and higher 2.x versions. The ``global`` engine mode isn't required for Aurora MySQL version 1.22 and higher 1.x versions, and ``global`` engine mode isn't required for any 2.x versions. The ``multimaster`` engine mode only applies for DB clusters created with Aurora MySQL version 5.6.10a. For Aurora PostgreSQL, the ``global`` engine mode isn't required, and both the ``parallelquery`` and the ``multimaster`` engine modes currently aren't supported. Limitations and requirements apply to some DB engine modes. For more information, see the following sections in the *Amazon Aurora User Guide* : - `Limitations of Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations>`_ - `Limitations of Parallel Query <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-mysql-parallel-query.html#aurora-mysql-parallel-query-limitations>`_ - `Limitations of Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html#aurora-global-database.limitations>`_ - `Limitations of Multi-Master Clusters <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-multi-master.html#aurora-multi-master-limitations>`_ Valid for: Aurora DB clusters only
        :param engine_version: The version number of the database engine to use. To list all of the available engine versions for ``aurora`` (for MySQL 5.6-compatible Aurora), use the following command: ``aws rds describe-db-engine-versions --engine aurora --query "DBEngineVersions[].EngineVersion"`` To list all of the available engine versions for ``aurora-mysql`` (for MySQL 5.7-compatible Aurora), use the following command: ``aws rds describe-db-engine-versions --engine aurora-mysql --query "DBEngineVersions[].EngineVersion"`` To list all of the available engine versions for ``aurora-postgresql`` , use the following command: ``aws rds describe-db-engine-versions --engine aurora-postgresql --query "DBEngineVersions[].EngineVersion"`` Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param global_cluster_identifier: If you are configuring an Aurora global database cluster and want your Aurora DB cluster to be a secondary member in the global database cluster, specify the global cluster ID of the global database cluster. To define the primary database cluster of the global cluster, use the `AWS::RDS::GlobalCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html>`_ resource. If you aren't configuring a global database cluster, don't specify this property. .. epigraph:: To remove the DB cluster from a global database cluster, specify an empty value for the ``GlobalClusterIdentifier`` property. For information about Aurora global databases, see `Working with Amazon Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html>`_ in the *Amazon Aurora User Guide* . Valid for: Aurora DB clusters only
        :param iops: The amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster. For information about valid IOPS values, see `Amazon RDS Provisioned IOPS storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* . This setting is required to create a Multi-AZ DB cluster. Constraints: Must be a multiple between .5 and 50 of the storage amount for the DB cluster. Valid for: Multi-AZ DB clusters only
        :param kms_key_id: The Amazon Resource Name (ARN) of the AWS KMS key that is used to encrypt the database instances in the DB cluster, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` . If you enable the ``StorageEncrypted`` property but don't specify this property, the default KMS key is used. If you specify this property, you must set the ``StorageEncrypted`` property to ``true`` . If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param manage_master_user_password: A value that indicates whether to manage the master user password with AWS Secrets Manager. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.* Constraints: - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param master_username: The name of the master user for the DB cluster. .. epigraph:: If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param master_user_password: The master password for the DB instance. .. epigraph:: If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param master_user_secret: Contains the secret managed by RDS in AWS Secrets Manager for the master user password. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*
        :param monitoring_interval: The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB cluster. To turn off collecting Enhanced Monitoring metrics, specify 0. The default is 0. If ``MonitoringRoleArn`` is specified, also set ``MonitoringInterval`` to a value other than 0. Valid Values: ``0, 1, 5, 10, 15, 30, 60`` Valid for: Multi-AZ DB clusters only
        :param monitoring_role_arn: The Amazon Resource Name (ARN) for the IAM role that permits RDS to send Enhanced Monitoring metrics to Amazon CloudWatch Logs. An example is ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting up and enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* . If ``MonitoringInterval`` is set to a value other than 0, supply a ``MonitoringRoleArn`` value. Valid for: Multi-AZ DB clusters only
        :param network_type: The network type of the DB cluster. Valid values: - ``IPV4`` - ``DUAL`` The network type is determined by the ``DBSubnetGroup`` specified for the DB cluster. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ). For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon Aurora User Guide.* Valid for: Aurora DB clusters only
        :param performance_insights_enabled: A value that indicates whether to turn on Performance Insights for the DB cluster. For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* . Valid for: Multi-AZ DB clusters only
        :param performance_insights_kms_key_id: The AWS KMS key identifier for encryption of Performance Insights data. The AWS KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. If you don't specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account . Your AWS account has a different default KMS key for each AWS Region . Valid for: Multi-AZ DB clusters only
        :param performance_insights_retention_period: The number of days to retain Performance Insights data. The default is 7 days. The following values are valid:. - 7 - *month* * 31, where *month* is a number of months from 1-23 - 731 For example, the following values are valid: - 93 (3 months * 31) - 341 (11 months * 31) - 589 (19 months * 31) - 731 If you specify a retention period such as 94, which isn't a valid value, RDS issues an error. Valid for: Multi-AZ DB clusters only
        :param port: The port number on which the DB instances in the DB cluster accept connections. Default: - When ``EngineMode`` is ``provisioned`` , ``3306`` (for both Aurora MySQL and Aurora PostgreSQL) - When ``EngineMode`` is ``serverless`` : - ``3306`` when ``Engine`` is ``aurora`` or ``aurora-mysql`` - ``5432`` when ``Engine`` is ``aurora-postgresql`` .. epigraph:: The ``No interruption`` on update behavior only applies to DB clusters. If you are updating a DB instance, see `Port <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-port>`_ for the AWS::RDS::DBInstance resource. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param preferred_backup_window: The daily time range during which automated backups are created. For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html#Aurora.Managing.Backups.BackupWindow>`_ in the *Amazon Aurora User Guide.* Constraints: - Must be in the format ``hh24:mi-hh24:mi`` . - Must be in Universal Coordinated Time (UTC). - Must not conflict with the preferred maintenance window. - Must be at least 30 minutes. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param preferred_maintenance_window: The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC). Format: ``ddd:hh24:mi-ddd:hh24:mi`` The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Cluster Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow.Aurora>`_ in the *Amazon Aurora User Guide.* Valid Days: Mon, Tue, Wed, Thu, Fri, Sat, Sun. Constraints: Minimum 30-minute window. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param publicly_accessible: A value that indicates whether the DB cluster is publicly accessible. When the DB cluster is publicly accessible, its Domain Name System (DNS) endpoint resolves to the private IP address from within the DB cluster's virtual private cloud (VPC). It resolves to the public IP address from outside of the DB cluster's VPC. Access to the DB cluster is ultimately controlled by the security group it uses. That public access isn't permitted if the security group assigned to the DB cluster doesn't permit it. When the DB cluster isn't publicly accessible, it is an internal DB cluster with a DNS name that resolves to a private IP address. Default: The default behavior varies depending on whether ``DBSubnetGroupName`` is specified. If ``DBSubnetGroupName`` isn't specified, and ``PubliclyAccessible`` isn't specified, the following applies: - If the default VPC in the target Region doesn’t have an internet gateway attached to it, the DB cluster is private. - If the default VPC in the target Region has an internet gateway attached to it, the DB cluster is public. If ``DBSubnetGroupName`` is specified, and ``PubliclyAccessible`` isn't specified, the following applies: - If the subnets are part of a VPC that doesn’t have an internet gateway attached to it, the DB cluster is private. - If the subnets are part of a VPC that has an internet gateway attached to it, the DB cluster is public. Valid for: Multi-AZ DB clusters only
        :param replication_source_identifier: The Amazon Resource Name (ARN) of the source DB instance or DB cluster if this DB cluster is created as a read replica. Valid for: Aurora DB clusters only
        :param restore_type: The type of restore to be performed. You can specify one of the following values:. - ``full-copy`` - The new DB cluster is restored as a full copy of the source DB cluster. - ``copy-on-write`` - The new DB cluster is restored as a clone of the source DB cluster. Constraints: You can't specify ``copy-on-write`` if the engine version of the source DB cluster is earlier than 1.11. If you don't specify a ``RestoreType`` value, then the new DB cluster is restored as a full copy of the source DB cluster. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param scaling_configuration: The ``ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless DB cluster. This property is only supported for Aurora Serverless v1. For Aurora Serverless v2, use ``ServerlessV2ScalingConfiguration`` property. Valid for: Aurora DB clusters only
        :param serverless_v2_scaling_configuration: The ``ServerlessV2ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless V2 DB cluster. This property is only supported for Aurora Serverless v2. For Aurora Serverless v1, use ``ScalingConfiguration`` property. Valid for: Aurora DB clusters only
        :param snapshot_identifier: The identifier for the DB snapshot or DB cluster snapshot to restore from. You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot. However, you can use only the ARN to specify a DB snapshot. After you restore a DB cluster with a ``SnapshotIdentifier`` property, you must specify the same ``SnapshotIdentifier`` property for any future updates to the DB cluster. When you specify this property for an update, the DB cluster is not restored from the snapshot again, and the data in the database is not changed. However, if you don't specify the ``SnapshotIdentifier`` property, an empty DB cluster is created, and the original DB cluster is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB cluster is restored from the specified ``SnapshotIdentifier`` property, and the original DB cluster is deleted. If you specify the ``SnapshotIdentifier`` property to restore a DB cluster (as opposed to specifying it for DB cluster updates), then don't specify the following properties: - ``GlobalClusterIdentifier`` - ``MasterUsername`` - ``MasterUserPassword`` - ``ReplicationSourceIdentifier`` - ``RestoreType`` - ``SourceDBClusterIdentifier`` - ``SourceRegion`` - ``StorageEncrypted`` (for an encrypted snapshot) - ``UseLatestRestorableTime`` Constraints: - Must match the identifier of an existing Snapshot. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param source_db_cluster_identifier: When restoring a DB cluster to a point in time, the identifier of the source DB cluster from which to restore. Constraints: - Must match the identifier of an existing DBCluster. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param source_region: The AWS Region which contains the source DB cluster when replicating a DB cluster. For example, ``us-east-1`` . Valid for: Aurora DB clusters only
        :param storage_encrypted: Indicates whether the DB cluster is encrypted. If you specify the ``KmsKeyId`` property, then you must enable encryption. If you specify the ``SourceDBClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB cluster is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB cluster to be encrypted, then don't set this property or set it to ``false`` . Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param storage_type: Specifies the storage type to be associated with the DB cluster. This setting is required to create a Multi-AZ DB cluster. Valid values: ``io1`` When specified, a value for the ``Iops`` parameter is required. Default: ``io1`` Valid for: Multi-AZ DB clusters only
        :param tags: An optional array of key-value pairs to apply to this DB cluster. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param use_latest_restorable_time: A value that indicates whether to restore the DB cluster to the latest restorable backup time. By default, the DB cluster is not restored to the latest restorable backup time. Valid for: Aurora DB clusters and Multi-AZ DB clusters
        :param vpc_security_group_ids: A list of EC2 VPC security groups to associate with this DB cluster. If you plan to update the resource, don't specify VPC security groups in a shared VPC. Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            cfn_dBCluster_props = rds.CfnDBClusterProps(
                allocated_storage=123,
                associated_roles=[rds.CfnDBCluster.DBClusterRoleProperty(
                    role_arn="roleArn",
            
                    # the properties below are optional
                    feature_name="featureName"
                )],
                auto_minor_version_upgrade=False,
                availability_zones=["availabilityZones"],
                backtrack_window=123,
                backup_retention_period=123,
                copy_tags_to_snapshot=False,
                database_name="databaseName",
                db_cluster_identifier="dbClusterIdentifier",
                db_cluster_instance_class="dbClusterInstanceClass",
                db_cluster_parameter_group_name="dbClusterParameterGroupName",
                db_instance_parameter_group_name="dbInstanceParameterGroupName",
                db_subnet_group_name="dbSubnetGroupName",
                db_system_id="dbSystemId",
                deletion_protection=False,
                domain="domain",
                domain_iam_role_name="domainIamRoleName",
                enable_cloudwatch_logs_exports=["enableCloudwatchLogsExports"],
                enable_http_endpoint=False,
                enable_iam_database_authentication=False,
                engine="engine",
                engine_mode="engineMode",
                engine_version="engineVersion",
                global_cluster_identifier="globalClusterIdentifier",
                iops=123,
                kms_key_id="kmsKeyId",
                manage_master_user_password=False,
                master_username="masterUsername",
                master_user_password="masterUserPassword",
                master_user_secret=rds.CfnDBCluster.MasterUserSecretProperty(
                    kms_key_id="kmsKeyId",
                    secret_arn="secretArn"
                ),
                monitoring_interval=123,
                monitoring_role_arn="monitoringRoleArn",
                network_type="networkType",
                performance_insights_enabled=False,
                performance_insights_kms_key_id="performanceInsightsKmsKeyId",
                performance_insights_retention_period=123,
                port=123,
                preferred_backup_window="preferredBackupWindow",
                preferred_maintenance_window="preferredMaintenanceWindow",
                publicly_accessible=False,
                replication_source_identifier="replicationSourceIdentifier",
                restore_type="restoreType",
                scaling_configuration=rds.CfnDBCluster.ScalingConfigurationProperty(
                    auto_pause=False,
                    max_capacity=123,
                    min_capacity=123,
                    seconds_before_timeout=123,
                    seconds_until_auto_pause=123,
                    timeout_action="timeoutAction"
                ),
                serverless_v2_scaling_configuration=rds.CfnDBCluster.ServerlessV2ScalingConfigurationProperty(
                    max_capacity=123,
                    min_capacity=123
                ),
                snapshot_identifier="snapshotIdentifier",
                source_db_cluster_identifier="sourceDbClusterIdentifier",
                source_region="sourceRegion",
                storage_encrypted=False,
                storage_type="storageType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                use_latest_restorable_time=False,
                vpc_security_group_ids=["vpcSecurityGroupIds"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__27867aab287edd7364f7efdf2cd7f99fd2fa8047c5855beb7bcece83a6ff1051)
            check_type(argname="argument allocated_storage", value=allocated_storage, expected_type=type_hints["allocated_storage"])
            check_type(argname="argument associated_roles", value=associated_roles, expected_type=type_hints["associated_roles"])
            check_type(argname="argument auto_minor_version_upgrade", value=auto_minor_version_upgrade, expected_type=type_hints["auto_minor_version_upgrade"])
            check_type(argname="argument availability_zones", value=availability_zones, expected_type=type_hints["availability_zones"])
            check_type(argname="argument backtrack_window", value=backtrack_window, expected_type=type_hints["backtrack_window"])
            check_type(argname="argument backup_retention_period", value=backup_retention_period, expected_type=type_hints["backup_retention_period"])
            check_type(argname="argument copy_tags_to_snapshot", value=copy_tags_to_snapshot, expected_type=type_hints["copy_tags_to_snapshot"])
            check_type(argname="argument database_name", value=database_name, expected_type=type_hints["database_name"])
            check_type(argname="argument db_cluster_identifier", value=db_cluster_identifier, expected_type=type_hints["db_cluster_identifier"])
            check_type(argname="argument db_cluster_instance_class", value=db_cluster_instance_class, expected_type=type_hints["db_cluster_instance_class"])
            check_type(argname="argument db_cluster_parameter_group_name", value=db_cluster_parameter_group_name, expected_type=type_hints["db_cluster_parameter_group_name"])
            check_type(argname="argument db_instance_parameter_group_name", value=db_instance_parameter_group_name, expected_type=type_hints["db_instance_parameter_group_name"])
            check_type(argname="argument db_subnet_group_name", value=db_subnet_group_name, expected_type=type_hints["db_subnet_group_name"])
            check_type(argname="argument db_system_id", value=db_system_id, expected_type=type_hints["db_system_id"])
            check_type(argname="argument deletion_protection", value=deletion_protection, expected_type=type_hints["deletion_protection"])
            check_type(argname="argument domain", value=domain, expected_type=type_hints["domain"])
            check_type(argname="argument domain_iam_role_name", value=domain_iam_role_name, expected_type=type_hints["domain_iam_role_name"])
            check_type(argname="argument enable_cloudwatch_logs_exports", value=enable_cloudwatch_logs_exports, expected_type=type_hints["enable_cloudwatch_logs_exports"])
            check_type(argname="argument enable_http_endpoint", value=enable_http_endpoint, expected_type=type_hints["enable_http_endpoint"])
            check_type(argname="argument enable_iam_database_authentication", value=enable_iam_database_authentication, expected_type=type_hints["enable_iam_database_authentication"])
            check_type(argname="argument engine", value=engine, expected_type=type_hints["engine"])
            check_type(argname="argument engine_mode", value=engine_mode, expected_type=type_hints["engine_mode"])
            check_type(argname="argument engine_version", value=engine_version, expected_type=type_hints["engine_version"])
            check_type(argname="argument global_cluster_identifier", value=global_cluster_identifier, expected_type=type_hints["global_cluster_identifier"])
            check_type(argname="argument iops", value=iops, expected_type=type_hints["iops"])
            check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
            check_type(argname="argument manage_master_user_password", value=manage_master_user_password, expected_type=type_hints["manage_master_user_password"])
            check_type(argname="argument master_username", value=master_username, expected_type=type_hints["master_username"])
            check_type(argname="argument master_user_password", value=master_user_password, expected_type=type_hints["master_user_password"])
            check_type(argname="argument master_user_secret", value=master_user_secret, expected_type=type_hints["master_user_secret"])
            check_type(argname="argument monitoring_interval", value=monitoring_interval, expected_type=type_hints["monitoring_interval"])
            check_type(argname="argument monitoring_role_arn", value=monitoring_role_arn, expected_type=type_hints["monitoring_role_arn"])
            check_type(argname="argument network_type", value=network_type, expected_type=type_hints["network_type"])
            check_type(argname="argument performance_insights_enabled", value=performance_insights_enabled, expected_type=type_hints["performance_insights_enabled"])
            check_type(argname="argument performance_insights_kms_key_id", value=performance_insights_kms_key_id, expected_type=type_hints["performance_insights_kms_key_id"])
            check_type(argname="argument performance_insights_retention_period", value=performance_insights_retention_period, expected_type=type_hints["performance_insights_retention_period"])
            check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            check_type(argname="argument preferred_backup_window", value=preferred_backup_window, expected_type=type_hints["preferred_backup_window"])
            check_type(argname="argument preferred_maintenance_window", value=preferred_maintenance_window, expected_type=type_hints["preferred_maintenance_window"])
            check_type(argname="argument publicly_accessible", value=publicly_accessible, expected_type=type_hints["publicly_accessible"])
            check_type(argname="argument replication_source_identifier", value=replication_source_identifier, expected_type=type_hints["replication_source_identifier"])
            check_type(argname="argument restore_type", value=restore_type, expected_type=type_hints["restore_type"])
            check_type(argname="argument scaling_configuration", value=scaling_configuration, expected_type=type_hints["scaling_configuration"])
            check_type(argname="argument serverless_v2_scaling_configuration", value=serverless_v2_scaling_configuration, expected_type=type_hints["serverless_v2_scaling_configuration"])
            check_type(argname="argument snapshot_identifier", value=snapshot_identifier, expected_type=type_hints["snapshot_identifier"])
            check_type(argname="argument source_db_cluster_identifier", value=source_db_cluster_identifier, expected_type=type_hints["source_db_cluster_identifier"])
            check_type(argname="argument source_region", value=source_region, expected_type=type_hints["source_region"])
            check_type(argname="argument storage_encrypted", value=storage_encrypted, expected_type=type_hints["storage_encrypted"])
            check_type(argname="argument storage_type", value=storage_type, expected_type=type_hints["storage_type"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument use_latest_restorable_time", value=use_latest_restorable_time, expected_type=type_hints["use_latest_restorable_time"])
            check_type(argname="argument vpc_security_group_ids", value=vpc_security_group_ids, expected_type=type_hints["vpc_security_group_ids"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allocated_storage is not None:
            self._values["allocated_storage"] = allocated_storage
        if associated_roles is not None:
            self._values["associated_roles"] = associated_roles
        if auto_minor_version_upgrade is not None:
            self._values["auto_minor_version_upgrade"] = auto_minor_version_upgrade
        if availability_zones is not None:
            self._values["availability_zones"] = availability_zones
        if backtrack_window is not None:
            self._values["backtrack_window"] = backtrack_window
        if backup_retention_period is not None:
            self._values["backup_retention_period"] = backup_retention_period
        if copy_tags_to_snapshot is not None:
            self._values["copy_tags_to_snapshot"] = copy_tags_to_snapshot
        if database_name is not None:
            self._values["database_name"] = database_name
        if db_cluster_identifier is not None:
            self._values["db_cluster_identifier"] = db_cluster_identifier
        if db_cluster_instance_class is not None:
            self._values["db_cluster_instance_class"] = db_cluster_instance_class
        if db_cluster_parameter_group_name is not None:
            self._values["db_cluster_parameter_group_name"] = db_cluster_parameter_group_name
        if db_instance_parameter_group_name is not None:
            self._values["db_instance_parameter_group_name"] = db_instance_parameter_group_name
        if db_subnet_group_name is not None:
            self._values["db_subnet_group_name"] = db_subnet_group_name
        if db_system_id is not None:
            self._values["db_system_id"] = db_system_id
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if domain is not None:
            self._values["domain"] = domain
        if domain_iam_role_name is not None:
            self._values["domain_iam_role_name"] = domain_iam_role_name
        if enable_cloudwatch_logs_exports is not None:
            self._values["enable_cloudwatch_logs_exports"] = enable_cloudwatch_logs_exports
        if enable_http_endpoint is not None:
            self._values["enable_http_endpoint"] = enable_http_endpoint
        if enable_iam_database_authentication is not None:
            self._values["enable_iam_database_authentication"] = enable_iam_database_authentication
        if engine is not None:
            self._values["engine"] = engine
        if engine_mode is not None:
            self._values["engine_mode"] = engine_mode
        if engine_version is not None:
            self._values["engine_version"] = engine_version
        if global_cluster_identifier is not None:
            self._values["global_cluster_identifier"] = global_cluster_identifier
        if iops is not None:
            self._values["iops"] = iops
        if kms_key_id is not None:
            self._values["kms_key_id"] = kms_key_id
        if manage_master_user_password is not None:
            self._values["manage_master_user_password"] = manage_master_user_password
        if master_username is not None:
            self._values["master_username"] = master_username
        if master_user_password is not None:
            self._values["master_user_password"] = master_user_password
        if master_user_secret is not None:
            self._values["master_user_secret"] = master_user_secret
        if monitoring_interval is not None:
            self._values["monitoring_interval"] = monitoring_interval
        if monitoring_role_arn is not None:
            self._values["monitoring_role_arn"] = monitoring_role_arn
        if network_type is not None:
            self._values["network_type"] = network_type
        if performance_insights_enabled is not None:
            self._values["performance_insights_enabled"] = performance_insights_enabled
        if performance_insights_kms_key_id is not None:
            self._values["performance_insights_kms_key_id"] = performance_insights_kms_key_id
        if performance_insights_retention_period is not None:
            self._values["performance_insights_retention_period"] = performance_insights_retention_period
        if port is not None:
            self._values["port"] = port
        if preferred_backup_window is not None:
            self._values["preferred_backup_window"] = preferred_backup_window
        if preferred_maintenance_window is not None:
            self._values["preferred_maintenance_window"] = preferred_maintenance_window
        if publicly_accessible is not None:
            self._values["publicly_accessible"] = publicly_accessible
        if replication_source_identifier is not None:
            self._values["replication_source_identifier"] = replication_source_identifier
        if restore_type is not None:
            self._values["restore_type"] = restore_type
        if scaling_configuration is not None:
            self._values["scaling_configuration"] = scaling_configuration
        if serverless_v2_scaling_configuration is not None:
            self._values["serverless_v2_scaling_configuration"] = serverless_v2_scaling_configuration
        if snapshot_identifier is not None:
            self._values["snapshot_identifier"] = snapshot_identifier
        if source_db_cluster_identifier is not None:
            self._values["source_db_cluster_identifier"] = source_db_cluster_identifier
        if source_region is not None:
            self._values["source_region"] = source_region
        if storage_encrypted is not None:
            self._values["storage_encrypted"] = storage_encrypted
        if storage_type is not None:
            self._values["storage_type"] = storage_type
        if tags is not None:
            self._values["tags"] = tags
        if use_latest_restorable_time is not None:
            self._values["use_latest_restorable_time"] = use_latest_restorable_time
        if vpc_security_group_ids is not None:
            self._values["vpc_security_group_ids"] = vpc_security_group_ids

    @builtins.property
    def allocated_storage(self) -> typing.Optional[jsii.Number]:
        '''The amount of storage in gibibytes (GiB) to allocate to each DB instance in the Multi-AZ DB cluster.

        This setting is required to create a Multi-AZ DB cluster.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-allocatedstorage
        '''
        result = self._values.get("allocated_storage")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def associated_roles(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[CfnDBCluster.DBClusterRoleProperty, _aws_cdk_core_f4b25747.IResolvable]]]]:
        '''Provides a list of the AWS Identity and Access Management (IAM) roles that are associated with the DB cluster.

        IAM roles that are associated with a DB cluster grant permission for the DB cluster to access other Amazon Web Services on your behalf.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-associatedroles
        '''
        result = self._values.get("associated_roles")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[CfnDBCluster.DBClusterRoleProperty, _aws_cdk_core_f4b25747.IResolvable]]]], result)

    @builtins.property
    def auto_minor_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether minor engine upgrades are applied automatically to the DB cluster during the maintenance window.

        By default, minor engine upgrades are applied automatically.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-autominorversionupgrade
        '''
        result = self._values.get("auto_minor_version_upgrade")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def availability_zones(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of Availability Zones (AZs) where instances in the DB cluster can be created.

        For information on AWS Regions and Availability Zones, see `Choosing the Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-availabilityzones
        '''
        result = self._values.get("availability_zones")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def backtrack_window(self) -> typing.Optional[jsii.Number]:
        '''The target backtrack window, in seconds. To disable backtracking, set this value to 0.

        .. epigraph::

           Currently, Backtrack is only supported for Aurora MySQL DB clusters.

        Default: 0

        Constraints:

        - If specified, this value must be set to a number from 0 to 259,200 (72 hours).

        Valid for: Aurora MySQL DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-backtrackwindow
        '''
        result = self._values.get("backtrack_window")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def backup_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days for which automated backups are retained.

        Default: 1

        Constraints:

        - Must be a value from 1 to 35

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-backupretentionperiod
        '''
        result = self._values.get("backup_retention_period")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def copy_tags_to_snapshot(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to copy all tags from the DB cluster to snapshots of the DB cluster.

        The default is not to copy them.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-copytagstosnapshot
        '''
        result = self._values.get("copy_tags_to_snapshot")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def database_name(self) -> typing.Optional[builtins.str]:
        '''The name of your database.

        If you don't provide a name, then Amazon RDS won't create a database in this DB cluster. For naming constraints, see `Naming Constraints <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-databasename
        '''
        result = self._values.get("database_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The DB cluster identifier. This parameter is stored as a lowercase string.

        Constraints:

        - Must contain from 1 to 63 letters, numbers, or hyphens.
        - First character must be a letter.
        - Can't end with a hyphen or contain two consecutive hyphens.

        Example: ``my-cluster1``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusteridentifier
        '''
        result = self._values.get("db_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_cluster_instance_class(self) -> typing.Optional[builtins.str]:
        '''The compute and memory capacity of each DB instance in the Multi-AZ DB cluster, for example db.m6gd.xlarge. Not all DB instance classes are available in all AWS Regions , or for all database engines.

        For the full list of DB instance classes and availability for your engine, see `DB instance class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide* .

        This setting is required to create a Multi-AZ DB cluster.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusterinstanceclass
        '''
        result = self._values.get("db_cluster_instance_class")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_cluster_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB cluster parameter group to associate with this DB cluster.

        .. epigraph::

           If you apply a parameter group to an existing DB cluster, then its DB instances might need to reboot. This can result in an outage while the DB instances are rebooting.

           If you apply a change to parameter group associated with a stopped DB cluster, then the update stack waits until the DB cluster is started.

        To list all of the available DB cluster parameter group names, use the following command:

        ``aws rds describe-db-cluster-parameter-groups --query "DBClusterParameterGroups[].DBClusterParameterGroupName" --output text``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbclusterparametergroupname
        '''
        result = self._values.get("db_cluster_parameter_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_instance_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB parameter group to apply to all instances of the DB cluster.

        .. epigraph::

           When you apply a parameter group using the ``DBInstanceParameterGroupName`` parameter, the DB cluster isn't rebooted automatically. Also, parameter changes are applied immediately rather than during the next maintenance window.

        Default: The existing name setting

        Constraints:

        - The DB parameter group must be in the same DB parameter group family as this DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbinstanceparametergroupname
        '''
        result = self._values.get("db_instance_parameter_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        '''A DB subnet group that you want to associate with this DB cluster.

        If you are restoring a DB cluster to a point in time with ``RestoreType`` set to ``copy-on-write`` , and don't specify a DB subnet group name, then the DB cluster is restored with a default DB subnet group.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbsubnetgroupname
        '''
        result = self._values.get("db_subnet_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_system_id(self) -> typing.Optional[builtins.str]:
        '''Reserved for future use.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-dbsystemid
        '''
        result = self._values.get("db_system_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether the DB cluster has deletion protection enabled.

        The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-deletionprotection
        '''
        result = self._values.get("deletion_protection")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def domain(self) -> typing.Optional[builtins.str]:
        '''Indicates the directory ID of the Active Directory to create the DB cluster.

        For Amazon Aurora DB clusters, Amazon RDS can use Kerberos authentication to authenticate users that connect to the DB cluster.

        For more information, see `Kerberos authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/kerberos-authentication.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-domain
        '''
        result = self._values.get("domain")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def domain_iam_role_name(self) -> typing.Optional[builtins.str]:
        '''Specifies the name of the IAM role to use when making API calls to the Directory Service.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-domainiamrolename
        '''
        result = self._values.get("domain_iam_role_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enable_cloudwatch_logs_exports(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of log types that need to be enabled for exporting to CloudWatch Logs.

        The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Aurora User Guide* .

        *Aurora MySQL*

        Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery``

        *Aurora PostgreSQL*

        Valid values: ``postgresql``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enablecloudwatchlogsexports
        '''
        result = self._values.get("enable_cloudwatch_logs_exports")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def enable_http_endpoint(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to enable the HTTP endpoint for an Aurora Serverless DB cluster.

        By default, the HTTP endpoint is disabled.

        When enabled, the HTTP endpoint provides a connectionless web service API for running SQL queries on the Aurora Serverless DB cluster. You can also query your database from inside the RDS console with the query editor.

        For more information, see `Using the Data API for Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enablehttpendpoint
        '''
        result = self._values.get("enable_http_endpoint")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def enable_iam_database_authentication(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.

        By default, mapping is disabled.

        For more information, see `IAM Database Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon Aurora User Guide.*

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enableiamdatabaseauthentication
        '''
        result = self._values.get("enable_iam_database_authentication")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def engine(self) -> typing.Optional[builtins.str]:
        '''The name of the database engine to be used for this DB cluster.

        Valid Values:

        - ``aurora`` (for MySQL 5.6-compatible Aurora)
        - ``aurora-mysql`` (for MySQL 5.7-compatible Aurora)
        - ``aurora-postgresql``
        - ``mysql``
        - ``postgres``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engine
        '''
        result = self._values.get("engine")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def engine_mode(self) -> typing.Optional[builtins.str]:
        '''The DB engine mode of the DB cluster, either ``provisioned`` , ``serverless`` , ``parallelquery`` , ``global`` , or ``multimaster`` .

        The ``parallelquery`` engine mode isn't required for Aurora MySQL version 1.23 and higher 1.x versions, and version 2.09 and higher 2.x versions.

        The ``global`` engine mode isn't required for Aurora MySQL version 1.22 and higher 1.x versions, and ``global`` engine mode isn't required for any 2.x versions.

        The ``multimaster`` engine mode only applies for DB clusters created with Aurora MySQL version 5.6.10a.

        For Aurora PostgreSQL, the ``global`` engine mode isn't required, and both the ``parallelquery`` and the ``multimaster`` engine modes currently aren't supported.

        Limitations and requirements apply to some DB engine modes. For more information, see the following sections in the *Amazon Aurora User Guide* :

        - `Limitations of Aurora Serverless <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations>`_
        - `Limitations of Parallel Query <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-mysql-parallel-query.html#aurora-mysql-parallel-query-limitations>`_
        - `Limitations of Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html#aurora-global-database.limitations>`_
        - `Limitations of Multi-Master Clusters <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-multi-master.html#aurora-multi-master-limitations>`_

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-enginemode
        '''
        result = self._values.get("engine_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def engine_version(self) -> typing.Optional[builtins.str]:
        '''The version number of the database engine to use.

        To list all of the available engine versions for ``aurora`` (for MySQL 5.6-compatible Aurora), use the following command:

        ``aws rds describe-db-engine-versions --engine aurora --query "DBEngineVersions[].EngineVersion"``

        To list all of the available engine versions for ``aurora-mysql`` (for MySQL 5.7-compatible Aurora), use the following command:

        ``aws rds describe-db-engine-versions --engine aurora-mysql --query "DBEngineVersions[].EngineVersion"``

        To list all of the available engine versions for ``aurora-postgresql`` , use the following command:

        ``aws rds describe-db-engine-versions --engine aurora-postgresql --query "DBEngineVersions[].EngineVersion"``

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-engineversion
        '''
        result = self._values.get("engine_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def global_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''If you are configuring an Aurora global database cluster and want your Aurora DB cluster to be a secondary member in the global database cluster, specify the global cluster ID of the global database cluster.

        To define the primary database cluster of the global cluster, use the `AWS::RDS::GlobalCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html>`_ resource.

        If you aren't configuring a global database cluster, don't specify this property.
        .. epigraph::

           To remove the DB cluster from a global database cluster, specify an empty value for the ``GlobalClusterIdentifier`` property.

        For information about Aurora global databases, see `Working with Amazon Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html>`_ in the *Amazon Aurora User Guide* .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-globalclusteridentifier
        '''
        result = self._values.get("global_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def iops(self) -> typing.Optional[jsii.Number]:
        '''The amount of Provisioned IOPS (input/output operations per second) to be initially allocated for each DB instance in the Multi-AZ DB cluster.

        For information about valid IOPS values, see `Amazon RDS Provisioned IOPS storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* .

        This setting is required to create a Multi-AZ DB cluster.

        Constraints: Must be a multiple between .5 and 50 of the storage amount for the DB cluster.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-iops
        '''
        result = self._values.get("iops")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the AWS KMS key that is used to encrypt the database instances in the DB cluster, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` .

        If you enable the ``StorageEncrypted`` property but don't specify this property, the default KMS key is used. If you specify this property, you must set the ``StorageEncrypted`` property to ``true`` .

        If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-kmskeyid
        '''
        result = self._values.get("kms_key_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def manage_master_user_password(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to manage the master user password with AWS Secrets Manager.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*

        Constraints:

        - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-managemasteruserpassword
        '''
        result = self._values.get("manage_master_user_password")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def master_username(self) -> typing.Optional[builtins.str]:
        '''The name of the master user for the DB cluster.

        .. epigraph::

           If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masterusername
        '''
        result = self._values.get("master_username")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def master_user_password(self) -> typing.Optional[builtins.str]:
        '''The master password for the DB instance.

        .. epigraph::

           If you specify the ``SourceDBClusterIdentifier`` , ``SnapshotIdentifier`` , or ``GlobalClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, the snapshot, or the primary DB cluster for the global database cluster, respectively.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masteruserpassword
        '''
        result = self._values.get("master_user_password")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def master_user_secret(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBCluster.MasterUserSecretProperty]]:
        '''Contains the secret managed by RDS in AWS Secrets Manager for the master user password.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-masterusersecret
        '''
        result = self._values.get("master_user_secret")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBCluster.MasterUserSecretProperty]], result)

    @builtins.property
    def monitoring_interval(self) -> typing.Optional[jsii.Number]:
        '''The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB cluster.

        To turn off collecting Enhanced Monitoring metrics, specify 0. The default is 0.

        If ``MonitoringRoleArn`` is specified, also set ``MonitoringInterval`` to a value other than 0.

        Valid Values: ``0, 1, 5, 10, 15, 30, 60``

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-monitoringinterval
        '''
        result = self._values.get("monitoring_interval")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def monitoring_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) for the IAM role that permits RDS to send Enhanced Monitoring metrics to Amazon CloudWatch Logs.

        An example is ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting up and enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* .

        If ``MonitoringInterval`` is set to a value other than 0, supply a ``MonitoringRoleArn`` value.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-monitoringrolearn
        '''
        result = self._values.get("monitoring_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def network_type(self) -> typing.Optional[builtins.str]:
        '''The network type of the DB cluster.

        Valid values:

        - ``IPV4``
        - ``DUAL``

        The network type is determined by the ``DBSubnetGroup`` specified for the DB cluster. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ).

        For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon Aurora User Guide.*

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-networktype
        '''
        result = self._values.get("network_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def performance_insights_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to turn on Performance Insights for the DB cluster.

        For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* .

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-performanceinsightsenabled
        '''
        result = self._values.get("performance_insights_enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def performance_insights_kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key identifier for encryption of Performance Insights data.

        The AWS KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.

        If you don't specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account . Your AWS account has a different default KMS key for each AWS Region .

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-performanceinsightskmskeyid
        '''
        result = self._values.get("performance_insights_kms_key_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def performance_insights_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days to retain Performance Insights data. The default is 7 days. The following values are valid:.

        - 7
        - *month* * 31, where *month* is a number of months from 1-23
        - 731

        For example, the following values are valid:

        - 93 (3 months * 31)
        - 341 (11 months * 31)
        - 589 (19 months * 31)
        - 731

        If you specify a retention period such as 94, which isn't a valid value, RDS issues an error.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-performanceinsightsretentionperiod
        '''
        result = self._values.get("performance_insights_retention_period")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        '''The port number on which the DB instances in the DB cluster accept connections.

        Default:

        - When ``EngineMode`` is ``provisioned`` , ``3306`` (for both Aurora MySQL and Aurora PostgreSQL)
        - When ``EngineMode`` is ``serverless`` :
        - ``3306`` when ``Engine`` is ``aurora`` or ``aurora-mysql``
        - ``5432`` when ``Engine`` is ``aurora-postgresql``

        .. epigraph::

           The ``No interruption`` on update behavior only applies to DB clusters. If you are updating a DB instance, see `Port <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-port>`_ for the AWS::RDS::DBInstance resource.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-port
        '''
        result = self._values.get("port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        '''The daily time range during which automated backups are created.

        For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html#Aurora.Managing.Backups.BackupWindow>`_ in the *Amazon Aurora User Guide.*

        Constraints:

        - Must be in the format ``hh24:mi-hh24:mi`` .
        - Must be in Universal Coordinated Time (UTC).
        - Must not conflict with the preferred maintenance window.
        - Must be at least 30 minutes.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-preferredbackupwindow
        '''
        result = self._values.get("preferred_backup_window")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        '''The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).

        Format: ``ddd:hh24:mi-ddd:hh24:mi``

        The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Cluster Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow.Aurora>`_ in the *Amazon Aurora User Guide.*

        Valid Days: Mon, Tue, Wed, Thu, Fri, Sat, Sun.

        Constraints: Minimum 30-minute window.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-preferredmaintenancewindow
        '''
        result = self._values.get("preferred_maintenance_window")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def publicly_accessible(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether the DB cluster is publicly accessible.

        When the DB cluster is publicly accessible, its Domain Name System (DNS) endpoint resolves to the private IP address from within the DB cluster's virtual private cloud (VPC). It resolves to the public IP address from outside of the DB cluster's VPC. Access to the DB cluster is ultimately controlled by the security group it uses. That public access isn't permitted if the security group assigned to the DB cluster doesn't permit it.

        When the DB cluster isn't publicly accessible, it is an internal DB cluster with a DNS name that resolves to a private IP address.

        Default: The default behavior varies depending on whether ``DBSubnetGroupName`` is specified.

        If ``DBSubnetGroupName`` isn't specified, and ``PubliclyAccessible`` isn't specified, the following applies:

        - If the default VPC in the target Region doesn’t have an internet gateway attached to it, the DB cluster is private.
        - If the default VPC in the target Region has an internet gateway attached to it, the DB cluster is public.

        If ``DBSubnetGroupName`` is specified, and ``PubliclyAccessible`` isn't specified, the following applies:

        - If the subnets are part of a VPC that doesn’t have an internet gateway attached to it, the DB cluster is private.
        - If the subnets are part of a VPC that has an internet gateway attached to it, the DB cluster is public.

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-publiclyaccessible
        '''
        result = self._values.get("publicly_accessible")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def replication_source_identifier(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the source DB instance or DB cluster if this DB cluster is created as a read replica.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-replicationsourceidentifier
        '''
        result = self._values.get("replication_source_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def restore_type(self) -> typing.Optional[builtins.str]:
        '''The type of restore to be performed. You can specify one of the following values:.

        - ``full-copy`` - The new DB cluster is restored as a full copy of the source DB cluster.
        - ``copy-on-write`` - The new DB cluster is restored as a clone of the source DB cluster.

        Constraints: You can't specify ``copy-on-write`` if the engine version of the source DB cluster is earlier than 1.11.

        If you don't specify a ``RestoreType`` value, then the new DB cluster is restored as a full copy of the source DB cluster.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-restoretype
        '''
        result = self._values.get("restore_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scaling_configuration(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBCluster.ScalingConfigurationProperty]]:
        '''The ``ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless DB cluster.

        This property is only supported for Aurora Serverless v1. For Aurora Serverless v2, use ``ServerlessV2ScalingConfiguration`` property.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-scalingconfiguration
        '''
        result = self._values.get("scaling_configuration")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBCluster.ScalingConfigurationProperty]], result)

    @builtins.property
    def serverless_v2_scaling_configuration(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBCluster.ServerlessV2ScalingConfigurationProperty]]:
        '''The ``ServerlessV2ScalingConfiguration`` property type specifies the scaling configuration of an Aurora Serverless V2 DB cluster.

        This property is only supported for Aurora Serverless v2. For Aurora Serverless v1, use ``ScalingConfiguration`` property.

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-serverlessv2scalingconfiguration
        '''
        result = self._values.get("serverless_v2_scaling_configuration")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBCluster.ServerlessV2ScalingConfigurationProperty]], result)

    @builtins.property
    def snapshot_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier for the DB snapshot or DB cluster snapshot to restore from.

        You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot. However, you can use only the ARN to specify a DB snapshot.

        After you restore a DB cluster with a ``SnapshotIdentifier`` property, you must specify the same ``SnapshotIdentifier`` property for any future updates to the DB cluster. When you specify this property for an update, the DB cluster is not restored from the snapshot again, and the data in the database is not changed. However, if you don't specify the ``SnapshotIdentifier`` property, an empty DB cluster is created, and the original DB cluster is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB cluster is restored from the specified ``SnapshotIdentifier`` property, and the original DB cluster is deleted.

        If you specify the ``SnapshotIdentifier`` property to restore a DB cluster (as opposed to specifying it for DB cluster updates), then don't specify the following properties:

        - ``GlobalClusterIdentifier``
        - ``MasterUsername``
        - ``MasterUserPassword``
        - ``ReplicationSourceIdentifier``
        - ``RestoreType``
        - ``SourceDBClusterIdentifier``
        - ``SourceRegion``
        - ``StorageEncrypted`` (for an encrypted snapshot)
        - ``UseLatestRestorableTime``

        Constraints:

        - Must match the identifier of an existing Snapshot.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-snapshotidentifier
        '''
        result = self._values.get("snapshot_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''When restoring a DB cluster to a point in time, the identifier of the source DB cluster from which to restore.

        Constraints:

        - Must match the identifier of an existing DBCluster.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-sourcedbclusteridentifier
        '''
        result = self._values.get("source_db_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_region(self) -> typing.Optional[builtins.str]:
        '''The AWS Region which contains the source DB cluster when replicating a DB cluster. For example, ``us-east-1`` .

        Valid for: Aurora DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-sourceregion
        '''
        result = self._values.get("source_region")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''Indicates whether the DB cluster is encrypted.

        If you specify the ``KmsKeyId`` property, then you must enable encryption.

        If you specify the ``SourceDBClusterIdentifier`` property, don't specify this property. The value is inherited from the source DB cluster, and if the DB cluster is encrypted, the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB cluster is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB cluster to be encrypted, then don't set this property or set it to ``false`` .

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-storageencrypted
        '''
        result = self._values.get("storage_encrypted")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def storage_type(self) -> typing.Optional[builtins.str]:
        '''Specifies the storage type to be associated with the DB cluster.

        This setting is required to create a Multi-AZ DB cluster.

        Valid values: ``io1``

        When specified, a value for the ``Iops`` parameter is required.

        Default: ``io1``

        Valid for: Multi-AZ DB clusters only

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-storagetype
        '''
        result = self._values.get("storage_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]]:
        '''An optional array of key-value pairs to apply to this DB cluster.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]], result)

    @builtins.property
    def use_latest_restorable_time(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to restore the DB cluster to the latest restorable backup time.

        By default, the DB cluster is not restored to the latest restorable backup time.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-uselatestrestorabletime
        '''
        result = self._values.get("use_latest_restorable_time")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of EC2 VPC security groups to associate with this DB cluster.

        If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        Valid for: Aurora DB clusters and Multi-AZ DB clusters

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html#cfn-rds-dbcluster-vpcsecuritygroupids
        '''
        result = self._values.get("vpc_security_group_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnDBInstance(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBInstance",
):
    '''A CloudFormation ``AWS::RDS::DBInstance``.

    The ``AWS::RDS::DBInstance`` resource creates an Amazon DB instance. The new DB instance can be an RDS DB instance, or it can be a DB instance in an Aurora DB cluster.

    For more information about creating an RDS DB instance, see `Creating an Amazon RDS DB instance <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_CreateDBInstance.html>`_ in the *Amazon RDS User Guide* .

    For more information about creating a DB instance in an Aurora DB cluster, see `Creating an Amazon Aurora DB cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.CreateInstance.html>`_ in the *Amazon Aurora User Guide* .

    If you import an existing DB instance, and the template configuration doesn't match the actual configuration of the DB instance, AWS CloudFormation applies the changes in the template during the import operation.
    .. epigraph::

       If a DB instance is deleted or replaced during an update, AWS CloudFormation deletes all automated snapshots. However, it retains manual DB snapshots. During an update that requires replacement, you can apply a stack policy to prevent DB instances from being replaced. For more information, see `Prevent Updates to Stack Resources <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/protect-stack-resources.html>`_ .

    *Updating DB instances*

    When properties labeled " *Update requires:* `Replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ " are updated, AWS CloudFormation first creates a replacement DB instance, then changes references from other dependent resources to point to the replacement DB instance, and finally deletes the old DB instance.
    .. epigraph::

       We highly recommend that you take a snapshot of the database before updating the stack. If you don't, you lose the data when AWS CloudFormation replaces your DB instance. To preserve your data, perform the following procedure:

       - Deactivate any applications that are using the DB instance so that there's no activity on the DB instance.
       - Create a snapshot of the DB instance. For more information, see `Creating a DB Snapshot <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_CreateSnapshot.html>`_ .
       - If you want to restore your instance using a DB snapshot, modify the updated template with your DB instance changes and add the ``DBSnapshotIdentifier`` property with the ID of the DB snapshot that you want to use.

       After you restore a DB instance with a ``DBSnapshotIdentifier`` property, you must specify the same ``DBSnapshotIdentifier`` property for any future updates to the DB instance. When you specify this property for an update, the DB instance is not restored from the DB snapshot again, and the data in the database is not changed. However, if you don't specify the ``DBSnapshotIdentifier`` property, an empty DB instance is created, and the original DB instance is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB instance is restored from the specified ``DBSnapshotIdentifier`` property, and the original DB instance is deleted.

       - Update the stack.

    For more information about updating other properties of this resource, see ``[ModifyDBInstance](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_ModifyDBInstance.html)`` . For more information about updating stacks, see `AWS CloudFormation Stacks Updates <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks.html>`_ .

    *Deleting DB instances*

    For DB instances that are part of an Aurora DB cluster, you can set a deletion policy for your DB instance to control how AWS CloudFormation handles the DB instance when the stack is deleted. For Amazon RDS DB instances, you can choose to *retain* the DB instance, to *delete* the DB instance, or to *create a snapshot* of the DB instance. The default AWS CloudFormation behavior depends on the ``DBClusterIdentifier`` property:

    - For ``AWS::RDS::DBInstance`` resources that don't specify the ``DBClusterIdentifier`` property, AWS CloudFormation saves a snapshot of the DB instance.
    - For ``AWS::RDS::DBInstance`` resources that do specify the ``DBClusterIdentifier`` property, AWS CloudFormation deletes the DB instance.

    For more information, see `DeletionPolicy Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html>`_ .

    :cloudformationResource: AWS::RDS::DBInstance
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        cfn_dBInstance = rds.CfnDBInstance(self, "MyCfnDBInstance",
            allocated_storage="allocatedStorage",
            allow_major_version_upgrade=False,
            associated_roles=[rds.CfnDBInstance.DBInstanceRoleProperty(
                feature_name="featureName",
                role_arn="roleArn"
            )],
            auto_minor_version_upgrade=False,
            availability_zone="availabilityZone",
            backup_retention_period=123,
            ca_certificate_identifier="caCertificateIdentifier",
            certificate_details=rds.CfnDBInstance.CertificateDetailsProperty(
                ca_identifier="caIdentifier",
                valid_till="validTill"
            ),
            certificate_rotation_restart=False,
            character_set_name="characterSetName",
            copy_tags_to_snapshot=False,
            custom_iam_instance_profile="customIamInstanceProfile",
            db_cluster_identifier="dbClusterIdentifier",
            db_cluster_snapshot_identifier="dbClusterSnapshotIdentifier",
            db_instance_class="dbInstanceClass",
            db_instance_identifier="dbInstanceIdentifier",
            db_name="dbName",
            db_parameter_group_name="dbParameterGroupName",
            db_security_groups=["dbSecurityGroups"],
            db_snapshot_identifier="dbSnapshotIdentifier",
            db_subnet_group_name="dbSubnetGroupName",
            delete_automated_backups=False,
            deletion_protection=False,
            domain="domain",
            domain_iam_role_name="domainIamRoleName",
            enable_cloudwatch_logs_exports=["enableCloudwatchLogsExports"],
            enable_iam_database_authentication=False,
            enable_performance_insights=False,
            endpoint=rds.CfnDBInstance.EndpointProperty(
                address="address",
                hosted_zone_id="hostedZoneId",
                port="port"
            ),
            engine="engine",
            engine_version="engineVersion",
            iops=123,
            kms_key_id="kmsKeyId",
            license_model="licenseModel",
            manage_master_user_password=False,
            master_username="masterUsername",
            master_user_password="masterUserPassword",
            master_user_secret=rds.CfnDBInstance.MasterUserSecretProperty(
                kms_key_id="kmsKeyId",
                secret_arn="secretArn"
            ),
            max_allocated_storage=123,
            monitoring_interval=123,
            monitoring_role_arn="monitoringRoleArn",
            multi_az=False,
            nchar_character_set_name="ncharCharacterSetName",
            network_type="networkType",
            option_group_name="optionGroupName",
            performance_insights_kms_key_id="performanceInsightsKmsKeyId",
            performance_insights_retention_period=123,
            port="port",
            preferred_backup_window="preferredBackupWindow",
            preferred_maintenance_window="preferredMaintenanceWindow",
            processor_features=[rds.CfnDBInstance.ProcessorFeatureProperty(
                name="name",
                value="value"
            )],
            promotion_tier=123,
            publicly_accessible=False,
            replica_mode="replicaMode",
            restore_time="restoreTime",
            source_db_instance_automated_backups_arn="sourceDbInstanceAutomatedBackupsArn",
            source_db_instance_identifier="sourceDbInstanceIdentifier",
            source_dbi_resource_id="sourceDbiResourceId",
            source_region="sourceRegion",
            storage_encrypted=False,
            storage_throughput=123,
            storage_type="storageType",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            timezone="timezone",
            use_default_processor_features=False,
            use_latest_restorable_time=False,
            vpc_security_groups=["vpcSecurityGroups"]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        allocated_storage: typing.Optional[builtins.str] = None,
        allow_major_version_upgrade: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        associated_roles: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDBInstance.DBInstanceRoleProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        auto_minor_version_upgrade: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        backup_retention_period: typing.Optional[jsii.Number] = None,
        ca_certificate_identifier: typing.Optional[builtins.str] = None,
        certificate_details: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDBInstance.CertificateDetailsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        certificate_rotation_restart: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        character_set_name: typing.Optional[builtins.str] = None,
        copy_tags_to_snapshot: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        custom_iam_instance_profile: typing.Optional[builtins.str] = None,
        db_cluster_identifier: typing.Optional[builtins.str] = None,
        db_cluster_snapshot_identifier: typing.Optional[builtins.str] = None,
        db_instance_class: typing.Optional[builtins.str] = None,
        db_instance_identifier: typing.Optional[builtins.str] = None,
        db_name: typing.Optional[builtins.str] = None,
        db_parameter_group_name: typing.Optional[builtins.str] = None,
        db_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        db_snapshot_identifier: typing.Optional[builtins.str] = None,
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        delete_automated_backups: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_iam_role_name: typing.Optional[builtins.str] = None,
        enable_cloudwatch_logs_exports: typing.Optional[typing.Sequence[builtins.str]] = None,
        enable_iam_database_authentication: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        enable_performance_insights: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        endpoint: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDBInstance.EndpointProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        license_model: typing.Optional[builtins.str] = None,
        manage_master_user_password: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        master_username: typing.Optional[builtins.str] = None,
        master_user_password: typing.Optional[builtins.str] = None,
        master_user_secret: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDBInstance.MasterUserSecretProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        max_allocated_storage: typing.Optional[jsii.Number] = None,
        monitoring_interval: typing.Optional[jsii.Number] = None,
        monitoring_role_arn: typing.Optional[builtins.str] = None,
        multi_az: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        nchar_character_set_name: typing.Optional[builtins.str] = None,
        network_type: typing.Optional[builtins.str] = None,
        option_group_name: typing.Optional[builtins.str] = None,
        performance_insights_kms_key_id: typing.Optional[builtins.str] = None,
        performance_insights_retention_period: typing.Optional[jsii.Number] = None,
        port: typing.Optional[builtins.str] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        processor_features: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDBInstance.ProcessorFeatureProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        promotion_tier: typing.Optional[jsii.Number] = None,
        publicly_accessible: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        replica_mode: typing.Optional[builtins.str] = None,
        restore_time: typing.Optional[builtins.str] = None,
        source_db_instance_automated_backups_arn: typing.Optional[builtins.str] = None,
        source_db_instance_identifier: typing.Optional[builtins.str] = None,
        source_dbi_resource_id: typing.Optional[builtins.str] = None,
        source_region: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        storage_throughput: typing.Optional[jsii.Number] = None,
        storage_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
        timezone: typing.Optional[builtins.str] = None,
        use_default_processor_features: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        use_latest_restorable_time: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        vpc_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBInstance``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param allocated_storage: The amount of storage in gibibytes (GiB) to be initially allocated for the database instance. .. epigraph:: If any value is set in the ``Iops`` parameter, ``AllocatedStorage`` must be at least 100 GiB, which corresponds to the minimum Iops value of 1,000. If you increase the ``Iops`` value (in 1,000 IOPS increments), then you must also increase the ``AllocatedStorage`` value (in 100-GiB increments). *Amazon Aurora* Not applicable. Aurora cluster volumes automatically grow as the amount of data in your database increases, though you are only charged for the space that you use in an Aurora cluster volume. *MySQL* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 5 to 3072. *MariaDB* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 5 to 3072. *PostgreSQL* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 5 to 3072. *Oracle* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 10 to 3072. *SQL Server* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): - Enterprise and Standard editions: Must be an integer from 20 to 16384. - Web and Express editions: Must be an integer from 20 to 16384. - Provisioned IOPS storage (io1): - Enterprise and Standard editions: Must be an integer from 20 to 16384. - Web and Express editions: Must be an integer from 20 to 16384. - Magnetic storage (standard): - Enterprise and Standard editions: Must be an integer from 20 to 1024. - Web and Express editions: Must be an integer from 20 to 1024.
        :param allow_major_version_upgrade: A value that indicates whether major version upgrades are allowed. Changing this parameter doesn't result in an outage and the change is asynchronously applied as soon as possible. Constraints: Major version upgrades must be allowed when specifying a value for the ``EngineVersion`` parameter that is a different major version than the DB instance's current version.
        :param associated_roles: The AWS Identity and Access Management (IAM) roles associated with the DB instance. *Amazon Aurora* Not applicable. The associated roles are managed by the DB cluster.
        :param auto_minor_version_upgrade: A value that indicates whether minor engine upgrades are applied automatically to the DB instance during the maintenance window. By default, minor engine upgrades are applied automatically.
        :param availability_zone: The Availability Zone (AZ) where the database will be created. For information on AWS Regions and Availability Zones, see `Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ . *Amazon Aurora* Each Aurora DB cluster hosts copies of its storage in three separate Availability Zones. Specify one of these Availability Zones. Aurora automatically chooses an appropriate Availability Zone if you don't specify one. Default: A random, system-chosen Availability Zone in the endpoint's AWS Region . Example: ``us-east-1d`` Constraint: The ``AvailabilityZone`` parameter can't be specified if the DB instance is a Multi-AZ deployment. The specified Availability Zone must be in the same AWS Region as the current endpoint.
        :param backup_retention_period: The number of days for which automated backups are retained. Setting this parameter to a positive number enables backups. Setting this parameter to 0 disables automated backups. *Amazon Aurora* Not applicable. The retention period for automated backups is managed by the DB cluster. Default: 1 Constraints: - Must be a value from 0 to 35 - Can't be set to 0 if the DB instance is a source to read replicas
        :param ca_certificate_identifier: The identifier of the CA certificate for this DB instance. .. epigraph:: Specifying or updating this property triggers a reboot. For more information about CA certificate identifiers for RDS DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon RDS User Guide* . For more information about CA certificate identifiers for Aurora DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon Aurora User Guide* .
        :param certificate_details: ``AWS::RDS::DBInstance.CertificateDetails``.
        :param certificate_rotation_restart: ``AWS::RDS::DBInstance.CertificateRotationRestart``.
        :param character_set_name: For supported engines, indicates that the DB instance should be associated with the specified character set. *Amazon Aurora* Not applicable. The character set is managed by the DB cluster. For more information, see `AWS::RDS::DBCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html>`_ .
        :param copy_tags_to_snapshot: A value that indicates whether to copy tags from the DB instance to snapshots of the DB instance. By default, tags are not copied. *Amazon Aurora* Not applicable. Copying tags to snapshots is managed by the DB cluster. Setting this value for an Aurora DB instance has no effect on the DB cluster setting.
        :param custom_iam_instance_profile: The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance. The instance profile must meet the following requirements: - The profile must exist in your account. - The profile must have an IAM role that Amazon EC2 has permissions to assume. - The instance profile name and the associated IAM role name must start with the prefix ``AWSRDSCustom`` . For the list of permissions required for the IAM role, see `Configure IAM and your VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-setup-orcl.html#custom-setup-orcl.iam-vpc>`_ in the *Amazon RDS User Guide* . This setting is required for RDS Custom.
        :param db_cluster_identifier: The identifier of the DB cluster that the instance will belong to.
        :param db_cluster_snapshot_identifier: The identifier for the RDS for MySQL Multi-AZ DB cluster snapshot to restore from. For more information on Multi-AZ DB clusters, see `Multi-AZ deployments with two readable standby DB instances <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/multi-az-db-clusters-concepts.html>`_ in the *Amazon RDS User Guide* . Constraints: - Must match the identifier of an existing Multi-AZ DB cluster snapshot. - Can't be specified when ``DBSnapshotIdentifier`` is specified. - Must be specified when ``DBSnapshotIdentifier`` isn't specified. - If you are restoring from a shared manual Multi-AZ DB cluster snapshot, the ``DBClusterSnapshotIdentifier`` must be the ARN of the shared snapshot. - Can't be the identifier of an Aurora DB cluster snapshot. - Can't be the identifier of an RDS for PostgreSQL Multi-AZ DB cluster snapshot.
        :param db_instance_class: The compute and memory capacity of the DB instance, for example, ``db.m4.large`` . Not all DB instance classes are available in all AWS Regions, or for all database engines. For the full list of DB instance classes, and availability for your engine, see `DB Instance Class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide.* For more information about DB instance class pricing and AWS Region support for DB instance classes, see `Amazon RDS Pricing <https://docs.aws.amazon.com/rds/pricing/>`_ .
        :param db_instance_identifier: A name for the DB instance. If you specify a name, AWS CloudFormation converts it to lowercase. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the DB instance. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ . For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* . .. epigraph:: If you specify a name, you can't perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param db_name: The meaning of this parameter differs according to the database engine you use. .. epigraph:: If you specify the ``[DBSnapshotIdentifier](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsnapshotidentifier)`` property, this property only applies to RDS for Oracle. *Amazon Aurora* Not applicable. The database name is managed by the DB cluster. *MySQL* The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Constraints: - Must contain 1 to 64 letters or numbers. - Can't be a word reserved by the specified database engine *MariaDB* The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Constraints: - Must contain 1 to 64 letters or numbers. - Can't be a word reserved by the specified database engine *PostgreSQL* The name of the database to create when the DB instance is created. If this parameter is not specified, the default ``postgres`` database is created in the DB instance. Constraints: - Must begin with a letter. Subsequent characters can be letters, underscores, or digits (0-9). - Must contain 1 to 63 characters. - Can't be a word reserved by the specified database engine *Oracle* The Oracle System ID (SID) of the created DB instance. If you specify ``null`` , the default value ``ORCL`` is used. You can't specify the string NULL, or any other reserved word, for ``DBName`` . Default: ``ORCL`` Constraints: - Can't be longer than 8 characters *SQL Server* Not applicable. Must be null.
        :param db_parameter_group_name: The name of an existing DB parameter group or a reference to an `AWS::RDS::DBParameterGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html>`_ resource created in the template. To list all of the available DB parameter group names, use the following command: ``aws rds describe-db-parameter-groups --query "DBParameterGroups[].DBParameterGroupName" --output text`` .. epigraph:: If any of the data members of the referenced parameter group are changed during an update, the DB instance might need to be restarted, which causes some interruption. If the parameter group contains static parameters, whether they were changed or not, an update triggers a reboot. If you don't specify a value for ``DBParameterGroupName`` property, the default DB parameter group for the specified engine and engine version is used.
        :param db_security_groups: A list of the DB security groups to assign to the DB instance. The list can include both the name of existing DB security groups or references to AWS::RDS::DBSecurityGroup resources created in the template. If you set DBSecurityGroups, you must not set VPCSecurityGroups, and vice versa. Also, note that the DBSecurityGroups property exists only for backwards compatibility with older regions and is no longer recommended for providing security information to an RDS DB instance. Instead, use VPCSecurityGroups. .. epigraph:: If you specify this property, AWS CloudFormation sends only the following properties (if specified) to Amazon RDS during create operations: - ``AllocatedStorage`` - ``AutoMinorVersionUpgrade`` - ``AvailabilityZone`` - ``BackupRetentionPeriod`` - ``CharacterSetName`` - ``DBInstanceClass`` - ``DBName`` - ``DBParameterGroupName`` - ``DBSecurityGroups`` - ``DBSubnetGroupName`` - ``Engine`` - ``EngineVersion`` - ``Iops`` - ``LicenseModel`` - ``MasterUsername`` - ``MasterUserPassword`` - ``MultiAZ`` - ``OptionGroupName`` - ``PreferredBackupWindow`` - ``PreferredMaintenanceWindow`` All other properties are ignored. Specify a virtual private cloud (VPC) security group if you want to submit other properties, such as ``StorageType`` , ``StorageEncrypted`` , or ``KmsKeyId`` . If you're already using the ``DBSecurityGroups`` property, you can't use these other properties by updating your DB instance to use a VPC security group. You must recreate the DB instance.
        :param db_snapshot_identifier: The name or Amazon Resource Name (ARN) of the DB snapshot that's used to restore the DB instance. If you're restoring from a shared manual DB snapshot, you must specify the ARN of the snapshot. By specifying this property, you can create a DB instance from the specified DB snapshot. If the ``DBSnapshotIdentifier`` property is an empty string or the ``AWS::RDS::DBInstance`` declaration has no ``DBSnapshotIdentifier`` property, AWS CloudFormation creates a new database. If the property contains a value (other than an empty string), AWS CloudFormation creates a database from the specified snapshot. If a snapshot with the specified name doesn't exist, AWS CloudFormation can't create the database and it rolls back the stack. Some DB instance properties aren't valid when you restore from a snapshot, such as the ``MasterUsername`` and ``MasterUserPassword`` properties. For information about the properties that you can specify, see the ``RestoreDBInstanceFromDBSnapshot`` action in the *Amazon RDS API Reference* . After you restore a DB instance with a ``DBSnapshotIdentifier`` property, you must specify the same ``DBSnapshotIdentifier`` property for any future updates to the DB instance. When you specify this property for an update, the DB instance is not restored from the DB snapshot again, and the data in the database is not changed. However, if you don't specify the ``DBSnapshotIdentifier`` property, an empty DB instance is created, and the original DB instance is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB instance is restored from the specified ``DBSnapshotIdentifier`` property, and the original DB instance is deleted. If you specify the ``DBSnapshotIdentifier`` property to restore a DB instance (as opposed to specifying it for DB instance updates), then don't specify the following properties: - ``CharacterSetName`` - ``DBClusterIdentifier`` - ``DBName`` - ``DeleteAutomatedBackups`` - ``EnablePerformanceInsights`` - ``KmsKeyId`` - ``MasterUsername`` - ``MasterUserPassword`` - ``PerformanceInsightsKMSKeyId`` - ``PerformanceInsightsRetentionPeriod`` - ``PromotionTier`` - ``SourceDBInstanceIdentifier`` - ``SourceRegion`` - ``StorageEncrypted`` (for an encrypted snapshot) - ``Timezone`` *Amazon Aurora* Not applicable. Snapshot restore is managed by the DB cluster.
        :param db_subnet_group_name: A DB subnet group to associate with the DB instance. If you update this value, the new subnet group must be a subnet group in a new VPC. If there's no DB subnet group, then the DB instance isn't a VPC DB instance. For more information about using Amazon RDS in a VPC, see `Using Amazon RDS with Amazon Virtual Private Cloud (VPC) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. The DB subnet group is managed by the DB cluster. If specified, the setting must match the DB cluster setting.
        :param delete_automated_backups: A value that indicates whether to remove automated backups immediately after the DB instance is deleted. This parameter isn't case-sensitive. The default is to remove automated backups immediately after the DB instance is deleted. *Amazon Aurora* Not applicable. When you delete a DB cluster, all automated backups for that DB cluster are deleted and can't be recovered. Manual DB cluster snapshots of the DB cluster are not deleted.
        :param deletion_protection: A value that indicates whether the DB instance has deletion protection enabled. The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled. For more information, see `Deleting a DB Instance <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_DeleteInstance.html>`_ . *Amazon Aurora* Not applicable. You can enable or disable deletion protection for the DB cluster. For more information, see ``CreateDBCluster`` . DB instances in a DB cluster can be deleted even when deletion protection is enabled for the DB cluster.
        :param domain: The Active Directory directory ID to create the DB instance in. Currently, only Microsoft SQL Server, Oracle, and PostgreSQL DB instances can be created in an Active Directory Domain. For more information, see `Kerberos Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html>`_ in the *Amazon RDS User Guide* .
        :param domain_iam_role_name: Specify the name of the IAM role to be used when making API calls to the Directory Service. This setting doesn't apply to RDS Custom. *Amazon Aurora* Not applicable. The domain is managed by the DB cluster.
        :param enable_cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Relational Database Service User Guide* . *Amazon Aurora* Not applicable. CloudWatch Logs exports are managed by the DB cluster. *MariaDB* Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery`` *Microsoft SQL Server* Valid values: ``agent`` , ``error`` *MySQL* Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery`` *Oracle* Valid values: ``alert`` , ``audit`` , ``listener`` , ``trace`` *PostgreSQL* Valid values: ``postgresql`` , ``upgrade``
        :param enable_iam_database_authentication: A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts. By default, mapping is disabled. This property is supported for RDS for MariaDB, RDS for MySQL, and RDS for PostgreSQL. For more information, see `IAM Database Authentication for MariaDB, MySQL, and PostgreSQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon RDS User Guide.* *Amazon Aurora* Not applicable. Mapping AWS IAM accounts to database accounts is managed by the DB cluster.
        :param enable_performance_insights: A value that indicates whether to enable Performance Insights for the DB instance. For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* . This setting doesn't apply to RDS Custom.
        :param endpoint: Specifies the connection endpoint. .. epigraph:: The endpoint might not be shown for instances whose status is ``creating`` .
        :param engine: The name of the database engine that you want to use for this DB instance. .. epigraph:: When you are creating a DB instance, the ``Engine`` property is required. Valid Values: - ``aurora`` (for MySQL 5.6-compatible Aurora) - ``aurora-mysql`` (for MySQL 5.7-compatible Aurora) - ``aurora-postgresql`` - ``mariadb`` - ``mysql`` - ``oracle-ee`` - ``oracle-ee-cdb`` - ``oracle-se2`` - ``oracle-se2-cdb`` - ``postgres`` - ``sqlserver-ee`` - ``sqlserver-se`` - ``sqlserver-ex`` - ``sqlserver-web``
        :param engine_version: The version number of the database engine to use. For a list of valid engine versions, use the ``DescribeDBEngineVersions`` action. The following are the database engines and links to information about the major and minor versions that are available with Amazon RDS. Not every database engine is available for every AWS Region. *Amazon Aurora* Not applicable. The version number of the database engine to be used by the DB instance is managed by the DB cluster. *MariaDB* See `MariaDB on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MariaDB.html#MariaDB.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.* *Microsoft SQL Server* See `Microsoft SQL Server Versions on Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.VersionSupport>`_ in the *Amazon RDS User Guide.* *MySQL* See `MySQL on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MySQL.html#MySQL.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.* *Oracle* See `Oracle Database Engine Release Notes <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.Oracle.PatchComposition.html>`_ in the *Amazon RDS User Guide.* *PostgreSQL* See `Supported PostgreSQL Database Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_PostgreSQL.html#PostgreSQL.Concepts.General.DBVersions>`_ in the *Amazon RDS User Guide.*
        :param iops: The number of I/O operations per second (IOPS) that the database provisions. The value must be equal to or greater than 1000. If you specify this property, you must follow the range of allowed ratios of your requested IOPS rate to the amount of storage that you allocate (IOPS to allocated storage). For example, you can provision an Oracle database instance with 1000 IOPS and 200 GiB of storage (a ratio of 5:1), or specify 2000 IOPS with 200 GiB of storage (a ratio of 10:1). For more information, see `Amazon RDS Provisioned IOPS Storage to Improve Performance <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* . .. epigraph:: If you specify ``io1`` for the ``StorageType`` property, then you must also specify the ``Iops`` property.
        :param kms_key_id: The ARN of the AWS KMS key that's used to encrypt the DB instance, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` . If you enable the StorageEncrypted property but don't specify this property, AWS CloudFormation uses the default KMS key. If you specify this property, you must set the StorageEncrypted property to true. If you specify the ``SourceDBInstanceIdentifier`` property, the value is inherited from the source DB instance if the read replica is created in the same region. If you create an encrypted read replica in a different AWS Region, then you must specify a KMS key for the destination AWS Region. KMS encryption keys are specific to the region that they're created in, and you can't use encryption keys from one region in another region. If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used. If you specify ``DBSecurityGroups`` , AWS CloudFormation ignores this property. To specify both a security group and this property, you must use a VPC security group. For more information about Amazon RDS and VPC, see `Using Amazon RDS with Amazon VPC <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. The KMS key identifier is managed by the DB cluster.
        :param license_model: License model information for this DB instance. Valid values: - Aurora MySQL - ``general-public-license`` - Aurora PostgreSQL - ``postgresql-license`` - MariaDB - ``general-public-license`` - Microsoft SQL Server - ``license-included`` - MySQL - ``general-public-license`` - Oracle - ``bring-your-own-license`` or ``license-included`` - PostgreSQL - ``postgresql-license`` .. epigraph:: If you've specified ``DBSecurityGroups`` and then you update the license model, AWS CloudFormation replaces the underlying DB instance. This will incur some interruptions to database availability.
        :param manage_master_user_password: A value that indicates whether to manage the master user password with AWS Secrets Manager. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.* Constraints: - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.
        :param master_username: The master user name for the DB instance. .. epigraph:: If you specify the ``SourceDBInstanceIdentifier`` or ``DBSnapshotIdentifier`` property, don't specify this property. The value is inherited from the source DB instance or snapshot. *Amazon Aurora* Not applicable. The name for the master user is managed by the DB cluster. *MariaDB* Constraints: - Required for MariaDB. - Must be 1 to 16 letters or numbers. - Can't be a reserved word for the chosen database engine. *Microsoft SQL Server* Constraints: - Required for SQL Server. - Must be 1 to 128 letters or numbers. - The first character must be a letter. - Can't be a reserved word for the chosen database engine. *MySQL* Constraints: - Required for MySQL. - Must be 1 to 16 letters or numbers. - First character must be a letter. - Can't be a reserved word for the chosen database engine. *Oracle* Constraints: - Required for Oracle. - Must be 1 to 30 letters or numbers. - First character must be a letter. - Can't be a reserved word for the chosen database engine. *PostgreSQL* Constraints: - Required for PostgreSQL. - Must be 1 to 63 letters or numbers. - First character must be a letter. - Can't be a reserved word for the chosen database engine.
        :param master_user_password: The password for the master user. The password can include any printable ASCII character except "/", """, or "@". *Amazon Aurora* Not applicable. The password for the master user is managed by the DB cluster. *MariaDB* Constraints: Must contain from 8 to 41 characters. *Microsoft SQL Server* Constraints: Must contain from 8 to 128 characters. *MySQL* Constraints: Must contain from 8 to 41 characters. *Oracle* Constraints: Must contain from 8 to 30 characters. *PostgreSQL* Constraints: Must contain from 8 to 128 characters.
        :param master_user_secret: Contains the secret managed by RDS in AWS Secrets Manager for the master user password. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.*
        :param max_allocated_storage: The upper limit in gibibytes (GiB) to which Amazon RDS can automatically scale the storage of the DB instance. For more information about this setting, including limitations that apply to it, see `Managing capacity automatically with Amazon RDS storage autoscaling <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling>`_ in the *Amazon RDS User Guide* . This setting doesn't apply to RDS Custom. *Amazon Aurora* Not applicable. Storage is managed by the DB cluster.
        :param monitoring_interval: The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB instance. To disable collection of Enhanced Monitoring metrics, specify 0. The default is 0. If ``MonitoringRoleArn`` is specified, then you must set ``MonitoringInterval`` to a value other than 0. This setting doesn't apply to RDS Custom. Valid Values: ``0, 1, 5, 10, 15, 30, 60``
        :param monitoring_role_arn: The ARN for the IAM role that permits RDS to send enhanced monitoring metrics to Amazon CloudWatch Logs. For example, ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting Up and Enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* . If ``MonitoringInterval`` is set to a value other than 0, then you must supply a ``MonitoringRoleArn`` value. This setting doesn't apply to RDS Custom.
        :param multi_az: Specifies whether the database instance is a Multi-AZ DB instance deployment. You can't set the ``AvailabilityZone`` parameter if the ``MultiAZ`` parameter is set to true. Currently, you can't use AWS CloudFormation to create a Multi-AZ DB cluster deployment. For more information, see `Multi-AZ deployments for high availability <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.MultiAZ.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. Amazon Aurora storage is replicated across all of the Availability Zones and doesn't require the ``MultiAZ`` option to be set.
        :param nchar_character_set_name: The name of the NCHAR character set for the Oracle DB instance. This parameter doesn't apply to RDS Custom.
        :param network_type: The network type of the DB instance. Valid values: - ``IPV4`` - ``DUAL`` The network type is determined by the ``DBSubnetGroup`` specified for the DB instance. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ). For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon RDS User Guide.*
        :param option_group_name: Indicates that the DB instance should be associated with the specified option group. Permanent options, such as the TDE option for Oracle Advanced Security TDE, can't be removed from an option group. Also, that option group can't be removed from a DB instance once it is associated with a DB instance.
        :param performance_insights_kms_key_id: The AWS KMS key identifier for encryption of Performance Insights data. The KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. If you do not specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account. Your AWS account has a different default KMS key for each AWS Region. For information about enabling Performance Insights, see `EnablePerformanceInsights <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights>`_ .
        :param performance_insights_retention_period: The amount of time, in days, to retain Performance Insights data. Valid values are 7 or 731 (2 years). For information about enabling Performance Insights, see `EnablePerformanceInsights <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights>`_ .
        :param port: The port number on which the database accepts connections. *Amazon Aurora* Not applicable. The port number is managed by the DB cluster.
        :param preferred_backup_window: The daily time range during which automated backups are created if automated backups are enabled, using the ``BackupRetentionPeriod`` parameter. For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow>`_ in the *Amazon RDS User Guide.* Constraints: - Must be in the format ``hh24:mi-hh24:mi`` . - Must be in Universal Coordinated Time (UTC). - Must not conflict with the preferred maintenance window. - Must be at least 30 minutes. *Amazon Aurora* Not applicable. The daily time range for creating automated backups is managed by the DB cluster.
        :param preferred_maintenance_window: The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC). Format: ``ddd:hh24:mi-ddd:hh24:mi`` The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Instance Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow>`_ in the *Amazon RDS User Guide.* .. epigraph:: This property applies when AWS CloudFormation initially creates the DB instance. If you use AWS CloudFormation to update the DB instance, those updates are applied immediately. Constraints: Minimum 30-minute window.
        :param processor_features: The number of CPU cores and the number of threads per core for the DB instance class of the DB instance. This setting doesn't apply to RDS Custom. *Amazon Aurora* Not applicable.
        :param promotion_tier: A value that specifies the order in which an Aurora Replica is promoted to the primary instance after a failure of the existing primary instance. For more information, see `Fault Tolerance for an Aurora DB Cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html#Aurora.Managing.FaultTolerance>`_ in the *Amazon Aurora User Guide* . This setting doesn't apply to RDS Custom. Default: 1 Valid Values: 0 - 15
        :param publicly_accessible: Indicates whether the DB instance is an internet-facing instance. If you specify true, AWS CloudFormation creates an instance with a publicly resolvable DNS name, which resolves to a public IP address. If you specify false, AWS CloudFormation creates an internal instance with a DNS name that resolves to a private IP address. The default behavior value depends on your VPC setup and the database subnet group. For more information, see the ``PubliclyAccessible`` parameter in the `CreateDBInstance <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html>`_ in the *Amazon RDS API Reference* .
        :param replica_mode: The open mode of an Oracle read replica. For more information, see `Working with Oracle Read Replicas for Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html>`_ in the *Amazon RDS User Guide* . This setting is only supported in RDS for Oracle. Default: ``open-read-only`` Valid Values: ``open-read-only`` or ``mounted``
        :param restore_time: The date and time to restore from. Valid Values: Value must be a time in Universal Coordinated Time (UTC) format Constraints: - Must be before the latest restorable time for the DB instance - Can't be specified if the ``UseLatestRestorableTime`` parameter is enabled Example: ``2009-09-07T23:45:00Z``
        :param source_db_instance_automated_backups_arn: The Amazon Resource Name (ARN) of the replicated automated backups from which to restore, for example, ``arn:aws:rds:useast-1:123456789012:auto-backup:ab-L2IJCEXJP7XQ7HOJ4SIEXAMPLE`` . This setting doesn't apply to RDS Custom.
        :param source_db_instance_identifier: If you want to create a read replica DB instance, specify the ID of the source DB instance. Each DB instance can have a limited number of read replicas. For more information, see `Working with Read Replicas <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/USER_ReadRepl.html>`_ in the *Amazon RDS User Guide* . For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* . The ``SourceDBInstanceIdentifier`` property determines whether a DB instance is a read replica. If you remove the ``SourceDBInstanceIdentifier`` property from your template and then update your stack, AWS CloudFormation promotes the Read Replica to a standalone DB instance. .. epigraph:: - If you specify a source DB instance that uses VPC security groups, we recommend that you specify the ``VPCSecurityGroups`` property. If you don't specify the property, the read replica inherits the value of the ``VPCSecurityGroups`` property from the source DB when you create the replica. However, if you update the stack, AWS CloudFormation reverts the replica's ``VPCSecurityGroups`` property to the default value because it's not defined in the stack's template. This change might cause unexpected issues. - Read replicas don't support deletion policies. AWS CloudFormation ignores any deletion policy that's associated with a read replica. - If you specify ``SourceDBInstanceIdentifier`` , don't specify the ``DBSnapshotIdentifier`` property. You can't create a read replica from a snapshot. - Don't set the ``BackupRetentionPeriod`` , ``DBName`` , ``MasterUsername`` , ``MasterUserPassword`` , and ``PreferredBackupWindow`` properties. The database attributes are inherited from the source DB instance, and backups are disabled for read replicas. - If the source DB instance is in a different region than the read replica, specify the source region in ``SourceRegion`` , and specify an ARN for a valid DB instance in ``SourceDBInstanceIdentifier`` . For more information, see `Constructing a Amazon RDS Amazon Resource Name (ARN) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.html#USER_Tagging.ARN>`_ in the *Amazon RDS User Guide* . - For DB instances in Amazon Aurora clusters, don't specify this property. Amazon RDS automatically assigns writer and reader DB instances.
        :param source_dbi_resource_id: The resource ID of the source DB instance from which to restore.
        :param source_region: The ID of the region that contains the source DB instance for the read replica.
        :param storage_encrypted: A value that indicates whether the DB instance is encrypted. By default, it isn't encrypted. If you specify the ``KmsKeyId`` property, then you must enable encryption. If you specify the ``SourceDBInstanceIdentifier`` property, don't specify this property. The value is inherited from the source DB instance, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB instance is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB instance to be encrypted, then don't set this property or set it to ``false`` . *Amazon Aurora* Not applicable. The encryption for DB instances is managed by the DB cluster.
        :param storage_throughput: Specifies the storage throughput value for the DB instance. This setting applies only to the ``gp3`` storage type. This setting doesn't apply to RDS Custom or Amazon Aurora.
        :param storage_type: Specifies the storage type to be associated with the DB instance. Valid values: ``gp2 | gp3 | io1 | standard`` The ``standard`` value is also known as magnetic. If you specify ``io1`` or ``gp3`` , you must also include a value for the ``Iops`` parameter. Default: ``io1`` if the ``Iops`` parameter is specified, otherwise ``gp2`` For more information, see `Amazon RDS DB Instance Storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. Aurora data is stored in the cluster volume, which is a single, virtual volume that uses solid state drives (SSDs).
        :param tags: An optional array of key-value pairs to apply to this DB instance.
        :param timezone: The time zone of the DB instance. The time zone parameter is currently supported only by `Microsoft SQL Server <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone>`_ .
        :param use_default_processor_features: A value that indicates whether the DB instance class of the DB instance uses its default processor features. This setting doesn't apply to RDS Custom.
        :param use_latest_restorable_time: A value that indicates whether the DB instance is restored from the latest backup time. By default, the DB instance isn't restored from the latest backup time. Constraints: Can't be specified if the ``RestoreTime`` parameter is provided.
        :param vpc_security_groups: A list of the VPC security group IDs to assign to the DB instance. The list can include both the physical IDs of existing VPC security groups and references to `AWS::EC2::SecurityGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html>`_ resources created in the template. If you plan to update the resource, don't specify VPC security groups in a shared VPC. If you set ``VPCSecurityGroups`` , you must not set ```DBSecurityGroups`` <https://docs.aws.amazon.com//AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ , and vice versa. .. epigraph:: You can migrate a DB instance in your stack from an RDS DB security group to a VPC security group, but keep the following in mind: - You can't revert to using an RDS security group after you establish a VPC security group membership. - When you migrate your DB instance to VPC security groups, if your stack update rolls back because the DB instance update fails or because an update fails in another AWS CloudFormation resource, the rollback fails because it can't revert to an RDS security group. - To use the properties that are available when you use a VPC security group, you must recreate the DB instance. If you don't, AWS CloudFormation submits only the property values that are listed in the ```DBSecurityGroups`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ property. To avoid this situation, migrate your DB instance to using VPC security groups only when that is the only change in your stack template. *Amazon Aurora* Not applicable. The associated list of EC2 VPC security groups is managed by the DB cluster. If specified, the setting must match the DB cluster setting.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__deb228abaad7110d4ddbbeaa0e2d7c7cba849e2f311811030c578da422f6f11f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBInstanceProps(
            allocated_storage=allocated_storage,
            allow_major_version_upgrade=allow_major_version_upgrade,
            associated_roles=associated_roles,
            auto_minor_version_upgrade=auto_minor_version_upgrade,
            availability_zone=availability_zone,
            backup_retention_period=backup_retention_period,
            ca_certificate_identifier=ca_certificate_identifier,
            certificate_details=certificate_details,
            certificate_rotation_restart=certificate_rotation_restart,
            character_set_name=character_set_name,
            copy_tags_to_snapshot=copy_tags_to_snapshot,
            custom_iam_instance_profile=custom_iam_instance_profile,
            db_cluster_identifier=db_cluster_identifier,
            db_cluster_snapshot_identifier=db_cluster_snapshot_identifier,
            db_instance_class=db_instance_class,
            db_instance_identifier=db_instance_identifier,
            db_name=db_name,
            db_parameter_group_name=db_parameter_group_name,
            db_security_groups=db_security_groups,
            db_snapshot_identifier=db_snapshot_identifier,
            db_subnet_group_name=db_subnet_group_name,
            delete_automated_backups=delete_automated_backups,
            deletion_protection=deletion_protection,
            domain=domain,
            domain_iam_role_name=domain_iam_role_name,
            enable_cloudwatch_logs_exports=enable_cloudwatch_logs_exports,
            enable_iam_database_authentication=enable_iam_database_authentication,
            enable_performance_insights=enable_performance_insights,
            endpoint=endpoint,
            engine=engine,
            engine_version=engine_version,
            iops=iops,
            kms_key_id=kms_key_id,
            license_model=license_model,
            manage_master_user_password=manage_master_user_password,
            master_username=master_username,
            master_user_password=master_user_password,
            master_user_secret=master_user_secret,
            max_allocated_storage=max_allocated_storage,
            monitoring_interval=monitoring_interval,
            monitoring_role_arn=monitoring_role_arn,
            multi_az=multi_az,
            nchar_character_set_name=nchar_character_set_name,
            network_type=network_type,
            option_group_name=option_group_name,
            performance_insights_kms_key_id=performance_insights_kms_key_id,
            performance_insights_retention_period=performance_insights_retention_period,
            port=port,
            preferred_backup_window=preferred_backup_window,
            preferred_maintenance_window=preferred_maintenance_window,
            processor_features=processor_features,
            promotion_tier=promotion_tier,
            publicly_accessible=publicly_accessible,
            replica_mode=replica_mode,
            restore_time=restore_time,
            source_db_instance_automated_backups_arn=source_db_instance_automated_backups_arn,
            source_db_instance_identifier=source_db_instance_identifier,
            source_dbi_resource_id=source_dbi_resource_id,
            source_region=source_region,
            storage_encrypted=storage_encrypted,
            storage_throughput=storage_throughput,
            storage_type=storage_type,
            tags=tags,
            timezone=timezone,
            use_default_processor_features=use_default_processor_features,
            use_latest_restorable_time=use_latest_restorable_time,
            vpc_security_groups=vpc_security_groups,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c3f58b8db8477c69cc738e53fbc3cfb5c861af3b3dd230ccaddfccfad17e6575)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ac23ca716922c3d96e266199353087f20e9d81aeae0f3cd4de4bfcd546ff1339)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCertificateDetailsCaIdentifier")
    def attr_certificate_details_ca_identifier(self) -> builtins.str:
        '''
        :cloudformationAttribute: CertificateDetails.CAIdentifier
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCertificateDetailsCaIdentifier"))

    @builtins.property
    @jsii.member(jsii_name="attrCertificateDetailsValidTill")
    def attr_certificate_details_valid_till(self) -> builtins.str:
        '''
        :cloudformationAttribute: CertificateDetails.ValidTill
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCertificateDetailsValidTill"))

    @builtins.property
    @jsii.member(jsii_name="attrDbInstanceArn")
    def attr_db_instance_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) for the DB instance.

        :cloudformationAttribute: DBInstanceArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbInstanceArn"))

    @builtins.property
    @jsii.member(jsii_name="attrDbiResourceId")
    def attr_dbi_resource_id(self) -> builtins.str:
        '''The AWS Region-unique, immutable identifier for the DB instance.

        This identifier is found in AWS CloudTrail log entries whenever the AWS KMS key for the DB instance is accessed.

        :cloudformationAttribute: DbiResourceId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbiResourceId"))

    @builtins.property
    @jsii.member(jsii_name="attrDbSystemId")
    def attr_db_system_id(self) -> builtins.str:
        '''The Oracle system ID (Oracle SID) for a container database (CDB).

        The Oracle SID is also the name of the CDB.

        This setting is valid for RDS Custom only.

        :cloudformationAttribute: DBSystemId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbSystemId"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpointAddress")
    def attr_endpoint_address(self) -> builtins.str:
        '''The connection endpoint for the database. For example: ``mystack-mydb-1apw1j4phylrk.cg034hpkmmjt.us-east-2.rds.amazonaws.com``.

        For Aurora Serverless DB clusters, the connection endpoint only applies to the DB cluster.

        :cloudformationAttribute: Endpoint.Address
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpointAddress"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpointHostedZoneId")
    def attr_endpoint_hosted_zone_id(self) -> builtins.str:
        '''The ID that Amazon Route 53 assigns when you create a hosted zone.

        :cloudformationAttribute: Endpoint.HostedZoneId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpointHostedZoneId"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpointPort")
    def attr_endpoint_port(self) -> builtins.str:
        '''The port number on which the database accepts connections.

        For example: ``3306``

        :cloudformationAttribute: Endpoint.Port
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpointPort"))

    @builtins.property
    @jsii.member(jsii_name="attrMasterUserSecretSecretArn")
    def attr_master_user_secret_secret_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the secret.

        :cloudformationAttribute: MasterUserSecret.SecretArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMasterUserSecretSecretArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _aws_cdk_core_f4b25747.TagManager:
        '''An optional array of key-value pairs to apply to this DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-tags
        '''
        return typing.cast(_aws_cdk_core_f4b25747.TagManager, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="allocatedStorage")
    def allocated_storage(self) -> typing.Optional[builtins.str]:
        '''The amount of storage in gibibytes (GiB) to be initially allocated for the database instance.

        .. epigraph::

           If any value is set in the ``Iops`` parameter, ``AllocatedStorage`` must be at least 100 GiB, which corresponds to the minimum Iops value of 1,000. If you increase the ``Iops`` value (in 1,000 IOPS increments), then you must also increase the ``AllocatedStorage`` value (in 100-GiB increments).

        *Amazon Aurora*

        Not applicable. Aurora cluster volumes automatically grow as the amount of data in your database increases, though you are only charged for the space that you use in an Aurora cluster volume.

        *MySQL*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 5 to 3072.

        *MariaDB*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 5 to 3072.

        *PostgreSQL*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 5 to 3072.

        *Oracle*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 10 to 3072.

        *SQL Server*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2):
        - Enterprise and Standard editions: Must be an integer from 20 to 16384.
        - Web and Express editions: Must be an integer from 20 to 16384.
        - Provisioned IOPS storage (io1):
        - Enterprise and Standard editions: Must be an integer from 20 to 16384.
        - Web and Express editions: Must be an integer from 20 to 16384.
        - Magnetic storage (standard):
        - Enterprise and Standard editions: Must be an integer from 20 to 1024.
        - Web and Express editions: Must be an integer from 20 to 1024.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-allocatedstorage
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "allocatedStorage"))

    @allocated_storage.setter
    def allocated_storage(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e04a5257c516f9a9234ddb0fc3598fd16274c796aaba972beb0f769527098446)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allocatedStorage", value)

    @builtins.property
    @jsii.member(jsii_name="allowMajorVersionUpgrade")
    def allow_major_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether major version upgrades are allowed.

        Changing this parameter doesn't result in an outage and the change is asynchronously applied as soon as possible.

        Constraints: Major version upgrades must be allowed when specifying a value for the ``EngineVersion`` parameter that is a different major version than the DB instance's current version.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-allowmajorversionupgrade
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "allowMajorVersionUpgrade"))

    @allow_major_version_upgrade.setter
    def allow_major_version_upgrade(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0d1d411ac7e3db1173f0e2ee2a30e6a4c14c85f614b4d6628cd89d6991d31b22)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allowMajorVersionUpgrade", value)

    @builtins.property
    @jsii.member(jsii_name="associatedRoles")
    def associated_roles(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.DBInstanceRoleProperty"]]]]:
        '''The AWS Identity and Access Management (IAM) roles associated with the DB instance.

        *Amazon Aurora*

        Not applicable. The associated roles are managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-associatedroles
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.DBInstanceRoleProperty"]]]], jsii.get(self, "associatedRoles"))

    @associated_roles.setter
    def associated_roles(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.DBInstanceRoleProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f0cfd8db6050535ea5865d3750d15714a6e34c33225f5508b442513c18c2cc00)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "associatedRoles", value)

    @builtins.property
    @jsii.member(jsii_name="autoMinorVersionUpgrade")
    def auto_minor_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether minor engine upgrades are applied automatically to the DB instance during the maintenance window.

        By default, minor engine upgrades are applied automatically.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-autominorversionupgrade
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "autoMinorVersionUpgrade"))

    @auto_minor_version_upgrade.setter
    def auto_minor_version_upgrade(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__10009573e724ef5a958614302225bbaf636761262c9e3a5df149a269fe4979a0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoMinorVersionUpgrade", value)

    @builtins.property
    @jsii.member(jsii_name="availabilityZone")
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''The Availability Zone (AZ) where the database will be created.

        For information on AWS Regions and Availability Zones, see `Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ .

        *Amazon Aurora*

        Each Aurora DB cluster hosts copies of its storage in three separate Availability Zones. Specify one of these Availability Zones. Aurora automatically chooses an appropriate Availability Zone if you don't specify one.

        Default: A random, system-chosen Availability Zone in the endpoint's AWS Region .

        Example: ``us-east-1d``

        Constraint: The ``AvailabilityZone`` parameter can't be specified if the DB instance is a Multi-AZ deployment. The specified Availability Zone must be in the same AWS Region as the current endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-availabilityzone
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "availabilityZone"))

    @availability_zone.setter
    def availability_zone(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b5c94619c77a239fadd916d6ae1731226d5ffa91585fda6f25d3ac52db3adff4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "availabilityZone", value)

    @builtins.property
    @jsii.member(jsii_name="backupRetentionPeriod")
    def backup_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days for which automated backups are retained.

        Setting this parameter to a positive number enables backups. Setting this parameter to 0 disables automated backups.

        *Amazon Aurora*

        Not applicable. The retention period for automated backups is managed by the DB cluster.

        Default: 1

        Constraints:

        - Must be a value from 0 to 35
        - Can't be set to 0 if the DB instance is a source to read replicas

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-backupretentionperiod
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "backupRetentionPeriod"))

    @backup_retention_period.setter
    def backup_retention_period(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d6c2ec2d2c90076d5768afa56e26b2333ed4711f1b5b4afa23ea4866d7af118)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "backupRetentionPeriod", value)

    @builtins.property
    @jsii.member(jsii_name="caCertificateIdentifier")
    def ca_certificate_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier of the CA certificate for this DB instance.

        .. epigraph::

           Specifying or updating this property triggers a reboot.

        For more information about CA certificate identifiers for RDS DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon RDS User Guide* .

        For more information about CA certificate identifiers for Aurora DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon Aurora User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-cacertificateidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "caCertificateIdentifier"))

    @ca_certificate_identifier.setter
    def ca_certificate_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c7f3fb22638f1c047bf49963175961242627be42435c151d15bf47bee2699599)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "caCertificateIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="certificateDetails")
    def certificate_details(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.CertificateDetailsProperty"]]:
        '''``AWS::RDS::DBInstance.CertificateDetails``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-certificatedetails
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.CertificateDetailsProperty"]], jsii.get(self, "certificateDetails"))

    @certificate_details.setter
    def certificate_details(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.CertificateDetailsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__19d3bfe6a9f1925210e08b7293d31af5f60677ef3b8e2ce988f4f445dbb9539f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "certificateDetails", value)

    @builtins.property
    @jsii.member(jsii_name="certificateRotationRestart")
    def certificate_rotation_restart(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''``AWS::RDS::DBInstance.CertificateRotationRestart``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-certificaterotationrestart
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "certificateRotationRestart"))

    @certificate_rotation_restart.setter
    def certificate_rotation_restart(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__86b23f096ea426fe8c445da0c58a8fd83a6816a2e002e32c794c4b84479d9d46)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "certificateRotationRestart", value)

    @builtins.property
    @jsii.member(jsii_name="characterSetName")
    def character_set_name(self) -> typing.Optional[builtins.str]:
        '''For supported engines, indicates that the DB instance should be associated with the specified character set.

        *Amazon Aurora*

        Not applicable. The character set is managed by the DB cluster. For more information, see `AWS::RDS::DBCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-charactersetname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "characterSetName"))

    @character_set_name.setter
    def character_set_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fedcbc34f2b709efa40f9bff0c1efa684ac452db356507f9a8b370b6bc7c2eae)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "characterSetName", value)

    @builtins.property
    @jsii.member(jsii_name="copyTagsToSnapshot")
    def copy_tags_to_snapshot(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to copy tags from the DB instance to snapshots of the DB instance.

        By default, tags are not copied.

        *Amazon Aurora*

        Not applicable. Copying tags to snapshots is managed by the DB cluster. Setting this value for an Aurora DB instance has no effect on the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-copytagstosnapshot
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "copyTagsToSnapshot"))

    @copy_tags_to_snapshot.setter
    def copy_tags_to_snapshot(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b493562b8c02a3a0cff243ee4772c73c8267a709e3e1276cf600d3b8f3eb8ac1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "copyTagsToSnapshot", value)

    @builtins.property
    @jsii.member(jsii_name="customIamInstanceProfile")
    def custom_iam_instance_profile(self) -> typing.Optional[builtins.str]:
        '''The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.

        The instance profile must meet the following requirements:

        - The profile must exist in your account.
        - The profile must have an IAM role that Amazon EC2 has permissions to assume.
        - The instance profile name and the associated IAM role name must start with the prefix ``AWSRDSCustom`` .

        For the list of permissions required for the IAM role, see `Configure IAM and your VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-setup-orcl.html#custom-setup-orcl.iam-vpc>`_ in the *Amazon RDS User Guide* .

        This setting is required for RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-customiaminstanceprofile
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "customIamInstanceProfile"))

    @custom_iam_instance_profile.setter
    def custom_iam_instance_profile(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d34c7407000f331063e51d4520aab291313e3a4220720a2824311ad077d442f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "customIamInstanceProfile", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterIdentifier")
    def db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier of the DB cluster that the instance will belong to.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbClusterIdentifier"))

    @db_cluster_identifier.setter
    def db_cluster_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6701249d56fc8f0cffd198cc16a26d8063fd82b390782f65413992ffd8577893)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterSnapshotIdentifier")
    def db_cluster_snapshot_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier for the RDS for MySQL Multi-AZ DB cluster snapshot to restore from.

        For more information on Multi-AZ DB clusters, see `Multi-AZ deployments with two readable standby DB instances <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/multi-az-db-clusters-concepts.html>`_ in the *Amazon RDS User Guide* .

        Constraints:

        - Must match the identifier of an existing Multi-AZ DB cluster snapshot.
        - Can't be specified when ``DBSnapshotIdentifier`` is specified.
        - Must be specified when ``DBSnapshotIdentifier`` isn't specified.
        - If you are restoring from a shared manual Multi-AZ DB cluster snapshot, the ``DBClusterSnapshotIdentifier`` must be the ARN of the shared snapshot.
        - Can't be the identifier of an Aurora DB cluster snapshot.
        - Can't be the identifier of an RDS for PostgreSQL Multi-AZ DB cluster snapshot.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbclustersnapshotidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbClusterSnapshotIdentifier"))

    @db_cluster_snapshot_identifier.setter
    def db_cluster_snapshot_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ed3347a63e08b61db31e04e14ea6e861fa8915951c50a6bc4d52e3504b90d653)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterSnapshotIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="dbInstanceClass")
    def db_instance_class(self) -> typing.Optional[builtins.str]:
        '''The compute and memory capacity of the DB instance, for example, ``db.m4.large`` . Not all DB instance classes are available in all AWS Regions, or for all database engines.

        For the full list of DB instance classes, and availability for your engine, see `DB Instance Class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide.* For more information about DB instance class pricing and AWS Region support for DB instance classes, see `Amazon RDS Pricing <https://docs.aws.amazon.com/rds/pricing/>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbinstanceclass
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbInstanceClass"))

    @db_instance_class.setter
    def db_instance_class(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__75832377aa5307fe0c70245e64ba5eb5b197361687513549e3a913bc776a6314)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbInstanceClass", value)

    @builtins.property
    @jsii.member(jsii_name="dbInstanceIdentifier")
    def db_instance_identifier(self) -> typing.Optional[builtins.str]:
        '''A name for the DB instance.

        If you specify a name, AWS CloudFormation converts it to lowercase. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the DB instance. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ .

        For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* .
        .. epigraph::

           If you specify a name, you can't perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbinstanceidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbInstanceIdentifier"))

    @db_instance_identifier.setter
    def db_instance_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7ca12a8c3e1ae24ab9c69b14f4737d16387ce4d6f5d828b206c2cb6f600974c3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbInstanceIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="dbName")
    def db_name(self) -> typing.Optional[builtins.str]:
        '''The meaning of this parameter differs according to the database engine you use.

        .. epigraph::

           If you specify the ``[DBSnapshotIdentifier](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsnapshotidentifier)`` property, this property only applies to RDS for Oracle.

        *Amazon Aurora*

        Not applicable. The database name is managed by the DB cluster.

        *MySQL*

        The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance.

        Constraints:

        - Must contain 1 to 64 letters or numbers.
        - Can't be a word reserved by the specified database engine

        *MariaDB*

        The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance.

        Constraints:

        - Must contain 1 to 64 letters or numbers.
        - Can't be a word reserved by the specified database engine

        *PostgreSQL*

        The name of the database to create when the DB instance is created. If this parameter is not specified, the default ``postgres`` database is created in the DB instance.

        Constraints:

        - Must begin with a letter. Subsequent characters can be letters, underscores, or digits (0-9).
        - Must contain 1 to 63 characters.
        - Can't be a word reserved by the specified database engine

        *Oracle*

        The Oracle System ID (SID) of the created DB instance. If you specify ``null`` , the default value ``ORCL`` is used. You can't specify the string NULL, or any other reserved word, for ``DBName`` .

        Default: ``ORCL``

        Constraints:

        - Can't be longer than 8 characters

        *SQL Server*

        Not applicable. Must be null.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbName"))

    @db_name.setter
    def db_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b713cfc85cd02b5627311407e053d7d678c1db3c92306ea826dd17baa72c4ff8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbName", value)

    @builtins.property
    @jsii.member(jsii_name="dbParameterGroupName")
    def db_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of an existing DB parameter group or a reference to an `AWS::RDS::DBParameterGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html>`_ resource created in the template.

        To list all of the available DB parameter group names, use the following command:

        ``aws rds describe-db-parameter-groups --query "DBParameterGroups[].DBParameterGroupName" --output text``
        .. epigraph::

           If any of the data members of the referenced parameter group are changed during an update, the DB instance might need to be restarted, which causes some interruption. If the parameter group contains static parameters, whether they were changed or not, an update triggers a reboot.

        If you don't specify a value for ``DBParameterGroupName`` property, the default DB parameter group for the specified engine and engine version is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbparametergroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbParameterGroupName"))

    @db_parameter_group_name.setter
    def db_parameter_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fb68890ac01533acb928f8d23ad18c618fdfb600e943bcf2cb1cde97b07643dc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbParameterGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="dbSecurityGroups")
    def db_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of the DB security groups to assign to the DB instance.

        The list can include both the name of existing DB security groups or references to AWS::RDS::DBSecurityGroup resources created in the template.

        If you set DBSecurityGroups, you must not set VPCSecurityGroups, and vice versa. Also, note that the DBSecurityGroups property exists only for backwards compatibility with older regions and is no longer recommended for providing security information to an RDS DB instance. Instead, use VPCSecurityGroups.
        .. epigraph::

           If you specify this property, AWS CloudFormation sends only the following properties (if specified) to Amazon RDS during create operations:

           - ``AllocatedStorage``
           - ``AutoMinorVersionUpgrade``
           - ``AvailabilityZone``
           - ``BackupRetentionPeriod``
           - ``CharacterSetName``
           - ``DBInstanceClass``
           - ``DBName``
           - ``DBParameterGroupName``
           - ``DBSecurityGroups``
           - ``DBSubnetGroupName``
           - ``Engine``
           - ``EngineVersion``
           - ``Iops``
           - ``LicenseModel``
           - ``MasterUsername``
           - ``MasterUserPassword``
           - ``MultiAZ``
           - ``OptionGroupName``
           - ``PreferredBackupWindow``
           - ``PreferredMaintenanceWindow``

           All other properties are ignored. Specify a virtual private cloud (VPC) security group if you want to submit other properties, such as ``StorageType`` , ``StorageEncrypted`` , or ``KmsKeyId`` . If you're already using the ``DBSecurityGroups`` property, you can't use these other properties by updating your DB instance to use a VPC security group. You must recreate the DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbsecuritygroups
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "dbSecurityGroups"))

    @db_security_groups.setter
    def db_security_groups(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4fb75c933677306466e1b5453c9e91d243cac6aa08045adf6cd4bd8a80db5e79)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSecurityGroups", value)

    @builtins.property
    @jsii.member(jsii_name="dbSnapshotIdentifier")
    def db_snapshot_identifier(self) -> typing.Optional[builtins.str]:
        '''The name or Amazon Resource Name (ARN) of the DB snapshot that's used to restore the DB instance.

        If you're restoring from a shared manual DB snapshot, you must specify the ARN of the snapshot.

        By specifying this property, you can create a DB instance from the specified DB snapshot. If the ``DBSnapshotIdentifier`` property is an empty string or the ``AWS::RDS::DBInstance`` declaration has no ``DBSnapshotIdentifier`` property, AWS CloudFormation creates a new database. If the property contains a value (other than an empty string), AWS CloudFormation creates a database from the specified snapshot. If a snapshot with the specified name doesn't exist, AWS CloudFormation can't create the database and it rolls back the stack.

        Some DB instance properties aren't valid when you restore from a snapshot, such as the ``MasterUsername`` and ``MasterUserPassword`` properties. For information about the properties that you can specify, see the ``RestoreDBInstanceFromDBSnapshot`` action in the *Amazon RDS API Reference* .

        After you restore a DB instance with a ``DBSnapshotIdentifier`` property, you must specify the same ``DBSnapshotIdentifier`` property for any future updates to the DB instance. When you specify this property for an update, the DB instance is not restored from the DB snapshot again, and the data in the database is not changed. However, if you don't specify the ``DBSnapshotIdentifier`` property, an empty DB instance is created, and the original DB instance is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB instance is restored from the specified ``DBSnapshotIdentifier`` property, and the original DB instance is deleted.

        If you specify the ``DBSnapshotIdentifier`` property to restore a DB instance (as opposed to specifying it for DB instance updates), then don't specify the following properties:

        - ``CharacterSetName``
        - ``DBClusterIdentifier``
        - ``DBName``
        - ``DeleteAutomatedBackups``
        - ``EnablePerformanceInsights``
        - ``KmsKeyId``
        - ``MasterUsername``
        - ``MasterUserPassword``
        - ``PerformanceInsightsKMSKeyId``
        - ``PerformanceInsightsRetentionPeriod``
        - ``PromotionTier``
        - ``SourceDBInstanceIdentifier``
        - ``SourceRegion``
        - ``StorageEncrypted`` (for an encrypted snapshot)
        - ``Timezone``

        *Amazon Aurora*

        Not applicable. Snapshot restore is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbsnapshotidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbSnapshotIdentifier"))

    @db_snapshot_identifier.setter
    def db_snapshot_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e81e293ac3ccd7c752bf231337cd041e25bd8fdf05d0cf4b515c45943beee853)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSnapshotIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="dbSubnetGroupName")
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        '''A DB subnet group to associate with the DB instance.

        If you update this value, the new subnet group must be a subnet group in a new VPC.

        If there's no DB subnet group, then the DB instance isn't a VPC DB instance.

        For more information about using Amazon RDS in a VPC, see `Using Amazon RDS with Amazon Virtual Private Cloud (VPC) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. The DB subnet group is managed by the DB cluster. If specified, the setting must match the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbsubnetgroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbSubnetGroupName"))

    @db_subnet_group_name.setter
    def db_subnet_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8ad6cfb9c7b306c61b8411bad7268621f84a597be5c342e39692a888327c1e08)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSubnetGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="deleteAutomatedBackups")
    def delete_automated_backups(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to remove automated backups immediately after the DB instance is deleted.

        This parameter isn't case-sensitive. The default is to remove automated backups immediately after the DB instance is deleted.

        *Amazon Aurora*

        Not applicable. When you delete a DB cluster, all automated backups for that DB cluster are deleted and can't be recovered. Manual DB cluster snapshots of the DB cluster are not deleted.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-deleteautomatedbackups
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "deleteAutomatedBackups"))

    @delete_automated_backups.setter
    def delete_automated_backups(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d6410a1639e965aa72ad3f82ff25fd4f1c6a7bbf96da803392b9ca3ea8ee49d8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deleteAutomatedBackups", value)

    @builtins.property
    @jsii.member(jsii_name="deletionProtection")
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether the DB instance has deletion protection enabled.

        The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled. For more information, see `Deleting a DB Instance <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_DeleteInstance.html>`_ .

        *Amazon Aurora*

        Not applicable. You can enable or disable deletion protection for the DB cluster. For more information, see ``CreateDBCluster`` . DB instances in a DB cluster can be deleted even when deletion protection is enabled for the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-deletionprotection
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "deletionProtection"))

    @deletion_protection.setter
    def deletion_protection(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__54888fafddcd7702544a29c557433e1615cb2a50574e813b32bd2695858bb723)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deletionProtection", value)

    @builtins.property
    @jsii.member(jsii_name="domain")
    def domain(self) -> typing.Optional[builtins.str]:
        '''The Active Directory directory ID to create the DB instance in.

        Currently, only Microsoft SQL Server, Oracle, and PostgreSQL DB instances can be created in an Active Directory Domain.

        For more information, see `Kerberos Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html>`_ in the *Amazon RDS User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-domain
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domain"))

    @domain.setter
    def domain(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__efa5e7fb20d653e13fb3d2f2f656b47fe376d0af6c980791108e0ae633ee8d63)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domain", value)

    @builtins.property
    @jsii.member(jsii_name="domainIamRoleName")
    def domain_iam_role_name(self) -> typing.Optional[builtins.str]:
        '''Specify the name of the IAM role to be used when making API calls to the Directory Service.

        This setting doesn't apply to RDS Custom.

        *Amazon Aurora*

        Not applicable. The domain is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-domainiamrolename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domainIamRoleName"))

    @domain_iam_role_name.setter
    def domain_iam_role_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ed59ea9c16b268e02f5c5ea70b70bc9e46011ea97eae73bcd9739c651860777c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainIamRoleName", value)

    @builtins.property
    @jsii.member(jsii_name="enableCloudwatchLogsExports")
    def enable_cloudwatch_logs_exports(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of log types that need to be enabled for exporting to CloudWatch Logs.

        The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Relational Database Service User Guide* .

        *Amazon Aurora*

        Not applicable. CloudWatch Logs exports are managed by the DB cluster.

        *MariaDB*

        Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery``

        *Microsoft SQL Server*

        Valid values: ``agent`` , ``error``

        *MySQL*

        Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery``

        *Oracle*

        Valid values: ``alert`` , ``audit`` , ``listener`` , ``trace``

        *PostgreSQL*

        Valid values: ``postgresql`` , ``upgrade``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-enablecloudwatchlogsexports
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "enableCloudwatchLogsExports"))

    @enable_cloudwatch_logs_exports.setter
    def enable_cloudwatch_logs_exports(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__06f7fbebed4703139bd0a5d6019b6724bbd2be7dd1cf002d75d59458255f1e44)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableCloudwatchLogsExports", value)

    @builtins.property
    @jsii.member(jsii_name="enableIamDatabaseAuthentication")
    def enable_iam_database_authentication(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.

        By default, mapping is disabled.

        This property is supported for RDS for MariaDB, RDS for MySQL, and RDS for PostgreSQL. For more information, see `IAM Database Authentication for MariaDB, MySQL, and PostgreSQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon RDS User Guide.*

        *Amazon Aurora*

        Not applicable. Mapping AWS IAM accounts to database accounts is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-enableiamdatabaseauthentication
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "enableIamDatabaseAuthentication"))

    @enable_iam_database_authentication.setter
    def enable_iam_database_authentication(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__35a748593f5813bbf7a5d71bdf814330397969f0bc8b73a267904acb06dc94c5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableIamDatabaseAuthentication", value)

    @builtins.property
    @jsii.member(jsii_name="enablePerformanceInsights")
    def enable_performance_insights(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to enable Performance Insights for the DB instance.

        For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* .

        This setting doesn't apply to RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-enableperformanceinsights
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "enablePerformanceInsights"))

    @enable_performance_insights.setter
    def enable_performance_insights(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__710581d3b0f8ea18ba11d7cc7b9ca07b97488c5586a01718aa3c34a99b7f3115)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enablePerformanceInsights", value)

    @builtins.property
    @jsii.member(jsii_name="endpoint")
    def endpoint(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.EndpointProperty"]]:
        '''Specifies the connection endpoint.

        .. epigraph::

           The endpoint might not be shown for instances whose status is ``creating`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-endpoint
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.EndpointProperty"]], jsii.get(self, "endpoint"))

    @endpoint.setter
    def endpoint(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.EndpointProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3ca3ac29f593716aebc876e32c93d69d380ad4d068905f0bbeeccba6315df9f0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "endpoint", value)

    @builtins.property
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional[builtins.str]:
        '''The name of the database engine that you want to use for this DB instance.

        .. epigraph::

           When you are creating a DB instance, the ``Engine`` property is required.

        Valid Values:

        - ``aurora`` (for MySQL 5.6-compatible Aurora)
        - ``aurora-mysql`` (for MySQL 5.7-compatible Aurora)
        - ``aurora-postgresql``
        - ``mariadb``
        - ``mysql``
        - ``oracle-ee``
        - ``oracle-ee-cdb``
        - ``oracle-se2``
        - ``oracle-se2-cdb``
        - ``postgres``
        - ``sqlserver-ee``
        - ``sqlserver-se``
        - ``sqlserver-ex``
        - ``sqlserver-web``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-engine
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engine"))

    @engine.setter
    def engine(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__56b516830b55e762da8cd4477ac7ce54d492461f2e287811a0961ae45f858ab2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engine", value)

    @builtins.property
    @jsii.member(jsii_name="engineVersion")
    def engine_version(self) -> typing.Optional[builtins.str]:
        '''The version number of the database engine to use.

        For a list of valid engine versions, use the ``DescribeDBEngineVersions`` action.

        The following are the database engines and links to information about the major and minor versions that are available with Amazon RDS. Not every database engine is available for every AWS Region.

        *Amazon Aurora*

        Not applicable. The version number of the database engine to be used by the DB instance is managed by the DB cluster.

        *MariaDB*

        See `MariaDB on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MariaDB.html#MariaDB.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.*

        *Microsoft SQL Server*

        See `Microsoft SQL Server Versions on Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.VersionSupport>`_ in the *Amazon RDS User Guide.*

        *MySQL*

        See `MySQL on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MySQL.html#MySQL.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.*

        *Oracle*

        See `Oracle Database Engine Release Notes <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.Oracle.PatchComposition.html>`_ in the *Amazon RDS User Guide.*

        *PostgreSQL*

        See `Supported PostgreSQL Database Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_PostgreSQL.html#PostgreSQL.Concepts.General.DBVersions>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-engineversion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engineVersion"))

    @engine_version.setter
    def engine_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__67592a86152f510f7bee2d60ee9d0b2a97139c3ad1de17ddb01351b3f62bedcf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engineVersion", value)

    @builtins.property
    @jsii.member(jsii_name="iops")
    def iops(self) -> typing.Optional[jsii.Number]:
        '''The number of I/O operations per second (IOPS) that the database provisions.

        The value must be equal to or greater than 1000.

        If you specify this property, you must follow the range of allowed ratios of your requested IOPS rate to the amount of storage that you allocate (IOPS to allocated storage). For example, you can provision an Oracle database instance with 1000 IOPS and 200 GiB of storage (a ratio of 5:1), or specify 2000 IOPS with 200 GiB of storage (a ratio of 10:1). For more information, see `Amazon RDS Provisioned IOPS Storage to Improve Performance <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* .
        .. epigraph::

           If you specify ``io1`` for the ``StorageType`` property, then you must also specify the ``Iops`` property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-iops
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "iops"))

    @iops.setter
    def iops(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7ced1ccffc7461224f2a1ebd84c50902cbbf8d0dc67bb42aee7a5c08e73a5683)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "iops", value)

    @builtins.property
    @jsii.member(jsii_name="kmsKeyId")
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The ARN of the AWS KMS key that's used to encrypt the DB instance, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` .

        If you enable the StorageEncrypted property but don't specify this property, AWS CloudFormation uses the default KMS key. If you specify this property, you must set the StorageEncrypted property to true.

        If you specify the ``SourceDBInstanceIdentifier`` property, the value is inherited from the source DB instance if the read replica is created in the same region.

        If you create an encrypted read replica in a different AWS Region, then you must specify a KMS key for the destination AWS Region. KMS encryption keys are specific to the region that they're created in, and you can't use encryption keys from one region in another region.

        If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used.

        If you specify ``DBSecurityGroups`` , AWS CloudFormation ignores this property. To specify both a security group and this property, you must use a VPC security group. For more information about Amazon RDS and VPC, see `Using Amazon RDS with Amazon VPC <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. The KMS key identifier is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-kmskeyid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kmsKeyId"))

    @kms_key_id.setter
    def kms_key_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ecf324a93d021763a14d5f0045ced2caa4f25cb986370f9c0d13524e9159ede8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kmsKeyId", value)

    @builtins.property
    @jsii.member(jsii_name="licenseModel")
    def license_model(self) -> typing.Optional[builtins.str]:
        '''License model information for this DB instance.

        Valid values:

        - Aurora MySQL - ``general-public-license``
        - Aurora PostgreSQL - ``postgresql-license``
        - MariaDB - ``general-public-license``
        - Microsoft SQL Server - ``license-included``
        - MySQL - ``general-public-license``
        - Oracle - ``bring-your-own-license`` or ``license-included``
        - PostgreSQL - ``postgresql-license``

        .. epigraph::

           If you've specified ``DBSecurityGroups`` and then you update the license model, AWS CloudFormation replaces the underlying DB instance. This will incur some interruptions to database availability.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-licensemodel
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "licenseModel"))

    @license_model.setter
    def license_model(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3886eaaa0bfb686e0318377e35c8d231d0ab222b8eda2d158ee05472354c2634)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "licenseModel", value)

    @builtins.property
    @jsii.member(jsii_name="manageMasterUserPassword")
    def manage_master_user_password(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to manage the master user password with AWS Secrets Manager.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.*

        Constraints:

        - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-managemasteruserpassword
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "manageMasterUserPassword"))

    @manage_master_user_password.setter
    def manage_master_user_password(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7755fb94d52f5f0542d31d72b07773b61d2bfca7d2424fab2d68f26bb7ca6835)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "manageMasterUserPassword", value)

    @builtins.property
    @jsii.member(jsii_name="masterUsername")
    def master_username(self) -> typing.Optional[builtins.str]:
        '''The master user name for the DB instance.

        .. epigraph::

           If you specify the ``SourceDBInstanceIdentifier`` or ``DBSnapshotIdentifier`` property, don't specify this property. The value is inherited from the source DB instance or snapshot.

        *Amazon Aurora*

        Not applicable. The name for the master user is managed by the DB cluster.

        *MariaDB*

        Constraints:

        - Required for MariaDB.
        - Must be 1 to 16 letters or numbers.
        - Can't be a reserved word for the chosen database engine.

        *Microsoft SQL Server*

        Constraints:

        - Required for SQL Server.
        - Must be 1 to 128 letters or numbers.
        - The first character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        *MySQL*

        Constraints:

        - Required for MySQL.
        - Must be 1 to 16 letters or numbers.
        - First character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        *Oracle*

        Constraints:

        - Required for Oracle.
        - Must be 1 to 30 letters or numbers.
        - First character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        *PostgreSQL*

        Constraints:

        - Required for PostgreSQL.
        - Must be 1 to 63 letters or numbers.
        - First character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-masterusername
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "masterUsername"))

    @master_username.setter
    def master_username(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8b73839629418600a7da8e0a136cca4f9e654bc4a99120f7744f5c17cbbdacc7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "masterUsername", value)

    @builtins.property
    @jsii.member(jsii_name="masterUserPassword")
    def master_user_password(self) -> typing.Optional[builtins.str]:
        '''The password for the master user. The password can include any printable ASCII character except "/", """, or "@".

        *Amazon Aurora*

        Not applicable. The password for the master user is managed by the DB cluster.

        *MariaDB*

        Constraints: Must contain from 8 to 41 characters.

        *Microsoft SQL Server*

        Constraints: Must contain from 8 to 128 characters.

        *MySQL*

        Constraints: Must contain from 8 to 41 characters.

        *Oracle*

        Constraints: Must contain from 8 to 30 characters.

        *PostgreSQL*

        Constraints: Must contain from 8 to 128 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-masteruserpassword
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "masterUserPassword"))

    @master_user_password.setter
    def master_user_password(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__284b2c703d35873e828a8953e01f1d689eec5dc3ee8db34345ea678713542e3e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "masterUserPassword", value)

    @builtins.property
    @jsii.member(jsii_name="masterUserSecret")
    def master_user_secret(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.MasterUserSecretProperty"]]:
        '''Contains the secret managed by RDS in AWS Secrets Manager for the master user password.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-masterusersecret
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.MasterUserSecretProperty"]], jsii.get(self, "masterUserSecret"))

    @master_user_secret.setter
    def master_user_secret(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.MasterUserSecretProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__adb1db2b6df1c7b1e6216f786b9ffd44c59136a2ce02419738f6df65f14b6c24)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "masterUserSecret", value)

    @builtins.property
    @jsii.member(jsii_name="maxAllocatedStorage")
    def max_allocated_storage(self) -> typing.Optional[jsii.Number]:
        '''The upper limit in gibibytes (GiB) to which Amazon RDS can automatically scale the storage of the DB instance.

        For more information about this setting, including limitations that apply to it, see `Managing capacity automatically with Amazon RDS storage autoscaling <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling>`_ in the *Amazon RDS User Guide* .

        This setting doesn't apply to RDS Custom.

        *Amazon Aurora*

        Not applicable. Storage is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-maxallocatedstorage
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maxAllocatedStorage"))

    @max_allocated_storage.setter
    def max_allocated_storage(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__48cc30822ba46220f2e0bf8bc4a8a94c856bd7941c3476b8a0b4474c5188122d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maxAllocatedStorage", value)

    @builtins.property
    @jsii.member(jsii_name="monitoringInterval")
    def monitoring_interval(self) -> typing.Optional[jsii.Number]:
        '''The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB instance.

        To disable collection of Enhanced Monitoring metrics, specify 0. The default is 0.

        If ``MonitoringRoleArn`` is specified, then you must set ``MonitoringInterval`` to a value other than 0.

        This setting doesn't apply to RDS Custom.

        Valid Values: ``0, 1, 5, 10, 15, 30, 60``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-monitoringinterval
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "monitoringInterval"))

    @monitoring_interval.setter
    def monitoring_interval(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__314cde0e7d19a0895a5de1444a76e5da56e3b9efc38bd9a48f09eb9fd3494fb3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "monitoringInterval", value)

    @builtins.property
    @jsii.member(jsii_name="monitoringRoleArn")
    def monitoring_role_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN for the IAM role that permits RDS to send enhanced monitoring metrics to Amazon CloudWatch Logs.

        For example, ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting Up and Enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* .

        If ``MonitoringInterval`` is set to a value other than 0, then you must supply a ``MonitoringRoleArn`` value.

        This setting doesn't apply to RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-monitoringrolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "monitoringRoleArn"))

    @monitoring_role_arn.setter
    def monitoring_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c02d82d6607d570d7ca8de7872eb3f703b19600d6b3d3fc365dc7f79444c0ff1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "monitoringRoleArn", value)

    @builtins.property
    @jsii.member(jsii_name="multiAz")
    def multi_az(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''Specifies whether the database instance is a Multi-AZ DB instance deployment.

        You can't set the ``AvailabilityZone`` parameter if the ``MultiAZ`` parameter is set to true.

        Currently, you can't use AWS CloudFormation to create a Multi-AZ DB cluster deployment.

        For more information, see `Multi-AZ deployments for high availability <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.MultiAZ.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. Amazon Aurora storage is replicated across all of the Availability Zones and doesn't require the ``MultiAZ`` option to be set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-multiaz
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "multiAz"))

    @multi_az.setter
    def multi_az(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dd8c513de6ce2718d4a9e76136c040737714b288e7bfe92ba469dbad02530371)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "multiAz", value)

    @builtins.property
    @jsii.member(jsii_name="ncharCharacterSetName")
    def nchar_character_set_name(self) -> typing.Optional[builtins.str]:
        '''The name of the NCHAR character set for the Oracle DB instance.

        This parameter doesn't apply to RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-ncharcharactersetname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ncharCharacterSetName"))

    @nchar_character_set_name.setter
    def nchar_character_set_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5146bbc549ffe260914032bb36be7d422290062cd86065800291c471e4718080)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ncharCharacterSetName", value)

    @builtins.property
    @jsii.member(jsii_name="networkType")
    def network_type(self) -> typing.Optional[builtins.str]:
        '''The network type of the DB instance.

        Valid values:

        - ``IPV4``
        - ``DUAL``

        The network type is determined by the ``DBSubnetGroup`` specified for the DB instance. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ).

        For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-networktype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "networkType"))

    @network_type.setter
    def network_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4f547361024af3fd51c234b700ea1ccd35a342a7947e73560f6bb4b793a107d2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "networkType", value)

    @builtins.property
    @jsii.member(jsii_name="optionGroupName")
    def option_group_name(self) -> typing.Optional[builtins.str]:
        '''Indicates that the DB instance should be associated with the specified option group.

        Permanent options, such as the TDE option for Oracle Advanced Security TDE, can't be removed from an option group. Also, that option group can't be removed from a DB instance once it is associated with a DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-optiongroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "optionGroupName"))

    @option_group_name.setter
    def option_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__47c0c171676de61ed26c19dd1d218a116a583bd095318981de3524585adbb31e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "optionGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="performanceInsightsKmsKeyId")
    def performance_insights_kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key identifier for encryption of Performance Insights data.

        The KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.

        If you do not specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account. Your AWS account has a different default KMS key for each AWS Region.

        For information about enabling Performance Insights, see `EnablePerformanceInsights <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-performanceinsightskmskeyid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "performanceInsightsKmsKeyId"))

    @performance_insights_kms_key_id.setter
    def performance_insights_kms_key_id(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__754c4d6b2cccd75781d39bbcdf1b039cd2a1db06d1211efe133f85593833649e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "performanceInsightsKmsKeyId", value)

    @builtins.property
    @jsii.member(jsii_name="performanceInsightsRetentionPeriod")
    def performance_insights_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The amount of time, in days, to retain Performance Insights data. Valid values are 7 or 731 (2 years).

        For information about enabling Performance Insights, see `EnablePerformanceInsights <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-performanceinsightsretentionperiod
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "performanceInsightsRetentionPeriod"))

    @performance_insights_retention_period.setter
    def performance_insights_retention_period(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b7666b6ca978e82a85020a42ed57c9157b67a50fff45bdd35b95f30ce6b2b1cb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "performanceInsightsRetentionPeriod", value)

    @builtins.property
    @jsii.member(jsii_name="port")
    def port(self) -> typing.Optional[builtins.str]:
        '''The port number on which the database accepts connections.

        *Amazon Aurora*

        Not applicable. The port number is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-port
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "port"))

    @port.setter
    def port(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5f589b78681c747a8159a959d8bbb6e081c157067438f2e637709bdf31d0b26f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "port", value)

    @builtins.property
    @jsii.member(jsii_name="preferredBackupWindow")
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        '''The daily time range during which automated backups are created if automated backups are enabled, using the ``BackupRetentionPeriod`` parameter.

        For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow>`_ in the *Amazon RDS User Guide.*

        Constraints:

        - Must be in the format ``hh24:mi-hh24:mi`` .
        - Must be in Universal Coordinated Time (UTC).
        - Must not conflict with the preferred maintenance window.
        - Must be at least 30 minutes.

        *Amazon Aurora*

        Not applicable. The daily time range for creating automated backups is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-preferredbackupwindow
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "preferredBackupWindow"))

    @preferred_backup_window.setter
    def preferred_backup_window(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__506c9b400dd1dd8eeab9f2b251aaeb690e1d2f0ab48e5384e2081c18bbd0dfaf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "preferredBackupWindow", value)

    @builtins.property
    @jsii.member(jsii_name="preferredMaintenanceWindow")
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        '''The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).

        Format: ``ddd:hh24:mi-ddd:hh24:mi``

        The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Instance Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow>`_ in the *Amazon RDS User Guide.*
        .. epigraph::

           This property applies when AWS CloudFormation initially creates the DB instance. If you use AWS CloudFormation to update the DB instance, those updates are applied immediately.

        Constraints: Minimum 30-minute window.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-preferredmaintenancewindow
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "preferredMaintenanceWindow"))

    @preferred_maintenance_window.setter
    def preferred_maintenance_window(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__29fca9b55572d6d4dabdc57054c1da92d440fecaba4a6c6993ddd3671644a7e9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "preferredMaintenanceWindow", value)

    @builtins.property
    @jsii.member(jsii_name="processorFeatures")
    def processor_features(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.ProcessorFeatureProperty"]]]]:
        '''The number of CPU cores and the number of threads per core for the DB instance class of the DB instance.

        This setting doesn't apply to RDS Custom.

        *Amazon Aurora*

        Not applicable.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-processorfeatures
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.ProcessorFeatureProperty"]]]], jsii.get(self, "processorFeatures"))

    @processor_features.setter
    def processor_features(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBInstance.ProcessorFeatureProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9cd537d01eb7af5145d303f1a228158fedf4f7719c7b29778b6576f814ee9349)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "processorFeatures", value)

    @builtins.property
    @jsii.member(jsii_name="promotionTier")
    def promotion_tier(self) -> typing.Optional[jsii.Number]:
        '''A value that specifies the order in which an Aurora Replica is promoted to the primary instance after a failure of the existing primary instance.

        For more information, see `Fault Tolerance for an Aurora DB Cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html#Aurora.Managing.FaultTolerance>`_ in the *Amazon Aurora User Guide* .

        This setting doesn't apply to RDS Custom.

        Default: 1

        Valid Values: 0 - 15

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-promotiontier
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "promotionTier"))

    @promotion_tier.setter
    def promotion_tier(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__56533903d1f960ce2e3f0bda133d5da4bddf5d9978af37f13bf0af27536eb41c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "promotionTier", value)

    @builtins.property
    @jsii.member(jsii_name="publiclyAccessible")
    def publicly_accessible(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''Indicates whether the DB instance is an internet-facing instance.

        If you specify true, AWS CloudFormation creates an instance with a publicly resolvable DNS name, which resolves to a public IP address. If you specify false, AWS CloudFormation creates an internal instance with a DNS name that resolves to a private IP address.

        The default behavior value depends on your VPC setup and the database subnet group. For more information, see the ``PubliclyAccessible`` parameter in the `CreateDBInstance <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html>`_ in the *Amazon RDS API Reference* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-publiclyaccessible
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "publiclyAccessible"))

    @publicly_accessible.setter
    def publicly_accessible(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fadc633d5b1faaed0977c6bf35c181698ca79a5dca827425cdc38fd4996680a8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "publiclyAccessible", value)

    @builtins.property
    @jsii.member(jsii_name="replicaMode")
    def replica_mode(self) -> typing.Optional[builtins.str]:
        '''The open mode of an Oracle read replica.

        For more information, see `Working with Oracle Read Replicas for Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html>`_ in the *Amazon RDS User Guide* .

        This setting is only supported in RDS for Oracle.

        Default: ``open-read-only``

        Valid Values: ``open-read-only`` or ``mounted``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-replicamode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "replicaMode"))

    @replica_mode.setter
    def replica_mode(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f836135e7d1b66d40a6e8d1782dedb9d5f2deb496edcba78d6903ddaa8e78151)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "replicaMode", value)

    @builtins.property
    @jsii.member(jsii_name="restoreTime")
    def restore_time(self) -> typing.Optional[builtins.str]:
        '''The date and time to restore from.

        Valid Values: Value must be a time in Universal Coordinated Time (UTC) format

        Constraints:

        - Must be before the latest restorable time for the DB instance
        - Can't be specified if the ``UseLatestRestorableTime`` parameter is enabled

        Example: ``2009-09-07T23:45:00Z``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-restoretime
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "restoreTime"))

    @restore_time.setter
    def restore_time(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a6f89ee70efd90bfefd2361f1649d00696d668042e4b6128e57ca1055b4faa2b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restoreTime", value)

    @builtins.property
    @jsii.member(jsii_name="sourceDbInstanceAutomatedBackupsArn")
    def source_db_instance_automated_backups_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the replicated automated backups from which to restore, for example, ``arn:aws:rds:useast-1:123456789012:auto-backup:ab-L2IJCEXJP7XQ7HOJ4SIEXAMPLE`` .

        This setting doesn't apply to RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbinstanceautomatedbackupsarn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDbInstanceAutomatedBackupsArn"))

    @source_db_instance_automated_backups_arn.setter
    def source_db_instance_automated_backups_arn(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6a0cd5832191dbece1854d3173842bc82e07d72379381399c41598288c902dcf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceDbInstanceAutomatedBackupsArn", value)

    @builtins.property
    @jsii.member(jsii_name="sourceDbInstanceIdentifier")
    def source_db_instance_identifier(self) -> typing.Optional[builtins.str]:
        '''If you want to create a read replica DB instance, specify the ID of the source DB instance.

        Each DB instance can have a limited number of read replicas. For more information, see `Working with Read Replicas <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/USER_ReadRepl.html>`_ in the *Amazon RDS User Guide* .

        For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* .

        The ``SourceDBInstanceIdentifier`` property determines whether a DB instance is a read replica. If you remove the ``SourceDBInstanceIdentifier`` property from your template and then update your stack, AWS CloudFormation promotes the Read Replica to a standalone DB instance.
        .. epigraph::

           - If you specify a source DB instance that uses VPC security groups, we recommend that you specify the ``VPCSecurityGroups`` property. If you don't specify the property, the read replica inherits the value of the ``VPCSecurityGroups`` property from the source DB when you create the replica. However, if you update the stack, AWS CloudFormation reverts the replica's ``VPCSecurityGroups`` property to the default value because it's not defined in the stack's template. This change might cause unexpected issues.
           - Read replicas don't support deletion policies. AWS CloudFormation ignores any deletion policy that's associated with a read replica.
           - If you specify ``SourceDBInstanceIdentifier`` , don't specify the ``DBSnapshotIdentifier`` property. You can't create a read replica from a snapshot.
           - Don't set the ``BackupRetentionPeriod`` , ``DBName`` , ``MasterUsername`` , ``MasterUserPassword`` , and ``PreferredBackupWindow`` properties. The database attributes are inherited from the source DB instance, and backups are disabled for read replicas.
           - If the source DB instance is in a different region than the read replica, specify the source region in ``SourceRegion`` , and specify an ARN for a valid DB instance in ``SourceDBInstanceIdentifier`` . For more information, see `Constructing a Amazon RDS Amazon Resource Name (ARN) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.html#USER_Tagging.ARN>`_ in the *Amazon RDS User Guide* .
           - For DB instances in Amazon Aurora clusters, don't specify this property. Amazon RDS automatically assigns writer and reader DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbinstanceidentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDbInstanceIdentifier"))

    @source_db_instance_identifier.setter
    def source_db_instance_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b328c4d7bf48de5bb0796e6591e5ebafad00261647493e9de2c231448fe6b09d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceDbInstanceIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="sourceDbiResourceId")
    def source_dbi_resource_id(self) -> typing.Optional[builtins.str]:
        '''The resource ID of the source DB instance from which to restore.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbiresourceid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDbiResourceId"))

    @source_dbi_resource_id.setter
    def source_dbi_resource_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b58e92d9226f6937698b0ac34becf8c06afb0010cec16813fbc64dcec57d0b22)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceDbiResourceId", value)

    @builtins.property
    @jsii.member(jsii_name="sourceRegion")
    def source_region(self) -> typing.Optional[builtins.str]:
        '''The ID of the region that contains the source DB instance for the read replica.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourceregion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceRegion"))

    @source_region.setter
    def source_region(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__98e738fd61a45341a1bfb4ccd9f5e49e1fd7077c9ebac6faf13899fe3b7ece92)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceRegion", value)

    @builtins.property
    @jsii.member(jsii_name="storageEncrypted")
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether the DB instance is encrypted. By default, it isn't encrypted.

        If you specify the ``KmsKeyId`` property, then you must enable encryption.

        If you specify the ``SourceDBInstanceIdentifier`` property, don't specify this property. The value is inherited from the source DB instance, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB instance is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB instance to be encrypted, then don't set this property or set it to ``false`` .

        *Amazon Aurora*

        Not applicable. The encryption for DB instances is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-storageencrypted
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "storageEncrypted"))

    @storage_encrypted.setter
    def storage_encrypted(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3efa3e03d5cd36a73ebbe58412d6d19010405fa02338a50fde768aa96e3f491e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageEncrypted", value)

    @builtins.property
    @jsii.member(jsii_name="storageThroughput")
    def storage_throughput(self) -> typing.Optional[jsii.Number]:
        '''Specifies the storage throughput value for the DB instance. This setting applies only to the ``gp3`` storage type.

        This setting doesn't apply to RDS Custom or Amazon Aurora.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-storagethroughput
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "storageThroughput"))

    @storage_throughput.setter
    def storage_throughput(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1739d211b61384f64025ff7b26939c4c96c91d52a38561ea746c5eb906ac484d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageThroughput", value)

    @builtins.property
    @jsii.member(jsii_name="storageType")
    def storage_type(self) -> typing.Optional[builtins.str]:
        '''Specifies the storage type to be associated with the DB instance.

        Valid values: ``gp2 | gp3 | io1 | standard``

        The ``standard`` value is also known as magnetic.

        If you specify ``io1`` or ``gp3`` , you must also include a value for the ``Iops`` parameter.

        Default: ``io1`` if the ``Iops`` parameter is specified, otherwise ``gp2``

        For more information, see `Amazon RDS DB Instance Storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. Aurora data is stored in the cluster volume, which is a single, virtual volume that uses solid state drives (SSDs).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-storagetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "storageType"))

    @storage_type.setter
    def storage_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5955c8667ea4e672b1ef79c36a67d3d77f86f4f89531b37f38e8a997ca461dc2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageType", value)

    @builtins.property
    @jsii.member(jsii_name="timezone")
    def timezone(self) -> typing.Optional[builtins.str]:
        '''The time zone of the DB instance.

        The time zone parameter is currently supported only by `Microsoft SQL Server <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-timezone
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "timezone"))

    @timezone.setter
    def timezone(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__05d9f86448ecf3afbff41a89d96cd2e38ecbac36e02ea4e377ee9183c3d07d5c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "timezone", value)

    @builtins.property
    @jsii.member(jsii_name="useDefaultProcessorFeatures")
    def use_default_processor_features(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether the DB instance class of the DB instance uses its default processor features.

        This setting doesn't apply to RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-usedefaultprocessorfeatures
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "useDefaultProcessorFeatures"))

    @use_default_processor_features.setter
    def use_default_processor_features(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a4ffde1f38de98a6d1712f6b9e05de1346951e573b31a9df14cc0b444c2db039)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "useDefaultProcessorFeatures", value)

    @builtins.property
    @jsii.member(jsii_name="useLatestRestorableTime")
    def use_latest_restorable_time(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether the DB instance is restored from the latest backup time.

        By default, the DB instance isn't restored from the latest backup time.

        Constraints: Can't be specified if the ``RestoreTime`` parameter is provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-uselatestrestorabletime
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "useLatestRestorableTime"))

    @use_latest_restorable_time.setter
    def use_latest_restorable_time(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6cd9c6c8e6962a101bcae361a854676b00cb94197b651acaa8b7524455becccc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "useLatestRestorableTime", value)

    @builtins.property
    @jsii.member(jsii_name="vpcSecurityGroups")
    def vpc_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of the VPC security group IDs to assign to the DB instance.

        The list can include both the physical IDs of existing VPC security groups and references to `AWS::EC2::SecurityGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html>`_ resources created in the template.

        If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        If you set ``VPCSecurityGroups`` , you must not set ```DBSecurityGroups`` <https://docs.aws.amazon.com//AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ , and vice versa.
        .. epigraph::

           You can migrate a DB instance in your stack from an RDS DB security group to a VPC security group, but keep the following in mind:

           - You can't revert to using an RDS security group after you establish a VPC security group membership.
           - When you migrate your DB instance to VPC security groups, if your stack update rolls back because the DB instance update fails or because an update fails in another AWS CloudFormation resource, the rollback fails because it can't revert to an RDS security group.
           - To use the properties that are available when you use a VPC security group, you must recreate the DB instance. If you don't, AWS CloudFormation submits only the property values that are listed in the ```DBSecurityGroups`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ property.

        To avoid this situation, migrate your DB instance to using VPC security groups only when that is the only change in your stack template.

        *Amazon Aurora*

        Not applicable. The associated list of EC2 VPC security groups is managed by the DB cluster. If specified, the setting must match the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-vpcsecuritygroups
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "vpcSecurityGroups"))

    @vpc_security_groups.setter
    def vpc_security_groups(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e8bb864c8b26659cba7cf3f140691fed6555a05f3a0bc0c8d900b54789d74b55)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcSecurityGroups", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBInstance.CertificateDetailsProperty",
        jsii_struct_bases=[],
        name_mapping={"ca_identifier": "caIdentifier", "valid_till": "validTill"},
    )
    class CertificateDetailsProperty:
        def __init__(
            self,
            *,
            ca_identifier: typing.Optional[builtins.str] = None,
            valid_till: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param ca_identifier: ``CfnDBInstance.CertificateDetailsProperty.CAIdentifier``.
            :param valid_till: ``CfnDBInstance.CertificateDetailsProperty.ValidTill``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-certificatedetails.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                certificate_details_property = rds.CfnDBInstance.CertificateDetailsProperty(
                    ca_identifier="caIdentifier",
                    valid_till="validTill"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a0072ccc264eba2aed8e235946525cc93af6f16ca8db092613f59b69fba6ddaf)
                check_type(argname="argument ca_identifier", value=ca_identifier, expected_type=type_hints["ca_identifier"])
                check_type(argname="argument valid_till", value=valid_till, expected_type=type_hints["valid_till"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if ca_identifier is not None:
                self._values["ca_identifier"] = ca_identifier
            if valid_till is not None:
                self._values["valid_till"] = valid_till

        @builtins.property
        def ca_identifier(self) -> typing.Optional[builtins.str]:
            '''``CfnDBInstance.CertificateDetailsProperty.CAIdentifier``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-certificatedetails.html#cfn-rds-dbinstance-certificatedetails-caidentifier
            '''
            result = self._values.get("ca_identifier")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def valid_till(self) -> typing.Optional[builtins.str]:
            '''``CfnDBInstance.CertificateDetailsProperty.ValidTill``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-certificatedetails.html#cfn-rds-dbinstance-certificatedetails-validtill
            '''
            result = self._values.get("valid_till")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CertificateDetailsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBInstance.DBInstanceRoleProperty",
        jsii_struct_bases=[],
        name_mapping={"feature_name": "featureName", "role_arn": "roleArn"},
    )
    class DBInstanceRoleProperty:
        def __init__(
            self,
            *,
            feature_name: builtins.str,
            role_arn: builtins.str,
        ) -> None:
            '''Describes an AWS Identity and Access Management (IAM) role that is associated with a DB instance.

            :param feature_name: The name of the feature associated with the AWS Identity and Access Management (IAM) role. IAM roles that are associated with a DB instance grant permission for the DB instance to access other AWS services on your behalf. For the list of supported feature names, see the ``SupportedFeatureNames`` description in `DBEngineVersion <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html>`_ in the *Amazon RDS API Reference* .
            :param role_arn: The Amazon Resource Name (ARN) of the IAM role that is associated with the DB instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-dbinstancerole.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                d_bInstance_role_property = rds.CfnDBInstance.DBInstanceRoleProperty(
                    feature_name="featureName",
                    role_arn="roleArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b7284939ad90d03cc192b42f7a4a45268cbecffb1e7310ed44c0ffa54100260e)
                check_type(argname="argument feature_name", value=feature_name, expected_type=type_hints["feature_name"])
                check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "feature_name": feature_name,
                "role_arn": role_arn,
            }

        @builtins.property
        def feature_name(self) -> builtins.str:
            '''The name of the feature associated with the AWS Identity and Access Management (IAM) role.

            IAM roles that are associated with a DB instance grant permission for the DB instance to access other AWS services on your behalf. For the list of supported feature names, see the ``SupportedFeatureNames`` description in `DBEngineVersion <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html>`_ in the *Amazon RDS API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-dbinstancerole.html#cfn-rds-dbinstance-dbinstancerole-featurename
            '''
            result = self._values.get("feature_name")
            assert result is not None, "Required property 'feature_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def role_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the IAM role that is associated with the DB instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-dbinstancerole.html#cfn-rds-dbinstance-dbinstancerole-rolearn
            '''
            result = self._values.get("role_arn")
            assert result is not None, "Required property 'role_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DBInstanceRoleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBInstance.EndpointProperty",
        jsii_struct_bases=[],
        name_mapping={
            "address": "address",
            "hosted_zone_id": "hostedZoneId",
            "port": "port",
        },
    )
    class EndpointProperty:
        def __init__(
            self,
            *,
            address: typing.Optional[builtins.str] = None,
            hosted_zone_id: typing.Optional[builtins.str] = None,
            port: typing.Optional[builtins.str] = None,
        ) -> None:
            '''This data type represents the information you need to connect to an Amazon RDS DB instance.

            This data type is used as a response element in the following actions:

            - ``CreateDBInstance``
            - ``DescribeDBInstances``
            - ``DeleteDBInstance``

            For the data structure that represents Amazon Aurora DB cluster endpoints, see ``DBClusterEndpoint`` .

            :param address: Specifies the DNS address of the DB instance.
            :param hosted_zone_id: Specifies the ID that Amazon Route 53 assigns when you create a hosted zone.
            :param port: Specifies the port that the database engine is listening on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-endpoint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                endpoint_property = rds.CfnDBInstance.EndpointProperty(
                    address="address",
                    hosted_zone_id="hostedZoneId",
                    port="port"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__abe761be64288aa6f437b1478c50e6344d635e08497a8a4a7820705566c88b86)
                check_type(argname="argument address", value=address, expected_type=type_hints["address"])
                check_type(argname="argument hosted_zone_id", value=hosted_zone_id, expected_type=type_hints["hosted_zone_id"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if address is not None:
                self._values["address"] = address
            if hosted_zone_id is not None:
                self._values["hosted_zone_id"] = hosted_zone_id
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def address(self) -> typing.Optional[builtins.str]:
            '''Specifies the DNS address of the DB instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-endpoint.html#cfn-rds-dbinstance-endpoint-address
            '''
            result = self._values.get("address")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def hosted_zone_id(self) -> typing.Optional[builtins.str]:
            '''Specifies the ID that Amazon Route 53 assigns when you create a hosted zone.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-endpoint.html#cfn-rds-dbinstance-endpoint-hostedzoneid
            '''
            result = self._values.get("hosted_zone_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[builtins.str]:
            '''Specifies the port that the database engine is listening on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-endpoint.html#cfn-rds-dbinstance-endpoint-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EndpointProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBInstance.MasterUserSecretProperty",
        jsii_struct_bases=[],
        name_mapping={"kms_key_id": "kmsKeyId", "secret_arn": "secretArn"},
    )
    class MasterUserSecretProperty:
        def __init__(
            self,
            *,
            kms_key_id: typing.Optional[builtins.str] = None,
            secret_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains the secret managed by RDS in AWS Secrets Manager for the master user password.

            For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide* and `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-secrets-manager.html>`_ in the *Amazon Aurora User Guide.*

            :param kms_key_id: The AWS KMS key identifier that is used to encrypt the secret.
            :param secret_arn: The Amazon Resource Name (ARN) of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-masterusersecret.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                master_user_secret_property = rds.CfnDBInstance.MasterUserSecretProperty(
                    kms_key_id="kmsKeyId",
                    secret_arn="secretArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__88d83f70a2f5ba98c1acf351e9598277e28abc682312ac336ab9cf0303e1322e)
                check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
                check_type(argname="argument secret_arn", value=secret_arn, expected_type=type_hints["secret_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if kms_key_id is not None:
                self._values["kms_key_id"] = kms_key_id
            if secret_arn is not None:
                self._values["secret_arn"] = secret_arn

        @builtins.property
        def kms_key_id(self) -> typing.Optional[builtins.str]:
            '''The AWS KMS key identifier that is used to encrypt the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-masterusersecret.html#cfn-rds-dbinstance-masterusersecret-kmskeyid
            '''
            result = self._values.get("kms_key_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def secret_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-masterusersecret.html#cfn-rds-dbinstance-masterusersecret-secretarn
            '''
            result = self._values.get("secret_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MasterUserSecretProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBInstance.ProcessorFeatureProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class ProcessorFeatureProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``ProcessorFeature`` property type specifies the processor features of a DB instance class status.

            :param name: The name of the processor feature. Valid names are ``coreCount`` and ``threadsPerCore`` .
            :param value: The value of a processor feature name.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-processorfeature.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                processor_feature_property = rds.CfnDBInstance.ProcessorFeatureProperty(
                    name="name",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8c6e95553fa694cc2f3779aba0ee0755618cd8e6bcaa02f89d3f3965fea690e7)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the processor feature.

            Valid names are ``coreCount`` and ``threadsPerCore`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-processorfeature.html#cfn-rds-dbinstance-processorfeature-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value of a processor feature name.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbinstance-processorfeature.html#cfn-rds-dbinstance-processorfeature-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProcessorFeatureProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBInstanceProps",
    jsii_struct_bases=[],
    name_mapping={
        "allocated_storage": "allocatedStorage",
        "allow_major_version_upgrade": "allowMajorVersionUpgrade",
        "associated_roles": "associatedRoles",
        "auto_minor_version_upgrade": "autoMinorVersionUpgrade",
        "availability_zone": "availabilityZone",
        "backup_retention_period": "backupRetentionPeriod",
        "ca_certificate_identifier": "caCertificateIdentifier",
        "certificate_details": "certificateDetails",
        "certificate_rotation_restart": "certificateRotationRestart",
        "character_set_name": "characterSetName",
        "copy_tags_to_snapshot": "copyTagsToSnapshot",
        "custom_iam_instance_profile": "customIamInstanceProfile",
        "db_cluster_identifier": "dbClusterIdentifier",
        "db_cluster_snapshot_identifier": "dbClusterSnapshotIdentifier",
        "db_instance_class": "dbInstanceClass",
        "db_instance_identifier": "dbInstanceIdentifier",
        "db_name": "dbName",
        "db_parameter_group_name": "dbParameterGroupName",
        "db_security_groups": "dbSecurityGroups",
        "db_snapshot_identifier": "dbSnapshotIdentifier",
        "db_subnet_group_name": "dbSubnetGroupName",
        "delete_automated_backups": "deleteAutomatedBackups",
        "deletion_protection": "deletionProtection",
        "domain": "domain",
        "domain_iam_role_name": "domainIamRoleName",
        "enable_cloudwatch_logs_exports": "enableCloudwatchLogsExports",
        "enable_iam_database_authentication": "enableIamDatabaseAuthentication",
        "enable_performance_insights": "enablePerformanceInsights",
        "endpoint": "endpoint",
        "engine": "engine",
        "engine_version": "engineVersion",
        "iops": "iops",
        "kms_key_id": "kmsKeyId",
        "license_model": "licenseModel",
        "manage_master_user_password": "manageMasterUserPassword",
        "master_username": "masterUsername",
        "master_user_password": "masterUserPassword",
        "master_user_secret": "masterUserSecret",
        "max_allocated_storage": "maxAllocatedStorage",
        "monitoring_interval": "monitoringInterval",
        "monitoring_role_arn": "monitoringRoleArn",
        "multi_az": "multiAz",
        "nchar_character_set_name": "ncharCharacterSetName",
        "network_type": "networkType",
        "option_group_name": "optionGroupName",
        "performance_insights_kms_key_id": "performanceInsightsKmsKeyId",
        "performance_insights_retention_period": "performanceInsightsRetentionPeriod",
        "port": "port",
        "preferred_backup_window": "preferredBackupWindow",
        "preferred_maintenance_window": "preferredMaintenanceWindow",
        "processor_features": "processorFeatures",
        "promotion_tier": "promotionTier",
        "publicly_accessible": "publiclyAccessible",
        "replica_mode": "replicaMode",
        "restore_time": "restoreTime",
        "source_db_instance_automated_backups_arn": "sourceDbInstanceAutomatedBackupsArn",
        "source_db_instance_identifier": "sourceDbInstanceIdentifier",
        "source_dbi_resource_id": "sourceDbiResourceId",
        "source_region": "sourceRegion",
        "storage_encrypted": "storageEncrypted",
        "storage_throughput": "storageThroughput",
        "storage_type": "storageType",
        "tags": "tags",
        "timezone": "timezone",
        "use_default_processor_features": "useDefaultProcessorFeatures",
        "use_latest_restorable_time": "useLatestRestorableTime",
        "vpc_security_groups": "vpcSecurityGroups",
    },
)
class CfnDBInstanceProps:
    def __init__(
        self,
        *,
        allocated_storage: typing.Optional[builtins.str] = None,
        allow_major_version_upgrade: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        associated_roles: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnDBInstance.DBInstanceRoleProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        auto_minor_version_upgrade: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        backup_retention_period: typing.Optional[jsii.Number] = None,
        ca_certificate_identifier: typing.Optional[builtins.str] = None,
        certificate_details: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnDBInstance.CertificateDetailsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        certificate_rotation_restart: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        character_set_name: typing.Optional[builtins.str] = None,
        copy_tags_to_snapshot: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        custom_iam_instance_profile: typing.Optional[builtins.str] = None,
        db_cluster_identifier: typing.Optional[builtins.str] = None,
        db_cluster_snapshot_identifier: typing.Optional[builtins.str] = None,
        db_instance_class: typing.Optional[builtins.str] = None,
        db_instance_identifier: typing.Optional[builtins.str] = None,
        db_name: typing.Optional[builtins.str] = None,
        db_parameter_group_name: typing.Optional[builtins.str] = None,
        db_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        db_snapshot_identifier: typing.Optional[builtins.str] = None,
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        delete_automated_backups: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        domain: typing.Optional[builtins.str] = None,
        domain_iam_role_name: typing.Optional[builtins.str] = None,
        enable_cloudwatch_logs_exports: typing.Optional[typing.Sequence[builtins.str]] = None,
        enable_iam_database_authentication: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        enable_performance_insights: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        endpoint: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnDBInstance.EndpointProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        iops: typing.Optional[jsii.Number] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        license_model: typing.Optional[builtins.str] = None,
        manage_master_user_password: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        master_username: typing.Optional[builtins.str] = None,
        master_user_password: typing.Optional[builtins.str] = None,
        master_user_secret: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnDBInstance.MasterUserSecretProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        max_allocated_storage: typing.Optional[jsii.Number] = None,
        monitoring_interval: typing.Optional[jsii.Number] = None,
        monitoring_role_arn: typing.Optional[builtins.str] = None,
        multi_az: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        nchar_character_set_name: typing.Optional[builtins.str] = None,
        network_type: typing.Optional[builtins.str] = None,
        option_group_name: typing.Optional[builtins.str] = None,
        performance_insights_kms_key_id: typing.Optional[builtins.str] = None,
        performance_insights_retention_period: typing.Optional[jsii.Number] = None,
        port: typing.Optional[builtins.str] = None,
        preferred_backup_window: typing.Optional[builtins.str] = None,
        preferred_maintenance_window: typing.Optional[builtins.str] = None,
        processor_features: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnDBInstance.ProcessorFeatureProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        promotion_tier: typing.Optional[jsii.Number] = None,
        publicly_accessible: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        replica_mode: typing.Optional[builtins.str] = None,
        restore_time: typing.Optional[builtins.str] = None,
        source_db_instance_automated_backups_arn: typing.Optional[builtins.str] = None,
        source_db_instance_identifier: typing.Optional[builtins.str] = None,
        source_dbi_resource_id: typing.Optional[builtins.str] = None,
        source_region: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        storage_throughput: typing.Optional[jsii.Number] = None,
        storage_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
        timezone: typing.Optional[builtins.str] = None,
        use_default_processor_features: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        use_latest_restorable_time: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        vpc_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBInstance``.

        :param allocated_storage: The amount of storage in gibibytes (GiB) to be initially allocated for the database instance. .. epigraph:: If any value is set in the ``Iops`` parameter, ``AllocatedStorage`` must be at least 100 GiB, which corresponds to the minimum Iops value of 1,000. If you increase the ``Iops`` value (in 1,000 IOPS increments), then you must also increase the ``AllocatedStorage`` value (in 100-GiB increments). *Amazon Aurora* Not applicable. Aurora cluster volumes automatically grow as the amount of data in your database increases, though you are only charged for the space that you use in an Aurora cluster volume. *MySQL* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 5 to 3072. *MariaDB* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 5 to 3072. *PostgreSQL* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 5 to 3072. *Oracle* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536. - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536. - Magnetic storage (standard): Must be an integer from 10 to 3072. *SQL Server* Constraints to the amount of storage for each storage type are the following: - General Purpose (SSD) storage (gp2): - Enterprise and Standard editions: Must be an integer from 20 to 16384. - Web and Express editions: Must be an integer from 20 to 16384. - Provisioned IOPS storage (io1): - Enterprise and Standard editions: Must be an integer from 20 to 16384. - Web and Express editions: Must be an integer from 20 to 16384. - Magnetic storage (standard): - Enterprise and Standard editions: Must be an integer from 20 to 1024. - Web and Express editions: Must be an integer from 20 to 1024.
        :param allow_major_version_upgrade: A value that indicates whether major version upgrades are allowed. Changing this parameter doesn't result in an outage and the change is asynchronously applied as soon as possible. Constraints: Major version upgrades must be allowed when specifying a value for the ``EngineVersion`` parameter that is a different major version than the DB instance's current version.
        :param associated_roles: The AWS Identity and Access Management (IAM) roles associated with the DB instance. *Amazon Aurora* Not applicable. The associated roles are managed by the DB cluster.
        :param auto_minor_version_upgrade: A value that indicates whether minor engine upgrades are applied automatically to the DB instance during the maintenance window. By default, minor engine upgrades are applied automatically.
        :param availability_zone: The Availability Zone (AZ) where the database will be created. For information on AWS Regions and Availability Zones, see `Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ . *Amazon Aurora* Each Aurora DB cluster hosts copies of its storage in three separate Availability Zones. Specify one of these Availability Zones. Aurora automatically chooses an appropriate Availability Zone if you don't specify one. Default: A random, system-chosen Availability Zone in the endpoint's AWS Region . Example: ``us-east-1d`` Constraint: The ``AvailabilityZone`` parameter can't be specified if the DB instance is a Multi-AZ deployment. The specified Availability Zone must be in the same AWS Region as the current endpoint.
        :param backup_retention_period: The number of days for which automated backups are retained. Setting this parameter to a positive number enables backups. Setting this parameter to 0 disables automated backups. *Amazon Aurora* Not applicable. The retention period for automated backups is managed by the DB cluster. Default: 1 Constraints: - Must be a value from 0 to 35 - Can't be set to 0 if the DB instance is a source to read replicas
        :param ca_certificate_identifier: The identifier of the CA certificate for this DB instance. .. epigraph:: Specifying or updating this property triggers a reboot. For more information about CA certificate identifiers for RDS DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon RDS User Guide* . For more information about CA certificate identifiers for Aurora DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon Aurora User Guide* .
        :param certificate_details: ``AWS::RDS::DBInstance.CertificateDetails``.
        :param certificate_rotation_restart: ``AWS::RDS::DBInstance.CertificateRotationRestart``.
        :param character_set_name: For supported engines, indicates that the DB instance should be associated with the specified character set. *Amazon Aurora* Not applicable. The character set is managed by the DB cluster. For more information, see `AWS::RDS::DBCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html>`_ .
        :param copy_tags_to_snapshot: A value that indicates whether to copy tags from the DB instance to snapshots of the DB instance. By default, tags are not copied. *Amazon Aurora* Not applicable. Copying tags to snapshots is managed by the DB cluster. Setting this value for an Aurora DB instance has no effect on the DB cluster setting.
        :param custom_iam_instance_profile: The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance. The instance profile must meet the following requirements: - The profile must exist in your account. - The profile must have an IAM role that Amazon EC2 has permissions to assume. - The instance profile name and the associated IAM role name must start with the prefix ``AWSRDSCustom`` . For the list of permissions required for the IAM role, see `Configure IAM and your VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-setup-orcl.html#custom-setup-orcl.iam-vpc>`_ in the *Amazon RDS User Guide* . This setting is required for RDS Custom.
        :param db_cluster_identifier: The identifier of the DB cluster that the instance will belong to.
        :param db_cluster_snapshot_identifier: The identifier for the RDS for MySQL Multi-AZ DB cluster snapshot to restore from. For more information on Multi-AZ DB clusters, see `Multi-AZ deployments with two readable standby DB instances <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/multi-az-db-clusters-concepts.html>`_ in the *Amazon RDS User Guide* . Constraints: - Must match the identifier of an existing Multi-AZ DB cluster snapshot. - Can't be specified when ``DBSnapshotIdentifier`` is specified. - Must be specified when ``DBSnapshotIdentifier`` isn't specified. - If you are restoring from a shared manual Multi-AZ DB cluster snapshot, the ``DBClusterSnapshotIdentifier`` must be the ARN of the shared snapshot. - Can't be the identifier of an Aurora DB cluster snapshot. - Can't be the identifier of an RDS for PostgreSQL Multi-AZ DB cluster snapshot.
        :param db_instance_class: The compute and memory capacity of the DB instance, for example, ``db.m4.large`` . Not all DB instance classes are available in all AWS Regions, or for all database engines. For the full list of DB instance classes, and availability for your engine, see `DB Instance Class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide.* For more information about DB instance class pricing and AWS Region support for DB instance classes, see `Amazon RDS Pricing <https://docs.aws.amazon.com/rds/pricing/>`_ .
        :param db_instance_identifier: A name for the DB instance. If you specify a name, AWS CloudFormation converts it to lowercase. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the DB instance. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ . For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* . .. epigraph:: If you specify a name, you can't perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param db_name: The meaning of this parameter differs according to the database engine you use. .. epigraph:: If you specify the ``[DBSnapshotIdentifier](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsnapshotidentifier)`` property, this property only applies to RDS for Oracle. *Amazon Aurora* Not applicable. The database name is managed by the DB cluster. *MySQL* The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Constraints: - Must contain 1 to 64 letters or numbers. - Can't be a word reserved by the specified database engine *MariaDB* The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance. Constraints: - Must contain 1 to 64 letters or numbers. - Can't be a word reserved by the specified database engine *PostgreSQL* The name of the database to create when the DB instance is created. If this parameter is not specified, the default ``postgres`` database is created in the DB instance. Constraints: - Must begin with a letter. Subsequent characters can be letters, underscores, or digits (0-9). - Must contain 1 to 63 characters. - Can't be a word reserved by the specified database engine *Oracle* The Oracle System ID (SID) of the created DB instance. If you specify ``null`` , the default value ``ORCL`` is used. You can't specify the string NULL, or any other reserved word, for ``DBName`` . Default: ``ORCL`` Constraints: - Can't be longer than 8 characters *SQL Server* Not applicable. Must be null.
        :param db_parameter_group_name: The name of an existing DB parameter group or a reference to an `AWS::RDS::DBParameterGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html>`_ resource created in the template. To list all of the available DB parameter group names, use the following command: ``aws rds describe-db-parameter-groups --query "DBParameterGroups[].DBParameterGroupName" --output text`` .. epigraph:: If any of the data members of the referenced parameter group are changed during an update, the DB instance might need to be restarted, which causes some interruption. If the parameter group contains static parameters, whether they were changed or not, an update triggers a reboot. If you don't specify a value for ``DBParameterGroupName`` property, the default DB parameter group for the specified engine and engine version is used.
        :param db_security_groups: A list of the DB security groups to assign to the DB instance. The list can include both the name of existing DB security groups or references to AWS::RDS::DBSecurityGroup resources created in the template. If you set DBSecurityGroups, you must not set VPCSecurityGroups, and vice versa. Also, note that the DBSecurityGroups property exists only for backwards compatibility with older regions and is no longer recommended for providing security information to an RDS DB instance. Instead, use VPCSecurityGroups. .. epigraph:: If you specify this property, AWS CloudFormation sends only the following properties (if specified) to Amazon RDS during create operations: - ``AllocatedStorage`` - ``AutoMinorVersionUpgrade`` - ``AvailabilityZone`` - ``BackupRetentionPeriod`` - ``CharacterSetName`` - ``DBInstanceClass`` - ``DBName`` - ``DBParameterGroupName`` - ``DBSecurityGroups`` - ``DBSubnetGroupName`` - ``Engine`` - ``EngineVersion`` - ``Iops`` - ``LicenseModel`` - ``MasterUsername`` - ``MasterUserPassword`` - ``MultiAZ`` - ``OptionGroupName`` - ``PreferredBackupWindow`` - ``PreferredMaintenanceWindow`` All other properties are ignored. Specify a virtual private cloud (VPC) security group if you want to submit other properties, such as ``StorageType`` , ``StorageEncrypted`` , or ``KmsKeyId`` . If you're already using the ``DBSecurityGroups`` property, you can't use these other properties by updating your DB instance to use a VPC security group. You must recreate the DB instance.
        :param db_snapshot_identifier: The name or Amazon Resource Name (ARN) of the DB snapshot that's used to restore the DB instance. If you're restoring from a shared manual DB snapshot, you must specify the ARN of the snapshot. By specifying this property, you can create a DB instance from the specified DB snapshot. If the ``DBSnapshotIdentifier`` property is an empty string or the ``AWS::RDS::DBInstance`` declaration has no ``DBSnapshotIdentifier`` property, AWS CloudFormation creates a new database. If the property contains a value (other than an empty string), AWS CloudFormation creates a database from the specified snapshot. If a snapshot with the specified name doesn't exist, AWS CloudFormation can't create the database and it rolls back the stack. Some DB instance properties aren't valid when you restore from a snapshot, such as the ``MasterUsername`` and ``MasterUserPassword`` properties. For information about the properties that you can specify, see the ``RestoreDBInstanceFromDBSnapshot`` action in the *Amazon RDS API Reference* . After you restore a DB instance with a ``DBSnapshotIdentifier`` property, you must specify the same ``DBSnapshotIdentifier`` property for any future updates to the DB instance. When you specify this property for an update, the DB instance is not restored from the DB snapshot again, and the data in the database is not changed. However, if you don't specify the ``DBSnapshotIdentifier`` property, an empty DB instance is created, and the original DB instance is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB instance is restored from the specified ``DBSnapshotIdentifier`` property, and the original DB instance is deleted. If you specify the ``DBSnapshotIdentifier`` property to restore a DB instance (as opposed to specifying it for DB instance updates), then don't specify the following properties: - ``CharacterSetName`` - ``DBClusterIdentifier`` - ``DBName`` - ``DeleteAutomatedBackups`` - ``EnablePerformanceInsights`` - ``KmsKeyId`` - ``MasterUsername`` - ``MasterUserPassword`` - ``PerformanceInsightsKMSKeyId`` - ``PerformanceInsightsRetentionPeriod`` - ``PromotionTier`` - ``SourceDBInstanceIdentifier`` - ``SourceRegion`` - ``StorageEncrypted`` (for an encrypted snapshot) - ``Timezone`` *Amazon Aurora* Not applicable. Snapshot restore is managed by the DB cluster.
        :param db_subnet_group_name: A DB subnet group to associate with the DB instance. If you update this value, the new subnet group must be a subnet group in a new VPC. If there's no DB subnet group, then the DB instance isn't a VPC DB instance. For more information about using Amazon RDS in a VPC, see `Using Amazon RDS with Amazon Virtual Private Cloud (VPC) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. The DB subnet group is managed by the DB cluster. If specified, the setting must match the DB cluster setting.
        :param delete_automated_backups: A value that indicates whether to remove automated backups immediately after the DB instance is deleted. This parameter isn't case-sensitive. The default is to remove automated backups immediately after the DB instance is deleted. *Amazon Aurora* Not applicable. When you delete a DB cluster, all automated backups for that DB cluster are deleted and can't be recovered. Manual DB cluster snapshots of the DB cluster are not deleted.
        :param deletion_protection: A value that indicates whether the DB instance has deletion protection enabled. The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled. For more information, see `Deleting a DB Instance <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_DeleteInstance.html>`_ . *Amazon Aurora* Not applicable. You can enable or disable deletion protection for the DB cluster. For more information, see ``CreateDBCluster`` . DB instances in a DB cluster can be deleted even when deletion protection is enabled for the DB cluster.
        :param domain: The Active Directory directory ID to create the DB instance in. Currently, only Microsoft SQL Server, Oracle, and PostgreSQL DB instances can be created in an Active Directory Domain. For more information, see `Kerberos Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html>`_ in the *Amazon RDS User Guide* .
        :param domain_iam_role_name: Specify the name of the IAM role to be used when making API calls to the Directory Service. This setting doesn't apply to RDS Custom. *Amazon Aurora* Not applicable. The domain is managed by the DB cluster.
        :param enable_cloudwatch_logs_exports: The list of log types that need to be enabled for exporting to CloudWatch Logs. The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Relational Database Service User Guide* . *Amazon Aurora* Not applicable. CloudWatch Logs exports are managed by the DB cluster. *MariaDB* Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery`` *Microsoft SQL Server* Valid values: ``agent`` , ``error`` *MySQL* Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery`` *Oracle* Valid values: ``alert`` , ``audit`` , ``listener`` , ``trace`` *PostgreSQL* Valid values: ``postgresql`` , ``upgrade``
        :param enable_iam_database_authentication: A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts. By default, mapping is disabled. This property is supported for RDS for MariaDB, RDS for MySQL, and RDS for PostgreSQL. For more information, see `IAM Database Authentication for MariaDB, MySQL, and PostgreSQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon RDS User Guide.* *Amazon Aurora* Not applicable. Mapping AWS IAM accounts to database accounts is managed by the DB cluster.
        :param enable_performance_insights: A value that indicates whether to enable Performance Insights for the DB instance. For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* . This setting doesn't apply to RDS Custom.
        :param endpoint: Specifies the connection endpoint. .. epigraph:: The endpoint might not be shown for instances whose status is ``creating`` .
        :param engine: The name of the database engine that you want to use for this DB instance. .. epigraph:: When you are creating a DB instance, the ``Engine`` property is required. Valid Values: - ``aurora`` (for MySQL 5.6-compatible Aurora) - ``aurora-mysql`` (for MySQL 5.7-compatible Aurora) - ``aurora-postgresql`` - ``mariadb`` - ``mysql`` - ``oracle-ee`` - ``oracle-ee-cdb`` - ``oracle-se2`` - ``oracle-se2-cdb`` - ``postgres`` - ``sqlserver-ee`` - ``sqlserver-se`` - ``sqlserver-ex`` - ``sqlserver-web``
        :param engine_version: The version number of the database engine to use. For a list of valid engine versions, use the ``DescribeDBEngineVersions`` action. The following are the database engines and links to information about the major and minor versions that are available with Amazon RDS. Not every database engine is available for every AWS Region. *Amazon Aurora* Not applicable. The version number of the database engine to be used by the DB instance is managed by the DB cluster. *MariaDB* See `MariaDB on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MariaDB.html#MariaDB.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.* *Microsoft SQL Server* See `Microsoft SQL Server Versions on Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.VersionSupport>`_ in the *Amazon RDS User Guide.* *MySQL* See `MySQL on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MySQL.html#MySQL.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.* *Oracle* See `Oracle Database Engine Release Notes <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.Oracle.PatchComposition.html>`_ in the *Amazon RDS User Guide.* *PostgreSQL* See `Supported PostgreSQL Database Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_PostgreSQL.html#PostgreSQL.Concepts.General.DBVersions>`_ in the *Amazon RDS User Guide.*
        :param iops: The number of I/O operations per second (IOPS) that the database provisions. The value must be equal to or greater than 1000. If you specify this property, you must follow the range of allowed ratios of your requested IOPS rate to the amount of storage that you allocate (IOPS to allocated storage). For example, you can provision an Oracle database instance with 1000 IOPS and 200 GiB of storage (a ratio of 5:1), or specify 2000 IOPS with 200 GiB of storage (a ratio of 10:1). For more information, see `Amazon RDS Provisioned IOPS Storage to Improve Performance <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* . .. epigraph:: If you specify ``io1`` for the ``StorageType`` property, then you must also specify the ``Iops`` property.
        :param kms_key_id: The ARN of the AWS KMS key that's used to encrypt the DB instance, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` . If you enable the StorageEncrypted property but don't specify this property, AWS CloudFormation uses the default KMS key. If you specify this property, you must set the StorageEncrypted property to true. If you specify the ``SourceDBInstanceIdentifier`` property, the value is inherited from the source DB instance if the read replica is created in the same region. If you create an encrypted read replica in a different AWS Region, then you must specify a KMS key for the destination AWS Region. KMS encryption keys are specific to the region that they're created in, and you can't use encryption keys from one region in another region. If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used. If you specify ``DBSecurityGroups`` , AWS CloudFormation ignores this property. To specify both a security group and this property, you must use a VPC security group. For more information about Amazon RDS and VPC, see `Using Amazon RDS with Amazon VPC <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. The KMS key identifier is managed by the DB cluster.
        :param license_model: License model information for this DB instance. Valid values: - Aurora MySQL - ``general-public-license`` - Aurora PostgreSQL - ``postgresql-license`` - MariaDB - ``general-public-license`` - Microsoft SQL Server - ``license-included`` - MySQL - ``general-public-license`` - Oracle - ``bring-your-own-license`` or ``license-included`` - PostgreSQL - ``postgresql-license`` .. epigraph:: If you've specified ``DBSecurityGroups`` and then you update the license model, AWS CloudFormation replaces the underlying DB instance. This will incur some interruptions to database availability.
        :param manage_master_user_password: A value that indicates whether to manage the master user password with AWS Secrets Manager. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.* Constraints: - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.
        :param master_username: The master user name for the DB instance. .. epigraph:: If you specify the ``SourceDBInstanceIdentifier`` or ``DBSnapshotIdentifier`` property, don't specify this property. The value is inherited from the source DB instance or snapshot. *Amazon Aurora* Not applicable. The name for the master user is managed by the DB cluster. *MariaDB* Constraints: - Required for MariaDB. - Must be 1 to 16 letters or numbers. - Can't be a reserved word for the chosen database engine. *Microsoft SQL Server* Constraints: - Required for SQL Server. - Must be 1 to 128 letters or numbers. - The first character must be a letter. - Can't be a reserved word for the chosen database engine. *MySQL* Constraints: - Required for MySQL. - Must be 1 to 16 letters or numbers. - First character must be a letter. - Can't be a reserved word for the chosen database engine. *Oracle* Constraints: - Required for Oracle. - Must be 1 to 30 letters or numbers. - First character must be a letter. - Can't be a reserved word for the chosen database engine. *PostgreSQL* Constraints: - Required for PostgreSQL. - Must be 1 to 63 letters or numbers. - First character must be a letter. - Can't be a reserved word for the chosen database engine.
        :param master_user_password: The password for the master user. The password can include any printable ASCII character except "/", """, or "@". *Amazon Aurora* Not applicable. The password for the master user is managed by the DB cluster. *MariaDB* Constraints: Must contain from 8 to 41 characters. *Microsoft SQL Server* Constraints: Must contain from 8 to 128 characters. *MySQL* Constraints: Must contain from 8 to 41 characters. *Oracle* Constraints: Must contain from 8 to 30 characters. *PostgreSQL* Constraints: Must contain from 8 to 128 characters.
        :param master_user_secret: Contains the secret managed by RDS in AWS Secrets Manager for the master user password. For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.*
        :param max_allocated_storage: The upper limit in gibibytes (GiB) to which Amazon RDS can automatically scale the storage of the DB instance. For more information about this setting, including limitations that apply to it, see `Managing capacity automatically with Amazon RDS storage autoscaling <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling>`_ in the *Amazon RDS User Guide* . This setting doesn't apply to RDS Custom. *Amazon Aurora* Not applicable. Storage is managed by the DB cluster.
        :param monitoring_interval: The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB instance. To disable collection of Enhanced Monitoring metrics, specify 0. The default is 0. If ``MonitoringRoleArn`` is specified, then you must set ``MonitoringInterval`` to a value other than 0. This setting doesn't apply to RDS Custom. Valid Values: ``0, 1, 5, 10, 15, 30, 60``
        :param monitoring_role_arn: The ARN for the IAM role that permits RDS to send enhanced monitoring metrics to Amazon CloudWatch Logs. For example, ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting Up and Enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* . If ``MonitoringInterval`` is set to a value other than 0, then you must supply a ``MonitoringRoleArn`` value. This setting doesn't apply to RDS Custom.
        :param multi_az: Specifies whether the database instance is a Multi-AZ DB instance deployment. You can't set the ``AvailabilityZone`` parameter if the ``MultiAZ`` parameter is set to true. Currently, you can't use AWS CloudFormation to create a Multi-AZ DB cluster deployment. For more information, see `Multi-AZ deployments for high availability <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.MultiAZ.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. Amazon Aurora storage is replicated across all of the Availability Zones and doesn't require the ``MultiAZ`` option to be set.
        :param nchar_character_set_name: The name of the NCHAR character set for the Oracle DB instance. This parameter doesn't apply to RDS Custom.
        :param network_type: The network type of the DB instance. Valid values: - ``IPV4`` - ``DUAL`` The network type is determined by the ``DBSubnetGroup`` specified for the DB instance. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ). For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon RDS User Guide.*
        :param option_group_name: Indicates that the DB instance should be associated with the specified option group. Permanent options, such as the TDE option for Oracle Advanced Security TDE, can't be removed from an option group. Also, that option group can't be removed from a DB instance once it is associated with a DB instance.
        :param performance_insights_kms_key_id: The AWS KMS key identifier for encryption of Performance Insights data. The KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key. If you do not specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account. Your AWS account has a different default KMS key for each AWS Region. For information about enabling Performance Insights, see `EnablePerformanceInsights <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights>`_ .
        :param performance_insights_retention_period: The amount of time, in days, to retain Performance Insights data. Valid values are 7 or 731 (2 years). For information about enabling Performance Insights, see `EnablePerformanceInsights <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights>`_ .
        :param port: The port number on which the database accepts connections. *Amazon Aurora* Not applicable. The port number is managed by the DB cluster.
        :param preferred_backup_window: The daily time range during which automated backups are created if automated backups are enabled, using the ``BackupRetentionPeriod`` parameter. For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow>`_ in the *Amazon RDS User Guide.* Constraints: - Must be in the format ``hh24:mi-hh24:mi`` . - Must be in Universal Coordinated Time (UTC). - Must not conflict with the preferred maintenance window. - Must be at least 30 minutes. *Amazon Aurora* Not applicable. The daily time range for creating automated backups is managed by the DB cluster.
        :param preferred_maintenance_window: The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC). Format: ``ddd:hh24:mi-ddd:hh24:mi`` The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Instance Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow>`_ in the *Amazon RDS User Guide.* .. epigraph:: This property applies when AWS CloudFormation initially creates the DB instance. If you use AWS CloudFormation to update the DB instance, those updates are applied immediately. Constraints: Minimum 30-minute window.
        :param processor_features: The number of CPU cores and the number of threads per core for the DB instance class of the DB instance. This setting doesn't apply to RDS Custom. *Amazon Aurora* Not applicable.
        :param promotion_tier: A value that specifies the order in which an Aurora Replica is promoted to the primary instance after a failure of the existing primary instance. For more information, see `Fault Tolerance for an Aurora DB Cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html#Aurora.Managing.FaultTolerance>`_ in the *Amazon Aurora User Guide* . This setting doesn't apply to RDS Custom. Default: 1 Valid Values: 0 - 15
        :param publicly_accessible: Indicates whether the DB instance is an internet-facing instance. If you specify true, AWS CloudFormation creates an instance with a publicly resolvable DNS name, which resolves to a public IP address. If you specify false, AWS CloudFormation creates an internal instance with a DNS name that resolves to a private IP address. The default behavior value depends on your VPC setup and the database subnet group. For more information, see the ``PubliclyAccessible`` parameter in the `CreateDBInstance <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html>`_ in the *Amazon RDS API Reference* .
        :param replica_mode: The open mode of an Oracle read replica. For more information, see `Working with Oracle Read Replicas for Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html>`_ in the *Amazon RDS User Guide* . This setting is only supported in RDS for Oracle. Default: ``open-read-only`` Valid Values: ``open-read-only`` or ``mounted``
        :param restore_time: The date and time to restore from. Valid Values: Value must be a time in Universal Coordinated Time (UTC) format Constraints: - Must be before the latest restorable time for the DB instance - Can't be specified if the ``UseLatestRestorableTime`` parameter is enabled Example: ``2009-09-07T23:45:00Z``
        :param source_db_instance_automated_backups_arn: The Amazon Resource Name (ARN) of the replicated automated backups from which to restore, for example, ``arn:aws:rds:useast-1:123456789012:auto-backup:ab-L2IJCEXJP7XQ7HOJ4SIEXAMPLE`` . This setting doesn't apply to RDS Custom.
        :param source_db_instance_identifier: If you want to create a read replica DB instance, specify the ID of the source DB instance. Each DB instance can have a limited number of read replicas. For more information, see `Working with Read Replicas <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/USER_ReadRepl.html>`_ in the *Amazon RDS User Guide* . For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* . The ``SourceDBInstanceIdentifier`` property determines whether a DB instance is a read replica. If you remove the ``SourceDBInstanceIdentifier`` property from your template and then update your stack, AWS CloudFormation promotes the Read Replica to a standalone DB instance. .. epigraph:: - If you specify a source DB instance that uses VPC security groups, we recommend that you specify the ``VPCSecurityGroups`` property. If you don't specify the property, the read replica inherits the value of the ``VPCSecurityGroups`` property from the source DB when you create the replica. However, if you update the stack, AWS CloudFormation reverts the replica's ``VPCSecurityGroups`` property to the default value because it's not defined in the stack's template. This change might cause unexpected issues. - Read replicas don't support deletion policies. AWS CloudFormation ignores any deletion policy that's associated with a read replica. - If you specify ``SourceDBInstanceIdentifier`` , don't specify the ``DBSnapshotIdentifier`` property. You can't create a read replica from a snapshot. - Don't set the ``BackupRetentionPeriod`` , ``DBName`` , ``MasterUsername`` , ``MasterUserPassword`` , and ``PreferredBackupWindow`` properties. The database attributes are inherited from the source DB instance, and backups are disabled for read replicas. - If the source DB instance is in a different region than the read replica, specify the source region in ``SourceRegion`` , and specify an ARN for a valid DB instance in ``SourceDBInstanceIdentifier`` . For more information, see `Constructing a Amazon RDS Amazon Resource Name (ARN) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.html#USER_Tagging.ARN>`_ in the *Amazon RDS User Guide* . - For DB instances in Amazon Aurora clusters, don't specify this property. Amazon RDS automatically assigns writer and reader DB instances.
        :param source_dbi_resource_id: The resource ID of the source DB instance from which to restore.
        :param source_region: The ID of the region that contains the source DB instance for the read replica.
        :param storage_encrypted: A value that indicates whether the DB instance is encrypted. By default, it isn't encrypted. If you specify the ``KmsKeyId`` property, then you must enable encryption. If you specify the ``SourceDBInstanceIdentifier`` property, don't specify this property. The value is inherited from the source DB instance, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used. If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB instance is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB instance to be encrypted, then don't set this property or set it to ``false`` . *Amazon Aurora* Not applicable. The encryption for DB instances is managed by the DB cluster.
        :param storage_throughput: Specifies the storage throughput value for the DB instance. This setting applies only to the ``gp3`` storage type. This setting doesn't apply to RDS Custom or Amazon Aurora.
        :param storage_type: Specifies the storage type to be associated with the DB instance. Valid values: ``gp2 | gp3 | io1 | standard`` The ``standard`` value is also known as magnetic. If you specify ``io1`` or ``gp3`` , you must also include a value for the ``Iops`` parameter. Default: ``io1`` if the ``Iops`` parameter is specified, otherwise ``gp2`` For more information, see `Amazon RDS DB Instance Storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html>`_ in the *Amazon RDS User Guide* . *Amazon Aurora* Not applicable. Aurora data is stored in the cluster volume, which is a single, virtual volume that uses solid state drives (SSDs).
        :param tags: An optional array of key-value pairs to apply to this DB instance.
        :param timezone: The time zone of the DB instance. The time zone parameter is currently supported only by `Microsoft SQL Server <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone>`_ .
        :param use_default_processor_features: A value that indicates whether the DB instance class of the DB instance uses its default processor features. This setting doesn't apply to RDS Custom.
        :param use_latest_restorable_time: A value that indicates whether the DB instance is restored from the latest backup time. By default, the DB instance isn't restored from the latest backup time. Constraints: Can't be specified if the ``RestoreTime`` parameter is provided.
        :param vpc_security_groups: A list of the VPC security group IDs to assign to the DB instance. The list can include both the physical IDs of existing VPC security groups and references to `AWS::EC2::SecurityGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html>`_ resources created in the template. If you plan to update the resource, don't specify VPC security groups in a shared VPC. If you set ``VPCSecurityGroups`` , you must not set ```DBSecurityGroups`` <https://docs.aws.amazon.com//AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ , and vice versa. .. epigraph:: You can migrate a DB instance in your stack from an RDS DB security group to a VPC security group, but keep the following in mind: - You can't revert to using an RDS security group after you establish a VPC security group membership. - When you migrate your DB instance to VPC security groups, if your stack update rolls back because the DB instance update fails or because an update fails in another AWS CloudFormation resource, the rollback fails because it can't revert to an RDS security group. - To use the properties that are available when you use a VPC security group, you must recreate the DB instance. If you don't, AWS CloudFormation submits only the property values that are listed in the ```DBSecurityGroups`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ property. To avoid this situation, migrate your DB instance to using VPC security groups only when that is the only change in your stack template. *Amazon Aurora* Not applicable. The associated list of EC2 VPC security groups is managed by the DB cluster. If specified, the setting must match the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            cfn_dBInstance_props = rds.CfnDBInstanceProps(
                allocated_storage="allocatedStorage",
                allow_major_version_upgrade=False,
                associated_roles=[rds.CfnDBInstance.DBInstanceRoleProperty(
                    feature_name="featureName",
                    role_arn="roleArn"
                )],
                auto_minor_version_upgrade=False,
                availability_zone="availabilityZone",
                backup_retention_period=123,
                ca_certificate_identifier="caCertificateIdentifier",
                certificate_details=rds.CfnDBInstance.CertificateDetailsProperty(
                    ca_identifier="caIdentifier",
                    valid_till="validTill"
                ),
                certificate_rotation_restart=False,
                character_set_name="characterSetName",
                copy_tags_to_snapshot=False,
                custom_iam_instance_profile="customIamInstanceProfile",
                db_cluster_identifier="dbClusterIdentifier",
                db_cluster_snapshot_identifier="dbClusterSnapshotIdentifier",
                db_instance_class="dbInstanceClass",
                db_instance_identifier="dbInstanceIdentifier",
                db_name="dbName",
                db_parameter_group_name="dbParameterGroupName",
                db_security_groups=["dbSecurityGroups"],
                db_snapshot_identifier="dbSnapshotIdentifier",
                db_subnet_group_name="dbSubnetGroupName",
                delete_automated_backups=False,
                deletion_protection=False,
                domain="domain",
                domain_iam_role_name="domainIamRoleName",
                enable_cloudwatch_logs_exports=["enableCloudwatchLogsExports"],
                enable_iam_database_authentication=False,
                enable_performance_insights=False,
                endpoint=rds.CfnDBInstance.EndpointProperty(
                    address="address",
                    hosted_zone_id="hostedZoneId",
                    port="port"
                ),
                engine="engine",
                engine_version="engineVersion",
                iops=123,
                kms_key_id="kmsKeyId",
                license_model="licenseModel",
                manage_master_user_password=False,
                master_username="masterUsername",
                master_user_password="masterUserPassword",
                master_user_secret=rds.CfnDBInstance.MasterUserSecretProperty(
                    kms_key_id="kmsKeyId",
                    secret_arn="secretArn"
                ),
                max_allocated_storage=123,
                monitoring_interval=123,
                monitoring_role_arn="monitoringRoleArn",
                multi_az=False,
                nchar_character_set_name="ncharCharacterSetName",
                network_type="networkType",
                option_group_name="optionGroupName",
                performance_insights_kms_key_id="performanceInsightsKmsKeyId",
                performance_insights_retention_period=123,
                port="port",
                preferred_backup_window="preferredBackupWindow",
                preferred_maintenance_window="preferredMaintenanceWindow",
                processor_features=[rds.CfnDBInstance.ProcessorFeatureProperty(
                    name="name",
                    value="value"
                )],
                promotion_tier=123,
                publicly_accessible=False,
                replica_mode="replicaMode",
                restore_time="restoreTime",
                source_db_instance_automated_backups_arn="sourceDbInstanceAutomatedBackupsArn",
                source_db_instance_identifier="sourceDbInstanceIdentifier",
                source_dbi_resource_id="sourceDbiResourceId",
                source_region="sourceRegion",
                storage_encrypted=False,
                storage_throughput=123,
                storage_type="storageType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                timezone="timezone",
                use_default_processor_features=False,
                use_latest_restorable_time=False,
                vpc_security_groups=["vpcSecurityGroups"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__093efbec5ec8c6daae2883e473927b634da436281b356e753f03c4b3fa3bac5e)
            check_type(argname="argument allocated_storage", value=allocated_storage, expected_type=type_hints["allocated_storage"])
            check_type(argname="argument allow_major_version_upgrade", value=allow_major_version_upgrade, expected_type=type_hints["allow_major_version_upgrade"])
            check_type(argname="argument associated_roles", value=associated_roles, expected_type=type_hints["associated_roles"])
            check_type(argname="argument auto_minor_version_upgrade", value=auto_minor_version_upgrade, expected_type=type_hints["auto_minor_version_upgrade"])
            check_type(argname="argument availability_zone", value=availability_zone, expected_type=type_hints["availability_zone"])
            check_type(argname="argument backup_retention_period", value=backup_retention_period, expected_type=type_hints["backup_retention_period"])
            check_type(argname="argument ca_certificate_identifier", value=ca_certificate_identifier, expected_type=type_hints["ca_certificate_identifier"])
            check_type(argname="argument certificate_details", value=certificate_details, expected_type=type_hints["certificate_details"])
            check_type(argname="argument certificate_rotation_restart", value=certificate_rotation_restart, expected_type=type_hints["certificate_rotation_restart"])
            check_type(argname="argument character_set_name", value=character_set_name, expected_type=type_hints["character_set_name"])
            check_type(argname="argument copy_tags_to_snapshot", value=copy_tags_to_snapshot, expected_type=type_hints["copy_tags_to_snapshot"])
            check_type(argname="argument custom_iam_instance_profile", value=custom_iam_instance_profile, expected_type=type_hints["custom_iam_instance_profile"])
            check_type(argname="argument db_cluster_identifier", value=db_cluster_identifier, expected_type=type_hints["db_cluster_identifier"])
            check_type(argname="argument db_cluster_snapshot_identifier", value=db_cluster_snapshot_identifier, expected_type=type_hints["db_cluster_snapshot_identifier"])
            check_type(argname="argument db_instance_class", value=db_instance_class, expected_type=type_hints["db_instance_class"])
            check_type(argname="argument db_instance_identifier", value=db_instance_identifier, expected_type=type_hints["db_instance_identifier"])
            check_type(argname="argument db_name", value=db_name, expected_type=type_hints["db_name"])
            check_type(argname="argument db_parameter_group_name", value=db_parameter_group_name, expected_type=type_hints["db_parameter_group_name"])
            check_type(argname="argument db_security_groups", value=db_security_groups, expected_type=type_hints["db_security_groups"])
            check_type(argname="argument db_snapshot_identifier", value=db_snapshot_identifier, expected_type=type_hints["db_snapshot_identifier"])
            check_type(argname="argument db_subnet_group_name", value=db_subnet_group_name, expected_type=type_hints["db_subnet_group_name"])
            check_type(argname="argument delete_automated_backups", value=delete_automated_backups, expected_type=type_hints["delete_automated_backups"])
            check_type(argname="argument deletion_protection", value=deletion_protection, expected_type=type_hints["deletion_protection"])
            check_type(argname="argument domain", value=domain, expected_type=type_hints["domain"])
            check_type(argname="argument domain_iam_role_name", value=domain_iam_role_name, expected_type=type_hints["domain_iam_role_name"])
            check_type(argname="argument enable_cloudwatch_logs_exports", value=enable_cloudwatch_logs_exports, expected_type=type_hints["enable_cloudwatch_logs_exports"])
            check_type(argname="argument enable_iam_database_authentication", value=enable_iam_database_authentication, expected_type=type_hints["enable_iam_database_authentication"])
            check_type(argname="argument enable_performance_insights", value=enable_performance_insights, expected_type=type_hints["enable_performance_insights"])
            check_type(argname="argument endpoint", value=endpoint, expected_type=type_hints["endpoint"])
            check_type(argname="argument engine", value=engine, expected_type=type_hints["engine"])
            check_type(argname="argument engine_version", value=engine_version, expected_type=type_hints["engine_version"])
            check_type(argname="argument iops", value=iops, expected_type=type_hints["iops"])
            check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
            check_type(argname="argument license_model", value=license_model, expected_type=type_hints["license_model"])
            check_type(argname="argument manage_master_user_password", value=manage_master_user_password, expected_type=type_hints["manage_master_user_password"])
            check_type(argname="argument master_username", value=master_username, expected_type=type_hints["master_username"])
            check_type(argname="argument master_user_password", value=master_user_password, expected_type=type_hints["master_user_password"])
            check_type(argname="argument master_user_secret", value=master_user_secret, expected_type=type_hints["master_user_secret"])
            check_type(argname="argument max_allocated_storage", value=max_allocated_storage, expected_type=type_hints["max_allocated_storage"])
            check_type(argname="argument monitoring_interval", value=monitoring_interval, expected_type=type_hints["monitoring_interval"])
            check_type(argname="argument monitoring_role_arn", value=monitoring_role_arn, expected_type=type_hints["monitoring_role_arn"])
            check_type(argname="argument multi_az", value=multi_az, expected_type=type_hints["multi_az"])
            check_type(argname="argument nchar_character_set_name", value=nchar_character_set_name, expected_type=type_hints["nchar_character_set_name"])
            check_type(argname="argument network_type", value=network_type, expected_type=type_hints["network_type"])
            check_type(argname="argument option_group_name", value=option_group_name, expected_type=type_hints["option_group_name"])
            check_type(argname="argument performance_insights_kms_key_id", value=performance_insights_kms_key_id, expected_type=type_hints["performance_insights_kms_key_id"])
            check_type(argname="argument performance_insights_retention_period", value=performance_insights_retention_period, expected_type=type_hints["performance_insights_retention_period"])
            check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            check_type(argname="argument preferred_backup_window", value=preferred_backup_window, expected_type=type_hints["preferred_backup_window"])
            check_type(argname="argument preferred_maintenance_window", value=preferred_maintenance_window, expected_type=type_hints["preferred_maintenance_window"])
            check_type(argname="argument processor_features", value=processor_features, expected_type=type_hints["processor_features"])
            check_type(argname="argument promotion_tier", value=promotion_tier, expected_type=type_hints["promotion_tier"])
            check_type(argname="argument publicly_accessible", value=publicly_accessible, expected_type=type_hints["publicly_accessible"])
            check_type(argname="argument replica_mode", value=replica_mode, expected_type=type_hints["replica_mode"])
            check_type(argname="argument restore_time", value=restore_time, expected_type=type_hints["restore_time"])
            check_type(argname="argument source_db_instance_automated_backups_arn", value=source_db_instance_automated_backups_arn, expected_type=type_hints["source_db_instance_automated_backups_arn"])
            check_type(argname="argument source_db_instance_identifier", value=source_db_instance_identifier, expected_type=type_hints["source_db_instance_identifier"])
            check_type(argname="argument source_dbi_resource_id", value=source_dbi_resource_id, expected_type=type_hints["source_dbi_resource_id"])
            check_type(argname="argument source_region", value=source_region, expected_type=type_hints["source_region"])
            check_type(argname="argument storage_encrypted", value=storage_encrypted, expected_type=type_hints["storage_encrypted"])
            check_type(argname="argument storage_throughput", value=storage_throughput, expected_type=type_hints["storage_throughput"])
            check_type(argname="argument storage_type", value=storage_type, expected_type=type_hints["storage_type"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument timezone", value=timezone, expected_type=type_hints["timezone"])
            check_type(argname="argument use_default_processor_features", value=use_default_processor_features, expected_type=type_hints["use_default_processor_features"])
            check_type(argname="argument use_latest_restorable_time", value=use_latest_restorable_time, expected_type=type_hints["use_latest_restorable_time"])
            check_type(argname="argument vpc_security_groups", value=vpc_security_groups, expected_type=type_hints["vpc_security_groups"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allocated_storage is not None:
            self._values["allocated_storage"] = allocated_storage
        if allow_major_version_upgrade is not None:
            self._values["allow_major_version_upgrade"] = allow_major_version_upgrade
        if associated_roles is not None:
            self._values["associated_roles"] = associated_roles
        if auto_minor_version_upgrade is not None:
            self._values["auto_minor_version_upgrade"] = auto_minor_version_upgrade
        if availability_zone is not None:
            self._values["availability_zone"] = availability_zone
        if backup_retention_period is not None:
            self._values["backup_retention_period"] = backup_retention_period
        if ca_certificate_identifier is not None:
            self._values["ca_certificate_identifier"] = ca_certificate_identifier
        if certificate_details is not None:
            self._values["certificate_details"] = certificate_details
        if certificate_rotation_restart is not None:
            self._values["certificate_rotation_restart"] = certificate_rotation_restart
        if character_set_name is not None:
            self._values["character_set_name"] = character_set_name
        if copy_tags_to_snapshot is not None:
            self._values["copy_tags_to_snapshot"] = copy_tags_to_snapshot
        if custom_iam_instance_profile is not None:
            self._values["custom_iam_instance_profile"] = custom_iam_instance_profile
        if db_cluster_identifier is not None:
            self._values["db_cluster_identifier"] = db_cluster_identifier
        if db_cluster_snapshot_identifier is not None:
            self._values["db_cluster_snapshot_identifier"] = db_cluster_snapshot_identifier
        if db_instance_class is not None:
            self._values["db_instance_class"] = db_instance_class
        if db_instance_identifier is not None:
            self._values["db_instance_identifier"] = db_instance_identifier
        if db_name is not None:
            self._values["db_name"] = db_name
        if db_parameter_group_name is not None:
            self._values["db_parameter_group_name"] = db_parameter_group_name
        if db_security_groups is not None:
            self._values["db_security_groups"] = db_security_groups
        if db_snapshot_identifier is not None:
            self._values["db_snapshot_identifier"] = db_snapshot_identifier
        if db_subnet_group_name is not None:
            self._values["db_subnet_group_name"] = db_subnet_group_name
        if delete_automated_backups is not None:
            self._values["delete_automated_backups"] = delete_automated_backups
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if domain is not None:
            self._values["domain"] = domain
        if domain_iam_role_name is not None:
            self._values["domain_iam_role_name"] = domain_iam_role_name
        if enable_cloudwatch_logs_exports is not None:
            self._values["enable_cloudwatch_logs_exports"] = enable_cloudwatch_logs_exports
        if enable_iam_database_authentication is not None:
            self._values["enable_iam_database_authentication"] = enable_iam_database_authentication
        if enable_performance_insights is not None:
            self._values["enable_performance_insights"] = enable_performance_insights
        if endpoint is not None:
            self._values["endpoint"] = endpoint
        if engine is not None:
            self._values["engine"] = engine
        if engine_version is not None:
            self._values["engine_version"] = engine_version
        if iops is not None:
            self._values["iops"] = iops
        if kms_key_id is not None:
            self._values["kms_key_id"] = kms_key_id
        if license_model is not None:
            self._values["license_model"] = license_model
        if manage_master_user_password is not None:
            self._values["manage_master_user_password"] = manage_master_user_password
        if master_username is not None:
            self._values["master_username"] = master_username
        if master_user_password is not None:
            self._values["master_user_password"] = master_user_password
        if master_user_secret is not None:
            self._values["master_user_secret"] = master_user_secret
        if max_allocated_storage is not None:
            self._values["max_allocated_storage"] = max_allocated_storage
        if monitoring_interval is not None:
            self._values["monitoring_interval"] = monitoring_interval
        if monitoring_role_arn is not None:
            self._values["monitoring_role_arn"] = monitoring_role_arn
        if multi_az is not None:
            self._values["multi_az"] = multi_az
        if nchar_character_set_name is not None:
            self._values["nchar_character_set_name"] = nchar_character_set_name
        if network_type is not None:
            self._values["network_type"] = network_type
        if option_group_name is not None:
            self._values["option_group_name"] = option_group_name
        if performance_insights_kms_key_id is not None:
            self._values["performance_insights_kms_key_id"] = performance_insights_kms_key_id
        if performance_insights_retention_period is not None:
            self._values["performance_insights_retention_period"] = performance_insights_retention_period
        if port is not None:
            self._values["port"] = port
        if preferred_backup_window is not None:
            self._values["preferred_backup_window"] = preferred_backup_window
        if preferred_maintenance_window is not None:
            self._values["preferred_maintenance_window"] = preferred_maintenance_window
        if processor_features is not None:
            self._values["processor_features"] = processor_features
        if promotion_tier is not None:
            self._values["promotion_tier"] = promotion_tier
        if publicly_accessible is not None:
            self._values["publicly_accessible"] = publicly_accessible
        if replica_mode is not None:
            self._values["replica_mode"] = replica_mode
        if restore_time is not None:
            self._values["restore_time"] = restore_time
        if source_db_instance_automated_backups_arn is not None:
            self._values["source_db_instance_automated_backups_arn"] = source_db_instance_automated_backups_arn
        if source_db_instance_identifier is not None:
            self._values["source_db_instance_identifier"] = source_db_instance_identifier
        if source_dbi_resource_id is not None:
            self._values["source_dbi_resource_id"] = source_dbi_resource_id
        if source_region is not None:
            self._values["source_region"] = source_region
        if storage_encrypted is not None:
            self._values["storage_encrypted"] = storage_encrypted
        if storage_throughput is not None:
            self._values["storage_throughput"] = storage_throughput
        if storage_type is not None:
            self._values["storage_type"] = storage_type
        if tags is not None:
            self._values["tags"] = tags
        if timezone is not None:
            self._values["timezone"] = timezone
        if use_default_processor_features is not None:
            self._values["use_default_processor_features"] = use_default_processor_features
        if use_latest_restorable_time is not None:
            self._values["use_latest_restorable_time"] = use_latest_restorable_time
        if vpc_security_groups is not None:
            self._values["vpc_security_groups"] = vpc_security_groups

    @builtins.property
    def allocated_storage(self) -> typing.Optional[builtins.str]:
        '''The amount of storage in gibibytes (GiB) to be initially allocated for the database instance.

        .. epigraph::

           If any value is set in the ``Iops`` parameter, ``AllocatedStorage`` must be at least 100 GiB, which corresponds to the minimum Iops value of 1,000. If you increase the ``Iops`` value (in 1,000 IOPS increments), then you must also increase the ``AllocatedStorage`` value (in 100-GiB increments).

        *Amazon Aurora*

        Not applicable. Aurora cluster volumes automatically grow as the amount of data in your database increases, though you are only charged for the space that you use in an Aurora cluster volume.

        *MySQL*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 5 to 3072.

        *MariaDB*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 5 to 3072.

        *PostgreSQL*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 5 to 3072.

        *Oracle*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2): Must be an integer from 20 to 65536.
        - Provisioned IOPS storage (io1): Must be an integer from 100 to 65536.
        - Magnetic storage (standard): Must be an integer from 10 to 3072.

        *SQL Server*

        Constraints to the amount of storage for each storage type are the following:

        - General Purpose (SSD) storage (gp2):
        - Enterprise and Standard editions: Must be an integer from 20 to 16384.
        - Web and Express editions: Must be an integer from 20 to 16384.
        - Provisioned IOPS storage (io1):
        - Enterprise and Standard editions: Must be an integer from 20 to 16384.
        - Web and Express editions: Must be an integer from 20 to 16384.
        - Magnetic storage (standard):
        - Enterprise and Standard editions: Must be an integer from 20 to 1024.
        - Web and Express editions: Must be an integer from 20 to 1024.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-allocatedstorage
        '''
        result = self._values.get("allocated_storage")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def allow_major_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether major version upgrades are allowed.

        Changing this parameter doesn't result in an outage and the change is asynchronously applied as soon as possible.

        Constraints: Major version upgrades must be allowed when specifying a value for the ``EngineVersion`` parameter that is a different major version than the DB instance's current version.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-allowmajorversionupgrade
        '''
        result = self._values.get("allow_major_version_upgrade")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def associated_roles(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBInstance.DBInstanceRoleProperty]]]]:
        '''The AWS Identity and Access Management (IAM) roles associated with the DB instance.

        *Amazon Aurora*

        Not applicable. The associated roles are managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-associatedroles
        '''
        result = self._values.get("associated_roles")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBInstance.DBInstanceRoleProperty]]]], result)

    @builtins.property
    def auto_minor_version_upgrade(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether minor engine upgrades are applied automatically to the DB instance during the maintenance window.

        By default, minor engine upgrades are applied automatically.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-autominorversionupgrade
        '''
        result = self._values.get("auto_minor_version_upgrade")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''The Availability Zone (AZ) where the database will be created.

        For information on AWS Regions and Availability Zones, see `Regions and Availability Zones <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.RegionsAndAvailabilityZones.html>`_ .

        *Amazon Aurora*

        Each Aurora DB cluster hosts copies of its storage in three separate Availability Zones. Specify one of these Availability Zones. Aurora automatically chooses an appropriate Availability Zone if you don't specify one.

        Default: A random, system-chosen Availability Zone in the endpoint's AWS Region .

        Example: ``us-east-1d``

        Constraint: The ``AvailabilityZone`` parameter can't be specified if the DB instance is a Multi-AZ deployment. The specified Availability Zone must be in the same AWS Region as the current endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-availabilityzone
        '''
        result = self._values.get("availability_zone")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def backup_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The number of days for which automated backups are retained.

        Setting this parameter to a positive number enables backups. Setting this parameter to 0 disables automated backups.

        *Amazon Aurora*

        Not applicable. The retention period for automated backups is managed by the DB cluster.

        Default: 1

        Constraints:

        - Must be a value from 0 to 35
        - Can't be set to 0 if the DB instance is a source to read replicas

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-backupretentionperiod
        '''
        result = self._values.get("backup_retention_period")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def ca_certificate_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier of the CA certificate for this DB instance.

        .. epigraph::

           Specifying or updating this property triggers a reboot.

        For more information about CA certificate identifiers for RDS DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon RDS User Guide* .

        For more information about CA certificate identifiers for Aurora DB engines, see `Rotating Your SSL/TLS Certificate <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL-certificate-rotation.html>`_ in the *Amazon Aurora User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-cacertificateidentifier
        '''
        result = self._values.get("ca_certificate_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def certificate_details(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBInstance.CertificateDetailsProperty]]:
        '''``AWS::RDS::DBInstance.CertificateDetails``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-certificatedetails
        '''
        result = self._values.get("certificate_details")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBInstance.CertificateDetailsProperty]], result)

    @builtins.property
    def certificate_rotation_restart(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''``AWS::RDS::DBInstance.CertificateRotationRestart``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-certificaterotationrestart
        '''
        result = self._values.get("certificate_rotation_restart")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def character_set_name(self) -> typing.Optional[builtins.str]:
        '''For supported engines, indicates that the DB instance should be associated with the specified character set.

        *Amazon Aurora*

        Not applicable. The character set is managed by the DB cluster. For more information, see `AWS::RDS::DBCluster <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbcluster.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-charactersetname
        '''
        result = self._values.get("character_set_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def copy_tags_to_snapshot(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to copy tags from the DB instance to snapshots of the DB instance.

        By default, tags are not copied.

        *Amazon Aurora*

        Not applicable. Copying tags to snapshots is managed by the DB cluster. Setting this value for an Aurora DB instance has no effect on the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-copytagstosnapshot
        '''
        result = self._values.get("copy_tags_to_snapshot")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def custom_iam_instance_profile(self) -> typing.Optional[builtins.str]:
        '''The instance profile associated with the underlying Amazon EC2 instance of an RDS Custom DB instance.

        The instance profile must meet the following requirements:

        - The profile must exist in your account.
        - The profile must have an IAM role that Amazon EC2 has permissions to assume.
        - The instance profile name and the associated IAM role name must start with the prefix ``AWSRDSCustom`` .

        For the list of permissions required for the IAM role, see `Configure IAM and your VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/custom-setup-orcl.html#custom-setup-orcl.iam-vpc>`_ in the *Amazon RDS User Guide* .

        This setting is required for RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-customiaminstanceprofile
        '''
        result = self._values.get("custom_iam_instance_profile")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier of the DB cluster that the instance will belong to.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbclusteridentifier
        '''
        result = self._values.get("db_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_cluster_snapshot_identifier(self) -> typing.Optional[builtins.str]:
        '''The identifier for the RDS for MySQL Multi-AZ DB cluster snapshot to restore from.

        For more information on Multi-AZ DB clusters, see `Multi-AZ deployments with two readable standby DB instances <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/multi-az-db-clusters-concepts.html>`_ in the *Amazon RDS User Guide* .

        Constraints:

        - Must match the identifier of an existing Multi-AZ DB cluster snapshot.
        - Can't be specified when ``DBSnapshotIdentifier`` is specified.
        - Must be specified when ``DBSnapshotIdentifier`` isn't specified.
        - If you are restoring from a shared manual Multi-AZ DB cluster snapshot, the ``DBClusterSnapshotIdentifier`` must be the ARN of the shared snapshot.
        - Can't be the identifier of an Aurora DB cluster snapshot.
        - Can't be the identifier of an RDS for PostgreSQL Multi-AZ DB cluster snapshot.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbclustersnapshotidentifier
        '''
        result = self._values.get("db_cluster_snapshot_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_instance_class(self) -> typing.Optional[builtins.str]:
        '''The compute and memory capacity of the DB instance, for example, ``db.m4.large`` . Not all DB instance classes are available in all AWS Regions, or for all database engines.

        For the full list of DB instance classes, and availability for your engine, see `DB Instance Class <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html>`_ in the *Amazon RDS User Guide.* For more information about DB instance class pricing and AWS Region support for DB instance classes, see `Amazon RDS Pricing <https://docs.aws.amazon.com/rds/pricing/>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbinstanceclass
        '''
        result = self._values.get("db_instance_class")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_instance_identifier(self) -> typing.Optional[builtins.str]:
        '''A name for the DB instance.

        If you specify a name, AWS CloudFormation converts it to lowercase. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the DB instance. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ .

        For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* .
        .. epigraph::

           If you specify a name, you can't perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbinstanceidentifier
        '''
        result = self._values.get("db_instance_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_name(self) -> typing.Optional[builtins.str]:
        '''The meaning of this parameter differs according to the database engine you use.

        .. epigraph::

           If you specify the ``[DBSnapshotIdentifier](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsnapshotidentifier)`` property, this property only applies to RDS for Oracle.

        *Amazon Aurora*

        Not applicable. The database name is managed by the DB cluster.

        *MySQL*

        The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance.

        Constraints:

        - Must contain 1 to 64 letters or numbers.
        - Can't be a word reserved by the specified database engine

        *MariaDB*

        The name of the database to create when the DB instance is created. If this parameter is not specified, no database is created in the DB instance.

        Constraints:

        - Must contain 1 to 64 letters or numbers.
        - Can't be a word reserved by the specified database engine

        *PostgreSQL*

        The name of the database to create when the DB instance is created. If this parameter is not specified, the default ``postgres`` database is created in the DB instance.

        Constraints:

        - Must begin with a letter. Subsequent characters can be letters, underscores, or digits (0-9).
        - Must contain 1 to 63 characters.
        - Can't be a word reserved by the specified database engine

        *Oracle*

        The Oracle System ID (SID) of the created DB instance. If you specify ``null`` , the default value ``ORCL`` is used. You can't specify the string NULL, or any other reserved word, for ``DBName`` .

        Default: ``ORCL``

        Constraints:

        - Can't be longer than 8 characters

        *SQL Server*

        Not applicable. Must be null.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbname
        '''
        result = self._values.get("db_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of an existing DB parameter group or a reference to an `AWS::RDS::DBParameterGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbparametergroup.html>`_ resource created in the template.

        To list all of the available DB parameter group names, use the following command:

        ``aws rds describe-db-parameter-groups --query "DBParameterGroups[].DBParameterGroupName" --output text``
        .. epigraph::

           If any of the data members of the referenced parameter group are changed during an update, the DB instance might need to be restarted, which causes some interruption. If the parameter group contains static parameters, whether they were changed or not, an update triggers a reboot.

        If you don't specify a value for ``DBParameterGroupName`` property, the default DB parameter group for the specified engine and engine version is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbparametergroupname
        '''
        result = self._values.get("db_parameter_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of the DB security groups to assign to the DB instance.

        The list can include both the name of existing DB security groups or references to AWS::RDS::DBSecurityGroup resources created in the template.

        If you set DBSecurityGroups, you must not set VPCSecurityGroups, and vice versa. Also, note that the DBSecurityGroups property exists only for backwards compatibility with older regions and is no longer recommended for providing security information to an RDS DB instance. Instead, use VPCSecurityGroups.
        .. epigraph::

           If you specify this property, AWS CloudFormation sends only the following properties (if specified) to Amazon RDS during create operations:

           - ``AllocatedStorage``
           - ``AutoMinorVersionUpgrade``
           - ``AvailabilityZone``
           - ``BackupRetentionPeriod``
           - ``CharacterSetName``
           - ``DBInstanceClass``
           - ``DBName``
           - ``DBParameterGroupName``
           - ``DBSecurityGroups``
           - ``DBSubnetGroupName``
           - ``Engine``
           - ``EngineVersion``
           - ``Iops``
           - ``LicenseModel``
           - ``MasterUsername``
           - ``MasterUserPassword``
           - ``MultiAZ``
           - ``OptionGroupName``
           - ``PreferredBackupWindow``
           - ``PreferredMaintenanceWindow``

           All other properties are ignored. Specify a virtual private cloud (VPC) security group if you want to submit other properties, such as ``StorageType`` , ``StorageEncrypted`` , or ``KmsKeyId`` . If you're already using the ``DBSecurityGroups`` property, you can't use these other properties by updating your DB instance to use a VPC security group. You must recreate the DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbsecuritygroups
        '''
        result = self._values.get("db_security_groups")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def db_snapshot_identifier(self) -> typing.Optional[builtins.str]:
        '''The name or Amazon Resource Name (ARN) of the DB snapshot that's used to restore the DB instance.

        If you're restoring from a shared manual DB snapshot, you must specify the ARN of the snapshot.

        By specifying this property, you can create a DB instance from the specified DB snapshot. If the ``DBSnapshotIdentifier`` property is an empty string or the ``AWS::RDS::DBInstance`` declaration has no ``DBSnapshotIdentifier`` property, AWS CloudFormation creates a new database. If the property contains a value (other than an empty string), AWS CloudFormation creates a database from the specified snapshot. If a snapshot with the specified name doesn't exist, AWS CloudFormation can't create the database and it rolls back the stack.

        Some DB instance properties aren't valid when you restore from a snapshot, such as the ``MasterUsername`` and ``MasterUserPassword`` properties. For information about the properties that you can specify, see the ``RestoreDBInstanceFromDBSnapshot`` action in the *Amazon RDS API Reference* .

        After you restore a DB instance with a ``DBSnapshotIdentifier`` property, you must specify the same ``DBSnapshotIdentifier`` property for any future updates to the DB instance. When you specify this property for an update, the DB instance is not restored from the DB snapshot again, and the data in the database is not changed. However, if you don't specify the ``DBSnapshotIdentifier`` property, an empty DB instance is created, and the original DB instance is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB instance is restored from the specified ``DBSnapshotIdentifier`` property, and the original DB instance is deleted.

        If you specify the ``DBSnapshotIdentifier`` property to restore a DB instance (as opposed to specifying it for DB instance updates), then don't specify the following properties:

        - ``CharacterSetName``
        - ``DBClusterIdentifier``
        - ``DBName``
        - ``DeleteAutomatedBackups``
        - ``EnablePerformanceInsights``
        - ``KmsKeyId``
        - ``MasterUsername``
        - ``MasterUserPassword``
        - ``PerformanceInsightsKMSKeyId``
        - ``PerformanceInsightsRetentionPeriod``
        - ``PromotionTier``
        - ``SourceDBInstanceIdentifier``
        - ``SourceRegion``
        - ``StorageEncrypted`` (for an encrypted snapshot)
        - ``Timezone``

        *Amazon Aurora*

        Not applicable. Snapshot restore is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbsnapshotidentifier
        '''
        result = self._values.get("db_snapshot_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        '''A DB subnet group to associate with the DB instance.

        If you update this value, the new subnet group must be a subnet group in a new VPC.

        If there's no DB subnet group, then the DB instance isn't a VPC DB instance.

        For more information about using Amazon RDS in a VPC, see `Using Amazon RDS with Amazon Virtual Private Cloud (VPC) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. The DB subnet group is managed by the DB cluster. If specified, the setting must match the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-dbsubnetgroupname
        '''
        result = self._values.get("db_subnet_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def delete_automated_backups(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to remove automated backups immediately after the DB instance is deleted.

        This parameter isn't case-sensitive. The default is to remove automated backups immediately after the DB instance is deleted.

        *Amazon Aurora*

        Not applicable. When you delete a DB cluster, all automated backups for that DB cluster are deleted and can't be recovered. Manual DB cluster snapshots of the DB cluster are not deleted.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-deleteautomatedbackups
        '''
        result = self._values.get("delete_automated_backups")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether the DB instance has deletion protection enabled.

        The database can't be deleted when deletion protection is enabled. By default, deletion protection is disabled. For more information, see `Deleting a DB Instance <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_DeleteInstance.html>`_ .

        *Amazon Aurora*

        Not applicable. You can enable or disable deletion protection for the DB cluster. For more information, see ``CreateDBCluster`` . DB instances in a DB cluster can be deleted even when deletion protection is enabled for the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-deletionprotection
        '''
        result = self._values.get("deletion_protection")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def domain(self) -> typing.Optional[builtins.str]:
        '''The Active Directory directory ID to create the DB instance in.

        Currently, only Microsoft SQL Server, Oracle, and PostgreSQL DB instances can be created in an Active Directory Domain.

        For more information, see `Kerberos Authentication <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html>`_ in the *Amazon RDS User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-domain
        '''
        result = self._values.get("domain")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def domain_iam_role_name(self) -> typing.Optional[builtins.str]:
        '''Specify the name of the IAM role to be used when making API calls to the Directory Service.

        This setting doesn't apply to RDS Custom.

        *Amazon Aurora*

        Not applicable. The domain is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-domainiamrolename
        '''
        result = self._values.get("domain_iam_role_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enable_cloudwatch_logs_exports(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of log types that need to be enabled for exporting to CloudWatch Logs.

        The values in the list depend on the DB engine being used. For more information, see `Publishing Database Logs to Amazon CloudWatch Logs <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_LogAccess.html#USER_LogAccess.Procedural.UploadtoCloudWatch>`_ in the *Amazon Relational Database Service User Guide* .

        *Amazon Aurora*

        Not applicable. CloudWatch Logs exports are managed by the DB cluster.

        *MariaDB*

        Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery``

        *Microsoft SQL Server*

        Valid values: ``agent`` , ``error``

        *MySQL*

        Valid values: ``audit`` , ``error`` , ``general`` , ``slowquery``

        *Oracle*

        Valid values: ``alert`` , ``audit`` , ``listener`` , ``trace``

        *PostgreSQL*

        Valid values: ``postgresql`` , ``upgrade``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-enablecloudwatchlogsexports
        '''
        result = self._values.get("enable_cloudwatch_logs_exports")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def enable_iam_database_authentication(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.

        By default, mapping is disabled.

        This property is supported for RDS for MariaDB, RDS for MySQL, and RDS for PostgreSQL. For more information, see `IAM Database Authentication for MariaDB, MySQL, and PostgreSQL <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html>`_ in the *Amazon RDS User Guide.*

        *Amazon Aurora*

        Not applicable. Mapping AWS IAM accounts to database accounts is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-enableiamdatabaseauthentication
        '''
        result = self._values.get("enable_iam_database_authentication")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def enable_performance_insights(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to enable Performance Insights for the DB instance.

        For more information, see `Using Amazon Performance Insights <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html>`_ in the *Amazon RDS User Guide* .

        This setting doesn't apply to RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-enableperformanceinsights
        '''
        result = self._values.get("enable_performance_insights")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def endpoint(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBInstance.EndpointProperty]]:
        '''Specifies the connection endpoint.

        .. epigraph::

           The endpoint might not be shown for instances whose status is ``creating`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-endpoint
        '''
        result = self._values.get("endpoint")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBInstance.EndpointProperty]], result)

    @builtins.property
    def engine(self) -> typing.Optional[builtins.str]:
        '''The name of the database engine that you want to use for this DB instance.

        .. epigraph::

           When you are creating a DB instance, the ``Engine`` property is required.

        Valid Values:

        - ``aurora`` (for MySQL 5.6-compatible Aurora)
        - ``aurora-mysql`` (for MySQL 5.7-compatible Aurora)
        - ``aurora-postgresql``
        - ``mariadb``
        - ``mysql``
        - ``oracle-ee``
        - ``oracle-ee-cdb``
        - ``oracle-se2``
        - ``oracle-se2-cdb``
        - ``postgres``
        - ``sqlserver-ee``
        - ``sqlserver-se``
        - ``sqlserver-ex``
        - ``sqlserver-web``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-engine
        '''
        result = self._values.get("engine")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def engine_version(self) -> typing.Optional[builtins.str]:
        '''The version number of the database engine to use.

        For a list of valid engine versions, use the ``DescribeDBEngineVersions`` action.

        The following are the database engines and links to information about the major and minor versions that are available with Amazon RDS. Not every database engine is available for every AWS Region.

        *Amazon Aurora*

        Not applicable. The version number of the database engine to be used by the DB instance is managed by the DB cluster.

        *MariaDB*

        See `MariaDB on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MariaDB.html#MariaDB.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.*

        *Microsoft SQL Server*

        See `Microsoft SQL Server Versions on Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.VersionSupport>`_ in the *Amazon RDS User Guide.*

        *MySQL*

        See `MySQL on Amazon RDS Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_MySQL.html#MySQL.Concepts.VersionMgmt>`_ in the *Amazon RDS User Guide.*

        *Oracle*

        See `Oracle Database Engine Release Notes <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.Oracle.PatchComposition.html>`_ in the *Amazon RDS User Guide.*

        *PostgreSQL*

        See `Supported PostgreSQL Database Versions <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_PostgreSQL.html#PostgreSQL.Concepts.General.DBVersions>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-engineversion
        '''
        result = self._values.get("engine_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def iops(self) -> typing.Optional[jsii.Number]:
        '''The number of I/O operations per second (IOPS) that the database provisions.

        The value must be equal to or greater than 1000.

        If you specify this property, you must follow the range of allowed ratios of your requested IOPS rate to the amount of storage that you allocate (IOPS to allocated storage). For example, you can provision an Oracle database instance with 1000 IOPS and 200 GiB of storage (a ratio of 5:1), or specify 2000 IOPS with 200 GiB of storage (a ratio of 10:1). For more information, see `Amazon RDS Provisioned IOPS Storage to Improve Performance <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/CHAP_Storage.html#USER_PIOPS>`_ in the *Amazon RDS User Guide* .
        .. epigraph::

           If you specify ``io1`` for the ``StorageType`` property, then you must also specify the ``Iops`` property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-iops
        '''
        result = self._values.get("iops")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The ARN of the AWS KMS key that's used to encrypt the DB instance, such as ``arn:aws:kms:us-east-1:012345678910:key/abcd1234-a123-456a-a12b-a123b4cd56ef`` .

        If you enable the StorageEncrypted property but don't specify this property, AWS CloudFormation uses the default KMS key. If you specify this property, you must set the StorageEncrypted property to true.

        If you specify the ``SourceDBInstanceIdentifier`` property, the value is inherited from the source DB instance if the read replica is created in the same region.

        If you create an encrypted read replica in a different AWS Region, then you must specify a KMS key for the destination AWS Region. KMS encryption keys are specific to the region that they're created in, and you can't use encryption keys from one region in another region.

        If you specify the ``SnapshotIdentifier`` property, the ``StorageEncrypted`` property value is inherited from the snapshot, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used.

        If you specify ``DBSecurityGroups`` , AWS CloudFormation ignores this property. To specify both a security group and this property, you must use a VPC security group. For more information about Amazon RDS and VPC, see `Using Amazon RDS with Amazon VPC <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. The KMS key identifier is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-kmskeyid
        '''
        result = self._values.get("kms_key_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def license_model(self) -> typing.Optional[builtins.str]:
        '''License model information for this DB instance.

        Valid values:

        - Aurora MySQL - ``general-public-license``
        - Aurora PostgreSQL - ``postgresql-license``
        - MariaDB - ``general-public-license``
        - Microsoft SQL Server - ``license-included``
        - MySQL - ``general-public-license``
        - Oracle - ``bring-your-own-license`` or ``license-included``
        - PostgreSQL - ``postgresql-license``

        .. epigraph::

           If you've specified ``DBSecurityGroups`` and then you update the license model, AWS CloudFormation replaces the underlying DB instance. This will incur some interruptions to database availability.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-licensemodel
        '''
        result = self._values.get("license_model")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def manage_master_user_password(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to manage the master user password with AWS Secrets Manager.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.*

        Constraints:

        - Can't manage the master user password with AWS Secrets Manager if ``MasterUserPassword`` is specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-managemasteruserpassword
        '''
        result = self._values.get("manage_master_user_password")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def master_username(self) -> typing.Optional[builtins.str]:
        '''The master user name for the DB instance.

        .. epigraph::

           If you specify the ``SourceDBInstanceIdentifier`` or ``DBSnapshotIdentifier`` property, don't specify this property. The value is inherited from the source DB instance or snapshot.

        *Amazon Aurora*

        Not applicable. The name for the master user is managed by the DB cluster.

        *MariaDB*

        Constraints:

        - Required for MariaDB.
        - Must be 1 to 16 letters or numbers.
        - Can't be a reserved word for the chosen database engine.

        *Microsoft SQL Server*

        Constraints:

        - Required for SQL Server.
        - Must be 1 to 128 letters or numbers.
        - The first character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        *MySQL*

        Constraints:

        - Required for MySQL.
        - Must be 1 to 16 letters or numbers.
        - First character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        *Oracle*

        Constraints:

        - Required for Oracle.
        - Must be 1 to 30 letters or numbers.
        - First character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        *PostgreSQL*

        Constraints:

        - Required for PostgreSQL.
        - Must be 1 to 63 letters or numbers.
        - First character must be a letter.
        - Can't be a reserved word for the chosen database engine.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-masterusername
        '''
        result = self._values.get("master_username")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def master_user_password(self) -> typing.Optional[builtins.str]:
        '''The password for the master user. The password can include any printable ASCII character except "/", """, or "@".

        *Amazon Aurora*

        Not applicable. The password for the master user is managed by the DB cluster.

        *MariaDB*

        Constraints: Must contain from 8 to 41 characters.

        *Microsoft SQL Server*

        Constraints: Must contain from 8 to 128 characters.

        *MySQL*

        Constraints: Must contain from 8 to 41 characters.

        *Oracle*

        Constraints: Must contain from 8 to 30 characters.

        *PostgreSQL*

        Constraints: Must contain from 8 to 128 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-masteruserpassword
        '''
        result = self._values.get("master_user_password")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def master_user_secret(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBInstance.MasterUserSecretProperty]]:
        '''Contains the secret managed by RDS in AWS Secrets Manager for the master user password.

        For more information, see `Password management with AWS Secrets Manager <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-secrets-manager.html>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-masterusersecret
        '''
        result = self._values.get("master_user_secret")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBInstance.MasterUserSecretProperty]], result)

    @builtins.property
    def max_allocated_storage(self) -> typing.Optional[jsii.Number]:
        '''The upper limit in gibibytes (GiB) to which Amazon RDS can automatically scale the storage of the DB instance.

        For more information about this setting, including limitations that apply to it, see `Managing capacity automatically with Amazon RDS storage autoscaling <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling>`_ in the *Amazon RDS User Guide* .

        This setting doesn't apply to RDS Custom.

        *Amazon Aurora*

        Not applicable. Storage is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-maxallocatedstorage
        '''
        result = self._values.get("max_allocated_storage")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def monitoring_interval(self) -> typing.Optional[jsii.Number]:
        '''The interval, in seconds, between points when Enhanced Monitoring metrics are collected for the DB instance.

        To disable collection of Enhanced Monitoring metrics, specify 0. The default is 0.

        If ``MonitoringRoleArn`` is specified, then you must set ``MonitoringInterval`` to a value other than 0.

        This setting doesn't apply to RDS Custom.

        Valid Values: ``0, 1, 5, 10, 15, 30, 60``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-monitoringinterval
        '''
        result = self._values.get("monitoring_interval")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def monitoring_role_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN for the IAM role that permits RDS to send enhanced monitoring metrics to Amazon CloudWatch Logs.

        For example, ``arn:aws:iam:123456789012:role/emaccess`` . For information on creating a monitoring role, see `Setting Up and Enabling Enhanced Monitoring <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Monitoring.OS.html#USER_Monitoring.OS.Enabling>`_ in the *Amazon RDS User Guide* .

        If ``MonitoringInterval`` is set to a value other than 0, then you must supply a ``MonitoringRoleArn`` value.

        This setting doesn't apply to RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-monitoringrolearn
        '''
        result = self._values.get("monitoring_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def multi_az(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''Specifies whether the database instance is a Multi-AZ DB instance deployment.

        You can't set the ``AvailabilityZone`` parameter if the ``MultiAZ`` parameter is set to true.

        Currently, you can't use AWS CloudFormation to create a Multi-AZ DB cluster deployment.

        For more information, see `Multi-AZ deployments for high availability <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.MultiAZ.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. Amazon Aurora storage is replicated across all of the Availability Zones and doesn't require the ``MultiAZ`` option to be set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-multiaz
        '''
        result = self._values.get("multi_az")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def nchar_character_set_name(self) -> typing.Optional[builtins.str]:
        '''The name of the NCHAR character set for the Oracle DB instance.

        This parameter doesn't apply to RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-ncharcharactersetname
        '''
        result = self._values.get("nchar_character_set_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def network_type(self) -> typing.Optional[builtins.str]:
        '''The network type of the DB instance.

        Valid values:

        - ``IPV4``
        - ``DUAL``

        The network type is determined by the ``DBSubnetGroup`` specified for the DB instance. A ``DBSubnetGroup`` can support only the IPv4 protocol or the IPv4 and IPv6 protocols ( ``DUAL`` ).

        For more information, see `Working with a DB instance in a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html>`_ in the *Amazon RDS User Guide.*

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-networktype
        '''
        result = self._values.get("network_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def option_group_name(self) -> typing.Optional[builtins.str]:
        '''Indicates that the DB instance should be associated with the specified option group.

        Permanent options, such as the TDE option for Oracle Advanced Security TDE, can't be removed from an option group. Also, that option group can't be removed from a DB instance once it is associated with a DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-optiongroupname
        '''
        result = self._values.get("option_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def performance_insights_kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key identifier for encryption of Performance Insights data.

        The KMS key identifier is the key ARN, key ID, alias ARN, or alias name for the KMS key.

        If you do not specify a value for ``PerformanceInsightsKMSKeyId`` , then Amazon RDS uses your default KMS key. There is a default KMS key for your AWS account. Your AWS account has a different default KMS key for each AWS Region.

        For information about enabling Performance Insights, see `EnablePerformanceInsights <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-performanceinsightskmskeyid
        '''
        result = self._values.get("performance_insights_kms_key_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def performance_insights_retention_period(self) -> typing.Optional[jsii.Number]:
        '''The amount of time, in days, to retain Performance Insights data. Valid values are 7 or 731 (2 years).

        For information about enabling Performance Insights, see `EnablePerformanceInsights <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-enableperformanceinsights>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-performanceinsightsretentionperiod
        '''
        result = self._values.get("performance_insights_retention_period")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def port(self) -> typing.Optional[builtins.str]:
        '''The port number on which the database accepts connections.

        *Amazon Aurora*

        Not applicable. The port number is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-port
        '''
        result = self._values.get("port")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def preferred_backup_window(self) -> typing.Optional[builtins.str]:
        '''The daily time range during which automated backups are created if automated backups are enabled, using the ``BackupRetentionPeriod`` parameter.

        For more information, see `Backup Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow>`_ in the *Amazon RDS User Guide.*

        Constraints:

        - Must be in the format ``hh24:mi-hh24:mi`` .
        - Must be in Universal Coordinated Time (UTC).
        - Must not conflict with the preferred maintenance window.
        - Must be at least 30 minutes.

        *Amazon Aurora*

        Not applicable. The daily time range for creating automated backups is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-preferredbackupwindow
        '''
        result = self._values.get("preferred_backup_window")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def preferred_maintenance_window(self) -> typing.Optional[builtins.str]:
        '''The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).

        Format: ``ddd:hh24:mi-ddd:hh24:mi``

        The default is a 30-minute window selected at random from an 8-hour block of time for each AWS Region, occurring on a random day of the week. To see the time blocks available, see `Adjusting the Preferred DB Instance Maintenance Window <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#AdjustingTheMaintenanceWindow>`_ in the *Amazon RDS User Guide.*
        .. epigraph::

           This property applies when AWS CloudFormation initially creates the DB instance. If you use AWS CloudFormation to update the DB instance, those updates are applied immediately.

        Constraints: Minimum 30-minute window.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-preferredmaintenancewindow
        '''
        result = self._values.get("preferred_maintenance_window")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def processor_features(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBInstance.ProcessorFeatureProperty]]]]:
        '''The number of CPU cores and the number of threads per core for the DB instance class of the DB instance.

        This setting doesn't apply to RDS Custom.

        *Amazon Aurora*

        Not applicable.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-processorfeatures
        '''
        result = self._values.get("processor_features")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBInstance.ProcessorFeatureProperty]]]], result)

    @builtins.property
    def promotion_tier(self) -> typing.Optional[jsii.Number]:
        '''A value that specifies the order in which an Aurora Replica is promoted to the primary instance after a failure of the existing primary instance.

        For more information, see `Fault Tolerance for an Aurora DB Cluster <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Aurora.Managing.Backups.html#Aurora.Managing.FaultTolerance>`_ in the *Amazon Aurora User Guide* .

        This setting doesn't apply to RDS Custom.

        Default: 1

        Valid Values: 0 - 15

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-promotiontier
        '''
        result = self._values.get("promotion_tier")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def publicly_accessible(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''Indicates whether the DB instance is an internet-facing instance.

        If you specify true, AWS CloudFormation creates an instance with a publicly resolvable DNS name, which resolves to a public IP address. If you specify false, AWS CloudFormation creates an internal instance with a DNS name that resolves to a private IP address.

        The default behavior value depends on your VPC setup and the database subnet group. For more information, see the ``PubliclyAccessible`` parameter in the `CreateDBInstance <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_CreateDBInstance.html>`_ in the *Amazon RDS API Reference* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-publiclyaccessible
        '''
        result = self._values.get("publicly_accessible")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def replica_mode(self) -> typing.Optional[builtins.str]:
        '''The open mode of an Oracle read replica.

        For more information, see `Working with Oracle Read Replicas for Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-read-replicas.html>`_ in the *Amazon RDS User Guide* .

        This setting is only supported in RDS for Oracle.

        Default: ``open-read-only``

        Valid Values: ``open-read-only`` or ``mounted``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-replicamode
        '''
        result = self._values.get("replica_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def restore_time(self) -> typing.Optional[builtins.str]:
        '''The date and time to restore from.

        Valid Values: Value must be a time in Universal Coordinated Time (UTC) format

        Constraints:

        - Must be before the latest restorable time for the DB instance
        - Can't be specified if the ``UseLatestRestorableTime`` parameter is enabled

        Example: ``2009-09-07T23:45:00Z``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-restoretime
        '''
        result = self._values.get("restore_time")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_db_instance_automated_backups_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the replicated automated backups from which to restore, for example, ``arn:aws:rds:useast-1:123456789012:auto-backup:ab-L2IJCEXJP7XQ7HOJ4SIEXAMPLE`` .

        This setting doesn't apply to RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbinstanceautomatedbackupsarn
        '''
        result = self._values.get("source_db_instance_automated_backups_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_db_instance_identifier(self) -> typing.Optional[builtins.str]:
        '''If you want to create a read replica DB instance, specify the ID of the source DB instance.

        Each DB instance can have a limited number of read replicas. For more information, see `Working with Read Replicas <https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/USER_ReadRepl.html>`_ in the *Amazon RDS User Guide* .

        For information about constraints that apply to DB instance identifiers, see `Naming constraints in Amazon RDS <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.Constraints>`_ in the *Amazon RDS User Guide* .

        The ``SourceDBInstanceIdentifier`` property determines whether a DB instance is a read replica. If you remove the ``SourceDBInstanceIdentifier`` property from your template and then update your stack, AWS CloudFormation promotes the Read Replica to a standalone DB instance.
        .. epigraph::

           - If you specify a source DB instance that uses VPC security groups, we recommend that you specify the ``VPCSecurityGroups`` property. If you don't specify the property, the read replica inherits the value of the ``VPCSecurityGroups`` property from the source DB when you create the replica. However, if you update the stack, AWS CloudFormation reverts the replica's ``VPCSecurityGroups`` property to the default value because it's not defined in the stack's template. This change might cause unexpected issues.
           - Read replicas don't support deletion policies. AWS CloudFormation ignores any deletion policy that's associated with a read replica.
           - If you specify ``SourceDBInstanceIdentifier`` , don't specify the ``DBSnapshotIdentifier`` property. You can't create a read replica from a snapshot.
           - Don't set the ``BackupRetentionPeriod`` , ``DBName`` , ``MasterUsername`` , ``MasterUserPassword`` , and ``PreferredBackupWindow`` properties. The database attributes are inherited from the source DB instance, and backups are disabled for read replicas.
           - If the source DB instance is in a different region than the read replica, specify the source region in ``SourceRegion`` , and specify an ARN for a valid DB instance in ``SourceDBInstanceIdentifier`` . For more information, see `Constructing a Amazon RDS Amazon Resource Name (ARN) <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Tagging.html#USER_Tagging.ARN>`_ in the *Amazon RDS User Guide* .
           - For DB instances in Amazon Aurora clusters, don't specify this property. Amazon RDS automatically assigns writer and reader DB instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbinstanceidentifier
        '''
        result = self._values.get("source_db_instance_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_dbi_resource_id(self) -> typing.Optional[builtins.str]:
        '''The resource ID of the source DB instance from which to restore.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourcedbiresourceid
        '''
        result = self._values.get("source_dbi_resource_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_region(self) -> typing.Optional[builtins.str]:
        '''The ID of the region that contains the source DB instance for the read replica.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-sourceregion
        '''
        result = self._values.get("source_region")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether the DB instance is encrypted. By default, it isn't encrypted.

        If you specify the ``KmsKeyId`` property, then you must enable encryption.

        If you specify the ``SourceDBInstanceIdentifier`` property, don't specify this property. The value is inherited from the source DB instance, and if the DB instance is encrypted, the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot is encrypted, don't specify this property. The value is inherited from the snapshot, and the specified ``KmsKeyId`` property is used.

        If you specify the ``SnapshotIdentifier`` and the specified snapshot isn't encrypted, you can use this property to specify that the restored DB instance is encrypted. Specify the ``KmsKeyId`` property for the KMS key to use for encryption. If you don't want the restored DB instance to be encrypted, then don't set this property or set it to ``false`` .

        *Amazon Aurora*

        Not applicable. The encryption for DB instances is managed by the DB cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-storageencrypted
        '''
        result = self._values.get("storage_encrypted")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def storage_throughput(self) -> typing.Optional[jsii.Number]:
        '''Specifies the storage throughput value for the DB instance. This setting applies only to the ``gp3`` storage type.

        This setting doesn't apply to RDS Custom or Amazon Aurora.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-storagethroughput
        '''
        result = self._values.get("storage_throughput")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def storage_type(self) -> typing.Optional[builtins.str]:
        '''Specifies the storage type to be associated with the DB instance.

        Valid values: ``gp2 | gp3 | io1 | standard``

        The ``standard`` value is also known as magnetic.

        If you specify ``io1`` or ``gp3`` , you must also include a value for the ``Iops`` parameter.

        Default: ``io1`` if the ``Iops`` parameter is specified, otherwise ``gp2``

        For more information, see `Amazon RDS DB Instance Storage <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Storage.html>`_ in the *Amazon RDS User Guide* .

        *Amazon Aurora*

        Not applicable. Aurora data is stored in the cluster volume, which is a single, virtual volume that uses solid state drives (SSDs).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-storagetype
        '''
        result = self._values.get("storage_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]]:
        '''An optional array of key-value pairs to apply to this DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]], result)

    @builtins.property
    def timezone(self) -> typing.Optional[builtins.str]:
        '''The time zone of the DB instance.

        The time zone parameter is currently supported only by `Microsoft SQL Server <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_SQLServer.html#SQLServer.Concepts.General.TimeZone>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-timezone
        '''
        result = self._values.get("timezone")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def use_default_processor_features(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether the DB instance class of the DB instance uses its default processor features.

        This setting doesn't apply to RDS Custom.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-usedefaultprocessorfeatures
        '''
        result = self._values.get("use_default_processor_features")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def use_latest_restorable_time(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether the DB instance is restored from the latest backup time.

        By default, the DB instance isn't restored from the latest backup time.

        Constraints: Can't be specified if the ``RestoreTime`` parameter is provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-uselatestrestorabletime
        '''
        result = self._values.get("use_latest_restorable_time")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def vpc_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of the VPC security group IDs to assign to the DB instance.

        The list can include both the physical IDs of existing VPC security groups and references to `AWS::EC2::SecurityGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html>`_ resources created in the template.

        If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        If you set ``VPCSecurityGroups`` , you must not set ```DBSecurityGroups`` <https://docs.aws.amazon.com//AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ , and vice versa.
        .. epigraph::

           You can migrate a DB instance in your stack from an RDS DB security group to a VPC security group, but keep the following in mind:

           - You can't revert to using an RDS security group after you establish a VPC security group membership.
           - When you migrate your DB instance to VPC security groups, if your stack update rolls back because the DB instance update fails or because an update fails in another AWS CloudFormation resource, the rollback fails because it can't revert to an RDS security group.
           - To use the properties that are available when you use a VPC security group, you must recreate the DB instance. If you don't, AWS CloudFormation submits only the property values that are listed in the ```DBSecurityGroups`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html#cfn-rds-dbinstance-dbsecuritygroups>`_ property.

        To avoid this situation, migrate your DB instance to using VPC security groups only when that is the only change in your stack template.

        *Amazon Aurora*

        Not applicable. The associated list of EC2 VPC security groups is managed by the DB cluster. If specified, the setting must match the DB cluster setting.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbinstance.html#cfn-rds-dbinstance-vpcsecuritygroups
        '''
        result = self._values.get("vpc_security_groups")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBInstanceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnDBParameterGroup(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBParameterGroup",
):
    '''A CloudFormation ``AWS::RDS::DBParameterGroup``.

    The ``AWS::RDS::DBParameterGroup`` resource creates a custom parameter group for an RDS database family.

    This type can be declared in a template and referenced in the ``DBParameterGroupName`` property of an ``[AWS::RDS::DBInstance](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-database-instance.html)`` resource.

    For information about configuring parameters for Amazon RDS DB instances, see `Working with parameter groups <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon RDS User Guide* .

    For information about configuring parameters for Amazon Aurora DB instances, see `Working with parameter groups <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon Aurora User Guide* .
    .. epigraph::

       Applying a parameter group to a DB instance may require the DB instance to reboot, resulting in a database outage for the duration of the reboot.

    :cloudformationResource: AWS::RDS::DBParameterGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        # parameters: Any
        
        cfn_dBParameter_group = rds.CfnDBParameterGroup(self, "MyCfnDBParameterGroup",
            description="description",
            family="family",
        
            # the properties below are optional
            db_parameter_group_name="dbParameterGroupName",
            parameters=parameters,
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        description: builtins.str,
        family: builtins.str,
        db_parameter_group_name: typing.Optional[builtins.str] = None,
        parameters: typing.Any = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBParameterGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param description: Provides the customer-specified description for this DB parameter group.
        :param family: The DB parameter group family name. A DB parameter group can be associated with one and only one DB parameter group family, and can be applied only to a DB instance running a DB engine and engine version compatible with that DB parameter group family. .. epigraph:: The DB parameter group family can't be changed when updating a DB parameter group. To list all of the available parameter group families, use the following command: ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"`` The output contains duplicates. For more information, see ``[CreateDBParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html)`` .
        :param db_parameter_group_name: The name of the DB parameter group. Constraints: - Must be 1 to 255 letters, numbers, or hyphens. - First character must be a letter - Can't end with a hyphen or contain two consecutive hyphens If you don't specify a value for ``DBParameterGroupName`` property, a name is automatically created for the DB parameter group. .. epigraph:: This value is stored as a lowercase string.
        :param parameters: An array of parameter names and values for the parameter update. At least one parameter name and value must be supplied. Subsequent arguments are optional. For more information about DB parameters and DB parameter groups for Amazon RDS DB engines, see `Working with DB Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon RDS User Guide* . For more information about DB cluster and DB instance parameters and parameter groups for Amazon Aurora DB engines, see `Working with DB Parameter Groups and DB Cluster Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon Aurora User Guide* . .. epigraph:: AWS CloudFormation doesn't support specifying an apply method for each individual parameter. The default apply method for each parameter is used.
        :param tags: An optional array of key-value pairs to apply to this DB parameter group.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6750decc9586f551a85785e78ff173b466d818fcc5f547bb41085dac31578506)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBParameterGroupProps(
            description=description,
            family=family,
            db_parameter_group_name=db_parameter_group_name,
            parameters=parameters,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2243dc21bd17537969215e71559587414acc20fd06e09179e66d3655ef676901)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4759dd05dbc70dfa4ecb9d827a1173b68f19ee557ca4e13b8f903905a554ebc1)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDbParameterGroupName")
    def attr_db_parameter_group_name(self) -> builtins.str:
        '''The name of the DB parameter group.

        :cloudformationAttribute: DBParameterGroupName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbParameterGroupName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _aws_cdk_core_f4b25747.TagManager:
        '''An optional array of key-value pairs to apply to this DB parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-tags
        '''
        return typing.cast(_aws_cdk_core_f4b25747.TagManager, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> builtins.str:
        '''Provides the customer-specified description for this DB parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-description
        '''
        return typing.cast(builtins.str, jsii.get(self, "description"))

    @description.setter
    def description(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aea04ce607b31db926f2ead857803f875fc4a80bd53f6ecb578f0cc370eeddfe)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value)

    @builtins.property
    @jsii.member(jsii_name="family")
    def family(self) -> builtins.str:
        '''The DB parameter group family name.

        A DB parameter group can be associated with one and only one DB parameter group family, and can be applied only to a DB instance running a DB engine and engine version compatible with that DB parameter group family.
        .. epigraph::

           The DB parameter group family can't be changed when updating a DB parameter group.

        To list all of the available parameter group families, use the following command:

        ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"``

        The output contains duplicates.

        For more information, see ``[CreateDBParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html)`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-family
        '''
        return typing.cast(builtins.str, jsii.get(self, "family"))

    @family.setter
    def family(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8071d2374c5199cdf341d642918a2d5077adaad969a3bf6777665217e7258cab)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "family", value)

    @builtins.property
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Any:
        '''An array of parameter names and values for the parameter update.

        At least one parameter name and value must be supplied. Subsequent arguments are optional.

        For more information about DB parameters and DB parameter groups for Amazon RDS DB engines, see `Working with DB Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon RDS User Guide* .

        For more information about DB cluster and DB instance parameters and parameter groups for Amazon Aurora DB engines, see `Working with DB Parameter Groups and DB Cluster Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon Aurora User Guide* .
        .. epigraph::

           AWS CloudFormation doesn't support specifying an apply method for each individual parameter. The default apply method for each parameter is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-parameters
        '''
        return typing.cast(typing.Any, jsii.get(self, "parameters"))

    @parameters.setter
    def parameters(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dfb0f6e255e194c8400cee67510afbc98985b82dfa0f48d6b673e36dfc84e194)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "parameters", value)

    @builtins.property
    @jsii.member(jsii_name="dbParameterGroupName")
    def db_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB parameter group.

        Constraints:

        - Must be 1 to 255 letters, numbers, or hyphens.
        - First character must be a letter
        - Can't end with a hyphen or contain two consecutive hyphens

        If you don't specify a value for ``DBParameterGroupName`` property, a name is automatically created for the DB parameter group.
        .. epigraph::

           This value is stored as a lowercase string.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-dbparametergroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbParameterGroupName"))

    @db_parameter_group_name.setter
    def db_parameter_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d05f893ba193ab806869b417d2122874bf9ec913239aadf503a6ec85a6596ede)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbParameterGroupName", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBParameterGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "family": "family",
        "db_parameter_group_name": "dbParameterGroupName",
        "parameters": "parameters",
        "tags": "tags",
    },
)
class CfnDBParameterGroupProps:
    def __init__(
        self,
        *,
        description: builtins.str,
        family: builtins.str,
        db_parameter_group_name: typing.Optional[builtins.str] = None,
        parameters: typing.Any = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBParameterGroup``.

        :param description: Provides the customer-specified description for this DB parameter group.
        :param family: The DB parameter group family name. A DB parameter group can be associated with one and only one DB parameter group family, and can be applied only to a DB instance running a DB engine and engine version compatible with that DB parameter group family. .. epigraph:: The DB parameter group family can't be changed when updating a DB parameter group. To list all of the available parameter group families, use the following command: ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"`` The output contains duplicates. For more information, see ``[CreateDBParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html)`` .
        :param db_parameter_group_name: The name of the DB parameter group. Constraints: - Must be 1 to 255 letters, numbers, or hyphens. - First character must be a letter - Can't end with a hyphen or contain two consecutive hyphens If you don't specify a value for ``DBParameterGroupName`` property, a name is automatically created for the DB parameter group. .. epigraph:: This value is stored as a lowercase string.
        :param parameters: An array of parameter names and values for the parameter update. At least one parameter name and value must be supplied. Subsequent arguments are optional. For more information about DB parameters and DB parameter groups for Amazon RDS DB engines, see `Working with DB Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon RDS User Guide* . For more information about DB cluster and DB instance parameters and parameter groups for Amazon Aurora DB engines, see `Working with DB Parameter Groups and DB Cluster Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon Aurora User Guide* . .. epigraph:: AWS CloudFormation doesn't support specifying an apply method for each individual parameter. The default apply method for each parameter is used.
        :param tags: An optional array of key-value pairs to apply to this DB parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            # parameters: Any
            
            cfn_dBParameter_group_props = rds.CfnDBParameterGroupProps(
                description="description",
                family="family",
            
                # the properties below are optional
                db_parameter_group_name="dbParameterGroupName",
                parameters=parameters,
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__69fa1f5c8876359dd30265d35f6cc9d88c6114370ace05cda75b48b06579cc1d)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument family", value=family, expected_type=type_hints["family"])
            check_type(argname="argument db_parameter_group_name", value=db_parameter_group_name, expected_type=type_hints["db_parameter_group_name"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "description": description,
            "family": family,
        }
        if db_parameter_group_name is not None:
            self._values["db_parameter_group_name"] = db_parameter_group_name
        if parameters is not None:
            self._values["parameters"] = parameters
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def description(self) -> builtins.str:
        '''Provides the customer-specified description for this DB parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-description
        '''
        result = self._values.get("description")
        assert result is not None, "Required property 'description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def family(self) -> builtins.str:
        '''The DB parameter group family name.

        A DB parameter group can be associated with one and only one DB parameter group family, and can be applied only to a DB instance running a DB engine and engine version compatible with that DB parameter group family.
        .. epigraph::

           The DB parameter group family can't be changed when updating a DB parameter group.

        To list all of the available parameter group families, use the following command:

        ``aws rds describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"``

        The output contains duplicates.

        For more information, see ``[CreateDBParameterGroup](https://docs.aws.amazon.com//AmazonRDS/latest/APIReference/API_CreateDBParameterGroup.html)`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-family
        '''
        result = self._values.get("family")
        assert result is not None, "Required property 'family' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def db_parameter_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the DB parameter group.

        Constraints:

        - Must be 1 to 255 letters, numbers, or hyphens.
        - First character must be a letter
        - Can't end with a hyphen or contain two consecutive hyphens

        If you don't specify a value for ``DBParameterGroupName`` property, a name is automatically created for the DB parameter group.
        .. epigraph::

           This value is stored as a lowercase string.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-dbparametergroupname
        '''
        result = self._values.get("db_parameter_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(self) -> typing.Any:
        '''An array of parameter names and values for the parameter update.

        At least one parameter name and value must be supplied. Subsequent arguments are optional.

        For more information about DB parameters and DB parameter groups for Amazon RDS DB engines, see `Working with DB Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon RDS User Guide* .

        For more information about DB cluster and DB instance parameters and parameter groups for Amazon Aurora DB engines, see `Working with DB Parameter Groups and DB Cluster Parameter Groups <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_WorkingWithParamGroups.html>`_ in the *Amazon Aurora User Guide* .
        .. epigraph::

           AWS CloudFormation doesn't support specifying an apply method for each individual parameter. The default apply method for each parameter is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Any, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]]:
        '''An optional array of key-value pairs to apply to this DB parameter group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbparametergroup.html#cfn-rds-dbparametergroup-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBParameterGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnDBProxy(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBProxy",
):
    '''A CloudFormation ``AWS::RDS::DBProxy``.

    The ``AWS::RDS::DBProxy`` resource creates or updates a DB proxy.

    For information about RDS Proxy for Amazon RDS, see `Managing Connections with Amazon RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html>`_ in the *Amazon RDS User Guide* .

    For information about RDS Proxy for Amazon Aurora, see `Managing Connections with Amazon RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html>`_ in the *Amazon Aurora User Guide* .
    .. epigraph::

       Limitations apply to RDS Proxy, including DB engine version limitations and AWS Region limitations.

       For information about limitations that apply to RDS Proxy for Amazon RDS, see `Limitations for RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy.limitations>`_ in the *Amazon RDS User Guide* .

       For information about that apply to RDS Proxy for Amazon Aurora, see `Limitations for RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html#rds-proxy.limitations>`_ in the *Amazon Aurora User Guide* .

    :cloudformationResource: AWS::RDS::DBProxy
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        cfn_dBProxy = rds.CfnDBProxy(self, "MyCfnDBProxy",
            auth=[rds.CfnDBProxy.AuthFormatProperty(
                auth_scheme="authScheme",
                client_password_auth_type="clientPasswordAuthType",
                description="description",
                iam_auth="iamAuth",
                secret_arn="secretArn",
                user_name="userName"
            )],
            db_proxy_name="dbProxyName",
            engine_family="engineFamily",
            role_arn="roleArn",
            vpc_subnet_ids=["vpcSubnetIds"],
        
            # the properties below are optional
            debug_logging=False,
            idle_client_timeout=123,
            require_tls=False,
            tags=[rds.CfnDBProxy.TagFormatProperty(
                key="key",
                value="value"
            )],
            vpc_security_group_ids=["vpcSecurityGroupIds"]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        auth: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDBProxy.AuthFormatProperty", typing.Dict[builtins.str, typing.Any]]]]],
        db_proxy_name: builtins.str,
        engine_family: builtins.str,
        role_arn: builtins.str,
        vpc_subnet_ids: typing.Sequence[builtins.str],
        debug_logging: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        idle_client_timeout: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union["CfnDBProxy.TagFormatProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        vpc_security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBProxy``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param auth: The authorization mechanism that the proxy uses.
        :param db_proxy_name: The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified AWS Region . An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens.
        :param engine_family: The kinds of databases that the proxy can connect to. This value determines which database network protocol the proxy recognizes when it interprets network traffic to and from the database. For Aurora MySQL, RDS for MariaDB, and RDS for MySQL databases, specify ``MYSQL`` . For Aurora PostgreSQL and RDS for PostgreSQL databases, specify ``POSTGRESQL`` . For RDS for Microsoft SQL Server, specify ``SQLSERVER`` . *Valid values* : ``MYSQL`` | ``POSTGRESQL`` | ``SQLSERVER``
        :param role_arn: The Amazon Resource Name (ARN) of the IAM role that the proxy uses to access secrets in AWS Secrets Manager.
        :param vpc_subnet_ids: One or more VPC subnet IDs to associate with the new proxy.
        :param debug_logging: Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs.
        :param idle_client_timeout: The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it. You can set this value higher or lower than the connection timeout limit for the associated database.
        :param require_tls: A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy. By enabling this setting, you can enforce encrypted TLS connections to the proxy.
        :param tags: An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.
        :param vpc_security_group_ids: One or more VPC security group IDs to associate with the new proxy. If you plan to update the resource, don't specify VPC security groups in a shared VPC.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__373873f28cb5087c50682df65265fc4c4156f3d30d28b4a667a8b22d35315edd)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBProxyProps(
            auth=auth,
            db_proxy_name=db_proxy_name,
            engine_family=engine_family,
            role_arn=role_arn,
            vpc_subnet_ids=vpc_subnet_ids,
            debug_logging=debug_logging,
            idle_client_timeout=idle_client_timeout,
            require_tls=require_tls,
            tags=tags,
            vpc_security_group_ids=vpc_security_group_ids,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__640d300990f70de7798f46641f72659dc51e52f1f7a1311dc051d486b9511695)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b75624ddb4eef74f75e2dfd0d9efe9abcb0e4b7b8b4c9c35e237e2540c21d7c7)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDbProxyArn")
    def attr_db_proxy_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) representing the target group.

        :cloudformationAttribute: DBProxyArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbProxyArn"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpoint")
    def attr_endpoint(self) -> builtins.str:
        '''The writer endpoint for the RDS DB instance or Aurora DB cluster.

        :cloudformationAttribute: Endpoint
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpoint"))

    @builtins.property
    @jsii.member(jsii_name="attrVpcId")
    def attr_vpc_id(self) -> builtins.str:
        '''The VPC ID to associate with the DB proxy.

        :cloudformationAttribute: VpcId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVpcId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="auth")
    def auth(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBProxy.AuthFormatProperty"]]]:
        '''The authorization mechanism that the proxy uses.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-auth
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBProxy.AuthFormatProperty"]]], jsii.get(self, "auth"))

    @auth.setter
    def auth(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBProxy.AuthFormatProperty"]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d49dd6af1c9d1fc322614a3d27e05dde69b2d72565453af035211c797ffbaabf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "auth", value)

    @builtins.property
    @jsii.member(jsii_name="dbProxyName")
    def db_proxy_name(self) -> builtins.str:
        '''The identifier for the proxy.

        This name must be unique for all proxies owned by your AWS account in the specified AWS Region . An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-dbproxyname
        '''
        return typing.cast(builtins.str, jsii.get(self, "dbProxyName"))

    @db_proxy_name.setter
    def db_proxy_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3ef836132a73853e53dc41faeddd1f75780d177d257990bbb682097c77c7cb9e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbProxyName", value)

    @builtins.property
    @jsii.member(jsii_name="engineFamily")
    def engine_family(self) -> builtins.str:
        '''The kinds of databases that the proxy can connect to.

        This value determines which database network protocol the proxy recognizes when it interprets network traffic to and from the database. For Aurora MySQL, RDS for MariaDB, and RDS for MySQL databases, specify ``MYSQL`` . For Aurora PostgreSQL and RDS for PostgreSQL databases, specify ``POSTGRESQL`` . For RDS for Microsoft SQL Server, specify ``SQLSERVER`` .

        *Valid values* : ``MYSQL`` | ``POSTGRESQL`` | ``SQLSERVER``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-enginefamily
        '''
        return typing.cast(builtins.str, jsii.get(self, "engineFamily"))

    @engine_family.setter
    def engine_family(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d9c13fa60fb54dd733069decf47a39578934e000bfe80ac15c954640439b4212)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engineFamily", value)

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the IAM role that the proxy uses to access secrets in AWS Secrets Manager.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-rolearn
        '''
        return typing.cast(builtins.str, jsii.get(self, "roleArn"))

    @role_arn.setter
    def role_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__62206d4977bfcf5c056cbc41b8c864b20026021a3bda1ab156089e661627cf5a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "roleArn", value)

    @builtins.property
    @jsii.member(jsii_name="vpcSubnetIds")
    def vpc_subnet_ids(self) -> typing.List[builtins.str]:
        '''One or more VPC subnet IDs to associate with the new proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-vpcsubnetids
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "vpcSubnetIds"))

    @vpc_subnet_ids.setter
    def vpc_subnet_ids(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e1cbdd6ee4b45d52bec4be29c964f231ac1d764ba394b81f18d91d5ddb564608)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcSubnetIds", value)

    @builtins.property
    @jsii.member(jsii_name="debugLogging")
    def debug_logging(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''Whether the proxy includes detailed information about SQL statements in its logs.

        This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-debuglogging
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "debugLogging"))

    @debug_logging.setter
    def debug_logging(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ad07002632a37a2e3d902395caa9f74ada5f7f896bb3d10d0e5bb69f3fdf3498)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "debugLogging", value)

    @builtins.property
    @jsii.member(jsii_name="idleClientTimeout")
    def idle_client_timeout(self) -> typing.Optional[jsii.Number]:
        '''The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it.

        You can set this value higher or lower than the connection timeout limit for the associated database.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-idleclienttimeout
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "idleClientTimeout"))

    @idle_client_timeout.setter
    def idle_client_timeout(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__549fcfe8f364b5b7aef4b7a6acbc68010d895d2fce6775c0538d89cc710d0ee3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "idleClientTimeout", value)

    @builtins.property
    @jsii.member(jsii_name="requireTls")
    def require_tls(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy.

        By enabling this setting, you can enforce encrypted TLS connections to the proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-requiretls
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "requireTls"))

    @require_tls.setter
    def require_tls(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__72cb81cd333abc249defe7382fae5def91310c969b001c71b9fa56aec78d18b2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requireTls", value)

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.List["CfnDBProxy.TagFormatProperty"]]:
        '''An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-tags
        '''
        return typing.cast(typing.Optional[typing.List["CfnDBProxy.TagFormatProperty"]], jsii.get(self, "tags"))

    @tags.setter
    def tags(
        self,
        value: typing.Optional[typing.List["CfnDBProxy.TagFormatProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eaffe0bfc4e57599c6e2c600fcfc7e48e166b9e8924ff7a6bc43496c4fa83cbb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value)

    @builtins.property
    @jsii.member(jsii_name="vpcSecurityGroupIds")
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more VPC security group IDs to associate with the new proxy.

        If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-vpcsecuritygroupids
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "vpcSecurityGroupIds"))

    @vpc_security_group_ids.setter
    def vpc_security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a72a22a1f498d496a461634563331f371742d823f9ec2a2954a1eb33804b3181)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcSecurityGroupIds", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBProxy.AuthFormatProperty",
        jsii_struct_bases=[],
        name_mapping={
            "auth_scheme": "authScheme",
            "client_password_auth_type": "clientPasswordAuthType",
            "description": "description",
            "iam_auth": "iamAuth",
            "secret_arn": "secretArn",
            "user_name": "userName",
        },
    )
    class AuthFormatProperty:
        def __init__(
            self,
            *,
            auth_scheme: typing.Optional[builtins.str] = None,
            client_password_auth_type: typing.Optional[builtins.str] = None,
            description: typing.Optional[builtins.str] = None,
            iam_auth: typing.Optional[builtins.str] = None,
            secret_arn: typing.Optional[builtins.str] = None,
            user_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies the details of authentication used by a proxy to log in as a specific database user.

            :param auth_scheme: The type of authentication that the proxy uses for connections from the proxy to the underlying database. Valid Values: ``SECRETS``
            :param client_password_auth_type: ``CfnDBProxy.AuthFormatProperty.ClientPasswordAuthType``.
            :param description: A user-specified description about the authentication used by a proxy to log in as a specific database user.
            :param iam_auth: Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy. The ``ENABLED`` value is valid only for proxies with RDS for Microsoft SQL Server. Valid Values: ``ENABLED | DISABLED | REQUIRED``
            :param secret_arn: The Amazon Resource Name (ARN) representing the secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster. These secrets are stored within Amazon Secrets Manager.
            :param user_name: The name of the database user to which the proxy connects.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                auth_format_property = rds.CfnDBProxy.AuthFormatProperty(
                    auth_scheme="authScheme",
                    client_password_auth_type="clientPasswordAuthType",
                    description="description",
                    iam_auth="iamAuth",
                    secret_arn="secretArn",
                    user_name="userName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b3e1383760bdc14f1d49656f6c5663b737aa0f124ad6273693010c4ef8fdf79c)
                check_type(argname="argument auth_scheme", value=auth_scheme, expected_type=type_hints["auth_scheme"])
                check_type(argname="argument client_password_auth_type", value=client_password_auth_type, expected_type=type_hints["client_password_auth_type"])
                check_type(argname="argument description", value=description, expected_type=type_hints["description"])
                check_type(argname="argument iam_auth", value=iam_auth, expected_type=type_hints["iam_auth"])
                check_type(argname="argument secret_arn", value=secret_arn, expected_type=type_hints["secret_arn"])
                check_type(argname="argument user_name", value=user_name, expected_type=type_hints["user_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if auth_scheme is not None:
                self._values["auth_scheme"] = auth_scheme
            if client_password_auth_type is not None:
                self._values["client_password_auth_type"] = client_password_auth_type
            if description is not None:
                self._values["description"] = description
            if iam_auth is not None:
                self._values["iam_auth"] = iam_auth
            if secret_arn is not None:
                self._values["secret_arn"] = secret_arn
            if user_name is not None:
                self._values["user_name"] = user_name

        @builtins.property
        def auth_scheme(self) -> typing.Optional[builtins.str]:
            '''The type of authentication that the proxy uses for connections from the proxy to the underlying database.

            Valid Values: ``SECRETS``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-authscheme
            '''
            result = self._values.get("auth_scheme")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def client_password_auth_type(self) -> typing.Optional[builtins.str]:
            '''``CfnDBProxy.AuthFormatProperty.ClientPasswordAuthType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-clientpasswordauthtype
            '''
            result = self._values.get("client_password_auth_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''A user-specified description about the authentication used by a proxy to log in as a specific database user.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def iam_auth(self) -> typing.Optional[builtins.str]:
            '''Whether to require or disallow AWS Identity and Access Management (IAM) authentication for connections to the proxy.

            The ``ENABLED`` value is valid only for proxies with RDS for Microsoft SQL Server.

            Valid Values: ``ENABLED | DISABLED | REQUIRED``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-iamauth
            '''
            result = self._values.get("iam_auth")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def secret_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) representing the secret that the proxy uses to authenticate to the RDS DB instance or Aurora DB cluster.

            These secrets are stored within Amazon Secrets Manager.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-secretarn
            '''
            result = self._values.get("secret_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def user_name(self) -> typing.Optional[builtins.str]:
            '''The name of the database user to which the proxy connects.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-authformat.html#cfn-rds-dbproxy-authformat-username
            '''
            result = self._values.get("user_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AuthFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBProxy.TagFormatProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class TagFormatProperty:
        def __init__(
            self,
            *,
            key: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Metadata assigned to a DB proxy consisting of a key-value pair.

            :param key: A key is the required name of the tag. The string value can be 1-128 Unicode characters in length and can't be prefixed with ``aws:`` . The string can contain only the set of Unicode letters, digits, white-space, '*', '.', '/', '=', '+', '-' (Java regex: "^([\\p{L}\\p{Z}\\p{N}*.:/=+\\-]*)$").
            :param value: A value is the optional value of the tag. The string value can be 1-256 Unicode characters in length and can't be prefixed with ``aws:`` . The string can contain only the set of Unicode letters, digits, white-space, '*', '.', '/', '=', '+', '-' (Java regex: "^([\\p{L}\\p{Z}\\p{N}*.:/=+\\-]*)$").

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-tagformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                tag_format_property = rds.CfnDBProxy.TagFormatProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f68733838aafaa11776717c370ac9fb3b0e7302bb61c0e85637e729d9b98b251)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if key is not None:
                self._values["key"] = key
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def key(self) -> typing.Optional[builtins.str]:
            '''A key is the required name of the tag.

            The string value can be 1-128 Unicode characters in length and can't be prefixed with ``aws:`` . The string can contain only the set of Unicode letters, digits, white-space, '*', '.', '/', '=', '+', '-' (Java regex: "^([\\p{L}\\p{Z}\\p{N}*.:/=+-]*)$").

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-tagformat.html#cfn-rds-dbproxy-tagformat-key
            '''
            result = self._values.get("key")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''A value is the optional value of the tag.

            The string value can be 1-256 Unicode characters in length and can't be prefixed with ``aws:`` . The string can contain only the set of Unicode letters, digits, white-space, '*', '.', '/', '=', '+', '-' (Java regex: "^([\\p{L}\\p{Z}\\p{N}*.:/=+-]*)$").

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxy-tagformat.html#cfn-rds-dbproxy-tagformat-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnDBProxyEndpoint(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBProxyEndpoint",
):
    '''A CloudFormation ``AWS::RDS::DBProxyEndpoint``.

    The ``AWS::RDS::DBProxyEndpoint`` resource creates or updates a DB proxy endpoint. You can use custom proxy endpoints to access a proxy through a different VPC than the proxy's default VPC.

    For more information about RDS Proxy, see `AWS::RDS::DBProxy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html>`_ .

    :cloudformationResource: AWS::RDS::DBProxyEndpoint
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        cfn_dBProxy_endpoint = rds.CfnDBProxyEndpoint(self, "MyCfnDBProxyEndpoint",
            db_proxy_endpoint_name="dbProxyEndpointName",
            db_proxy_name="dbProxyName",
            vpc_subnet_ids=["vpcSubnetIds"],
        
            # the properties below are optional
            tags=[rds.CfnDBProxyEndpoint.TagFormatProperty(
                key="key",
                value="value"
            )],
            target_role="targetRole",
            vpc_security_group_ids=["vpcSecurityGroupIds"]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        db_proxy_endpoint_name: builtins.str,
        db_proxy_name: builtins.str,
        vpc_subnet_ids: typing.Sequence[builtins.str],
        tags: typing.Optional[typing.Sequence[typing.Union["CfnDBProxyEndpoint.TagFormatProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        target_role: typing.Optional[builtins.str] = None,
        vpc_security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBProxyEndpoint``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_proxy_endpoint_name: The name of the DB proxy endpoint to create.
        :param db_proxy_name: The name of the DB proxy associated with the DB proxy endpoint that you create.
        :param vpc_subnet_ids: The VPC subnet IDs for the DB proxy endpoint that you create. You can specify a different set of subnet IDs than for the original DB proxy.
        :param tags: An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.
        :param target_role: A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations. Valid Values: ``READ_WRITE | READ_ONLY``
        :param vpc_security_group_ids: The VPC security group IDs for the DB proxy endpoint that you create. You can specify a different set of security group IDs than for the original DB proxy. The default is the default security group for the VPC.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__227203e90b5cde07f764e055389de0937ad35053d38a31129795e9902be7b61f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBProxyEndpointProps(
            db_proxy_endpoint_name=db_proxy_endpoint_name,
            db_proxy_name=db_proxy_name,
            vpc_subnet_ids=vpc_subnet_ids,
            tags=tags,
            target_role=target_role,
            vpc_security_group_ids=vpc_security_group_ids,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3b089384555d10eb40f7233b944524b38616e3eb2ce5f6d4adeaf7658cab14dd)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__38a7f2f9d39089db422de913096ff8675cd96a71b15b569788b5504f5b861886)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDbProxyEndpointArn")
    def attr_db_proxy_endpoint_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) representing the DB proxy endpoint.

        :cloudformationAttribute: DBProxyEndpointArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDbProxyEndpointArn"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpoint")
    def attr_endpoint(self) -> builtins.str:
        '''The custom endpoint for the RDS DB instance or Aurora DB cluster.

        :cloudformationAttribute: Endpoint
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpoint"))

    @builtins.property
    @jsii.member(jsii_name="attrIsDefault")
    def attr_is_default(self) -> _aws_cdk_core_f4b25747.IResolvable:
        '''A value that indicates whether this endpoint is the default endpoint for the associated DB proxy.

        Default DB proxy endpoints always have read/write capability. Other endpoints that you associate with the DB proxy can be either read/write or read-only.

        :cloudformationAttribute: IsDefault
        '''
        return typing.cast(_aws_cdk_core_f4b25747.IResolvable, jsii.get(self, "attrIsDefault"))

    @builtins.property
    @jsii.member(jsii_name="attrVpcId")
    def attr_vpc_id(self) -> builtins.str:
        '''The VPC ID of the DB proxy endpoint.

        :cloudformationAttribute: VpcId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVpcId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="dbProxyEndpointName")
    def db_proxy_endpoint_name(self) -> builtins.str:
        '''The name of the DB proxy endpoint to create.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-dbproxyendpointname
        '''
        return typing.cast(builtins.str, jsii.get(self, "dbProxyEndpointName"))

    @db_proxy_endpoint_name.setter
    def db_proxy_endpoint_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6ec1487d4fb6148b0af74ccce7a61f41bf3293125b698c5e8c3247a691f076e8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbProxyEndpointName", value)

    @builtins.property
    @jsii.member(jsii_name="dbProxyName")
    def db_proxy_name(self) -> builtins.str:
        '''The name of the DB proxy associated with the DB proxy endpoint that you create.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-dbproxyname
        '''
        return typing.cast(builtins.str, jsii.get(self, "dbProxyName"))

    @db_proxy_name.setter
    def db_proxy_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__907c77a39705745d5c051b254cb4ba33841bf06b02ee0fb2f3f2025bb3fc5446)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbProxyName", value)

    @builtins.property
    @jsii.member(jsii_name="vpcSubnetIds")
    def vpc_subnet_ids(self) -> typing.List[builtins.str]:
        '''The VPC subnet IDs for the DB proxy endpoint that you create.

        You can specify a different set of subnet IDs than for the original DB proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-vpcsubnetids
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "vpcSubnetIds"))

    @vpc_subnet_ids.setter
    def vpc_subnet_ids(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__246d2e87b8cf8c778e702407b17838fa2a5cecc16eacff8cc9f3ed7473965b4f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcSubnetIds", value)

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(
        self,
    ) -> typing.Optional[typing.List["CfnDBProxyEndpoint.TagFormatProperty"]]:
        '''An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-tags
        '''
        return typing.cast(typing.Optional[typing.List["CfnDBProxyEndpoint.TagFormatProperty"]], jsii.get(self, "tags"))

    @tags.setter
    def tags(
        self,
        value: typing.Optional[typing.List["CfnDBProxyEndpoint.TagFormatProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__69ef4532eac40033a12349f62b703476f08fb77716ef8063782f7a0e90bffa9c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value)

    @builtins.property
    @jsii.member(jsii_name="targetRole")
    def target_role(self) -> typing.Optional[builtins.str]:
        '''A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations.

        Valid Values: ``READ_WRITE | READ_ONLY``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-targetrole
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "targetRole"))

    @target_role.setter
    def target_role(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8f02c373611b760ca314fd34d46a6d46e6d7ae4f98bbd88b0230df693e6ddc5a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetRole", value)

    @builtins.property
    @jsii.member(jsii_name="vpcSecurityGroupIds")
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The VPC security group IDs for the DB proxy endpoint that you create.

        You can specify a different set of security group IDs than for the original DB proxy. The default is the default security group for the VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-vpcsecuritygroupids
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "vpcSecurityGroupIds"))

    @vpc_security_group_ids.setter
    def vpc_security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0cdba725a15093a3f1ff11c061b75d7a3854316822e55c565fb36ffffd50082e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcSecurityGroupIds", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBProxyEndpoint.TagFormatProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class TagFormatProperty:
        def __init__(
            self,
            *,
            key: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Metadata assigned to a DB proxy endpoint consisting of a key-value pair.

            :param key: A value is the optional value of the tag. The string value can be 1-256 Unicode characters in length and can't be prefixed with ``aws:`` . The string can contain only the set of Unicode letters, digits, white-space, '*', '.', '/', '=', '+', '-' (Java regex: "^([\\p{L}\\p{Z}\\p{N}*.:/=+\\-]*)$").
            :param value: Metadata assigned to a DB instance consisting of a key-value pair.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxyendpoint-tagformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                tag_format_property = rds.CfnDBProxyEndpoint.TagFormatProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c6a48fdb49832127c65a46ea35f5a24a8faade033bed26213837ed4fe4494c0c)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if key is not None:
                self._values["key"] = key
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def key(self) -> typing.Optional[builtins.str]:
            '''A value is the optional value of the tag.

            The string value can be 1-256 Unicode characters in length and can't be prefixed with ``aws:`` . The string can contain only the set of Unicode letters, digits, white-space, '*', '.', '/', '=', '+', '-' (Java regex: "^([\\p{L}\\p{Z}\\p{N}*.:/=+-]*)$").

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxyendpoint-tagformat.html#cfn-rds-dbproxyendpoint-tagformat-key
            '''
            result = self._values.get("key")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''Metadata assigned to a DB instance consisting of a key-value pair.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxyendpoint-tagformat.html#cfn-rds-dbproxyendpoint-tagformat-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBProxyEndpointProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_proxy_endpoint_name": "dbProxyEndpointName",
        "db_proxy_name": "dbProxyName",
        "vpc_subnet_ids": "vpcSubnetIds",
        "tags": "tags",
        "target_role": "targetRole",
        "vpc_security_group_ids": "vpcSecurityGroupIds",
    },
)
class CfnDBProxyEndpointProps:
    def __init__(
        self,
        *,
        db_proxy_endpoint_name: builtins.str,
        db_proxy_name: builtins.str,
        vpc_subnet_ids: typing.Sequence[builtins.str],
        tags: typing.Optional[typing.Sequence[typing.Union[CfnDBProxyEndpoint.TagFormatProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        target_role: typing.Optional[builtins.str] = None,
        vpc_security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBProxyEndpoint``.

        :param db_proxy_endpoint_name: The name of the DB proxy endpoint to create.
        :param db_proxy_name: The name of the DB proxy associated with the DB proxy endpoint that you create.
        :param vpc_subnet_ids: The VPC subnet IDs for the DB proxy endpoint that you create. You can specify a different set of subnet IDs than for the original DB proxy.
        :param tags: An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.
        :param target_role: A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations. Valid Values: ``READ_WRITE | READ_ONLY``
        :param vpc_security_group_ids: The VPC security group IDs for the DB proxy endpoint that you create. You can specify a different set of security group IDs than for the original DB proxy. The default is the default security group for the VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            cfn_dBProxy_endpoint_props = rds.CfnDBProxyEndpointProps(
                db_proxy_endpoint_name="dbProxyEndpointName",
                db_proxy_name="dbProxyName",
                vpc_subnet_ids=["vpcSubnetIds"],
            
                # the properties below are optional
                tags=[rds.CfnDBProxyEndpoint.TagFormatProperty(
                    key="key",
                    value="value"
                )],
                target_role="targetRole",
                vpc_security_group_ids=["vpcSecurityGroupIds"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__01a96abb75e170a8eba7d747c1d3f1a18e887fd6c1c1c90e19ee4491aa5937e4)
            check_type(argname="argument db_proxy_endpoint_name", value=db_proxy_endpoint_name, expected_type=type_hints["db_proxy_endpoint_name"])
            check_type(argname="argument db_proxy_name", value=db_proxy_name, expected_type=type_hints["db_proxy_name"])
            check_type(argname="argument vpc_subnet_ids", value=vpc_subnet_ids, expected_type=type_hints["vpc_subnet_ids"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument target_role", value=target_role, expected_type=type_hints["target_role"])
            check_type(argname="argument vpc_security_group_ids", value=vpc_security_group_ids, expected_type=type_hints["vpc_security_group_ids"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "db_proxy_endpoint_name": db_proxy_endpoint_name,
            "db_proxy_name": db_proxy_name,
            "vpc_subnet_ids": vpc_subnet_ids,
        }
        if tags is not None:
            self._values["tags"] = tags
        if target_role is not None:
            self._values["target_role"] = target_role
        if vpc_security_group_ids is not None:
            self._values["vpc_security_group_ids"] = vpc_security_group_ids

    @builtins.property
    def db_proxy_endpoint_name(self) -> builtins.str:
        '''The name of the DB proxy endpoint to create.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-dbproxyendpointname
        '''
        result = self._values.get("db_proxy_endpoint_name")
        assert result is not None, "Required property 'db_proxy_endpoint_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def db_proxy_name(self) -> builtins.str:
        '''The name of the DB proxy associated with the DB proxy endpoint that you create.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-dbproxyname
        '''
        result = self._values.get("db_proxy_name")
        assert result is not None, "Required property 'db_proxy_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def vpc_subnet_ids(self) -> typing.List[builtins.str]:
        '''The VPC subnet IDs for the DB proxy endpoint that you create.

        You can specify a different set of subnet IDs than for the original DB proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-vpcsubnetids
        '''
        result = self._values.get("vpc_subnet_ids")
        assert result is not None, "Required property 'vpc_subnet_ids' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def tags(
        self,
    ) -> typing.Optional[typing.List[CfnDBProxyEndpoint.TagFormatProperty]]:
        '''An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[CfnDBProxyEndpoint.TagFormatProperty]], result)

    @builtins.property
    def target_role(self) -> typing.Optional[builtins.str]:
        '''A value that indicates whether the DB proxy endpoint can be used for read/write or read-only operations.

        Valid Values: ``READ_WRITE | READ_ONLY``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-targetrole
        '''
        result = self._values.get("target_role")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The VPC security group IDs for the DB proxy endpoint that you create.

        You can specify a different set of security group IDs than for the original DB proxy. The default is the default security group for the VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxyendpoint.html#cfn-rds-dbproxyendpoint-vpcsecuritygroupids
        '''
        result = self._values.get("vpc_security_group_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBProxyEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBProxyProps",
    jsii_struct_bases=[],
    name_mapping={
        "auth": "auth",
        "db_proxy_name": "dbProxyName",
        "engine_family": "engineFamily",
        "role_arn": "roleArn",
        "vpc_subnet_ids": "vpcSubnetIds",
        "debug_logging": "debugLogging",
        "idle_client_timeout": "idleClientTimeout",
        "require_tls": "requireTls",
        "tags": "tags",
        "vpc_security_group_ids": "vpcSecurityGroupIds",
    },
)
class CfnDBProxyProps:
    def __init__(
        self,
        *,
        auth: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnDBProxy.AuthFormatProperty, typing.Dict[builtins.str, typing.Any]]]]],
        db_proxy_name: builtins.str,
        engine_family: builtins.str,
        role_arn: builtins.str,
        vpc_subnet_ids: typing.Sequence[builtins.str],
        debug_logging: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        idle_client_timeout: typing.Optional[jsii.Number] = None,
        require_tls: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[CfnDBProxy.TagFormatProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        vpc_security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBProxy``.

        :param auth: The authorization mechanism that the proxy uses.
        :param db_proxy_name: The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified AWS Region . An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens.
        :param engine_family: The kinds of databases that the proxy can connect to. This value determines which database network protocol the proxy recognizes when it interprets network traffic to and from the database. For Aurora MySQL, RDS for MariaDB, and RDS for MySQL databases, specify ``MYSQL`` . For Aurora PostgreSQL and RDS for PostgreSQL databases, specify ``POSTGRESQL`` . For RDS for Microsoft SQL Server, specify ``SQLSERVER`` . *Valid values* : ``MYSQL`` | ``POSTGRESQL`` | ``SQLSERVER``
        :param role_arn: The Amazon Resource Name (ARN) of the IAM role that the proxy uses to access secrets in AWS Secrets Manager.
        :param vpc_subnet_ids: One or more VPC subnet IDs to associate with the new proxy.
        :param debug_logging: Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs.
        :param idle_client_timeout: The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it. You can set this value higher or lower than the connection timeout limit for the associated database.
        :param require_tls: A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy. By enabling this setting, you can enforce encrypted TLS connections to the proxy.
        :param tags: An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.
        :param vpc_security_group_ids: One or more VPC security group IDs to associate with the new proxy. If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            cfn_dBProxy_props = rds.CfnDBProxyProps(
                auth=[rds.CfnDBProxy.AuthFormatProperty(
                    auth_scheme="authScheme",
                    client_password_auth_type="clientPasswordAuthType",
                    description="description",
                    iam_auth="iamAuth",
                    secret_arn="secretArn",
                    user_name="userName"
                )],
                db_proxy_name="dbProxyName",
                engine_family="engineFamily",
                role_arn="roleArn",
                vpc_subnet_ids=["vpcSubnetIds"],
            
                # the properties below are optional
                debug_logging=False,
                idle_client_timeout=123,
                require_tls=False,
                tags=[rds.CfnDBProxy.TagFormatProperty(
                    key="key",
                    value="value"
                )],
                vpc_security_group_ids=["vpcSecurityGroupIds"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c3108e6674ac8b78e0075c0bc125e135dfc58e4c878c145b8c9918f9d87daadf)
            check_type(argname="argument auth", value=auth, expected_type=type_hints["auth"])
            check_type(argname="argument db_proxy_name", value=db_proxy_name, expected_type=type_hints["db_proxy_name"])
            check_type(argname="argument engine_family", value=engine_family, expected_type=type_hints["engine_family"])
            check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
            check_type(argname="argument vpc_subnet_ids", value=vpc_subnet_ids, expected_type=type_hints["vpc_subnet_ids"])
            check_type(argname="argument debug_logging", value=debug_logging, expected_type=type_hints["debug_logging"])
            check_type(argname="argument idle_client_timeout", value=idle_client_timeout, expected_type=type_hints["idle_client_timeout"])
            check_type(argname="argument require_tls", value=require_tls, expected_type=type_hints["require_tls"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument vpc_security_group_ids", value=vpc_security_group_ids, expected_type=type_hints["vpc_security_group_ids"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "auth": auth,
            "db_proxy_name": db_proxy_name,
            "engine_family": engine_family,
            "role_arn": role_arn,
            "vpc_subnet_ids": vpc_subnet_ids,
        }
        if debug_logging is not None:
            self._values["debug_logging"] = debug_logging
        if idle_client_timeout is not None:
            self._values["idle_client_timeout"] = idle_client_timeout
        if require_tls is not None:
            self._values["require_tls"] = require_tls
        if tags is not None:
            self._values["tags"] = tags
        if vpc_security_group_ids is not None:
            self._values["vpc_security_group_ids"] = vpc_security_group_ids

    @builtins.property
    def auth(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBProxy.AuthFormatProperty]]]:
        '''The authorization mechanism that the proxy uses.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-auth
        '''
        result = self._values.get("auth")
        assert result is not None, "Required property 'auth' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBProxy.AuthFormatProperty]]], result)

    @builtins.property
    def db_proxy_name(self) -> builtins.str:
        '''The identifier for the proxy.

        This name must be unique for all proxies owned by your AWS account in the specified AWS Region . An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens; it can't end with a hyphen or contain two consecutive hyphens.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-dbproxyname
        '''
        result = self._values.get("db_proxy_name")
        assert result is not None, "Required property 'db_proxy_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def engine_family(self) -> builtins.str:
        '''The kinds of databases that the proxy can connect to.

        This value determines which database network protocol the proxy recognizes when it interprets network traffic to and from the database. For Aurora MySQL, RDS for MariaDB, and RDS for MySQL databases, specify ``MYSQL`` . For Aurora PostgreSQL and RDS for PostgreSQL databases, specify ``POSTGRESQL`` . For RDS for Microsoft SQL Server, specify ``SQLSERVER`` .

        *Valid values* : ``MYSQL`` | ``POSTGRESQL`` | ``SQLSERVER``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-enginefamily
        '''
        result = self._values.get("engine_family")
        assert result is not None, "Required property 'engine_family' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the IAM role that the proxy uses to access secrets in AWS Secrets Manager.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-rolearn
        '''
        result = self._values.get("role_arn")
        assert result is not None, "Required property 'role_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def vpc_subnet_ids(self) -> typing.List[builtins.str]:
        '''One or more VPC subnet IDs to associate with the new proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-vpcsubnetids
        '''
        result = self._values.get("vpc_subnet_ids")
        assert result is not None, "Required property 'vpc_subnet_ids' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def debug_logging(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''Whether the proxy includes detailed information about SQL statements in its logs.

        This information helps you to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting when needed for debugging, and only when you have security measures in place to safeguard any sensitive information that appears in the logs.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-debuglogging
        '''
        result = self._values.get("debug_logging")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def idle_client_timeout(self) -> typing.Optional[jsii.Number]:
        '''The number of seconds that a connection to the proxy can be inactive before the proxy disconnects it.

        You can set this value higher or lower than the connection timeout limit for the associated database.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-idleclienttimeout
        '''
        result = self._values.get("idle_client_timeout")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def require_tls(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A Boolean parameter that specifies whether Transport Layer Security (TLS) encryption is required for connections to the proxy.

        By enabling this setting, you can enforce encrypted TLS connections to the proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-requiretls
        '''
        result = self._values.get("require_tls")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[CfnDBProxy.TagFormatProperty]]:
        '''An optional set of key-value pairs to associate arbitrary data of your choosing with the proxy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[CfnDBProxy.TagFormatProperty]], result)

    @builtins.property
    def vpc_security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more VPC security group IDs to associate with the new proxy.

        If you plan to update the resource, don't specify VPC security groups in a shared VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#cfn-rds-dbproxy-vpcsecuritygroupids
        '''
        result = self._values.get("vpc_security_group_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBProxyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnDBProxyTargetGroup(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBProxyTargetGroup",
):
    '''A CloudFormation ``AWS::RDS::DBProxyTargetGroup``.

    The ``AWS::RDS::DBProxyTargetGroup`` resource represents a set of RDS DB instances, Aurora DB clusters, or both that a proxy can connect to. Currently, each target group is associated with exactly one RDS DB instance or Aurora DB cluster.

    This data type is used as a response element in the ``DescribeDBProxyTargetGroups`` action.

    For information about RDS Proxy for Amazon RDS, see `Managing Connections with Amazon RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html>`_ in the *Amazon RDS User Guide* .

    For information about RDS Proxy for Amazon Aurora, see `Managing Connections with Amazon RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html>`_ in the *Amazon Aurora User Guide* .

    For a sample template that creates a DB proxy and registers a DB instance, see `Examples <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxy.html#aws-resource-rds-dbproxy--examples>`_ in AWS::RDS::DBProxy.
    .. epigraph::

       Limitations apply to RDS Proxy, including DB engine version limitations and AWS Region limitations.

       For information about limitations that apply to RDS Proxy for Amazon RDS, see `Limitations for RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy.limitations>`_ in the *Amazon RDS User Guide* .

       For information about that apply to RDS Proxy for Amazon Aurora, see `Limitations for RDS Proxy <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html#rds-proxy.limitations>`_ in the *Amazon Aurora User Guide* .

    :cloudformationResource: AWS::RDS::DBProxyTargetGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        cfn_dBProxy_target_group = rds.CfnDBProxyTargetGroup(self, "MyCfnDBProxyTargetGroup",
            db_proxy_name="dbProxyName",
            target_group_name="targetGroupName",
        
            # the properties below are optional
            connection_pool_configuration_info=rds.CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty(
                connection_borrow_timeout=123,
                init_query="initQuery",
                max_connections_percent=123,
                max_idle_connections_percent=123,
                session_pinning_filters=["sessionPinningFilters"]
            ),
            db_cluster_identifiers=["dbClusterIdentifiers"],
            db_instance_identifiers=["dbInstanceIdentifiers"]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        db_proxy_name: builtins.str,
        target_group_name: builtins.str,
        connection_pool_configuration_info: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        db_cluster_identifiers: typing.Optional[typing.Sequence[builtins.str]] = None,
        db_instance_identifiers: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBProxyTargetGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_proxy_name: The identifier of the ``DBProxy`` that is associated with the ``DBProxyTargetGroup`` .
        :param target_group_name: The identifier for the target group. .. epigraph:: Currently, this property must be set to ``default`` .
        :param connection_pool_configuration_info: Settings that control the size and behavior of the connection pool associated with a ``DBProxyTargetGroup`` .
        :param db_cluster_identifiers: One or more DB cluster identifiers.
        :param db_instance_identifiers: One or more DB instance identifiers.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eb8088785dc40f5dfd2e46d6798229b2879e92c2278bc23be304b2e9dca958d0)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBProxyTargetGroupProps(
            db_proxy_name=db_proxy_name,
            target_group_name=target_group_name,
            connection_pool_configuration_info=connection_pool_configuration_info,
            db_cluster_identifiers=db_cluster_identifiers,
            db_instance_identifiers=db_instance_identifiers,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__98b0741af1fc77a7ae200941c1bb201dd7ae55835e6ae9c64178392777ea2c8d)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6dbec70ce3ef8007665a15e392c322a9e6be73df40deb96cb6ab6fcb1b3b9b96)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrTargetGroupArn")
    def attr_target_group_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) representing the target group.

        :cloudformationAttribute: TargetGroupArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrTargetGroupArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="dbProxyName")
    def db_proxy_name(self) -> builtins.str:
        '''The identifier of the ``DBProxy`` that is associated with the ``DBProxyTargetGroup`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbproxyname
        '''
        return typing.cast(builtins.str, jsii.get(self, "dbProxyName"))

    @db_proxy_name.setter
    def db_proxy_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__efcbcdc79eb6db3995af3e72f402d769e9f8625b5a25408dde86021e1a094797)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbProxyName", value)

    @builtins.property
    @jsii.member(jsii_name="targetGroupName")
    def target_group_name(self) -> builtins.str:
        '''The identifier for the target group.

        .. epigraph::

           Currently, this property must be set to ``default`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-targetgroupname
        '''
        return typing.cast(builtins.str, jsii.get(self, "targetGroupName"))

    @target_group_name.setter
    def target_group_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__98ceca502125baff21a90ab4b8bf834d1943bd968831520ea412ba62857e8a97)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="connectionPoolConfigurationInfo")
    def connection_pool_configuration_info(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty"]]:
        '''Settings that control the size and behavior of the connection pool associated with a ``DBProxyTargetGroup`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfo
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty"]], jsii.get(self, "connectionPoolConfigurationInfo"))

    @connection_pool_configuration_info.setter
    def connection_pool_configuration_info(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__01ee43a4a027535f796a6b111229152a18d2751968e9f1ce20e44d1a01f5eb96)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "connectionPoolConfigurationInfo", value)

    @builtins.property
    @jsii.member(jsii_name="dbClusterIdentifiers")
    def db_cluster_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more DB cluster identifiers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbclusteridentifiers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "dbClusterIdentifiers"))

    @db_cluster_identifiers.setter
    def db_cluster_identifiers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f58e488268e9a0ce317329c7450ba465b0163da4d6aa62f1e9e72d9fcf561273)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbClusterIdentifiers", value)

    @builtins.property
    @jsii.member(jsii_name="dbInstanceIdentifiers")
    def db_instance_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more DB instance identifiers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbinstanceidentifiers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "dbInstanceIdentifiers"))

    @db_instance_identifiers.setter
    def db_instance_identifiers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9169e90fb8418f9626684279e280a068fcf61dafe5fb761fd4a75e9878147909)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbInstanceIdentifiers", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty",
        jsii_struct_bases=[],
        name_mapping={
            "connection_borrow_timeout": "connectionBorrowTimeout",
            "init_query": "initQuery",
            "max_connections_percent": "maxConnectionsPercent",
            "max_idle_connections_percent": "maxIdleConnectionsPercent",
            "session_pinning_filters": "sessionPinningFilters",
        },
    )
    class ConnectionPoolConfigurationInfoFormatProperty:
        def __init__(
            self,
            *,
            connection_borrow_timeout: typing.Optional[jsii.Number] = None,
            init_query: typing.Optional[builtins.str] = None,
            max_connections_percent: typing.Optional[jsii.Number] = None,
            max_idle_connections_percent: typing.Optional[jsii.Number] = None,
            session_pinning_filters: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''Specifies the settings that control the size and behavior of the connection pool associated with a ``DBProxyTargetGroup`` .

            :param connection_borrow_timeout: The number of seconds for a proxy to wait for a connection to become available in the connection pool. Only applies when the proxy has opened its maximum number of connections and all connections are busy with client sessions. Default: 120 Constraints: between 1 and 3600, or 0 representing unlimited
            :param init_query: One or more SQL statements for the proxy to run when opening each new database connection. Typically used with ``SET`` statements to make sure that each connection has identical settings such as time zone and character set. For multiple statements, use semicolons as the separator. You can also include multiple variables in a single ``SET`` statement, such as ``SET x=1, y=2`` . Default: no initialization query
            :param max_connections_percent: The maximum size of the connection pool for each target in a target group. The value is expressed as a percentage of the ``max_connections`` setting for the RDS DB instance or Aurora DB cluster used by the target group. If you specify ``MaxIdleConnectionsPercent`` , then you must also include a value for this parameter. Default: 10 for RDS for Microsoft SQL Server, and 100 for all other engines Constraints: Must be between 1 and 100.
            :param max_idle_connections_percent: Controls how actively the proxy closes idle database connections in the connection pool. The value is expressed as a percentage of the ``max_connections`` setting for the RDS DB instance or Aurora DB cluster used by the target group. With a high value, the proxy leaves a high percentage of idle database connections open. A low value causes the proxy to close more idle connections and return them to the database. If you specify this parameter, then you must also include a value for ``MaxConnectionsPercent`` . Default: The default value is half of the value of ``MaxConnectionsPercent`` . For example, if ``MaxConnectionsPercent`` is 80, then the default value of ``MaxIdleConnectionsPercent`` is 40. If the value of ``MaxConnectionsPercent`` isn't specified, then for SQL Server, ``MaxIdleConnectionsPercent`` is 5, and for all other engines, the default is 50. Constraints: Must be between 0 and the value of ``MaxConnectionsPercent`` .
            :param session_pinning_filters: Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection. Including an item in the list exempts that class of SQL operations from the pinning behavior. Default: no session pinning filters

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                connection_pool_configuration_info_format_property = rds.CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty(
                    connection_borrow_timeout=123,
                    init_query="initQuery",
                    max_connections_percent=123,
                    max_idle_connections_percent=123,
                    session_pinning_filters=["sessionPinningFilters"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__da025d6d1c9f1d4d2158074cd0625237cbac0513bffa1a2397886cf80b8ee2de)
                check_type(argname="argument connection_borrow_timeout", value=connection_borrow_timeout, expected_type=type_hints["connection_borrow_timeout"])
                check_type(argname="argument init_query", value=init_query, expected_type=type_hints["init_query"])
                check_type(argname="argument max_connections_percent", value=max_connections_percent, expected_type=type_hints["max_connections_percent"])
                check_type(argname="argument max_idle_connections_percent", value=max_idle_connections_percent, expected_type=type_hints["max_idle_connections_percent"])
                check_type(argname="argument session_pinning_filters", value=session_pinning_filters, expected_type=type_hints["session_pinning_filters"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if connection_borrow_timeout is not None:
                self._values["connection_borrow_timeout"] = connection_borrow_timeout
            if init_query is not None:
                self._values["init_query"] = init_query
            if max_connections_percent is not None:
                self._values["max_connections_percent"] = max_connections_percent
            if max_idle_connections_percent is not None:
                self._values["max_idle_connections_percent"] = max_idle_connections_percent
            if session_pinning_filters is not None:
                self._values["session_pinning_filters"] = session_pinning_filters

        @builtins.property
        def connection_borrow_timeout(self) -> typing.Optional[jsii.Number]:
            '''The number of seconds for a proxy to wait for a connection to become available in the connection pool.

            Only applies when the proxy has opened its maximum number of connections and all connections are busy with client sessions.

            Default: 120

            Constraints: between 1 and 3600, or 0 representing unlimited

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-connectionborrowtimeout
            '''
            result = self._values.get("connection_borrow_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def init_query(self) -> typing.Optional[builtins.str]:
            '''One or more SQL statements for the proxy to run when opening each new database connection.

            Typically used with ``SET`` statements to make sure that each connection has identical settings such as time zone and character set. For multiple statements, use semicolons as the separator. You can also include multiple variables in a single ``SET`` statement, such as ``SET x=1, y=2`` .

            Default: no initialization query

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-initquery
            '''
            result = self._values.get("init_query")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def max_connections_percent(self) -> typing.Optional[jsii.Number]:
            '''The maximum size of the connection pool for each target in a target group.

            The value is expressed as a percentage of the ``max_connections`` setting for the RDS DB instance or Aurora DB cluster used by the target group.

            If you specify ``MaxIdleConnectionsPercent`` , then you must also include a value for this parameter.

            Default: 10 for RDS for Microsoft SQL Server, and 100 for all other engines

            Constraints: Must be between 1 and 100.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-maxconnectionspercent
            '''
            result = self._values.get("max_connections_percent")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def max_idle_connections_percent(self) -> typing.Optional[jsii.Number]:
            '''Controls how actively the proxy closes idle database connections in the connection pool.

            The value is expressed as a percentage of the ``max_connections`` setting for the RDS DB instance or Aurora DB cluster used by the target group. With a high value, the proxy leaves a high percentage of idle database connections open. A low value causes the proxy to close more idle connections and return them to the database.

            If you specify this parameter, then you must also include a value for ``MaxConnectionsPercent`` .

            Default: The default value is half of the value of ``MaxConnectionsPercent`` . For example, if ``MaxConnectionsPercent`` is 80, then the default value of ``MaxIdleConnectionsPercent`` is 40. If the value of ``MaxConnectionsPercent`` isn't specified, then for SQL Server, ``MaxIdleConnectionsPercent`` is 5, and for all other engines, the default is 50.

            Constraints: Must be between 0 and the value of ``MaxConnectionsPercent`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-maxidleconnectionspercent
            '''
            result = self._values.get("max_idle_connections_percent")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def session_pinning_filters(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Each item in the list represents a class of SQL operations that normally cause all later statements in a session using a proxy to be pinned to the same underlying database connection.

            Including an item in the list exempts that class of SQL operations from the pinning behavior.

            Default: no session pinning filters

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfoformat-sessionpinningfilters
            '''
            result = self._values.get("session_pinning_filters")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConnectionPoolConfigurationInfoFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBProxyTargetGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_proxy_name": "dbProxyName",
        "target_group_name": "targetGroupName",
        "connection_pool_configuration_info": "connectionPoolConfigurationInfo",
        "db_cluster_identifiers": "dbClusterIdentifiers",
        "db_instance_identifiers": "dbInstanceIdentifiers",
    },
)
class CfnDBProxyTargetGroupProps:
    def __init__(
        self,
        *,
        db_proxy_name: builtins.str,
        target_group_name: builtins.str,
        connection_pool_configuration_info: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        db_cluster_identifiers: typing.Optional[typing.Sequence[builtins.str]] = None,
        db_instance_identifiers: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBProxyTargetGroup``.

        :param db_proxy_name: The identifier of the ``DBProxy`` that is associated with the ``DBProxyTargetGroup`` .
        :param target_group_name: The identifier for the target group. .. epigraph:: Currently, this property must be set to ``default`` .
        :param connection_pool_configuration_info: Settings that control the size and behavior of the connection pool associated with a ``DBProxyTargetGroup`` .
        :param db_cluster_identifiers: One or more DB cluster identifiers.
        :param db_instance_identifiers: One or more DB instance identifiers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            cfn_dBProxy_target_group_props = rds.CfnDBProxyTargetGroupProps(
                db_proxy_name="dbProxyName",
                target_group_name="targetGroupName",
            
                # the properties below are optional
                connection_pool_configuration_info=rds.CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty(
                    connection_borrow_timeout=123,
                    init_query="initQuery",
                    max_connections_percent=123,
                    max_idle_connections_percent=123,
                    session_pinning_filters=["sessionPinningFilters"]
                ),
                db_cluster_identifiers=["dbClusterIdentifiers"],
                db_instance_identifiers=["dbInstanceIdentifiers"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__696b81288dea41f185f9393384a5e9d87e62ceb1fa9b71340eade3f1dd909adb)
            check_type(argname="argument db_proxy_name", value=db_proxy_name, expected_type=type_hints["db_proxy_name"])
            check_type(argname="argument target_group_name", value=target_group_name, expected_type=type_hints["target_group_name"])
            check_type(argname="argument connection_pool_configuration_info", value=connection_pool_configuration_info, expected_type=type_hints["connection_pool_configuration_info"])
            check_type(argname="argument db_cluster_identifiers", value=db_cluster_identifiers, expected_type=type_hints["db_cluster_identifiers"])
            check_type(argname="argument db_instance_identifiers", value=db_instance_identifiers, expected_type=type_hints["db_instance_identifiers"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "db_proxy_name": db_proxy_name,
            "target_group_name": target_group_name,
        }
        if connection_pool_configuration_info is not None:
            self._values["connection_pool_configuration_info"] = connection_pool_configuration_info
        if db_cluster_identifiers is not None:
            self._values["db_cluster_identifiers"] = db_cluster_identifiers
        if db_instance_identifiers is not None:
            self._values["db_instance_identifiers"] = db_instance_identifiers

    @builtins.property
    def db_proxy_name(self) -> builtins.str:
        '''The identifier of the ``DBProxy`` that is associated with the ``DBProxyTargetGroup`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbproxyname
        '''
        result = self._values.get("db_proxy_name")
        assert result is not None, "Required property 'db_proxy_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_group_name(self) -> builtins.str:
        '''The identifier for the target group.

        .. epigraph::

           Currently, this property must be set to ``default`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-targetgroupname
        '''
        result = self._values.get("target_group_name")
        assert result is not None, "Required property 'target_group_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def connection_pool_configuration_info(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty]]:
        '''Settings that control the size and behavior of the connection pool associated with a ``DBProxyTargetGroup`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-connectionpoolconfigurationinfo
        '''
        result = self._values.get("connection_pool_configuration_info")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBProxyTargetGroup.ConnectionPoolConfigurationInfoFormatProperty]], result)

    @builtins.property
    def db_cluster_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more DB cluster identifiers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbclusteridentifiers
        '''
        result = self._values.get("db_cluster_identifiers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def db_instance_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more DB instance identifiers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbproxytargetgroup.html#cfn-rds-dbproxytargetgroup-dbinstanceidentifiers
        '''
        result = self._values.get("db_instance_identifiers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBProxyTargetGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnDBSecurityGroup(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBSecurityGroup",
):
    '''A CloudFormation ``AWS::RDS::DBSecurityGroup``.

    The ``AWS::RDS::DBSecurityGroup`` resource creates or updates an Amazon RDS DB security group.
    .. epigraph::

       EC2-Classic was retired on August 15, 2022. If you haven't migrated from EC2-Classic to a VPC, we recommend that you migrate as soon as possible. For more information, see `Migrate from EC2-Classic to a VPC <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/vpc-migrate.html>`_ in the *Amazon EC2 User Guide* , the blog `EC2-Classic Networking is Retiring – Here’s How to Prepare <https://docs.aws.amazon.com/aws/ec2-classic-is-retiring-heres-how-to-prepare/>`_ , and `Moving a DB instance not in a VPC into a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.Non-VPC2VPC.html>`_ in the *Amazon RDS User Guide* .

    :cloudformationResource: AWS::RDS::DBSecurityGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        cfn_dBSecurity_group = rds.CfnDBSecurityGroup(self, "MyCfnDBSecurityGroup",
            db_security_group_ingress=[rds.CfnDBSecurityGroup.IngressProperty(
                cidrip="cidrip",
                ec2_security_group_id="ec2SecurityGroupId",
                ec2_security_group_name="ec2SecurityGroupName",
                ec2_security_group_owner_id="ec2SecurityGroupOwnerId"
            )],
            group_description="groupDescription",
        
            # the properties below are optional
            ec2_vpc_id="ec2VpcId",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        db_security_group_ingress: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDBSecurityGroup.IngressProperty", typing.Dict[builtins.str, typing.Any]]]]],
        group_description: builtins.str,
        ec2_vpc_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBSecurityGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_security_group_ingress: Ingress rules to be applied to the DB security group.
        :param group_description: Provides the description of the DB security group.
        :param ec2_vpc_id: The identifier of an Amazon VPC. This property indicates the VPC that this DB security group belongs to. .. epigraph:: The ``EC2VpcId`` property is for backward compatibility with older regions, and is no longer recommended for providing security information to an RDS DB instance.
        :param tags: An optional array of key-value pairs to apply to this DB security group.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c5c7680faf5c2338d69e68949d97e5d71f842a1f204a366ea6bc4af0e919520c)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBSecurityGroupProps(
            db_security_group_ingress=db_security_group_ingress,
            group_description=group_description,
            ec2_vpc_id=ec2_vpc_id,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5d602fa45f7d09e41ed7e8acd626cc1fc99c3a1ef2d6c7514038c77df5930b72)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0351f063696d3aa6285c25db6816ce60e6f76882fcd180a521ec611b69390866)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _aws_cdk_core_f4b25747.TagManager:
        '''An optional array of key-value pairs to apply to this DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-tags
        '''
        return typing.cast(_aws_cdk_core_f4b25747.TagManager, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="dbSecurityGroupIngress")
    def db_security_group_ingress(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBSecurityGroup.IngressProperty"]]]:
        '''Ingress rules to be applied to the DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-dbsecuritygroupingress
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBSecurityGroup.IngressProperty"]]], jsii.get(self, "dbSecurityGroupIngress"))

    @db_security_group_ingress.setter
    def db_security_group_ingress(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDBSecurityGroup.IngressProperty"]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ffad8c538492ab19a75b41eb615f19fd1a9f37234a4587a34f7c5b6ed28f344c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSecurityGroupIngress", value)

    @builtins.property
    @jsii.member(jsii_name="groupDescription")
    def group_description(self) -> builtins.str:
        '''Provides the description of the DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-groupdescription
        '''
        return typing.cast(builtins.str, jsii.get(self, "groupDescription"))

    @group_description.setter
    def group_description(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__afff9a35b7f992a41edb6971111d5b4aacb912e5b6a2a58aa12b69073008c45a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "groupDescription", value)

    @builtins.property
    @jsii.member(jsii_name="ec2VpcId")
    def ec2_vpc_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of an Amazon VPC. This property indicates the VPC that this DB security group belongs to.

        .. epigraph::

           The ``EC2VpcId`` property is for backward compatibility with older regions, and is no longer recommended for providing security information to an RDS DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-ec2vpcid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ec2VpcId"))

    @ec2_vpc_id.setter
    def ec2_vpc_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__48c637225ac3089218eeef2fc5db94e687fd377c66872254aaab0843c6ca8abe)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ec2VpcId", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnDBSecurityGroup.IngressProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cidrip": "cidrip",
            "ec2_security_group_id": "ec2SecurityGroupId",
            "ec2_security_group_name": "ec2SecurityGroupName",
            "ec2_security_group_owner_id": "ec2SecurityGroupOwnerId",
        },
    )
    class IngressProperty:
        def __init__(
            self,
            *,
            cidrip: typing.Optional[builtins.str] = None,
            ec2_security_group_id: typing.Optional[builtins.str] = None,
            ec2_security_group_name: typing.Optional[builtins.str] = None,
            ec2_security_group_owner_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``Ingress`` property type specifies an individual ingress rule within an ``AWS::RDS::DBSecurityGroup`` resource.

            .. epigraph::

               EC2-Classic was retired on August 15, 2022. If you haven't migrated from EC2-Classic to a VPC, we recommend that you migrate as soon as possible. For more information, see `Migrate from EC2-Classic to a VPC <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/vpc-migrate.html>`_ in the *Amazon EC2 User Guide* , the blog `EC2-Classic Networking is Retiring – Here’s How to Prepare <https://docs.aws.amazon.com/aws/ec2-classic-is-retiring-heres-how-to-prepare/>`_ , and `Moving a DB instance not in a VPC into a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.Non-VPC2VPC.html>`_ in the *Amazon RDS User Guide* .

            :param cidrip: The IP range to authorize.
            :param ec2_security_group_id: Id of the EC2 security group to authorize. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
            :param ec2_security_group_name: Name of the EC2 security group to authorize. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
            :param ec2_security_group_owner_id: AWS account number of the owner of the EC2 security group specified in the ``EC2SecurityGroupName`` parameter. The AWS access key ID isn't an acceptable value. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                ingress_property = rds.CfnDBSecurityGroup.IngressProperty(
                    cidrip="cidrip",
                    ec2_security_group_id="ec2SecurityGroupId",
                    ec2_security_group_name="ec2SecurityGroupName",
                    ec2_security_group_owner_id="ec2SecurityGroupOwnerId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e4438c307c4e8cc623e5bb87869dbb669eb306cb639707d550db575e4342ad70)
                check_type(argname="argument cidrip", value=cidrip, expected_type=type_hints["cidrip"])
                check_type(argname="argument ec2_security_group_id", value=ec2_security_group_id, expected_type=type_hints["ec2_security_group_id"])
                check_type(argname="argument ec2_security_group_name", value=ec2_security_group_name, expected_type=type_hints["ec2_security_group_name"])
                check_type(argname="argument ec2_security_group_owner_id", value=ec2_security_group_owner_id, expected_type=type_hints["ec2_security_group_owner_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cidrip is not None:
                self._values["cidrip"] = cidrip
            if ec2_security_group_id is not None:
                self._values["ec2_security_group_id"] = ec2_security_group_id
            if ec2_security_group_name is not None:
                self._values["ec2_security_group_name"] = ec2_security_group_name
            if ec2_security_group_owner_id is not None:
                self._values["ec2_security_group_owner_id"] = ec2_security_group_owner_id

        @builtins.property
        def cidrip(self) -> typing.Optional[builtins.str]:
            '''The IP range to authorize.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html#cfn-rds-securitygroup-cidrip
            '''
            result = self._values.get("cidrip")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ec2_security_group_id(self) -> typing.Optional[builtins.str]:
            '''Id of the EC2 security group to authorize.

            For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html#cfn-rds-securitygroup-ec2securitygroupid
            '''
            result = self._values.get("ec2_security_group_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ec2_security_group_name(self) -> typing.Optional[builtins.str]:
            '''Name of the EC2 security group to authorize.

            For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html#cfn-rds-securitygroup-ec2securitygroupname
            '''
            result = self._values.get("ec2_security_group_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ec2_security_group_owner_id(self) -> typing.Optional[builtins.str]:
            '''AWS account number of the owner of the EC2 security group specified in the ``EC2SecurityGroupName`` parameter.

            The AWS access key ID isn't an acceptable value. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group-rule.html#cfn-rds-securitygroup-ec2securitygroupownerid
            '''
            result = self._values.get("ec2_security_group_owner_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IngressProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnDBSecurityGroupIngress(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBSecurityGroupIngress",
):
    '''A CloudFormation ``AWS::RDS::DBSecurityGroupIngress``.

    The ``AWS::RDS::DBSecurityGroupIngress`` resource enables ingress to a DB security group using one of two forms of authorization. First, you can add EC2 or VPC security groups to the DB security group if the application using the database is running on EC2 or VPC instances. Second, IP ranges are available if the application accessing your database is running on the Internet.

    This type supports updates. For more information about updating stacks, see `AWS CloudFormation Stacks Updates <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks.html>`_ .

    For details about the settings for DB security group ingress, see `AuthorizeDBSecurityGroupIngress <https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_AuthorizeDBSecurityGroupIngress.html>`_ .
    .. epigraph::

       EC2-Classic was retired on August 15, 2022. If you haven't migrated from EC2-Classic to a VPC, we recommend that you migrate as soon as possible. For more information, see `Migrate from EC2-Classic to a VPC <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/vpc-migrate.html>`_ in the *Amazon EC2 User Guide* , the blog `EC2-Classic Networking is Retiring – Here’s How to Prepare <https://docs.aws.amazon.com/aws/ec2-classic-is-retiring-heres-how-to-prepare/>`_ , and `Moving a DB instance not in a VPC into a VPC <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.Non-VPC2VPC.html>`_ in the *Amazon RDS User Guide* .

    :cloudformationResource: AWS::RDS::DBSecurityGroupIngress
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        cfn_dBSecurity_group_ingress = rds.CfnDBSecurityGroupIngress(self, "MyCfnDBSecurityGroupIngress",
            db_security_group_name="dbSecurityGroupName",
        
            # the properties below are optional
            cidrip="cidrip",
            ec2_security_group_id="ec2SecurityGroupId",
            ec2_security_group_name="ec2SecurityGroupName",
            ec2_security_group_owner_id="ec2SecurityGroupOwnerId"
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        db_security_group_name: builtins.str,
        cidrip: typing.Optional[builtins.str] = None,
        ec2_security_group_id: typing.Optional[builtins.str] = None,
        ec2_security_group_name: typing.Optional[builtins.str] = None,
        ec2_security_group_owner_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBSecurityGroupIngress``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_security_group_name: The name of the DB security group to add authorization to.
        :param cidrip: The IP range to authorize.
        :param ec2_security_group_id: Id of the EC2 security group to authorize. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
        :param ec2_security_group_name: Name of the EC2 security group to authorize. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
        :param ec2_security_group_owner_id: AWS account number of the owner of the EC2 security group specified in the ``EC2SecurityGroupName`` parameter. The AWS access key ID isn't an acceptable value. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d81ca270129fface5b5c10d07c689a85f9153cdc45dac9348550d8677a1dd543)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBSecurityGroupIngressProps(
            db_security_group_name=db_security_group_name,
            cidrip=cidrip,
            ec2_security_group_id=ec2_security_group_id,
            ec2_security_group_name=ec2_security_group_name,
            ec2_security_group_owner_id=ec2_security_group_owner_id,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__87934cf9cfc0c85433b929b1eb9e8bbd7fed443e08ca2a693861f59f3ed500d5)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cb5dfbe5f880f810e16feac129021b50ff9763da413af26ce890e9f97419d8d7)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="dbSecurityGroupName")
    def db_security_group_name(self) -> builtins.str:
        '''The name of the DB security group to add authorization to.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-dbsecuritygroupname
        '''
        return typing.cast(builtins.str, jsii.get(self, "dbSecurityGroupName"))

    @db_security_group_name.setter
    def db_security_group_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4285c774cc0c7c26201fd406a8396338af67833c57ff495749489e806419b108)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSecurityGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="cidrip")
    def cidrip(self) -> typing.Optional[builtins.str]:
        '''The IP range to authorize.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-cidrip
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cidrip"))

    @cidrip.setter
    def cidrip(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b8bc4a03887533b809db5c14a6728ae4db936ea9de88a6988b397fe9a1f0a0cf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cidrip", value)

    @builtins.property
    @jsii.member(jsii_name="ec2SecurityGroupId")
    def ec2_security_group_id(self) -> typing.Optional[builtins.str]:
        '''Id of the EC2 security group to authorize.

        For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ec2SecurityGroupId"))

    @ec2_security_group_id.setter
    def ec2_security_group_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7e0e1dcdc7ccf5ce05efacd2c1033024f3bd03bd541d0aa0b0dfb82762d59f73)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ec2SecurityGroupId", value)

    @builtins.property
    @jsii.member(jsii_name="ec2SecurityGroupName")
    def ec2_security_group_name(self) -> typing.Optional[builtins.str]:
        '''Name of the EC2 security group to authorize.

        For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ec2SecurityGroupName"))

    @ec2_security_group_name.setter
    def ec2_security_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d3e21fdc6ea1896dbe1b04848b9b6c2472836d90b62a4ce2c87cfdcc1190b011)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ec2SecurityGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="ec2SecurityGroupOwnerId")
    def ec2_security_group_owner_id(self) -> typing.Optional[builtins.str]:
        '''AWS account number of the owner of the EC2 security group specified in the ``EC2SecurityGroupName`` parameter.

        The AWS access key ID isn't an acceptable value. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupownerid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ec2SecurityGroupOwnerId"))

    @ec2_security_group_owner_id.setter
    def ec2_security_group_owner_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__82046ff77daf5df2853ff3ddf93822461831fba6542ec5d95cbe5caf17ca739f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ec2SecurityGroupOwnerId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBSecurityGroupIngressProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_security_group_name": "dbSecurityGroupName",
        "cidrip": "cidrip",
        "ec2_security_group_id": "ec2SecurityGroupId",
        "ec2_security_group_name": "ec2SecurityGroupName",
        "ec2_security_group_owner_id": "ec2SecurityGroupOwnerId",
    },
)
class CfnDBSecurityGroupIngressProps:
    def __init__(
        self,
        *,
        db_security_group_name: builtins.str,
        cidrip: typing.Optional[builtins.str] = None,
        ec2_security_group_id: typing.Optional[builtins.str] = None,
        ec2_security_group_name: typing.Optional[builtins.str] = None,
        ec2_security_group_owner_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBSecurityGroupIngress``.

        :param db_security_group_name: The name of the DB security group to add authorization to.
        :param cidrip: The IP range to authorize.
        :param ec2_security_group_id: Id of the EC2 security group to authorize. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
        :param ec2_security_group_name: Name of the EC2 security group to authorize. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.
        :param ec2_security_group_owner_id: AWS account number of the owner of the EC2 security group specified in the ``EC2SecurityGroupName`` parameter. The AWS access key ID isn't an acceptable value. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            cfn_dBSecurity_group_ingress_props = rds.CfnDBSecurityGroupIngressProps(
                db_security_group_name="dbSecurityGroupName",
            
                # the properties below are optional
                cidrip="cidrip",
                ec2_security_group_id="ec2SecurityGroupId",
                ec2_security_group_name="ec2SecurityGroupName",
                ec2_security_group_owner_id="ec2SecurityGroupOwnerId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__48b744409d785e8778fc98b2030f2dd68b635095dd09f5d0d70eb981021e2ff4)
            check_type(argname="argument db_security_group_name", value=db_security_group_name, expected_type=type_hints["db_security_group_name"])
            check_type(argname="argument cidrip", value=cidrip, expected_type=type_hints["cidrip"])
            check_type(argname="argument ec2_security_group_id", value=ec2_security_group_id, expected_type=type_hints["ec2_security_group_id"])
            check_type(argname="argument ec2_security_group_name", value=ec2_security_group_name, expected_type=type_hints["ec2_security_group_name"])
            check_type(argname="argument ec2_security_group_owner_id", value=ec2_security_group_owner_id, expected_type=type_hints["ec2_security_group_owner_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "db_security_group_name": db_security_group_name,
        }
        if cidrip is not None:
            self._values["cidrip"] = cidrip
        if ec2_security_group_id is not None:
            self._values["ec2_security_group_id"] = ec2_security_group_id
        if ec2_security_group_name is not None:
            self._values["ec2_security_group_name"] = ec2_security_group_name
        if ec2_security_group_owner_id is not None:
            self._values["ec2_security_group_owner_id"] = ec2_security_group_owner_id

    @builtins.property
    def db_security_group_name(self) -> builtins.str:
        '''The name of the DB security group to add authorization to.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-dbsecuritygroupname
        '''
        result = self._values.get("db_security_group_name")
        assert result is not None, "Required property 'db_security_group_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cidrip(self) -> typing.Optional[builtins.str]:
        '''The IP range to authorize.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-cidrip
        '''
        result = self._values.get("cidrip")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ec2_security_group_id(self) -> typing.Optional[builtins.str]:
        '''Id of the EC2 security group to authorize.

        For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupid
        '''
        result = self._values.get("ec2_security_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ec2_security_group_name(self) -> typing.Optional[builtins.str]:
        '''Name of the EC2 security group to authorize.

        For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupname
        '''
        result = self._values.get("ec2_security_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ec2_security_group_owner_id(self) -> typing.Optional[builtins.str]:
        '''AWS account number of the owner of the EC2 security group specified in the ``EC2SecurityGroupName`` parameter.

        The AWS access key ID isn't an acceptable value. For VPC DB security groups, ``EC2SecurityGroupId`` must be provided. Otherwise, ``EC2SecurityGroupOwnerId`` and either ``EC2SecurityGroupName`` or ``EC2SecurityGroupId`` must be provided.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-security-group-ingress.html#cfn-rds-securitygroup-ingress-ec2securitygroupownerid
        '''
        result = self._values.get("ec2_security_group_owner_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBSecurityGroupIngressProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBSecurityGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_security_group_ingress": "dbSecurityGroupIngress",
        "group_description": "groupDescription",
        "ec2_vpc_id": "ec2VpcId",
        "tags": "tags",
    },
)
class CfnDBSecurityGroupProps:
    def __init__(
        self,
        *,
        db_security_group_ingress: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnDBSecurityGroup.IngressProperty, typing.Dict[builtins.str, typing.Any]]]]],
        group_description: builtins.str,
        ec2_vpc_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBSecurityGroup``.

        :param db_security_group_ingress: Ingress rules to be applied to the DB security group.
        :param group_description: Provides the description of the DB security group.
        :param ec2_vpc_id: The identifier of an Amazon VPC. This property indicates the VPC that this DB security group belongs to. .. epigraph:: The ``EC2VpcId`` property is for backward compatibility with older regions, and is no longer recommended for providing security information to an RDS DB instance.
        :param tags: An optional array of key-value pairs to apply to this DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            cfn_dBSecurity_group_props = rds.CfnDBSecurityGroupProps(
                db_security_group_ingress=[rds.CfnDBSecurityGroup.IngressProperty(
                    cidrip="cidrip",
                    ec2_security_group_id="ec2SecurityGroupId",
                    ec2_security_group_name="ec2SecurityGroupName",
                    ec2_security_group_owner_id="ec2SecurityGroupOwnerId"
                )],
                group_description="groupDescription",
            
                # the properties below are optional
                ec2_vpc_id="ec2VpcId",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a8084d545963380141e7e10e78a82051633a202fa47da301078af9d18e6a7066)
            check_type(argname="argument db_security_group_ingress", value=db_security_group_ingress, expected_type=type_hints["db_security_group_ingress"])
            check_type(argname="argument group_description", value=group_description, expected_type=type_hints["group_description"])
            check_type(argname="argument ec2_vpc_id", value=ec2_vpc_id, expected_type=type_hints["ec2_vpc_id"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "db_security_group_ingress": db_security_group_ingress,
            "group_description": group_description,
        }
        if ec2_vpc_id is not None:
            self._values["ec2_vpc_id"] = ec2_vpc_id
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def db_security_group_ingress(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBSecurityGroup.IngressProperty]]]:
        '''Ingress rules to be applied to the DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-dbsecuritygroupingress
        '''
        result = self._values.get("db_security_group_ingress")
        assert result is not None, "Required property 'db_security_group_ingress' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDBSecurityGroup.IngressProperty]]], result)

    @builtins.property
    def group_description(self) -> builtins.str:
        '''Provides the description of the DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-groupdescription
        '''
        result = self._values.get("group_description")
        assert result is not None, "Required property 'group_description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def ec2_vpc_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of an Amazon VPC. This property indicates the VPC that this DB security group belongs to.

        .. epigraph::

           The ``EC2VpcId`` property is for backward compatibility with older regions, and is no longer recommended for providing security information to an RDS DB instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-ec2vpcid
        '''
        result = self._values.get("ec2_vpc_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]]:
        '''An optional array of key-value pairs to apply to this DB security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-security-group.html#cfn-rds-dbsecuritygroup-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBSecurityGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnDBSubnetGroup(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnDBSubnetGroup",
):
    '''A CloudFormation ``AWS::RDS::DBSubnetGroup``.

    The ``AWS::RDS::DBSubnetGroup`` resource creates a database subnet group. Subnet groups must contain at least two subnets in two different Availability Zones in the same region.

    For more information, see `Working with DB subnet groups <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_VPC.WorkingWithRDSInstanceinaVPC.html#USER_VPC.Subnets>`_ in the *Amazon RDS User Guide* .

    :cloudformationResource: AWS::RDS::DBSubnetGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        cfn_dBSubnet_group = rds.CfnDBSubnetGroup(self, "MyCfnDBSubnetGroup",
            db_subnet_group_description="dbSubnetGroupDescription",
            subnet_ids=["subnetIds"],
        
            # the properties below are optional
            db_subnet_group_name="dbSubnetGroupName",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        db_subnet_group_description: builtins.str,
        subnet_ids: typing.Sequence[builtins.str],
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::DBSubnetGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param db_subnet_group_description: The description for the DB subnet group.
        :param subnet_ids: The EC2 Subnet IDs for the DB subnet group.
        :param db_subnet_group_name: The name for the DB subnet group. This value is stored as a lowercase string. Constraints: Must contain no more than 255 lowercase alphanumeric characters or hyphens. Must not be "Default". Example: ``mysubnetgroup``
        :param tags: An optional array of key-value pairs to apply to this DB subnet group.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__86d442aac7265645880b06d42133b4d379a84e5a6e90efa1588a9b9b907b79d5)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDBSubnetGroupProps(
            db_subnet_group_description=db_subnet_group_description,
            subnet_ids=subnet_ids,
            db_subnet_group_name=db_subnet_group_name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__53567afc486f23ee4ffb2da3591ec9f8653c99bb2bc4322dba8333b1f78f6c02)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4903a2fe40536aa1ac7284c3c0216bc8f393b976195d097551a672cf3f69a5a2)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _aws_cdk_core_f4b25747.TagManager:
        '''An optional array of key-value pairs to apply to this DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-tags
        '''
        return typing.cast(_aws_cdk_core_f4b25747.TagManager, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="dbSubnetGroupDescription")
    def db_subnet_group_description(self) -> builtins.str:
        '''The description for the DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-dbsubnetgroupdescription
        '''
        return typing.cast(builtins.str, jsii.get(self, "dbSubnetGroupDescription"))

    @db_subnet_group_description.setter
    def db_subnet_group_description(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8f224ebe8df57583a1cede71a29f433b9f8a4bf6063ef28d538166e6dae054df)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSubnetGroupDescription", value)

    @builtins.property
    @jsii.member(jsii_name="subnetIds")
    def subnet_ids(self) -> typing.List[builtins.str]:
        '''The EC2 Subnet IDs for the DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-subnetids
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "subnetIds"))

    @subnet_ids.setter
    def subnet_ids(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d64fbaf49a64fb89fd1557e6dbc058dc03f4b7831933a50ca525cc6aea867943)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "subnetIds", value)

    @builtins.property
    @jsii.member(jsii_name="dbSubnetGroupName")
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        '''The name for the DB subnet group. This value is stored as a lowercase string.

        Constraints: Must contain no more than 255 lowercase alphanumeric characters or hyphens. Must not be "Default".

        Example: ``mysubnetgroup``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-dbsubnetgroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dbSubnetGroupName"))

    @db_subnet_group_name.setter
    def db_subnet_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__45c73e76a4e5ac105b4040708e6dfc191bf273b4a752aeb44dcd3bcced89accb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dbSubnetGroupName", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnDBSubnetGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "db_subnet_group_description": "dbSubnetGroupDescription",
        "subnet_ids": "subnetIds",
        "db_subnet_group_name": "dbSubnetGroupName",
        "tags": "tags",
    },
)
class CfnDBSubnetGroupProps:
    def __init__(
        self,
        *,
        db_subnet_group_description: builtins.str,
        subnet_ids: typing.Sequence[builtins.str],
        db_subnet_group_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDBSubnetGroup``.

        :param db_subnet_group_description: The description for the DB subnet group.
        :param subnet_ids: The EC2 Subnet IDs for the DB subnet group.
        :param db_subnet_group_name: The name for the DB subnet group. This value is stored as a lowercase string. Constraints: Must contain no more than 255 lowercase alphanumeric characters or hyphens. Must not be "Default". Example: ``mysubnetgroup``
        :param tags: An optional array of key-value pairs to apply to this DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            cfn_dBSubnet_group_props = rds.CfnDBSubnetGroupProps(
                db_subnet_group_description="dbSubnetGroupDescription",
                subnet_ids=["subnetIds"],
            
                # the properties below are optional
                db_subnet_group_name="dbSubnetGroupName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ca9992a87825ebd1312c1c69904864fa3e696e09e831508bfcc324fc4c54b96c)
            check_type(argname="argument db_subnet_group_description", value=db_subnet_group_description, expected_type=type_hints["db_subnet_group_description"])
            check_type(argname="argument subnet_ids", value=subnet_ids, expected_type=type_hints["subnet_ids"])
            check_type(argname="argument db_subnet_group_name", value=db_subnet_group_name, expected_type=type_hints["db_subnet_group_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "db_subnet_group_description": db_subnet_group_description,
            "subnet_ids": subnet_ids,
        }
        if db_subnet_group_name is not None:
            self._values["db_subnet_group_name"] = db_subnet_group_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def db_subnet_group_description(self) -> builtins.str:
        '''The description for the DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-dbsubnetgroupdescription
        '''
        result = self._values.get("db_subnet_group_description")
        assert result is not None, "Required property 'db_subnet_group_description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def subnet_ids(self) -> typing.List[builtins.str]:
        '''The EC2 Subnet IDs for the DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-subnetids
        '''
        result = self._values.get("subnet_ids")
        assert result is not None, "Required property 'subnet_ids' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def db_subnet_group_name(self) -> typing.Optional[builtins.str]:
        '''The name for the DB subnet group. This value is stored as a lowercase string.

        Constraints: Must contain no more than 255 lowercase alphanumeric characters or hyphens. Must not be "Default".

        Example: ``mysubnetgroup``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-dbsubnetgroupname
        '''
        result = self._values.get("db_subnet_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]]:
        '''An optional array of key-value pairs to apply to this DB subnet group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-dbsubnetgroup.html#cfn-rds-dbsubnetgroup-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDBSubnetGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnEventSubscription(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnEventSubscription",
):
    '''A CloudFormation ``AWS::RDS::EventSubscription``.

    The ``AWS::RDS::EventSubscription`` resource allows you to receive notifications for Amazon Relational Database Service events through the Amazon Simple Notification Service (Amazon SNS). For more information, see `Using Amazon RDS Event Notification <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.html>`_ in the *Amazon RDS User Guide* .

    :cloudformationResource: AWS::RDS::EventSubscription
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        cfn_event_subscription = rds.CfnEventSubscription(self, "MyCfnEventSubscription",
            sns_topic_arn="snsTopicArn",
        
            # the properties below are optional
            enabled=False,
            event_categories=["eventCategories"],
            source_ids=["sourceIds"],
            source_type="sourceType",
            subscription_name="subscriptionName",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        sns_topic_arn: builtins.str,
        enabled: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        event_categories: typing.Optional[typing.Sequence[builtins.str]] = None,
        source_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        source_type: typing.Optional[builtins.str] = None,
        subscription_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::EventSubscription``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param sns_topic_arn: The Amazon Resource Name (ARN) of the SNS topic created for event notification. The ARN is created by Amazon SNS when you create a topic and subscribe to it.
        :param enabled: A value that indicates whether to activate the subscription. If the event notification subscription isn't activated, the subscription is created but not active.
        :param event_categories: A list of event categories for a particular source type ( ``SourceType`` ) that you want to subscribe to. You can see a list of the categories for a given source type in the "Amazon RDS event categories and event messages" section of the `*Amazon RDS User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.Messages.html>`_ or the `*Amazon Aurora User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_Events.Messages.html>`_ . You can also see this list by using the ``DescribeEventCategories`` operation.
        :param source_ids: The list of identifiers of the event sources for which events are returned. If not specified, then all sources are included in the response. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens. It can't end with a hyphen or contain two consecutive hyphens. Constraints: - If a ``SourceIds`` value is supplied, ``SourceType`` must also be provided. - If the source type is a DB instance, a ``DBInstanceIdentifier`` value must be supplied. - If the source type is a DB cluster, a ``DBClusterIdentifier`` value must be supplied. - If the source type is a DB parameter group, a ``DBParameterGroupName`` value must be supplied. - If the source type is a DB security group, a ``DBSecurityGroupName`` value must be supplied. - If the source type is a DB snapshot, a ``DBSnapshotIdentifier`` value must be supplied. - If the source type is a DB cluster snapshot, a ``DBClusterSnapshotIdentifier`` value must be supplied.
        :param source_type: The type of source that is generating the events. For example, if you want to be notified of events generated by a DB instance, set this parameter to ``db-instance`` . If this value isn't specified, all events are returned. Valid values: ``db-instance`` | ``db-cluster`` | ``db-parameter-group`` | ``db-security-group`` | ``db-snapshot`` | ``db-cluster-snapshot``
        :param subscription_name: The name of the subscription. Constraints: The name must be less than 255 characters.
        :param tags: An optional array of key-value pairs to apply to this subscription.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__626f9455ff3afe230c8efec1a5a6e14edd11560ae1d61fe064045859f2ebd22a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnEventSubscriptionProps(
            sns_topic_arn=sns_topic_arn,
            enabled=enabled,
            event_categories=event_categories,
            source_ids=source_ids,
            source_type=source_type,
            subscription_name=subscription_name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d4e408c787e8cc670049da5d084b75085f9cbd175aa8bfd42203dfde23b4543a)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d512b690c0558bf1cffadc2826e8a80484fc16debafb5266b46a6a651395b8ae)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _aws_cdk_core_f4b25747.TagManager:
        '''An optional array of key-value pairs to apply to this subscription.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-tags
        '''
        return typing.cast(_aws_cdk_core_f4b25747.TagManager, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="snsTopicArn")
    def sns_topic_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the SNS topic created for event notification.

        The ARN is created by Amazon SNS when you create a topic and subscribe to it.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-snstopicarn
        '''
        return typing.cast(builtins.str, jsii.get(self, "snsTopicArn"))

    @sns_topic_arn.setter
    def sns_topic_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f9373da1f966331058b554457b3b72794c358f0c7afe3689a2553c34264d7093)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "snsTopicArn", value)

    @builtins.property
    @jsii.member(jsii_name="enabled")
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to activate the subscription.

        If the event notification subscription isn't activated, the subscription is created but not active.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-enabled
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "enabled"))

    @enabled.setter
    def enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3e63898d7283acdf28adc41a2cafafb3828c63e918cd541a670d7e6a65487ab9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enabled", value)

    @builtins.property
    @jsii.member(jsii_name="eventCategories")
    def event_categories(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of event categories for a particular source type ( ``SourceType`` ) that you want to subscribe to.

        You can see a list of the categories for a given source type in the "Amazon RDS event categories and event messages" section of the `*Amazon RDS User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.Messages.html>`_ or the `*Amazon Aurora User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_Events.Messages.html>`_ . You can also see this list by using the ``DescribeEventCategories`` operation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-eventcategories
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "eventCategories"))

    @event_categories.setter
    def event_categories(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__94f9396278415a74e007272901b43424b322dc8acc06f1ab1ef2d96d718da939)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "eventCategories", value)

    @builtins.property
    @jsii.member(jsii_name="sourceIds")
    def source_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of identifiers of the event sources for which events are returned.

        If not specified, then all sources are included in the response. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens. It can't end with a hyphen or contain two consecutive hyphens.

        Constraints:

        - If a ``SourceIds`` value is supplied, ``SourceType`` must also be provided.
        - If the source type is a DB instance, a ``DBInstanceIdentifier`` value must be supplied.
        - If the source type is a DB cluster, a ``DBClusterIdentifier`` value must be supplied.
        - If the source type is a DB parameter group, a ``DBParameterGroupName`` value must be supplied.
        - If the source type is a DB security group, a ``DBSecurityGroupName`` value must be supplied.
        - If the source type is a DB snapshot, a ``DBSnapshotIdentifier`` value must be supplied.
        - If the source type is a DB cluster snapshot, a ``DBClusterSnapshotIdentifier`` value must be supplied.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-sourceids
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "sourceIds"))

    @source_ids.setter
    def source_ids(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__86bd64fd8eb2b010a73036ae5683cc6e72a677c39a1ab9b68ec775a0a0d780d8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceIds", value)

    @builtins.property
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> typing.Optional[builtins.str]:
        '''The type of source that is generating the events.

        For example, if you want to be notified of events generated by a DB instance, set this parameter to ``db-instance`` . If this value isn't specified, all events are returned.

        Valid values: ``db-instance`` | ``db-cluster`` | ``db-parameter-group`` | ``db-security-group`` | ``db-snapshot`` | ``db-cluster-snapshot``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-sourcetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceType"))

    @source_type.setter
    def source_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cdefa57886f35f79d17c05f4d891bc3ebd2725772a8c22b8a116a19418f76e40)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceType", value)

    @builtins.property
    @jsii.member(jsii_name="subscriptionName")
    def subscription_name(self) -> typing.Optional[builtins.str]:
        '''The name of the subscription.

        Constraints: The name must be less than 255 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-subscriptionname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "subscriptionName"))

    @subscription_name.setter
    def subscription_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ba12f9b75c6cd164b6b5aa81a1fd058ea117460a5de98f9ee0793d83c094d1a5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "subscriptionName", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnEventSubscriptionProps",
    jsii_struct_bases=[],
    name_mapping={
        "sns_topic_arn": "snsTopicArn",
        "enabled": "enabled",
        "event_categories": "eventCategories",
        "source_ids": "sourceIds",
        "source_type": "sourceType",
        "subscription_name": "subscriptionName",
        "tags": "tags",
    },
)
class CfnEventSubscriptionProps:
    def __init__(
        self,
        *,
        sns_topic_arn: builtins.str,
        enabled: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        event_categories: typing.Optional[typing.Sequence[builtins.str]] = None,
        source_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        source_type: typing.Optional[builtins.str] = None,
        subscription_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnEventSubscription``.

        :param sns_topic_arn: The Amazon Resource Name (ARN) of the SNS topic created for event notification. The ARN is created by Amazon SNS when you create a topic and subscribe to it.
        :param enabled: A value that indicates whether to activate the subscription. If the event notification subscription isn't activated, the subscription is created but not active.
        :param event_categories: A list of event categories for a particular source type ( ``SourceType`` ) that you want to subscribe to. You can see a list of the categories for a given source type in the "Amazon RDS event categories and event messages" section of the `*Amazon RDS User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.Messages.html>`_ or the `*Amazon Aurora User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_Events.Messages.html>`_ . You can also see this list by using the ``DescribeEventCategories`` operation.
        :param source_ids: The list of identifiers of the event sources for which events are returned. If not specified, then all sources are included in the response. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens. It can't end with a hyphen or contain two consecutive hyphens. Constraints: - If a ``SourceIds`` value is supplied, ``SourceType`` must also be provided. - If the source type is a DB instance, a ``DBInstanceIdentifier`` value must be supplied. - If the source type is a DB cluster, a ``DBClusterIdentifier`` value must be supplied. - If the source type is a DB parameter group, a ``DBParameterGroupName`` value must be supplied. - If the source type is a DB security group, a ``DBSecurityGroupName`` value must be supplied. - If the source type is a DB snapshot, a ``DBSnapshotIdentifier`` value must be supplied. - If the source type is a DB cluster snapshot, a ``DBClusterSnapshotIdentifier`` value must be supplied.
        :param source_type: The type of source that is generating the events. For example, if you want to be notified of events generated by a DB instance, set this parameter to ``db-instance`` . If this value isn't specified, all events are returned. Valid values: ``db-instance`` | ``db-cluster`` | ``db-parameter-group`` | ``db-security-group`` | ``db-snapshot`` | ``db-cluster-snapshot``
        :param subscription_name: The name of the subscription. Constraints: The name must be less than 255 characters.
        :param tags: An optional array of key-value pairs to apply to this subscription.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            cfn_event_subscription_props = rds.CfnEventSubscriptionProps(
                sns_topic_arn="snsTopicArn",
            
                # the properties below are optional
                enabled=False,
                event_categories=["eventCategories"],
                source_ids=["sourceIds"],
                source_type="sourceType",
                subscription_name="subscriptionName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b1a0db2777533b66422db525157d52082e4619934b45ad9559f503eaaff65fa8)
            check_type(argname="argument sns_topic_arn", value=sns_topic_arn, expected_type=type_hints["sns_topic_arn"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument event_categories", value=event_categories, expected_type=type_hints["event_categories"])
            check_type(argname="argument source_ids", value=source_ids, expected_type=type_hints["source_ids"])
            check_type(argname="argument source_type", value=source_type, expected_type=type_hints["source_type"])
            check_type(argname="argument subscription_name", value=subscription_name, expected_type=type_hints["subscription_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "sns_topic_arn": sns_topic_arn,
        }
        if enabled is not None:
            self._values["enabled"] = enabled
        if event_categories is not None:
            self._values["event_categories"] = event_categories
        if source_ids is not None:
            self._values["source_ids"] = source_ids
        if source_type is not None:
            self._values["source_type"] = source_type
        if subscription_name is not None:
            self._values["subscription_name"] = subscription_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def sns_topic_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the SNS topic created for event notification.

        The ARN is created by Amazon SNS when you create a topic and subscribe to it.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-snstopicarn
        '''
        result = self._values.get("sns_topic_arn")
        assert result is not None, "Required property 'sns_topic_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A value that indicates whether to activate the subscription.

        If the event notification subscription isn't activated, the subscription is created but not active.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-enabled
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def event_categories(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of event categories for a particular source type ( ``SourceType`` ) that you want to subscribe to.

        You can see a list of the categories for a given source type in the "Amazon RDS event categories and event messages" section of the `*Amazon RDS User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.Messages.html>`_ or the `*Amazon Aurora User Guide* <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_Events.Messages.html>`_ . You can also see this list by using the ``DescribeEventCategories`` operation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-eventcategories
        '''
        result = self._values.get("event_categories")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def source_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of identifiers of the event sources for which events are returned.

        If not specified, then all sources are included in the response. An identifier must begin with a letter and must contain only ASCII letters, digits, and hyphens. It can't end with a hyphen or contain two consecutive hyphens.

        Constraints:

        - If a ``SourceIds`` value is supplied, ``SourceType`` must also be provided.
        - If the source type is a DB instance, a ``DBInstanceIdentifier`` value must be supplied.
        - If the source type is a DB cluster, a ``DBClusterIdentifier`` value must be supplied.
        - If the source type is a DB parameter group, a ``DBParameterGroupName`` value must be supplied.
        - If the source type is a DB security group, a ``DBSecurityGroupName`` value must be supplied.
        - If the source type is a DB snapshot, a ``DBSnapshotIdentifier`` value must be supplied.
        - If the source type is a DB cluster snapshot, a ``DBClusterSnapshotIdentifier`` value must be supplied.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-sourceids
        '''
        result = self._values.get("source_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def source_type(self) -> typing.Optional[builtins.str]:
        '''The type of source that is generating the events.

        For example, if you want to be notified of events generated by a DB instance, set this parameter to ``db-instance`` . If this value isn't specified, all events are returned.

        Valid values: ``db-instance`` | ``db-cluster`` | ``db-parameter-group`` | ``db-security-group`` | ``db-snapshot`` | ``db-cluster-snapshot``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-sourcetype
        '''
        result = self._values.get("source_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def subscription_name(self) -> typing.Optional[builtins.str]:
        '''The name of the subscription.

        Constraints: The name must be less than 255 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-subscriptionname
        '''
        result = self._values.get("subscription_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]]:
        '''An optional array of key-value pairs to apply to this subscription.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-eventsubscription.html#cfn-rds-eventsubscription-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEventSubscriptionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnGlobalCluster(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnGlobalCluster",
):
    '''A CloudFormation ``AWS::RDS::GlobalCluster``.

    The ``AWS::RDS::GlobalCluster`` resource creates or updates an Amazon Aurora global database spread across multiple AWS Regions.

    The global database contains a single primary cluster with read-write capability, and a read-only secondary cluster that receives data from the primary cluster through high-speed replication performed by the Aurora storage subsystem.

    You can create a global database that is initially empty, and then add a primary cluster and a secondary cluster to it.

    For information about Aurora global databases, see `Working with Amazon Aurora Global Databases <https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-global-database.html>`_ in the *Amazon Aurora User Guide* .

    :cloudformationResource: AWS::RDS::GlobalCluster
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        cfn_global_cluster = rds.CfnGlobalCluster(self, "MyCfnGlobalCluster",
            deletion_protection=False,
            engine="engine",
            engine_version="engineVersion",
            global_cluster_identifier="globalClusterIdentifier",
            source_db_cluster_identifier="sourceDbClusterIdentifier",
            storage_encrypted=False
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        global_cluster_identifier: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::GlobalCluster``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param deletion_protection: The deletion protection setting for the new global database. The global database can't be deleted when deletion protection is enabled.
        :param engine: The name of the database engine to be used for this DB cluster. If this property isn't specified, the database engine is derived from the source DB cluster specified by the ``SourceDBClusterIdentifier`` property. .. epigraph:: If the ``SourceDBClusterIdentifier`` property isn't specified, this property is required. If the ``SourceDBClusterIdentifier`` property is specified, make sure this property isn't specified.
        :param engine_version: The engine version of the Aurora global database.
        :param global_cluster_identifier: The cluster identifier of the global database cluster.
        :param source_db_cluster_identifier: The DB cluster identifier or Amazon Resource Name (ARN) to use as the primary cluster of the global database. .. epigraph:: If the ``Engine`` property isn't specified, this property is required. If the ``Engine`` property is specified, make sure this property isn't specified.
        :param storage_encrypted: The storage encryption setting for the global database cluster.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8b272f2c3e9ae7d1f4e3baed33182a410d790163399556efc5a64de5a03eecb7)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnGlobalClusterProps(
            deletion_protection=deletion_protection,
            engine=engine,
            engine_version=engine_version,
            global_cluster_identifier=global_cluster_identifier,
            source_db_cluster_identifier=source_db_cluster_identifier,
            storage_encrypted=storage_encrypted,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0b6158f9ba739154c04eee56eeed275146ffb1580abae7febefcf81b1c393f9b)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__29035d4112548ff74e18b71daf1abbcfc19d8389f3683f7fbe01da3f8c8543ad)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="deletionProtection")
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''The deletion protection setting for the new global database.

        The global database can't be deleted when deletion protection is enabled.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-deletionprotection
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "deletionProtection"))

    @deletion_protection.setter
    def deletion_protection(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2a623661863e61a35dc1032c5769e87f9121beaadb579fee0a1cf0b776cabd50)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deletionProtection", value)

    @builtins.property
    @jsii.member(jsii_name="engine")
    def engine(self) -> typing.Optional[builtins.str]:
        '''The name of the database engine to be used for this DB cluster.

        If this property isn't specified, the database engine is derived from the source DB cluster specified by the ``SourceDBClusterIdentifier`` property.
        .. epigraph::

           If the ``SourceDBClusterIdentifier`` property isn't specified, this property is required. If the ``SourceDBClusterIdentifier`` property is specified, make sure this property isn't specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-engine
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engine"))

    @engine.setter
    def engine(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d4543bb5f6a10ae810ab4d0262e708f45fdb78d368a0c9a214799d2d4a6ad7c3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engine", value)

    @builtins.property
    @jsii.member(jsii_name="engineVersion")
    def engine_version(self) -> typing.Optional[builtins.str]:
        '''The engine version of the Aurora global database.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-engineversion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "engineVersion"))

    @engine_version.setter
    def engine_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d22189b2cc51468507669fb6edd0d498ab09898ec3a685303aa39bfca04479b0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engineVersion", value)

    @builtins.property
    @jsii.member(jsii_name="globalClusterIdentifier")
    def global_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The cluster identifier of the global database cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-globalclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "globalClusterIdentifier"))

    @global_cluster_identifier.setter
    def global_cluster_identifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__722b520a68fde56c21d303894305e36aff017c4a31ae4d51d8cb84621410f5f6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "globalClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="sourceDbClusterIdentifier")
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The DB cluster identifier or Amazon Resource Name (ARN) to use as the primary cluster of the global database.

        .. epigraph::

           If the ``Engine`` property isn't specified, this property is required. If the ``Engine`` property is specified, make sure this property isn't specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-sourcedbclusteridentifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDbClusterIdentifier"))

    @source_db_cluster_identifier.setter
    def source_db_cluster_identifier(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d94e0ca92a1da4024d7a94a6209eb2787506bf32f11bac2e0b66e625df6238bd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceDbClusterIdentifier", value)

    @builtins.property
    @jsii.member(jsii_name="storageEncrypted")
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''The storage encryption setting for the global database cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-storageencrypted
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "storageEncrypted"))

    @storage_encrypted.setter
    def storage_encrypted(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fc45d7154389ea54c4ecda284b6d1f5a147871d33ca69ae15cd65d18c3abbb56)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageEncrypted", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnGlobalClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "deletion_protection": "deletionProtection",
        "engine": "engine",
        "engine_version": "engineVersion",
        "global_cluster_identifier": "globalClusterIdentifier",
        "source_db_cluster_identifier": "sourceDbClusterIdentifier",
        "storage_encrypted": "storageEncrypted",
    },
)
class CfnGlobalClusterProps:
    def __init__(
        self,
        *,
        deletion_protection: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        engine: typing.Optional[builtins.str] = None,
        engine_version: typing.Optional[builtins.str] = None,
        global_cluster_identifier: typing.Optional[builtins.str] = None,
        source_db_cluster_identifier: typing.Optional[builtins.str] = None,
        storage_encrypted: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
    ) -> None:
        '''Properties for defining a ``CfnGlobalCluster``.

        :param deletion_protection: The deletion protection setting for the new global database. The global database can't be deleted when deletion protection is enabled.
        :param engine: The name of the database engine to be used for this DB cluster. If this property isn't specified, the database engine is derived from the source DB cluster specified by the ``SourceDBClusterIdentifier`` property. .. epigraph:: If the ``SourceDBClusterIdentifier`` property isn't specified, this property is required. If the ``SourceDBClusterIdentifier`` property is specified, make sure this property isn't specified.
        :param engine_version: The engine version of the Aurora global database.
        :param global_cluster_identifier: The cluster identifier of the global database cluster.
        :param source_db_cluster_identifier: The DB cluster identifier or Amazon Resource Name (ARN) to use as the primary cluster of the global database. .. epigraph:: If the ``Engine`` property isn't specified, this property is required. If the ``Engine`` property is specified, make sure this property isn't specified.
        :param storage_encrypted: The storage encryption setting for the global database cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            cfn_global_cluster_props = rds.CfnGlobalClusterProps(
                deletion_protection=False,
                engine="engine",
                engine_version="engineVersion",
                global_cluster_identifier="globalClusterIdentifier",
                source_db_cluster_identifier="sourceDbClusterIdentifier",
                storage_encrypted=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2ba97009e0ccd48fb5d92f95f41b4f37db115c23295e942b053190733112426a)
            check_type(argname="argument deletion_protection", value=deletion_protection, expected_type=type_hints["deletion_protection"])
            check_type(argname="argument engine", value=engine, expected_type=type_hints["engine"])
            check_type(argname="argument engine_version", value=engine_version, expected_type=type_hints["engine_version"])
            check_type(argname="argument global_cluster_identifier", value=global_cluster_identifier, expected_type=type_hints["global_cluster_identifier"])
            check_type(argname="argument source_db_cluster_identifier", value=source_db_cluster_identifier, expected_type=type_hints["source_db_cluster_identifier"])
            check_type(argname="argument storage_encrypted", value=storage_encrypted, expected_type=type_hints["storage_encrypted"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if engine is not None:
            self._values["engine"] = engine
        if engine_version is not None:
            self._values["engine_version"] = engine_version
        if global_cluster_identifier is not None:
            self._values["global_cluster_identifier"] = global_cluster_identifier
        if source_db_cluster_identifier is not None:
            self._values["source_db_cluster_identifier"] = source_db_cluster_identifier
        if storage_encrypted is not None:
            self._values["storage_encrypted"] = storage_encrypted

    @builtins.property
    def deletion_protection(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''The deletion protection setting for the new global database.

        The global database can't be deleted when deletion protection is enabled.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-deletionprotection
        '''
        result = self._values.get("deletion_protection")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def engine(self) -> typing.Optional[builtins.str]:
        '''The name of the database engine to be used for this DB cluster.

        If this property isn't specified, the database engine is derived from the source DB cluster specified by the ``SourceDBClusterIdentifier`` property.
        .. epigraph::

           If the ``SourceDBClusterIdentifier`` property isn't specified, this property is required. If the ``SourceDBClusterIdentifier`` property is specified, make sure this property isn't specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-engine
        '''
        result = self._values.get("engine")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def engine_version(self) -> typing.Optional[builtins.str]:
        '''The engine version of the Aurora global database.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-engineversion
        '''
        result = self._values.get("engine_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def global_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The cluster identifier of the global database cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-globalclusteridentifier
        '''
        result = self._values.get("global_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_db_cluster_identifier(self) -> typing.Optional[builtins.str]:
        '''The DB cluster identifier or Amazon Resource Name (ARN) to use as the primary cluster of the global database.

        .. epigraph::

           If the ``Engine`` property isn't specified, this property is required. If the ``Engine`` property is specified, make sure this property isn't specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-sourcedbclusteridentifier
        '''
        result = self._values.get("source_db_cluster_identifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def storage_encrypted(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''The storage encryption setting for the global database cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-globalcluster.html#cfn-rds-globalcluster-storageencrypted
        '''
        result = self._values.get("storage_encrypted")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnGlobalClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnOptionGroup(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-rds.CfnOptionGroup",
):
    '''A CloudFormation ``AWS::RDS::OptionGroup``.

    The ``AWS::RDS::OptionGroup`` resource creates or updates an option group, to enable and configure features that are specific to a particular DB engine.

    :cloudformationResource: AWS::RDS::OptionGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_rds as rds
        
        cfn_option_group = rds.CfnOptionGroup(self, "MyCfnOptionGroup",
            engine_name="engineName",
            major_engine_version="majorEngineVersion",
            option_group_description="optionGroupDescription",
        
            # the properties below are optional
            option_configurations=[rds.CfnOptionGroup.OptionConfigurationProperty(
                option_name="optionName",
        
                # the properties below are optional
                db_security_group_memberships=["dbSecurityGroupMemberships"],
                option_settings=[rds.CfnOptionGroup.OptionSettingProperty(
                    name="name",
                    value="value"
                )],
                option_version="optionVersion",
                port=123,
                vpc_security_group_memberships=["vpcSecurityGroupMemberships"]
            )],
            option_group_name="optionGroupName",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        engine_name: builtins.str,
        major_engine_version: builtins.str,
        option_group_description: builtins.str,
        option_configurations: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnOptionGroup.OptionConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        option_group_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::RDS::OptionGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param engine_name: Specifies the name of the engine that this option group should be associated with. Valid Values: - ``mariadb`` - ``mysql`` - ``oracle-ee`` - ``oracle-ee-cdb`` - ``oracle-se2`` - ``oracle-se2-cdb`` - ``postgres`` - ``sqlserver-ee`` - ``sqlserver-se`` - ``sqlserver-ex`` - ``sqlserver-web``
        :param major_engine_version: Specifies the major version of the engine that this option group should be associated with.
        :param option_group_description: The description of the option group.
        :param option_configurations: A list of options and the settings for each option.
        :param option_group_name: The name of the option group to be created. Constraints: - Must be 1 to 255 letters, numbers, or hyphens - First character must be a letter - Can't end with a hyphen or contain two consecutive hyphens Example: ``myoptiongroup`` If you don't specify a value for ``OptionGroupName`` property, a name is automatically created for the option group. .. epigraph:: This value is stored as a lowercase string.
        :param tags: An optional array of key-value pairs to apply to this option group.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1173aa6264caa624db0c47dcd4e54f5bdc2518299b147f39772c8836c1cf9a59)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnOptionGroupProps(
            engine_name=engine_name,
            major_engine_version=major_engine_version,
            option_group_description=option_group_description,
            option_configurations=option_configurations,
            option_group_name=option_group_name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7202cff31157a2596281fdd0e91bdc46025a6cc78541ccc6b8942ffb13fae5bf)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d1a80b2fc5b8bc4209cc14151a51aa9d482981a16b98f723904cba52474d6de8)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _aws_cdk_core_f4b25747.TagManager:
        '''An optional array of key-value pairs to apply to this option group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-tags
        '''
        return typing.cast(_aws_cdk_core_f4b25747.TagManager, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="engineName")
    def engine_name(self) -> builtins.str:
        '''Specifies the name of the engine that this option group should be associated with.

        Valid Values:

        - ``mariadb``
        - ``mysql``
        - ``oracle-ee``
        - ``oracle-ee-cdb``
        - ``oracle-se2``
        - ``oracle-se2-cdb``
        - ``postgres``
        - ``sqlserver-ee``
        - ``sqlserver-se``
        - ``sqlserver-ex``
        - ``sqlserver-web``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-enginename
        '''
        return typing.cast(builtins.str, jsii.get(self, "engineName"))

    @engine_name.setter
    def engine_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3e0083fd8b1ba4d70249e2469f90bef67b98e100be48ba5143ff6a3a3c28c4c7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "engineName", value)

    @builtins.property
    @jsii.member(jsii_name="majorEngineVersion")
    def major_engine_version(self) -> builtins.str:
        '''Specifies the major version of the engine that this option group should be associated with.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-majorengineversion
        '''
        return typing.cast(builtins.str, jsii.get(self, "majorEngineVersion"))

    @major_engine_version.setter
    def major_engine_version(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__abb98243cf0cc1232af9d6c95d0456eec83779b2babdca84dda6eca5c2b13216)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "majorEngineVersion", value)

    @builtins.property
    @jsii.member(jsii_name="optionGroupDescription")
    def option_group_description(self) -> builtins.str:
        '''The description of the option group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-optiongroupdescription
        '''
        return typing.cast(builtins.str, jsii.get(self, "optionGroupDescription"))

    @option_group_description.setter
    def option_group_description(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__63218a0764ee31751dd92a2672286ee93c0ca6b198891cbda1c0a27315b253ac)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "optionGroupDescription", value)

    @builtins.property
    @jsii.member(jsii_name="optionConfigurations")
    def option_configurations(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOptionGroup.OptionConfigurationProperty"]]]]:
        '''A list of options and the settings for each option.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-optionconfigurations
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOptionGroup.OptionConfigurationProperty"]]]], jsii.get(self, "optionConfigurations"))

    @option_configurations.setter
    def option_configurations(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOptionGroup.OptionConfigurationProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b47caa7f7e728d6accedb847272b006f60f1e669da40205468e4fe5827db36e0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "optionConfigurations", value)

    @builtins.property
    @jsii.member(jsii_name="optionGroupName")
    def option_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the option group to be created.

        Constraints:

        - Must be 1 to 255 letters, numbers, or hyphens
        - First character must be a letter
        - Can't end with a hyphen or contain two consecutive hyphens

        Example: ``myoptiongroup``

        If you don't specify a value for ``OptionGroupName`` property, a name is automatically created for the option group.
        .. epigraph::

           This value is stored as a lowercase string.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-optiongroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "optionGroupName"))

    @option_group_name.setter
    def option_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3f742d46ce6da6e0764f5b6bfe107957c08c8abf7189415b3700ad3886708478)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "optionGroupName", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnOptionGroup.OptionConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "option_name": "optionName",
            "db_security_group_memberships": "dbSecurityGroupMemberships",
            "option_settings": "optionSettings",
            "option_version": "optionVersion",
            "port": "port",
            "vpc_security_group_memberships": "vpcSecurityGroupMemberships",
        },
    )
    class OptionConfigurationProperty:
        def __init__(
            self,
            *,
            option_name: builtins.str,
            db_security_group_memberships: typing.Optional[typing.Sequence[builtins.str]] = None,
            option_settings: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnOptionGroup.OptionSettingProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            option_version: typing.Optional[builtins.str] = None,
            port: typing.Optional[jsii.Number] = None,
            vpc_security_group_memberships: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The ``OptionConfiguration`` property type specifies an individual option, and its settings, within an ``AWS::RDS::OptionGroup`` resource.

            :param option_name: The configuration of options to include in a group.
            :param db_security_group_memberships: A list of DBSecurityGroupMembership name strings used for this option.
            :param option_settings: The option settings to include in an option group.
            :param option_version: The version for the option.
            :param port: The optional port for the option.
            :param vpc_security_group_memberships: A list of VpcSecurityGroupMembership name strings used for this option.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                option_configuration_property = rds.CfnOptionGroup.OptionConfigurationProperty(
                    option_name="optionName",
                
                    # the properties below are optional
                    db_security_group_memberships=["dbSecurityGroupMemberships"],
                    option_settings=[rds.CfnOptionGroup.OptionSettingProperty(
                        name="name",
                        value="value"
                    )],
                    option_version="optionVersion",
                    port=123,
                    vpc_security_group_memberships=["vpcSecurityGroupMemberships"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__479299edabc04cf4bb7b6245cfeb4df26fffa2ca9908253fbb7ef1aa10603823)
                check_type(argname="argument option_name", value=option_name, expected_type=type_hints["option_name"])
                check_type(argname="argument db_security_group_memberships", value=db_security_group_memberships, expected_type=type_hints["db_security_group_memberships"])
                check_type(argname="argument option_settings", value=option_settings, expected_type=type_hints["option_settings"])
                check_type(argname="argument option_version", value=option_version, expected_type=type_hints["option_version"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument vpc_security_group_memberships", value=vpc_security_group_memberships, expected_type=type_hints["vpc_security_group_memberships"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "option_name": option_name,
            }
            if db_security_group_memberships is not None:
                self._values["db_security_group_memberships"] = db_security_group_memberships
            if option_settings is not None:
                self._values["option_settings"] = option_settings
            if option_version is not None:
                self._values["option_version"] = option_version
            if port is not None:
                self._values["port"] = port
            if vpc_security_group_memberships is not None:
                self._values["vpc_security_group_memberships"] = vpc_security_group_memberships

        @builtins.property
        def option_name(self) -> builtins.str:
            '''The configuration of options to include in a group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfiguration.html#cfn-rds-optiongroup-optionconfiguration-optionname
            '''
            result = self._values.get("option_name")
            assert result is not None, "Required property 'option_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def db_security_group_memberships(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of DBSecurityGroupMembership name strings used for this option.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfiguration.html#cfn-rds-optiongroup-optionconfiguration-dbsecuritygroupmemberships
            '''
            result = self._values.get("db_security_group_memberships")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def option_settings(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOptionGroup.OptionSettingProperty"]]]]:
            '''The option settings to include in an option group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfiguration.html#cfn-rds-optiongroup-optionconfiguration-optionsettings
            '''
            result = self._values.get("option_settings")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOptionGroup.OptionSettingProperty"]]]], result)

        @builtins.property
        def option_version(self) -> typing.Optional[builtins.str]:
            '''The version for the option.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfiguration.html#cfn-rds-optiongroup-optionconfiguration-optionversion
            '''
            result = self._values.get("option_version")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The optional port for the option.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfiguration.html#cfn-rds-optiongroup-optionconfiguration-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def vpc_security_group_memberships(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of VpcSecurityGroupMembership name strings used for this option.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionconfiguration.html#cfn-rds-optiongroup-optionconfiguration-vpcsecuritygroupmemberships
            '''
            result = self._values.get("vpc_security_group_memberships")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OptionConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-rds.CfnOptionGroup.OptionSettingProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class OptionSettingProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``OptionSetting`` property type specifies the value for an option within an ``OptionSetting`` property.

            :param name: The name of the option that has settings that you can set.
            :param value: The current value of the option setting.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionsetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_rds as rds
                
                option_setting_property = rds.CfnOptionGroup.OptionSettingProperty(
                    name="name",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0b8943145c8118ea2143febb92d838b5fe9c28ab3d93824c21b88f726847c630)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the option that has settings that you can set.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionsetting.html#cfn-rds-optiongroup-optionsetting-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The current value of the option setting.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-rds-optiongroup-optionsetting.html#cfn-rds-optiongroup-optionsetting-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OptionSettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CfnOptionGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "engine_name": "engineName",
        "major_engine_version": "majorEngineVersion",
        "option_group_description": "optionGroupDescription",
        "option_configurations": "optionConfigurations",
        "option_group_name": "optionGroupName",
        "tags": "tags",
    },
)
class CfnOptionGroupProps:
    def __init__(
        self,
        *,
        engine_name: builtins.str,
        major_engine_version: builtins.str,
        option_group_description: builtins.str,
        option_configurations: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnOptionGroup.OptionConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        option_group_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnOptionGroup``.

        :param engine_name: Specifies the name of the engine that this option group should be associated with. Valid Values: - ``mariadb`` - ``mysql`` - ``oracle-ee`` - ``oracle-ee-cdb`` - ``oracle-se2`` - ``oracle-se2-cdb`` - ``postgres`` - ``sqlserver-ee`` - ``sqlserver-se`` - ``sqlserver-ex`` - ``sqlserver-web``
        :param major_engine_version: Specifies the major version of the engine that this option group should be associated with.
        :param option_group_description: The description of the option group.
        :param option_configurations: A list of options and the settings for each option.
        :param option_group_name: The name of the option group to be created. Constraints: - Must be 1 to 255 letters, numbers, or hyphens - First character must be a letter - Can't end with a hyphen or contain two consecutive hyphens Example: ``myoptiongroup`` If you don't specify a value for ``OptionGroupName`` property, a name is automatically created for the option group. .. epigraph:: This value is stored as a lowercase string.
        :param tags: An optional array of key-value pairs to apply to this option group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            cfn_option_group_props = rds.CfnOptionGroupProps(
                engine_name="engineName",
                major_engine_version="majorEngineVersion",
                option_group_description="optionGroupDescription",
            
                # the properties below are optional
                option_configurations=[rds.CfnOptionGroup.OptionConfigurationProperty(
                    option_name="optionName",
            
                    # the properties below are optional
                    db_security_group_memberships=["dbSecurityGroupMemberships"],
                    option_settings=[rds.CfnOptionGroup.OptionSettingProperty(
                        name="name",
                        value="value"
                    )],
                    option_version="optionVersion",
                    port=123,
                    vpc_security_group_memberships=["vpcSecurityGroupMemberships"]
                )],
                option_group_name="optionGroupName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__64a04fa714f24960debf9140da1e620aa546468d74857ae93860cfe09f1837a3)
            check_type(argname="argument engine_name", value=engine_name, expected_type=type_hints["engine_name"])
            check_type(argname="argument major_engine_version", value=major_engine_version, expected_type=type_hints["major_engine_version"])
            check_type(argname="argument option_group_description", value=option_group_description, expected_type=type_hints["option_group_description"])
            check_type(argname="argument option_configurations", value=option_configurations, expected_type=type_hints["option_configurations"])
            check_type(argname="argument option_group_name", value=option_group_name, expected_type=type_hints["option_group_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "engine_name": engine_name,
            "major_engine_version": major_engine_version,
            "option_group_description": option_group_description,
        }
        if option_configurations is not None:
            self._values["option_configurations"] = option_configurations
        if option_group_name is not None:
            self._values["option_group_name"] = option_group_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def engine_name(self) -> builtins.str:
        '''Specifies the name of the engine that this option group should be associated with.

        Valid Values:

        - ``mariadb``
        - ``mysql``
        - ``oracle-ee``
        - ``oracle-ee-cdb``
        - ``oracle-se2``
        - ``oracle-se2-cdb``
        - ``postgres``
        - ``sqlserver-ee``
        - ``sqlserver-se``
        - ``sqlserver-ex``
        - ``sqlserver-web``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-enginename
        '''
        result = self._values.get("engine_name")
        assert result is not None, "Required property 'engine_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def major_engine_version(self) -> builtins.str:
        '''Specifies the major version of the engine that this option group should be associated with.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-majorengineversion
        '''
        result = self._values.get("major_engine_version")
        assert result is not None, "Required property 'major_engine_version' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def option_group_description(self) -> builtins.str:
        '''The description of the option group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-optiongroupdescription
        '''
        result = self._values.get("option_group_description")
        assert result is not None, "Required property 'option_group_description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def option_configurations(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnOptionGroup.OptionConfigurationProperty]]]]:
        '''A list of options and the settings for each option.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-optionconfigurations
        '''
        result = self._values.get("option_configurations")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnOptionGroup.OptionConfigurationProperty]]]], result)

    @builtins.property
    def option_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the option group to be created.

        Constraints:

        - Must be 1 to 255 letters, numbers, or hyphens
        - First character must be a letter
        - Can't end with a hyphen or contain two consecutive hyphens

        Example: ``myoptiongroup``

        If you don't specify a value for ``OptionGroupName`` property, a name is automatically created for the option group.
        .. epigraph::

           This value is stored as a lowercase string.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-optiongroupname
        '''
        result = self._values.get("option_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]]:
        '''An optional array of key-value pairs to apply to this option group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-rds-optiongroup.html#cfn-rds-optiongroup-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnOptionGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ClusterEngineBindOptions",
    jsii_struct_bases=[],
    name_mapping={
        "parameter_group": "parameterGroup",
        "s3_export_role": "s3ExportRole",
        "s3_import_role": "s3ImportRole",
    },
)
class ClusterEngineBindOptions:
    def __init__(
        self,
        *,
        parameter_group: typing.Optional["IParameterGroup"] = None,
        s3_export_role: typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole] = None,
        s3_import_role: typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole] = None,
    ) -> None:
        '''The extra options passed to the {@link IClusterEngine.bindToCluster} method.

        :param parameter_group: The customer-provided ParameterGroup. Default: - none
        :param s3_export_role: The role used for S3 exporting. Default: - none
        :param s3_import_role: The role used for S3 importing. Default: - none

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_iam as iam
            import aws_cdk.aws_rds as rds
            
            # parameter_group: rds.ParameterGroup
            # role: iam.Role
            
            cluster_engine_bind_options = rds.ClusterEngineBindOptions(
                parameter_group=parameter_group,
                s3_export_role=role,
                s3_import_role=role
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dbf67d85350e0e43eb11196293d9fc74eddec15cf0fabab7499eee64ca4b18fd)
            check_type(argname="argument parameter_group", value=parameter_group, expected_type=type_hints["parameter_group"])
            check_type(argname="argument s3_export_role", value=s3_export_role, expected_type=type_hints["s3_export_role"])
            check_type(argname="argument s3_import_role", value=s3_import_role, expected_type=type_hints["s3_import_role"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if parameter_group is not None:
            self._values["parameter_group"] = parameter_group
        if s3_export_role is not None:
            self._values["s3_export_role"] = s3_export_role
        if s3_import_role is not None:
            self._values["s3_import_role"] = s3_import_role

    @builtins.property
    def parameter_group(self) -> typing.Optional["IParameterGroup"]:
        '''The customer-provided ParameterGroup.

        :default: - none
        '''
        result = self._values.get("parameter_group")
        return typing.cast(typing.Optional["IParameterGroup"], result)

    @builtins.property
    def s3_export_role(self) -> typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole]:
        '''The role used for S3 exporting.

        :default: - none
        '''
        result = self._values.get("s3_export_role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole], result)

    @builtins.property
    def s3_import_role(self) -> typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole]:
        '''The role used for S3 importing.

        :default: - none
        '''
        result = self._values.get("s3_import_role")
        return typing.cast(typing.Optional[_aws_cdk_aws_iam_940a1ce0.IRole], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ClusterEngineBindOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ClusterEngineConfig",
    jsii_struct_bases=[],
    name_mapping={
        "features": "features",
        "parameter_group": "parameterGroup",
        "port": "port",
    },
)
class ClusterEngineConfig:
    def __init__(
        self,
        *,
        features: typing.Optional[typing.Union["ClusterEngineFeatures", typing.Dict[builtins.str, typing.Any]]] = None,
        parameter_group: typing.Optional["IParameterGroup"] = None,
        port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''The type returned from the {@link IClusterEngine.bindToCluster} method.

        :param features: Features supported by the database engine. Default: - no features
        :param parameter_group: The ParameterGroup to use for the cluster. Default: - no ParameterGroup will be used
        :param port: The port to use for this cluster, unless the customer specified the port directly. Default: - use the default port for clusters (3306)

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            # parameter_group: rds.ParameterGroup
            
            cluster_engine_config = rds.ClusterEngineConfig(
                features=rds.ClusterEngineFeatures(
                    s3_export="s3Export",
                    s3_import="s3Import"
                ),
                parameter_group=parameter_group,
                port=123
            )
        '''
        if isinstance(features, dict):
            features = ClusterEngineFeatures(**features)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7893088bf8af6168ed3bc54c2749884362aa0fee70017bcf76bb14fd54f900c5)
            check_type(argname="argument features", value=features, expected_type=type_hints["features"])
            check_type(argname="argument parameter_group", value=parameter_group, expected_type=type_hints["parameter_group"])
            check_type(argname="argument port", value=port, expected_type=type_hints["port"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if features is not None:
            self._values["features"] = features
        if parameter_group is not None:
            self._values["parameter_group"] = parameter_group
        if port is not None:
            self._values["port"] = port

    @builtins.property
    def features(self) -> typing.Optional["ClusterEngineFeatures"]:
        '''Features supported by the database engine.

        :default: - no features

        :see: https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DBEngineVersion.html
        '''
        result = self._values.get("features")
        return typing.cast(typing.Optional["ClusterEngineFeatures"], result)

    @builtins.property
    def parameter_group(self) -> typing.Optional["IParameterGroup"]:
        '''The ParameterGroup to use for the cluster.

        :default: - no ParameterGroup will be used
        '''
        result = self._values.get("parameter_group")
        return typing.cast(typing.Optional["IParameterGroup"], result)

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        '''The port to use for this cluster, unless the customer specified the port directly.

        :default: - use the default port for clusters (3306)
        '''
        result = self._values.get("port")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ClusterEngineConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.ClusterEngineFeatures",
    jsii_struct_bases=[],
    name_mapping={"s3_export": "s3Export", "s3_import": "s3Import"},
)
class ClusterEngineFeatures:
    def __init__(
        self,
        *,
        s3_export: typing.Optional[builtins.str] = None,
        s3_import: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Represents Database Engine features.

        :param s3_export: Feature name for the DB instance that the IAM role to export to S3 bucket is to be associated with. Default: - no s3Export feature name
        :param s3_import: Feature name for the DB instance that the IAM role to access the S3 bucket for import is to be associated with. Default: - no s3Import feature name

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_rds as rds
            
            cluster_engine_features = rds.ClusterEngineFeatures(
                s3_export="s3Export",
                s3_import="s3Import"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3bd9f7e1837f694c3cff0f16cc0fb5667906c3e9c0c01fc5e400df6103aed694)
            check_type(argname="argument s3_export", value=s3_export, expected_type=type_hints["s3_export"])
            check_type(argname="argument s3_import", value=s3_import, expected_type=type_hints["s3_import"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if s3_export is not None:
            self._values["s3_export"] = s3_export
        if s3_import is not None:
            self._values["s3_import"] = s3_import

    @builtins.property
    def s3_export(self) -> typing.Optional[builtins.str]:
        '''Feature name for the DB instance that the IAM role to export to S3 bucket is to be associated with.

        :default: - no s3Export feature name
        '''
        result = self._values.get("s3_export")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def s3_import(self) -> typing.Optional[builtins.str]:
        '''Feature name for the DB instance that the IAM role to access the S3 bucket for import is to be associated with.

        :default: - no s3Import feature name
        '''
        result = self._values.get("s3_import")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ClusterEngineFeatures(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CommonRotationUserOptions",
    jsii_struct_bases=[],
    name_mapping={
        "automatically_after": "automaticallyAfter",
        "endpoint": "endpoint",
        "exclude_characters": "excludeCharacters",
        "vpc_subnets": "vpcSubnets",
    },
)
class CommonRotationUserOptions:
    def __init__(
        self,
        *,
        automatically_after: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        endpoint: typing.Optional[_aws_cdk_aws_ec2_67de8e8d.IInterfaceVpcEndpoint] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
        vpc_subnets: typing.Optional[typing.Union[_aws_cdk_aws_ec2_67de8e8d.SubnetSelection, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Properties common to single-user and multi-user rotation options.

        :param automatically_after: Specifies the number of days after the previous rotation before Secrets Manager triggers the next automatic rotation. Default: - 30 days
        :param endpoint: The VPC interface endpoint to use for the Secrets Manager API. If you enable private DNS hostnames for your VPC private endpoint (the default), you don't need to specify an endpoint. The standard Secrets Manager DNS hostname the Secrets Manager CLI and SDKs use by default (https://secretsmanager..amazonaws.com) automatically resolves to your VPC endpoint. Default: https://secretsmanager..amazonaws.com
        :param exclude_characters: Specifies characters to not include in generated passwords. Default: " %+~`#$&*()|[]{}:;<>?!'/
        :param vpc_subnets: Where to place the rotation Lambda function. Default: - same placement as instance or cluster

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            import aws_cdk.aws_rds as rds
            import aws_cdk.core as cdk
            
            # interface_vpc_endpoint: ec2.InterfaceVpcEndpoint
            # subnet: ec2.Subnet
            # subnet_filter: ec2.SubnetFilter
            
            common_rotation_user_options = rds.CommonRotationUserOptions(
                automatically_after=cdk.Duration.minutes(30),
                endpoint=interface_vpc_endpoint,
                exclude_characters="excludeCharacters",
                vpc_subnets=ec2.SubnetSelection(
                    availability_zones=["availabilityZones"],
                    one_per_az=False,
                    subnet_filters=[subnet_filter],
                    subnet_group_name="subnetGroupName",
                    subnet_name="subnetName",
                    subnets=[subnet],
                    subnet_type=ec2.SubnetType.ISOLATED
                )
            )
        '''
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _aws_cdk_aws_ec2_67de8e8d.SubnetSelection(**vpc_subnets)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__84e7059de52b64b5ce19ca6789d61059b761d6f27c7c566e3c384dd3136d4a1f)
            check_type(argname="argument automatically_after", value=automatically_after, expected_type=type_hints["automatically_after"])
            check_type(argname="argument endpoint", value=endpoint, expected_type=type_hints["endpoint"])
            check_type(argname="argument exclude_characters", value=exclude_characters, expected_type=type_hints["exclude_characters"])
            check_type(argname="argument vpc_subnets", value=vpc_subnets, expected_type=type_hints["vpc_subnets"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if automatically_after is not None:
            self._values["automatically_after"] = automatically_after
        if endpoint is not None:
            self._values["endpoint"] = endpoint
        if exclude_characters is not None:
            self._values["exclude_characters"] = exclude_characters
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def automatically_after(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''Specifies the number of days after the previous rotation before Secrets Manager triggers the next automatic rotation.

        :default: - 30 days
        '''
        result = self._values.get("automatically_after")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def endpoint(
        self,
    ) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.IInterfaceVpcEndpoint]:
        '''The VPC interface endpoint to use for the Secrets Manager API.

        If you enable private DNS hostnames for your VPC private endpoint (the default), you don't
        need to specify an endpoint. The standard Secrets Manager DNS hostname the Secrets Manager
        CLI and SDKs use by default (https://secretsmanager..amazonaws.com) automatically
        resolves to your VPC endpoint.

        :default: https://secretsmanager..amazonaws.com
        '''
        result = self._values.get("endpoint")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.IInterfaceVpcEndpoint], result)

    @builtins.property
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        '''Specifies characters to not include in generated passwords.

        :default: " %+~`#$&*()|[]{}:;<>?!'/

        :: "\\"
        '''
        result = self._values.get("exclude_characters")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_aws_cdk_aws_ec2_67de8e8d.SubnetSelection]:
        '''Where to place the rotation Lambda function.

        :default: - same placement as instance or cluster
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_aws_cdk_aws_ec2_67de8e8d.SubnetSelection], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommonRotationUserOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Credentials(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-rds.Credentials",
):
    '''Username and password combination.

    :exampleMetadata: infused

    Example::

        # Build a data source for AppSync to access the database.
        # api: appsync.GraphqlApi
        # Create username and password secret for DB Cluster
        secret = rds.DatabaseSecret(self, "AuroraSecret",
            username="clusteradmin"
        )
        
        # The VPC to place the cluster in
        vpc = ec2.Vpc(self, "AuroraVpc")
        
        # Create the serverless cluster, provide all values needed to customise the database.
        cluster = rds.ServerlessCluster(self, "AuroraCluster",
            engine=rds.DatabaseClusterEngine.AURORA_MYSQL,
            vpc=vpc,
            credentials={"username": "clusteradmin"},
            cluster_identifier="db-endpoint-test",
            default_database_name="demos"
        )
        rds_dS = api.add_rds_data_source("rds", cluster, secret, "demos")
        
        # Set up a resolver for an RDS query.
        rds_dS.create_resolver(
            type_name="Query",
            field_name="getDemosRds",
            request_mapping_template=appsync.MappingTemplate.from_string("""
                  {
                    "version": "2018-05-29",
                    "statements": [
                      "SELECT * FROM demos"
                    ]
                  }
                  """),
            response_mapping_template=appsync.MappingTemplate.from_string("""
                    $utils.toJson($utils.rds.toJsonObject($ctx.result)[0])
                  """)
        )
        
        # Set up a resolver for an RDS mutation.
        rds_dS.create_resolver(
            type_name="Mutation",
            field_name="addDemoRds",
            request_mapping_template=appsync.MappingTemplate.from_string("""
                  {
                    "version": "2018-05-29",
                    "statements": [
                      "INSERT INTO demos VALUES (:id, :version)",
                      "SELECT * WHERE id = :id"
                    ],
                    "variableMap": {
                      ":id": $util.toJson($util.autoId()),
                      ":version": $util.toJson($ctx.args.version)
                    }
                  }
                  """),
            response_mapping_template=appsync.MappingTemplate.from_string("""
                    $utils.toJson($utils.rds.toJsonObject($ctx.result)[1][0])
                  """)
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromGeneratedSecret")
    @builtins.classmethod
    def from_generated_secret(
        cls,
        username: builtins.str,
        *,
        encryption_key: typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
        replica_regions: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_aws_secretsmanager_72af8d6f.ReplicaRegion, typing.Dict[builtins.str, typing.Any]]]] = None,
        secret_name: typing.Optional[builtins.str] = None,
    ) -> "Credentials":
        '''Creates Credentials with a password generated and stored in Secrets Manager.

        :param username: -
        :param encryption_key: KMS encryption key to encrypt the generated secret. Default: - default master key
        :param exclude_characters: The characters to exclude from the generated password. Has no effect if {@link password} has been provided. Default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/
        :param replica_regions: A list of regions where to replicate this secret. Default: - Secret is not replicated
        :param secret_name: The name of the secret. Default: - A name is generated by CloudFormation.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__95da9d833ca8d28ff43a7db2a2f078b2486002bfbccab3f333c24ace5c66bd2a)
            check_type(argname="argument username", value=username, expected_type=type_hints["username"])
        options = CredentialsBaseOptions(
            encryption_key=encryption_key,
            exclude_characters=exclude_characters,
            replica_regions=replica_regions,
            secret_name=secret_name,
        )

        return typing.cast("Credentials", jsii.sinvoke(cls, "fromGeneratedSecret", [username, options]))

    @jsii.member(jsii_name="fromPassword")
    @builtins.classmethod
    def from_password(
        cls,
        username: builtins.str,
        password: _aws_cdk_core_f4b25747.SecretValue,
    ) -> "Credentials":
        '''Creates Credentials from a password.

        Do not put passwords in your CDK code directly.

        :param username: -
        :param password: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__da185d8c77ffbb371a873606abe0ff5b79904c0b3d3314d09b0b8d352b7779eb)
            check_type(argname="argument username", value=username, expected_type=type_hints["username"])
            check_type(argname="argument password", value=password, expected_type=type_hints["password"])
        return typing.cast("Credentials", jsii.sinvoke(cls, "fromPassword", [username, password]))

    @jsii.member(jsii_name="fromSecret")
    @builtins.classmethod
    def from_secret(
        cls,
        secret: _aws_cdk_aws_secretsmanager_72af8d6f.ISecret,
        username: typing.Optional[builtins.str] = None,
    ) -> "Credentials":
        '''Creates Credentials from an existing Secrets Manager ``Secret`` (or ``DatabaseSecret``).

        The Secret must be a JSON string with a ``username`` and ``password`` field::

           {
              ...
              "username": <required: username>,
              "password": <required: password>,
           }

        :param secret: The secret where the credentials are stored.
        :param username: The username defined in the secret. If specified the username will be referenced as a string and not a dynamic reference to the username field in the secret. This allows to replace the secret without replacing the instance or cluster.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__86ef7e5b6296cded6719a20863281101f23c75c11d8373678e433b6a00141b8e)
            check_type(argname="argument secret", value=secret, expected_type=type_hints["secret"])
            check_type(argname="argument username", value=username, expected_type=type_hints["username"])
        return typing.cast("Credentials", jsii.sinvoke(cls, "fromSecret", [secret, username]))

    @jsii.member(jsii_name="fromUsername")
    @builtins.classmethod
    def from_username(
        cls,
        username: builtins.str,
        *,
        password: typing.Optional[_aws_cdk_core_f4b25747.SecretValue] = None,
        encryption_key: typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
        replica_regions: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_aws_secretsmanager_72af8d6f.ReplicaRegion, typing.Dict[builtins.str, typing.Any]]]] = None,
        secret_name: typing.Optional[builtins.str] = None,
    ) -> "Credentials":
        '''Creates Credentials for the given username, and optional password and key.

        If no password is provided, one will be generated and stored in Secrets Manager.

        :param username: -
        :param password: Password. Do not put passwords in your CDK code directly. Default: - a Secrets Manager generated password
        :param encryption_key: KMS encryption key to encrypt the generated secret. Default: - default master key
        :param exclude_characters: The characters to exclude from the generated password. Has no effect if {@link password} has been provided. Default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/
        :param replica_regions: A list of regions where to replicate this secret. Default: - Secret is not replicated
        :param secret_name: The name of the secret. Default: - A name is generated by CloudFormation.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4693114d131c0948b9256dc19d9fa021101d60b0837540beecabd6e671375006)
            check_type(argname="argument username", value=username, expected_type=type_hints["username"])
        options = CredentialsFromUsernameOptions(
            password=password,
            encryption_key=encryption_key,
            exclude_characters=exclude_characters,
            replica_regions=replica_regions,
            secret_name=secret_name,
        )

        return typing.cast("Credentials", jsii.sinvoke(cls, "fromUsername", [username, options]))

    @builtins.property
    @jsii.member(jsii_name="username")
    @abc.abstractmethod
    def username(self) -> builtins.str:
        '''Username.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="encryptionKey")
    @abc.abstractmethod
    def encryption_key(self) -> typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey]:
        '''KMS encryption key to encrypt the generated secret.

        :default: - default master key
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="excludeCharacters")
    @abc.abstractmethod
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        '''The characters to exclude from the generated password.

        Only used if {@link password} has not been set.

        :default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :: "\\")
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="password")
    @abc.abstractmethod
    def password(self) -> typing.Optional[_aws_cdk_core_f4b25747.SecretValue]:
        '''Password.

        Do not put passwords in your CDK code directly.

        :default: - a Secrets Manager generated password
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="replicaRegions")
    @abc.abstractmethod
    def replica_regions(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_secretsmanager_72af8d6f.ReplicaRegion]]:
        '''A list of regions where to replicate the generated secret.

        :default: - Secret is not replicated
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="secret")
    @abc.abstractmethod
    def secret(self) -> typing.Optional[_aws_cdk_aws_secretsmanager_72af8d6f.ISecret]:
        '''Secret used to instantiate this Login.

        :default: - none
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="secretName")
    @abc.abstractmethod
    def secret_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the Secret if a new Secret is to be generated in SecretsManager for these Credentials.

        :default: - A name is generated by CloudFormation.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="usernameAsString")
    @abc.abstractmethod
    def username_as_string(self) -> typing.Optional[builtins.bool]:
        '''Whether the username should be referenced as a string and not as a dynamic reference to the username in the secret.

        :default: false
        '''
        ...


class _CredentialsProxy(Credentials):
    @builtins.property
    @jsii.member(jsii_name="username")
    def username(self) -> builtins.str:
        '''Username.'''
        return typing.cast(builtins.str, jsii.get(self, "username"))

    @builtins.property
    @jsii.member(jsii_name="encryptionKey")
    def encryption_key(self) -> typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey]:
        '''KMS encryption key to encrypt the generated secret.

        :default: - default master key
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey], jsii.get(self, "encryptionKey"))

    @builtins.property
    @jsii.member(jsii_name="excludeCharacters")
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        '''The characters to exclude from the generated password.

        Only used if {@link password} has not been set.

        :default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :: "\\")
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "excludeCharacters"))

    @builtins.property
    @jsii.member(jsii_name="password")
    def password(self) -> typing.Optional[_aws_cdk_core_f4b25747.SecretValue]:
        '''Password.

        Do not put passwords in your CDK code directly.

        :default: - a Secrets Manager generated password
        '''
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.SecretValue], jsii.get(self, "password"))

    @builtins.property
    @jsii.member(jsii_name="replicaRegions")
    def replica_regions(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_secretsmanager_72af8d6f.ReplicaRegion]]:
        '''A list of regions where to replicate the generated secret.

        :default: - Secret is not replicated
        '''
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_secretsmanager_72af8d6f.ReplicaRegion]], jsii.get(self, "replicaRegions"))

    @builtins.property
    @jsii.member(jsii_name="secret")
    def secret(self) -> typing.Optional[_aws_cdk_aws_secretsmanager_72af8d6f.ISecret]:
        '''Secret used to instantiate this Login.

        :default: - none
        '''
        return typing.cast(typing.Optional[_aws_cdk_aws_secretsmanager_72af8d6f.ISecret], jsii.get(self, "secret"))

    @builtins.property
    @jsii.member(jsii_name="secretName")
    def secret_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the Secret if a new Secret is to be generated in SecretsManager for these Credentials.

        :default: - A name is generated by CloudFormation.
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "secretName"))

    @builtins.property
    @jsii.member(jsii_name="usernameAsString")
    def username_as_string(self) -> typing.Optional[builtins.bool]:
        '''Whether the username should be referenced as a string and not as a dynamic reference to the username in the secret.

        :default: false
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "usernameAsString"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Credentials).__jsii_proxy_class__ = lambda : _CredentialsProxy


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CredentialsBaseOptions",
    jsii_struct_bases=[],
    name_mapping={
        "encryption_key": "encryptionKey",
        "exclude_characters": "excludeCharacters",
        "replica_regions": "replicaRegions",
        "secret_name": "secretName",
    },
)
class CredentialsBaseOptions:
    def __init__(
        self,
        *,
        encryption_key: typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
        replica_regions: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_aws_secretsmanager_72af8d6f.ReplicaRegion, typing.Dict[builtins.str, typing.Any]]]] = None,
        secret_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Base options for creating Credentials.

        :param encryption_key: KMS encryption key to encrypt the generated secret. Default: - default master key
        :param exclude_characters: The characters to exclude from the generated password. Has no effect if {@link password} has been provided. Default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/
        :param replica_regions: A list of regions where to replicate this secret. Default: - Secret is not replicated
        :param secret_name: The name of the secret. Default: - A name is generated by CloudFormation.

        :exampleMetadata: infused

        Example::

            # vpc: ec2.Vpc
            
            engine = rds.DatabaseInstanceEngine.postgres(version=rds.PostgresEngineVersion.VER_12_3)
            my_key = kms.Key(self, "MyKey")
            
            rds.DatabaseInstance(self, "InstanceWithCustomizedSecret",
                engine=engine,
                vpc=vpc,
                credentials=rds.Credentials.from_generated_secret("postgres",
                    secret_name="my-cool-name",
                    encryption_key=my_key,
                    exclude_characters="!&*^#@()",
                    replica_regions=[secretsmanager.ReplicaRegion(region="eu-west-1"), secretsmanager.ReplicaRegion(region="eu-west-2")]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8dc70b24b85ce95e1fd2ac7e9c72c7887e869c9cba427ac396f991da87edfb75)
            check_type(argname="argument encryption_key", value=encryption_key, expected_type=type_hints["encryption_key"])
            check_type(argname="argument exclude_characters", value=exclude_characters, expected_type=type_hints["exclude_characters"])
            check_type(argname="argument replica_regions", value=replica_regions, expected_type=type_hints["replica_regions"])
            check_type(argname="argument secret_name", value=secret_name, expected_type=type_hints["secret_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if encryption_key is not None:
            self._values["encryption_key"] = encryption_key
        if exclude_characters is not None:
            self._values["exclude_characters"] = exclude_characters
        if replica_regions is not None:
            self._values["replica_regions"] = replica_regions
        if secret_name is not None:
            self._values["secret_name"] = secret_name

    @builtins.property
    def encryption_key(self) -> typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey]:
        '''KMS encryption key to encrypt the generated secret.

        :default: - default master key
        '''
        result = self._values.get("encryption_key")
        return typing.cast(typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey], result)

    @builtins.property
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        '''The characters to exclude from the generated password.

        Has no effect if {@link password} has been provided.

        :default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :: "\\")
        '''
        result = self._values.get("exclude_characters")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def replica_regions(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_secretsmanager_72af8d6f.ReplicaRegion]]:
        '''A list of regions where to replicate this secret.

        :default: - Secret is not replicated
        '''
        result = self._values.get("replica_regions")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_secretsmanager_72af8d6f.ReplicaRegion]], result)

    @builtins.property
    def secret_name(self) -> typing.Optional[builtins.str]:
        '''The name of the secret.

        :default: - A name is generated by CloudFormation.
        '''
        result = self._values.get("secret_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CredentialsBaseOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.CredentialsFromUsernameOptions",
    jsii_struct_bases=[CredentialsBaseOptions],
    name_mapping={
        "encryption_key": "encryptionKey",
        "exclude_characters": "excludeCharacters",
        "replica_regions": "replicaRegions",
        "secret_name": "secretName",
        "password": "password",
    },
)
class CredentialsFromUsernameOptions(CredentialsBaseOptions):
    def __init__(
        self,
        *,
        encryption_key: typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey] = None,
        exclude_characters: typing.Optional[builtins.str] = None,
        replica_regions: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_aws_secretsmanager_72af8d6f.ReplicaRegion, typing.Dict[builtins.str, typing.Any]]]] = None,
        secret_name: typing.Optional[builtins.str] = None,
        password: typing.Optional[_aws_cdk_core_f4b25747.SecretValue] = None,
    ) -> None:
        '''Options for creating Credentials from a username.

        :param encryption_key: KMS encryption key to encrypt the generated secret. Default: - default master key
        :param exclude_characters: The characters to exclude from the generated password. Has no effect if {@link password} has been provided. Default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/
        :param replica_regions: A list of regions where to replicate this secret. Default: - Secret is not replicated
        :param secret_name: The name of the secret. Default: - A name is generated by CloudFormation.
        :param password: Password. Do not put passwords in your CDK code directly. Default: - a Secrets Manager generated password

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_kms as kms
            import aws_cdk.aws_rds as rds
            import aws_cdk.core as cdk
            
            # key: kms.Key
            # secret_value: cdk.SecretValue
            
            credentials_from_username_options = rds.CredentialsFromUsernameOptions(
                encryption_key=key,
                exclude_characters="excludeCharacters",
                password=secret_value,
                replica_regions=[ReplicaRegion(
                    region="region",
            
                    # the properties below are optional
                    encryption_key=key
                )],
                secret_name="secretName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d2a8adce94f9fa257beebcfd8ac34b88106d443bf1f83e6cca958a30373da4f)
            check_type(argname="argument encryption_key", value=encryption_key, expected_type=type_hints["encryption_key"])
            check_type(argname="argument exclude_characters", value=exclude_characters, expected_type=type_hints["exclude_characters"])
            check_type(argname="argument replica_regions", value=replica_regions, expected_type=type_hints["replica_regions"])
            check_type(argname="argument secret_name", value=secret_name, expected_type=type_hints["secret_name"])
            check_type(argname="argument password", value=password, expected_type=type_hints["password"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if encryption_key is not None:
            self._values["encryption_key"] = encryption_key
        if exclude_characters is not None:
            self._values["exclude_characters"] = exclude_characters
        if replica_regions is not None:
            self._values["replica_regions"] = replica_regions
        if secret_name is not None:
            self._values["secret_name"] = secret_name
        if password is not None:
            self._values["password"] = password

    @builtins.property
    def encryption_key(self) -> typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey]:
        '''KMS encryption key to encrypt the generated secret.

        :default: - default master key
        '''
        result = self._values.get("encryption_key")
        return typing.cast(typing.Optional[_aws_cdk_aws_kms_e491a92b.IKey], result)

    @builtins.property
    def exclude_characters(self) -> typing.Optional[builtins.str]:
        '''The characters to exclude from the generated password.

        Has no effect if {@link password} has been provided.

        :default: - the DatabaseSecret default exclude character set (" %+~`#$&*()|[]{}:;<>?!'/

        :: "\\")
        '''
        result = self._values.get("exclude_characters")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def replica_regions(
        self,
    ) -> typing.Optional[typing.List[_aws_cdk_aws_secretsmanager_72af8d6f.ReplicaRegion]]:
        '''A list of regions where to replicate this secret.

        :default: - Secret is not replicated
        '''
        result = self._values.get("replica_regions")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_secretsmanager_72af8d6f.ReplicaRegion]], result)

    @builtins.property
    def secret_name(self) -> typing.Optional[builtins.str]:
        '''The name of the secret.

        :default: - A name is generated by CloudFormation.
        '''
        result = self._values.get("secret_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def password(self) -> typing.Optional[_aws_cdk_core_f4b25747.SecretValue]:
        '''Password.

        Do not put passwords in your CDK code directly.

        :default: - a Secrets Manager generated password
        '''
        result = self._values.get("password")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.SecretValue], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CredentialsFromUsernameOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-rds.DatabaseClusterAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "cluster_identifier": "clusterIdentifier",
        "cluster_endpoint_address": "clusterEndpointAddress",
        "engine": "engine",
        "instance_endpoint_addresses": "instanceEndpointAddresses",
        "instance_identifiers": "instanceIdentifiers",
        "port": "port",
        "reader_endpoint_address": "readerEndpointAddress",
        "security_groups": "securityGroups",
    },
)
class DatabaseClusterAttributes:
    def __init__(
        self,
        *,
        cluster_identifier: builtins.str,
        cluster_endpoint_address: typing.Optional[builtins.str] = None,
        engine: typing.Optional["IClusterEngine"] = None,
        instance_endpoint_addresses: typing.Optional[typing.Sequence[builtins.str]] = None,
        instance_identifiers: typing.Optional[typing.Sequence[builtins.str]] = None,
        port: typing.Optional[jsii.Number] = None,
        reader_endpoint_address: typing.Optional[builtins.str] = None,
        security_groups: typing.Optional[typing.Sequence[_aws_cdk_aws_ec2_67de8e8d.ISecurityGroup]] = None,
    ) -> None:
        '''Properties that describe an existing cluster instance.

        :param cluster_identifier: Identifier for the cluster.
        :param cluster_endpoint_address: Cluster endpoint address. Default: - no endpoint address
        :param engine: The engine of the existing Cluster. Default: - the imported Cluster's engine is unknown
        :param instance_endpoint_addresses: Endpoint addresses of individual instances. Default: - no instance endpoints
        :param instance_identifiers: Identifier for the instances. Default: - no instance identifiers
        :param port: The database port. Default: - none
        :param reader_endpoint_address: Reader endpoint address. Default: - no reader address
        :param security_groups: The security groups of the database cluster. Default: - no security groups

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            import aws_cdk.aws_rds as rds
            
            # cluster_engine: rds.IClusterEngine
            # security_group: ec2.SecurityGroup
            
            database_cluster_attributes = rds.DatabaseClusterAttributes(
                cluster_identifier="clusterIdentifier",
            
                # the properties below are optional
                cluster_endpoint_address="clusterEndpointAddress",
                engine=cluster_engine,
                instance_endpoint_addresses=["instanceEndpointAddresses"],
                instance_identifiers=["instanceIdentifiers"],
                port=123,
                reader_endpoint_address="readerEndpointAddress",
                security_groups=[security_group]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c4f32bee099ed97eb0de1106d000c1a7c34f53025a3b46b716a85fcb0d7b3e27)
            check_type(argname="argument cluster_identifier", value=cluster_identifier, expected_type=type_hints["cluster_identifier"])
            check_type(argname="argument cluster_endpoint_address", value=cluster_endpoint_address, expected_type=type_hints["cluster_endpoint_address"])
            check_type(argname="argument engine", value=engine, expected_type=type_hints["engine"])
            check_type(argname="argument instance_endpoint_addresses", value=instance_endpoint_addresses, expected_ty