"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const monocdk_1 = require("monocdk");
const aws_ec2_1 = require("monocdk/aws-ec2");
const blockedPorts = [20, 21, 3389, 3309, 3306, 4333];
/**
 * EC2 instances have all common TCP ports restricted for ingress IPv4 traffic - (AC-4, CM-2, SC-7, SC-7(3)).
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/default-custom-security-groups.html
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnSecurityGroup) {
        const ingressRules = monocdk_1.Stack.of(node).resolve(node.securityGroupIngress);
        if (ingressRules != undefined) {
            //For each ingress rule, ensure that it does not allow unrestricted SSH traffic.
            for (const rule of ingressRules) {
                const resolvedRule = monocdk_1.Stack.of(node).resolve(rule);
                for (const portNum of blockedPorts) {
                    if (!testPort(resolvedRule, portNum)) {
                        return false;
                    }
                }
            }
        }
    }
    else if (node instanceof aws_ec2_1.CfnSecurityGroupIngress) {
        for (const portNum of blockedPorts) {
            if (!testPort(node, portNum)) {
                return false;
            }
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to identify if the given port number is unrestricted
 * @param rule the CfnSecurityGroupIngress rule to check
 * @param portNum the number of the port to check
 */
function testPort(rule, portNum) {
    //Does this rule apply to TCP traffic?
    if (rule.ipProtocol != undefined && rule.ipProtocol == 'tcp') {
        //Does this rule allow all IPv4 addresses (unrestricted access)?
        if (rule.cidrIp != undefined && rule.cidrIp.includes('/0')) {
            //Is a port range specified?
            if (rule.fromPort != undefined && rule.toPort != undefined) {
                if ((rule.fromPort <= portNum && rule.toPort >= portNum) ||
                    rule.fromPort == -1 ||
                    rule.toPort == -1) {
                    return false;
                }
            }
            else {
                if (rule.fromPort == portNum) {
                    return false;
                }
            }
        }
    }
    //Are all ports allowed?
    if (rule.ipProtocol != undefined && rule.ipProtocol == '-1') {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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