"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HIPAASecurityChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const apigw_1 = require("./rules/apigw");
const autoscaling_1 = require("./rules/autoscaling");
const cloudtrail_1 = require("./rules/cloudtrail");
const cloudwatch_1 = require("./rules/cloudwatch");
const codebuild_1 = require("./rules/codebuild");
const dms_1 = require("./rules/dms");
const dynamodb_1 = require("./rules/dynamodb");
const ec2_1 = require("./rules/ec2");
const ecs_1 = require("./rules/ecs");
const efs_1 = require("./rules/efs");
const elasticache_1 = require("./rules/elasticache");
const elasticbeanstalk_1 = require("./rules/elasticbeanstalk");
const elb_1 = require("./rules/elb");
const opensearch_1 = require("./rules/opensearch");
/**
 * Check for HIPAA Security compliance.
 *
 * Based on the HIPAA Security AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-hipaa_security.html
 *
 * @stability stable
 */
class HIPAASecurityChecks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkAPIGW(node, ignores);
            this.checkAutoScaling(node, ignores);
            this.checkCloudTrail(node, ignores);
            this.checkCloudWatch(node, ignores);
            this.checkCodeBuild(node, ignores);
            this.checkDMS(node, ignores);
            this.checkDynamoDB(node, ignores);
            this.checkEC2(node, ignores);
            this.checkECS(node, ignores);
            this.checkEFS(node, ignores);
            this.checkElastiCache(node, ignores);
            this.checkElasticBeanstalk(node, ignores);
            this.checkELB(node, ignores);
            // this.checkEMR(node, ignores);
            // this.checkIAM(node, ignores);
            // this.checkLambda(node, ignores);
            this.checkOpenSearch(node, ignores);
            // this.checkRDS(node, ignores);
            // this.checkRedshift(node, ignores);
            // this.checkS3(node, ignores);
            // this.checkSageMaker(node, ignores);
            // this.checkSecretsManager(node, ignores);
            // this.checkSNS(node, ignores);
            // this.checkVPC(node, ignores);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-APIGWCacheEnabledAndEncrypted') &&
            !apigw_1.hipaaSecurityAPIGWCacheEnabledAndEncrypted(node)) {
            const ruleId = 'HIPAA.Security-APIGWCacheEnabledAndEncrypted';
            const info = 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-APIGWExecutionLoggingEnabled') &&
            !apigw_1.hipaaSecurityAPIGWExecutionLoggingEnabled(node)) {
            const ruleId = 'HIPAA.Security-APIGWExecutionLoggingEnabled';
            const info = 'The API Gateway stage does not have execution logging enabled for all methods - (Control ID: 164.312(b)).';
            const explanation = 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-APIGWSSLEnabled') &&
            !apigw_1.hipaaSecurityAPIGWSSLEnabled(node)) {
            const ruleId = 'HIPAA.Security-APIGWSSLEnabled';
            const info = 'The API Gateway REST API stage is not configured with SSL certificates - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).';
            const explanation = 'Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-APIGWXrayEnabled') &&
            !apigw_1.hipaaSecurityAPIGWXrayEnabled(node)) {
            const ruleId = 'HIPAA.Security-APIGWXrayEnabled';
            const info = 'The API Gateway REST API stage does not have X-Ray enabled - (Control ID: 164.312(b)).';
            const explanation = 'AWS X-Ray collects data about requests that your application serves, and provides tools you can use to view, filter, and gain insights into that data to identify issues and opportunities for optimization. Ensure X-Ray is enables so you can see detailed information not only about the request and response, but also about calls that your application makes to downstream AWS resources, microservices, databases and HTTP web APIs.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Auto Scaling Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-AutoscalingGroupELBHealthCheckRequired') &&
            !autoscaling_1.hipaaSecurityAutoscalingGroupELBHealthCheckRequired(node)) {
            const ruleId = 'HIPAA.Security-AutoscalingGroupELBHealthCheckRequired';
            const info = 'The Auto Scaling group utilizes a load balancer and does not have an ELB health check configured - (Control ID: 164.312(b)).';
            const explanation = 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-AutoscalingLaunchConfigPublicIpDisabled') &&
            !autoscaling_1.hipaaSecurityAutoscalingLaunchConfigPublicIpDisabled(node)) {
            const ruleId = 'HIPAA.Security-AutoscalingLaunchConfigPublicIpDisabled';
            const info = 'The Auto Scaling launch configuration does not have public IP addresses disabled - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CloudTrail Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudTrailCloudWatchLogsEnabled') &&
            !cloudtrail_1.hipaaSecurityCloudTrailCloudWatchLogsEnabled(node)) {
            const ruleId = 'HIPAA.Security-CloudTrailCloudWatchLogsEnabled';
            const info = 'The trail does not have CloudWatch logs enabled - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).';
            const explanation = 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudTrailEncryptionEnabled') &&
            !cloudtrail_1.hipaaSecurityCloudTrailEncryptionEnabled(node)) {
            const ruleId = 'HIPAA.Security-CloudTrailEncryptionEnabled';
            const info = 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudTrailLogFileValidationEnabled') &&
            !cloudtrail_1.hipaaSecurityCloudTrailLogFileValidationEnabled(node)) {
            const ruleId = 'HIPAA.Security-CloudTrailLogFileValidationEnabled';
            const info = 'The trail does not have log file validation enabled - (Control ID: 164.312(c)(1), 164.312(c)(2)).';
            const explanation = 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CloudWatch Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudWatchAlarmAction') &&
            !cloudwatch_1.hipaaSecurityCloudWatchAlarmAction(node)) {
            const ruleId = 'HIPAA.Security-CloudWatchAlarmAction';
            const info = 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control ID: 164.312(b)).';
            const explanation = 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudWatchLogGroupEncrypted') &&
            !cloudwatch_1.hipaaSecurityCloudWatchLogGroupEncrypted(node)) {
            const ruleId = 'HIPAA.Security-CloudWatchLogGroupEncrypted';
            const info = 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CodeBuild Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CodeBuildProjectEnvVarAwsCred') &&
            !codebuild_1.hipaaSecurityCodeBuildProjectEnvVarAwsCred(node)) {
            const ruleId = 'HIPAA.Security-CodeBuildProjectEnvVarAwsCred';
            const info = 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)).';
            const explanation = 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CodeBuildProjectSourceRepoUrl') &&
            !codebuild_1.hipaaSecurityCodeBuildProjectSourceRepoUrl(node)) {
            const ruleId = 'HIPAA.Security-CodeBuildProjectSourceRepoUrl';
            const info = 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAUTH - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)).';
            const explanation = 'OAUTH is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check DMS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDMS(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-DMSReplicationNotPublic') &&
            !dms_1.hipaaSecurityDMSReplicationNotPublic(node)) {
            const ruleId = 'HIPAA.Security-DMSReplicationNotPublic';
            const info = 'The DMS replication instance is public - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'DMS replication instances can contain sensitive information and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check DynamoDB Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-DynamoDBPITREnabled') &&
            !dynamodb_1.hipaaSecurityDynamoDBPITREnabled(node)) {
            const ruleId = 'HIPAA.Security-DynamoDBPITREnabled';
            const info = 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).';
            const explanation = 'The recovery maintains continuous backups of your table for the last 35 days.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EC2 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEC2(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EC2InstanceDetailedMonitoringEnabled') &&
            !ec2_1.hipaaSecurityEC2InstanceDetailedMonitoringEnabled(node)) {
            const ruleId = 'HIPAA.Security-EC2InstanceDetailedMonitoringEnabled';
            const info = 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: 164.312(b)).';
            const explanation = 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EC2InstancesInVPC') &&
            !ec2_1.hipaaSecurityEC2InstancesInVPC(node)) {
            const ruleId = 'HIPAA.Security-EC2InstancesInVPC';
            const info = 'The EC2 instance is not within a VPC - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EC2CheckNoPublicIPs') &&
            !ec2_1.hipaaSecurityEC2InstanceNoPublicIp(node)) {
            const ruleId = 'HIPAA.Security-EC2CheckNoPublicIPs';
            const info = 'The EC2 instance is associated with a public IP address - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check ECS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkECS(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ECSTaskDefinitionUserForHostMode') &&
            !ecs_1.hipaaSecurityECSTaskDefinitionUserForHostMode(node)) {
            const ruleId = 'HIPAA.Security-ECSTaskDefinitionUserForHostMode';
            const info = "The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)).";
            const explanation = 'If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EFS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEFS(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EFSEncrypted') &&
            !efs_1.hipaaSecurityEFSEncrypted(node)) {
            const ruleId = 'HIPAA.Security-EFSEncrypted';
            const info = 'The EFS does not have encryption at rest enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check ElastiCache Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ElasticacheRedisClusterAutomaticBackup') &&
            !elasticache_1.hipaaSecurityElastiCacheRedisClusterAutomaticBackup(node)) {
            const ruleId = 'HIPAA.Security-ElasticacheRedisClusterAutomaticBackup';
            const info = 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).';
            const explanation = 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Elastic Beanstalk Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkElasticBeanstalk(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ElasticBeanstalkEnhancedHealthReportingEnabled') &&
            !elasticbeanstalk_1.hipaaSecurityElasticBeanstalkEnhancedHealthReportingEnabled(node)) {
            const ruleId = 'HIPAA.Security-ElasticBeanstalkEnhancedHealthReportingEnabled';
            const info = 'The Elastic Beanstalk environment does not have enhanced health reporting enabled - (Control ID: 164.312(b)).';
            const explanation = 'AWS Elastic Beanstalk enhanced health reporting enables a more rapid response to changes in the health of the underlying infrastructure. These changes could result in a lack of availability of the application. Elastic Beanstalk enhanced health reporting provides a status descriptor to gauge the severity of the identified issues and identify possible causes to investigate.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkELB(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ALBHttpDropInvalidHeaderEnabled') &&
            !elb_1.hipaaSecurityALBHttpDropInvalidHeaderEnabled(node)) {
            const ruleId = 'HIPAA.Security-ALBHttpDropInvalidHeaderEnabled';
            const info = 'The ALB does not have invalid HTTP header dropping enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).';
            const explanation = 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ALBHttpToHttpsRedirection') &&
            !elb_1.hipaaSecurityALBHttpToHttpsRedirection(node)) {
            const ruleId = 'HIPAA.Security-ALBHttpToHttpsRedirection';
            const info = "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).";
            const explanation = 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ELBACMCertificateRequired') &&
            !elb_1.hipaaSecurityELBACMCertificateRequired(node)) {
            const ruleId = 'HIPAA.Security-ELBACMCertificateRequired';
            const info = 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ELBCrossZoneBalancingEnabled') &&
            !elb_1.hipaaSecurityELBCrossZoneBalancingEnabled(node)) {
            const ruleId = 'HIPAA.Security-ELBCrossZoneBalancingEnabled';
            const info = 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).';
            const explanation = 'The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ELBDeletionProtectionEnabled') &&
            !elb_1.hipaaSecurityELBDeletionProtectionEnabled(node)) {
            const ruleId = 'HIPAA.Security-ELBDeletionProtectionEnabled';
            const info = 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).';
            const explanation = 'Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ELBLoggingEnabled') &&
            !elb_1.hipaaSecurityELBLoggingEnabled(node)) {
            const ruleId = 'HIPAA.Security-ELBLoggingEnabled';
            const info = 'The ELB does not have logging enabled - (Control ID: 164.312(b)).';
            const explanation = "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ELBTlsHttpsListenersOnly') &&
            !elb_1.hipaaSecurityELBTlsHttpsListenersOnly(node)) {
            const ruleId = 'HIPAA.Security-ELBTlsHttpsListenersOnly';
            const info = 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ELBv2ACMCertificateRequired') &&
            !elb_1.hipaaSecurityELBv2ACMCertificateRequired(node)) {
            const ruleId = 'HIPAA.Security-ELBv2ACMCertificateRequired';
            const info = 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    //   /**
    //    * Check EMR Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkEMR(node: CfnResource, ignores: any): void {}
    //   /**
    //    * Check IAM Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkIAM(node: CfnResource, ignores: any): void {}
    //   /**
    //    * Check Lambda Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkLambda(node: CfnResource, ignores: any): void {}
    /**
     * Check OpenSearch Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-OpenSearchEncryptedAtRest') &&
            !opensearch_1.hipaaSecurityOpenSearchEncryptedAtRest(node)) {
            const ruleId = 'HIPAA.Security-OpenSearchEncryptedAtRest';
            const info = 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-OpenSearchInVPCOnly') &&
            !opensearch_1.hipaaSecurityOpenSearchInVPCOnly(node)) {
            const ruleId = 'HIPAA.Security-OpenSearchInVPCOnly';
            const info = 'The OpenSearch Service domain is not running within a VPC - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'VPCs help secure your AWS resources and provide an extra layer of protection.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-OpenSearchLogsToCloudWatch') &&
            !opensearch_1.hipaaSecurityOpenSearchLogsToCloudWatch(node)) {
            const ruleId = 'HIPAA.Security-OpenSearchLogsToCloudWatch';
            const info = 'The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).';
            const explanation = 'Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-OpenSearchNodeToNodeEncryption') &&
            !opensearch_1.hipaaSecurityOpenSearchNodeToNodeEncryption(node)) {
            const ruleId = 'HIPAA.Security-OpenSearchNodeToNodeEncryption';
            const info = 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
}
exports.HIPAASecurityChecks = HIPAASecurityChecks;
_a = JSII_RTTI_SYMBOL_1;
HIPAASecurityChecks[_a] = { fqn: "monocdk-nag.HIPAASecurityChecks", version: "0.0.89" };
//# sourceMappingURL=data:application/json;base64,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