"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsSolutionsChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const analytics_1 = require("./rules/analytics");
const application_integration_1 = require("./rules/application_integration");
const index_1 = require("./rules/compute/index");
const index_2 = require("./rules/databases/index");
const developer_tools_1 = require("./rules/developer_tools");
const index_3 = require("./rules/machine_learning/index");
const management_and_governance_1 = require("./rules/management_and_governance");
const media_services_1 = require("./rules/media_services");
const index_4 = require("./rules/network_and_delivery/index");
const security_and_compliance_1 = require("./rules/security_and_compliance");
const serverless_1 = require("./rules/serverless");
const index_5 = require("./rules/storage/index");
/**
 * Check Best practices based on AWS Solutions Security Matrix.
 *
 * @stability stable
 */
class AwsSolutionsChecks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkCompute(node, ignores);
            this.checkStorage(node, ignores);
            this.checkDatabases(node, ignores);
            this.checkNetworkDelivery(node, ignores);
            this.checkManagementGovernance(node, ignores);
            this.checkMachineLearning(node, ignores);
            this.checkAnalytics(node, ignores);
            this.checkSecurityCompliance(node, ignores);
            this.checkServerless(node, ignores);
            this.checkApplicationIntegration(node, ignores);
            this.checkMediaServices(node, ignores);
            this.checkDeveloperTools(node, ignores);
        }
    }
    /**
     * Check Compute Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCompute(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-EC23') &&
            !index_1.awsSolutionsEc23(node)) {
            const ruleId = 'AwsSolutions-EC23';
            const info = 'The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.';
            const explanation = 'Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when  all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EC26') &&
            !index_1.awsSolutionsEc26(node)) {
            const ruleId = 'AwsSolutions-EC26';
            const info = 'The EBS volume has encryption disabled.';
            const explanation = "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EC27') &&
            !index_1.awsSolutionsEc27(node)) {
            const ruleId = 'AwsSolutions-EC27';
            const info = 'The Security Group does not have a description.';
            const explanation = 'Descriptions help simplify operations and remove any opportunities for operator errors.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EC28') &&
            !index_1.awsSolutionsEc28(node)) {
            const ruleId = 'AwsSolutions-EC28';
            const info = 'The EC2 instance/AutoScaling launch configuration does not have detailed monitoring enabled.';
            const explanation = 'Monitoring data helps make better decisions on architecting and managing compute resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EC29') &&
            !index_1.awsSolutionsEc29(node)) {
            const ruleId = 'AwsSolutions-EC29';
            const info = 'The EC2 instance is not part of an ASG and has Termination Protection disabled.';
            const explanation = 'Termination Protection safety feature enabled in order to protect the instances from being accidentally terminated.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ECR1') &&
            !index_1.awsSolutionsEcr1(node)) {
            const ruleId = 'AwsSolutions-ECR1';
            const info = 'The ECR Repository allows open access.';
            const explanation = 'Removing * principals in an ECR Repository helps protect against unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ECS4') &&
            !index_1.awsSolutionsEcs4(node)) {
            const ruleId = 'AwsSolutions-ECS4';
            const info = 'The ECS Cluster has CloudWatch Container Insights disabled.';
            const explanation = 'CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ECS7') &&
            !index_1.awsSolutionsEcs7(node)) {
            const ruleId = 'AwsSolutions-ECS7';
            const info = 'The ECS Task Definition does not have awslogs logging enabled at the minimum.';
            const explanation = 'Container logging allows operators to view and aggregate the logs from the container.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ELB1') &&
            !index_1.awsSolutionsElb1(node)) {
            const ruleId = 'AwsSolutions-ELB1';
            const info = 'The CLB is used for incoming HTTP/HTTPS traffic. Use ALBs instead.';
            const explanation = 'HTTP/HTTPS applications (monolithic or containerized) should use the ALB instead of The CLB for enhanced incoming traffic distribution, better performance and lower costs.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ELB2a') &&
            !index_1.awsSolutionsElb2a(node)) {
            const ruleId = 'AwsSolutions-ELB2a';
            const info = 'The ALB does not have access logs enabled.';
            const explanation = 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ELB2e') &&
            !index_1.awsSolutionsElb2e(node)) {
            const ruleId = 'AwsSolutions-ELB2e';
            const info = 'The CLB does not have access logs enabled.';
            const explanation = 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ELB3') &&
            !index_1.awsSolutionsElb3(node)) {
            const ruleId = 'AwsSolutions-ELB3';
            const info = 'The CLB does not have connection draining enabled.';
            const explanation = 'With Connection Draining feature enabled, if an EC2 backend instance fails health checks The CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ELB4') &&
            !index_1.awsSolutionsElb4(node)) {
            const ruleId = 'AwsSolutions-ELB4';
            const info = 'The CLB does not use at least two AZs with the Cross-Zone Load Balancing feature enabled.';
            const explanation = 'CLBs can distribute the traffic evenly across all backend instances. To use Cross-Zone Load Balancing at optimal level, the system should maintain an equal EC2 capacity distribution in each of the AZs registered with the load balancer.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-ELB5') &&
            !index_1.awsSolutionsElb5(node)) {
            const ruleId = 'AwsSolutions-ELB5';
            const info = 'The CLB listener is not configured for secure (HTTPs or SSL) protocols for client communication.';
            const explanation = 'The HTTPs or SSL protocols enable secure communication by encrypting the communication between the client and the load balancer.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Storage Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkStorage(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-S1') && !index_5.awsSolutionsS1(node)) {
            const ruleId = 'AwsSolutions-S1';
            const info = 'The S3 Bucket has server access logs disabled.';
            const explanation = 'The bucket should have server access logging enabled to provide detailed records for the requests that are made to the bucket.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-S2') && !index_5.awsSolutionsS2(node)) {
            const ruleId = 'AwsSolutions-S2';
            const info = 'The S3 Bucket does not have public access restricted and blocked.';
            const explanation = 'The bucket should have public access restricted and blocked to prevent unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-S3') && !index_5.awsSolutionsS3(node)) {
            const ruleId = 'AwsSolutions-S3';
            const info = 'The S3 Bucket does not default encryption enabled.';
            const explanation = 'The bucket should minimally have SSE enabled to help protect data-at-rest.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EFS1') &&
            !index_5.awsSolutionsEfs1(node)) {
            const ruleId = 'AwsSolutions-EFS1';
            const info = 'The EFS is not configured for encryption at rest.';
            const explanation = 'By using an encrypted file system, data and metadata are automatically encrypted before being written to the file system. Similarly, as data and metadata are read, they are automatically decrypted before being presented to the application. These processes are handled transparently by EFS without requiring modification of applications.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Database Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDatabases(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS2') &&
            !index_2.awsSolutionsRds2(node)) {
            const ruleId = 'AwsSolutions-RDS2';
            const info = 'The RDS instance or Aurora cluster does not have storage encryption enabled.';
            const explanation = 'Storage encryption helps protect data-at-rest by encrypting the underlying storage, automated backups, read replicas, and snapshots for the database.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS6') &&
            !index_2.awsSolutionsRds6(node)) {
            const ruleId = 'AwsSolutions-RDS6';
            const info = 'The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.';
            const explanation = "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS10') &&
            !index_2.awsSolutionsRds10(node)) {
            const ruleId = 'AwsSolutions-RDS10';
            const info = 'The RDS instance or Aurora cluster does not have deletion protection enabled.';
            const explanation = 'The deletion protection feature helps protect the database from being accidentally deleted.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS11') &&
            !index_2.awsSolutionsRds11(node)) {
            const ruleId = 'AwsSolutions-RDS11';
            const info = 'The RDS instance or Aurora cluster uses the default endpoint port.';
            const explanation = 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS13') &&
            !index_2.awsSolutionsRds13(node)) {
            const ruleId = 'AwsSolutions-RDS13';
            const info = 'The RDS instance is not configured for automated backups.';
            const explanation = 'Automated backups allow for point-in-time recovery.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS14') &&
            !index_2.awsSolutionsRds14(node)) {
            const ruleId = 'AwsSolutions-RDS14';
            const info = 'The RDS Aurora MySQL cluster does not have Backtrack enabled.';
            const explanation = 'Backtrack helps order to rewind cluster tables to a specific time, without using backups.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RDS16') &&
            !index_2.awsSolutionsRds16(node)) {
            const ruleId = 'AwsSolutions-RDS16';
            const info = 'The RDS Aurora MySQL serverless cluster does not have audit, error, general, and slowquery Log Exports enabled.';
            const explanation = 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DDB3') &&
            !index_2.awsSolutionsDdb3(node)) {
            const ruleId = 'AwsSolutions-DDB3';
            const info = 'The DynamoDB table does not have Point-in-time Recovery enabled.';
            const explanation = 'DynamoDB continuous backups represent an additional layer of insurance against accidental loss of data on top of on-demand backups. The DynamoDB service can back up the data with per-second granularity and restore it to any single second from the time PITR was enabled up to the prior 35 days.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DDB4') &&
            !index_2.awsSolutionsDdb4(node)) {
            const ruleId = 'AwsSolutions-DDB4';
            const info = 'The DAX cluster does not have server-side encryption enabled.';
            const explanation = 'Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AEC1') &&
            !index_2.awsSolutionsAec1(node)) {
            const ruleId = 'AwsSolutions-AEC1';
            const info = 'The ElastiCache cluster is not provisioned in a VPC.';
            const explanation = 'Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AEC3') &&
            !index_2.awsSolutionsAec3(node)) {
            const ruleId = 'AwsSolutions-AEC3';
            const info = 'The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.';
            const explanation = 'Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AEC4') &&
            !index_2.awsSolutionsAec4(node)) {
            const ruleId = 'AwsSolutions-AEC4';
            const info = 'The ElastiCache Redis cluster is not deployed in a Multi-AZ configuration.';
            const explanation = 'The cluster should use a Multi-AZ deployment configuration for high availability.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AEC5') &&
            !index_2.awsSolutionsAec5(node)) {
            const ruleId = 'AwsSolutions-AEC5';
            const info = 'The ElastiCache cluster uses the default endpoint port.';
            const explanation = 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AEC6') &&
            !index_2.awsSolutionsAec6(node)) {
            const ruleId = 'AwsSolutions-AEC6';
            const info = 'The ElastiCache Redis cluster does not use Redis AUTH for user authentication.';
            const explanation = 'Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-N1') && !index_2.awsSolutionsN1(node)) {
            const ruleId = 'AwsSolutions-N1';
            const info = 'The Neptune DB cluster is not deployed in a Multi-AZ configuration.';
            const explanation = 'The cluster should use a Multi-AZ deployment configuration for high availability.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-N2') && !index_2.awsSolutionsN2(node)) {
            const ruleId = 'AwsSolutions-N2';
            const info = 'The Neptune DB instance does have Auto Minor Version Upgrade enabled.';
            const explanation = 'The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-N3') && !index_2.awsSolutionsN3(node)) {
            const ruleId = 'AwsSolutions-N3';
            const info = 'The Neptune DB cluster does not have a reasonable minimum backup retention period configured.';
            const explanation = 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-N4') && !index_2.awsSolutionsN4(node)) {
            const ruleId = 'AwsSolutions-N4';
            const info = 'The Neptune DB cluster does not have encryption at rest enabled.';
            const explanation = 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-N5') && !index_2.awsSolutionsN5(node)) {
            const ruleId = 'AwsSolutions-N5';
            const info = 'The Neptune DB cluster does not have IAM Database Authentication enabled.';
            const explanation = "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS1') &&
            !index_2.awsSolutionsRs1(node)) {
            const ruleId = 'AwsSolutions-RS1';
            const info = 'The Redshift cluster parameter group must have the "require_ssl" parameter enabled.';
            const explanation = 'Enabling the "require_ssl" parameter secures data-in-transit by encrypting the connection between the clients and the Redshift clusters.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS2') &&
            !index_2.awsSolutionsRs2(node)) {
            const ruleId = 'AwsSolutions-RS2';
            const info = 'The Redshift cluster is not provisioned in a VPC.';
            const explanation = 'Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS3') &&
            !index_2.awsSolutionsRs3(node)) {
            const ruleId = 'AwsSolutions-RS3';
            const info = 'The Redshift cluster uses the default "awsuser" username.';
            const explanation = 'Using a custom master user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS4') &&
            !index_2.awsSolutionsRs4(node)) {
            const ruleId = 'AwsSolutions-RS4';
            const info = 'The Redshift cluster uses the default endpoint port.';
            const explanation = 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS5') &&
            !index_2.awsSolutionsRs5(node)) {
            const ruleId = 'AwsSolutions-RS5';
            const info = 'The Redshift cluster does not have audit logging enabled.';
            const explanation = 'Audit logging helps operators troubleshoot issues and ensure security.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS6') &&
            !index_2.awsSolutionsRs6(node)) {
            const ruleId = 'AwsSolutions-RS6';
            const info = 'The Redshift cluster does not have encryption at rest enabled.';
            const explanation = 'Encrypting data-at-rest protects data confidentiality.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS8') &&
            !index_2.awsSolutionsRs8(node)) {
            const ruleId = 'AwsSolutions-RS8';
            const info = 'The Redshift cluster is publicly accessible.';
            const explanation = 'Disabling public accessibility helps minimize security risks.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS9') &&
            !index_2.awsSolutionsRs9(node)) {
            const ruleId = 'AwsSolutions-RS9';
            const info = 'The Redshift cluster does not have version upgrade enabled.';
            const explanation = 'Version Upgrade must enabled to enable the cluster to automatically receive upgrades during the maintenance window.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-RS10') &&
            !index_2.awsSolutionsRs10(node)) {
            const ruleId = 'AwsSolutions-RS10';
            const info = 'The Redshift cluster does not have a retention period for automated snapshots configured.';
            const explanation = 'The retention period represents the number of days to retain automated snapshots. A positive retention period should be set to configure this feature.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DOC1') &&
            !index_2.awsSolutionsDoc1(node)) {
            const ruleId = 'AwsSolutions-DOC1';
            const info = 'The Document DB cluster does not have encryption at rest enabled.';
            const explanation = 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DOC2') &&
            !index_2.awsSolutionsDoc2(node)) {
            const ruleId = 'AwsSolutions-DOC2';
            const info = 'The Document DB cluster uses the default endpoint port.';
            const explanation = 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DOC3') &&
            !index_2.awsSolutionsDoc3(node)) {
            const ruleId = 'AwsSolutions-DOC3';
            const info = 'The Document DB cluster does not have the username and password stored in Secrets Manager.';
            const explanation = "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DOC4') &&
            !index_2.awsSolutionsDoc4(node)) {
            const ruleId = 'AwsSolutions-DOC4';
            const info = 'The Document DB cluster does not have a reasonable minimum backup retention period configured.';
            const explanation = 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-DOC5') &&
            !index_2.awsSolutionsDoc5(node)) {
            const ruleId = 'AwsSolutions-DOC5';
            const info = 'The Document DB cluster does not have authenticate, createIndex, and dropCollection Log Exports enabled.';
            const explanation = 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database. The events recorded by the AWS DocumentDB audit logs include successful and failed authentication attempts, creating indexes or dropping a collection in a database within the DocumentDB cluster.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Network and Delivery Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkNetworkDelivery(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-VPC3') &&
            !index_4.awsSolutionsVpc3(node)) {
            const ruleId = 'AwsSolutions-VPC3';
            const info = 'A Network ACL or Network ACL entry has been implemented.';
            const explanation = 'Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CFR1') &&
            !index_4.awsSolutionsCfr1(node)) {
            const ruleId = 'AwsSolutions-CFR1';
            const info = 'The CloudFront distribution may require Geo restrictions.';
            const explanation = 'Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CFR2') &&
            !index_4.awsSolutionsCfr2(node)) {
            const ruleId = 'AwsSolutions-CFR2';
            const info = 'The CloudFront distribution may require integration with AWS WAF.';
            const explanation = 'The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CFR3') &&
            !index_4.awsSolutionsCfr3(node)) {
            const ruleId = 'AwsSolutions-CFR3';
            const info = 'The CloudFront distributions does not have access logging enabled.';
            const explanation = 'Enabling access logs helps operators track all viewer requests for the content delivered through the Content Delivery Network.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CFR5') &&
            !index_4.awsSolutionsCfr5(node)) {
            const ruleId = 'AwsSolutions-CFR5';
            const info = 'The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.';
            const explanation = 'Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CFR6') &&
            !index_4.awsSolutionsCfr6(node)) {
            const ruleId = 'AwsSolutions-CFR6';
            const info = 'The CloudFront distribution does not use an origin access identity an S3 origin.';
            const explanation = 'Origin access identities help with security by restricting any direct access to objects through S3 URLs.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-APIG1') &&
            !index_4.awsSolutionsApig1(node)) {
            const ruleId = 'AwsSolutions-APIG1';
            const info = 'The API does not have access logging enabled.';
            const explanation = 'Enabling access logs helps operators view who accessed an API and how the caller accessed the API.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-APIG4') &&
            !index_4.awsSolutionsApig4(node)) {
            const ruleId = 'AwsSolutions-APIG4';
            const info = 'The API does not implement authorization.';
            const explanation = 'In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-APIG6') &&
            !index_4.awsSolutionsApig6(node)) {
            const ruleId = 'AwsSolutions-APIG6';
            const info = 'The REST API Stage does not have CloudWatch logging enabled for all methods.';
            const explanation = 'Enabling CloudWatch logs at the stage level helps operators to track and analyze execution behavior at the API stage level.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Management and Governance Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkManagementGovernance(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-AS1') &&
            !management_and_governance_1.awsSolutionsAs1(node)) {
            const ruleId = 'AwsSolutions-AS1';
            const info = 'The Auto Scaling Group does not have a cooldown period.';
            const explanation = 'A cooldown period temporarily suspends any scaling activities in order to allow the newly launched EC2 instance(s) some time to start handling the application traffic.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AS2') &&
            !management_and_governance_1.awsSolutionsAs2(node)) {
            const ruleId = 'AwsSolutions-AS2';
            const info = 'The Auto Scaling Group does not have properly configured health checks.';
            const explanation = 'The health check feature enables the service to detect whether its registered EC2 instances are healthy or not.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-AS3') &&
            !management_and_governance_1.awsSolutionsAs3(node)) {
            const ruleId = 'AwsSolutions-AS3';
            const info = 'The Auto Scaling Group does not have notifications configured for all scaling events.';
            const explanation = 'Notifications on EC2 instance launch, launch error, termination, and termination errors allow operators to gain better insights into systems attributes such as activity and health.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Machine Learning Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkMachineLearning(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-SM1') &&
            !index_3.awsSolutionsSm1(node)) {
            const ruleId = 'AwsSolutions-SM1';
            const info = 'The SageMaker notebook instance is not provisioned inside a VPC.';
            const explanation = 'Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-SM2') &&
            !index_3.awsSolutionsSm2(node)) {
            const ruleId = 'AwsSolutions-SM2';
            const info = 'The SageMaker notebook instance does not have an encrypted storage volume.';
            const explanation = 'Encrypting storage volumes helps protect SageMaker data-at-rest.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-SM3') &&
            !index_3.awsSolutionsSm3(node)) {
            const ruleId = 'AwsSolutions-SM3';
            const info = 'The SageMaker notebook instance has direct internet access enabled.';
            const explanation = 'Disabling public accessibility helps minimize security risks.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Analytics Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAnalytics(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-ATH1') &&
            !analytics_1.awsSolutionsAth1(node)) {
            const ruleId = 'AwsSolutions-ATH1';
            const info = 'The Athena workgroup does not encrypt query results.';
            const explanation = 'Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EMR2') &&
            !analytics_1.awsSolutionsEmr2(node)) {
            const ruleId = 'AwsSolutions-EMR2';
            const info = 'The EMR cluster does not have S3 logging enabled.';
            const explanation = 'Uploading logs to S3 enables the system to keep the logging data for historical purposes or to track and analyze the clusters behavior.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-EMR6') &&
            !analytics_1.awsSolutionsEmr6(node)) {
            const ruleId = 'AwsSolutions-EMR6';
            const info = 'The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.';
            const explanation = 'SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-KDA3') &&
            !analytics_1.awsSolutionsKda3(node)) {
            const ruleId = 'AwsSolutions-KDA3';
            const info = 'The Kinesis Data Analytics Flink Application does not have checkpointing enabled.';
            const explanation = 'Checkpoints are backups of application state that KDA automatically creates periodically and uses to restore from faults.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-KDS1') &&
            !analytics_1.awsSolutionsKds1(node)) {
            const ruleId = 'AwsSolutions-KDS1';
            const info = 'The Kinesis Data Stream does not has server-side encryption enabled.';
            const explanation = "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-KDF1') &&
            !analytics_1.awsSolutionsKdf1(node)) {
            const ruleId = 'AwsSolutions-KDF1';
            const info = 'The Kinesis Data Firehose delivery stream does have server-side encryption enabled.';
            const explanation = 'This allows the system to meet strict regulatory requirements and enhance the security of system data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MSK2') &&
            !analytics_1.awsSolutionsMsk2(node)) {
            const ruleId = 'AwsSolutions-MSK2';
            const info = 'The MSK cluster uses plaintext communication between clients and brokers.';
            const explanation = 'TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MSK3') &&
            !analytics_1.awsSolutionsMsk3(node)) {
            const ruleId = 'AwsSolutions-MSK3';
            const info = 'The MSK cluster uses plaintext communication between brokers.';
            const explanation = 'TLS communication secures data-in-transit by encrypting the connection between brokers.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MSK6') &&
            !analytics_1.awsSolutionsMsk6(node)) {
            const ruleId = 'AwsSolutions-MSK6';
            const info = 'The MSK cluster does not send broker logs to a supported destination.';
            const explanation = 'Broker logs enable operators to troubleshoot Apache Kafka applications and to analyze their communications with the MSK cluster. The cluster can deliver logs to the following resources: a CloudWatch log group, an S3 bucket, a Kinesis Data Firehose delivery stream.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-OS1') &&
            !analytics_1.awsSolutionsOs1(node)) {
            const ruleId = 'AwsSolutions-OS1';
            const info = 'The OpenSearch Service domain is not provisioned inside a VPC.';
            const explanation = 'Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-OS2') &&
            !analytics_1.awsSolutionsOs2(node)) {
            const ruleId = 'AwsSolutions-OS2';
            const info = 'The OpenSearch Service domain does not have node-to-node encryption enabled.';
            const explanation = 'Enabling the node-to-node encryption feature adds an extra layer of data protection on top of the existing ES security features such as HTTPS client to cluster encryption and data-at-rest encryption.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-OS3') &&
            !analytics_1.awsSolutionsOs3(node)) {
            const ruleId = 'AwsSolutions-OS3';
            const info = 'The OpenSearch Service domain does not only grant access via allowlisted IP addresses.';
            const explanation = 'Using allowlisted IP addresses helps protect the domain against unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-OS4') &&
            !analytics_1.awsSolutionsOs4(node)) {
            const ruleId = 'AwsSolutions-OS4';
            const info = 'The OpenSearch Service domain does not use dedicated master nodes.';
            const explanation = 'Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-OS5') &&
            !analytics_1.awsSolutionsOs5(node)) {
            const ruleId = 'AwsSolutions-OS5';
            const info = 'The OpenSearch Service domain does not allow for unsigned requests or anonymous access.';
            const explanation = 'Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-OS7') &&
            !analytics_1.awsSolutionsOs7(node)) {
            const ruleId = 'AwsSolutions-OS7';
            const info = 'The OpenSearch Service domain does not have Zone Awareness enabled.';
            const explanation = 'Enabling cross-zone replication (Zone Awareness) increases the availability of the OpenSearch Service domain by allocating the nodes and replicate the data across two AZs in the same region in order to prevent data loss and minimize downtime in the event of node or AZ failure.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-OS8') &&
            !analytics_1.awsSolutionsOs8(node)) {
            const ruleId = 'AwsSolutions-OS8';
            const info = 'The OpenSearch Service domain does not have encryption at rest enabled.';
            const explanation = 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-OS9') &&
            !analytics_1.awsSolutionsOs9(node)) {
            const ruleId = 'AwsSolutions-OS9';
            const info = 'The OpenSearch Service domain does not minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs.';
            const explanation = 'These logs enable operators to gain full insight into the performance of these operations.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-QS1') &&
            !analytics_1.awsSolutionsQs1(node)) {
            const ruleId = 'AwsSolutions-QS1';
            const info = 'The Quicksight data sources connection is not configured to use SSL.';
            const explanation = 'SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Security and Compliance Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkSecurityCompliance(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-IAM4') &&
            !security_and_compliance_1.awsSolutionsIam4(node)) {
            const ruleId = 'AwsSolutions-IAM4';
            const info = 'The IAM user, role, or group uses AWS managed policies.';
            const explanation = 'An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-IAM5') &&
            !security_and_compliance_1.awsSolutionsIam5(node)) {
            const ruleId = 'AwsSolutions-IAM5';
            const info = 'The IAM entity contains wildcard permissions and does not have a cdk_nag rule suppression with evidence for those permission.';
            const explanation = 'Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-COG1') &&
            !security_and_compliance_1.awsSolutionsCog1(node)) {
            const ruleId = 'AwsSolutions-COG1';
            const info = 'The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.';
            const explanation = 'Strong password policies increase system security by encouraging users to create reliable and secure passwords.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-COG2') &&
            !security_and_compliance_1.awsSolutionsCog2(node)) {
            const ruleId = 'AwsSolutions-COG2';
            const info = 'The Cognito user pool does not require MFA.';
            const explanation = 'Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-COG3') &&
            !security_and_compliance_1.awsSolutionsCog3(node)) {
            const ruleId = 'AwsSolutions-COG3';
            const info = 'The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.';
            const explanation = 'Advanced security features enable the system to detect and act upon malicious sign-in attempts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-COG7') &&
            !security_and_compliance_1.awsSolutionsCog7(node)) {
            const ruleId = 'AwsSolutions-COG7';
            const info = 'The Cognito identity pool allows for unauthenticated logins and does not have a cdk_nag rule suppression with a reason.';
            const explanation = 'In many cases applications do not warrant unauthenticated guest access applications. Metadata explaining the use case allows for transparency to operators.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-KMS5') &&
            !security_and_compliance_1.awsSolutionsKms5(node)) {
            const ruleId = 'AwsSolutions-KMS5';
            const info = 'The KMS Symmetric key does not have Key Rotation enabled.';
            const explanation = 'KMS Key Rotation allow a system to set an yearly rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Serverless Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkServerless(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-SF1') &&
            !serverless_1.awsSolutionsSf1(node)) {
            const ruleId = 'AwsSolutions-SF1';
            const info = 'The Step Function does not log "ALL" events to CloudWatch Logs.';
            const explanation = 'Logging "ALL" events to CloudWatch logs help operators troubleshoot and audit systems.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-SF2') &&
            !serverless_1.awsSolutionsSf2(node)) {
            const ruleId = 'AwsSolutions-SF2';
            const info = 'The Step Function does not have X-Ray tracing enabled.';
            const explanation = 'X-ray provides an end-to-end view of how an application is performing. This helps operators to discover performance issues, detect permission problems, and track requests made to and from other AWS services.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Application Integration Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkApplicationIntegration(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-SNS2') &&
            !application_integration_1.awsSolutionsSns2(node)) {
            const ruleId = 'AwsSolutions-SNS2';
            const info = 'The SNS Topic does not have server-side encryption enabled.';
            const explanation = 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-SQS2') &&
            !application_integration_1.awsSolutionsSqs2(node)) {
            const ruleId = 'AwsSolutions-SQS2';
            const info = 'The SQS Queue does not have server-side encryption enabled.';
            const explanation = 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-SQS3') &&
            !application_integration_1.awsSolutionsSqs3(node)) {
            const ruleId = 'AwsSolutions-SQS3';
            const info = 'The SQS queue does not have a dead-letter queue (DLQ) enabled or have a cdk_nag rule suppression indicating it is a DLQ.';
            const explanation = 'Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Media Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkMediaServices(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-MS1') &&
            !media_services_1.awsSolutionsMs1(node)) {
            const ruleId = 'AwsSolutions-MS1';
            const info = 'The MediaStore container does not have container access logging enabled.';
            const explanation = 'The container should have access logging enabled to provide detailed records for the requests that are made to the container.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MS4') &&
            !media_services_1.awsSolutionsMs4(node)) {
            const ruleId = 'AwsSolutions-MS4';
            const info = 'The MediaStore container does not define a metric policy to send metrics to CloudWatch.';
            const explanation = 'Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MS7') &&
            !media_services_1.awsSolutionsMs7(node)) {
            const ruleId = 'AwsSolutions-MS7';
            const info = 'The MediaStore container does not define a container policy.';
            const explanation = 'Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MS8') &&
            !media_services_1.awsSolutionsMs8(node)) {
            const ruleId = 'AwsSolutions-MS8';
            const info = 'The MediaStore container does not define a CORS policy.';
            const explanation = 'Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-MS10') &&
            !media_services_1.awsSolutionsMs10(node)) {
            const ruleId = 'AwsSolutions-MS10';
            const info = 'The MediaStore container does not define a lifecycle policy.';
            const explanation = 'Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Developer Tools Services
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDeveloperTools(node, ignores) {
        if (!this.ignoreRule(ignores, 'AwsSolutions-CB3') &&
            !developer_tools_1.awsSolutionsCb3(node)) {
            const ruleId = 'AwsSolutions-CB3';
            const info = 'The CodeBuild project has privileged mode enabled.';
            const explanation = 'Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CB4') &&
            !developer_tools_1.awsSolutionsCb4(node)) {
            const ruleId = 'AwsSolutions-CB4';
            const info = 'The CodeBuild project does not use an AWS KMS key for encryption.';
            const explanation = 'Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-CB5') &&
            !developer_tools_1.awsSolutionsCb5(node)) {
            const ruleId = 'AwsSolutions-CB5';
            const info = 'The Codebuild project does not use images provided by the CodeBuild service or have a cdk_nag suppression rule explaining the need for a custom image.';
            const explanation = 'Explaining differences/edits to Docker images helps operators better understand system dependencies.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'AwsSolutions-C91') &&
            !developer_tools_1.awsSolutionsC91(node)) {
            const ruleId = 'AwsSolutions-C91';
            const info = 'The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.';
            const explanation = 'SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
}
exports.AwsSolutionsChecks = AwsSolutionsChecks;
_a = JSII_RTTI_SYMBOL_1;
AwsSolutionsChecks[_a] = { fqn: "monocdk-nag.AwsSolutionsChecks", version: "0.0.89" };
//# sourceMappingURL=data:application/json;base64,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