"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Portfolio = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const util_1 = require("./private/util");
const validation_1 = require("./private/validation");
const servicecatalog_generated_1 = require("./servicecatalog.generated");
class PortfolioBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.associatedPrincipals = new Set();
    }
    /**
     * (experimental) Associate portfolio with an IAM Role.
     *
     * @experimental
     */
    giveAccessToRole(role) {
        this.associatePrincipal(role.roleArn, role.node.addr);
    }
    /**
     * (experimental) Associate portfolio with an IAM User.
     *
     * @experimental
     */
    giveAccessToUser(user) {
        this.associatePrincipal(user.userArn, user.node.addr);
    }
    /**
     * (experimental) Associate portfolio with an IAM Group.
     *
     * @experimental
     */
    giveAccessToGroup(group) {
        this.associatePrincipal(group.groupArn, group.node.addr);
    }
    /**
     * (experimental) Initiate a portfolio share with another account.
     *
     * @experimental
     */
    shareWithAccount(accountId, options = {}) {
        const hashId = this.generateUniqueHash(accountId);
        new servicecatalog_generated_1.CfnPortfolioShare(this, `PortfolioShare${hashId}`, {
            portfolioId: this.portfolioId,
            accountId: accountId,
            shareTagOptions: options.shareTagOptions,
            acceptLanguage: options.acceptLanguage,
        });
    }
    /**
     * Associate a principal with the portfolio.
     * If the principal is already associated, it will skip.
     */
    associatePrincipal(principalArn, principalId) {
        if (!this.associatedPrincipals.has(principalArn)) {
            const hashId = this.generateUniqueHash(principalId);
            new servicecatalog_generated_1.CfnPortfolioPrincipalAssociation(this, `PortolioPrincipalAssociation${hashId}`, {
                portfolioId: this.portfolioId,
                principalArn: principalArn,
                principalType: 'IAM',
            });
            this.associatedPrincipals.add(principalArn);
        }
    }
}
/**
 * (experimental) A Service Catalog portfolio.
 *
 * @experimental
 */
class Portfolio extends PortfolioBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.validatePortfolioProps(props);
        this.portfolio = new servicecatalog_generated_1.CfnPortfolio(this, 'Resource', {
            displayName: props.displayName,
            providerName: props.providerName,
            description: props.description,
            acceptLanguage: props.acceptLanguage,
        });
        this.portfolioId = this.portfolio.ref;
        this.portfolioArn = core_1.Stack.of(this).formatArn({
            service: 'servicecatalog',
            resource: 'portfolio',
            resourceName: this.portfolioId,
        });
    }
    /**
     * (experimental) Creates a Portfolio construct that represents an external portfolio.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param portfolioArn the Amazon Resource Name of the existing portfolio.
     * @experimental
     */
    static fromPortfolioArn(scope, id, portfolioArn) {
        const arn = core_1.Stack.of(scope).parseArn(portfolioArn);
        const portfolioId = arn.resourceName;
        if (!portfolioId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + portfolioArn);
        }
        class Import extends PortfolioBase {
            constructor() {
                super(...arguments);
                this.portfolioArn = portfolioArn;
                this.portfolioId = portfolioId;
            }
            generateUniqueHash(value) {
                return util_1.hashValues(this.portfolioArn, value);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: portfolioArn,
        });
    }
    /**
     * (experimental) Create a unique id based off the L1 CfnPortfolio or the arn of an imported portfolio.
     *
     * @experimental
     */
    generateUniqueHash(value) {
        return util_1.hashValues(core_1.Names.nodeUniqueId(this.portfolio.node), value);
    }
    validatePortfolioProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio display name', 1, 100, props.displayName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio provider name', 1, 50, props.providerName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio description', 0, 2000, props.description);
    }
}
exports.Portfolio = Portfolio;
_a = JSII_RTTI_SYMBOL_1;
Portfolio[_a] = { fqn: "@aws-cdk/aws-servicecatalog.Portfolio", version: "1.109.0" };
//# sourceMappingURL=data:application/json;base64,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