"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
/**
 * Use an SQS queue as a subscription target
 */
class SqsSubscription {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
    }
    bind(topic) {
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!core_1.Construct.isConstruct(this.queue)) {
            throw new Error('The supplied Queue object must be an instance of Construct');
        }
        // add a statement to the queue resource policy which allows this topic
        // to send messages to the queue.
        this.queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.queue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [new iam.ServicePrincipal('sns.amazonaws.com')],
            conditions: {
                ArnEquals: { 'aws:SourceArn': topic.topicArn },
            },
        }));
        return {
            subscriberScope: this.queue,
            subscriberId: topic.node.uniqueId,
            endpoint: this.queue.queueArn,
            protocol: sns.SubscriptionProtocol.SQS,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
            region: this.regionFromArn(topic),
        };
    }
    regionFromArn(topic) {
        // no need to specify `region` for topics defined within the same stack
        if (topic instanceof sns.Topic) {
            return undefined;
        }
        return core_1.Stack.of(topic).parseArn(topic.topicArn).region;
    }
}
exports.SqsSubscription = SqsSubscription;
//# sourceMappingURL=data:application/json;base64,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