"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscalingManager = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const runner_configuration_1 = require("../runner-configuration");
const DEFAULT_SSH_KEY_PATH = "/etc/gitlab-runner/ssh-custom";
/**
 * Settings for the manager (coordinator)
 *
 *  Manager coordinates the placement of runner (job executor) instances
 */
class GitlabRunnerAutoscalingManager extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.globalConfiguration =
            props.globalConfiguration ||
                {
                    concurrent: 10,
                    checkInterval: 0,
                    logFormat: "runner",
                    logLevel: "info",
                };
        this.machineImage =
            props.machineImage ??
                aws_ec2_1.MachineImage.latestAmazonLinux({
                    generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
                    edition: aws_ec2_1.AmazonLinuxEdition.STANDARD,
                    virtualization: aws_ec2_1.AmazonLinuxVirt.HVM,
                    storage: aws_ec2_1.AmazonLinuxStorage.EBS,
                    cpuType: aws_ec2_1.AmazonLinuxCpuType.X86_64,
                });
        this.instanceType = props.instanceType ?? aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.NANO);
        this.keyPairName = props.keyPairName;
        this.runners = props.runners;
        this.network = props.network;
        this.cacheBucket = props.cacheBucket;
        this.runnersSecurityGroupName = props.runnersSecurityGroup.securityGroupName;
        this.role =
            props.role ||
                new aws_iam_1.Role(scope, "ManagerRole", {
                    assumedBy: new aws_iam_1.ServicePrincipal("ec2.amazonaws.com", {}),
                    managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName("AmazonSSMManagedInstanceCore")],
                    inlinePolicies: {
                        Cache: aws_iam_1.PolicyDocument.fromJson({
                            Version: "2012-10-17",
                            Statement: [
                                {
                                    Effect: "Allow",
                                    Action: ["s3:ListObjects*", "s3:GetObject*", "s3:DeleteObject*", "s3:PutObject*"],
                                    Resource: [`${this.cacheBucket.bucketArn}/*`],
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["s3:ListBucket"],
                                    Resource: [`${this.cacheBucket.bucketArn}`],
                                },
                            ],
                        }),
                        Runners: aws_iam_1.PolicyDocument.fromJson({
                            Version: "2012-10-17",
                            Statement: [
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:CreateKeyPair", "ec2:DeleteKeyPair", "ec2:ImportKeyPair", "ec2:Describe*"],
                                    Resource: ["*"],
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:CreateTags", "ssm:UpdateInstanceInformation"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringLike: {
                                            "aws:RequestTag/Name": "*gitlab-runner-*",
                                        },
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["Name"],
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:RequestSpotInstances", "ec2:CancelSpotInstanceRequests"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringEqualsIfExists: {
                                            "ec2:Region": `${aws_cdk_lib_1.Stack.of(this).region}`,
                                        },
                                        ArnEqualsIfExists: {
                                            "ec2:Vpc": `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(this).region}:${aws_cdk_lib_1.Stack.of(this).account}:vpc/${this.network.vpc.vpcId}`,
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:RunInstances"],
                                    Resource: ["*"],
                                    Condition: {
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["InstanceProfile"],
                                            "ec2:InstanceType": (this.runners || []).map((runner) => {
                                                const runnersInstanceType = (this.runners && runner.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
                                                return runnersInstanceType.toString();
                                            }),
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:TerminateInstances", "ec2:StopInstances", "ec2:StartInstances", "ec2:RebootInstances"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringLike: {
                                            "ec2:ResourceTag/Name": "*gitlab-runner-*",
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["iam:PassRole"],
                                    Resource: ["*"],
                                    Condition: {
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["RunnersRole"],
                                        },
                                    },
                                },
                            ],
                        }),
                    },
                });
        this.userData = aws_ec2_1.UserData.forLinux({});
        this.userData.addCommands(`yum update -y aws-cfn-bootstrap` // !/bin/bash -xe
        );
        // https://github.com/awslabs/amazon-ecr-credential-helper
        const userDataRunners = aws_ec2_1.UserData.forLinux({});
        userDataRunners.addCommands(`[ ! -z "$(which apt-get)" ] && apt-get install -y amazon-ecr-credential-helper`, `[ ! -z "$(which yum)" ] && yum install -y amazon-ecr-credential-helper`);
        const gitlabRunnerConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        gitlabRunnerConfigRestartHandle._addFile("/etc/gitlab-runner/config.toml");
        const rsyslogConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        rsyslogConfigRestartHandle._addFile("/etc/rsyslog.d/25-gitlab-runner.conf");
        /**
         * Config set keys
         * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html#aws-resource-init-configsets
         */
        const REPOSITORIES = "repositories";
        const PACKAGES = "packages";
        const CONFIG = "config";
        const RESTART = "restart";
        this.initConfig = aws_ec2_1.CloudFormationInit.fromConfigSets({
            configSets: {
                default: [REPOSITORIES, PACKAGES, CONFIG, RESTART],
            },
            configs: {
                [REPOSITORIES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://packages.gitlab.com/install/repositories/runner/gitlab-runner/script.rpm.sh | bash", { key: "10-gitlab-runner" }),
                ]),
                [PACKAGES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitPackage.yum("docker"),
                    aws_ec2_1.InitPackage.yum("gitlab-runner"),
                    aws_ec2_1.InitPackage.yum("tzdata"),
                    aws_ec2_1.InitPackage.yum("jq"),
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://gitlab-docker-machine-downloads.s3.amazonaws.com/v0.16.2-gitlab.12/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine", 
                    //"curl -L https://github.com/docker/machine/releases/download/v0.16.2/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine",
                    { key: "10-docker-machine" }),
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner start", {
                        key: "20-gitlab-runner-start",
                    }),
                ]),
                [CONFIG]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitFile.fromString("/etc/gitlab-runner/config.toml", runner_configuration_1.ConfigurationMapper.withDefaults({
                        globalConfiguration: this.globalConfiguration,
                        runnersConfiguration: this.runners.map((runner) => {
                            const configuration = runner.configuration;
                            return {
                                ...configuration,
                                machine: {
                                    ...configuration.machine,
                                    machineOptions: {
                                        sshKeypath: 
                                        // drivers/amazonec2/amazonec2.go SSHPrivateKeyPath
                                        runner.keyPair
                                            ? `${DEFAULT_SSH_KEY_PATH}/${configuration.machine?.machineOptions?.keypairName}`
                                            : "",
                                        ...configuration.machine?.machineOptions,
                                        instanceType: runner.instanceType.toString(),
                                        ami: runner.machineImage.getImage(scope).imageId,
                                        region: aws_cdk_lib_1.Stack.of(this).region,
                                        vpcId: this.network.vpc.vpcId,
                                        zone: this.network.availabilityZone.slice(-1),
                                        subnetId: this.network.subnet.subnetId,
                                        securityGroup: this.runnersSecurityGroupName,
                                        privateAddressOnly: configuration.machine?.machineOptions?.privateAddressOnly ?? this.network.hasPrivateSubnets(),
                                        usePrivateAddress: configuration.machine?.machineOptions?.usePrivateAddress ?? true,
                                        iamInstanceProfile: runner.instanceProfile.ref,
                                        userdata: "/etc/gitlab-runner/user_data_runners",
                                    },
                                },
                                cache: {
                                    s3: {
                                        serverAddress: `s3.${aws_cdk_lib_1.Stack.of(this).urlSuffix}`,
                                        bucketName: `${this.cacheBucket.bucketName}`,
                                        bucketLocation: `${aws_cdk_lib_1.Stack.of(this).region}`,
                                        authenticationType: "iam",
                                    },
                                },
                            };
                        }),
                    }).toToml(), {
                        owner: "gitlab-runner",
                        group: "gitlab-runner",
                        mode: "000600",
                    }),
                    aws_ec2_1.InitFile.fromString("/etc/rsyslog.d/25-gitlab-runner.conf", `:programname, isequal, "gitlab-runner" /var/log/gitlab-runner.log`, {
                        owner: "root",
                        group: "root",
                        mode: "000644",
                    }),
                    aws_ec2_1.InitService.enable("gitlab-runner", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: gitlabRunnerConfigRestartHandle,
                    }),
                    aws_ec2_1.InitService.enable("rsyslog", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: rsyslogConfigRestartHandle,
                    }),
                    aws_ec2_1.InitCommand.shellCommand(
                    // Download custom EC2 key pair for manager <> runner ssh connect
                    this.runners
                        .map((runner) => {
                        if (!runner.keyPair) {
                            return "";
                        }
                        runner.keyPair.grantRead(this.role);
                        const region = aws_cdk_lib_1.Stack.of(this).region;
                        const secretArn = runner.keyPair.secretArn;
                        const keyPairName = runner.configuration.machine.machineOptions.keypairName;
                        const sshKeyPath = runner.configuration.machine.machineOptions.sshKeypath ?? DEFAULT_SSH_KEY_PATH;
                        return [
                            `mkdir -p ${sshKeyPath};`,
                            `echo $(aws secretsmanager get-secret-value --region ${region} --secret-id ${secretArn} --query SecretString --output text) | jq -r '."${keyPairName}"' > ${sshKeyPath}/${keyPairName};`,
                            `echo $(aws secretsmanager get-secret-value --region ${region} --secret-id ${secretArn} --query SecretString --output text) | jq -r '."${keyPairName}.pub"' > ${sshKeyPath}/${keyPairName}.pub;`,
                        ].join("\n");
                    })
                        .filter((s) => s.length > 0)
                        .join("\n"), {
                        key: "999-retrieve-ec2-key-pair",
                    }),
                    aws_ec2_1.InitFile.fromString("/etc/gitlab-runner/user_data_runners", userDataRunners.render(), {
                        owner: "gitlab-runner",
                        group: "gitlab-runner",
                        mode: "000600",
                    }),
                ]),
                [RESTART]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner restart", {
                        key: "10-gitlab-runner-restart",
                    }),
                ]),
            },
        });
    }
}
exports.GitlabRunnerAutoscalingManager = GitlabRunnerAutoscalingManager;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscalingManager[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.GitlabRunnerAutoscalingManager", version: "0.2.231" };
//# sourceMappingURL=data:application/json;base64,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