"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
function testStack() {
    const stack = new cdk.Stack(undefined, undefined, { env: { account: '12345', region: 'us-test-1' } });
    stack.node.setContext('availability-zones:12345:us-test-1', ['us-test-1a', 'us-test-1b']);
    return stack;
}
module.exports = {
    'check that instantiation works'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Properties: {
                Engine: "aurora",
                DBSubnetGroupName: { Ref: "DatabaseSubnets56F17B9A" },
                MasterUsername: "admin",
                MasterUserPassword: "tooshort",
                VpcSecurityGroupIds: [{ "Fn::GetAtt": ["DatabaseSecurityGroup5C91FDCB", "GroupId"] }]
            },
            DeletionPolicy: 'Retain',
            UpdateReplacePolicy: 'Retain'
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DeletionPolicy: 'Retain',
            UpdateReplacePolicy: 'Retain'
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'can create a cluster with a single instance'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: "aurora",
            DBSubnetGroupName: { Ref: "DatabaseSubnets56F17B9A" },
            MasterUsername: "admin",
            MasterUserPassword: "tooshort",
            VpcSecurityGroupIds: [{ "Fn::GetAtt": ["DatabaseSecurityGroup5C91FDCB", "GroupId"] }]
        }));
        test.done();
    },
    'can create a cluster with imported vpc and security group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = ec2.Vpc.fromLookup(stack, 'VPC', {
            vpcId: "VPC12345"
        });
        const sg = ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', "SecurityGroupId12345");
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
                securityGroup: sg
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: "aurora",
            DBSubnetGroupName: { Ref: "DatabaseSubnets56F17B9A" },
            MasterUsername: "admin",
            MasterUserPassword: "tooshort",
            VpcSecurityGroupIds: ["SecurityGroupId12345"]
        }));
        test.done();
    },
    'cluster with parameter group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const group = new lib_1.ClusterParameterGroup(stack, 'Params', {
            family: 'hello',
            description: 'bye',
            parameters: {
                param: 'value'
            }
        });
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc
            },
            parameterGroup: group
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            DBClusterParameterGroupName: { Ref: 'ParamsA8366201' },
        }));
        test.done();
    },
    'creates a secret when master credentials are not specified'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: {
                username: 'admin'
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            MasterUsername: {
                'Fn::Join': [
                    '',
                    [
                        '{{resolve:secretsmanager:',
                        {
                            Ref: 'DatabaseSecret3B817195'
                        },
                        ':SecretString:username::}}'
                    ]
                ]
            },
            MasterUserPassword: {
                'Fn::Join': [
                    '',
                    [
                        '{{resolve:secretsmanager:',
                        {
                            Ref: 'DatabaseSecret3B817195'
                        },
                        ':SecretString:password::}}'
                    ]
                ]
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                ExcludeCharacters: '\"@/\\',
                GenerateStringKey: 'password',
                PasswordLength: 30,
                SecretStringTemplate: '{"username":"admin"}'
            }
        }));
        test.done();
    },
    'create an encrypted cluster with custom KMS key'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: {
                username: 'admin'
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc
            },
            kmsKey: new kms.Key(stack, 'Key')
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            KmsKeyId: {
                'Fn::GetAtt': [
                    'Key961B73FD',
                    'Arn'
                ]
            }
        }));
        test.done();
    },
    'cluster with instance parameter group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const parameterGroup = new lib_1.ParameterGroup(stack, 'ParameterGroup', {
            family: 'hello',
            parameters: {
                key: 'value'
            }
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                parameterGroup,
                vpc
            }
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DBParameterGroupName: {
                Ref: 'ParameterGroup5E32DECB'
            }
        }));
        test.done();
    },
    'create a cluster using a specific version of MySQL'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            engineVersion: "5.7.mysql_aurora.2.04.4",
            masterUser: {
                username: 'admin'
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: "aurora-mysql",
            EngineVersion: "5.7.mysql_aurora.2.04.4",
        }));
        test.done();
    },
    'create a cluster using a specific version of Postgresql'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_POSTGRESQL,
            engineVersion: "10.7",
            masterUser: {
                username: 'admin'
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: "aurora-postgresql",
            EngineVersion: "10.7",
        }));
        test.done();
    },
    'cluster exposes different read and write endpoints'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const cluster = new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc
            }
        });
        // THEN
        test.notDeepEqual(stack.resolve(cluster.clusterEndpoint), stack.resolve(cluster.clusterReadEndpoint));
        test.done();
    },
    'imported cluster with imported security group honors allowAllOutbound'(test) {
        // GIVEN
        const stack = testStack();
        const cluster = lib_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroup: ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false
            }),
        });
        // WHEN
        cluster.connections.allowToAnyIpv4(ec2.Port.tcp(443));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: 'sg-123456789',
        }));
        test.done();
    },
    "cluster with enabled monitoring"(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, "VPC");
        // WHEN
        new lib_1.DatabaseCluster(stack, "Database", {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: "admin"
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc
            },
            monitoringInterval: cdk.Duration.minutes(1),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource("AWS::RDS::DBInstance", {
            MonitoringInterval: 60,
            MonitoringRoleArn: {
                "Fn::GetAtt": ["DatabaseMonitoringRole576991DA", "Arn"]
            }
        }, assert_1.ResourcePart.Properties));
        assert_1.expect(stack).to(assert_1.haveResource("AWS::IAM::Role", {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "monitoring.rds.amazonaws.com"
                        }
                    }
                ],
                Version: "2012-10-17"
            },
            ManagedPolicyArns: [
                {
                    "Fn::Join": [
                        "",
                        [
                            "arn:",
                            {
                                Ref: "AWS::Partition"
                            },
                            ":iam::aws:policy/service-role/AmazonRDSEnhancedMonitoringRole"
                        ]
                    ]
                }
            ]
        }));
        test.done();
    },
    'create a cluster with imported monitoring role'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, "VPC");
        const monitoringRole = new aws_iam_1.Role(stack, "MonitoringRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("monitoring.rds.amazonaws.com"),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')
            ]
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, "Database", {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: "admin"
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc
            },
            monitoringInterval: cdk.Duration.minutes(1),
            monitoringRole
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource("AWS::RDS::DBInstance", {
            MonitoringInterval: 60,
            MonitoringRoleArn: {
                "Fn::GetAtt": ["MonitoringRole90457BF9", "Arn"]
            }
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'throws when trying to add rotation to a cluster without secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const cluster = new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort')
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc
            }
        });
        // THEN
        test.throws(() => cluster.addRotationSingleUser(), /without secret/);
        test.done();
    },
    'throws when trying to add single user rotation multiple timet'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: { username: 'admin' },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc
            }
        });
        // WHEN
        cluster.addRotationSingleUser();
        // THEN
        test.throws(() => cluster.addRotationSingleUser(), /A single user rotation was already added to this cluster/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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