"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeECRRepositoryAspect = exports.BatchJobDefinition = exports.StepFunctionsSageMakerTrainingJob = exports.ECSTaskDefinition = exports.ECRRepositoryAspect = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const batch = require("@aws-cdk/aws-batch-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const FN_SUB = 'Fn::Sub';
/**
 * Abtract aspect for ECR repository.
 *
 * You must provide the account id in props or set BSS_IMAGE_ASSET_ACCOUNT_ID in env
 *
 * @stability stable
 */
class ECRRepositoryAspect {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        var _f;
        this.account = (_f = props.imageAssetAccountId) !== null && _f !== void 0 ? _f : process.env.BSS_IMAGE_ASSET_ACCOUNT_ID;
    }
    /**
     * @stability stable
     */
    getRepoName(imageUri) {
        const matches = /\d{12}\.dkr\.ecr\..*\/(.*):.*/g.exec(imageUri);
        if (matches) {
            return matches[1];
        }
        return undefined;
    }
    /**
     * @stability stable
     */
    crossAccountECRPolicy(stack, repoName) {
        const policy = ECRRepositoryAspect._repoPolicies.get(repoName);
        if (policy) {
            return policy;
        }
        const newPolicy = new aws_iam_1.Policy(stack, `CrossAccountECR-${repoName}`, {
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'ecr:BatchCheckLayerAvailability',
                        'ecr:GetDownloadUrlForLayer',
                        'ecr:BatchGetImage',
                    ],
                    resources: [
                        aws_cdk_lib_1.Arn.format({
                            account: this.account,
                            service: 'ecr',
                            resource: 'repository',
                            resourceName: repoName,
                        }, stack),
                    ],
                }),
            ],
        });
        ECRRepositoryAspect._repoPolicies.set(repoName, newPolicy);
        return newPolicy;
    }
}
exports.ECRRepositoryAspect = ECRRepositoryAspect;
_a = JSII_RTTI_SYMBOL_1;
ECRRepositoryAspect[_a] = { fqn: "cdk-bootstrapless-synthesizer.ECRRepositoryAspect", version: "2.2.2" };
/**
 * @internal
 */
ECRRepositoryAspect._repoPolicies = new Map();
/**
 * Process the image assets in ECS task definition.
 *
 * @stability stable
 */
class ECSTaskDefinition extends ECRRepositoryAspect {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        super(props);
    }
    /**
     * @stability stable
     */
    hasBeReplaced(prop) {
        if (typeof prop.image === 'object' && FN_SUB in prop.image &&
            prop.image[FN_SUB].indexOf(this.account) > -1) {
            return prop.image[FN_SUB];
        }
        else if (prop.image && prop.image && prop.image.indexOf(this.account) > -1) {
            return prop.image;
        }
        return undefined;
    }
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(construct) {
        if (construct instanceof aws_ecs_1.TaskDefinition) {
            const containers = construct.stack.resolve(construct.node.defaultChild.containerDefinitions);
            let imageUri = undefined;
            if (containers instanceof Array) {
                for (const container of containers) {
                    if (container) {
                        imageUri = this.hasBeReplaced(container);
                        if (imageUri) {
                            break;
                        }
                    }
                }
            }
            else if (containers) {
                imageUri = this.hasBeReplaced(containers);
            }
            if (imageUri) {
                const repoName = this.getRepoName(imageUri);
                if (repoName) {
                    construct.executionRole.attachInlinePolicy(this.crossAccountECRPolicy(construct.stack, repoName));
                }
            }
        }
    }
}
exports.ECSTaskDefinition = ECSTaskDefinition;
_b = JSII_RTTI_SYMBOL_1;
ECSTaskDefinition[_b] = { fqn: "cdk-bootstrapless-synthesizer.ECSTaskDefinition", version: "2.2.2" };
/**
 * Process the image assets in SageMaker training job in Step Functions.
 *
 * @stability stable
 */
class StepFunctionsSageMakerTrainingJob extends ECRRepositoryAspect {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        super(props);
    }
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(construct) {
        if (construct instanceof aws_stepfunctions_tasks_1.SageMakerCreateTrainingJob) {
            const stack = aws_cdk_lib_1.Stack.of(construct);
            const state = construct.toStateJson();
            const image = stack.resolve(state.Parameters.AlgorithmSpecification.TrainingImage);
            if (FN_SUB in image) {
                const repoName = this.getRepoName(image[FN_SUB]);
                if (repoName) {
                    construct.role.attachInlinePolicy(this.crossAccountECRPolicy(stack, repoName));
                }
            }
        }
    }
}
exports.StepFunctionsSageMakerTrainingJob = StepFunctionsSageMakerTrainingJob;
_c = JSII_RTTI_SYMBOL_1;
StepFunctionsSageMakerTrainingJob[_c] = { fqn: "cdk-bootstrapless-synthesizer.StepFunctionsSageMakerTrainingJob", version: "2.2.2" };
/**
 * Process the image assets in AWS Batch job.
 *
 * @stability stable
 */
class BatchJobDefinition extends ECRRepositoryAspect {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        super(props);
        this._repoNames = [];
        this._allRolesMap = new Map();
    }
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(construct) {
        if (construct instanceof batch.JobDefinition) {
            const stack = construct.stack;
            this._executionRoleArn = construct.node.defaultChild
                .containerProperties.executionRoleArn;
            if (this._executionRoleArn && this._allRolesMap.get(this._executionRoleArn)) {
                this._executionRole = this._allRolesMap.get(this._executionRoleArn);
            }
            const image = construct.node.defaultChild
                .containerProperties.image;
            const image_resolved = stack.resolve(image);
            if (FN_SUB in image_resolved) {
                const repoName = this.getRepoName(image_resolved[FN_SUB]);
                if (repoName) {
                    if (this._executionRole) {
                        this._executionRole.attachInlinePolicy(this.crossAccountECRPolicy(stack, repoName));
                    }
                    else {
                        if (this._repoNames.indexOf(repoName) < 0) {
                            this._repoNames.push(repoName);
                        }
                    }
                }
            }
        }
        if (construct instanceof aws_iam_1.Role) {
            this._allRolesMap.set(construct.roleArn, construct);
            if (construct.roleArn == this._executionRoleArn) {
                const stack = construct.stack;
                this._executionRole = construct;
                while (this._repoNames.length > 0) {
                    const repoName = this._repoNames.pop();
                    if (repoName) {
                        this._executionRole.attachInlinePolicy(this.crossAccountECRPolicy(stack, repoName));
                    }
                }
            }
        }
    }
}
exports.BatchJobDefinition = BatchJobDefinition;
_d = JSII_RTTI_SYMBOL_1;
BatchJobDefinition[_d] = { fqn: "cdk-bootstrapless-synthesizer.BatchJobDefinition", version: "2.2.2" };
/**
 * Default ECR asset aspect, support using ECR assets in below services,.
 *
 * - ECS task definition
 * - SageMaker training job in Step Functions
 * - AWS Batch job
 * - AWS Lambda container image
 *
 * @stability stable
 */
class CompositeECRRepositoryAspect extends ECRRepositoryAspect {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        super(props);
        this._aspects = [
            new ECSTaskDefinition(props),
            new StepFunctionsSageMakerTrainingJob(props),
            new BatchJobDefinition(props),
        ];
    }
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(construct) {
        for (const _aspect of this._aspects) {
            _aspect.visit(construct);
        }
    }
}
exports.CompositeECRRepositoryAspect = CompositeECRRepositoryAspect;
_e = JSII_RTTI_SYMBOL_1;
CompositeECRRepositoryAspect[_e] = { fqn: "cdk-bootstrapless-synthesizer.CompositeECRRepositoryAspect", version: "2.2.2" };
//# sourceMappingURL=data:application/json;base64,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