"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NIST80053Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const ec2_1 = require("./rules/ec2");
const efs_1 = require("./rules/efs");
const iam_1 = require("./rules/iam");
/**
 * Check for NIST 800-53 compliance.
 *
 * Based on the NIST 800-53 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_4.html
 *
 * @stability stable
 */
class NIST80053Checks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkEC2(node, ignores);
            this.checkIAM(node, ignores);
            this.checkEFS(node, ignores);
        }
    }
    /**
     * Check EC2 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEC2(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckDetailedMonitoring') &&
            !ec2_1.nist80053EC2CheckDetailedMonitoring(node)) {
            const ruleId = 'NIST.800.53-EC2CheckDetailedMonitoring';
            const info = 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: CA-7(a)(b), SI-4(2), SI-4(a)(b)(c)).';
            const explanation = 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckInsideVPC') &&
            !ec2_1.nist80053EC2CheckInsideVPC(node)) {
            const ruleId = 'NIST.800.53-EC2CheckInsideVPC';
            const info = 'The EC2 instance is not within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckNoPublicIPs') &&
            !ec2_1.nist80053EC2CheckNoPublicIPs(node)) {
            const ruleId = 'NIST.800.53-EC2CheckNoPublicIPs';
            const info = 'The EC2 instance is associated with a public IP address - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)). ';
            const explanation = 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckSSHRestricted') &&
            !ec2_1.nist80053EC2CheckSSHRestricted(node)) {
            const ruleId = 'NIST.800.53-EC2CheckSSHRestricted';
            const info = 'The Security Group allows unrestricted SSH access - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EFS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEFS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EFSEncrypted') &&
            !efs_1.nist80053EFSEncrypted(node)) {
            const ruleId = 'NIST.800.53-EFSEncrypted';
            const info = 'The EFS does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check IAM Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkIAM(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMGroupMembershipCheck') &&
            !iam_1.nist80053IamGroupMembership(node)) {
            const ruleId = 'NIST.800.53-IAMGroupMembershipCheck';
            const info = 'The IAM user does not belong to any group(s) - (Control IDs: AC-2(1), AC-2(j), AC-3, and AC-6).';
            const explanation = 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations, by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMUserNoPoliciesCheck') &&
            !iam_1.nist80053IamUserNoPolicies(node)) {
            const ruleId = 'NIST.800.53-IAMUserNoPoliciesCheck';
            const info = 'The IAM policy is attached at the user level - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).';
            const explanation = 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMNoInlinePolicyCheck') &&
            !iam_1.nist80053IamNoInlinePolicy(node)) {
            const ruleId = 'NIST.800.53-IAMNoInlinePolicyCheck';
            const info = 'The IAM Group, User, or Role contains an inline policy - (Control ID: AC-6).';
            const explanation = 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMPolicyNoStatementsWithAdminAccess') &&
            !iam_1.nist80053IamPolicyNoStatementsWithAdminAccess(node)) {
            const ruleId = 'NIST.800.53-IAMPolicyNoStatementsWithAdminAccess';
            const info = 'The IAM policy grants admin access - (Control IDs AC-2(1), AC-2(j), AC-3, AC-6).';
            const explanation = 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, restricting policies from containing "Effect": "Allow" with "Action": "*" over "Resource": "*". Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
}
exports.NIST80053Checks = NIST80053Checks;
_a = JSII_RTTI_SYMBOL_1;
NIST80053Checks[_a] = { fqn: "monocdk-nag.NIST80053Checks", version: "0.0.7" };
//# sourceMappingURL=data:application/json;base64,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