# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['shinyutils', 'tests']

package_data = \
{'': ['*'], 'shinyutils': ['data/*']}

extras_require = \
{'color': ['crayons', 'rich>=6.0.0,<7.0.0'],
 'plotting': ['matplotlib>=3.3,<4.0', 'seaborn>=0.10,<0.11'],
 'pytorch': ['torch>=1.5.0,<2.0.0',
             'torchvision',
             'tensorboard[python]>=2.2.1,<3.0.0',
             'tqdm>=4.46.0,<5.0.0']}

setup_kwargs = {
    'name': 'shinyutils',
    'version': '5.0.1',
    'description': 'Personal collection of common utilities',
    'long_description': '# shinyutils\nVarious utilities for common tasks. :sparkles: :sparkles: :sparkles:\n\n## Setup\nInstall with `pip`. Additional features can be enabled with the `[<feature>]` syntax shown below. Available optional features are:\n* `color`: color support for logging and argument parsing\n* `plotting`: support for `matplotlib` and `seaborn`\n```bash\npip install shinyutils  # basic install\npip install "shinyutils[color]"  # install with color support\npip install "shinyutils[color,plotting]"  # install with color and plotting support\n```\n\n## Components\n\n### `subcls`\nUtility functions for dealing with subclasses.\n\n#### Functions\n* __`get_subclasses(cls)`__: returns a list of all the subclasses of `cls`.\n* __`get_subclass_names(cls)`__: returns a list of names of all subclasses of `cls`.\n* __`get_subclass_from_name(base_cls, cls_name)`__: return the subclass of `base_cls` named `cls_name`.\n\n### `argp`\nUtilities for argument parsing.\n\n#### `LazyHelpFormatter`\n`HelpFormatter` with sane defaults, and colors (courtesy of `crayons`)! To use, simply pass `formatter_class=LazyHelpFormatter` when creating `ArgumentParser` instances.\n```python\narg_parser = ArgumentParser(formatter_class=LazyHelpFormatter)\nsub_parsers = arg_parser.add_subparsers(dest="cmd")\nsub_parsers.required = True\n# `formatter_class` needs to be set for sub parsers as well.\ncmd1_parser = sub_parsers.add_parser("cmd1", formatter_class=LazyHelpFormatter)\n```\n\n#### `CommaSeparatedInts`\n`ArgumentParser` type representing a list of `int` values. Accepts a string of comma separated values, e.g., `\'1,2,3\'`.\n\n#### `InputFileType`\n`FileType` restricted to input files, (with `\'-\'` for `stdin`). Returns a `file` object.\n\n#### `OutputFileType`\n`FileType` restricted to output files (with `\'-\'` for `stdout`). The file\'s parent directories are created if needed. Returns a `file` object.\n\n#### `InputDirectoryType`\n`ArgumentParser` type representing a directory. Returns a `Path` object.\n\n#### `OutputDirectoryType`\n`ArgumentParser` type representing an output directory. The directory is created if it doesn\'t exist. Returns a `Path` object.\n\n#### `ClassType`\n`ArgumentParser` type representing sub-classes of a given base class. The returned value is a `class`.\n```python\nclass Base:\n    pass\n\nclass A(Base):\n    pass\n\nclass B(Base):\n    pass\n\narg_parser.add_argument("--cls", type=ClassType(Base), default=A)\n```\n\n#### `KeyValuePairsType`\n`ArgumentParser` type representing mappings. Accepts inputs of the form `str=val,[...]` where val is `int/float/str`. Returns a `dict`.\n\n#### `shiny_arg_parser`\n`ArgumentParser` object with `LazyHelpFormatter`, and arguments from sub-modules.\n\n### `logng`\nUtilities for logging.\n#### `build_log_argp`\nCreates an argument group with logging arguments.\n```python\n>>> arg_parser = ArgumentParser()\n>>> _ = build_log_argp(arg_parser)  # returns the parser\n>>> arg_parser.print_help()\nusage: -c [-h] [--log-level {DEBUG,INFO,WARNING,ERROR,CRITICAL}]\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --log-level {DEBUG,INFO,WARNING,ERROR,CRITICAL}\n```\nThis function is called on `shiny_arg_parser` when `shinyutils` is imported.\n\n#### `conf_logging`\nConfigures global logging using arguments returned by `ArgumentParser.parse_args`. `log_level` can be over-ridden with the keyword argument. Colors (enabled by default if `rich` is installed) can be toggled.\n```python\nargs = arg_parser.parse_args()\nconf_logging(args)\nconf_logging(args, log_level="INFO")  # override `log_level`\nconf_logging(use_colors=False)  # disable colors\n```\nWhen imported, `shinyutils` calls `conf_logging` without any arguments.\n\n### `matwrap`\nWrapper around `matplotlib` and `seaborn`.\n\n#### `MatWrap`\n```python\nfrom shinyutils.matwrap import MatWrap as mw  # do not import `matplotlib`, `seaborn`\n\nmw.configure()  # this should be called before importing any packages that import matplotlib\n\nfig = mw.plt().figure()\nax = fig.add_subplot(111)  # `ax` can be used normally now\n\n# Use class methods in `MatWrap` to access `matplotlib`/`seaborn` functions.\nmw.mpl()  # returns `matplotlib` module\nmw.plt()  # returns `matplotlib.pyplot` module\nmw.sns()  # returns `seaborn` module\n```\n\nUse `mw.configure` to configure plots. Arguments (defaults in bold) are:\n* `context`: seaborn context (__paper__/poster/talk/notebook)\n* `style`: seaborn style (white/whitegrid/dark/darkgrid/__ticks__)\n* `font`: any font available to fontspec (default __Latin Modern Roman__)\n* `latex_pkgs`: additional latex packages to be included before defaults\n* `**rc_extra`: matplotlib rc parameters to override defaults\n`mw.configure()` is called when `shinyutils.matwrap` is imported.\n\nUse `add_parser_config_args` to add matwrap config options to an argument parser.\n```python\n>>> arg_parser = ArgumentParser()\n>>> _ = mw.add_parser_config_args(arg_parser, group_title="plotting options")  # returns the parser group\n>>> arg_parser.print_help()\nusage: -c [-h] [--plotting-context {paper,notebook,talk,poster}]\n          [--plotting-style {white,dark,whitegrid,darkgrid,ticks}]\n          [--plotting-font PLOTTING_FONT]\n          [--plotting-latex-pkgs PLOTTING_LATEX_PKGS [PLOTTING_LATEX_PKGS ...]]\n          [--plotting-rc-extra PLOTTING_RC_EXTRA]\n\noptional arguments:\n  -h, --help            show this help message and exit\n\nplotting options:\n  --plotting-context {paper,notebook,talk,poster}\n  --plotting-style {white,dark,whitegrid,darkgrid,ticks}\n  --plotting-font PLOTTING_FONT\n  --plotting-latex-pkgs PLOTTING_LATEX_PKGS [PLOTTING_LATEX_PKGS ...]\n  --plotting-rc-extra PLOTTING_RC_EXTRA\n```\n`group_title` is optional, and if omitted, matwrap options will not be put in a separate group. When `shinyutils.matwrap` is imported, this function is called on `shiny_arg_parser`.\n\n#### Plot\n`Plot` is a wrapper around a single matplotlib plot, designed to be used as a context manager.\n```python\nfrom shinyutils.matwrap import Plot\n\nwith Plot(save_file, title, sizexy, labelxy, logxy) as ax:\n  ...\n```\nOnly the `save_file` argument is mandatory. When entering the context, `Plot` returns the plot axes, and when leaving, the plot is saved to the provided path.\n\n### `pt`\nUtilities for pytorch.\n\n#### `PTOpt`\nWrapper around pytorch optimizer and learning rate scheduler.\n```python\nfrom shinyutils.pt import PTOpt\nopt = PTOpt(\n        weights,  # iterable of parameters to update\n        optim_cls,  # subclass of torch.optim.Optimizer\n        optim_params,  # arguments to initialize optim_cls\n        lr_sched_cls,  # subclass of torch.optim.lr_scheduler._LRScheduler\n        lr_sched_params,\n)\n...\nopt.zero_grad()\nloss.backward()\nopt.step()\n```\n`lr_sched_` arguments are optional, and control the learning rate schedule. The class can also be used with argument parsers.\n```python\n>>> arg_parser = ArgumentParser(formatter_class=LazyHelpFormatter)\n>>> PTOpt.add_parser_args(\n        arg_parser,\n        arg_prefix="test",  # all options will be prefixed with "test-"\n        group_title="pt test",  # if None, separate group will not be created\n        default_optim_cls=Adam,\n        default_optim_params=None,  # if None, default is an empty dict\n        add_lr_decay=True,\n    )\n>>> arg_parser.print_help()\noptions:\n  -h, --help                            show this help message and exit (optional)\n\npt test:\n  --test-optim-cls cls                  ({Adadelta / Adagrad / Adam / AdamW / SparseAdam /\n                                          Adamax / ASGD / SGD / Rprop / RMSprop / LBFGS}\n                                          default: Adam)\n  --test-optim-params str=val[,...]     (default: {})\n  --test-lr-sched-cls cls               ({LambdaLR / MultiplicativeLR / StepLR / MultiStepLR /\n                                        ExponentialLR / CosineAnnealingLR / CyclicLR /\n                                        CosineAnnealingWarmRestarts / OneCycleLR} optional)\n  --test-lr-sched-params str=val[,...]  (default: {})\n>>> args = arg_parser.parse_args(...)\n>>> opt = PTOpt.from_args(weights, args, arg_prefix="test")\n```\n`PTOpt` can also add help options to argument parsers to display signatures for optimizer and learning rate schedule classes.\n```python\n>>> arg_parser = ArgumentParser()\n>>> PTOpt.add_help(arg_parser)\n>>> arg_parser.print_help()\nusage: -c [-h] [--explain-optimizer EXPLAIN_OPTIMIZER]\n          [--explain-lr-sched EXPLAIN_LR_SCHED]\n\noptional arguments:\n  -h, --help            show this help message and exit\n\npytorch help:\n  --explain-optimizer EXPLAIN_OPTIMIZER\n                        describe arguments of a torch optimizer\n  --explain-lr-sched EXPLAIN_LR_SCHED\n                        describe arguments of a torch lr scheduler\n\n>>> arg_parser.parse_args(["--explain-optimizer", "Adam"])\nAdam(params, lr=0.001, betas=(0.9, 0.999), eps=1e-08, weight_decay=0, amsgrad=False)\n...\n```\nThe help options are added to `shiny_arg_parser` when `shinyutils.pt` is imported.\n\n#### `FCNet`\n`FCNet` is a template class for fully connected networks.\n```python\nfrom shinyutils.pt import FCNet\nnet = FCNet(\n        in_dim,  # input dimension\n        out_dim,  # output dimension\n        hidden_sizes,  # list of hidden layer sizes\n        hidden_act,  # hidden layer activations (default relu)\n        out_act,  # output layer activation (default None)\n)\n```\nLike `PTOpt`, this class also supports construction through command line arguments.\n```python\n>>> arg_parser = ArgumentParser(formatter_class=LazyHelpFormatter)\n>>> FCNet.add_parser_args(\n        arg_parser,\n        arg_prefix="test",\n        group_title="fcnet",\n        default_indim=None,  # None means the option is mandatory\n        default_outdim=1,\n        default_hidden_sizes=None,\n        default_hidden_act=F.relu,\n        default_out_act=None,  # here, None means no output activation\n)\n>>> arg_parser.print_help()\noptions:\n  -h, --help                           show this help message and exit (optional)\n\nfcnet:\n  --test-fcnet-indim int               (required)\n  --test-fcnet-outdim int              (default: 1)\n  --test-fcnet-hidden-sizes int,[...]  (required)\n  --test-fcnet-hidden-act func         (default: relu)\n  --test-fcnet-out-act func            (optional)\n>>> args = arg_parser.parse_args(...)\n>>> net = FCNet.from_args(args, arg_prefix="test")\n```\n\n#### `NNTrainer`\nThis class trains a model on a dataset, and accepts multiple dataset "formats".\n```python\nfrom shinyutils.pt import *\nnn_trainer = NNTrainer(\n    batch_size,  # only mandatory argument\n    data_load_workers,  # default 0\n    shuffle,  # default True\n    pin_memory,  # default True\n    drop_last,  # default True\n    device,  # default cuda if available else cpu\n)\nnn_trainer.set_dataset(\n    dataset,  # can be a torch Dataset, a tuple of torch Tensors, or a tuple of numpy arrays\n)\nmodel = FCNet(...)\nopt = PTOpt(...)\nloss_fn = torch.nn.functional.mse_loss\nnn_trainer.train(model, opt, loss_fn, iters)\n```\n\n#### `SetTBWriterAction`\n`argparse` action to create a tensorboard summary writer. The writer is stored in the `tb_writer` attribute of the argument namespace; this can be overridden by setting `SetTBWriterAction.attr`. The usage is shown below with the tensorboard option that is added to `shiny_arg_parser` on importing the module.\n```python\nshiny_arg_parser.add_argument(\n    "--tb-dir",\n    type=OutputDirectoryType(),\n    help="tensorboard log directory",\n    default=None,\n    action=SetTBWriterAction,\n)\nshiny_arg_parser.set_defaults(**{SetTBWriterAction.attr: Mock(SummaryWriter)})\n```\n`shiny_arg_parser.tb_writer` will contain a `SummaryWriter` like object. If no log directory is provided through the command line, this object will be a dummy. So tensorboard functions can be called on the writer without extra checks.\n',
    'author': 'Jayanth Koushik',
    'author_email': 'jnkoushik@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/jayanthkoushik/shinyutils',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
