# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/rolling.ipynb (unless otherwise specified).

__all__ = ['rolling_mean', 'rolling_std', 'rolling_max', 'rolling_min']

# Cell
from math import sqrt
from typing import Callable, Optional

import numpy as np
import pandas as pd
from numba import njit

# Cell
@njit
def rolling_mean(x: np.ndarray,
                 window_size: int,
                 min_samples: Optional[int] = None) -> np.ndarray:
    if min_samples is None:
        min_samples = window_size
    n_samples = x.size
    out = np.full(n_samples, np.nan, dtype=np.float32)
    accum = 0.
    for i in range(min_samples - 1):
        accum += x[i]
    for i in range(min_samples - 1, window_size):
        accum += x[i]
        out[i] = accum / (i+1)
    for i in range(window_size, n_samples):
        accum += x[i] - x[i - window_size]
        out[i] = accum / window_size
    return out

# Cell
@njit
def rolling_std(x: np.ndarray,
                window_size: int,
                min_samples: Optional[int] = None) -> np.ndarray:
    if min_samples is None:
        min_samples = window_size
    if min_samples < 2:
        raise ValueError('min_samples must be greater than 1')
    n_samples = x.size
    out = np.full(n_samples, np.nan, dtype=np.float32)
    rolling_means = rolling_mean(x, window_size, min_samples)
    accum_xsq = 0.
    for i in range(min_samples - 1):
        accum_xsq += x[i]**2
    for i in range(min_samples - 1, window_size):
        accum_xsq += x[i]**2
        if i > 0:
            out[i] = sqrt((accum_xsq - (i+1) * rolling_means[i]**2) / i)
    out[0] = np.nan
    for i in range(window_size, n_samples):
        accum_xsq += x[i]**2 - x[i-window_size]**2
        out[i] = sqrt((accum_xsq - window_size * rolling_means[i]**2) / (window_size-1))
    return out

# Internal Cell
@njit
def _rolling_comp(comp: Callable,
                  x: np.ndarray,
                  window_size: int,
                  min_samples: Optional[int] = None):
    if min_samples is None:
        min_samples = window_size
    n_samples = x.size
    out = np.full(n_samples, np.nan, dtype=np.float32)
    for i in range(min_samples - 1, n_samples):
        pivot = x[i]
        ws = min(i+1, window_size)
        for j in range(1, ws):
            if comp(x[i - j], pivot):
                pivot = x[i - j]
        out[i] = pivot
    return out

@njit
def _gt(x: float, y: float) -> bool:
    return x > y

@njit
def _lt(x: float, y: float) -> bool:
    return ~_gt(x, y)

# Cell
def rolling_max(x: np.ndarray,
                window_size: int,
                min_samples: Optional[int] = None):
    return _rolling_comp(_gt, x, window_size, min_samples)

# Cell
def rolling_min(x: np.ndarray,
                window_size: int,
                min_samples: Optional[int] = None):
    return _rolling_comp(_lt, x, window_size, min_samples)