import pytest
import numpy as np
from dmtools.transform import rescale
from dmtools.colorspace import gray_to_RGB

# Checkerboard tests derived from https://legacy.imagemagick.org/Usage/filter/

# -----------
# TEST IMAGES
# -----------

# 5x5 checkerboard
CHECKS_5 = \
    np.array([[  0, 255,   0, 255,   0],
              [255,   0, 255,   0, 255],
              [  0, 255,   0, 255,   0],
              [255,   0, 255,   0, 255],
              [  0, 255,   0, 255,   0]])

# 10x10 checkerboard
CHECKS_10 = \
    np.array([[  0, 255,   0, 255,   0, 255,   0, 255,   0, 255],
              [255,   0, 255,   0, 255,   0, 255,   0, 255,   0],
              [  0, 255,   0, 255,   0, 255,   0, 255,   0, 255],
              [255,   0, 255,   0, 255,   0, 255,   0, 255,   0],
              [  0, 255,   0, 255,   0, 255,   0, 255,   0, 255],
              [255,   0, 255,   0, 255,   0, 255,   0, 255,   0],
              [  0, 255,   0, 255,   0, 255,   0, 255,   0, 255],
              [255,   0, 255,   0, 255,   0, 255,   0, 255,   0],
              [  0, 255,   0, 255,   0, 255,   0, 255,   0, 255],
              [255,   0, 255,   0, 255,   0, 255,   0, 255,   0]])

# --------------------------------------------------
# POINT FILTER (SCALE DOWN)
# https://legacy.imagemagick.org/Usage/filter/#point
# --------------------------------------------------

# 10x10 | point filter | 0.9
CHECKS_10_POINT_09 = \
    np.array([[  0, 255,   0, 255, 255,   0, 255,   0, 255],
              [255,   0, 255,   0,   0, 255,   0, 255,   0],
              [  0, 255,   0, 255, 255,   0, 255,   0, 255],
              [255,   0, 255,   0,   0, 255,   0, 255,   0],
              [255,   0, 255,   0,   0, 255,   0, 255,   0],
              [  0, 255,   0, 255, 255,   0, 255,   0, 255],
              [255,   0, 255,   0,   0, 255,   0, 255,   0],
              [  0, 255,   0, 255, 255,   0, 255,   0, 255],
              [255,   0, 255,   0,   0, 255,   0, 255,   0]])

# 10x10 | point filter | 0.8
CHECKS_10_POINT_08 = \
    np.array([[  0, 255, 255,   0, 255,   0,   0, 255],
              [255,   0,   0, 255,   0, 255, 255,   0],
              [255,   0,   0, 255,   0, 255, 255,   0],
              [  0, 255, 255,   0, 255,   0,   0, 255],
              [255,   0,   0, 255,   0, 255, 255,   0],
              [  0, 255, 255,   0, 255,   0,   0, 255],
              [  0, 255, 255,   0, 255,   0,   0, 255],
              [255,   0,   0, 255,   0, 255, 255,   0]])

# 10x10 | point filter | 0.7
CHECKS_10_POINT_07 = \
    np.array([[  0,   0, 255, 255,   0, 255, 255],
              [  0,   0, 255, 255,   0, 255, 255],
              [255, 255,   0,   0, 255,   0,   0],
              [255, 255,   0,   0, 255,   0,   0],
              [  0,   0, 255, 255,   0, 255, 255],
              [255, 255,   0,   0, 255,   0,   0],
              [255, 255,   0,   0, 255,   0,   0]])

# 10x10 | point filter | 0.6
CHECKS_10_POINT_06 = \
    np.array([[  0,   0,   0, 255, 255, 255],
              [  0,   0,   0, 255, 255, 255],
              [  0,   0,   0, 255, 255, 255],
              [255, 255, 255,   0,   0,   0],
              [255, 255, 255,   0,   0,   0],
              [255, 255, 255,   0,   0,   0]])

# 10x10 | point filter | 0.5
CHECKS_10_POINT_05 = \
    np.array([[0, 0, 0, 0, 0],
              [0, 0, 0, 0, 0],
              [0, 0, 0, 0, 0],
              [0, 0, 0, 0, 0],
              [0, 0, 0, 0, 0]])

# ------------------------------------------------
# BOX FILTER (SCALE DOWN)
# https://legacy.imagemagick.org/Usage/filter/#box
# ------------------------------------------------

# 10x10 | box filter | 0.9
CHECKS_10_BOX_09 = \
    np.array([[  0, 255,   0, 255, 127,   0, 255,   0, 255],
              [255,   0, 255,   0, 127, 255,   0, 255,   0],
              [  0, 255,   0, 255, 127,   0, 255,   0, 255],
              [255,   0, 255,   0, 127, 255,   0, 255,   0],
              [127, 127, 127, 127, 127, 127, 127, 127, 127],
              [  0, 255,   0, 255, 127,   0, 255,   0, 255],
              [255,   0, 255,   0, 127, 255,   0, 255,   0],
              [  0, 255,   0, 255, 127,   0, 255,   0, 255],
              [255,   0, 255,   0, 127, 255,   0, 255,   0]])

# 10x10 | box filter | 0.8
# NOTE: 0.8 example was omitted as it strangely differs from ImageMagick

# 10x10 | box filter | 0.7
CHECKS_10_BOX_07 = \
    np.array([[  0, 127, 255, 127,   0, 127, 255],
              [127, 127, 127, 127, 127, 127, 127],
              [255, 127,   0, 127, 255, 127,   0],
              [127, 127, 127, 127, 127, 127, 127],
              [  0, 127, 255, 127,   0, 127, 255],
              [127, 127, 127, 127, 127, 127, 127],
              [255, 127,   0, 127, 255, 127,   0]])

# 10x10 | box filter | 0.6
CHECKS_10_BOX_06 = \
    np.array([[127, 127, 127, 127, 127, 127],
              [127,   0, 127, 127, 255, 127],
              [127, 127, 127, 127, 127, 127],
              [127, 127, 127, 127, 127, 127],
              [127, 255, 127, 127,   0, 127],
              [127, 127, 127, 127, 127, 127]])

# 10x10 | box filter | 0.5
CHECKS_10_BOX_05 = \
    np.array([[127, 127, 127, 127, 127],
              [127, 127, 127, 127, 127],
              [127, 127, 127, 127, 127],
              [127, 127, 127, 127, 127],
              [127, 127, 127, 127, 127]])

# ------------------------------------------------
# POINT / BOX FILTER (SCALE UP)
# https://legacy.imagemagick.org/Usage/filter/#box
# ------------------------------------------------

# 5x5 | point / box filter | 1.2
CHECKS_5_BOX_POINT_12 = \
    np.array([[  0, 255,   0,   0, 255,   0],
              [255,   0, 255, 255,   0, 255],
              [  0, 255,   0,   0, 255,   0],
              [  0, 255,   0,   0, 255,   0],
              [255,   0, 255, 255,   0, 255],
              [  0, 255,   0,   0, 255,   0]])

# 5x5 | point / box filter | 1.4
CHECKS_5_BOX_POINT_14 = \
    np.array([[  0, 255, 255,   0, 255, 255,   0],
              [255,   0,   0, 255,   0,   0, 255],
              [255,   0,   0, 255,   0,   0, 255],
              [  0, 255, 255,   0, 255, 255,   0],
              [255,   0,   0, 255,   0,   0, 255],
              [255,   0,   0, 255,   0,   0, 255],
              [  0, 255, 255,   0, 255, 255,   0]])

# 5x5 | point / box filter | 1.6
CHECKS_5_BOX_POINT_16 = \
    np.array([[  0,   0, 255,   0,   0, 255,   0,   0],
              [  0,   0, 255,   0,   0, 255,   0,   0],
              [255, 255,   0, 255, 255,   0, 255, 255],
              [  0,   0, 255,   0,   0, 255,   0,   0],
              [  0,   0, 255,   0,   0, 255,   0,   0],
              [255, 255,   0, 255, 255,   0, 255, 255],
              [  0,   0, 255,   0,   0, 255,   0,   0],
              [  0,   0, 255,   0,   0, 255,   0,   0]])

# 5x5 | point / box filter | 1.8
CHECKS_5_BOX_POINT_18 = \
    np.array([[  0,   0, 255, 255,   0, 255, 255,   0,   0],
              [  0,   0, 255, 255,   0, 255, 255,   0,   0],
              [255, 255,   0,   0, 255,   0,   0, 255, 255],
              [255, 255,   0,   0, 255,   0,   0, 255, 255],
              [  0,   0, 255, 255,   0, 255, 255,   0,   0],
              [255, 255,   0,   0, 255,   0,   0, 255, 255],
              [255, 255,   0,   0, 255,   0,   0, 255, 255],
              [  0,   0, 255, 255,   0, 255, 255,   0,   0],
              [  0,   0, 255, 255,   0, 255, 255,   0,   0]])

# 5x5 | point / box filter | 2.0
CHECKS_5_BOX_POINT_20 = \
    np.array([[  0,   0, 255, 255,   0,   0, 255, 255,   0,   0],
              [  0,   0, 255, 255,   0,   0, 255, 255,   0,   0],
              [255, 255,   0,   0, 255, 255,   0,   0, 255, 255],
              [255, 255,   0,   0, 255, 255,   0,   0, 255, 255],
              [  0,   0, 255, 255,   0,   0, 255, 255,   0,   0],
              [  0,   0, 255, 255,   0,   0, 255, 255,   0,   0],
              [255, 255,   0,   0, 255, 255,   0,   0, 255, 255],
              [255, 255,   0,   0, 255, 255,   0,   0, 255, 255],
              [  0,   0, 255, 255,   0,   0, 255, 255,   0,   0],
              [  0,   0, 255, 255,   0,   0, 255, 255,   0,   0]])

# -----------------------------------------------------
# TRIANGLE FILTER (SCALE DOWN)
# https://legacy.imagemagick.org/Usage/filter/#triangle
# -----------------------------------------------------

# 10x10 | triangle filter | 0.9
CHECKS_10_TRIANGLE_09 = \
    np.array([[ 59, 177,  93, 143, 127, 110, 160,  76, 194],
              [177,  88, 152, 114, 127, 139, 101, 165,  76],
              [ 93, 152, 110, 135, 127, 118, 143, 101, 160],
              [143, 114, 135, 122, 126, 131, 118, 139, 110],
              [126, 126, 126, 126, 127, 126, 126, 126, 126],
              [110, 139, 118, 131, 126, 122, 135, 114, 143],
              [160, 101, 143, 118, 127, 135, 110, 152,  93],
              [ 76, 165, 101, 139, 127, 114, 152,  88, 177],
              [194,  76, 160, 110, 127, 143,  93, 177,  59]])

# 10x10 | triangle filter | 0.8
CHECKS_10_TRIANGLE_08 = \
    np.array([[ 95, 137, 137, 102, 151, 116, 116, 159],
              [137, 123, 123, 135, 118, 130, 130, 116],
              [137, 123, 123, 135, 118, 130, 130, 116],
              [102, 135, 135, 108, 145, 118, 118, 151],
              [151, 118, 118, 145, 108, 135, 135, 102],
              [116, 130, 130, 118, 135, 123, 123, 137],
              [116, 130, 130, 118, 135, 123, 123, 137],
              [159, 116, 116, 151, 102, 137, 137,  95]])

# 10x10 | triangle filter | 0.7
CHECKS_10_TRIANGLE_07 = \
    np.array([[115, 122, 135, 127, 118, 131, 139],
              [122, 125, 130, 127, 123, 128, 131],
              [135, 130, 120, 127, 133, 123, 118],
              [127, 127, 127, 126, 126, 127, 127],
              [118, 123, 133, 127, 120, 130, 135],
              [131, 128, 123, 127, 130, 125, 122],
              [139, 131, 118, 127, 135, 122, 114]])

# 10x10 | triangle filter | 0.6
CHECKS_10_TRIANGLE_06 = \
    np.array([[124, 125, 127, 126, 129, 129],
              [124, 125, 127, 126, 128, 129],
              [127, 127, 127, 126, 127, 127],
              [127, 127, 127, 126, 127, 127],
              [129, 128, 127, 126, 125, 125],
              [129, 129, 127, 126, 125, 124]])

# 10x10 | triangle filter | 0.5
CHECKS_10_TRIANGLE_05 = \
    np.array([[124, 127, 127, 127, 129],
              [127, 127, 127, 127, 127],
              [127, 127, 127, 127, 127],
              [127, 127, 127, 127, 127],
              [129, 127, 127, 127, 124]])

# -----------------------------------------------------
# TRIANGLE FILTER (SCALE UP)
# https://legacy.imagemagick.org/Usage/filter/#triangle
# -----------------------------------------------------

# 5x5 | triangle filter | 1.2
CHECKS_5_TRIANGLE_12 = \
    np.array([[  0, 191, 106, 106, 191,   0],
              [191,  95, 137, 137,  95, 191],
              [106, 137, 123, 123, 137, 106],
              [106, 137, 123, 123, 137, 106],
              [191,  95, 137, 137,  95, 191],
              [  0, 191, 106, 106, 191,   0]])

# 5x5 | triangle filter | 1.4
CHECKS_5_TRIANGLE_14 = \
    np.array([[  0, 145, 182,   0, 182, 145,   0],
              [145, 124, 119, 145, 119, 124, 145],
              [182, 119, 103, 182, 103, 119, 182],
              [  0, 145, 182,   0, 182, 145,   0],
              [182, 119, 103, 182, 103, 119, 182],
              [145, 124, 119, 145, 119, 124, 145],
              [  0, 145, 182,   0, 182, 145,   0]])

# 5x5 | triangle filter | 1.6
CHECKS_5_TRIANGLE_16 = \
    np.array([[  0, 111, 239,  79,  79, 239, 111,   0],
              [111, 125, 141, 121, 121, 141, 125, 111],
              [239, 141,  29, 169, 169,  29, 141, 239],
              [ 79, 121, 169, 109, 109, 169, 121,  79],
              [ 79, 121, 169, 109, 109, 169, 121,  79],
              [239, 141,  29, 169, 169,  29, 141, 239],
              [111, 125, 141, 121, 121, 141, 125, 111],
              [  0, 111, 239,  79,  79, 239, 111,   0]])

# 5x5 | triangle filter | 1.8
CHECKS_5_TRIANGLE_18 = \
    np.array([[  0,  84, 226, 141,   0, 141, 226,  84,   0],
              [ 83, 112, 160, 131,  84, 131, 160, 112,  84],
              [226, 160,  50, 116, 226, 116,  50, 160, 225],
              [141, 131, 116, 125, 141, 125, 116, 131, 141],
              [  0,  84, 226, 141,   0, 141, 226,  84,   0],
              [141, 131, 116, 125, 141, 125, 116, 131, 141],
              [226, 160,  50, 116, 226, 116,  50, 160, 225],
              [ 83, 112, 160, 131,  84, 131, 160, 112,  84],
              [  0,  84, 226, 141,   0, 141, 226,  84,   0]])

# 5x5 | triangle filter | 2.0
CHECKS_5_TRIANGLE_20 = \
    np.array([[  0,  63, 191, 191,  63,  63, 191, 191,  63,   0],
              [ 63,  95, 159, 159,  95,  95, 159, 159,  95,  63],
              [191, 159,  95,  95, 159, 159,  95,  95, 159, 191],
              [191, 159,  95,  95, 159, 159,  95,  95, 159, 191],
              [ 63,  95, 159, 159,  95,  95, 159, 159,  95,  63],
              [ 63,  95, 159, 159,  95,  95, 159, 159,  95,  63],
              [191, 159,  95,  95, 159, 159,  95,  95, 159, 191],
              [191, 159,  95,  95, 159, 159,  95,  95, 159, 191],
              [ 63,  95, 159, 159,  95,  95, 159, 159,  95,  63],
              [  0,  63, 191, 191,  63,  63, 191, 191,  63,   0]])


@pytest.mark.parametrize("source,filter,k,new",[
    (CHECKS_10, 'point', 0.9, CHECKS_10_POINT_09),
    (CHECKS_10, 'point', 0.8, CHECKS_10_POINT_08),
    (CHECKS_10, 'point', 0.7, CHECKS_10_POINT_07),
    (CHECKS_10, 'point', 0.6, CHECKS_10_POINT_06),
    (CHECKS_10, 'point', 0.5, CHECKS_10_POINT_05),
    (CHECKS_10, 'box', 0.9, CHECKS_10_BOX_09),
    (CHECKS_10, 'box', 0.7, CHECKS_10_BOX_07),
    (CHECKS_10, 'box', 0.6, CHECKS_10_BOX_06),
    (CHECKS_10, 'box', 0.5, CHECKS_10_BOX_05),
    (CHECKS_5, 'box', 1.2, CHECKS_5_BOX_POINT_12),
    (CHECKS_5, 'point', 1.2, CHECKS_5_BOX_POINT_12),
    (CHECKS_5, 'box', 1.4, CHECKS_5_BOX_POINT_14),
    (CHECKS_5, 'point', 1.4, CHECKS_5_BOX_POINT_14),
    (CHECKS_5, 'box', 1.6, CHECKS_5_BOX_POINT_16),
    (CHECKS_5, 'point', 1.6, CHECKS_5_BOX_POINT_16),
    (CHECKS_5, 'box', 1.8, CHECKS_5_BOX_POINT_18),
    (CHECKS_5, 'point', 1.8, CHECKS_5_BOX_POINT_18),
    (CHECKS_5, 'box', 2.0, CHECKS_5_BOX_POINT_20),
    (CHECKS_5, 'point', 2.0, CHECKS_5_BOX_POINT_20),
    (CHECKS_10, 'triangle', 0.9, CHECKS_10_TRIANGLE_09),
    (CHECKS_10, 'triangle', 0.8, CHECKS_10_TRIANGLE_08),
    (CHECKS_10, 'triangle', 0.7, CHECKS_10_TRIANGLE_07),
    (CHECKS_10, 'triangle', 0.6, CHECKS_10_TRIANGLE_06),
    (CHECKS_10, 'triangle', 0.5, CHECKS_10_TRIANGLE_05),
    (CHECKS_5, 'triangle', 1.2, CHECKS_5_TRIANGLE_12),
    (CHECKS_5, 'triangle', 1.4, CHECKS_5_TRIANGLE_14),
    (CHECKS_5, 'triangle', 1.6, CHECKS_5_TRIANGLE_16),
    (CHECKS_5, 'triangle', 1.8, CHECKS_5_TRIANGLE_18),
    (CHECKS_5, 'triangle', 2.0, CHECKS_5_TRIANGLE_20)])
def test_rescale(source, filter, k, new):
    # single channel
    assert np.array_equal(new, rescale(image=source, k=k, filter=filter))
    # three channel
    source = gray_to_RGB(source)
    new = gray_to_RGB(new)
    assert np.array_equal(new, rescale(image=source, k=k, filter=filter))
