# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mate3', 'mate3.examples', 'mate3.sunspec', 'mate3.sunspec.scripts']

package_data = \
{'': ['*'], 'mate3.sunspec': ['doc/*']}

install_requires = \
['loguru>=0.5.3,<0.6.0', 'pymodbus>=2.4,<3.0']

entry_points = \
{'console_scripts': ['mate3 = mate3.main:main']}

setup_kwargs = {
    'name': 'mate3',
    'version': '0.5.2',
    'description': 'Outback Mate3s Python Library',
    'long_description': '# Outback Mate 3 & 3s Python library & command line interface\n\n[![PyPI version](https://badge.fury.io/py/mate3.svg)](https://badge.fury.io/py/mate3)\n\nThis library provides complete support for all Outback devices (at least in theory, \nI don\'t own all the devices so cannot test it). Writing data is also supported.\n\nThis data is accessed though the Outback Mate3s\' Modbus interface. You must therefore \nhave a Mate3s which is connected to your local network using its ethernet port.\n\nTested on Python 3.7. May work on 3.6.\n\n## Warnings\n\nFirst, the big one:\n\n> **WARNING!** Please make sure you read [the license](https://github.com/adamcharnock/mate3/blob/master/LICENSE) before using any of the `write` functionality. You could easily damage your equipment by setting incorrect values (directly or indirectly).\n\nIn addition, there are other edges cases that may cause problems, mostly related to if a device is re-assigned a new port. For example, you have two inverters, read some values, then switch their ports over in the Hub before writing some values - which may now go to the \'wrong\' one. For now, it\'s safest not to do that, unless you restart the `Mate3Client` each time. On that note, the recommended approach if you need to poll over time is:\n\n```python\nwhile True:\n    with Mate3Client(...) as client:\n        client...\n    sleep(1)\n```\n\nAs opposed to\n\n```python\nwith Mate3Client(...) as client:\n    while True:\n        client...\n    sleep(1)\n```\n\nWhy? It means you\'re getting point-in-time values, and don\'t have to worry about changes (such as ports being switched). There are exceptions, but you should know why you\'re doing it.\n\n## Installation\n\nThe recommended installation is as follows:\n\n```sh\npip install mate3\n```\n\nAfter this you should be able to run the `mate3` command.\n\n## Using the library\n\nMore documentation is needed, but you can get a pretty code idea from [./examples/getting_started.py](./examples/getting_started.py), copied (somewhat) below. \n\n```python\nwith Mate3Client("192.168.1.12") as client:\n        # Read all devices:\n        client.read()\n        \n        # What\'s the system name?\n        client.devices.mate3.system_name\n        # >>> FieldValue[system_name] | Implemented | Read @ 2020-07-19 21:27:57.747231 | Value: --- | Clean\n        \n        # Get the battery voltage. Note that it\'s auto-scaled appropriately.\n        client.devices.fndc.battery_voltage\n        # >>> FieldValue[battery_voltage] | Implemented | Read @ 2020-07-19 21:27:57.795158 | Scale factor: -1 | Unscaled value: 544 | Value: 54.4 | Clean\n\n        # Get the (raw) values for the same device type on different ports\n        for port in client.devices.fx_inverters:\n            print(f"FET temp on port {port} = {client.devices.fx_inverters[port].fet_temperature.value}")\n        # >>> FET temp on port 1 = 36\n        # >>> FET temp on port 2 = 35\n\n        # Read only battery voltage again and check only it\'s read time was updated but not system name\n        time.sleep(1)\n        client.read(only=[client.devices.fndc.battery_voltage])\n        client.devices.mate3.system_name\n        client.devices.fndc.battery_voltage\n        # >>> FieldValue[system_name] ... 2020-07-19 21:27:57 ...\n        # >>> FieldValue[battery_voltage] ... 2020-07-19 21:27:58 ...\n        \n        # Nice. What about modbus fields that aren\'t implemented?\n        client.devices.mate3.sched_1_ac_mode.implemented\n        # >>> False\n\n        # Cool. Can we set a new value? Note that we don\'t need to worry about scaling etc.\n        volts = client.devices.charge_controller.config.absorb_volts\n        # >>> ... | Scale factor: -1 | Unscaled value: 535 | Value: 53.5 | Clean\n        client.devices.chjarge_controller.config.absorb_volts.value = volts.value + 0.1\n        # >>> ... | Scale factor: -1 | Unscaled value: 535 | Value: 53.5 | Dirty (value to write: 536)\n        \n        # OK, but what about fun fields like Enums? It\'s doable, though a bit gross ...\n        new_value = client.devices.charge_controller.config.grid_tie_mode.field.options["Grid Tie Mode disabled"]\n        client.devices.charge_controller.config.grid_tie_mode.value = new_value\n\n\n        # Finally, write any values that have changed to the device itself - BE CAREFUL!\n        client.write()\n```\n\n\n## Using the command line interface (CLI)\n\nA simple CLI is available, with four main sub-commands:\n\n- `read` - reads all of the values from the Mate3 and prints to stdout in a variety of formats.\n- `write` - writes values to the Mate3. (If you\'re doing anything serious you should use the python API.)\n- `devices` - shows the connected devices.\n- `dump` - dumps all of the raw modbus values to a (JSON) file in a format compatible with `CachingModbusClient` which you can then share with others to help in debugging any problems you may have. \n\nFor each you can access the help (i.e. `mate3 <cmd> -h`) for more information.\n\n## Troubleshooting\n\nSome ideas (which can be helpful for issues)\n\n### Set log-level to DEBUG\n\nSee `mate3 -h` for the CLI, otherwise the following (or similar) for python code:\n\n```python\nfrom loguru import logger\nlogger.remove()\nlogger.add(sys.stderr, level="DEBUG")\n```\n\n### List the devices\n\n```sh\n$ mate3 devices --host ...\nname                                               address    port\n----                                               -------    ----\nMate3                                              40069      None\nChargeController                                   40986      4\nChargeControllerConfiguration                      41014      4\n...\n```\nAre they all there?\n\n### Create a dump of the raw modbus values\n\nSee `mate3 dump -h`. You can send the resulting JSON file to someone to help debug. (Just note that it includes all the data about the Mate, e.g. any passwords etc.)\n\n## Writing data to Postgres\n\n> NB: this used to be in `mate3_pg` command, but has been moved to `./examples/postgres_monitor.py`.\n\nThe `postgress_monitor.py` command reads data from your Mate3 and writes it to a Postgres database.\n\nIn addition to a Mate3s connected to your network, you will need:\n\n* A running Postgres database\n* A definitions YAML file. ([example](https://github.com/adamcharnock/mate3/blob/master/pg_config.yaml))\n\nExample use:\n\n```\n$ python postgres_monitor.py \\\n    -H 192.168.0.123 \\ \n    --definitions /path/to/my/definitions.yaml \\\n    --database-url postgres://username:password@host:5432/database_name \\\n    --debug\n```\n\nYou will need to replace `192.168.0.123` with your Mate3s\' IP. Replace `/path/to/my/pg_config.yaml` with \na path to your definitions file (see [example](https://github.com/adamcharnock/mate3/blob/master/pg_config.yaml)).\nReplace the `username`, `password`, `host`, and `database_name` values with those for your Postgres database.\n\nFull details of the `postgres_monitor.py` command:\n\n```\n$ python postgres_monitor.py --help\nusage: mate3_pg [-h] --host HOST [--port PORT] [--interval INTERVAL] [--database-url DATABASE_URL] --definitions DEFINITIONS [--hypertables] [--quiet] [--debug]\n\nRead all available data from the Mate3 controller\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --host HOST, -H HOST  The host name or IP address of the Mate3\n  --port PORT, -p PORT  The port number address of the Mate3\n  --interval INTERVAL, -i INTERVAL\n                        Polling interval in seconds\n  --database-url DATABASE_URL\n                        Postgres database URL\n  --definitions DEFINITIONS\n                        YAML definition file\n  --hypertables         Should we create tables as hypertables? Use only if you are using TimescaleDB\n  --quiet, -q           Hide status output. Only errors will be shown\n  --debug               Show debug logging\n```  \n\n## Contributing\n\nIf you wish to edit the mate3 source (contributions are gladly received!), \nthen you can get the project directly from GitHub:\n\n```sh\n# Install poetry if you don\'t have it already (if you\'re unsure, you don\'t have it)\npip install poetry\n\n# Get the source\ngit clone https://github.com/adamcharnock/mate3.git\ncd mate3\n\n# Install mate3 and its dependencies. This also makes the mate3 command available.\npoetry install\n```\n\nAfter this you should be able to run the `mate3` command and edit the project\'s source code.\n\n## Release process\n\n```sh\n# Check everything has been comitted\ngit diff\n\n# Update setup.py et al\ndephell deps convert\n\n# Up the version\npoetry version {major|minor|bug}\n\n# Review the resulting changes\ngit diff\n\n# Build\npoetry publish --build\n\n# Docker: build & push\ndocker build -t adamcharnock/mate3:{VERSION_HERE} .\ndocker push adamcharnock/mate3:{VERSION_HERE}\n\n# Commit\ngit ci  -m "Version bump"\ngit push\ngit push --tags\n```\n\n## Credits\n\nThis was originally a heavily refactored version of\n[basrijn\'s Outback_Mate3 library](https://github.com/basrijn/Outback_Mate3), though has largely been completely rewritten since. Thanks anyway basrijn!\n',
    'author': 'Adam Charnock',
    'author_email': 'adam@adamcharnock.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/adamcharnock/mate3/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
