"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaSubscription = void 0;
const iam = require("@aws-cdk/aws-iam");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * Use a Lambda function as a subscription target.
 *
 * @stability stable
 */
class LambdaSubscription {
    /**
     * @stability stable
     */
    constructor(fn, props = {}) {
        this.fn = fn;
        this.props = props;
    }
    /**
     * Returns a configuration for a Lambda function to subscribe to an SNS topic.
     *
     * @stability stable
     */
    bind(topic) {
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!core_2.Construct.isConstruct(this.fn)) {
            throw new Error('The supplied lambda Function object must be an instance of Construct');
        }
        this.fn.addPermission(`AllowInvoke:${core_1.Names.nodeUniqueId(topic.node)}`, {
            sourceArn: topic.topicArn,
            principal: new iam.ServicePrincipal('sns.amazonaws.com'),
        });
        return {
            subscriberScope: this.fn,
            subscriberId: topic.node.id,
            endpoint: this.fn.functionArn,
            protocol: sns.SubscriptionProtocol.LAMBDA,
            filterPolicy: this.props.filterPolicy,
            region: this.regionFromArn(topic),
            deadLetterQueue: this.props.deadLetterQueue,
        };
    }
    regionFromArn(topic) {
        // no need to specify `region` for topics defined within the same stack.
        if (topic instanceof sns.Topic) {
            return undefined;
        }
        return core_1.Stack.of(topic).parseArn(topic.topicArn).region;
    }
}
exports.LambdaSubscription = LambdaSubscription;
//# sourceMappingURL=data:application/json;base64,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