# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['webcrypt']

package_data = \
{'': ['*']}

install_requires = \
['cryptography>=37.0,<38.0', 'pydantic>=1.9,<2.0']

setup_kwargs = {
    'name': 'webcrypt',
    'version': '0.8.0',
    'description': 'High level Constructs for Web App cryptography and JOSE spec implementation',
    'long_description': '# Webcrypt\n\nWebcrypt is a collection of Python3 tools and constructs that aim to simplify the implementation\nof all sorts of high-level cryptographic functionality commonly needed in the \ndevelopment of modern, distributed, and security-focused web applications.\n\nAt its core, Webcrypt relies entirely and **only** on the Python Library ``cryptography`` for all \ncryptographic operations, and acts as a thin wrapper around this library\'s primitives to provide \na high level API that is easier to use in the context of business software development.\n\nThis project borrows ideas from, and is inspired by other Python libraries including:\n\n* ``pycryptodome``\n* ``python-jose``\n\n``pip install "git+https://github.com/plataux/webcrypt@v0.6.1"``\n\n## Project Goals\n\n* Provide all the essential cryptographic tools:\n  * That can be used directly: encryption/decryption, signing/verification\n  * upon which higher level business protocols can be established (for example, nesting of JWTs)\n* Implement most of the essential aspects of JOSE spec to maximize interoperability with other frameworks\n* optimize for Production use:\n  * Focusing on Performance (by caching and reusing validated cryptographic constructs as much as possible)\n  * Focusing on Security (apply thorough validation steps when handling external signatures and other external entities)\n* Promote secure practices:\n  * wherever applicable, rejecting algorithms of insufficient strengths or known vulnerabilities\n  * Defaults to algorithms / key lengths that are reasonably secure and reasonably fast\n  * Make it easier to create new keys, and retire old ones\n\n## Current Features\n\n* Support for most of the JWS signature algorithms: https://datatracker.ietf.org/doc/html/rfc7518#section-3.1:\n  * All HMAC algorithms: ``HS256``, ``HS384``, ``HS512``\n  * All RSA algorithms: ``RS256``, ``RS384``, ``RS512`` and ``PS256``, ``PS384``, ``PS512``\n  * All Elliptic Curve Algorithms: ``ES256``, ``ES384``, ``ES512``\n  * Leaving out only ``none`` algorithm for JWT signatures\n\n* Support for all the JWE Encryption and Key Wrapping Algorithms https://datatracker.ietf.org/doc/html/rfc7518#section-4.1:\n  * All content encryption algorithms: ``A128GCM``, ``A192GCM``, ``A256GCM``, ``A128CBC-HS256``, ``A192CBC-HS384``, ``A256CBC-HS512``\n  * direct ``dir`` encryption using any of the Encryption Algorithms defined by the standard\n  * AES key wrapping of a newly, randomly Generated CEK: ``A128KW``, ``A192KW`` and ``A256KW``\n  * AES-GCM encryption of a newly, randomly generated CEK: ``A128GCMKW``, ``A192GCMKW`` and ``A256GCMKW``\n  * Password-Based Encryption algorithms: ``PBES2-HS256+A128KW``, ``PBES2-HS384+A192KW``, ``PBES2-HS512+A256KW``\n  * RSA key wrapping of CEKs: ``RSA1_5``, ``RSA-OAEP`` and ``RSA-OAEP-256``\n  * ECDH-ES key derivation for direct use, or wrapping of a CEK: ``ECDH-ES``, ``ECDH-ES+A128KW``, ``ECDH-ES+A192KW``, ``ECDH-ES+A256KW``\n\n* Simple API to export and import key sets (JWKS), and public JWKS from Private JWKS\n* Using ``Pydantic`` to validate, serialize and deserialize JWT Tokens \n\n\n## Usage\n\n### JWS Signing and Verification\n\n#### Default Creation and usage of JWS Signing Objects\n\nSign and retrieve byte payloads to and from unicode JWTs. The ``verify`` method will raise\nmany kinds of ``TokenException`` if the JWT is fabricated, corrupted or tampered with in any way\n```python\nfrom webcrypt.jws import JWS\n\n# Creates a new signing key with algorithm ES256 by default - it is fast, and can be verified by clients\nsigner = JWS()\n\npayload = b\'Byte Data to be signed and verified\'\n\ntoken: str = signer.sign(payload)\n\nprint(token)\n# will produce something like this\n# eyJhbGciOiJFUzI1NiIsImt0eSI6IkVDIiwia2lkIjoiZWFjNTgyMWMtZDQ3Yi00ZTA4LWEwMTMtOWQxOWUzNmNkNGRkIn0.\n# RGF0YSB0byBiZSBzaWduZWQgYW5kIHZlcmlmaWVk.tvQcT6S33H9auuGqNyYm_VHsA8I0Bw6NaLGi6plJCwmnr9oKXS78lZYI\n# 9ndlju6dnNXdP3nCAxZuyR9I0vxS-A\n\ndecoded_payload = signer.verify(token)\n\nassert payload == decoded_payload\n```\n\n#### Creation and usage of other JWS algorithms\nThe ``Algorithm`` Enum in the JWS class contains all Algorithms defined by the JOSE spec.\nThe following is an example of newly created signature keys:\n\n* All newly created RSA keys are 2048 bits (minimal recommended, and faster than 3072 and 4096 bit key sizes)\n* All newly created HMAC and EC keys match the length of the Hashing Algorithm used\n```python\nfrom webcrypt.jws import JWS\n\nk1 = JWS(JWS.Algorithm.RS512) # new RSA 2048 bit key with SHA512 hashing and PKCS1v15 Padding\nk2 = JWS(JWS.Algorithm.PS384) # new RSA 2048 bit with SHA384 hashing and PSS Padding\nk3 = JWS(JWS.Algorithm.HS256) # new HMAC signing key with SHA256 Hashing\nk4 = JWS(JWS.Algorithm.ES512) # new Elliptic Curve P-521 (SECP521R1) key, with SHA512 Hashing\n```\n\n#### Loading Existing Keys\n\nJWS Signing keys can be loaded from existing keys in various formats:\n\n* PEM formats\n* ``cryptography`` key objects\n* JWK (JSON Web Key) JSON Format\n\n##### From PEM\n```python\nfrom webcrypt.jws import JWS\n\n# This is a P-256 Curve EC key\nprivkey_pem = """-----BEGIN PRIVATE KEY-----\nMIGHAgEAMBMGByqGSM49AgEGCCqGSM49AwEHBG0wawIBAQQg2/Hi1u+D8HYixWoY\nCl0uQnq9KscIlSw5N2sGJJaWcv+hRANCAASmX7fu++yJAxOCUODmf9ZX14zU0IXb\ndXn5a9lL4Dswt/LLzVAo2DQQWe9nviYx0xb2txYXbtssaqEDUPeKAklF\n-----END PRIVATE KEY-----\n"""\n\n# Since this is a P-256 curve key, a The JWS Algorithm ES256 MUST be used, as per the JOSE spec\n# This key can sign, and verify\nec_jws = JWS.from_pem(privkey_pem,algorithm=JWS.Algorithm.ES256)\n\n\n# this is a 3072 RSA Public Key - can only verify JWTs, but cannot be used to sign\npubkey_pem="""-----BEGIN PUBLIC KEY-----\nMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAqBALJZGWr4zAvQsnj8e5\nxIX5KA74Nel7+Q8RR9HW80y+5t2pGpQuObX7WvuXhrkCFGgWxqvaIu7Z9XsxYAv/\nI4waVtoYOjUVH4w4aDI+SpOe8duiF5nAxpiaHp5h4ubDjQcipzsJyp3QQS7qqUAf\nwwNRwJQcYrdchJrFO9AQ/Wg7Actd7O8Ijhh0mSXID/hEVanBwrRuAq8GwQfDSv1Q\nMXNzC9k6ZlecKkUD2aRpTHPUbjbTZcGfPBnACdih2QMkQ+8XvOfIQqKuCM/MDVmD\nqbmQG6t/Bbsxa32cGBXVGmb+JW+e1TnQCF9i5xnOwKj6F08NVCzWaub8+heQeUJX\ngwpdVeejQS9w1q+eO5Ts6EGxOvnTbEs2WboRP7Hi8y1NE7PlKmOCHRQrdWdTaqTH\nj/BXR2V9LT0nNwC2/SnpDtqz9lbFJSvEKpLp4h7BDBuhnjfw6kywyRYaM5Q3HvHM\niiiRP4MeERT4cYKkGiV+GtpjzKqRqYpkDSzcueTNRUhlAgMBAAE=\n-----END PUBLIC KEY-----\n"""\n\n# all JWS RSA Algorithms are applicable with this (or any other) RSA keys (of size 2048 bits or larger)\nrsa_verifier = JWS.from_pem(pubkey_pem,JWS.Algorithm.PS512)\n\n# based on the above parameters, this key can verify signatures by the corresponding Private Key\n# With the specified PS512 JWS Algorithm\n```\n\n##### From cryptography objects\n```python\nfrom webcrypt.jws import JWS\nfrom cryptography.hazmat.primitives.asymmetric import rsa\n\n# assuming this is an existing 4096 bit RSA Private key\nprivkey = rsa.generate_private_key(public_exponent=65537,key_size=1024*4)\n\n# The corresponding Public key to be shared with partners, clients and other third parties\npubkey = privkey.public_key()\n\ndata = b\'Some data to be signed and verified\'\n\n# Uses SHA384 and PKCS1v15 Padding. Can be used to sign and verify\npriv_jwk = JWS(algorithm=JWS.Algorithm.RS384, key_obj=privkey)\n\ntoken = priv_jwk.sign(data)\n\n#######################\n# Somewhere else, construct the JWS object with the public key, and the agreed upon algorithm\n\n# cannot sign, but can verify signatures\npub_jwk = JWS(algorithm=JWS.Algorithm.RS384,key_obj=pubkey)\n\nsigned_data = pub_jwk.verify(token)\nassert signed_data == data\n```\n\n\n##### Exporting and Importing JWK JSON objects\n\nThis is the preferred and easiest method to store, and restore JWK objects, since it includes\nthe private and/or the public key components, as well as the algorithm and the intended usage of the key\n\n```python\nprivkey_jwk="""\n{\n  "use": "sig",\n  "kid": "23b5973e-7257-4fbc-944b-3f79e01da799",\n  "kty": "EC",\n  "alg": "ES384",\n  "key_ops": [\n    "sign",\n    "verify"\n  ],\n  "crv": "P-384",\n  "x": "xcICJQvPvomxkue8ZOE9AsKSSlGwYhEOBpscwdpiFK4jzkh2zGvaq1Ek5wY1BkxU",\n  "y": "Q6VVuYPTlVvZLZYTbtOoxfNUD3kqJs4ZEqQ6mt5cxfOHCc0mGqrGGcnhAZ95YKZ0",\n  "d": "mhKUB-5-leY-XBciNcSRFDEeUJuA4h6rzwaDoxyCeNkTLtauElWoWsRvN8Xu9rIh"\n}\n"""\n\nfrom webcrypt.jws import JWS\nimport json\n\n# can sign and verify\nsigner = JWS.from_jwk(json.loads(privkey_jwk))\n\n# export public components\npublic_jwk = signer.public_jwk()\n\n# which looks something like this:\npubkey_jwk = """{\n  "use": "sig",\n  "kid": "23b5973e-7257-4fbc-944b-3f79e01da799",\n  "kty": "EC",\n  "alg": "ES384",\n  "key_ops": [\n    "verify"\n  ],\n  "crv": "P-384",\n  "x": "xcICJQvPvomxkue8ZOE9AsKSSlGwYhEOBpscwdpiFK4jzkh2zGvaq1Ek5wY1BkxU",\n  "y": "Q6VVuYPTlVvZLZYTbtOoxfNUD3kqJs4ZEqQ6mt5cxfOHCc0mGqrGGcnhAZ95YKZ0"\n}"""\n\n# can verify, but cannot sign\nverifier = JWS.from_jwk(json.loads(pubkey_jwk))\n\n```\n\n### JWE Key Wrapping and Encryption\n\nMost JWE Algorithm involve using a private key to directly encrypt, or to wrap\na newly created CEK (Content Encryption Key)\n\n#### JWE private AES keys for direct ``dir`` encryption:\n\n```python\nfrom webcrypt.jwe import JWE\nimport json\n\n# generate a new 192-bit key used directly in content Encryption\njwk1 = JWE(algorithm=JWE.Algorithm.DIR, encryption=JWE.Encryption.A192GCM)\n\n# generate a new 256-bit key used directly in content Encryption + HMAC Authentication\njwk2 = JWE(algorithm=JWE.Algorithm.DIR, encryption=JWE.Encryption.A128CBC_HS256)\n\n# export jwk1:\nprint(json.dumps(jwk1.to_jwk(),indent=4))\n\n# will look like this:\nprivkey = """{\n    "use": "enc",\n    "kid": "be29da9a-3a89-4839-a664-68de669f145a",\n    "kty": "oct",\n    "alg": "dir",\n    "enc": "A128GCM",\n    "key_ops": [\n        "encrypt",\n        "decrypt"\n    ],\n    "k": "L35wm0tFTg12nKcZviyv1Q"\n}\n"""\n\njwk_reloaded = JWE.from_jwk(json.loads(privkey))\n\ndata = b\'Some byte data to be encrypted then decrypted\'\n\ntoken = jwk_reloaded.encrypt(data,compress=True) # option to compress the data\n\ndata_decrypted = jwk_reloaded.decrypt(token)\n\nassert data_decrypted == data\n\n```\n\n\n#### JWE Key-wrapping Algorithms:\n\n```python\nfrom webcrypt.jwe import JWE\nimport json\n\n# Generate a 192-bit private key to wrap a 256-bit CEK for encrypting and decrypting data\njwk1 = JWE(algorithm=JWE.Algorithm.A192KW, encryption=JWE.Encryption.A256GCM)\n\n# Generate a 256-bit private key to encrypt and wrap a 512-bit key for Encryption + Authentication\njwk2 = JWE(algorithm=JWE.Algorithm.A256GCMKW, encryption=JWE.Encryption.A256CBC_HS512)\n\n# Generate a 128-bit private key to wrap a 192-bit CEK for encrypting and decrypting data\njwk3 = JWE(algorithm=JWE.Algorithm.A128KW, encryption=JWE.Encryption.A192GCM)\n\nprint(json.dumps(jwk2.to_jwk(),indent=4))\n\n# will produce something like this:\n\njwk_json = """{\n    "use": "enc",\n    "kid": "f47a54c3-85d8-46b8-a9cb-8a1b5f47eddb",\n    "kty": "oct",\n    "alg": "A256GCMKW",\n    "enc": "A256CBC-HS512",\n    "key_ops": [\n        "wrapKey",\n        "unwrapKey"\n    ],\n    "k": "dkcM5Fnj7oYN4r4NGs7RMVxSX1jcT9gwvoRgxXJ4um8"\n}\n\n"""\n\n# which can later be reloaded for encryption / decryption operations\njwe_key = JWE.from_jwk(json.loads(jwk_json))\n```\n\n\n#### JWE RSA Key-wrapping Algorithms\n\n```python\nfrom webcrypt.jwe import JWE\n\n# Examples of all RSA Algorithms, with different CEK sizes\n\njwe1 = JWE(algorithm=JWE.Algorithm.RSA_OAEP_256, encryption=JWE.Encryption.A192GCM)\njwe2 = JWE(algorithm=JWE.Algorithm.RSA_OAEP, encryption=JWE.Encryption.A128CBC_HS256)\njwe3 = JWE(algorithm=JWE.Algorithm.RSA1_5, encryption=JWE.Encryption.A256GCM)\n\n\n# Load a Public JWE key from the JWK of a private one\npub_jwe = JWE.from_jwk(jwe1.public_jwk())\n\ndata = b\'Byte data to be encrypted and decrypted\'\n\n# encrypt data, and wrap the CEK\ntoken = pub_jwe.encrypt(data)\n\n# Raises an Error, a public key cannot decrypt the CEK!\npub_jwe.decrypt(token)\n\n# only the corresponding private key can unwrap the CEK and decrypt the data\ndata_decrypted = jwe1.decrypt(token)\n\nassert data_decrypted == data\n```\n\n\n#### JWE PBE (Passphrase based Encryption) Algorithms\n\n```python\nfrom webcrypt.jwe import JWE\nimport json\n\n# Generate a 192-bit private key to wrap a 384-bit key for Authentication + Encryption\njwk = JWE(algorithm=JWE.Algorithm.PBES2_HS384_A192KW,\n          encryption=JWE.Encryption.A256GCM,\n          key="I love python")\n\ndata = b\'Some secret data\'\n\ntoken = jwk.encrypt(data)\n\nprint(json.dumps(JWE.decode_header(token), indent=4))\n\n# the Token header will look something like this, including the alg, enc and\n# the PBE salt and iteration count (p2s and p2c)\nheader="""{\n    "alg": "PBES2-HS384+A192KW",\n    "enc": "A256GCM",\n    "kid": "08842033-5f83-477b-9be3-c91ab6e7635c",\n    "p2s": "7jByuyCgOWc4aEfkoAJ0VQ",\n    "p2c": 1644\n}"""\n```\n',
    'author': 'mk',
    'author_email': 'mk@plataux.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://plataux.com',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
