"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NIST80053Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const apigw_1 = require("./rules/apigw");
const autoscaling_1 = require("./rules/autoscaling");
const cloudtrail_1 = require("./rules/cloudtrail");
const cloudwatch_1 = require("./rules/cloudwatch");
const codebuild_1 = require("./rules/codebuild");
const dms_1 = require("./rules/dms");
const dynamodb_1 = require("./rules/dynamodb");
const ec2_1 = require("./rules/ec2");
const efs_1 = require("./rules/efs");
const elasticache_1 = require("./rules/elasticache");
const elasticsearch_1 = require("./rules/elasticsearch");
const elb_1 = require("./rules/elb");
const emr_1 = require("./rules/emr");
const iam_1 = require("./rules/iam");
const lambda_1 = require("./rules/lambda");
const rds_1 = require("./rules/rds");
const redshift_1 = require("./rules/redshift");
const s3_1 = require("./rules/s3");
const sagemaker_1 = require("./rules/sagemaker");
const sns_1 = require("./rules/sns");
/**
 * Check for NIST 800-53 compliance.
 *
 * Based on the NIST 800-53 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_4.html
 *
 * @stability stable
 */
class NIST80053Checks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkAPIGW(node, ignores);
            this.checkAutoScaling(node, ignores);
            this.checkCloudTrail(node, ignores);
            this.checkCloudWatch(node, ignores);
            this.checkCodeBuild(node, ignores);
            this.checkDMS(node, ignores);
            this.checkDynamoDB(node, ignores);
            this.checkEC2(node, ignores);
            this.checkEFS(node, ignores);
            this.checkElastiCache(node, ignores);
            this.checkElasticsearch(node, ignores);
            this.checkELB(node, ignores);
            this.checkEMR(node, ignores);
            this.checkIAM(node, ignores);
            this.checkLambda(node, ignores);
            this.checkRDS(node, ignores);
            this.checkRedshift(node, ignores);
            this.checkS3(node, ignores);
            this.checkSageMaker(node, ignores);
            this.checkSNS(node, ignores);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-APIGWCacheEnabledAndEncrypted') &&
            !apigw_1.nist80053APIGWCacheEnabledAndEncrypted(node)) {
            const ruleId = 'NIST.800.53-APIGWCacheEnabledAndEncrypted';
            const info = 'The  API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: SC-13, SC-28).';
            const explanation = "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-APIGWExecutionLoggingEnabled') &&
            !apigw_1.nist80053APIGWExecutionLoggingEnabled(node)) {
            const ruleId = 'NIST.800.53-APIGWExecutionLoggingEnabled';
            const info = 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: AU-2(a)(d), AU-3, AU-12(a)(c)).';
            const explanation = 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Auto Scaling Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-AutoScalingHealthChecks') &&
            !autoscaling_1.nist80053AutoScalingHealthChecks(node)) {
            const ruleId = 'NIST.800.53-AutoScalingHealthChecks';
            const info = 'The Auto Scaling group utilizes a load balancer and does not have an ELB health check configured - (Control IDs: SC-5).';
            const explanation = 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CloudTrail Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudTrailCloudWatchLogsEnabled') &&
            !cloudtrail_1.nist80053CloudTrailCloudWatchLogsEnabled(node)) {
            const ruleId = 'NIST.800.53-CloudTrailCloudWatchLogsEnabled';
            const info = 'The trail does not have CloudWatch logs enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-6(1)(3), AU-7(1), AU-12(a)(c), CA-7(a)(b), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).';
            const explanation = 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudTrailEncryptionEnabled') &&
            !cloudtrail_1.nist80053CloudTrailEncryptionEnabled(node)) {
            const ruleId = 'NIST.800.53-CloudTrailEncryptionEnabled';
            const info = 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: AU-9).';
            const explanation = 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudTrailLogFileValidationEnabled') &&
            !cloudtrail_1.nist80053CloudTrailLogFileValidationEnabled(node)) {
            const ruleId = 'NIST.800.53-CloudTrailLogFileValidationEnabled';
            const info = 'The trail does not have log file validation enabled - (Control ID: AC-6).';
            const explanation = 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CloudWatch Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudWatchAlarmAction') &&
            !cloudwatch_1.nist80053CloudWatchAlarmAction(node)) {
            const ruleId = 'NIST.800.53-CloudWatchAlarmAction';
            const info = 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control IDs: AC-2(4), AU-6(1)(3), AU-7(1), CA-7(a)(b), IR-4(1), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).';
            const explanation = 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudWatchLogGroupEncrypted') &&
            !cloudwatch_1.nist80053CloudWatchLogGroupEncrypted(node)) {
            const ruleId = 'NIST.800.53-CloudWatchLogGroupEncrypted';
            const info = 'The CloudWatch Log Group is not encrypted with a KMS Customer Master Key (CMK) - (Control IDs: AU-9, SC-13, SC-28).';
            const explanation = 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CodeBuild Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-CodeBuildCheckEnvVars') &&
            !codebuild_1.nist80053CodeBuildCheckEnvVars(node)) {
            const ruleId = 'NIST.800.53-CodeBuildCheckEnvVars';
            const info = 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control IDs: AC-6, IA-5(7), SA-3(a)).';
            const explanation = 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-CodeBuildURLCheck') &&
            !codebuild_1.nist80053CodeBuildURLCheck(node)) {
            const ruleId = 'NIST.800.53-CodeBuildURLCheck';
            const info = 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAUTH - (Control IDs: SA-3(a).';
            const explanation = 'OAUTH is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check DMS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDMS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-DMSReplicationNotPublic') &&
            !dms_1.nist80053DMSReplicationNotPublic(node)) {
            const ruleId = 'NIST.800.53-DMSReplicationNotPublic';
            const info = 'The DMS replication instance is public - (Control IDs: AC-3).';
            const explanation = 'DMS replication instances can contain sensitive information and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check DynamoDB Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-DynamoDBPITREnabled') &&
            !dynamodb_1.nist80053DynamoDBPITREnabled(node)) {
            const ruleId = 'NIST.800.53-DynamoDBPITREnabled';
            const info = 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: CP-9(b), CP-10, SI-12).';
            const explanation = 'The recovery maintains continuous backups of your table for the last 35 days.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EC2 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEC2(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckDefaultSecurityGroupClosed') &&
            !ec2_1.nist80053EC2CheckDefaultSecurityGroupClosed(node)) {
            const ruleId = 'NIST.800.53-EC2CheckDefaultSecurityGroupClosed';
            const info = "The VPC's default security group allows inbound or outbound traffic - (Control IDs: AC-4, SC-7, SC-7(3)).";
            const explanation = 'When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckDetailedMonitoring') &&
            !ec2_1.nist80053EC2CheckDetailedMonitoring(node)) {
            const ruleId = 'NIST.800.53-EC2CheckDetailedMonitoring';
            const info = 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: CA-7(a)(b), SI-4(2), SI-4(a)(b)(c)).';
            const explanation = 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckInsideVPC') &&
            !ec2_1.nist80053EC2CheckInsideVPC(node)) {
            const ruleId = 'NIST.800.53-EC2CheckInsideVPC';
            const info = 'The EC2 instance is not within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckNoPublicIPs') &&
            !ec2_1.nist80053EC2CheckNoPublicIPs(node)) {
            const ruleId = 'NIST.800.53-EC2CheckNoPublicIPs';
            const info = 'The EC2 instance is associated with a public IP address - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)). ';
            const explanation = 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckSSHRestricted') &&
            !ec2_1.nist80053EC2CheckSSHRestricted(node)) {
            const ruleId = 'NIST.800.53-EC2CheckSSHRestricted';
            const info = 'The Security Group allows unrestricted SSH access - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckCommonPortsRestricted') &&
            !ec2_1.nist80053EC2CheckCommonPortsRestricted(node)) {
            const ruleId = 'NIST.800.53-EC2CheckCommonPortsRestricted';
            const info = 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on common ports (20, 21, 3389, 3306, 4333) - (Control IDs: AC-4, CM-2, SC-7, SC-7(3)).';
            const explanation = 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EFS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEFS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EFSEncrypted') &&
            !efs_1.nist80053EFSEncrypted(node)) {
            const ruleId = 'NIST.800.53-EFSEncrypted';
            const info = 'The EFS does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check ElastiCache Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-ElastiCacheRedisClusterAutomaticBackup') &&
            !elasticache_1.nist80053ElastiCacheRedisClusterAutomaticBackup(node)) {
            const ruleId = 'NIST.800.53-ElastiCacheRedisClusterAutomaticBackup';
            const info = 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: CP-9(b), CP-10, SI-12).';
            const explanation = 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Elasticsearch Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkElasticsearch(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-ElasticsearchEncryptedAtRest') &&
            !elasticsearch_1.nist80053ElasticsearchEncryptedAtRest(node)) {
            const ruleId = 'NIST.800.53-ElasticsearchEncryptedAtRest';
            const info = 'The Elasticsearch domain does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elasticsearch Service (Amazon ES) domains.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ElasticsearchNodeToNodeEncrypted') &&
            !elasticsearch_1.nist80053ElasticsearchNodeToNodeEncrypted(node)) {
            const ruleId = 'NIST.800.53-ElasticsearchNodeToNodeEncrypted';
            const info = 'The Elasticsearch domain does not have node-to-node encryption enabled - (Control IDs: SC-7, SC-8, SC-8(1)).';
            const explanation = 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon Elasticsearch Service (Amazon ES) domains.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ElasticsearchRunningWithinVPC') &&
            !elasticsearch_1.nist80053ElasticsearchRunningWithinVPC(node)) {
            const ruleId = 'NIST.800.53-ElasticsearchRunningWithinVPC';
            const info = 'The Elasticsearch domain is not running within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'VPCs help secure your AWS resources and provide an extra layer of protection.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkELB(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-ALBHttpDropInvalidHeaderEnabled') &&
            !elb_1.nist80053ALBHttpDropInvalidHeaderEnabled(node)) {
            const ruleId = 'NIST.800.53-ALBHttpDropInvalidHeaderEnabled';
            const info = 'The ALB does not have invalid http header dropping enabled - (Control ID: AC-17(2)).';
            const explanation = 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ALBHttpToHttpsRedirection') &&
            !elb_1.nist80053ALBHttpToHttpsRedirection(node)) {
            const ruleId = 'NIST.800.53-ALBHttpToHttpsRedirection';
            const info = 'The ALB\'s Http listeners are not configured to redirect to https - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13, SC-23).';
            const explanation = 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBCrossZoneBalancing') &&
            !elb_1.nist80053ELBCrossZoneBalancing(node)) {
            const ruleId = 'NIST.800.53-ELBCrossZoneBalancing';
            const info = 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: SC-5, CP-10).';
            const explanation = 'The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBDeletionProtectionEnabled') &&
            !elb_1.nist80053ELBDeletionProtectionEnabled(node)) {
            const ruleId = 'NIST.800.53-ELBDeletionProtectionEnabled';
            const info = 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: CM-2, CP-10).';
            const explanation = 'Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBListenersUseSSLOrHTTPS') &&
            !elb_1.nist80053ELBListenersUseSSLOrHTTPS(node)) {
            const ruleId = 'NIST.800.53-ELBListenersUseSSLOrHTTPS';
            const info = 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-23).';
            const explanation = 'Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBLoggingEnabled') &&
            !elb_1.nist80053ELBLoggingEnabled(node)) {
            const ruleId = 'NIST.800.53-ELBLoggingEnabled';
            const info = 'The ELB does not have logging enabled - (Control ID: AU-2(a)(d)).';
            const explanation = "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBUseACMCerts') &&
            !elb_1.nist80053ELBUseACMCerts(node)) {
            const ruleId = 'NIST.800.53-ELBUseACMCerts';
            const info = 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13).';
            const explanation = 'Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EMR Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEMR(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EMRKerberosEnabled') &&
            !emr_1.nist80053EMRKerberosEnabled(node)) {
            const ruleId = 'NIST.800.53-EMRKerberosEnabled';
            const info = 'The EMR cluster does not have Kerberos enabled - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).';
            const explanation = 'The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check IAM Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkIAM(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMGroupMembershipCheck') &&
            !iam_1.nist80053IamGroupMembership(node)) {
            const ruleId = 'NIST.800.53-IAMGroupMembershipCheck';
            const info = 'The IAM user does not belong to any group(s) - (Control IDs: AC-2(1), AC-2(j), AC-3, and AC-6).';
            const explanation = 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations, by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMNoInlinePolicyCheck') &&
            !iam_1.nist80053IamNoInlinePolicy(node)) {
            const ruleId = 'NIST.800.53-IAMNoInlinePolicyCheck';
            const info = 'The IAM Group, User, or Role contains an inline policy - (Control ID: AC-6).';
            const explanation = 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMPolicyNoStatementsWithAdminAccess') &&
            !iam_1.nist80053IamPolicyNoStatementsWithAdminAccess(node)) {
            const ruleId = 'NIST.800.53-IAMPolicyNoStatementsWithAdminAccess';
            const info = 'The IAM policy grants admin access - (Control IDs AC-2(1), AC-2(j), AC-3, AC-6).';
            const explanation = 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, restricting policies from containing "Effect": "Allow" with "Action": "*" over "Resource": "*". Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMUserNoPoliciesCheck') &&
            !iam_1.nist80053IamUserNoPolicies(node)) {
            const ruleId = 'NIST.800.53-IAMUserNoPoliciesCheck';
            const info = 'The IAM policy is attached at the user level - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).';
            const explanation = 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Lambda Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkLambda(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-LambdaFunctionsInsideVPC') &&
            !lambda_1.nist80053LambdaFunctionsInsideVPC(node)) {
            const ruleId = 'NIST.800.53-LambdaFunctionsInsideVPC';
            const info = 'The Lambda function is not defined within within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Amazon RDS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkRDS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSEnhancedMonitoringEnabled') &&
            !rds_1.nist80053RDSEnhancedMonitoringEnabled(node)) {
            const ruleId = 'NIST.800.53-RDSEnhancedMonitoringEnabled';
            const info = 'The RDS DB Instance does not enhanced monitoring enabled - (Control ID: CA-7(a)(b)).';
            const explanation = 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSInstanceBackupEnabled') &&
            !rds_1.nist80053RDSInstanceBackupEnabled(node)) {
            const ruleId = 'NIST.800.53-RDSInstanceBackupEnabled';
            const info = 'The RDS DB Instance does not have backups enabled - (Control IDs: CP-9(b), CP-10, SI-12).';
            const explanation = 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSInstanceDeletionProtectionEnabled') &&
            !rds_1.nist80053RDSInstanceDeletionProtectionEnabled(node)) {
            const ruleId = 'NIST.800.53-RDSInstanceDeletionProtectionEnabled';
            const info = 'The RDS DB Instance or Aurora Cluster does not have deletion protection enabled - (Control ID: SC-5).';
            const explanation = 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSInstanceMultiAzSupport') &&
            !rds_1.nist80053RDSInstanceMultiAZSupport(node)) {
            const ruleId = 'NIST.800.53-RDSInstanceMultiAzSupport';
            const info = 'The RDS DB Instance does not have multi-AZ support - (Control IDs: CP-10, SC-5, SC-36).';
            const explanation = 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSInstancePublicAccess') &&
            !rds_1.nist80053RDSInstancePublicAccess(node)) {
            const ruleId = 'NIST.800.53-RDSInstancePublicAccess';
            const info = 'The RDS DB Instance allows public access - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).';
            const explanation = 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSLoggingEnabled') &&
            !rds_1.nist80053RDSLoggingEnabled(node)) {
            const ruleId = 'NIST.800.53-RDSLoggingEnabled';
            const info = 'The RDS DB Instance does not have all CloudWatch log types exported - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).';
            const explanation = 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSStorageEncrypted') &&
            !rds_1.nist80053RDSStorageEncrypted(node)) {
            const ruleId = 'NIST.800.53-RDSStorageEncrypted';
            const info = 'The RDS DB Instance or Aurora Cluster does not have storage encrypted - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist at rest in Amazon RDS instances and clusters, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Redshift Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-RedshiftClusterConfiguration') &&
            !redshift_1.nist80053RedshiftClusterConfiguration(node)) {
            const ruleId = 'NIST.800.53-RedshiftClusterConfiguration';
            const info = 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c), SC-13).';
            const explanation = 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RedshiftClusterPublicAccess') &&
            !redshift_1.nist80053RedshiftClusterPublicAccess(node)) {
            const ruleId = 'NIST.800.53-RedshiftClusterPublicAccess';
            const info = 'The Redshift cluster allows public access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).';
            const explanation = 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Amazon S3 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkS3(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-S3BucketLoggingEnabled') &&
            !s3_1.nist80053S3BucketLoggingEnabled(node)) {
            const ruleId = 'NIST.800.53-S3BucketLoggingEnabled';
            const info = 'The S3 Bucket does not have server access logs enabled - (Control IDs: AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).';
            const explanation = 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check SageMaker Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-SageMakerEndpointKMS') &&
            !sagemaker_1.nist80053SageMakerEndpointKMS(node)) {
            const ruleId = 'NIST.800.53-SageMakerEndpointKMS';
            const info = 'The SageMaker endpoint is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-SageMakerNotebookDirectInternetAccessDisabled') &&
            !sagemaker_1.nist80053SageMakerNotebookDirectInternetAccessDisabled(node)) {
            const ruleId = 'NIST.800.53-SageMakerNotebookDirectInternetAccessDisbabled';
            const info = 'The SageMaker notebook does not disable direct internet access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).';
            const explanation = 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-SageMakerNotebookKMS') &&
            !sagemaker_1.nist80053SageMakerNotebookKMS(node)) {
            const ruleId = 'NIST.800.53-SageMakerNotebookKMS';
            const info = 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Amazon SNS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkSNS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-SNSEncryptedKMS') &&
            !sns_1.nist80053SNSEncryptedKMS(node)) {
            const ruleId = 'NIST.800.53-SNSEncryptedKMS';
            const info = 'The SNS topic does not have KMS encryption enabled - (Control ID: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
}
exports.NIST80053Checks = NIST80053Checks;
_a = JSII_RTTI_SYMBOL_1;
NIST80053Checks[_a] = { fqn: "monocdk-nag.NIST80053Checks", version: "0.0.54" };
//# sourceMappingURL=data:application/json;base64,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