"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HIPAASecurityChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const cloudtrail_1 = require("./rules/cloudtrail");
const ec2_1 = require("./rules/ec2");
/**
 * Check for HIPAA Security compliance.
 *
 * Based on the HIPAA Security AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-hipaa_security.html
 *
 * @stability stable
 */
class HIPAASecurityChecks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            // this.checkAPIGW(node, ignores);
            // this.checkAutoScaling(node, ignores);
            this.checkCloudTrail(node, ignores);
            // this.checkCloudWatch(node, ignores);
            // this.checkCodeBuild(node, ignores);
            // this.checkDMS(node, ignores);
            // this.checkDynamoDB(node, ignores);
            this.checkEC2(node, ignores);
            // this.checkECS(node, ignores);
            // this.checkEFS(node, ignores);
            // this.checkElastiCache(node, ignores);
            // this.checkElasticBeanstalk(node, ignores);
            // this.checkElasticsearch(node, ignores);
            // this.checkELB(node, ignores);
            // this.checkEMR(node, ignores);
            // this.checkIAM(node, ignores);
            // this.checkLambda(node, ignores);
            // this.checkRDS(node, ignores);
            // this.checkRedshift(node, ignores);
            // this.checkS3(node, ignores);
            // this.checkSageMaker(node, ignores);
            // this.checkSecretsManager(node, ignores);
            // this.checkSNS(node, ignores);
            // this.checkVPC(node, ignores);
        }
    }
    //   /**
    //    * Check API Gateway Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkAPIGW(node: CfnResource, ignores: any): void {}
    //   /**
    //    * Check Auto Scaling Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkAutoScaling(node: CfnResource, ignores: any): void {}
    /**
     * Check CloudTrail Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudTrailCloudWatchLogsEnabled') &&
            !cloudtrail_1.hipaaSecurityCloudTrailCloudWatchLogsEnabled(node)) {
            const ruleId = 'HIPAA.Security-CloudTrailCloudWatchLogsEnabled';
            const info = 'The trail does not have CloudWatch logs enabled - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).';
            const explanation = 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudTrailEncryptionEnabled') &&
            !cloudtrail_1.hipaaSecurityCloudTrailEncryptionEnabled(node)) {
            const ruleId = 'HIPAA.Security-CloudTrailEncryptionEnabled';
            const info = 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudTrailLogFileValidationEnabled') &&
            !cloudtrail_1.hipaaSecurityCloudTrailLogFileValidationEnabled(node)) {
            const ruleId = 'HIPAA.Security-CloudTrailLogFileValidationEnabled';
            const info = 'The trail does not have log file validation enabled - (Control ID: 164.312(c)(1), 164.312(c)(2)).';
            const explanation = 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    //   /**
    //    * Check CloudWatch Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkCloudWatch(node: CfnResource, ignores: any): void {}
    //   /**
    //    * Check CodeBuild Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkCodeBuild(node: CfnResource, ignores: any): void {}
    //   /**
    //    * Check DMS Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkDMS(node: CfnResource, ignores: any): void {}
    //   /**
    //    * Check DynamoDB Resources
    //    * @param node the IConstruct to evaluate
    //    * @param ignores list of ignores for the resource
    //    */
    //   private checkDynamoDB(node: CfnResource, ignores: any): void {}
    /**
     * Check EC2 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEC2(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EC2InstanceDetailedMonitoringEnabled') &&
            !ec2_1.hipaaSecurityEC2InstanceDetailedMonitoringEnabled(node)) {
            const ruleId = 'HIPAA.Security-C2InstanceDetailedMonitoringEnabled';
            const info = 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: 164.312(b)).';
            const explanation = 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EC2InstancesInVPC') &&
            !ec2_1.hipaaSecurityEC2InstancesInVPC(node)) {
            const ruleId = 'HIPAA.Security-EC2InstancesInVPC';
            const info = 'The EC2 instance is not within a VPC - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EC2CheckNoPublicIPs') &&
            !ec2_1.hipaaSecurityEC2InstanceNoPublicIp(node)) {
            const ruleId = 'HIPAA.Security-EC2CheckNoPublicIPs';
            const info = 'The EC2 instance is associated with a public IP address - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
}
exports.HIPAASecurityChecks = HIPAASecurityChecks;
_a = JSII_RTTI_SYMBOL_1;
HIPAASecurityChecks[_a] = { fqn: "monocdk-nag.HIPAASecurityChecks", version: "0.0.54" };
//# sourceMappingURL=data:application/json;base64,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