"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsSubscription = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * Use an SQS queue as a subscription target.
 *
 * @stability stable
 */
class SqsSubscription {
    /**
     * @stability stable
     */
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        jsiiDeprecationWarnings._aws_cdk_aws_sns_subscriptions_SqsSubscriptionProps(props);
    }
    /**
     * Returns a configuration for an SQS queue to subscribe to an SNS topic.
     *
     * @stability stable
     */
    bind(topic) {
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!core_2.Construct.isConstruct(this.queue)) {
            throw new Error('The supplied Queue object must be an instance of Construct');
        }
        const snsServicePrincipal = new iam.ServicePrincipal('sns.amazonaws.com');
        // add a statement to the queue resource policy which allows this topic
        // to send messages to the queue.
        this.queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.queue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [snsServicePrincipal],
            conditions: {
                ArnEquals: { 'aws:SourceArn': topic.topicArn },
            },
        }));
        // if the queue is encrypted, add a statement to the key resource policy
        // which allows this topic to decrypt KMS keys
        if (this.queue.encryptionMasterKey) {
            this.queue.encryptionMasterKey.addToResourcePolicy(new iam.PolicyStatement({
                resources: ['*'],
                actions: ['kms:Decrypt', 'kms:GenerateDataKey'],
                principals: [snsServicePrincipal],
            }));
        }
        // if the topic and queue are created in different stacks
        // then we need to make sure the topic is created first
        if (topic instanceof sns.Topic && topic.stack !== this.queue.stack) {
            this.queue.stack.addDependency(topic.stack);
        }
        return {
            subscriberScope: this.queue,
            subscriberId: core_1.Names.nodeUniqueId(topic.node),
            endpoint: this.queue.queueArn,
            protocol: sns.SubscriptionProtocol.SQS,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
            region: this.regionFromArn(topic),
            deadLetterQueue: this.props.deadLetterQueue,
        };
    }
    regionFromArn(topic) {
        // no need to specify `region` for topics defined within the same stack
        if (topic instanceof sns.Topic) {
            if (topic.stack !== this.queue.stack) {
                // only if we know the region, will not work for
                // env agnostic stacks
                if (!core_1.Token.isUnresolved(topic.stack.region) &&
                    (topic.stack.region !== this.queue.stack.region)) {
                    return topic.stack.region;
                }
            }
            return undefined;
        }
        return core_1.Stack.of(topic).splitArn(topic.topicArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).region;
    }
}
exports.SqsSubscription = SqsSubscription;
_a = JSII_RTTI_SYMBOL_1;
SqsSubscription[_a] = { fqn: "@aws-cdk/aws-sns-subscriptions.SqsSubscription", version: "1.142.0" };
//# sourceMappingURL=data:application/json;base64,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