"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ifw = require(".");
class IntegTesting {
    constructor() {
        const app = new cdk.App();
        const env = {
            region: process.env.CDK_INTEG_REGION || process.env.CDK_DEFAULT_REGION || 'us-east-1',
            account: process.env.CDK_INTEG_ACCOUNT || process.env.CDK_DEFAULT_ACCOUNT,
        };
        const stack = new cdk.Stack(app, 'integ-stack', { env });
        const databaseName = 'FleetWise';
        const tableName = 'FleetWise';
        const database = new aws_cdk_lib_1.aws_timestream.CfnDatabase(stack, 'Database', {
            databaseName,
        });
        const table = new aws_cdk_lib_1.aws_timestream.CfnTable(stack, 'Table', {
            databaseName,
            tableName,
        });
        table.node.addDependency(database);
        const signalCatalog = new ifw.SignalCatalog(stack, 'SignalCatalog', {
            database,
            table,
            description: 'my signal catalog',
            nodes: [
                new ifw.SignalCatalogBranch('Vehicle'),
                new ifw.SignalCatalogSensor('Vehicle.EngineTorque', 'DOUBLE'),
            ],
        });
        const model_a = new ifw.VehicleModel(stack, 'ModelA', {
            signalCatalog,
            name: 'modelA',
            description: 'Model A vehicle',
            networkInterfaces: [new ifw.CanVehicleInterface('1', 'vcan0')],
            signals: [
                new ifw.CanVehicleSignal('Vehicle.EngineTorque', '1', 401, // messageId
                1.0, // factor
                true, // isBigEndian
                false, // isSigned
                8, // length
                0.0, // offset
                9),
            ],
        });
        const vin100 = new ifw.Vehicle(stack, 'vin100', {
            vehicleName: 'vin100',
            vehicleModel: model_a,
            createIotThing: true,
        });
        const vpc = aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(stack, 'VPC', { isDefault: true });
        const securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(stack, 'SecurityGroup', {
            vpc,
            allowAllOutbound: true,
        });
        securityGroup.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.anyIpv4(), aws_cdk_lib_1.aws_ec2.Port.tcp(22), 'SSH access');
        securityGroup.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.anyIpv4(), aws_cdk_lib_1.aws_ec2.Port.allIcmp(), 'ping');
        // EC2 role
        const ec2_role = new aws_cdk_lib_1.aws_iam.Role(stack, 'ec2Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            ],
        });
        ec2_role.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                's3:List*',
                's3:Get*',
            ],
            resources: ['arn:aws:s3:::*'],
        }));
        // Ubuntu 18.04 for Arm64
        const machineImage = aws_cdk_lib_1.aws_ec2.MachineImage.fromSsmParameter('/aws/service/canonical/ubuntu/server/18.04/stable/current/arm64/hvm/ebs-gp2/ami-id', { os: aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX });
        // Create the Vehicle simulator
        const keyName = stack.node.tryGetContext('key_name');
        const instance = new aws_cdk_lib_1.aws_ec2.Instance(stack, 'VehicleSim', {
            vpc: vpc,
            instanceType: new aws_cdk_lib_1.aws_ec2.InstanceType('m6g.xlarge'),
            machineImage,
            securityGroup,
            keyName,
            role: ec2_role,
            vpcSubnets: {
                subnetGroupName: 'Public',
            },
            resourceSignalTimeout: cdk.Duration.minutes(30),
        });
        const sourceUrl = 'https://github.com/aws/aws-iot-fleetwise-edge.git';
        const userData = `\
    set -euo pipefail

    # Wait for any existing package install to finish
    i=0
    while true; do
        if sudo fuser /var/{lib/{dpkg,apt/lists},cache/apt/archives}/lock >/dev/null 2>&1; then
            i=0
        else
            i=\`expr $i + 1\`
            if expr $i \\>= 10 > /dev/null; then
                break
            fi
        fi
        sleep 1
    done

    # Fix to ubuntu upgrade circular dep >>>
    #apt update && apt --only-upgrade -y install grub-efi-arm64-signed
    # <<<

    # Upgrade system and reboot if required
    apt update && apt upgrade -y
    if [ -f /var/run/reboot-required ]; then
      # Delete the UserData info file so that we run again after reboot
      rm -f /var/lib/cloud/instances/*/sem/config_scripts_user
      reboot
      exit
    fi

    # Install helper scripts:
    apt update && apt install -y python3-setuptools
    mkdir -p /opt/aws/bin
    wget https://s3.amazonaws.com/cloudformation-examples/aws-cfn-bootstrap-py3-latest.tar.gz
    python3 -m easy_install --script-dir /opt/aws/bin aws-cfn-bootstrap-py3-latest.tar.gz
    rm aws-cfn-bootstrap-py3-latest.tar.gz

    # On error, signal back to cfn:
    error_handler() {
      /opt/aws/bin/cfn-signal --success false --stack ${stack.stackName} --resource ${instance.instance.logicalId} --region ${stack.region}
    }
    trap error_handler ERR

    # Increase pid_max:
    echo 1048576 > /proc/sys/kernel/pid_max
    # Disable syslog:
    systemctl stop syslog.socket rsyslog.service
    # Remove journald rate limiting and set max size:
    printf "RateLimitBurst=0\nSystemMaxUse=1G\n" >> /etc/systemd/journald.conf

    # Install packages
    apt update && apt install -y git ec2-instance-connect htop jq unzip

    # Install AWS CLI:
    curl "https://awscli.amazonaws.com/awscli-exe-linux-aarch64.zip" -o "awscliv2.zip"
    unzip awscliv2.zip
    ./aws/install
    rm awscliv2.zip

    # Download source
    cd /home/ubuntu
    if echo ${sourceUrl} | grep -q 's3://'; then
      sudo -u ubuntu aws s3 cp ${sourceUrl} aws-iot-fleetwise-edge.zip
      sudo -u ubuntu unzip aws-iot-fleetwise-edge.zip -d aws-iot-fleetwise-edge
    else
      sudo -u ubuntu git clone ${sourceUrl} aws-iot-fleetwise-edge
    fi
    cd aws-iot-fleetwise-edge
    
    # Install SocketCAN modules:
    ./tools/install-socketcan.sh --bus-count 1
    
    # Install CAN Simulator
    ./tools/install-cansim.sh --bus-count 1
    
    # Install FWE credentials and config file
    mkdir /etc/aws-iot-fleetwise
    echo -n "${vin100.certificatePem}" > /etc/aws-iot-fleetwise/certificate.pem
    echo -n "${vin100.privateKey}" > /etc/aws-iot-fleetwise/private-key.key
    ./tools/configure-fwe.sh \
      --input-config-file "configuration/static-config.json" \
      --output-config-file "/etc/aws-iot-fleetwise/config-0.json" \
      --vehicle-name vin100 \
      --endpoint-url "${vin100.endpointAddress}" \
      --topic-prefix '$aws/iotfleetwise/' \
      --can-bus0 "vcan0"

    # Install source deps
    ./tools/install-deps-native.sh

    # Build source
    sudo -u ubuntu ./tools/build-fwe-native.sh

    # Install FWE
    ./tools/install-fwe.sh
    
    # Signal init complete:
    /opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${instance.instance.logicalId} --region ${stack.region}`;
        instance.addUserData(userData);
        new cdk.CfnOutput(stack, 'Vehicle Sim ssh command', { value: `ssh -i ${keyName}.pem ubuntu@${instance.instancePublicIp}` });
        new ifw.Campaign(stack, 'Campaign', {
            name: 'FwTimeBasedCampaign',
            target: vin100,
            collectionScheme: new ifw.TimeBasedCollectionScheme(cdk.Duration.seconds(10)),
            signals: [
                new ifw.CampaignSignal('Vehicle.EngineTorque'),
            ],
            autoApprove: true,
        });
        new ifw.Fleet(stack, 'Fleet', {
            fleetId: 'fleet',
            signalCatalog,
            vehicles: [vin100],
        });
        this.stack = [stack];
    }
}
exports.IntegTesting = IntegTesting;
new IntegTesting();
//# sourceMappingURL=data:application/json;base64,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