"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionedModelTransformer = void 0;
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphql_1 = require("graphql");
const graphql_mapping_template_1 = require("graphql-mapping-template");
const graphql_transformer_common_1 = require("graphql-transformer-common");
class VersionedModelTransformer extends graphql_transformer_core_1.Transformer {
    constructor() {
        super('VersionedModelTransformer', graphql_transformer_core_1.gql `
        directive @versioned(versionField: String = "version", versionInput: String = "expectedVersion") on OBJECT
      `);
        this.object = (def, directive, ctx) => {
            const modelDirective = def.directives.find(dir => dir.name.value === 'model');
            if (!modelDirective) {
                throw new graphql_transformer_core_1.InvalidDirectiveError('Types annotated with @versioned must also be annotated with @model.');
            }
            const isArg = (s) => (arg) => arg.name.value === s;
            const getArg = (arg, dflt) => {
                const argument = directive.arguments.find(isArg(arg));
                return argument ? graphql_1.valueFromASTUntyped(argument.value) : dflt;
            };
            const versionField = getArg('versionField', 'version');
            const versionInput = getArg('versionInput', 'expectedVersion');
            const typeName = def.name.value;
            this.augmentCreateMutation(ctx, typeName, versionField, versionInput);
            this.augmentUpdateMutation(ctx, typeName, versionField, versionInput);
            this.augmentDeleteMutation(ctx, typeName, versionField, versionInput);
            this.stripCreateInputVersionedField(ctx, typeName, versionField);
            this.addVersionedInputToDeleteInput(ctx, typeName, versionInput);
            this.addVersionedInputToUpdateInput(ctx, typeName, versionInput);
            this.enforceVersionedFieldOnType(ctx, typeName, versionField);
        };
    }
    augmentCreateMutation(ctx, typeName, versionField, versionInput) {
        const snippet = graphql_mapping_template_1.printBlock(`Setting "${versionField}" to 1`)(graphql_mapping_template_1.qref(`$ctx.args.input.put("${versionField}", 1)`));
        const mutationResolverLogicalId = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBCreateResolverResourceID(typeName);
        const resolver = ctx.getResource(mutationResolverLogicalId);
        if (resolver) {
            resolver.Properties.RequestMappingTemplate = snippet + '\n\n' + resolver.Properties.RequestMappingTemplate;
            ctx.setResource(mutationResolverLogicalId, resolver);
        }
    }
    augmentDeleteMutation(ctx, typeName, versionField, versionInput) {
        const mutationResolverLogicalId = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBDeleteResolverResourceID(typeName);
        const snippet = graphql_mapping_template_1.printBlock(`Inject @versioned condition.`)(graphql_mapping_template_1.compoundExpression([
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.VersionedCondition), graphql_mapping_template_1.obj({
                expression: graphql_mapping_template_1.str(`#${versionField} = :${versionInput}`),
                expressionValues: graphql_mapping_template_1.obj({
                    [`:${versionInput}`]: graphql_mapping_template_1.raw(`$util.dynamodb.toDynamoDB($ctx.args.input.${versionInput})`),
                }),
                expressionNames: graphql_mapping_template_1.obj({
                    [`#${versionField}`]: graphql_mapping_template_1.str(`${versionField}`),
                }),
            })),
            graphql_mapping_template_1.qref(`$ctx.args.input.remove("${versionInput}")`),
        ]));
        const resolver = ctx.getResource(mutationResolverLogicalId);
        if (resolver) {
            resolver.Properties.RequestMappingTemplate = snippet + '\n\n' + resolver.Properties.RequestMappingTemplate;
            ctx.setResource(mutationResolverLogicalId, resolver);
        }
    }
    augmentUpdateMutation(ctx, typeName, versionField, versionInput) {
        const mutationResolverLogicalId = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBUpdateResolverResourceID(typeName);
        const snippet = graphql_mapping_template_1.printBlock(`Inject @versioned condition.`)(graphql_mapping_template_1.compoundExpression([
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.VersionedCondition), graphql_mapping_template_1.obj({
                expression: graphql_mapping_template_1.str(`#${versionField} = :${versionInput}`),
                expressionValues: graphql_mapping_template_1.obj({
                    [`:${versionInput}`]: graphql_mapping_template_1.raw(`$util.dynamodb.toDynamoDB($ctx.args.input.${versionInput})`),
                }),
                expressionNames: graphql_mapping_template_1.obj({
                    [`#${versionField}`]: graphql_mapping_template_1.str(`${versionField}`),
                }),
            })),
            graphql_mapping_template_1.set(graphql_mapping_template_1.ref('newVersion'), graphql_mapping_template_1.raw(`$ctx.args.input.${versionInput} + 1`)),
            graphql_mapping_template_1.qref(`$ctx.args.input.put("${versionField}", $newVersion)`),
            graphql_mapping_template_1.qref(`$ctx.args.input.remove("${versionInput}")`),
        ]));
        const resolver = ctx.getResource(mutationResolverLogicalId);
        if (resolver) {
            resolver.Properties.RequestMappingTemplate = snippet + '\n\n' + resolver.Properties.RequestMappingTemplate;
            ctx.setResource(mutationResolverLogicalId, resolver);
        }
    }
    stripCreateInputVersionedField(ctx, typeName, versionField) {
        const createInputName = graphql_transformer_common_1.ModelResourceIDs.ModelCreateInputObjectName(typeName);
        const input = ctx.getType(createInputName);
        if (input && input.kind === graphql_1.Kind.INPUT_OBJECT_TYPE_DEFINITION) {
            const updatedFields = input.fields.filter(f => f.name.value !== versionField);
            if (updatedFields.length === 0) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`After stripping away version field "${versionField}", \
                    the create input for type "${typeName}" cannot be created \
                    with 0 fields. Add another field to type "${typeName}" to continue.`);
            }
            const updatedInput = {
                ...input,
                fields: updatedFields,
            };
            ctx.putType(updatedInput);
        }
    }
    addVersionedInputToUpdateInput(ctx, typeName, versionInput) {
        return this.addVersionedInputToInput(ctx, graphql_transformer_common_1.ModelResourceIDs.ModelUpdateInputObjectName(typeName), versionInput);
    }
    addVersionedInputToDeleteInput(ctx, typeName, versionInput) {
        return this.addVersionedInputToInput(ctx, graphql_transformer_common_1.ModelResourceIDs.ModelDeleteInputObjectName(typeName), versionInput);
    }
    addVersionedInputToInput(ctx, inputName, versionInput) {
        const input = ctx.getType(inputName);
        if (input && input.kind === graphql_1.Kind.INPUT_OBJECT_TYPE_DEFINITION) {
            const updatedFields = [...input.fields, graphql_transformer_common_1.makeInputValueDefinition(versionInput, graphql_transformer_common_1.makeNonNullType(graphql_transformer_common_1.makeNamedType('Int')))];
            const updatedInput = {
                ...input,
                fields: updatedFields,
            };
            ctx.putType(updatedInput);
        }
    }
    enforceVersionedFieldOnType(ctx, typeName, versionField) {
        const type = ctx.getType(typeName);
        if (type && type.kind === graphql_1.Kind.OBJECT_TYPE_DEFINITION) {
            let updatedFields = type.fields;
            const versionFieldImpl = type.fields.find(f => f.name.value === versionField);
            let updatedField = versionFieldImpl;
            if (versionFieldImpl) {
                const baseType = graphql_transformer_common_1.getBaseType(versionFieldImpl.type);
                if (baseType === 'Int' || baseType === 'BigInt') {
                    if (versionFieldImpl.type.kind !== graphql_1.Kind.NON_NULL_TYPE) {
                        updatedField = {
                            ...updatedField,
                            type: graphql_transformer_common_1.makeNonNullType(versionFieldImpl.type),
                        };
                        updatedFields = updatedFields.map(f => (f.name.value === versionField ? updatedField : f));
                    }
                }
                else {
                    throw new graphql_transformer_core_1.TransformerContractError(`The versionField "${versionField}" is required to be of type "Int" or "BigInt".`);
                }
            }
            else {
                updatedField = graphql_transformer_common_1.makeField(versionField, [], graphql_transformer_common_1.makeNonNullType(graphql_transformer_common_1.makeNamedType('Int')));
                updatedFields = [...updatedFields, updatedField];
            }
            const updatedType = {
                ...type,
                fields: updatedFields,
            };
            ctx.putType(updatedType);
        }
    }
}
exports.VersionedModelTransformer = VersionedModelTransformer;
//# sourceMappingURL=VersionedModelTransformer.js.map